/*! \file   janus_streaming.c
 * \author Lorenzo Miniero <lorenzo@meetecho.com>
 * \copyright GNU General Public License v3
 * \brief  Janus Streaming plugin
 * \details  This is a streaming plugin for Janus, allowing WebRTC peers
 * to watch/listen to pre-recorded files or media generated by another tool.
 * Specifically, the plugin currently supports three different type of streams:
 *
 * -# on-demand streaming of pre-recorded media files (different
 * streaming context for each peer);
 * -# live streaming of pre-recorded media files (shared streaming
 * context for all peers attached to the stream);
 * -# live streaming of media generated by another tool (shared
 * streaming context for all peers attached to the stream).
 *
 * For what concerns types 1. and 2., considering the proof of concept
 * nature of the implementation the only pre-recorded media files
 * that the plugins supports right now are raw mu-Law and a-Law files:
 * support is of course planned for other additional widespread formats
 * as well.
 *
 * For what concerns type 3., instead, the plugin is configured
 * to listen on a couple of ports for RTP: this means that the plugin
 * is implemented to receive RTP on those ports and relay them to all
 * peers attached to that stream. Any tool that can generate audio/video
 * RTP streams and specify a destination is good for the purpose: the
 * examples section contains samples that make use of GStreamer (http://gstreamer.freedesktop.org/)
 * but other tools like FFmpeg (http://www.ffmpeg.org/), LibAV (http://libav.org/)
 * or others are fine as well. This makes it really easy to capture and
 * encode whatever you want using your favourite tool, and then have it
 * transparently broadcasted via WebRTC using Janus. Notice that we recently
 * added  the possibility to also add a datachannel track to an RTP streaming
 * mountpoint: this allows you to send, via UDP, a text-based message to
 * relay via datachannels (e.g., the title of the current song, if this
 * is a radio streaming channel). When using this feature, though, beware
 * that you'll have to stay within the boundaries of the MTU, as each
 * message will have to stay within the size of an UDP packet.
 *
 * Streams to make available are listed in the plugin configuration file.
 * A pre-filled configuration file is provided in \c conf/janus.plugin.streaming.cfg
 * and includes a stream of every type.
 *
 * To add more streams or modify the existing ones, you can use the following
 * syntax:
 *
 * \verbatim
[stream-name]
type = rtp|live|ondemand|rtsp
       rtp = stream originated by an external tool (e.g., gstreamer or
             ffmpeg) and sent to the plugin via RTP
       live = local file streamed live to multiple listeners
              (multiple listeners = same streaming context)
       ondemand = local file streamed on-demand to a single listener
                  (multiple listeners = different streaming contexts)
       rtsp = stream originated by an external RTSP feed (only
              available if libcurl support was compiled)
id = <unique numeric ID>
description = This is my awesome stream
is_private = yes|no (private streams don't appear when you do a 'list' request)
filename = path to the local file to stream (only for live/ondemand)
secret = <optional password needed for manipulating (e.g., destroying
		or enabling/disabling) the stream>
pin = <optional password needed for watching the stream>
audio = yes|no (do/don't stream audio)
video = yes|no (do/don't stream video)
   The following options are only valid for the 'rtp' type:
data = yes|no (do/don't stream text via datachannels)
audioport = local port for receiving audio frames
audiomcast = multicast group port for receiving audio frames, if any
audioiface = network interface or IP address to bind to, if any (binds to all otherwise)
audiopt = <audio RTP payload type> (e.g., 111)
audiortpmap = RTP map of the audio codec (e.g., opus/48000/2)
audiofmtp = Codec specific parameters, if any
videoport = local port for receiving video frames (only for rtp)
videomcast = multicast group port for receiving video frames, if any
videoiface = network interface or IP address to bind to, if any (binds to all otherwise)
videopt = <video RTP payload type> (e.g., 100)
videortpmap = RTP map of the video codec (e.g., VP8/90000)
videofmtp = Codec specific parameters, if any
videobufferkf = yes|no (whether the plugin should store the latest
	keyframe and send it immediately for new viewers, EXPERIMENTAL)
videosimulcast = yes|no (do|don't enable video simulcasting)
videoport2 = second local port for receiving video frames (only for rtp, and simulcasting)
videoport3 = third local port for receiving video frames (only for rtp, and simulcasting)
dataport = local port for receiving data messages to relay
dataiface = network interface or IP address to bind to, if any (binds to all otherwise)
databuffermsg = yes|no (whether the plugin should store the latest
	message and send it immediately for new viewers)

The following options are only valid for the 'rstp' type:
url = RTSP stream URL (only if type=rtsp)
rtsp_user = RTSP authorization username (only if type=rtsp)
rtsp_pwd = RTSP authorization password (only if type=rtsp)
rtspiface = network interface IP address or device name to listen on when receiving RTSP streams
\endverbatim
 *
 * \section streamapi Streaming API
 *
 * The Streaming API supports several requests, some of which are
 * synchronous and some asynchronous. There are some situations, though,
 * (invalid JSON, invalid request) which will always result in a
 * synchronous error response even for asynchronous requests.
 *
 * \c list , \c create , \c destroy , \c recording , \c enable and
 * \c disable are synchronous requests, which means you'll
 * get a response directly within the context of the transaction. \c list
 * lists all the available streams; \c create allows you to create a new
 * mountpoint dynamically, as an alternative to using the configuration
 * file; \c destroy removes a mountpoint and destroys it; \c recording
 * instructs the plugin on whether or not a live RTP stream should be
 * recorded while it's broadcasted; \c enable and \c disable respectively
 * enable and disable a mountpoint, that is decide whether or not a
 * mountpoint should be available to users without destroying it.
 *
 * The \c watch , \c start , \c pause , \c switch and \c stop requests
 * instead are all asynchronous, which means you'll get a notification
 * about their success or failure in an event. \c watch asks the plugin
 * to prepare the playout of one of the available streams; \c start
 * starts the actual playout; \c pause allows you to pause a playout
 * without tearing down the PeerConnection; \c switch allows you to
 * switch to a different mountpoint of the same kind (note: only live
 * RTP mountpoints supported as of now) without having to stop and watch
 * the new one; \c stop stops the playout and tears the PeerConnection
 * down.
 *
 * Notice that, in general, all users can create mountpoints, no matter
 * what type they are. If you want to limit this functionality, you can
 * configure an admin \c admin_key in the plugin settings. When
 * configured, only "create" requests that include the correct
 * \c admin_key value in an "admin_key" property will succeed, and will
 * be rejected otherwise.
 *
 * Actual API docs: TBD.
 *
 * \ingroup plugins
 * \ref plugins
 */

#include "plugin.h"

#include <jansson.h>
#include <errno.h>
#include <sys/poll.h>
#include <sys/time.h>

#ifdef HAVE_LIBCURL
#include <curl/curl.h>
#endif

#include "../debug.h"
#include "../apierror.h"
#include "../config.h"
#include "../mutex.h"
#include "../rtp.h"
#include "../rtcp.h"
#include "../record.h"
#include "../utils.h"
#include "../ip-utils.h"


/* Plugin information */
#define JANUS_STREAMING_VERSION			8
#define JANUS_STREAMING_VERSION_STRING	"0.0.8"
#define JANUS_STREAMING_DESCRIPTION		"This is a streaming plugin for Janus, allowing WebRTC peers to watch/listen to pre-recorded files or media generated by gstreamer."
#define JANUS_STREAMING_NAME			"JANUS Streaming plugin"
#define JANUS_STREAMING_AUTHOR			"Meetecho s.r.l."
#define JANUS_STREAMING_PACKAGE			"janus.plugin.streaming"

/* Plugin methods */
janus_plugin *create(void);
int janus_streaming_init(janus_callbacks *callback, const char *config_path);
void janus_streaming_destroy(void);
int janus_streaming_get_api_compatibility(void);
int janus_streaming_get_version(void);
const char *janus_streaming_get_version_string(void);
const char *janus_streaming_get_description(void);
const char *janus_streaming_get_name(void);
const char *janus_streaming_get_author(void);
const char *janus_streaming_get_package(void);
void janus_streaming_create_session(janus_plugin_session *handle, int *error);
struct janus_plugin_result *janus_streaming_handle_message(janus_plugin_session *handle, char *transaction, json_t *message, json_t *jsep);
void janus_streaming_setup_media(janus_plugin_session *handle);
void janus_streaming_incoming_rtp(janus_plugin_session *handle, int video, char *buf, int len);
void janus_streaming_incoming_rtcp(janus_plugin_session *handle, int video, char *buf, int len);
void janus_streaming_hangup_media(janus_plugin_session *handle);
void janus_streaming_destroy_session(janus_plugin_session *handle, int *error);
json_t *janus_streaming_query_session(janus_plugin_session *handle);
static int janus_streaming_get_fd_port(int fd);

/* Plugin setup */
static janus_plugin janus_streaming_plugin =
	JANUS_PLUGIN_INIT (
		.init = janus_streaming_init,
		.destroy = janus_streaming_destroy,

		.get_api_compatibility = janus_streaming_get_api_compatibility,
		.get_version = janus_streaming_get_version,
		.get_version_string = janus_streaming_get_version_string,
		.get_description = janus_streaming_get_description,
		.get_name = janus_streaming_get_name,
		.get_author = janus_streaming_get_author,
		.get_package = janus_streaming_get_package,

		.create_session = janus_streaming_create_session,
		.handle_message = janus_streaming_handle_message,
		.setup_media = janus_streaming_setup_media,
		.incoming_rtp = janus_streaming_incoming_rtp,
		.incoming_rtcp = janus_streaming_incoming_rtcp,
		.hangup_media = janus_streaming_hangup_media,
		.destroy_session = janus_streaming_destroy_session,
		.query_session = janus_streaming_query_session,
	);

/* Plugin creator */
janus_plugin *create(void) {
	JANUS_LOG(LOG_VERB, "%s created!\n", JANUS_STREAMING_NAME);
	return &janus_streaming_plugin;
}

/* Parameter validation */
static struct janus_json_parameter request_parameters[] = {
	{"request", JSON_STRING, JANUS_JSON_PARAM_REQUIRED}
};
static struct janus_json_parameter id_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE}
};
static struct janus_json_parameter watch_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"pin", JSON_STRING, 0},
	{"offer_audio", JANUS_JSON_BOOL, 0},
	{"offer_video", JANUS_JSON_BOOL, 0},
	{"offer_data", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter adminkey_parameters[] = {
	{"admin_key", JSON_STRING, JANUS_JSON_PARAM_REQUIRED}
};
static struct janus_json_parameter create_parameters[] = {
	{"type", JSON_STRING, JANUS_JSON_PARAM_REQUIRED},
	{"secret", JSON_STRING, 0},
	{"pin", JSON_STRING, 0},
	{"permanent", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter rtp_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"name", JSON_STRING, 0},
	{"description", JSON_STRING, 0},
	{"is_private", JANUS_JSON_BOOL, 0},
	{"audio", JANUS_JSON_BOOL, 0},
	{"video", JANUS_JSON_BOOL, 0},
	{"data", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter live_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"name", JSON_STRING, 0},
	{"description", JSON_STRING, 0},
	{"is_private", JANUS_JSON_BOOL, 0},
	{"filename", JSON_STRING, JANUS_JSON_PARAM_REQUIRED},
	{"audio", JANUS_JSON_BOOL, 0},
	{"video", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter ondemand_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"name", JSON_STRING, 0},
	{"description", JSON_STRING, 0},
	{"is_private", JANUS_JSON_BOOL, 0},
	{"filename", JSON_STRING, JANUS_JSON_PARAM_REQUIRED},
	{"audio", JANUS_JSON_BOOL, 0},
	{"video", JANUS_JSON_BOOL, 0}
};
#ifdef HAVE_LIBCURL
static struct janus_json_parameter rtsp_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"name", JSON_STRING, 0},
	{"description", JSON_STRING, 0},
	{"is_private", JANUS_JSON_BOOL, 0},
	{"url", JSON_STRING, 0},
	{"rtsp_user", JSON_STRING, 0},
	{"rtsp_pwd", JSON_STRING, 0},
	{"audio", JANUS_JSON_BOOL, 0},
	{"audiortpmap", JSON_STRING, 0},
	{"audiofmtp", JSON_STRING, 0},
	{"video", JANUS_JSON_BOOL, 0},
	{"videortpmap", JSON_STRING, 0},
	{"videofmtp", JSON_STRING, 0},
	{"rtspiface", JSON_STRING, 0}
};
#endif
static struct janus_json_parameter rtp_audio_parameters[] = {
	{"audiomcast", JSON_STRING, 0},
	{"audioport", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"audiopt", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"audiortpmap", JSON_STRING, JANUS_JSON_PARAM_REQUIRED},
	{"audiofmtp", JSON_STRING, 0},
	{"audioiface", JSON_STRING, 0}
};
static struct janus_json_parameter rtp_video_parameters[] = {
	{"videomcast", JSON_STRING, 0},
	{"videoport", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"videopt", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"videortpmap", JSON_STRING, JANUS_JSON_PARAM_REQUIRED},
	{"videofmtp", JSON_STRING, 0},
	{"videobufferkf", JANUS_JSON_BOOL, 0},
	{"videoiface", JSON_STRING, 0},
	{"videosimulcast", JANUS_JSON_BOOL, 0},
	{"videoport2", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"videoport3", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
};
static struct janus_json_parameter rtp_data_parameters[] = {
	{"dataport", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"databuffermsg", JANUS_JSON_BOOL, 0},
	{"dataiface", JSON_STRING, 0}
};
static struct janus_json_parameter destroy_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"permanent", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter recording_parameters[] = {
	{"id", JSON_INTEGER, JANUS_JSON_PARAM_REQUIRED | JANUS_JSON_PARAM_POSITIVE},
	{"action", JSON_STRING, JANUS_JSON_PARAM_REQUIRED}
};
static struct janus_json_parameter recording_start_parameters[] = {
	{"audio", JSON_STRING, 0},
	{"video", JSON_STRING, 0},
	{"data", JSON_STRING, 0}
};
static struct janus_json_parameter recording_stop_parameters[] = {
	{"audio", JANUS_JSON_BOOL, 0},
	{"video", JANUS_JSON_BOOL, 0},
	{"data", JANUS_JSON_BOOL, 0}
};
static struct janus_json_parameter simulcast_parameters[] = {
	{"substream", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"temporal", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE}
};
static struct janus_json_parameter configure_parameters[] = {
	{"audio", JANUS_JSON_BOOL, 0},
	{"video", JANUS_JSON_BOOL, 0},
	{"data", JANUS_JSON_BOOL, 0},
	/* For VP8 simulcast */
	{"substream", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
	{"temporal", JSON_INTEGER, JANUS_JSON_PARAM_POSITIVE},
};

/* Static configuration instance */
static janus_config *config = NULL;
static const char *config_folder = NULL;
static janus_mutex config_mutex = JANUS_MUTEX_INITIALIZER;

/* Useful stuff */
static volatile gint initialized = 0, stopping = 0;
static gboolean notify_events = TRUE;
static janus_callbacks *gateway = NULL;
static GThread *handler_thread;
static GThread *watchdog;
static void *janus_streaming_handler(void *data);
static void *janus_streaming_ondemand_thread(void *data);
static void *janus_streaming_filesource_thread(void *data);
static void janus_streaming_relay_rtp_packet(gpointer data, gpointer user_data);
static void *janus_streaming_relay_thread(void *data);
static void janus_streaming_hangup_media_internal(janus_plugin_session *handle);

typedef enum janus_streaming_type {
	janus_streaming_type_none = 0,
	janus_streaming_type_live,
	janus_streaming_type_on_demand,
} janus_streaming_type;

typedef enum janus_streaming_source {
	janus_streaming_source_none = 0,
	janus_streaming_source_file,
	janus_streaming_source_rtp,
} janus_streaming_source;

typedef struct janus_streaming_rtp_keyframe {
	gboolean enabled;
	/* If enabled, we store the packets of the last keyframe, to immediately send them for new viewers */
	GList *latest_keyframe;
	/* This is where we store packets while we're still collecting the whole keyframe */
	GList *temp_keyframe;
	guint32 temp_ts;
	janus_mutex mutex;
} janus_streaming_rtp_keyframe;

#ifdef HAVE_LIBCURL
typedef struct janus_streaming_buffer {
	char *buffer;
	size_t size;
} janus_streaming_buffer;
#endif

typedef struct janus_streaming_rtp_source {
	gint audio_port;
	in_addr_t audio_mcast;
	gint video_port[3];
	in_addr_t video_mcast;
	gint data_port;
	janus_recorder *arc;	/* The Janus recorder instance for this streams's audio, if enabled */
	janus_recorder *vrc;	/* The Janus recorder instance for this streams's video, if enabled */
	janus_recorder *drc;	/* The Janus recorder instance for this streams's data, if enabled */
	janus_mutex rec_mutex;	/* Mutex to protect the recorders from race conditions */
	janus_rtp_switching_context context[3];
	int audio_fd;
	int video_fd[3];
	int data_fd;
	gboolean simulcast;
	gint64 last_received_audio;
	gint64 last_received_video;
	gint64 last_received_data;
#ifdef HAVE_LIBCURL
	gboolean rtsp;
	CURL *curl;
	janus_streaming_buffer *curldata;
	char *rtsp_url;
	char *rtsp_username, *rtsp_password;
	int ka_timeout;
	gboolean reconnecting;
	gint64 reconnect_timer;
	janus_mutex rtsp_mutex;
	int audio_rtcp_fd;
	int video_rtcp_fd;
#endif
	janus_streaming_rtp_keyframe keyframe;
	gboolean buffermsg;
	void *last_msg;
	janus_mutex buffermsg_mutex;
	janus_network_address audio_iface;
	janus_network_address video_iface;
	janus_network_address data_iface;
} janus_streaming_rtp_source;

typedef struct janus_streaming_file_source {
	char *filename;
} janus_streaming_file_source;

/* used for audio/video fd and rtcp fd */
typedef struct multiple_fds {
	int fd;
	int rtcp_fd;
} multiple_fds;

#define JANUS_STREAMING_VP8		0
#define JANUS_STREAMING_H264	1
#define JANUS_STREAMING_VP9		2
typedef struct janus_streaming_codecs {
	gint audio_pt;
	char *audio_rtpmap;
	char *audio_fmtp;
	gint video_codec;
	gint video_pt;
	char *video_rtpmap;
	char *video_fmtp;
} janus_streaming_codecs;

typedef struct janus_streaming_mountpoint {
	guint64 id;
	char *name;
	char *description;
	gboolean is_private;
	char *secret;
	char *pin;
	gboolean enabled;
	gboolean active;
	janus_streaming_type streaming_type;
	janus_streaming_source streaming_source;
	void *source;	/* Can differ according to the source type */
	GDestroyNotify source_destroy;
	janus_streaming_codecs codecs;
	gboolean audio, video, data;
	GList/*<unowned janus_streaming_session>*/ *listeners;
	gint64 destroyed;
	janus_mutex mutex;
} janus_streaming_mountpoint;
GHashTable *mountpoints;
static GList *old_mountpoints;
janus_mutex mountpoints_mutex;
static char *admin_key = NULL;

static void janus_streaming_mountpoint_free(janus_streaming_mountpoint *mp);

/* Helper to create an RTP live source (e.g., from gstreamer/ffmpeg/vlc/etc.) */
janus_streaming_mountpoint *janus_streaming_create_rtp_source(
		uint64_t id, char *name, char *desc,
		gboolean doaudio, char* amcast, const janus_network_address *aiface, uint16_t aport, uint8_t acodec, char *artpmap, char *afmtp,
		gboolean dovideo, char* vmcast, const janus_network_address *viface, uint16_t vport, uint8_t vcodec, char *vrtpmap, char *vfmtp, gboolean bufferkf,
			gboolean simulcast, uint16_t vport2, uint16_t vport3,
		gboolean dodata, const janus_network_address *diface, uint16_t dport, gboolean buffermsg);
/* Helper to create a file/ondemand live source */
janus_streaming_mountpoint *janus_streaming_create_file_source(
		uint64_t id, char *name, char *desc, char *filename,
		gboolean live, gboolean doaudio, gboolean dovideo);
/* Helper to create a rtsp live source */
janus_streaming_mountpoint *janus_streaming_create_rtsp_source(
		uint64_t id, char *name, char *desc,
		char *url, char *username, char *password,
		gboolean doaudio, char *artpmap, char *afmtp,
		gboolean dovideo, char *vrtpmap, char *vfmtp,
		const janus_network_address *iface);


typedef struct janus_streaming_message {
	janus_plugin_session *handle;
	char *transaction;
	json_t *message;
	json_t *jsep;
} janus_streaming_message;
static GAsyncQueue *messages = NULL;
static janus_streaming_message exit_message;

static void janus_streaming_message_free(janus_streaming_message *msg) {
	if(!msg || msg == &exit_message)
		return;

	msg->handle = NULL;

	g_free(msg->transaction);
	msg->transaction = NULL;
	if(msg->message)
		json_decref(msg->message);
	msg->message = NULL;
	if(msg->jsep)
		json_decref(msg->jsep);
	msg->jsep = NULL;

	g_free(msg);
}


typedef struct janus_streaming_session {
	janus_plugin_session *handle;
	janus_streaming_mountpoint *mountpoint;
	gboolean started;
	gboolean paused;
	gboolean audio, video, data;		/* Whether audio, video and/or data must be sent to this listener */
	janus_rtp_switching_context context;
	int substream;			/* Which simulcast substream we should forward, in case the mountpoint is simulcasting */
	int substream_target;	/* As above, but to handle transitions (e.g., wait for keyframe) */
	int templayer;			/* Which simulcast temporal layer we should forward, in case the mountpoint is simulcasting */
	int templayer_target;	/* As above, but to handle transitions (e.g., wait for keyframe) */
	gint64 last_relayed;	/* When we relayed the last packet (used to detect when substreams become unavailable) */
	janus_vp8_simulcast_context simulcast_context;
	gboolean stopping;
	volatile gint hangingup;
	gint64 destroyed;	/* Time at which this session was marked as destroyed */
} janus_streaming_session;
static GHashTable *sessions;
static GList *old_sessions;
static janus_mutex sessions_mutex = JANUS_MUTEX_INITIALIZER;

/* Packets we get from outside and relay */
typedef struct janus_streaming_rtp_relay_packet {
	janus_rtp_header *data;
	gint length;
	gboolean is_rtp;	/* This may be a data packet and not RTP */
	gboolean is_video;
	gboolean is_keyframe;
	gboolean simulcast;
	int codec, substream;
	uint32_t timestamp;
	uint16_t seq_number;
} janus_streaming_rtp_relay_packet;


/* Error codes */
#define JANUS_STREAMING_ERROR_NO_MESSAGE			450
#define JANUS_STREAMING_ERROR_INVALID_JSON			451
#define JANUS_STREAMING_ERROR_INVALID_REQUEST		452
#define JANUS_STREAMING_ERROR_MISSING_ELEMENT		453
#define JANUS_STREAMING_ERROR_INVALID_ELEMENT		454
#define JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT	455
#define JANUS_STREAMING_ERROR_CANT_CREATE			456
#define JANUS_STREAMING_ERROR_UNAUTHORIZED			457
#define JANUS_STREAMING_ERROR_CANT_SWITCH			458
#define JANUS_STREAMING_ERROR_CANT_RECORD			459
#define JANUS_STREAMING_ERROR_UNKNOWN_ERROR			470


/* Streaming watchdog/garbage collector (sort of) */
static void *janus_streaming_watchdog(void *data) {
	JANUS_LOG(LOG_INFO, "Streaming watchdog started\n");
	gint64 now = 0;
	while(g_atomic_int_get(&initialized) && !g_atomic_int_get(&stopping)) {
		janus_mutex_lock(&sessions_mutex);
		/* Iterate on all the sessions */
		now = janus_get_monotonic_time();
		if(old_sessions != NULL) {
			GList *sl = old_sessions;
			JANUS_LOG(LOG_HUGE, "Checking %d old Streaming sessions...\n", g_list_length(old_sessions));
			while(sl) {
				janus_streaming_session *session = (janus_streaming_session *)sl->data;
				if(!session) {
					sl = sl->next;
					continue;
				}
				if(now-session->destroyed >= 5*G_USEC_PER_SEC) {
					/* We're lazy and actually get rid of the stuff only after a few seconds */
					JANUS_LOG(LOG_VERB, "Freeing old Streaming session\n");
					GList *rm = sl->next;
					old_sessions = g_list_delete_link(old_sessions, sl);
					sl = rm;
					session->handle = NULL;
					g_free(session);
					session = NULL;
					continue;
				}
				sl = sl->next;
			}
		}
		janus_mutex_unlock(&sessions_mutex);
		janus_mutex_lock(&mountpoints_mutex);
		/* Iterate on all the mountpoints */
		if(old_mountpoints != NULL) {
			GList *sl = old_mountpoints;
			now = janus_get_monotonic_time();
			JANUS_LOG(LOG_HUGE, "Checking %d old Streaming mountpoints...\n", g_list_length(old_mountpoints));
			while(sl) {
				janus_streaming_mountpoint *mountpoint = (janus_streaming_mountpoint *)sl->data;
				if(!mountpoint) {
					sl = sl->next;
					continue;
				}
				if(now-mountpoint->destroyed >= 5*G_USEC_PER_SEC) {
					/* We're lazy and actually get rid of the stuff only after a few seconds */
					JANUS_LOG(LOG_VERB, "Freeing old Streaming mountpoint\n");
					GList *rm = sl->next;
					old_mountpoints = g_list_delete_link(old_mountpoints, sl);
					sl = rm;
					janus_streaming_mountpoint_free(mountpoint);
					mountpoint = NULL;
					continue;
				}
				sl = sl->next;
			}
		}
		janus_mutex_unlock(&mountpoints_mutex);
		g_usleep(500000);
	}
	JANUS_LOG(LOG_INFO, "Streaming watchdog stopped\n");
	return NULL;
}

/* Plugin implementation */
int janus_streaming_init(janus_callbacks *callback, const char *config_path) {
#ifdef HAVE_LIBCURL
	curl_global_init(CURL_GLOBAL_ALL);
#endif
	if(g_atomic_int_get(&stopping)) {
		/* Still stopping from before */
		return -1;
	}
	if(callback == NULL || config_path == NULL) {
		/* Invalid arguments */
		return -1;
	}

	struct ifaddrs *ifas = NULL;
	if(getifaddrs(&ifas) || ifas == NULL) {
		JANUS_LOG(LOG_ERR, "Unable to acquire list of network devices/interfaces; some configurations may not work as expected...\n");
	}

	/* Read configuration */
	char filename[255];
	g_snprintf(filename, 255, "%s/%s.cfg", config_path, JANUS_STREAMING_PACKAGE);
	JANUS_LOG(LOG_VERB, "Configuration file: %s\n", filename);
	config = janus_config_parse(filename);
	config_folder = config_path;
	if(config != NULL)
		janus_config_print(config);

	mountpoints = g_hash_table_new_full(g_int64_hash, g_int64_equal, (GDestroyNotify)g_free, NULL);

	/* Threads will expect this to be set */
	g_atomic_int_set(&initialized, 1);

	/* Parse configuration to populate the mountpoints */
	if(config != NULL) {
		/* Any admin key to limit who can "create"? */
		janus_config_item *key = janus_config_get_item_drilldown(config, "general", "admin_key");
		if(key != NULL && key->value != NULL)
			admin_key = g_strdup(key->value);
		janus_config_item *events = janus_config_get_item_drilldown(config, "general", "events");
		if(events != NULL && events->value != NULL)
			notify_events = janus_is_true(events->value);
		if(!notify_events && callback->events_is_enabled()) {
			JANUS_LOG(LOG_WARN, "Notification of events to handlers disabled for %s\n", JANUS_STREAMING_NAME);
		}
		/* Iterate on all rooms */
		GList *cl = janus_config_get_categories(config);
		while(cl != NULL) {
			janus_config_category *cat = (janus_config_category *)cl->data;
			if(cat->name == NULL || !strcasecmp(cat->name, "general")) {
				cl = cl->next;
				continue;
			}
			JANUS_LOG(LOG_VERB, "Adding stream '%s'\n", cat->name);
			janus_config_item *type = janus_config_get_item(cat, "type");
			if(type == NULL || type->value == NULL) {
				JANUS_LOG(LOG_WARN, "  -- Invalid type, skipping stream '%s'...\n", cat->name);
				cl = cl->next;
				continue;
			}
			if(!strcasecmp(type->value, "rtp")) {
				janus_network_address video_iface, audio_iface, data_iface;
				/* RTP live source (e.g., from gstreamer/ffmpeg/vlc/etc.) */
				janus_config_item *id = janus_config_get_item(cat, "id");
				janus_config_item *desc = janus_config_get_item(cat, "description");
				janus_config_item *priv = janus_config_get_item(cat, "is_private");
				janus_config_item *secret = janus_config_get_item(cat, "secret");
				janus_config_item *pin = janus_config_get_item(cat, "pin");
				janus_config_item *audio = janus_config_get_item(cat, "audio");
				janus_config_item *video = janus_config_get_item(cat, "video");
				janus_config_item *data = janus_config_get_item(cat, "data");
				janus_config_item *diface = janus_config_get_item(cat, "dataiface");
				janus_config_item *amcast = janus_config_get_item(cat, "audiomcast");
				janus_config_item *aiface = janus_config_get_item(cat, "audioiface");
				janus_config_item *aport = janus_config_get_item(cat, "audioport");
				janus_config_item *acodec = janus_config_get_item(cat, "audiopt");
				janus_config_item *artpmap = janus_config_get_item(cat, "audiortpmap");
				janus_config_item *afmtp = janus_config_get_item(cat, "audiofmtp");
				janus_config_item *vmcast = janus_config_get_item(cat, "videomcast");
				janus_config_item *viface = janus_config_get_item(cat, "videoiface");
				janus_config_item *vport = janus_config_get_item(cat, "videoport");
				janus_config_item *vcodec = janus_config_get_item(cat, "videopt");
				janus_config_item *vrtpmap = janus_config_get_item(cat, "videortpmap");
				janus_config_item *vfmtp = janus_config_get_item(cat, "videofmtp");
				janus_config_item *vkf = janus_config_get_item(cat, "videobufferkf");
				janus_config_item *vsc = janus_config_get_item(cat, "videosimulcast");
				janus_config_item *vport2 = janus_config_get_item(cat, "videoport2");
				janus_config_item *vport3 = janus_config_get_item(cat, "videoport3");
				janus_config_item *dport = janus_config_get_item(cat, "dataport");
				janus_config_item *dbm = janus_config_get_item(cat, "databuffermsg");
				gboolean is_private = priv && priv->value && janus_is_true(priv->value);
				gboolean doaudio = audio && audio->value && janus_is_true(audio->value);
				gboolean dovideo = video && video->value && janus_is_true(video->value);
				gboolean dodata = data && data->value && janus_is_true(data->value);
				gboolean bufferkf = video && vkf && vkf->value && janus_is_true(vkf->value);
				gboolean simulcast = video && vsc && vsc->value && janus_is_true(vsc->value);
				if(simulcast && bufferkf) {
					/* FIXME We'll need to take care of this */
					JANUS_LOG(LOG_WARN, "Simulcasting enabled, so disabling buffering of keyframes\n");
					bufferkf = FALSE;
				}
				gboolean buffermsg = data && dbm && dbm->value && janus_is_true(dbm->value);
				if(!doaudio && !dovideo && !dodata) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', no audio, video or data have to be streamed...\n", cat->name);
					cl = cl->next;
					continue;
				}
				if(doaudio &&
						(aport == NULL || aport->value == NULL || atoi(aport->value) == 0 ||
						acodec == NULL || acodec->value == NULL ||
						artpmap == NULL || artpmap->value == NULL)) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', missing mandatory information for audio...\n", cat->name);
					cl = cl->next;
					continue;
				}
				if(doaudio && aiface) {
					if(!ifas) {
						JANUS_LOG(LOG_ERR, "Skipping 'rtp' stream '%s', it relies on network configuration but network device information is unavailable...\n", cat->name);
						cl = cl->next;
						continue;
					}
					if(janus_network_lookup_interface(ifas, aiface->value, &audio_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for audio...\n", cat->name);
						cl = cl->next;
						continue;
					}
				}
				if(dovideo &&
						(vport == NULL || vport->value == NULL || atoi(vport->value) == 0 ||
						vcodec == NULL || vcodec->value == NULL ||
						vrtpmap == NULL || vrtpmap->value == NULL)) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', missing mandatory information for video...\n", cat->name);
					cl = cl->next;
					continue;
				}
				if(dodata && (dport == NULL || dport->value == NULL || atoi(dport->value) == 0)) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', missing mandatory information for data...\n", cat->name);
					cl = cl->next;
					continue;
				}
#ifndef HAVE_SCTP
				if(dodata) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s': no datachannels support......\n", cat->name);
					cl = cl->next;
					continue;
				}
#endif
				if(dodata && diface) {
					if(!ifas) {
						JANUS_LOG(LOG_ERR, "Skipping 'rtp' stream '%s', it relies on network configuration but network device information is unavailable...\n", cat->name);
						cl = cl->next;
						continue;
					}
					if(janus_network_lookup_interface(ifas, diface->value, &data_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for data...\n", cat->name);
						cl = cl->next;
						continue;
					}
				}
				if(dovideo && viface) {
					if(!ifas) {
						JANUS_LOG(LOG_ERR, "Skipping 'rtp' stream '%s', it relies on network configuration but network device information is unavailable...\n", cat->name);
						cl = cl->next;
						continue;
					}
					if(janus_network_lookup_interface(ifas, viface->value, &video_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for video...\n", cat->name);
						cl = cl->next;
						continue;
					}
				}
				if(id == NULL || id->value == NULL) {
					JANUS_LOG(LOG_VERB, "Missing id for stream '%s', will generate a random one...\n", cat->name);
				} else {
					janus_mutex_lock(&mountpoints_mutex);
					guint64 mpid = g_ascii_strtoull(id->value, 0, 10);
					janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &mpid);
					janus_mutex_unlock(&mountpoints_mutex);
					if(mp != NULL) {
						JANUS_LOG(LOG_ERR, "A stream with the provided ID %s already exists, skipping '%s'\n", id->value, cat->name);
						cl = cl->next;
						continue;
					}
				}
				JANUS_LOG(LOG_VERB, "Audio %s, Video %s, Data %s\n",
					doaudio ? "enabled" : "NOT enabled",
					dovideo ? "enabled" : "NOT enabled",
					dodata ? "enabled" : "NOT enabled");
				janus_streaming_mountpoint *mp = NULL;
				if((mp = janus_streaming_create_rtp_source(
						(id && id->value) ? g_ascii_strtoull(id->value, 0, 10) : 0,
						(char *)cat->name,
						desc ? (char *)desc->value : NULL,
						doaudio,
						amcast ? (char *)amcast->value : NULL,
						doaudio && aiface && aiface->value ? &audio_iface : NULL,
						(aport && aport->value) ? atoi(aport->value) : 0,
						(acodec && acodec->value) ? atoi(acodec->value) : 0,
						artpmap ? (char *)artpmap->value : NULL,
						afmtp ? (char *)afmtp->value : NULL,
						dovideo,
						vmcast ? (char *)vmcast->value : NULL,
						dovideo && viface && viface->value ? &video_iface : NULL,
						(vport && vport->value) ? atoi(vport->value) : 0,
						(vcodec && vcodec->value) ? atoi(vcodec->value) : 0,
						vrtpmap ? (char *)vrtpmap->value : NULL,
						vfmtp ? (char *)vfmtp->value : NULL,
						bufferkf,
						simulcast,
						(vport2 && vport2->value) ? atoi(vport2->value) : 0,
						(vport3 && vport3->value) ? atoi(vport3->value) : 0,
						dodata,
						dodata && diface && diface->value ? &data_iface : NULL,
						(dport && dport->value) ? atoi(dport->value) : 0,
						buffermsg)) == NULL) {
					JANUS_LOG(LOG_ERR, "Error creating 'rtp' stream '%s'...\n", cat->name);
					cl = cl->next;
					continue;
				}
				mp->is_private = is_private;
				if(secret && secret->value)
					mp->secret = g_strdup(secret->value);
				if(pin && pin->value)
					mp->pin = g_strdup(pin->value);
			} else if(!strcasecmp(type->value, "live")) {
				/* File live source */
				janus_config_item *id = janus_config_get_item(cat, "id");
				janus_config_item *desc = janus_config_get_item(cat, "description");
				janus_config_item *priv = janus_config_get_item(cat, "is_private");
				janus_config_item *secret = janus_config_get_item(cat, "secret");
				janus_config_item *pin = janus_config_get_item(cat, "pin");
				janus_config_item *file = janus_config_get_item(cat, "filename");
				janus_config_item *audio = janus_config_get_item(cat, "audio");
				janus_config_item *video = janus_config_get_item(cat, "video");
				if(file == NULL || file->value == NULL) {
					JANUS_LOG(LOG_ERR, "Can't add 'live' stream '%s', missing mandatory information...\n", cat->name);
					cl = cl->next;
					continue;
				}
				gboolean is_private = priv && priv->value && janus_is_true(priv->value);
				gboolean doaudio = audio && audio->value && janus_is_true(audio->value);
				gboolean dovideo = video && video->value && janus_is_true(video->value);
				/* TODO We should support something more than raw a-Law and mu-Law streams... */
				if(!doaudio || dovideo) {
					JANUS_LOG(LOG_ERR, "Can't add 'live' stream '%s', we only support audio file streaming right now...\n", cat->name);
					cl = cl->next;
					continue;
				}
				if(!strstr(file->value, ".alaw") && !strstr(file->value, ".mulaw")) {
					JANUS_LOG(LOG_ERR, "Can't add 'live' stream '%s', unsupported format (we only support raw mu-Law and a-Law files right now)\n", cat->name);
					cl = cl->next;
					continue;
				}
				FILE *audiofile = fopen(file->value, "rb");
				if(!audiofile) {
					JANUS_LOG(LOG_ERR, "Can't add 'live' stream, no such file '%s'...\n", file->value);
					cl = cl->next;
					continue;
				}
				fclose(audiofile);
				if(id == NULL || id->value == NULL) {
					JANUS_LOG(LOG_VERB, "Missing id for stream '%s', will generate a random one...\n", cat->name);
				} else {
					janus_mutex_lock(&mountpoints_mutex);
					guint64 mpid = g_ascii_strtoull(id->value, 0, 10);
					janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &mpid);
					janus_mutex_unlock(&mountpoints_mutex);
					if(mp != NULL) {
						JANUS_LOG(LOG_ERR, "A stream with the provided ID %s already exists, skipping '%s'\n", id->value, cat->name);
						cl = cl->next;
						continue;
					}
				}
				janus_streaming_mountpoint *mp = NULL;
				if((mp = janus_streaming_create_file_source(
						(id && id->value) ? g_ascii_strtoull(id->value, 0, 10) : 0,
						(char *)cat->name,
						desc ? (char *)desc->value : NULL,
						(char *)file->value,
						TRUE, doaudio, dovideo)) == NULL) {
					JANUS_LOG(LOG_ERR, "Error creating 'live' stream '%s'...\n", cat->name);
					cl = cl->next;
					continue;
				}
				mp->is_private = is_private;
				if(secret && secret->value)
					mp->secret = g_strdup(secret->value);
				if(pin && pin->value)
					mp->pin = g_strdup(pin->value);
			} else if(!strcasecmp(type->value, "ondemand")) {
				/* mu-Law file on demand source */
				janus_config_item *id = janus_config_get_item(cat, "id");
				janus_config_item *desc = janus_config_get_item(cat, "description");
				janus_config_item *priv = janus_config_get_item(cat, "is_private");
				janus_config_item *secret = janus_config_get_item(cat, "secret");
				janus_config_item *pin = janus_config_get_item(cat, "pin");
				janus_config_item *file = janus_config_get_item(cat, "filename");
				janus_config_item *audio = janus_config_get_item(cat, "audio");
				janus_config_item *video = janus_config_get_item(cat, "video");
				if(file == NULL || file->value == NULL) {
					JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream '%s', missing mandatory information...\n", cat->name);
					cl = cl->next;
					continue;
				}
				gboolean is_private = priv && priv->value && janus_is_true(priv->value);
				gboolean doaudio = audio && audio->value && janus_is_true(audio->value);
				gboolean dovideo = video && video->value && janus_is_true(video->value);
				/* TODO We should support something more than raw a-Law and mu-Law streams... */
				if(!doaudio || dovideo) {
					JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream '%s', we only support audio file streaming right now...\n", cat->name);
					cl = cl->next;
					continue;
				}
				if(!strstr(file->value, ".alaw") && !strstr(file->value, ".mulaw")) {
					JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream '%s', unsupported format (we only support raw mu-Law and a-Law files right now)\n", cat->name);
					cl = cl->next;
					continue;
				}
				FILE *audiofile = fopen(file->value, "rb");
				if(!audiofile) {
					JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream, no such file '%s'...\n", file->value);
					cl = cl->next;
					continue;
				}
				fclose(audiofile);
				if(id == NULL || id->value == NULL) {
					JANUS_LOG(LOG_VERB, "Missing id for stream '%s', will generate a random one...\n", cat->name);
				} else {
					janus_mutex_lock(&mountpoints_mutex);
					guint64 mpid = g_ascii_strtoull(id->value, 0, 10);
					janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &mpid);
					janus_mutex_unlock(&mountpoints_mutex);
					if(mp != NULL) {
						JANUS_LOG(LOG_ERR, "A stream with the provided ID %s already exists, skipping '%s'\n", id->value, cat->name);
						cl = cl->next;
						continue;
					}
				}
				janus_streaming_mountpoint *mp = NULL;
				if((mp = janus_streaming_create_file_source(
						(id && id->value) ? g_ascii_strtoull(id->value, 0, 10) : 0,
						(char *)cat->name,
						desc ? (char *)desc->value : NULL,
						(char *)file->value,
						FALSE, doaudio, dovideo)) == NULL) {
					JANUS_LOG(LOG_ERR, "Error creating 'ondemand' stream '%s'...\n", cat->name);
					cl = cl->next;
					continue;
				}
				mp->is_private = is_private;
				if(secret && secret->value)
					mp->secret = g_strdup(secret->value);
				if(pin && pin->value)
					mp->pin = g_strdup(pin->value);
			} else if(!strcasecmp(type->value, "rtsp")) {
#ifndef HAVE_LIBCURL
				JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream '%s', libcurl support not compiled...\n", cat->name);
				cl = cl->next;
				continue;
#else
				janus_config_item *id = janus_config_get_item(cat, "id");
				janus_config_item *desc = janus_config_get_item(cat, "description");
				janus_config_item *priv = janus_config_get_item(cat, "is_private");
				janus_config_item *secret = janus_config_get_item(cat, "secret");
				janus_config_item *pin = janus_config_get_item(cat, "pin");
				janus_config_item *file = janus_config_get_item(cat, "url");
				janus_config_item *username = janus_config_get_item(cat, "rtsp_user");
				janus_config_item *password = janus_config_get_item(cat, "rtsp_pwd");
				janus_config_item *audio = janus_config_get_item(cat, "audio");
				janus_config_item *artpmap = janus_config_get_item(cat, "audiortpmap");
				janus_config_item *afmtp = janus_config_get_item(cat, "audiofmtp");
				janus_config_item *video = janus_config_get_item(cat, "video");
				janus_config_item *vrtpmap = janus_config_get_item(cat, "videortpmap");
				janus_config_item *vfmtp = janus_config_get_item(cat, "videofmtp");
				janus_config_item *iface = janus_config_get_item(cat, "rtspiface");
				janus_network_address iface_value;
				if(file == NULL || file->value == NULL) {
					JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream '%s', missing mandatory information...\n", cat->name);
					cl = cl->next;
					continue;
				}
				gboolean is_private = priv && priv->value && janus_is_true(priv->value);
				gboolean doaudio = audio && audio->value && janus_is_true(audio->value);
				gboolean dovideo = video && video->value && janus_is_true(video->value);

				if((doaudio || dovideo) && iface && iface->value) {
					if(!ifas) {
						JANUS_LOG(LOG_ERR, "Skipping 'rtsp' stream '%s', it relies on network configuration but network device information is unavailable...\n", cat->name);
						cl = cl->next;
						continue;
					}
					if(janus_network_lookup_interface(ifas, iface->value, &iface_value) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream '%s', invalid network interface configuration for stream...\n", cat->name);
						cl = cl->next;
						continue;
					}
				}

				if(id == NULL || id->value == NULL) {
					JANUS_LOG(LOG_VERB, "Missing id for stream '%s', will generate a random one...\n", cat->name);
				} else {
					janus_mutex_lock(&mountpoints_mutex);
					guint64 mpid = g_ascii_strtoull(id->value, 0, 10);
					janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &mpid);
					janus_mutex_unlock(&mountpoints_mutex);
					if(mp != NULL) {
						JANUS_LOG(LOG_ERR, "A stream with the provided ID %s already exists, skipping '%s'\n", id->value, cat->name);
						cl = cl->next;
						continue;
					}
				}
				janus_streaming_mountpoint *mp = NULL;
				if((mp = janus_streaming_create_rtsp_source(
						(id && id->value) ? g_ascii_strtoull(id->value, 0, 10) : 0,
						(char *)cat->name,
						desc ? (char *)desc->value : NULL,
						(char *)file->value,
						username ? (char *)username->value : NULL,
						password ? (char *)password->value : NULL,
						doaudio,
						artpmap ? (char *)artpmap->value : NULL,
						afmtp ? (char *)afmtp->value : NULL,
						dovideo,
						vrtpmap ? (char *)vrtpmap->value : NULL,
						vfmtp ? (char *)vfmtp->value : NULL,
						iface && iface->value ? &iface_value : NULL)) == NULL) {
					JANUS_LOG(LOG_ERR, "Error creating 'rtsp' stream '%s'...\n", cat->name);
					cl = cl->next;
					continue;
				}
				mp->is_private = is_private;
				if(secret && secret->value)
					mp->secret = g_strdup(secret->value);
				if(pin && pin->value)
					mp->pin = g_strdup(pin->value);
#endif
			} else {
				JANUS_LOG(LOG_WARN, "Ignoring unknown stream type '%s' (%s)...\n", type->value, cat->name);
			}
			cl = cl->next;
		}
		/* Done: we keep the configuration file open in case we get a "create" or "destroy" with permanent=true */
	}
	if(ifas) {
		freeifaddrs(ifas);
	}

	/* Show available mountpoints */
	janus_mutex_lock(&mountpoints_mutex);
	GHashTableIter iter;
	gpointer value;
	g_hash_table_iter_init(&iter, mountpoints);
	while(g_hash_table_iter_next(&iter, NULL, &value)) {
		janus_streaming_mountpoint *mp = value;
		JANUS_LOG(LOG_VERB, "  ::: [%"SCNu64"][%s] %s (%s, %s, %s, pin: %s)\n", mp->id, mp->name, mp->description,
			mp->streaming_type == janus_streaming_type_live ? "live" : "on demand",
			mp->streaming_source == janus_streaming_source_rtp ? "RTP source" : "file source",
			mp->is_private ? "private" : "public",
			mp->pin ? mp->pin : "no pin");
	}
	janus_mutex_unlock(&mountpoints_mutex);

	sessions = g_hash_table_new(NULL, NULL);
	messages = g_async_queue_new_full((GDestroyNotify) janus_streaming_message_free);
	/* This is the callback we'll need to invoke to contact the gateway */
	gateway = callback;

	GError *error = NULL;
	/* Start the sessions watchdog */
	watchdog = g_thread_try_new("streaming watchdog", &janus_streaming_watchdog, NULL, &error);
	if(!watchdog) {
		g_atomic_int_set(&initialized, 0);
		JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the Streaming watchdog thread...\n", error->code, error->message ? error->message : "??");
		janus_config_destroy(config);
		return -1;
	}
	/* Launch the thread that will handle incoming messages */
	handler_thread = g_thread_try_new("streaming handler", janus_streaming_handler, NULL, &error);
	if(error != NULL) {
		g_atomic_int_set(&initialized, 0);
		JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the Streaming handler thread...\n", error->code, error->message ? error->message : "??");
		janus_config_destroy(config);
		return -1;
	}
	JANUS_LOG(LOG_INFO, "%s initialized!\n", JANUS_STREAMING_NAME);
	return 0;
}

void janus_streaming_destroy(void) {
	if(!g_atomic_int_get(&initialized))
		return;
	g_atomic_int_set(&stopping, 1);

	g_async_queue_push(messages, &exit_message);

	if(handler_thread != NULL) {
		g_thread_join(handler_thread);
		handler_thread = NULL;
	}

	/* Remove all mountpoints */
	janus_mutex_lock(&mountpoints_mutex);
	GHashTableIter iter;
	gpointer value;
	g_hash_table_iter_init(&iter, mountpoints);
	while(g_hash_table_iter_next(&iter, NULL, &value)) {
		janus_streaming_mountpoint *mp = value;
		if(!mp->destroyed) {
			mp->destroyed = janus_get_monotonic_time();
			old_mountpoints = g_list_append(old_mountpoints, mp);
		}
	}
	janus_mutex_unlock(&mountpoints_mutex);
	if(watchdog != NULL) {
		g_thread_join(watchdog);
		watchdog = NULL;
	}

	/* FIXME We should destroy the sessions cleanly */
	usleep(500000);
	janus_mutex_lock(&mountpoints_mutex);
	g_hash_table_destroy(mountpoints);
	janus_mutex_unlock(&mountpoints_mutex);
	janus_mutex_lock(&sessions_mutex);
	g_hash_table_destroy(sessions);
	janus_mutex_unlock(&sessions_mutex);
	g_async_queue_unref(messages);
	messages = NULL;
	sessions = NULL;

	janus_config_destroy(config);
	g_free(admin_key);

	g_atomic_int_set(&initialized, 0);
	g_atomic_int_set(&stopping, 0);
	JANUS_LOG(LOG_INFO, "%s destroyed!\n", JANUS_STREAMING_NAME);
}

int janus_streaming_get_api_compatibility(void) {
	/* Important! This is what your plugin MUST always return: don't lie here or bad things will happen */
	return JANUS_PLUGIN_API_VERSION;
}

int janus_streaming_get_version(void) {
	return JANUS_STREAMING_VERSION;
}

const char *janus_streaming_get_version_string(void) {
	return JANUS_STREAMING_VERSION_STRING;
}

const char *janus_streaming_get_description(void) {
	return JANUS_STREAMING_DESCRIPTION;
}

const char *janus_streaming_get_name(void) {
	return JANUS_STREAMING_NAME;
}

const char *janus_streaming_get_author(void) {
	return JANUS_STREAMING_AUTHOR;
}

const char *janus_streaming_get_package(void) {
	return JANUS_STREAMING_PACKAGE;
}

static janus_streaming_session *janus_streaming_lookup_session(janus_plugin_session *handle) {
	janus_streaming_session *session = NULL;
	if (g_hash_table_contains(sessions, handle)) {
		session = (janus_streaming_session *)handle->plugin_handle;
	}
	return session;
}

void janus_streaming_create_session(janus_plugin_session *handle, int *error) {
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized)) {
		*error = -1;
		return;
	}
	janus_streaming_session *session = (janus_streaming_session *)g_malloc0(sizeof(janus_streaming_session));
	session->handle = handle;
	session->mountpoint = NULL;	/* This will happen later */
	session->started = FALSE;	/* This will happen later */
	session->paused = FALSE;
	session->destroyed = 0;
	g_atomic_int_set(&session->hangingup, 0);
	handle->plugin_handle = session;
	janus_mutex_lock(&sessions_mutex);
	g_hash_table_insert(sessions, handle, session);
	janus_mutex_unlock(&sessions_mutex);

	return;
}

void janus_streaming_destroy_session(janus_plugin_session *handle, int *error) {
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized)) {
		*error = -1;
		return;
	}
	janus_mutex_lock(&sessions_mutex);
	janus_streaming_session *session = janus_streaming_lookup_session(handle);
	if(!session) {
		janus_mutex_unlock(&sessions_mutex);
		JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
		*error = -2;
		return;
	}
	if(!session->destroyed) {
		JANUS_LOG(LOG_VERB, "Removing streaming session...\n");
		janus_streaming_mountpoint *mp = session->mountpoint;
		if(mp) {
			janus_mutex_lock(&mp->mutex);
			mp->listeners = g_list_remove_all(mp->listeners, session);
			janus_mutex_unlock(&mp->mutex);
		}
		janus_streaming_hangup_media_internal(handle);
		session->destroyed = janus_get_monotonic_time();
		g_hash_table_remove(sessions, handle);
		/* Cleaning up and removing the session is done in a lazy way */
		old_sessions = g_list_append(old_sessions, session);
	}
	janus_mutex_unlock(&sessions_mutex);
	return;
}

json_t *janus_streaming_query_session(janus_plugin_session *handle) {
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized)) {
		return NULL;
	}
	janus_mutex_lock(&sessions_mutex);
	janus_streaming_session *session = janus_streaming_lookup_session(handle);
	if(!session) {
		janus_mutex_unlock(&sessions_mutex);
		JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
		return NULL;
	}
	/* What is this user watching, if anything? */
	json_t *info = json_object();
	janus_streaming_mountpoint *mp = session->mountpoint;
	json_object_set_new(info, "state", json_string(mp ? "watching" : "idle"));
	if(mp) {
		json_object_set_new(info, "mountpoint_id", json_integer(mp->id));
		json_object_set_new(info, "mountpoint_name", mp->name ? json_string(mp->name) : NULL);
	}
	json_object_set_new(info, "destroyed", json_integer(session->destroyed));
	janus_mutex_unlock(&sessions_mutex);
	return info;
}

struct janus_plugin_result *janus_streaming_handle_message(janus_plugin_session *handle, char *transaction, json_t *message, json_t *jsep) {
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized))
		return janus_plugin_result_new(JANUS_PLUGIN_ERROR, g_atomic_int_get(&stopping) ? "Shutting down" : "Plugin not initialized", NULL);

	/* Pre-parse the message */
	int error_code = 0;
	char error_cause[512];
	json_t *root = message;
	json_t *response = NULL;
	struct ifaddrs *ifas = NULL;

	janus_mutex_lock(&sessions_mutex);

	if(message == NULL) {
		JANUS_LOG(LOG_ERR, "No message??\n");
		error_code = JANUS_STREAMING_ERROR_NO_MESSAGE;
		g_snprintf(error_cause, 512, "%s", "No message??");
		goto plugin_response;
	}

	janus_streaming_session *session = janus_streaming_lookup_session(handle);
	if(!session) {
		JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
		error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
		g_snprintf(error_cause, 512, "%s", "session associated with this handle...");
		goto plugin_response;
	}
	if(session->destroyed) {
		JANUS_LOG(LOG_ERR, "Session has already been destroyed...\n");
		error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
		g_snprintf(error_cause, 512, "%s", "Session has already been destroyed...");
		goto plugin_response;
	}
	if(!json_is_object(root)) {
		JANUS_LOG(LOG_ERR, "JSON error: not an object\n");
		error_code = JANUS_STREAMING_ERROR_INVALID_JSON;
		g_snprintf(error_cause, 512, "JSON error: not an object");
		goto plugin_response;
	}
	/* Get the request first */
	JANUS_VALIDATE_JSON_OBJECT(root, request_parameters,
		error_code, error_cause, TRUE,
		JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
	if(error_code != 0)
		goto plugin_response;
	json_t *request = json_object_get(root, "request");
	/* Some requests ('create' and 'destroy') can be handled synchronously */
	const char *request_text = json_string_value(request);
	if(!strcasecmp(request_text, "list")) {
		json_t *list = json_array();
		JANUS_LOG(LOG_VERB, "Request for the list of mountpoints\n");
		/* Return a list of all available mountpoints */
		janus_mutex_lock(&mountpoints_mutex);
		GHashTableIter iter;
		gpointer value;
		g_hash_table_iter_init(&iter, mountpoints);
		while(g_hash_table_iter_next(&iter, NULL, &value)) {
			janus_streaming_mountpoint *mp = value;
			if(mp->is_private) {
				/* Skip private stream */
				JANUS_LOG(LOG_VERB, "Skipping private mountpoint '%s'\n", mp->description);
				continue;
			}
			json_t *ml = json_object();
			json_object_set_new(ml, "id", json_integer(mp->id));
			json_object_set_new(ml, "description", json_string(mp->description));
			json_object_set_new(ml, "type", json_string(mp->streaming_type == janus_streaming_type_live ? "live" : "on demand"));
			if(mp->streaming_source == janus_streaming_source_rtp) {
				janus_streaming_rtp_source *source = mp->source;
				gint64 now = janus_get_monotonic_time();
				if(source->audio_fd != -1)
					json_object_set_new(ml, "audio_age_ms", json_integer((now - source->last_received_audio) / 1000));
				if(source->video_fd[0] != -1 || source->video_fd[1] != -1 || source->video_fd[2] != -1)
					json_object_set_new(ml, "video_age_ms", json_integer((now - source->last_received_video) / 1000));
			}
			json_array_append_new(list, ml);
		}
		janus_mutex_unlock(&mountpoints_mutex);
		/* Send info back */
		response = json_object();
		json_object_set_new(response, "streaming", json_string("list"));
		json_object_set_new(response, "list", list);
		goto plugin_response;
	} else if(!strcasecmp(request_text, "info")) {
		JANUS_LOG(LOG_VERB, "Request info on a specific mountpoint\n");
		/* Return info on a specific mountpoint */
		JANUS_VALIDATE_JSON_OBJECT(root, id_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto plugin_response;
		json_t *id = json_object_get(root, "id");
		guint64 id_value = json_integer_value(id);
		janus_mutex_lock(&mountpoints_mutex);
		janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
		if(mp == NULL) {
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
			error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
			g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
			goto plugin_response;
		}
		/* Return more info if the right secret is provided */
		gboolean admin = FALSE;
		if(mp->secret) {
			json_t *secret = json_object_get(root, "secret");
			if(secret && json_string_value(secret) && janus_strcmp_const_time(mp->secret, json_string_value(secret)))
				admin = TRUE;
		}
		json_t *ml = json_object();
		json_object_set_new(ml, "id", json_integer(mp->id));
		if(admin && mp->name)
			json_object_set_new(ml, "name", json_string(mp->name));
		if(mp->description)
			json_object_set_new(ml, "description", json_string(mp->description));
		if(admin && mp->secret)
			json_object_set_new(ml, "secret", json_string(mp->secret));
		if(admin && mp->pin)
			json_object_set_new(ml, "pin", json_string(mp->pin));
		if(admin && mp->is_private)
			json_object_set_new(ml, "is_private", json_true());
		if(mp->audio) {
			json_object_set_new(ml, "audio", json_true());
			if(mp->codecs.audio_pt != -1)
				json_object_set_new(ml, "audiopt", json_integer(mp->codecs.audio_pt));
			if(mp->codecs.audio_rtpmap)
				json_object_set_new(ml, "audiortpmap", json_string(mp->codecs.audio_rtpmap));
			if(mp->codecs.audio_fmtp)
				json_object_set_new(ml, "audiofmtp", json_string(mp->codecs.audio_fmtp));
		}
		if(mp->video) {
			json_object_set_new(ml, "video", json_true());
			if(mp->codecs.video_pt != -1)
				json_object_set_new(ml, "videopt", json_integer(mp->codecs.video_pt));
			if(mp->codecs.video_rtpmap)
				json_object_set_new(ml, "videortpmap", json_string(mp->codecs.video_rtpmap));
			if(mp->codecs.video_fmtp)
				json_object_set_new(ml, "videofmtp", json_string(mp->codecs.video_fmtp));
		}
		if(mp->data) {
			json_object_set_new(ml, "data", json_true());
		}
		json_object_set_new(ml, "type", json_string(mp->streaming_type == janus_streaming_type_live ? "live" : "on demand"));
		if(mp->streaming_source == janus_streaming_source_file) {
			janus_streaming_file_source *source = mp->source;
			if(admin && source->filename)
				json_object_set_new(ml, "filename", json_string(source->filename));
		} else if(mp->streaming_source == janus_streaming_source_rtp) {
			janus_streaming_rtp_source *source = mp->source;
			gint64 now = janus_get_monotonic_time();
#ifdef HAVE_LIBCURL
			if(source->rtsp) {
				json_object_set_new(ml, "rtsp", json_true());
				if(admin) {
					if(source->rtsp_url)
						json_object_set_new(ml, "url", json_string(source->rtsp_url));
					if(source->rtsp_username)
						json_object_set_new(ml, "rtsp_user", json_string(source->rtsp_username));
					if(source->rtsp_password)
						json_object_set_new(ml, "rtsp_pwd", json_string(source->rtsp_password));
				}
			}
#endif
			if(source->keyframe.enabled) {
				json_object_set_new(ml, "videobufferkf", json_true());
			}
			if(source->simulcast) {
				json_object_set_new(ml, "videosimulcast", json_true());
			}
			if(admin) {
				if(mp->audio)
					json_object_set_new(ml, "audioport", json_integer(source->audio_port));
				if(mp->video) {
					json_object_set_new(ml, "videoport", json_integer(source->video_port[0]));
					if(source->video_port[1] > -1)
						json_object_set_new(ml, "videoport2", json_integer(source->video_port[1]));
					if(source->video_port[2] > -1)
						json_object_set_new(ml, "videoport3", json_integer(source->video_port[2]));
				}
				if(mp->data)
					json_object_set_new(ml, "dataport", json_integer(source->data_port));
			}
			if(source->audio_fd != -1)
				json_object_set_new(ml, "audio_age_ms", json_integer((now - source->last_received_audio) / 1000));
			if(source->video_fd[0] != -1 || source->video_fd[1] != -1 || source->video_fd[2] != -1)
				json_object_set_new(ml, "video_age_ms", json_integer((now - source->last_received_video) / 1000));
			if(source->data_fd != -1)
				json_object_set_new(ml, "data_age_ms", json_integer((now - source->last_received_data) / 1000));
			janus_mutex_lock(&source->rec_mutex);
			if(admin && (source->arc || source->vrc || source->drc)) {
				json_t *recording = json_object();
				if(source->arc && source->arc->filename)
					json_object_set_new(recording, "audio", json_string(source->arc->filename));
				if(source->vrc && source->vrc->filename)
					json_object_set_new(recording, "video", json_string(source->vrc->filename));
				if(source->drc && source->drc->filename)
					json_object_set_new(recording, "data", json_string(source->drc->filename));
				json_object_set_new(ml, "recording", recording);
			}
			janus_mutex_unlock(&source->rec_mutex);
		}
		janus_mutex_unlock(&mountpoints_mutex);
		/* Send info back */
		response = json_object();
		json_object_set_new(response, "streaming", json_string("info"));
		json_object_set_new(response, "info", ml);
		goto plugin_response;
	} else if(!strcasecmp(request_text, "create")) {
		/* Create a new stream */
		JANUS_VALIDATE_JSON_OBJECT(root, create_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto plugin_response;
		if(admin_key != NULL) {
			/* An admin key was specified: make sure it was provided, and that it's valid */
			JANUS_VALIDATE_JSON_OBJECT(root, adminkey_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto plugin_response;
			JANUS_CHECK_SECRET(admin_key, root, "admin_key", error_code, error_cause,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT, JANUS_STREAMING_ERROR_UNAUTHORIZED);
			if(error_code != 0)
				goto plugin_response;
		}

		if(getifaddrs(&ifas) || ifas == NULL) {
			JANUS_LOG(LOG_ERR, "Unable to acquire list of network devices/interfaces; some configurations may not work as expected...\n");
		}

		json_t *type = json_object_get(root, "type");
		const char *type_text = json_string_value(type);
		json_t *secret = json_object_get(root, "secret");
		json_t *pin = json_object_get(root, "pin");
		json_t *permanent = json_object_get(root, "permanent");
		gboolean save = permanent ? json_is_true(permanent) : FALSE;
		if(save && config == NULL) {
			JANUS_LOG(LOG_ERR, "No configuration file, can't create permanent mountpoint\n");
			error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
			g_snprintf(error_cause, 512, "No configuration file, can't create permanent mountpoint");
			goto plugin_response;
		}
		janus_streaming_mountpoint *mp = NULL;
		if(!strcasecmp(type_text, "rtp")) {
			janus_network_address audio_iface, video_iface, data_iface;
			/* RTP live source (e.g., from gstreamer/ffmpeg/vlc/etc.) */
			JANUS_VALIDATE_JSON_OBJECT(root, rtp_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto plugin_response;
			json_t *id = json_object_get(root, "id");
			json_t *name = json_object_get(root, "name");
			json_t *desc = json_object_get(root, "description");
			json_t *is_private = json_object_get(root, "is_private");
			json_t *audio = json_object_get(root, "audio");
			json_t *video = json_object_get(root, "video");
			json_t *data = json_object_get(root, "data");
			gboolean doaudio = audio ? json_is_true(audio) : FALSE;
			gboolean dovideo = video ? json_is_true(video) : FALSE;
			gboolean dodata = data ? json_is_true(data) : FALSE;
			if(!doaudio && !dovideo && !dodata) {
				JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream, no audio, video or data have to be streamed...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'rtp' stream, no audio or video have to be streamed...");
				goto plugin_response;
			}
			uint16_t aport = 0;
			uint8_t acodec = 0;
			char *artpmap = NULL, *afmtp = NULL, *amcast = NULL;
			if(doaudio) {
				JANUS_VALIDATE_JSON_OBJECT(root, rtp_audio_parameters,
					error_code, error_cause, TRUE,
					JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
				if(error_code != 0)
					goto plugin_response;
				json_t *audiomcast = json_object_get(root, "audiomcast");
				amcast = (char *)json_string_value(audiomcast);
				json_t *audioport = json_object_get(root, "audioport");
				aport = json_integer_value(audioport);
				json_t *audiopt = json_object_get(root, "audiopt");
				acodec = json_integer_value(audiopt);
				json_t *audiortpmap = json_object_get(root, "audiortpmap");
				artpmap = (char *)json_string_value(audiortpmap);
				json_t *audiofmtp = json_object_get(root, "audiofmtp");
				afmtp = (char *)json_string_value(audiofmtp);
				json_t *aiface = json_object_get(root, "audioiface");
				if(aiface) {
					const char *miface = (const char *)json_string_value(aiface);
					if(janus_network_lookup_interface(ifas, miface, &audio_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for audio...\n", (const char *)json_string_value(name));
						error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
						g_snprintf(error_cause, 512, ifas ? "Invalid network interface configuration for audio" : "Unable to query network device information");
						goto plugin_response;
					}
				} else {
					janus_network_address_nullify(&audio_iface);
				}
			}
			uint16_t vport = 0, vport2 = 0, vport3 = 0;
			uint8_t vcodec = 0;
			char *vrtpmap = NULL, *vfmtp = NULL, *vmcast = NULL;
			gboolean bufferkf = FALSE, simulcast = FALSE;
			if(dovideo) {
				JANUS_VALIDATE_JSON_OBJECT(root, rtp_video_parameters,
					error_code, error_cause, TRUE,
					JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
				if(error_code != 0)
					goto plugin_response;
				json_t *videomcast = json_object_get(root, "videomcast");
				vmcast = (char *)json_string_value(videomcast);
				json_t *videoport = json_object_get(root, "videoport");
				vport = json_integer_value(videoport);
				json_t *videopt = json_object_get(root, "videopt");
				vcodec = json_integer_value(videopt);
				json_t *videortpmap = json_object_get(root, "videortpmap");
				vrtpmap = (char *)json_string_value(videortpmap);
				json_t *videofmtp = json_object_get(root, "videofmtp");
				vfmtp = (char *)json_string_value(videofmtp);
				json_t *vkf = json_object_get(root, "videobufferkf");
				bufferkf = vkf ? json_is_true(vkf) : FALSE;
				json_t *vsc = json_object_get(root, "videosimulcast");
				simulcast = vsc ? json_is_true(vsc) : FALSE;
				if(simulcast && bufferkf) {
					/* FIXME We'll need to take care of this */
					JANUS_LOG(LOG_WARN, "Simulcasting enabled, so disabling buffering of keyframes\n");
					bufferkf = FALSE;
				}
				json_t *videoport2 = json_object_get(root, "videoport2");
				vport2 = json_integer_value(videoport2);
				json_t *videoport3 = json_object_get(root, "videoport3");
				vport3 = json_integer_value(videoport3);
				json_t *viface = json_object_get(root, "videoiface");
				if(viface) {
					const char *miface = (const char *)json_string_value(viface);
					if(janus_network_lookup_interface(ifas, miface, &video_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for video...\n", (const char *)json_string_value(name));
						error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
						g_snprintf(error_cause, 512, ifas ? "Invalid network interface configuration for video" : "Unable to query network device information");
						goto plugin_response;
					}
				} else {
					janus_network_address_nullify(&video_iface);
				}
			}
			uint16_t dport = 0;
			gboolean buffermsg = FALSE;
			if(dodata) {
				JANUS_VALIDATE_JSON_OBJECT(root, rtp_data_parameters,
					error_code, error_cause, TRUE,
					JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
				if(error_code != 0)
					goto plugin_response;
#ifdef HAVE_SCTP
				json_t *dataport = json_object_get(root, "dataport");
				dport = json_integer_value(dataport);
				json_t *dbm = json_object_get(root, "databuffermsg");
				buffermsg = dbm ? json_is_true(dbm) : FALSE;
				json_t *diface = json_object_get(root, "dataiface");
				if(diface) {
					const char *miface = (const char *)json_string_value(diface);
					if(janus_network_lookup_interface(ifas, miface, &data_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream '%s', invalid network interface configuration for data...\n", (const char *)json_string_value(name));
						error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
						g_snprintf(error_cause, 512, ifas ? "Invalid network interface configuration for data" : "Unable to query network device information");
						goto plugin_response;
					}
				} else {
					janus_network_address_nullify(&data_iface);
				}
#else
				JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream: no datachannels support...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'rtp' stream: no datachannels support...");
				goto plugin_response;
#endif
			}
			if(id == NULL) {
				JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
			} else {
				janus_mutex_lock(&mountpoints_mutex);
				guint64 mpid = json_integer_value(id);
				mp = g_hash_table_lookup(mountpoints, &mpid);
				janus_mutex_unlock(&mountpoints_mutex);
				if(mp != NULL) {
					JANUS_LOG(LOG_ERR, "A stream with the provided ID already exists\n");
					error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
					g_snprintf(error_cause, 512, "A stream with the provided ID already exists");
					goto plugin_response;
				}
			}
			JANUS_LOG(LOG_VERB, "Audio %s, Video %s\n", doaudio ? "enabled" : "NOT enabled", dovideo ? "enabled" : "NOT enabled");
			mp = janus_streaming_create_rtp_source(
					id ? json_integer_value(id) : 0,
					name ? (char *)json_string_value(name) : NULL,
					desc ? (char *)json_string_value(desc) : NULL,
					doaudio, amcast, &audio_iface, aport, acodec, artpmap, afmtp,
					dovideo, vmcast, &video_iface, vport, vcodec, vrtpmap, vfmtp, bufferkf,
					simulcast, vport2, vport3,
					dodata, &data_iface, dport, buffermsg);
			if(mp == NULL) {
				JANUS_LOG(LOG_ERR, "Error creating 'rtp' stream...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Error creating 'rtp' stream");
				goto plugin_response;
			}
			mp->is_private = is_private ? json_is_true(is_private) : FALSE;
		} else if(!strcasecmp(type_text, "live")) {
			/* File live source */
			JANUS_VALIDATE_JSON_OBJECT(root, live_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto plugin_response;
			json_t *id = json_object_get(root, "id");
			json_t *name = json_object_get(root, "name");
			json_t *desc = json_object_get(root, "description");
			json_t *is_private = json_object_get(root, "is_private");
			json_t *file = json_object_get(root, "filename");
			json_t *audio = json_object_get(root, "audio");
			json_t *video = json_object_get(root, "video");
			gboolean doaudio = audio ? json_is_true(audio) : FALSE;
			gboolean dovideo = video ? json_is_true(video) : FALSE;
			/* TODO We should support something more than raw a-Law and mu-Law streams... */
			if(!doaudio || dovideo) {
				JANUS_LOG(LOG_ERR, "Can't add 'live' stream, we only support audio file streaming right now...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'live' stream, we only support audio file streaming right now...");
				goto plugin_response;
			}
			char *filename = (char *)json_string_value(file);
			if(!strstr(filename, ".alaw") && !strstr(filename, ".mulaw")) {
				JANUS_LOG(LOG_ERR, "Can't add 'live' stream, unsupported format (we only support raw mu-Law and a-Law files right now)\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'live' stream, unsupported format (we only support raw mu-Law and a-Law files right now)");
				goto plugin_response;
			}
			FILE *audiofile = fopen(filename, "rb");
			if(!audiofile) {
				JANUS_LOG(LOG_ERR, "Can't add 'live' stream, no such file '%s'...\n", filename);
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'live' stream, no such file '%s'\n", filename);
				goto plugin_response;
			}
			fclose(audiofile);
			if(id == NULL) {
				JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
			} else {
				janus_mutex_lock(&mountpoints_mutex);
				guint64 mpid = json_integer_value(id);
				mp = g_hash_table_lookup(mountpoints, &mpid);
				janus_mutex_unlock(&mountpoints_mutex);
				if(mp != NULL) {
					JANUS_LOG(LOG_ERR, "A stream with the provided ID already exists\n");
					error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
					g_snprintf(error_cause, 512, "A stream with the provided ID already exists");
					goto plugin_response;
				}
			}
			mp = janus_streaming_create_file_source(
					id ? json_integer_value(id) : 0,
					name ? (char *)json_string_value(name) : NULL,
					desc ? (char *)json_string_value(desc) : NULL,
					filename,
					TRUE, doaudio, dovideo);
			if(mp == NULL) {
				JANUS_LOG(LOG_ERR, "Error creating 'live' stream...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Error creating 'live' stream");
				goto plugin_response;
			}
			mp->is_private = is_private ? json_is_true(is_private) : FALSE;
		} else if(!strcasecmp(type_text, "ondemand")) {
			/* mu-Law file on demand source */
			JANUS_VALIDATE_JSON_OBJECT(root, ondemand_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto plugin_response;
			json_t *id = json_object_get(root, "id");
			json_t *name = json_object_get(root, "name");
			json_t *desc = json_object_get(root, "description");
			json_t *is_private = json_object_get(root, "is_private");
			json_t *file = json_object_get(root, "filename");
			json_t *audio = json_object_get(root, "audio");
			json_t *video = json_object_get(root, "video");
			gboolean doaudio = audio ? json_is_true(audio) : FALSE;
			gboolean dovideo = video ? json_is_true(video) : FALSE;
			/* TODO We should support something more than raw a-Law and mu-Law streams... */
			if(!doaudio || dovideo) {
				JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream, we only support audio file streaming right now...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'ondemand' stream, we only support audio file streaming right now...");
				goto plugin_response;
			}
			char *filename = (char *)json_string_value(file);
			if(!strstr(filename, ".alaw") && !strstr(filename, ".mulaw")) {
				JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream, unsupported format (we only support raw mu-Law and a-Law files right now)\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'ondemand' stream, unsupported format (we only support raw mu-Law and a-Law files right now)");
				goto plugin_response;
			}
			FILE *audiofile = fopen(filename, "rb");
			if(!audiofile) {
				JANUS_LOG(LOG_ERR, "Can't add 'ondemand' stream, no such file '%s'...\n", filename);
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'ondemand' stream, no such file '%s'\n", filename);
				goto plugin_response;
			}
			fclose(audiofile);
			if(id == NULL) {
				JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
			} else {
				janus_mutex_lock(&mountpoints_mutex);
				guint64 mpid = json_integer_value(id);
				mp = g_hash_table_lookup(mountpoints, &mpid);
				janus_mutex_unlock(&mountpoints_mutex);
				if(mp != NULL) {
					JANUS_LOG(LOG_ERR, "A stream with the provided ID already exists\n");
					error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
					g_snprintf(error_cause, 512, "A stream with the provided ID already exists");
					goto plugin_response;
				}
			}
			mp = janus_streaming_create_file_source(
					id ? json_integer_value(id) : 0,
					name ? (char *)json_string_value(name) : NULL,
					desc ? (char *)json_string_value(desc) : NULL,
					filename,
					FALSE, doaudio, dovideo);
			if(mp == NULL) {
				JANUS_LOG(LOG_ERR, "Error creating 'ondemand' stream...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Error creating 'ondemand' stream");
				goto plugin_response;
			}
			mp->is_private = is_private ? json_is_true(is_private) : FALSE;
		} else if(!strcasecmp(type_text, "rtsp")) {
#ifndef HAVE_LIBCURL
			JANUS_LOG(LOG_ERR, "Can't create 'rtsp' mountpoint, libcurl support not compiled...\n");
			error_code = JANUS_STREAMING_ERROR_INVALID_ELEMENT;
			g_snprintf(error_cause, 512, "Can't create 'rtsp' mountpoint, libcurl support not compiled...\n");
			goto plugin_response;
#else
			JANUS_VALIDATE_JSON_OBJECT(root, rtsp_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto plugin_response;
			/* RTSP source*/
			janus_network_address multicast_iface;
			json_t *id = json_object_get(root, "id");
			json_t *name = json_object_get(root, "name");
			json_t *desc = json_object_get(root, "description");
			json_t *is_private = json_object_get(root, "is_private");
			json_t *audio = json_object_get(root, "audio");
			json_t *audiortpmap = json_object_get(root, "audiortpmap");
			json_t *audiofmtp = json_object_get(root, "audiofmtp");
			json_t *video = json_object_get(root, "video");
			json_t *videortpmap = json_object_get(root, "videortpmap");
			json_t *videofmtp = json_object_get(root, "videofmtp");
			json_t *url = json_object_get(root, "url");
			json_t *username = json_object_get(root, "rtsp_user");
			json_t *password = json_object_get(root, "rtsp_pwd");
			json_t *iface = json_object_get(root, "rtspiface");
			gboolean doaudio = audio ? json_is_true(audio) : FALSE;
			gboolean dovideo = video ? json_is_true(video) : FALSE;
			if(!doaudio && !dovideo) {
				JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream, no audio or video have to be streamed...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Can't add 'rtsp' stream, no audio or video have to be streamed...");
				goto plugin_response;
			} else {
				if(iface) {
					const char *miface = (const char *)json_string_value(iface);
					if(janus_network_lookup_interface(ifas, miface, &multicast_iface) != 0) {
						JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream '%s', invalid network interface configuration for stream...\n", (const char *)json_string_value(name));
						error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
						g_snprintf(error_cause, 512, ifas ? "Invalid network interface configuration for stream" : "Unable to query network device information");
						goto plugin_response;
					}
				} else {
					janus_network_address_nullify(&multicast_iface);
				}
			}
			mp = janus_streaming_create_rtsp_source(
					id ? json_integer_value(id) : 0,
					name ? (char *)json_string_value(name) : NULL,
					desc ? (char *)json_string_value(desc) : NULL,
					(char *)json_string_value(url),
					username ? (char *)json_string_value(username) : NULL,
					password ? (char *)json_string_value(password) : NULL,
					doaudio, (char *)json_string_value(audiortpmap), (char *)json_string_value(audiofmtp),
					dovideo, (char *)json_string_value(videortpmap), (char *)json_string_value(videofmtp),
					&multicast_iface);
			if(mp == NULL) {
				JANUS_LOG(LOG_ERR, "Error creating 'rtsp' stream...\n");
				error_code = JANUS_STREAMING_ERROR_CANT_CREATE;
				g_snprintf(error_cause, 512, "Error creating 'RTSP' stream");
				goto plugin_response;
			}
			mp->is_private = is_private ? json_is_true(is_private) : FALSE;
#endif
		} else {
			JANUS_LOG(LOG_ERR, "Unknown stream type '%s'...\n", type_text);
			error_code = JANUS_STREAMING_ERROR_INVALID_ELEMENT;
			g_snprintf(error_cause, 512, "Unknown stream type '%s'...\n", type_text);
			goto plugin_response;
		}
		/* Any secret? */
		if(secret)
			mp->secret = g_strdup(json_string_value(secret));
		/* Any PIN? */
		if(pin)
			mp->pin = g_strdup(json_string_value(pin));
		if(save) {
			/* This mountpoint is permanent: save to the configuration file too
			 * FIXME: We should check if anything fails... */
			JANUS_LOG(LOG_VERB, "Saving mountpoint %"SCNu64" permanently in config file\n", mp->id);
			janus_mutex_lock(&config_mutex);
			char value[BUFSIZ];
			/* The category to add is the mountpoint name */
			janus_config_add_category(config, mp->name);
			/* Now for the common values */
			janus_config_add_item(config, mp->name, "type", type_text);
			g_snprintf(value, BUFSIZ, "%"SCNu64, mp->id);
			janus_config_add_item(config, mp->name, "id", value);
			janus_config_add_item(config, mp->name, "description", mp->description);
			if(mp->is_private)
				janus_config_add_item(config, mp->name, "is_private", "yes");
			/* Per type values */
			if(!strcasecmp(type_text, "rtp")) {
				janus_config_add_item(config, mp->name, "audio", mp->codecs.audio_pt >= 0 ? "yes" : "no");
				janus_streaming_rtp_source *source = mp->source;
				if(mp->codecs.audio_pt >= 0) {
					g_snprintf(value, BUFSIZ, "%d", source->audio_port);
					janus_config_add_item(config, mp->name, "audioport", value);
					json_t *audiomcast = json_object_get(root, "audiomcast");
					if(audiomcast)
						janus_config_add_item(config, mp->name, "audiomcast", json_string_value(audiomcast));
					g_snprintf(value, BUFSIZ, "%d", mp->codecs.audio_pt);
					janus_config_add_item(config, mp->name, "audiopt", value);
					janus_config_add_item(config, mp->name, "audiortpmap", mp->codecs.audio_rtpmap);
					if(mp->codecs.audio_fmtp)
						janus_config_add_item(config, mp->name, "audiofmtp", mp->codecs.audio_fmtp);
					json_t *aiface = json_object_get(root, "audioiface");
					if(aiface)
						janus_config_add_item(config, mp->name, "audioiface", json_string_value(aiface));
				}
				janus_config_add_item(config, mp->name, "video", mp->codecs.video_pt >= 0? "yes" : "no");
				if(mp->codecs.video_pt >= 0) {
					g_snprintf(value, BUFSIZ, "%d", source->video_port[0]);
					janus_config_add_item(config, mp->name, "videoport", value);
					json_t *videomcast = json_object_get(root, "videomcast");
					if(videomcast)
						janus_config_add_item(config, mp->name, "videomcast", json_string_value(videomcast));
					g_snprintf(value, BUFSIZ, "%d", mp->codecs.video_pt);
					janus_config_add_item(config, mp->name, "videopt", value);
					janus_config_add_item(config, mp->name, "videortpmap", mp->codecs.video_rtpmap);
					if(mp->codecs.video_fmtp)
						janus_config_add_item(config, mp->name, "videofmtp", mp->codecs.video_fmtp);
					if(source->keyframe.enabled)
						janus_config_add_item(config, mp->name, "videobufferkf", "yes");
					if(source->simulcast) {
						janus_config_add_item(config, mp->name, "videosimulcast", "yes");
						if(source->video_port[1]) {
							g_snprintf(value, BUFSIZ, "%d", source->video_port[1]);
							janus_config_add_item(config, mp->name, "videoport2", value);
						}
						if(source->video_port[2]) {
							g_snprintf(value, BUFSIZ, "%d", source->video_port[2]);
							janus_config_add_item(config, mp->name, "videoport3", value);
						}
					}
					json_t *viface = json_object_get(root, "videoiface");
					if(viface)
						janus_config_add_item(config, mp->name, "videoiface", json_string_value(viface));
				}
				janus_config_add_item(config, mp->name, "data", mp->data ? "yes" : "no");
				if(source->data_port > -1) {
					g_snprintf(value, BUFSIZ, "%d", source->data_port);
					janus_config_add_item(config, mp->name, "dataport", value);
					if(source->buffermsg)
						janus_config_add_item(config, mp->name, "databuffermsg", "yes");
					json_t *diface = json_object_get(root, "dataiface");
					if(diface)
						janus_config_add_item(config, mp->name, "dataiface", json_string_value(diface));
				}
			} else if(!strcasecmp(type_text, "live") || !strcasecmp(type_text, "ondemand")) {
				janus_streaming_file_source *source = mp->source;
				janus_config_add_item(config, mp->name, "filename", source->filename);
				janus_config_add_item(config, mp->name, "audio", mp->codecs.audio_pt ? "yes" : "no");
				janus_config_add_item(config, mp->name, "video", mp->codecs.video_pt ? "yes" : "no");
			} else if(!strcasecmp(type_text, "rtsp")) {
#ifdef HAVE_LIBCURL
				janus_streaming_rtp_source *source = mp->source;
				if(source->rtsp_url)
					janus_config_add_item(config, mp->name, "url", source->rtsp_url);
				if(source->rtsp_username)
					janus_config_add_item(config, mp->name, "rtsp_user", source->rtsp_username);
				if(source->rtsp_password)
					janus_config_add_item(config, mp->name, "rtsp_pwd", source->rtsp_password);
#endif
				if(mp->codecs.audio_pt >= 0) {
					janus_config_add_item(config, mp->name, "audio", mp->codecs.audio_pt ? "yes" : "no");
					if(mp->codecs.audio_rtpmap)
						janus_config_add_item(config, mp->name, "audiortpmap", mp->codecs.audio_rtpmap);
					if(mp->codecs.audio_fmtp)
						janus_config_add_item(config, mp->name, "audiofmtp", mp->codecs.audio_fmtp);
				}
				if(mp->codecs.video_pt >= 0) {
					janus_config_add_item(config, mp->name, "video", mp->codecs.video_pt ? "yes" : "no");
					if(mp->codecs.video_rtpmap)
						janus_config_add_item(config, mp->name, "videortpmap", mp->codecs.video_rtpmap);
					if(mp->codecs.video_fmtp)
						janus_config_add_item(config, mp->name, "videofmtp", mp->codecs.video_fmtp);
				}
				json_t *iface = json_object_get(root, "rtspiface");
				if(iface)
					janus_config_add_item(config, mp->name, "rtspiface", json_string_value(iface));
			}
			/* Some more common values */
			if(mp->secret)
				janus_config_add_item(config, mp->name, "secret", mp->secret);
			if(mp->pin)
				janus_config_add_item(config, mp->name, "pin", mp->pin);
			/* Save modified configuration */
			if(janus_config_save(config, config_folder, JANUS_STREAMING_PACKAGE) < 0)
				save = FALSE;	/* This will notify the user the mountpoint is not permanent */
			janus_mutex_unlock(&config_mutex);
		}
		/* Send info back */
		response = json_object();
		json_object_set_new(response, "streaming", json_string("created"));
		json_object_set_new(response, "created", json_string(mp->name));
		json_object_set_new(response, "permanent", save ? json_true() : json_false());
		json_t *ml = json_object();
		json_object_set_new(ml, "id", json_integer(mp->id));
		json_object_set_new(ml, "description", json_string(mp->description));
		json_object_set_new(ml, "type", json_string(mp->streaming_type == janus_streaming_type_live ? "live" : "on demand"));
		json_object_set_new(ml, "is_private", mp->is_private ? json_true() : json_false());
		if(!strcasecmp(type_text, "rtp")) {
			janus_streaming_rtp_source *source = mp->source;
			if (source->audio_fd != -1) {
				json_object_set_new(ml, "audio_port", json_integer(janus_streaming_get_fd_port(source->audio_fd)));
			}
			if (source->video_fd[0] != -1) {
				json_object_set_new(ml, "video_port", json_integer(janus_streaming_get_fd_port(source->video_fd[0])));
			}
			if (source->video_fd[1] != -1) {
				json_object_set_new(ml, "video_port_2", json_integer(janus_streaming_get_fd_port(source->video_fd[1])));
			}
			if (source->video_fd[2] != -1) {
				json_object_set_new(ml, "video_port_3", json_integer(janus_streaming_get_fd_port(source->video_fd[2])));
			}
			if (source->data_fd != -1) {
				json_object_set_new(ml, "data_port", json_integer(janus_streaming_get_fd_port(source->data_fd)));
			}
		}
		json_object_set_new(response, "stream", ml);
		/* Also notify event handlers */
		if(notify_events && gateway->events_is_enabled()) {
			json_t *info = json_object();
			json_object_set_new(info, "event", json_string("created"));
			json_object_set_new(info, "id", json_integer(mp->id));
			json_object_set_new(info, "type", json_string(mp->streaming_type == janus_streaming_type_live ? "live" : "on demand"));
			gateway->notify_event(&janus_streaming_plugin, session->handle, info);
		}
		goto plugin_response;
	} else if(!strcasecmp(request_text, "destroy")) {
		/* Get rid of an existing stream (notice this doesn't remove it from the config file, though) */
		JANUS_VALIDATE_JSON_OBJECT(root, destroy_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto plugin_response;
		json_t *id = json_object_get(root, "id");
		json_t *permanent = json_object_get(root, "permanent");
		gboolean save = permanent ? json_is_true(permanent) : FALSE;
		if(save && config == NULL) {
			JANUS_LOG(LOG_ERR, "No configuration file, can't destroy mountpoint permanently\n");
			error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
			g_snprintf(error_cause, 512, "No configuration file, can't destroy mountpoint permanently");
			goto plugin_response;
		}
		guint64 id_value = json_integer_value(id);
		janus_mutex_lock(&mountpoints_mutex);
		janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
		if(mp == NULL) {
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
			error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
			g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
			goto plugin_response;
		}
		/* A secret may be required for this action */
		JANUS_CHECK_SECRET(mp->secret, root, "secret", error_code, error_cause,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT, JANUS_STREAMING_ERROR_UNAUTHORIZED);
		if(error_code != 0) {
			janus_mutex_unlock(&mountpoints_mutex);
			goto plugin_response;
		}
		JANUS_LOG(LOG_VERB, "Request to unmount mountpoint/stream %"SCNu64"\n", id_value);
		/* FIXME Should we kick the current viewers as well? */
		janus_mutex_lock(&mp->mutex);
		GList *viewer = g_list_first(mp->listeners);
		/* Prepare JSON event */
		json_t *event = json_object();
		json_object_set_new(event, "streaming", json_string("event"));
		json_t *result = json_object();
		json_object_set_new(result, "status", json_string("stopped"));
		json_object_set_new(event, "result", result);
		while(viewer) {
			janus_streaming_session *session = (janus_streaming_session *)viewer->data;
			if(session != NULL) {
				session->stopping = TRUE;
				session->started = FALSE;
				session->paused = FALSE;
				session->mountpoint = NULL;
				/* Tell the core to tear down the PeerConnection, hangup_media will do the rest */
				gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
				gateway->close_pc(session->handle);
			}
			mp->listeners = g_list_remove_all(mp->listeners, session);
			viewer = g_list_first(mp->listeners);
		}
		json_decref(event);
		janus_mutex_unlock(&mp->mutex);
		if(save) {
			/* This change is permanent: save to the configuration file too
			 * FIXME: We should check if anything fails... */
			JANUS_LOG(LOG_VERB, "Destroying mountpoint %"SCNu64" (%s) permanently in config file\n", mp->id, mp->name);
			janus_mutex_lock(&config_mutex);
			/* The category to remove is the mountpoint name */
			janus_config_remove_category(config, mp->name);
			/* Save modified configuration */
			janus_config_save(config, config_folder, JANUS_STREAMING_PACKAGE);
			janus_mutex_unlock(&config_mutex);
		}
		/* Remove mountpoint from the hashtable: this will get it destroyed */
		if(!mp->destroyed) {
			mp->destroyed = janus_get_monotonic_time();
			g_hash_table_remove(mountpoints, &id_value);
			/* Cleaning up and removing the mountpoint is done in a lazy way */
			old_mountpoints = g_list_append(old_mountpoints, mp);
		}
		/* Also notify event handlers */
		if(notify_events && gateway->events_is_enabled()) {
			json_t *info = json_object();
			json_object_set_new(info, "event", json_string("destroyed"));
			json_object_set_new(info, "id", json_integer(id_value));
			gateway->notify_event(&janus_streaming_plugin, session->handle, info);
		}
		janus_mutex_unlock(&mountpoints_mutex);
		/* Send info back */
		response = json_object();
		json_object_set_new(response, "streaming", json_string("destroyed"));
		json_object_set_new(response, "destroyed", json_integer(id_value));
		goto plugin_response;
	} else if(!strcasecmp(request_text, "recording")) {
		/* We can start/stop recording a live, RTP-based stream */
		JANUS_VALIDATE_JSON_OBJECT(root, recording_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto plugin_response;
		json_t *action = json_object_get(root, "action");
		const char *action_text = json_string_value(action);
		if(strcasecmp(action_text, "start") && strcasecmp(action_text, "stop")) {
			JANUS_LOG(LOG_ERR, "Invalid action (should be start|stop)\n");
			error_code = JANUS_STREAMING_ERROR_INVALID_ELEMENT;
			g_snprintf(error_cause, 512, "Invalid action (should be start|stop)");
			goto plugin_response;
		}
		json_t *id = json_object_get(root, "id");
		guint64 id_value = json_integer_value(id);
		janus_mutex_lock(&mountpoints_mutex);
		janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
		if(mp == NULL) {
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
			error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
			g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
			goto plugin_response;
		}
		if(mp->streaming_type != janus_streaming_type_live || mp->streaming_source != janus_streaming_source_rtp) {
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_ERR, "Recording is only available on RTP-based live streams\n");
			error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
			g_snprintf(error_cause, 512, "Recording is only available on RTP-based live streams");
			goto plugin_response;
		}
		/* A secret may be required for this action */
		JANUS_CHECK_SECRET(mp->secret, root, "secret", error_code, error_cause,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT, JANUS_STREAMING_ERROR_UNAUTHORIZED);
		if(error_code != 0) {
			janus_mutex_unlock(&mountpoints_mutex);
			goto plugin_response;
		}
		janus_streaming_rtp_source *source = mp->source;
		if(!strcasecmp(action_text, "start")) {
			/* Start a recording for audio and/or video */
			JANUS_VALIDATE_JSON_OBJECT(root, recording_start_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0) {
				janus_mutex_unlock(&mountpoints_mutex);
				goto plugin_response;
			}
			json_t *audio = json_object_get(root, "audio");
			json_t *video = json_object_get(root, "video");
			json_t *data = json_object_get(root, "data");
			janus_recorder *arc = NULL, *vrc = NULL, *drc = NULL;
			if((audio && source->arc) || (video && source->vrc) || (data && source->drc)) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_ERR, "Recording for audio, video and/or data already started for this stream\n");
				error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
				g_snprintf(error_cause, 512, "Recording for audio, video and/or data already started for this stream");
				goto plugin_response;
			}
			if(!audio && !video && !data) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_ERR, "Missing audio, video and/or data\n");
				error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
				g_snprintf(error_cause, 512, "Missing audio, video and/or data");
				goto plugin_response;
			}
			if(audio) {
				const char *codec = NULL;
				if(strstr(mp->codecs.audio_rtpmap, "opus") || strstr(mp->codecs.audio_rtpmap, "OPUS"))
					codec = "opus";
				else if(strstr(mp->codecs.audio_rtpmap, "pcm") || strstr(mp->codecs.audio_rtpmap, "PCM"))
					codec = "g711";
				else if(strstr(mp->codecs.audio_rtpmap, "g722") || strstr(mp->codecs.audio_rtpmap, "G722"))
					codec = "g722";
				const char *audiofile = json_string_value(audio);
				arc = janus_recorder_create(NULL, codec, (char *)audiofile);
				if(arc == NULL) {
					JANUS_LOG(LOG_ERR, "[%s] Error starting recorder for audio\n", mp->name);
					janus_mutex_unlock(&mountpoints_mutex);
					error_code = JANUS_STREAMING_ERROR_CANT_RECORD;
					g_snprintf(error_cause, 512, "Error starting recorder for audio");
					goto plugin_response;
				}
				JANUS_LOG(LOG_INFO, "[%s] Audio recording started\n", mp->name);
			}
			if(video) {
				const char *codec = NULL;
				if(strstr(mp->codecs.video_rtpmap, "vp8") || strstr(mp->codecs.video_rtpmap, "VP8"))
					codec = "vp8";
				else if(strstr(mp->codecs.video_rtpmap, "vp9") || strstr(mp->codecs.video_rtpmap, "VP9"))
					codec = "vp9";
				else if(strstr(mp->codecs.video_rtpmap, "h264") || strstr(mp->codecs.video_rtpmap, "H264"))
					codec = "h264";
				const char *videofile = json_string_value(video);
				vrc = janus_recorder_create(NULL, codec, (char *)videofile);
				if(vrc == NULL) {
					if(arc != NULL) {
						janus_recorder_close(arc);
						janus_recorder_free(arc);
						arc = NULL;
					}
					JANUS_LOG(LOG_ERR, "[%s] Error starting recorder for video\n", mp->name);
					janus_mutex_unlock(&mountpoints_mutex);
					error_code = JANUS_STREAMING_ERROR_CANT_RECORD;
					g_snprintf(error_cause, 512, "Error starting recorder for video");
					goto plugin_response;
				}
				JANUS_LOG(LOG_INFO, "[%s] Video recording started\n", mp->name);
			}
			if(data) {
				const char *datafile = json_string_value(data);
				drc = janus_recorder_create(NULL, "text", (char *)datafile);
				if(drc == NULL) {
					if(arc != NULL) {
						janus_recorder_close(arc);
						janus_recorder_free(arc);
						arc = NULL;
					}
					if(vrc != NULL) {
						janus_recorder_close(vrc);
						janus_recorder_free(vrc);
						vrc = NULL;
					}
					JANUS_LOG(LOG_ERR, "[%s] Error starting recorder for data\n", mp->name);
					janus_mutex_unlock(&mountpoints_mutex);
					error_code = JANUS_STREAMING_ERROR_CANT_RECORD;
					g_snprintf(error_cause, 512, "Error starting recorder for data");
					goto plugin_response;
				}
				JANUS_LOG(LOG_INFO, "[%s] Data recording started\n", mp->name);
			}
			if(arc != NULL)
				source->arc = arc;
			if(vrc != NULL)
				source->vrc = vrc;
			if(drc != NULL)
				source->drc = drc;
			janus_mutex_unlock(&mountpoints_mutex);
			/* Send a success response back */
			response = json_object();
			json_object_set_new(response, "streaming", json_string("ok"));
			goto plugin_response;
		} else if(!strcasecmp(action_text, "stop")) {
			/* Stop the recording */
			JANUS_VALIDATE_JSON_OBJECT(root, recording_stop_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0) {
				janus_mutex_unlock(&mountpoints_mutex);
				goto plugin_response;
			}
			json_t *audio = json_object_get(root, "audio");
			json_t *video = json_object_get(root, "video");
			json_t *data = json_object_get(root, "data");
			if(!audio && !video) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_ERR, "Missing audio and/or video\n");
				error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
				g_snprintf(error_cause, 512, "Missing audio and/or video");
				goto plugin_response;
			}
			janus_mutex_lock(&source->rec_mutex);
			if(audio && json_is_true(audio) && source->arc) {
				/* Close the audio recording */
				janus_recorder_close(source->arc);
				JANUS_LOG(LOG_INFO, "[%s] Closed audio recording %s\n", mp->name, source->arc->filename ? source->arc->filename : "??");
				janus_recorder *tmp = source->arc;
				source->arc = NULL;
				janus_recorder_free(tmp);
			}
			if(video && json_is_true(video) && source->vrc) {
				/* Close the video recording */
				janus_recorder_close(source->vrc);
				JANUS_LOG(LOG_INFO, "[%s] Closed video recording %s\n", mp->name, source->vrc->filename ? source->vrc->filename : "??");
				janus_recorder *tmp = source->vrc;
				source->vrc = NULL;
				janus_recorder_free(tmp);
			}
			if(data && json_is_true(data) && source->drc) {
				/* Close the data recording */
				janus_recorder_close(source->drc);
				JANUS_LOG(LOG_INFO, "[%s] Closed data recording %s\n", mp->name, source->drc->filename ? source->drc->filename : "??");
				janus_recorder *tmp = source->drc;
				source->drc = NULL;
				janus_recorder_free(tmp);
			}
			janus_mutex_unlock(&source->rec_mutex);
			janus_mutex_unlock(&mountpoints_mutex);
			/* Send a success response back */
			response = json_object();
			json_object_set_new(response, "streaming", json_string("ok"));
			goto plugin_response;
		}
	} else if(!strcasecmp(request_text, "enable") || !strcasecmp(request_text, "disable")) {
		/* A request to enable/disable a mountpoint */
		JANUS_VALIDATE_JSON_OBJECT(root, id_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto plugin_response;
		json_t *id = json_object_get(root, "id");
		guint64 id_value = json_integer_value(id);
		janus_mutex_lock(&mountpoints_mutex);
		janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
		if(mp == NULL) {
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
			error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
			g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
			goto plugin_response;
		}
		/* A secret may be required for this action */
		JANUS_CHECK_SECRET(mp->secret, root, "secret", error_code, error_cause,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT, JANUS_STREAMING_ERROR_UNAUTHORIZED);
		if(error_code != 0) {
			janus_mutex_unlock(&mountpoints_mutex);
			goto plugin_response;
		}
		if(!strcasecmp(request_text, "enable")) {
			/* Enable a previously disabled mountpoint */
			JANUS_LOG(LOG_INFO, "[%s] Stream enabled\n", mp->name);
			mp->enabled = TRUE;
			/* FIXME: Should we notify the listeners, or is this up to the controller application? */
		} else {
			/* Disable a previously enabled mountpoint */
			JANUS_LOG(LOG_INFO, "[%s] Stream disabled\n", mp->name);
			mp->enabled = FALSE;
			/* Any recording to close? */
			if(mp->streaming_source == janus_streaming_source_rtp) {
				janus_streaming_rtp_source *source = mp->source;
				janus_mutex_lock(&source->rec_mutex);
				if(source->arc) {
					janus_recorder_close(source->arc);
					JANUS_LOG(LOG_INFO, "[%s] Closed audio recording %s\n", mp->name, source->arc->filename ? source->arc->filename : "??");
					janus_recorder *tmp = source->arc;
					source->arc = NULL;
					janus_recorder_free(tmp);
				}
				if(source->vrc) {
					janus_recorder_close(source->vrc);
					JANUS_LOG(LOG_INFO, "[%s] Closed video recording %s\n", mp->name, source->vrc->filename ? source->vrc->filename : "??");
					janus_recorder *tmp = source->vrc;
					source->vrc = NULL;
					janus_recorder_free(tmp);
				}
				if(source->drc) {
					janus_recorder_close(source->drc);
					JANUS_LOG(LOG_INFO, "[%s] Closed data recording %s\n", mp->name, source->drc->filename ? source->drc->filename : "??");
					janus_recorder *tmp = source->drc;
					source->drc = NULL;
					janus_recorder_free(tmp);
				}
				janus_mutex_unlock(&source->rec_mutex);
			}
			/* FIXME: Should we notify the listeners, or is this up to the controller application? */
		}
		janus_mutex_unlock(&mountpoints_mutex);
		/* Send a success response back */
		response = json_object();
		json_object_set_new(response, "streaming", json_string("ok"));
		goto plugin_response;
	} else if(!strcasecmp(request_text, "watch") || !strcasecmp(request_text, "start")
			|| !strcasecmp(request_text, "pause") || !strcasecmp(request_text, "stop")
			|| !strcasecmp(request_text, "configure") || !strcasecmp(request_text, "switch")) {
		/* These messages are handled asynchronously */
		janus_mutex_unlock(&sessions_mutex);
		janus_streaming_message *msg = g_malloc0(sizeof(janus_streaming_message));
		msg->handle = handle;
		msg->transaction = transaction;
		msg->message = root;
		msg->jsep = jsep;

		g_async_queue_push(messages, msg);
		return janus_plugin_result_new(JANUS_PLUGIN_OK_WAIT, NULL, NULL);
	} else {
		JANUS_LOG(LOG_VERB, "Unknown request '%s'\n", request_text);
		error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
		g_snprintf(error_cause, 512, "Unknown request '%s'", request_text);
	}

plugin_response:
		{
			janus_mutex_unlock(&sessions_mutex);
			if(ifas) {
				freeifaddrs(ifas);
			}

			if(error_code == 0 && !response) {
				error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
				g_snprintf(error_cause, 512, "Invalid response");
			}
			if(error_code != 0) {
				/* Prepare JSON error event */
				json_t *event = json_object();
				json_object_set_new(event, "streaming", json_string("event"));
				json_object_set_new(event, "error_code", json_integer(error_code));
				json_object_set_new(event, "error", json_string(error_cause));
				response = event;
			}
			if(root != NULL)
				json_decref(root);
			if(jsep != NULL)
				json_decref(jsep);
			g_free(transaction);

			return janus_plugin_result_new(JANUS_PLUGIN_OK, NULL, response);
		}

}

void janus_streaming_setup_media(janus_plugin_session *handle) {
	JANUS_LOG(LOG_INFO, "WebRTC media is now available\n");
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized))
		return;
	janus_mutex_lock(&sessions_mutex);
	janus_streaming_session *session = janus_streaming_lookup_session(handle);
	if(!session) {
		janus_mutex_unlock(&sessions_mutex);
		JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
		return;
	}
	if(session->destroyed) {
		janus_mutex_unlock(&sessions_mutex);
		return;
	}
	g_atomic_int_set(&session->hangingup, 0);
	/* We only start streaming towards this user when we get this event */
	janus_rtp_switching_context_reset(&session->context);
	/* If this is related to a live RTP mountpoint, any keyframe we can shoot already? */
	janus_streaming_mountpoint *mountpoint = session->mountpoint;
	if(mountpoint->streaming_source == janus_streaming_source_rtp) {
		janus_streaming_rtp_source *source = mountpoint->source;
		if(source->keyframe.enabled) {
			JANUS_LOG(LOG_HUGE, "Any keyframe to send?\n");
			janus_mutex_lock(&source->keyframe.mutex);
			if(source->keyframe.latest_keyframe != NULL) {
				JANUS_LOG(LOG_HUGE, "Yep! %d packets\n", g_list_length(source->keyframe.latest_keyframe));
				GList *temp = source->keyframe.latest_keyframe;
				while(temp) {
					janus_streaming_relay_rtp_packet(session, temp->data);
					temp = temp->next;
				}
			}
			janus_mutex_unlock(&source->keyframe.mutex);
		}
		if(source->buffermsg) {
			JANUS_LOG(LOG_HUGE, "Any recent datachannel message to send?\n");
			janus_mutex_lock(&source->buffermsg_mutex);
			if(source->last_msg != NULL) {
				JANUS_LOG(LOG_HUGE, "Yep!\n");
				janus_streaming_relay_rtp_packet(session, source->last_msg);
			}
			janus_mutex_unlock(&source->buffermsg_mutex);
		}
	}
	session->started = TRUE;
	/* Prepare JSON event */
	json_t *event = json_object();
	json_object_set_new(event, "streaming", json_string("event"));
	json_t *result = json_object();
	json_object_set_new(result, "status", json_string("started"));
	json_object_set_new(event, "result", result);
	int ret = gateway->push_event(handle, &janus_streaming_plugin, NULL, event, NULL);
	JANUS_LOG(LOG_VERB, "  >> Pushing event: %d (%s)\n", ret, janus_get_api_error(ret));
	json_decref(event);
	janus_mutex_unlock(&sessions_mutex);
}

void janus_streaming_incoming_rtp(janus_plugin_session *handle, int video, char *buf, int len) {
	if(handle == NULL || handle->stopped || g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized))
		return;
	/* FIXME We don't care about what the browser sends us, we're sendonly */
}

void janus_streaming_incoming_rtcp(janus_plugin_session *handle, int video, char *buf, int len) {
	if(handle == NULL || handle->stopped || g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized))
		return;
	/* We might interested in the available bandwidth that the user advertizes */
	uint64_t bw = janus_rtcp_get_remb(buf, len);
	if(bw > 0) {
		JANUS_LOG(LOG_HUGE, "REMB for this PeerConnection: %"SCNu64"\n", bw);
		/* TODO Use this somehow (e.g., notification towards application?) */
	}
	/* FIXME Maybe we should care about RTCP, but not now */
}

void janus_streaming_hangup_media(janus_plugin_session *handle) {
	janus_mutex_lock(&sessions_mutex);
	janus_streaming_hangup_media_internal(handle);
	janus_mutex_unlock(&sessions_mutex);
}

static void janus_streaming_hangup_media_internal(janus_plugin_session *handle) {
	JANUS_LOG(LOG_INFO, "No WebRTC media anymore\n");
	if(g_atomic_int_get(&stopping) || !g_atomic_int_get(&initialized))
		return;
	janus_streaming_session *session = janus_streaming_lookup_session(handle);
	if(!session) {
		JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
		return;
	}
	if(session->destroyed) {
		return;
	}
	if(g_atomic_int_add(&session->hangingup, 1)) {
		return;
	}
	session->substream = -1;
	session->substream_target = 0;
	session->templayer = -1;
	session->templayer_target = 0;
	session->last_relayed = 0;
	janus_vp8_simulcast_context_reset(&session->simulcast_context);
	session->stopping = TRUE;
	session->started = FALSE;
	session->paused = FALSE;
	session->substream = -1;
	session->substream_target = 0;
	session->templayer = -1;
	session->templayer_target = 0;
	session->last_relayed = 0;
	janus_vp8_simulcast_context_reset(&session->simulcast_context);
	janus_streaming_mountpoint *mp = session->mountpoint;
	if(mp) {
		janus_mutex_lock(&mp->mutex);
		JANUS_LOG(LOG_VERB, "  -- Removing the session from the mountpoint listeners\n");
		if(g_list_find(mp->listeners, session) != NULL) {
			JANUS_LOG(LOG_VERB, "  -- -- Found!\n");
		}
		mp->listeners = g_list_remove_all(mp->listeners, session);
		janus_mutex_unlock(&mp->mutex);
	}
	session->mountpoint = NULL;
}

/* Thread to handle incoming messages */
static void *janus_streaming_handler(void *data) {
	JANUS_LOG(LOG_VERB, "Joining Streaming handler thread\n");
	janus_streaming_message *msg = NULL;
	int error_code = 0;
	char error_cause[512];
	json_t *root = NULL;
	while(g_atomic_int_get(&initialized) && !g_atomic_int_get(&stopping)) {
		msg = g_async_queue_pop(messages);
		if(msg == NULL)
			continue;
		if(msg == &exit_message)
			break;
		if(msg->handle == NULL) {
			janus_streaming_message_free(msg);
			continue;
		}
		janus_mutex_lock(&sessions_mutex);
		janus_streaming_session *session = janus_streaming_lookup_session(msg->handle);
		if(!session) {
			janus_mutex_unlock(&sessions_mutex);
			JANUS_LOG(LOG_ERR, "No session associated with this handle...\n");
			janus_streaming_message_free(msg);
			continue;
		}
		if(session->destroyed) {
			janus_mutex_unlock(&sessions_mutex);
			janus_streaming_message_free(msg);
			continue;
		}
		janus_mutex_unlock(&sessions_mutex);
		/* Handle request */
		error_code = 0;
		root = NULL;
		if(msg->message == NULL) {
			JANUS_LOG(LOG_ERR, "No message??\n");
			error_code = JANUS_STREAMING_ERROR_NO_MESSAGE;
			g_snprintf(error_cause, 512, "%s", "No message??");
			goto error;
		}
		root = msg->message;
		/* Get the request first */
		JANUS_VALIDATE_JSON_OBJECT(root, request_parameters,
			error_code, error_cause, TRUE,
			JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
		if(error_code != 0)
			goto error;
		json_t *request = json_object_get(root, "request");
		const char *request_text = json_string_value(request);
		json_t *result = NULL;
		const char *sdp_type = NULL;
		char *sdp = NULL;
		/* All these requests can only be handled asynchronously */
		if(!strcasecmp(request_text, "watch")) {
			JANUS_VALIDATE_JSON_OBJECT(root, watch_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto error;
			json_t *id = json_object_get(root, "id");
			json_t *offer_audio = json_object_get(root, "offer_audio");
			json_t *offer_video = json_object_get(root, "offer_video");
			json_t *offer_data = json_object_get(root, "offer_data");
			guint64 id_value = json_integer_value(id);
			janus_mutex_lock(&mountpoints_mutex);
			janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
			if(mp == NULL) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
				goto error;
			}
			/* A secret may be required for this action */
			JANUS_CHECK_SECRET(mp->pin, root, "pin", error_code, error_cause,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT, JANUS_STREAMING_ERROR_UNAUTHORIZED);
			if(error_code != 0) {
				janus_mutex_unlock(&mountpoints_mutex);
				goto error;
			}
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "Request to watch mountpoint/stream %"SCNu64"\n", id_value);
			session->stopping = FALSE;
			session->mountpoint = mp;
			/* Check what we should offer */
			session->audio = offer_audio ? json_is_true(offer_audio) : TRUE;	/* True by default */
			if(!mp->audio)
				session->audio = FALSE;	/* ... unless the mountpoint isn't sending any audio */
			session->video = offer_video ? json_is_true(offer_video) : TRUE;	/* True by default */
			if(!mp->video)
				session->video = FALSE;	/* ... unless the mountpoint isn't sending any video */
			session->data = offer_data ? json_is_true(offer_data) : TRUE;	/* True by default */
			if(!mp->data)
				session->data = FALSE;	/* ... unless the mountpoint isn't sending any data */
			if((!mp->audio || !session->audio) &&
					(!mp->video || !session->video) &&
					(!mp->data || !session->data)) {
				JANUS_LOG(LOG_ERR, "Can't offer an SDP with no audio, video or data for this mountpoint\n");
				error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
				g_snprintf(error_cause, 512, "Can't offer an SDP with no audio, video or data for this mountpoint");
				goto error;
			}
			if(mp->streaming_type == janus_streaming_type_on_demand) {
				GError *error = NULL;
				char tname[16];
				g_snprintf(tname, sizeof(tname), "mp %"SCNu64, id_value);
				g_thread_try_new(tname, &janus_streaming_ondemand_thread, session, &error);
				if(error != NULL) {
					JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the on-demand thread...\n", error->code, error->message ? error->message : "??");
					error_code = JANUS_STREAMING_ERROR_UNKNOWN_ERROR;
					g_snprintf(error_cause, 512, "Got error %d (%s) trying to launch the on-demand thread", error->code, error->message ? error->message : "??");
					goto error;
				}
			} else if(mp->streaming_source == janus_streaming_source_rtp) {
				janus_streaming_rtp_source *source = (janus_streaming_rtp_source *)mp->source;
				if(source && source->simulcast) {
					JANUS_VALIDATE_JSON_OBJECT(root, simulcast_parameters,
						error_code, error_cause, TRUE,
						JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
					if(error_code != 0)
						goto error;
					/* This mountpoint is simulcasting, let's aim high by default */
					session->substream = -1;
					session->substream_target = 2;
					session->templayer = -1;
					session->templayer_target = 2;
					janus_vp8_simulcast_context_reset(&session->simulcast_context);
					/* Unless the request contains a target */
					json_t *substream = json_object_get(root, "substream");
					if(substream) {
						session->substream_target = json_integer_value(substream);
						JANUS_LOG(LOG_VERB, "Setting video substream to let through (simulcast): %d (was %d)\n",
							session->substream_target, session->substream);
					}
					json_t *temporal = json_object_get(root, "temporal");
					if(temporal) {
						session->templayer_target = json_integer_value(temporal);
						JANUS_LOG(LOG_VERB, "Setting video temporal layer to let through (simulcast): %d (was %d)\n",
							session->templayer_target, session->templayer);
					}
				}
			}
			/* Let's prepare an offer now, but let's also check if there0s something we need to skip */
			sdp_type = "offer";	/* We're always going to do the offer ourselves, never answer */
			char sdptemp[2048];
			memset(sdptemp, 0, 2048);
			gchar buffer[512];
			memset(buffer, 0, 512);
			gint64 sessid = janus_get_real_time();
			gint64 version = sessid;	/* FIXME This needs to be increased when it changes, so time should be ok */
			g_snprintf(buffer, 512,
				"v=0\r\no=%s %"SCNu64" %"SCNu64" IN IP4 127.0.0.1\r\n",
					"-", sessid, version);
			g_strlcat(sdptemp, buffer, 2048);
			g_snprintf(buffer, 512,
				"s=Mountpoint %"SCNu64"\r\n", mp->id);
			g_strlcat(sdptemp, buffer, 2048);
			g_strlcat(sdptemp, "t=0 0\r\n", 2048);
			if(mp->codecs.audio_pt >= 0 && session->audio) {
				/* Add audio line */
				g_snprintf(buffer, 512,
					"m=audio 1 RTP/SAVPF %d\r\n"
					"c=IN IP4 1.1.1.1\r\n",
					mp->codecs.audio_pt);
				g_strlcat(sdptemp, buffer, 2048);
				if(mp->codecs.audio_rtpmap) {
					g_snprintf(buffer, 512,
						"a=rtpmap:%d %s\r\n",
						mp->codecs.audio_pt, mp->codecs.audio_rtpmap);
					g_strlcat(sdptemp, buffer, 2048);
				}
				if(mp->codecs.audio_fmtp) {
					g_snprintf(buffer, 512,
						"a=fmtp:%d %s\r\n",
						mp->codecs.audio_pt, mp->codecs.audio_fmtp);
					g_strlcat(sdptemp, buffer, 2048);
				}
				g_strlcat(sdptemp, "a=sendonly\r\n", 2048);
			}
			if(mp->codecs.video_pt >= 0 && session->video) {
				/* Add video line */
				g_snprintf(buffer, 512,
					"m=video 1 RTP/SAVPF %d\r\n"
					"c=IN IP4 1.1.1.1\r\n",
					mp->codecs.video_pt);
				g_strlcat(sdptemp, buffer, 2048);
				if(mp->codecs.video_rtpmap) {
					g_snprintf(buffer, 512,
						"a=rtpmap:%d %s\r\n",
						mp->codecs.video_pt, mp->codecs.video_rtpmap);
					g_strlcat(sdptemp, buffer, 2048);
				}
				if(mp->codecs.video_fmtp) {
					g_snprintf(buffer, 512,
						"a=fmtp:%d %s\r\n",
						mp->codecs.video_pt, mp->codecs.video_fmtp);
					g_strlcat(sdptemp, buffer, 2048);
				}
				g_snprintf(buffer, 512,
					"a=rtcp-fb:%d nack\r\n",
					mp->codecs.video_pt);
				g_strlcat(sdptemp, buffer, 2048);
				g_snprintf(buffer, 512,
					"a=rtcp-fb:%d goog-remb\r\n",
					mp->codecs.video_pt);
				g_strlcat(sdptemp, buffer, 2048);
				g_strlcat(sdptemp, "a=sendonly\r\n", 2048);
			}
#ifdef HAVE_SCTP
			if(mp->data && session->data) {
				/* Add data line */
				g_snprintf(buffer, 512,
					"m=application 1 DTLS/SCTP 5000\r\n"
					"c=IN IP4 1.1.1.1\r\n"
					"a=sctpmap:5000 webrtc-datachannel 16\r\n");
				g_strlcat(sdptemp, buffer, 2048);
			}
#endif
			sdp = g_strdup(sdptemp);
			JANUS_LOG(LOG_VERB, "Going to offer this SDP:\n%s\n", sdp);
			result = json_object();
			json_object_set_new(result, "status", json_string("preparing"));
			/* Add the user to the list of watchers and we're done */
			janus_mutex_lock(&mp->mutex);
			mp->listeners = g_list_append(mp->listeners, session);
			janus_mutex_unlock(&mp->mutex);
		} else if(!strcasecmp(request_text, "start")) {
			if(session->mountpoint == NULL) {
				JANUS_LOG(LOG_VERB, "Can't start: no mountpoint set\n");
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "Can't start: no mountpoint set");
				goto error;
			}
			JANUS_LOG(LOG_VERB, "Starting the streaming\n");
			session->paused = FALSE;
			result = json_object();
			/* We wait for the setup_media event to start: on the other hand, it may have already arrived */
			json_object_set_new(result, "status", json_string(session->started ? "started" : "starting"));
			/* Also notify event handlers */
			if(notify_events && gateway->events_is_enabled()) {
				json_t *info = json_object();
				json_object_set_new(info, "status", json_string("starting"));
				if(session->mountpoint != NULL)
					json_object_set_new(info, "id", json_integer(session->mountpoint->id));
				gateway->notify_event(&janus_streaming_plugin, session->handle, info);
			}
		} else if(!strcasecmp(request_text, "pause")) {
			if(session->mountpoint == NULL) {
				JANUS_LOG(LOG_VERB, "Can't pause: no mountpoint set\n");
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "Can't start: no mountpoint set");
				goto error;
			}
			JANUS_LOG(LOG_VERB, "Pausing the streaming\n");
			session->paused = TRUE;
			result = json_object();
			json_object_set_new(result, "status", json_string("pausing"));
			/* Also notify event handlers */
			if(notify_events && gateway->events_is_enabled()) {
				json_t *info = json_object();
				json_object_set_new(info, "status", json_string("pausing"));
				if(session->mountpoint != NULL)
					json_object_set_new(info, "id", json_integer(session->mountpoint->id));
				gateway->notify_event(&janus_streaming_plugin, session->handle, info);
			}
		} else if(!strcasecmp(request_text, "configure")) {
			janus_streaming_mountpoint *mp = session->mountpoint;
			if(mp == NULL) {
				JANUS_LOG(LOG_VERB, "Can't configure: not on a mountpoint\n");
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "Can't configure: not on a mountpoint");
				goto error;
			}
			JANUS_VALIDATE_JSON_OBJECT(root, configure_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			json_t *audio = json_object_get(root, "audio");
			if(audio)
				session->audio = json_is_true(audio);
			json_t *video = json_object_get(root, "video");
			if(video)
				session->video = json_is_true(video);
			json_t *data = json_object_get(root, "data");
			if(data)
				session->data = json_is_true(data);
			if(mp->streaming_source == janus_streaming_source_rtp) {
				janus_streaming_rtp_source *source = (janus_streaming_rtp_source *)mp->source;
				if(source && source->simulcast) {
					/* This mountpoint is simulcasting, let's aim high by default */
					session->substream = -1;
					session->substream_target = 2;
					session->templayer = -1;
					session->templayer_target = 2;
					janus_vp8_simulcast_context_reset(&session->simulcast_context);
					/* Unless the request contains a target */
					json_t *substream = json_object_get(root, "substream");
					if(substream) {
						session->substream_target = json_integer_value(substream);
						JANUS_LOG(LOG_VERB, "Setting video substream to let through (simulcast): %d (was %d)\n",
							session->substream_target, session->substream);
						if(session->substream_target == session->substream) {
							/* No need to do anything, we're already getting the right substream, so notify the viewer */
							json_t *event = json_object();
							json_object_set_new(event, "streaming", json_string("event"));
							json_t *result = json_object();
							json_object_set_new(result, "substream", json_integer(session->substream));
							json_object_set_new(event, "result", result);
							gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
							json_decref(event);
						}
					}
					json_t *temporal = json_object_get(root, "temporal");
					if(temporal) {
						session->templayer_target = json_integer_value(temporal);
						JANUS_LOG(LOG_VERB, "Setting video temporal layer to let through (simulcast): %d (was %d)\n",
							session->templayer_target, session->templayer);
						if(session->templayer_target == session->templayer) {
							/* No need to do anything, we're already getting the right temporal layer, so notify the viewer */
							json_t *event = json_object();
							json_object_set_new(event, "streaming", json_string("event"));
							json_t *result = json_object();
							json_object_set_new(result, "temporal", json_integer(session->templayer));
							json_object_set_new(event, "result", result);
							gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
							json_decref(event);
						}
					}
				}
			}
			/* Done */
			result = json_object();
			json_object_set_new(result, "event", json_string("configured"));
		} else if(!strcasecmp(request_text, "switch")) {
			/* This listener wants to switch to a different mountpoint
			 * NOTE: this only works for live RTP streams as of now: you
			 * cannot, for instance, switch from a live RTP mountpoint to
			 * an on demand one or viceversa (TBD.) */
			janus_streaming_mountpoint *oldmp = session->mountpoint;
			if(oldmp == NULL) {
				JANUS_LOG(LOG_VERB, "Can't switch: not on a mountpoint\n");
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "Can't switch: not on a mountpoint");
				goto error;
			}
			if(oldmp->streaming_type != janus_streaming_type_live ||
					oldmp->streaming_source != janus_streaming_source_rtp) {
				JANUS_LOG(LOG_VERB, "Can't switch: not on a live RTP mountpoint\n");
				error_code = JANUS_STREAMING_ERROR_CANT_SWITCH;
				g_snprintf(error_cause, 512, "Can't switch: not on a live RTP mountpoint");
				goto error;
			}
			JANUS_VALIDATE_JSON_OBJECT(root, id_parameters,
				error_code, error_cause, TRUE,
				JANUS_STREAMING_ERROR_MISSING_ELEMENT, JANUS_STREAMING_ERROR_INVALID_ELEMENT);
			if(error_code != 0)
				goto error;
			json_t *id = json_object_get(root, "id");
			guint64 id_value = json_integer_value(id);
			janus_mutex_lock(&mountpoints_mutex);
			janus_streaming_mountpoint *mp = g_hash_table_lookup(mountpoints, &id_value);
			if(mp == NULL) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_VERB, "No such mountpoint/stream %"SCNu64"\n", id_value);
				error_code = JANUS_STREAMING_ERROR_NO_SUCH_MOUNTPOINT;
				g_snprintf(error_cause, 512, "No such mountpoint/stream %"SCNu64"", id_value);
				goto error;
			}
			if(mp->streaming_type != janus_streaming_type_live ||
					mp->streaming_source != janus_streaming_source_rtp) {
				janus_mutex_unlock(&mountpoints_mutex);
				JANUS_LOG(LOG_VERB, "Can't switch: target is not a live RTP mountpoint\n");
				error_code = JANUS_STREAMING_ERROR_CANT_SWITCH;
				g_snprintf(error_cause, 512, "Can't switch: target is not a live RTP mountpoint");
				goto error;
			}
			janus_mutex_unlock(&mountpoints_mutex);
			JANUS_LOG(LOG_VERB, "Request to switch to mountpoint/stream %"SCNu64" (old: %"SCNu64")\n", id_value, oldmp->id);
			session->paused = TRUE;
			/* Unsubscribe from the previous mountpoint and subscribe to the new one */
			janus_mutex_lock(&oldmp->mutex);
			oldmp->listeners = g_list_remove_all(oldmp->listeners, session);
			janus_mutex_unlock(&oldmp->mutex);
			/* Subscribe to the new one */
			janus_mutex_lock(&mp->mutex);
			mp->listeners = g_list_append(mp->listeners, session);
			janus_mutex_unlock(&mp->mutex);
			session->mountpoint = mp;
			session->paused = FALSE;
			/* Done */
			result = json_object();
			json_object_set_new(result, "switched", json_string("ok"));
			json_object_set_new(result, "id", json_integer(id_value));
			/* Also notify event handlers */
			if(notify_events && gateway->events_is_enabled()) {
				json_t *info = json_object();
				json_object_set_new(info, "status", json_string("switching"));
				json_object_set_new(info, "id", json_integer(id_value));
				gateway->notify_event(&janus_streaming_plugin, session->handle, info);
			}
		} else if(!strcasecmp(request_text, "stop")) {
			if(session->stopping || !session->started) {
				/* Been there, done that: ignore */
				janus_streaming_message_free(msg);
				continue;
			}
			JANUS_LOG(LOG_VERB, "Stopping the streaming\n");
			result = json_object();
			json_object_set_new(result, "status", json_string("stopping"));
			/* Also notify event handlers */
			if(notify_events && gateway->events_is_enabled()) {
				json_t *info = json_object();
				json_object_set_new(info, "status", json_string("stopping"));
				janus_streaming_mountpoint *mp = session->mountpoint;
				if(mp)
					json_object_set_new(info, "id", json_integer(mp->id));
				gateway->notify_event(&janus_streaming_plugin, session->handle, info);
			}
			/* Tell the core to tear down the PeerConnection, hangup_media will do the rest */
			gateway->close_pc(session->handle);
		} else {
			JANUS_LOG(LOG_VERB, "Unknown request '%s'\n", request_text);
			error_code = JANUS_STREAMING_ERROR_INVALID_REQUEST;
			g_snprintf(error_cause, 512, "Unknown request '%s'", request_text);
			goto error;
		}

		/* Any SDP to handle? */
		const char *msg_sdp_type = json_string_value(json_object_get(msg->jsep, "type"));
		const char *msg_sdp = json_string_value(json_object_get(msg->jsep, "sdp"));
		if(msg_sdp) {
			JANUS_LOG(LOG_VERB, "This is involving a negotiation (%s) as well (but we really don't care):\n%s\n", msg_sdp_type, msg_sdp);
		}

		/* Prepare JSON event */
		json_t *jsep = json_pack("{ssss}", "type", sdp_type, "sdp", sdp);
		json_t *event = json_object();
		json_object_set_new(event, "streaming", json_string("event"));
		if(result != NULL)
			json_object_set_new(event, "result", result);
		int ret = gateway->push_event(msg->handle, &janus_streaming_plugin, msg->transaction, event, jsep);
		JANUS_LOG(LOG_VERB, "  >> Pushing event: %d (%s)\n", ret, janus_get_api_error(ret));
		g_free(sdp);
		json_decref(event);
		json_decref(jsep);
		janus_streaming_message_free(msg);
		continue;

error:
		{
			/* Prepare JSON error event */
			json_t *event = json_object();
			json_object_set_new(event, "streaming", json_string("event"));
			json_object_set_new(event, "error_code", json_integer(error_code));
			json_object_set_new(event, "error", json_string(error_cause));
			int ret = gateway->push_event(msg->handle, &janus_streaming_plugin, msg->transaction, event, NULL);
			JANUS_LOG(LOG_VERB, "  >> Pushing event: %d (%s)\n", ret, janus_get_api_error(ret));
			json_decref(event);
			janus_streaming_message_free(msg);
		}
	}
	JANUS_LOG(LOG_VERB, "Leaving Streaming handler thread\n");
	return NULL;
}

/* Helpers to create a listener filedescriptor */
static int janus_streaming_create_fd(int port, in_addr_t mcast, const janus_network_address *iface, const char *listenername, const char *medianame, const char *mountpointname) {
	struct sockaddr_in address;
	janus_network_address_string_buffer address_representation;
	int fd = socket(AF_INET, SOCK_DGRAM, 0);
	if(fd < 0) {
		JANUS_LOG(LOG_ERR, "[%s] Cannot create socket for %s...\n", mountpointname, medianame);
		return -1;
	}
	if(port > 0) {
		if(IN_MULTICAST(ntohl(mcast))) {
#ifdef IP_MULTICAST_ALL
			int mc_all = 0;
			if((setsockopt(fd, IPPROTO_IP, IP_MULTICAST_ALL, (void*) &mc_all, sizeof(mc_all))) < 0) {
				JANUS_LOG(LOG_ERR, "[%s] %s listener setsockopt IP_MULTICAST_ALL failed\n", mountpointname, listenername);
				close(fd);
				return -1;
			}
#endif
			struct ip_mreq mreq;
			memset(&mreq, '\0', sizeof(mreq));
			mreq.imr_multiaddr.s_addr = mcast;
			if(!janus_network_address_is_null(iface)) {
				if(iface->family == AF_INET) {
					mreq.imr_interface = iface->ipv4;
					(void) janus_network_address_to_string_buffer(iface, &address_representation); /* This is OK: if we get here iface must be non-NULL */
					JANUS_LOG(LOG_INFO, "[%s] %s listener using interface address: %s\n", mountpointname, listenername, janus_network_address_string_from_buffer(&address_representation));
				} else {
					JANUS_LOG(LOG_ERR, "[%s] %s listener: invalid multicast address type (only IPv4 is currently supported by this plugin)\n", mountpointname, listenername);
					close(fd);
					return -1;
				}
			} else {
				JANUS_LOG(LOG_WARN, "[%s] No multicast interface for: %s. This may not work as expected if you have multiple network devices (NICs)\n", mountpointname, listenername);
			}
			if(setsockopt(fd, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreq, sizeof(mreq)) == -1) {
				JANUS_LOG(LOG_ERR, "[%s] %s listener IP_ADD_MEMBERSHIP failed\n", mountpointname, listenername);
				close(fd);
				return -1;
			}
			JANUS_LOG(LOG_INFO, "[%s] %s listener IP_ADD_MEMBERSHIP ok\n", mountpointname, listenername);
		}
	}

	address.sin_family = AF_INET;
	address.sin_port = htons(port);
	address.sin_addr.s_addr = INADDR_ANY;
	/* If this is multicast, allow a re-use of the same ports (different groups may be used) */
	if(port > 0 && IN_MULTICAST(ntohl(mcast))) {
		int reuse = 1;
		if(setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &reuse, sizeof(reuse)) == -1) {
			JANUS_LOG(LOG_ERR, "[%s] %s listener setsockopt SO_REUSEADDR failed\n", mountpointname, listenername);
			close(fd);
			return -1;
		}
		address.sin_addr.s_addr = mcast;
	} else {
		if(!IN_MULTICAST(ntohl(mcast)) && !janus_network_address_is_null(iface)) {
			if(iface->family == AF_INET) {
				address.sin_addr = iface->ipv4;
				(void) janus_network_address_to_string_buffer(iface, &address_representation); /* This is OK: if we get here iface must be non-NULL */
				JANUS_LOG(LOG_INFO, "[%s] %s listener restricted to interface address: %s\n", mountpointname, listenername, janus_network_address_string_from_buffer(&address_representation));
			} else {
				JANUS_LOG(LOG_ERR, "[%s] %s listener: invalid address/restriction type (only IPv4 is currently supported by this plugin)\n", mountpointname, listenername);
				close(fd);
				return -1;
			}
		}
	}
	/* Bind to the specified port */
	if(bind(fd, (struct sockaddr *)(&address), sizeof(struct sockaddr)) < 0) {
		JANUS_LOG(LOG_ERR, "[%s] Bind failed for %s (port %d)...\n", mountpointname, medianame, port);
		close(fd);
		return -1;
	}
	return fd;
}

/* Helper to return fd port */
static int janus_streaming_get_fd_port(int fd) {
	struct sockaddr_in server;
	socklen_t len = sizeof(fd);
	if (getsockname(fd, &server, &len) == -1) {
		return -1;
	}

	return ntohs(server.sin_port);
}

/* Helpers to destroy a streaming mountpoint. */
static void janus_streaming_rtp_source_free(janus_streaming_rtp_source *source) {
	if(source->audio_fd > -1) {
		close(source->audio_fd);
	}
	if(source->video_fd[0] > -1) {
		close(source->video_fd[0]);
	}
	if(source->video_fd[1] > -1) {
		close(source->video_fd[1]);
	}
	if(source->video_fd[2] > -1) {
		close(source->video_fd[2]);
	}
	if(source->data_fd > -1) {
		close(source->data_fd);
	}
#ifdef HAVE_LIBCURL
	if(source->audio_rtcp_fd > -1) {
		close(source->audio_rtcp_fd);
	}
	if(source->video_rtcp_fd > -1) {
		close(source->video_rtcp_fd);
	}
#endif
	janus_mutex_lock(&source->keyframe.mutex);
	GList *temp = NULL;
	while(source->keyframe.latest_keyframe) {
		temp = g_list_first(source->keyframe.latest_keyframe);
		source->keyframe.latest_keyframe = g_list_remove_link(source->keyframe.latest_keyframe, temp);
		janus_streaming_rtp_relay_packet *pkt = (janus_streaming_rtp_relay_packet *)temp->data;
		g_free(pkt->data);
		g_free(pkt);
		g_list_free(temp);
	}
	source->keyframe.latest_keyframe = NULL;
	while(source->keyframe.temp_keyframe) {
		temp = g_list_first(source->keyframe.temp_keyframe);
		source->keyframe.temp_keyframe = g_list_remove_link(source->keyframe.temp_keyframe, temp);
		janus_streaming_rtp_relay_packet *pkt = (janus_streaming_rtp_relay_packet *)temp->data;
		g_free(pkt->data);
		g_free(pkt);
		g_list_free(temp);
	}
	source->keyframe.latest_keyframe = NULL;
	janus_mutex_unlock(&source->keyframe.mutex);
	janus_mutex_lock(&source->buffermsg_mutex);
	if(source->last_msg) {
		janus_streaming_rtp_relay_packet *pkt = (janus_streaming_rtp_relay_packet *)source->last_msg;
		g_free(pkt->data);
		g_free(pkt);
		source->last_msg = NULL;
	}
	janus_mutex_unlock(&source->buffermsg_mutex);
#ifdef HAVE_LIBCURL
	janus_mutex_lock(&source->rtsp_mutex);
	if(source->curl) {
		/* Send an RTSP TEARDOWN */
		curl_easy_setopt(source->curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_TEARDOWN);
		int res = curl_easy_perform(source->curl);
		if(res != CURLE_OK) {
			JANUS_LOG(LOG_ERR, "Couldn't send TEARDOWN request: %s\n", curl_easy_strerror(res));
		}
		curl_easy_cleanup(source->curl);
	}
	janus_streaming_buffer *curldata = source->curldata;
	if(curldata != NULL) {
		g_free(curldata->buffer);
		g_free(curldata);
	}
	g_free(source->rtsp_url);
	g_free(source->rtsp_username);
	g_free(source->rtsp_password);
	janus_mutex_unlock(&source->rtsp_mutex);
#endif
	g_free(source);
}

static void janus_streaming_file_source_free(janus_streaming_file_source *source) {
	g_free(source->filename);
	g_free(source);
}

static void janus_streaming_mountpoint_free(janus_streaming_mountpoint *mp) {
	mp->destroyed = janus_get_monotonic_time();

	g_free(mp->name);
	g_free(mp->description);
	g_free(mp->secret);
	g_free(mp->pin);
	janus_mutex_lock(&mp->mutex);
	g_list_free(mp->listeners);
	janus_mutex_unlock(&mp->mutex);

	if(mp->source != NULL && mp->source_destroy != NULL) {
		mp->source_destroy(mp->source);
	}

	g_free(mp->codecs.audio_rtpmap);
	g_free(mp->codecs.audio_fmtp);
	g_free(mp->codecs.video_rtpmap);
	g_free(mp->codecs.video_fmtp);

	g_free(mp);
}


/* Helper to create an RTP live source (e.g., from gstreamer/ffmpeg/vlc/etc.) */
janus_streaming_mountpoint *janus_streaming_create_rtp_source(
		uint64_t id, char *name, char *desc,
		gboolean doaudio, char *amcast, const janus_network_address *aiface, uint16_t aport, uint8_t acodec, char *artpmap, char *afmtp,
		gboolean dovideo, char *vmcast, const janus_network_address *viface, uint16_t vport, uint8_t vcodec, char *vrtpmap, char *vfmtp, gboolean bufferkf,
			gboolean simulcast, uint16_t vport2, uint16_t vport3,
		gboolean dodata, const janus_network_address *diface, uint16_t dport, gboolean buffermsg) {
	janus_mutex_lock(&mountpoints_mutex);
	if(id == 0) {
		JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
		while(id == 0) {
			id = janus_random_uint64();
			if(g_hash_table_lookup(mountpoints, &id) != NULL) {
				/* ID already in use, try another one */
				id = 0;
			}
		}
	}
	char tempname[255];
	if(name == NULL) {
		JANUS_LOG(LOG_VERB, "Missing name, will generate a random one...\n");
		memset(tempname, 0, 255);
		g_snprintf(tempname, 255, "%"SCNu64, id);
	}
	if(!doaudio && !dovideo && !dodata) {
		JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream, no audio, video or data have to be streamed...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	if(doaudio && (artpmap == NULL)) {
		JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream, missing mandatory information for audio...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	if(dovideo && (vcodec == 0 || vrtpmap == NULL)) {
		JANUS_LOG(LOG_ERR, "Can't add 'rtp' stream, missing mandatory information for video...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "Audio %s, Video %s, Data %s\n",
		doaudio ? "enabled" : "NOT enabled",
		dovideo ? "enabled" : "NOT enabled",
		dodata ? "enabled" : "NOT enabled");
	/* First of all, let's check if the requested ports are free */
	int audio_fd = -1;
	if(doaudio) {
		audio_fd = janus_streaming_create_fd(aport, amcast ? inet_addr(amcast) : INADDR_ANY, aiface,
			"Audio", "audio", name ? name : tempname);
		if(audio_fd < 0) {
			JANUS_LOG(LOG_ERR, "Can't bind to port %d for audio...\n", aport);
			janus_mutex_unlock(&mountpoints_mutex);
			return NULL;
		}
	}
	int video_fd[3] = {-1, -1, -1};
	if(dovideo) {
		video_fd[0] = janus_streaming_create_fd(vport, vmcast ? inet_addr(vmcast) : INADDR_ANY, viface,
			"Video", "video", name ? name : tempname);
		if(video_fd[0] < 0) {
			JANUS_LOG(LOG_ERR, "Can't bind to port %d for video...\n", vport);
			if(audio_fd > -1)
				close(audio_fd);
			janus_mutex_unlock(&mountpoints_mutex);
			return NULL;
		}
		if(simulcast) {
			if(vport2 > 0) {
				video_fd[1] = janus_streaming_create_fd(vport2, vmcast ? inet_addr(vmcast) : INADDR_ANY, viface,
					"Video", "video", name ? name : tempname);
				if(video_fd[1] < 0) {
					JANUS_LOG(LOG_ERR, "Can't bind to port %d for video (2nd port)...\n", vport2);
					if(audio_fd > -1)
						close(audio_fd);
					if(video_fd[0] > -1)
						close(video_fd[0]);
					janus_mutex_unlock(&mountpoints_mutex);
					return NULL;
				}
			}
			if(vport3 > 0) {
				video_fd[2] = janus_streaming_create_fd(vport3, vmcast ? inet_addr(vmcast) : INADDR_ANY, viface,
					"Video", "video", name ? name : tempname);
				if(video_fd[2] < 0) {
					JANUS_LOG(LOG_ERR, "Can't bind to port %d for video (3rd port)...\n", vport3);
					if(audio_fd > -1)
						close(audio_fd);
					if(video_fd[0] > -1)
						close(video_fd[0]);
					if(video_fd[1] > -1)
						close(video_fd[1]);
					janus_mutex_unlock(&mountpoints_mutex);
					return NULL;
				}
			}
		}
	}
	int data_fd = -1;
	if(dodata) {
#ifdef HAVE_SCTP
		data_fd = janus_streaming_create_fd(dport, INADDR_ANY, diface,
			"Data", "data", name ? name : tempname);
		if(data_fd < 0) {
			JANUS_LOG(LOG_ERR, "Can't bind to port %d for data...\n", dport);
			if(audio_fd > -1)
				close(audio_fd);
			if(video_fd[0] > -1)
				close(video_fd[0]);
			if(video_fd[1] > -1)
				close(video_fd[1]);
			if(video_fd[2] > -1)
				close(video_fd[2]);
			janus_mutex_unlock(&mountpoints_mutex);
			return NULL;
		}
#else
		JANUS_LOG(LOG_WARN, "Mountpoint wants to do datachannel relaying, but datachannels support was not compiled...\n");
		dodata = FALSE;
#endif
	}
	/* Create the mountpoint */
	janus_network_address nil;
	janus_network_address_nullify(&nil);

	janus_streaming_mountpoint *live_rtp = g_malloc0(sizeof(janus_streaming_mountpoint));
	live_rtp->id = id;
	live_rtp->name = g_strdup(name ? name : tempname);
	char *description = NULL;
	if(desc != NULL)
		description = g_strdup(desc);
	else
		description = g_strdup(name ? name : tempname);
	live_rtp->description = description;
	live_rtp->enabled = TRUE;
	live_rtp->active = FALSE;
	live_rtp->audio = doaudio;
	live_rtp->video = dovideo;
	live_rtp->data = dodata;
	live_rtp->streaming_type = janus_streaming_type_live;
	live_rtp->streaming_source = janus_streaming_source_rtp;
	janus_streaming_rtp_source *live_rtp_source = g_malloc0(sizeof(janus_streaming_rtp_source));
	live_rtp_source->audio_mcast = doaudio ? (amcast ? inet_addr(amcast) : INADDR_ANY) : INADDR_ANY;
	live_rtp_source->audio_iface = doaudio && !janus_network_address_is_null(aiface) ? *aiface : nil;
	live_rtp_source->audio_port = doaudio ? aport : -1;
	live_rtp_source->video_mcast = dovideo ? (vmcast ? inet_addr(vmcast) : INADDR_ANY) : INADDR_ANY;
	live_rtp_source->video_port[0] = dovideo ? vport : -1;
	live_rtp_source->simulcast = dovideo && simulcast;
	live_rtp_source->video_port[1] = live_rtp_source->simulcast ? vport2 : -1;
	live_rtp_source->video_port[2] = live_rtp_source->simulcast ? vport3 : -1;
	live_rtp_source->video_iface = dovideo && !janus_network_address_is_null(viface) ? *viface : nil;
	live_rtp_source->data_port = dodata ? dport : -1;
	live_rtp_source->data_iface = dodata && !janus_network_address_is_null(diface) ? *diface : nil;
	live_rtp_source->arc = NULL;
	live_rtp_source->vrc = NULL;
	live_rtp_source->drc = NULL;
	janus_rtp_switching_context_reset(&live_rtp_source->context[0]);
	janus_rtp_switching_context_reset(&live_rtp_source->context[1]);
	janus_rtp_switching_context_reset(&live_rtp_source->context[2]);
	janus_mutex_init(&live_rtp_source->rec_mutex);
	live_rtp_source->audio_fd = audio_fd;
	live_rtp_source->video_fd[0] = video_fd[0];
	live_rtp_source->video_fd[1] = video_fd[1];
	live_rtp_source->video_fd[2] = video_fd[2];
	live_rtp_source->data_fd = data_fd;
	live_rtp_source->last_received_audio = janus_get_monotonic_time();
	live_rtp_source->last_received_video = janus_get_monotonic_time();
	live_rtp_source->last_received_data = janus_get_monotonic_time();
	live_rtp_source->keyframe.enabled = bufferkf;
	live_rtp_source->keyframe.latest_keyframe = NULL;
	live_rtp_source->keyframe.temp_keyframe = NULL;
	live_rtp_source->keyframe.temp_ts = 0;
	janus_mutex_init(&live_rtp_source->keyframe.mutex);
	live_rtp_source->buffermsg = buffermsg;
	live_rtp_source->last_msg = NULL;
	janus_mutex_init(&live_rtp_source->buffermsg_mutex);
	live_rtp->source = live_rtp_source;
	live_rtp->source_destroy = (GDestroyNotify) janus_streaming_rtp_source_free;
	live_rtp->codecs.audio_pt = doaudio ? acodec : -1;
	live_rtp->codecs.audio_rtpmap = doaudio ? g_strdup(artpmap) : NULL;
	live_rtp->codecs.audio_fmtp = doaudio ? (afmtp ? g_strdup(afmtp) : NULL) : NULL;
	live_rtp->codecs.video_codec = -1;
	if(dovideo) {
		if(strstr(vrtpmap, "vp8") || strstr(vrtpmap, "VP8"))
			live_rtp->codecs.video_codec = JANUS_STREAMING_VP8;
		else if(strstr(vrtpmap, "vp9") || strstr(vrtpmap, "VP9"))
			live_rtp->codecs.video_codec = JANUS_STREAMING_VP9;
		else if(strstr(vrtpmap, "h264") || strstr(vrtpmap, "H264"))
			live_rtp->codecs.video_codec = JANUS_STREAMING_H264;
	}
	live_rtp->codecs.video_pt = dovideo ? vcodec : -1;
	live_rtp->codecs.video_rtpmap = dovideo ? g_strdup(vrtpmap) : NULL;
	live_rtp->codecs.video_fmtp = dovideo ? (vfmtp ? g_strdup(vfmtp) : NULL) : NULL;
	live_rtp->listeners = NULL;
	live_rtp->destroyed = 0;
	janus_mutex_init(&live_rtp->mutex);
	g_hash_table_insert(mountpoints, janus_uint64_dup(live_rtp->id), live_rtp);
	janus_mutex_unlock(&mountpoints_mutex);
	GError *error = NULL;
	char tname[16];
	g_snprintf(tname, sizeof(tname), "mp %"SCNu64, live_rtp->id);
	g_thread_try_new(tname, &janus_streaming_relay_thread, live_rtp, &error);
	if(error != NULL) {
		JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the RTP thread...\n", error->code, error->message ? error->message : "??");
		g_free(live_rtp->name);
		g_free(description);
		g_free(live_rtp_source);
		g_free(live_rtp);
		return NULL;
	}
	return live_rtp;
}

/* Helper to create a file/ondemand live source */
janus_streaming_mountpoint *janus_streaming_create_file_source(
		uint64_t id, char *name, char *desc, char *filename,
		gboolean live, gboolean doaudio, gboolean dovideo) {
	janus_mutex_lock(&mountpoints_mutex);
	if(filename == NULL) {
		JANUS_LOG(LOG_ERR, "Can't add 'live' stream, missing filename...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	if(name == NULL) {
		JANUS_LOG(LOG_VERB, "Missing name, will generate a random one...\n");
	}
	if(id == 0) {
		JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
		while(id == 0) {
			id = janus_random_uint64();
			if(g_hash_table_lookup(mountpoints, &id) != NULL) {
				/* ID already in use, try another one */
				id = 0;
			}
		}
	}
	if(!doaudio && !dovideo) {
		JANUS_LOG(LOG_ERR, "Can't add 'file' stream, no audio or video have to be streamed...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	/* FIXME We don't support video streaming from file yet */
	if(!doaudio || dovideo) {
		JANUS_LOG(LOG_ERR, "Can't add 'file' stream, we only support audio file streaming right now...\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	/* TODO We should support something more than raw a-Law and mu-Law streams... */
	if(!strstr(filename, ".alaw") && !strstr(filename, ".mulaw")) {
		JANUS_LOG(LOG_ERR, "Can't add 'file' stream, unsupported format (we only support raw mu-Law and a-Law files right now)\n");
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	janus_streaming_mountpoint *file_source = g_malloc0(sizeof(janus_streaming_mountpoint));
	file_source->id = id;
	char tempname[255];
	if(!name) {
		memset(tempname, 0, 255);
		g_snprintf(tempname, 255, "%"SCNu64, file_source->id);
	}
	file_source->name = g_strdup(name ? name : tempname);
	char *description = NULL;
	if(desc != NULL)
		description = g_strdup(desc);
	else
		description = g_strdup(name ? name : tempname);
	file_source->description = description;
	file_source->enabled = TRUE;
	file_source->active = FALSE;
	file_source->audio = TRUE;
	file_source->video = FALSE;
	file_source->data = FALSE;
	file_source->streaming_type = live ? janus_streaming_type_live : janus_streaming_type_on_demand;
	file_source->streaming_source = janus_streaming_source_file;
	janus_streaming_file_source *file_source_source = g_malloc0(sizeof(janus_streaming_file_source));
	file_source_source->filename = g_strdup(filename);
	file_source->source = file_source_source;
	file_source->source_destroy = (GDestroyNotify) janus_streaming_file_source_free;
	file_source->codecs.audio_pt = strstr(filename, ".alaw") ? 8 : 0;
	file_source->codecs.audio_rtpmap = g_strdup(strstr(filename, ".alaw") ? "PCMA/8000" : "PCMU/8000");
	file_source->codecs.video_pt = -1;	/* FIXME We don't support video for this type yet */
	file_source->codecs.video_rtpmap = NULL;
	file_source->listeners = NULL;
	file_source->destroyed = 0;
	janus_mutex_init(&file_source->mutex);
	g_hash_table_insert(mountpoints, janus_uint64_dup(file_source->id), file_source);
	janus_mutex_unlock(&mountpoints_mutex);
	if(live) {
		GError *error = NULL;
		char tname[16];
		g_snprintf(tname, sizeof(tname), "mp %"SCNu64, file_source->id);
		g_thread_try_new(tname, &janus_streaming_filesource_thread, file_source, &error);
		if(error != NULL) {
			JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the live filesource thread...\n", error->code, error->message ? error->message : "??");
			g_free(file_source->name);
			g_free(description);
			g_free(file_source_source);
			g_free(file_source);
			return NULL;
		}
	}
	return file_source;
}

#ifdef HAVE_LIBCURL
static size_t janus_streaming_rtsp_curl_callback(void *payload, size_t size, size_t nmemb, void *data) {
	size_t realsize = size * nmemb;
	janus_streaming_buffer *buf = (struct janus_streaming_buffer *)data;
	/* (Re)allocate if needed */
	buf->buffer = realloc(buf->buffer, buf->size+realsize+1);
	/* Update the buffer */
	memcpy(&(buf->buffer[buf->size]), payload, realsize);
	buf->size += realsize;
	buf->buffer[buf->size] = 0;
	/* Done! */
	return realsize;
}

static int janus_streaming_rtsp_parse_sdp(const char *buffer, const char *name, const char *media, int *pt,
		char *transport, char *rtpmap, char *fmtp, char *control, const janus_network_address *iface, multiple_fds *fds) {
	char pattern[256];
	g_snprintf(pattern, sizeof(pattern), "m=%s", media);
	char *m = strstr(buffer, pattern);
	if(m == NULL) {
		JANUS_LOG(LOG_VERB, "[%s] no media %s...\n", name, media);
		return -1;
	}
	sscanf(m, "m=%*s %*d %*s %d", pt);
	char *s = strstr(m, "a=control:");
	if(s == NULL) {
		JANUS_LOG(LOG_ERR, "[%s] no control for %s...\n", name, media);
		return -1;
	}
	sscanf(s, "a=control:%2047s", control);
	char *r = strstr(m, "a=rtpmap:");
	if(r != NULL) {
		sscanf(r, "a=rtpmap:%*d %2047s", rtpmap);
	}
	char *f = strstr(m, "a=fmtp:");
	if(f != NULL) {
		sscanf(f, "a=fmtp:%*d %2047[^\r\n]s", fmtp);
	}
	char *c = strstr(m, "c=IN IP4");
	char ip[256];
	in_addr_t mcast = INADDR_ANY;
	if(c != NULL) {
		if(sscanf(c, "c=IN IP4 %[^/]", ip) != 0) {
			mcast = inet_addr(ip);
		}
	}
	int port;
	struct sockaddr_in address;
	socklen_t len = sizeof(address);
	/* loop until can bind two adjacent ports for RTP and RTCP */
	do {
		fds->fd = janus_streaming_create_fd(0, mcast, iface, media, media, name);
		if(fds->fd < 0) {
			return -1;
		}
		if(getsockname(fds->fd, (struct sockaddr *)&address, &len) < 0) {
			JANUS_LOG(LOG_ERR, "[%s] Bind failed for %s...\n", name, media);
			close(fds->fd);
			return -1;
		}
		port = ntohs(address.sin_port);
		fds->rtcp_fd = janus_streaming_create_fd(port+1, mcast, iface, media, media, name);
		if(fds->rtcp_fd < 0) {
			close(fds->fd);
			port = -1;
		} else if(getsockname(fds->rtcp_fd, (struct sockaddr *)&address, &len) < 0) {
			close(fds->fd);
			close(fds->rtcp_fd);
			port = -1;
		}
	} while(port == -1);

	if(IN_MULTICAST(ntohl(mcast))) {
		g_snprintf(transport, 1024, "RTP/AVP/UDP;multicast;client_port=%d-%d", port, port+1);
	} else {
		g_snprintf(transport, 1024, "RTP/AVP/UDP;unicast;client_port=%d-%d", port, port+1);
	}

	return 0;
}

/* Static helper to connect to an RTSP server, considering we might do this either
 * when creating a new mountpoint, or when reconnecting after some failure */
static int janus_streaming_rtsp_connect_to_server(janus_streaming_mountpoint *mp) {
	if(mp == NULL)
		return -1;
	janus_streaming_rtp_source *source = (janus_streaming_rtp_source *)mp->source;
	if(source == NULL)
		return -1;

	char *name = mp->name;
	gboolean doaudio = mp->audio;
	gboolean dovideo = mp->video;

	CURL *curl = curl_easy_init();
	if(curl == NULL) {
		JANUS_LOG(LOG_ERR, "Can't init CURL\n");
		return -1;
	}
	if(janus_log_level > LOG_INFO)
		curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
	curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 1L);
	curl_easy_setopt(curl, CURLOPT_URL, source->rtsp_url);
	curl_easy_setopt(curl, CURLOPT_TIMEOUT, 10L);
	curl_easy_setopt(curl, CURLOPT_NOSIGNAL, 0L);
	/* Any authentication to take into account? */
	if(source->rtsp_username && source->rtsp_password) {
		/* Point out that digest authentication is only available is libcurl >= 7.45.0 */
		if(LIBCURL_VERSION_NUM < 0x072d00) {
			JANUS_LOG(LOG_WARN, "RTSP digest authentication unsupported (needs libcurl >= 7.45.0)\n");
		}
		curl_easy_setopt(curl, CURLOPT_HTTPAUTH, CURLAUTH_ANY);
		curl_easy_setopt(curl, CURLOPT_USERNAME, source->rtsp_username);
		curl_easy_setopt(curl, CURLOPT_PASSWORD, source->rtsp_password);
	}
	/* Send an RTSP DESCRIBE */
	janus_streaming_buffer *curldata = g_malloc0(sizeof(janus_streaming_buffer));
	curldata->buffer = g_malloc0(1);
	curldata->size = 0;
	curl_easy_setopt(curl, CURLOPT_RTSP_STREAM_URI, source->rtsp_url);
	curl_easy_setopt(curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_DESCRIBE);
	curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, janus_streaming_rtsp_curl_callback);
	curl_easy_setopt(curl, CURLOPT_WRITEDATA, curldata);
	curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, janus_streaming_rtsp_curl_callback);
	curl_easy_setopt(curl, CURLOPT_HEADERDATA, curldata);
	int res = curl_easy_perform(curl);
	if(res != CURLE_OK) {
		JANUS_LOG(LOG_ERR, "Couldn't send DESCRIBE request: %s\n", curl_easy_strerror(res));
		curl_easy_cleanup(curl);
		g_free(curldata->buffer);
		g_free(curldata);
		return -2;
	}
	long code = 0;
	res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code);
	if(res != CURLE_OK) {
		JANUS_LOG(LOG_ERR, "Couldn't get DESCRIBE answer: %s\n", curl_easy_strerror(res));
		curl_easy_cleanup(curl);
		g_free(curldata->buffer);
		g_free(curldata);
		return -3;
	} else if(code != 200) {
		JANUS_LOG(LOG_ERR, "Couldn't get DESCRIBE code: %ld\n", code);
		curl_easy_cleanup(curl);
		g_free(curldata->buffer);
		g_free(curldata);
		return -4;
	}
	JANUS_LOG(LOG_VERB, "DESCRIBE answer:%s\n", curldata->buffer);
	/* Parse the SDP we just got to figure out the negotiated media */
	int ka_timeout = 0;
	int vpt = -1;
	char vrtpmap[2048];
	char vfmtp[2048];
	char vcontrol[2048];
	char uri[1024];
	char vtransport[1024];
	multiple_fds video_fds = {-1, -1};

	int apt = -1;
	char artpmap[2048];
	char afmtp[2048];
	char acontrol[2048];
	char atransport[1024];
	multiple_fds audio_fds = {-1, -1};

	/* Parse both video and audio first before proceed to setup as curldata will be reused */
	int vresult;
	vresult = janus_streaming_rtsp_parse_sdp(curldata->buffer, name, "video", &vpt,
		vtransport, vrtpmap, vfmtp, vcontrol, &source->video_iface, &video_fds);

	int aresult;
	aresult = janus_streaming_rtsp_parse_sdp(curldata->buffer, name, "audio", &apt,
		atransport, artpmap, afmtp, acontrol, &source->audio_iface, &audio_fds);

	if(vresult != -1) {
		/* Send an RTSP SETUP for video */
		g_free(curldata->buffer);
		curldata->buffer = g_malloc0(1);
		curldata->size = 0;
		if(strstr(vcontrol, source->rtsp_url) == vcontrol) {
			/* The control attribute already contains the whole URL? */
			g_snprintf(uri, sizeof(uri), "%s", vcontrol);
		} else {
			/* Append the control attribute to the URL */
			g_snprintf(uri, sizeof(uri), "%s/%s", source->rtsp_url, vcontrol);
		}
		curl_easy_setopt(curl, CURLOPT_RTSP_STREAM_URI, uri);
		curl_easy_setopt(curl, CURLOPT_RTSP_TRANSPORT, vtransport);
		curl_easy_setopt(curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_SETUP);
		res = curl_easy_perform(curl);
		if(res != CURLE_OK) {
			JANUS_LOG(LOG_ERR, "Couldn't send SETUP request: %s\n", curl_easy_strerror(res));
			curl_easy_cleanup(curl);
			g_free(curldata->buffer);
			g_free(curldata);
			return -5;
		}
		JANUS_LOG(LOG_VERB, "SETUP answer:%s\n", curldata->buffer);
		const char *timeout = strstr(curldata->buffer, ";timeout=");
		if(timeout != NULL) {
			/* There's a timeout to take into account: take note of the
			 * value for sending GET_PARAMETER keepalives later on */
			ka_timeout = atoi(timeout+strlen(";timeout="));
			JANUS_LOG(LOG_VERB, "  -- RTSP session timeout (video): %d\n", ka_timeout);
		}
	}

	if(aresult != -1) {
		/* Send an RTSP SETUP for audio */
		g_free(curldata->buffer);
		curldata->buffer = g_malloc0(1);
		curldata->size = 0;
		g_snprintf(uri, sizeof(uri), "%s/%s", source->rtsp_url, acontrol);
		curl_easy_setopt(curl, CURLOPT_RTSP_STREAM_URI, uri);
		curl_easy_setopt(curl, CURLOPT_RTSP_TRANSPORT, atransport);
		curl_easy_setopt(curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_SETUP);
		res = curl_easy_perform(curl);
		if(res != CURLE_OK) {
			JANUS_LOG(LOG_ERR, "Couldn't send SETUP request: %s\n", curl_easy_strerror(res));
			curl_easy_cleanup(curl);
			g_free(curldata->buffer);
			g_free(curldata);
			return -6;
		}
		JANUS_LOG(LOG_VERB, "SETUP answer:%s\n", curldata->buffer);
		const char *timeout = strstr(curldata->buffer, ";timeout=");
		if(timeout != NULL) {
			/* There's a timeout to take into account: take note of the
			 * value for sending GET_PARAMETER keepalives later on */
			int temp_timeout = atoi(timeout+strlen(";timeout="));
			JANUS_LOG(LOG_VERB, "  -- RTSP session timeout (audio): %d\n", temp_timeout);
			if(temp_timeout > 0 && temp_timeout < ka_timeout)
				ka_timeout = temp_timeout;
		}
	}

	/* Update the source (but check if rtpmap/fmtp need to be overridden) */
	mp->codecs.audio_pt = doaudio ? apt : -1;
	if(mp->codecs.audio_rtpmap == NULL)
		mp->codecs.audio_rtpmap = doaudio ? g_strdup(artpmap) : NULL;
	if(mp->codecs.audio_fmtp == NULL)
		mp->codecs.audio_fmtp = doaudio ? g_strdup(afmtp) : NULL;
	mp->codecs.video_pt = dovideo ? vpt : -1;
	if(mp->codecs.video_rtpmap == NULL)
		mp->codecs.video_rtpmap = dovideo ? g_strdup(vrtpmap) : NULL;
	if(mp->codecs.video_fmtp == NULL)
		mp->codecs.video_fmtp = dovideo ? g_strdup(vfmtp) : NULL;
	source->audio_fd = audio_fds.fd;
	source->audio_rtcp_fd = audio_fds.rtcp_fd;
	source->video_fd[0] = video_fds.fd;
	source->video_rtcp_fd = video_fds.rtcp_fd;
	source->curl = curl;
	source->curldata = curldata;
	if(ka_timeout > 0)
		source->ka_timeout = ka_timeout;
	return 0;
}

/* Helper to send an RTSP PLAY (either when we create the mountpoint, or when we try reconnecting) */
static int janus_streaming_rtsp_play(janus_streaming_rtp_source *source) {
	if(source == NULL || source->curldata == NULL)
		return -1;
	/* Send an RTSP PLAY */
	janus_mutex_lock(&source->rtsp_mutex);
	g_free(source->curldata->buffer);
	source->curldata->buffer = g_malloc0(1);
	source->curldata->size = 0;
	JANUS_LOG(LOG_VERB, "Sending PLAY request...\n");
	curl_easy_setopt(source->curl, CURLOPT_RTSP_STREAM_URI, source->rtsp_url);
	curl_easy_setopt(source->curl, CURLOPT_RANGE, "0.000-");
	curl_easy_setopt(source->curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_PLAY);
	int res = curl_easy_perform(source->curl);
	if(res != CURLE_OK) {
		JANUS_LOG(LOG_ERR, "Couldn't send PLAY request: %s\n", curl_easy_strerror(res));
		janus_mutex_unlock(&source->rtsp_mutex);
		return -1;
	}
	JANUS_LOG(LOG_VERB, "PLAY answer:%s\n", source->curldata->buffer);
	janus_mutex_unlock(&source->rtsp_mutex);
	return 0;
}

/* Helper to create an RTSP source */
janus_streaming_mountpoint *janus_streaming_create_rtsp_source(
		uint64_t id, char *name, char *desc,
		char *url, char *username, char *password,
		gboolean doaudio, char *artpmap, char *afmtp,
		gboolean dovideo, char *vrtpmap, char *vfmtp,
		const janus_network_address *iface) {
	if(url == NULL) {
		JANUS_LOG(LOG_ERR, "Can't add 'rtsp' stream, missing url...\n");
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "Audio %s, Video %s\n", doaudio ? "enabled" : "NOT enabled", dovideo ? "enabled" : "NOT enabled");

	janus_mutex_lock(&mountpoints_mutex);
	/* Create an RTP source for the media we'll get */
	if(id == 0) {
		JANUS_LOG(LOG_VERB, "Missing id, will generate a random one...\n");
		while(id == 0) {
			id = janus_random_uint64();
			if(g_hash_table_lookup(mountpoints, &id) != NULL) {
				/* ID already in use, try another one */
				id = 0;
			}
		}
	}
	char tempname[255];
	if(name == NULL) {
		JANUS_LOG(LOG_VERB, "Missing name, will generate a random one...\n");
		memset(tempname, 0, 255);
		g_snprintf(tempname, 255, "%"SCNu64, id);
	}
	char *sourcename =  g_strdup(name ? name : tempname);
	char *description = NULL;
	if(desc != NULL) {
		description = g_strdup(desc);
	} else {
		description = g_strdup(name ? name : tempname);
	}

	janus_network_address nil;
	janus_network_address_nullify(&nil);

	/* Create the mountpoint and prepare the source */
	janus_streaming_mountpoint *live_rtsp = g_malloc0(sizeof(janus_streaming_mountpoint));
	live_rtsp->id = id;
	live_rtsp->name = sourcename;
	live_rtsp->description = description;
	live_rtsp->enabled = TRUE;
	live_rtsp->active = FALSE;
	live_rtsp->audio = doaudio;
	live_rtsp->video = dovideo;
	live_rtsp->data = FALSE;
	live_rtsp->streaming_type = janus_streaming_type_live;
	live_rtsp->streaming_source = janus_streaming_source_rtp;
	janus_streaming_rtp_source *live_rtsp_source = g_malloc0(sizeof(janus_streaming_rtp_source));
	live_rtsp_source->rtsp = TRUE;
	live_rtsp_source->rtsp_url = g_strdup(url);
	live_rtsp_source->rtsp_username = username ? g_strdup(username) : NULL;
	live_rtsp_source->rtsp_password = password ? g_strdup(password) : NULL;
	live_rtsp_source->arc = NULL;
	live_rtsp_source->vrc = NULL;
	live_rtsp_source->drc = NULL;
	live_rtsp_source->audio_fd = -1;
	live_rtsp_source->audio_rtcp_fd = -1;
	live_rtsp_source->audio_iface = iface ? *iface : nil;
	live_rtsp_source->video_fd[0] = -1;
	live_rtsp_source->video_fd[1] = -1;
	live_rtsp_source->video_fd[2] = -1;
	live_rtsp_source->video_rtcp_fd = -1;
	live_rtsp_source->video_iface = iface ? *iface : nil;
	live_rtsp_source->data_fd = -1;
	live_rtsp_source->data_iface = nil;
	live_rtsp_source->reconnect_timer = 0;
	janus_mutex_init(&live_rtsp_source->rtsp_mutex);
	live_rtsp->source = live_rtsp_source;
	live_rtsp->source_destroy = (GDestroyNotify) janus_streaming_rtp_source_free;
	live_rtsp->listeners = NULL;
	live_rtsp->destroyed = 0;
	janus_mutex_init(&live_rtsp->mutex);
	/* We may have to override the rtpmap and/or fmtp for audio and/or video */
	live_rtsp->codecs.audio_rtpmap = doaudio ? (artpmap ? g_strdup(artpmap) : NULL) : NULL;
	live_rtsp->codecs.audio_fmtp = doaudio ? (afmtp ? g_strdup(afmtp) : NULL) : NULL;
	live_rtsp->codecs.video_rtpmap = dovideo ? (vrtpmap ? g_strdup(vrtpmap) : NULL) : NULL;
	live_rtsp->codecs.video_fmtp = dovideo ? (vfmtp ? g_strdup(vfmtp) : NULL) : NULL;
	/* Now connect to the RTSP server */
	if(janus_streaming_rtsp_connect_to_server(live_rtsp) < 0) {
		/* Error connecting, get rid of the mountpoint */
		janus_streaming_mountpoint_free(live_rtsp);
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	/* Send an RTSP PLAY, now */
	if(janus_streaming_rtsp_play(live_rtsp_source) < 0) {
		/* Error trying to play, get rid of the mountpoint */
		janus_streaming_mountpoint_free(live_rtsp);
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	/* Start the thread that will receive the media packets */
	GError *error = NULL;
	char tname[16];
	g_snprintf(tname, sizeof(tname), "mp %"SCNu64, live_rtsp->id);
	g_thread_try_new(tname, &janus_streaming_relay_thread, live_rtsp, &error);
	if(error != NULL) {
		JANUS_LOG(LOG_ERR, "Got error %d (%s) trying to launch the RTSP thread...\n", error->code, error->message ? error->message : "??");
		janus_streaming_mountpoint_free(live_rtsp);
		janus_mutex_unlock(&mountpoints_mutex);
		return NULL;
	}
	g_hash_table_insert(mountpoints, janus_uint64_dup(live_rtsp->id), live_rtsp);
	janus_mutex_unlock(&mountpoints_mutex);
	return live_rtsp;
}
#else
/* Helper to create an RTSP source */
janus_streaming_mountpoint *janus_streaming_create_rtsp_source(
		uint64_t id, char *name, char *desc,
		char *url, char *username, char *password,
		gboolean doaudio, char *audiortpmap, char *audiofmtp,
		gboolean dovideo, char *videortpmap, char *videofmtp,
		const janus_network_address *iface) {
	JANUS_LOG(LOG_ERR, "RTSP need libcurl\n");
	return NULL;
}
#endif

/* FIXME Thread to send RTP packets from a file (on demand) */
static void *janus_streaming_ondemand_thread(void *data) {
	JANUS_LOG(LOG_VERB, "Filesource (on demand) RTP thread starting...\n");
	janus_streaming_session *session = (janus_streaming_session *)data;
	if(!session) {
		JANUS_LOG(LOG_ERR, "Invalid session!\n");
		g_thread_unref(g_thread_self());
		return NULL;
	}
	janus_streaming_mountpoint *mountpoint = session->mountpoint;
	if(!mountpoint) {
		JANUS_LOG(LOG_ERR, "Invalid mountpoint!\n");
		g_thread_unref(g_thread_self());
		return NULL;
	}
	if(mountpoint->streaming_source != janus_streaming_source_file) {
		JANUS_LOG(LOG_ERR, "[%s] Not an file source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	if(mountpoint->streaming_type != janus_streaming_type_on_demand) {
		JANUS_LOG(LOG_ERR, "[%s] Not an on-demand file source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	janus_streaming_file_source *source = mountpoint->source;
	if(source == NULL || source->filename == NULL) {
		g_thread_unref(g_thread_self());
		JANUS_LOG(LOG_ERR, "[%s] Invalid file source mountpoint!\n", mountpoint->name);
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "[%s] Opening file source %s...\n", mountpoint->name, source->filename);
	FILE *audio = fopen(source->filename, "rb");
	if(!audio) {
		JANUS_LOG(LOG_ERR, "[%s] Ooops, audio file missing!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "[%s] Streaming audio file: %s\n", mountpoint->name, source->filename);
	/* Buffer */
	char *buf = g_malloc0(1024);
	char *name = g_strdup(mountpoint->name ? mountpoint->name : "??");
	/* Set up RTP */
	gint16 seq = 1;
	gint32 ts = 0;
	janus_rtp_header *header = (janus_rtp_header *)buf;
	header->version = 2;
	header->markerbit = 1;
	header->type = mountpoint->codecs.audio_pt;
	header->seq_number = htons(seq);
	header->timestamp = htonl(ts);
	header->ssrc = htonl(1);	/* The gateway will fix this anyway */
	/* Timer */
	struct timeval now, before;
	gettimeofday(&before, NULL);
	now.tv_sec = before.tv_sec;
	now.tv_usec = before.tv_usec;
	time_t passed, d_s, d_us;
	/* Loop */
	gint read = 0;
	janus_streaming_rtp_relay_packet packet;
	while(!g_atomic_int_get(&stopping) && !mountpoint->destroyed && !session->stopping && !session->destroyed) {
		/* See if it's time to prepare a frame */
		gettimeofday(&now, NULL);
		d_s = now.tv_sec - before.tv_sec;
		d_us = now.tv_usec - before.tv_usec;
		if(d_us < 0) {
			d_us += 1000000;
			--d_s;
		}
		passed = d_s*1000000 + d_us;
		if(passed < 18000) {	/* Let's wait about 18ms */
			usleep(1000);
			continue;
		}
		/* Update the reference time */
		before.tv_usec += 20000;
		if(before.tv_usec > 1000000) {
			before.tv_sec++;
			before.tv_usec -= 1000000;
		}
		/* If not started or paused, wait some more */
		if(!session->started || session->paused || !mountpoint->enabled)
			continue;
		/* Read frame from file... */
		read = fread(buf + RTP_HEADER_SIZE, sizeof(char), 160, audio);
		if(feof(audio)) {
			/* FIXME We're doing this forever... should this be configurable? */
			JANUS_LOG(LOG_VERB, "[%s] Rewind! (%s)\n", name, source->filename);
			fseek(audio, 0, SEEK_SET);
			continue;
		}
		if(read < 0)
			break;
		if(mountpoint->active == FALSE)
			mountpoint->active = TRUE;
		//~ JANUS_LOG(LOG_VERB, " ... Preparing RTP packet (pt=%u, seq=%u, ts=%u)...\n",
			//~ header->type, ntohs(header->seq_number), ntohl(header->timestamp));
		//~ JANUS_LOG(LOG_VERB, " ... Read %d bytes from the audio file...\n", read);
		/* Relay on all sessions */
		packet.data = header;
		packet.length = RTP_HEADER_SIZE + read;
		packet.is_rtp = TRUE;
		packet.is_video = FALSE;
		packet.is_keyframe = FALSE;
		/* Backup the actual timestamp and sequence number */
		packet.timestamp = ntohl(packet.data->timestamp);
		packet.seq_number = ntohs(packet.data->seq_number);
		/* Go! */
		janus_streaming_relay_rtp_packet(session, &packet);
		/* Update header */
		seq++;
		header->seq_number = htons(seq);
		ts += 160;
		header->timestamp = htonl(ts);
		header->markerbit = 0;
	}
	JANUS_LOG(LOG_VERB, "[%s] Leaving filesource (ondemand) thread\n", name);
	g_free(name);
	g_free(buf);
	fclose(audio);
	g_thread_unref(g_thread_self());
	return NULL;
}

/* FIXME Thread to send RTP packets from a file (live) */
static void *janus_streaming_filesource_thread(void *data) {
	JANUS_LOG(LOG_VERB, "Filesource (live) thread starting...\n");
	janus_streaming_mountpoint *mountpoint = (janus_streaming_mountpoint *)data;
	if(!mountpoint) {
		JANUS_LOG(LOG_ERR, "Invalid mountpoint!\n");
		g_thread_unref(g_thread_self());
		return NULL;
	}
	if(mountpoint->streaming_source != janus_streaming_source_file) {
		JANUS_LOG(LOG_ERR, "[%s] Not an file source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	if(mountpoint->streaming_type != janus_streaming_type_live) {
		JANUS_LOG(LOG_ERR, "[%s] Not a live file source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	janus_streaming_file_source *source = mountpoint->source;
	if(source == NULL || source->filename == NULL) {
		JANUS_LOG(LOG_ERR, "[%s] Invalid file source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "[%s] Opening file source %s...\n", mountpoint->name, source->filename);
	FILE *audio = fopen(source->filename, "rb");
	if(!audio) {
		JANUS_LOG(LOG_ERR, "[%s] Ooops, audio file missing!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	JANUS_LOG(LOG_VERB, "[%s] Streaming audio file: %s\n", mountpoint->name, source->filename);
	/* Buffer */
	char *buf = g_malloc0(1024);
	char *name = g_strdup(mountpoint->name ? mountpoint->name : "??");
	/* Set up RTP */
	gint16 seq = 1;
	gint32 ts = 0;
	janus_rtp_header *header = (janus_rtp_header *)buf;
	header->version = 2;
	header->markerbit = 1;
	header->type = mountpoint->codecs.audio_pt;
	header->seq_number = htons(seq);
	header->timestamp = htonl(ts);
	header->ssrc = htonl(1);	/* The gateway will fix this anyway */
	/* Timer */
	struct timeval now, before;
	gettimeofday(&before, NULL);
	now.tv_sec = before.tv_sec;
	now.tv_usec = before.tv_usec;
	time_t passed, d_s, d_us;
	/* Loop */
	gint read = 0;
	janus_streaming_rtp_relay_packet packet;
	while(!g_atomic_int_get(&stopping) && !mountpoint->destroyed) {
		/* See if it's time to prepare a frame */
		gettimeofday(&now, NULL);
		d_s = now.tv_sec - before.tv_sec;
		d_us = now.tv_usec - before.tv_usec;
		if(d_us < 0) {
			d_us += 1000000;
			--d_s;
		}
		passed = d_s*1000000 + d_us;
		if(passed < 18000) {	/* Let's wait about 18ms */
			usleep(1000);
			continue;
		}
		/* Update the reference time */
		before.tv_usec += 20000;
		if(before.tv_usec > 1000000) {
			before.tv_sec++;
			before.tv_usec -= 1000000;
		}
		/* If paused, wait some more */
		if(!mountpoint->enabled)
			continue;
		/* Read frame from file... */
		read = fread(buf + RTP_HEADER_SIZE, sizeof(char), 160, audio);
		if(feof(audio)) {
			/* FIXME We're doing this forever... should this be configurable? */
			JANUS_LOG(LOG_VERB, "[%s] Rewind! (%s)\n", name, source->filename);
			fseek(audio, 0, SEEK_SET);
			continue;
		}
		if(read < 0)
			break;
		if(mountpoint->active == FALSE)
			mountpoint->active = TRUE;
		// JANUS_LOG(LOG_VERB, " ... Preparing RTP packet (pt=%u, seq=%u, ts=%u)...\n",
			// header->type, ntohs(header->seq_number), ntohl(header->timestamp));
		// JANUS_LOG(LOG_VERB, " ... Read %d bytes from the audio file...\n", read);
		/* Relay on all sessions */
		packet.data = header;
		packet.length = RTP_HEADER_SIZE + read;
		packet.is_rtp = TRUE;
		packet.is_video = FALSE;
		packet.is_keyframe = FALSE;
		/* Backup the actual timestamp and sequence number */
		packet.timestamp = ntohl(packet.data->timestamp);
		packet.seq_number = ntohs(packet.data->seq_number);
		/* Go! */
		janus_mutex_lock_nodebug(&mountpoint->mutex);
		g_list_foreach(mountpoint->listeners, janus_streaming_relay_rtp_packet, &packet);
		janus_mutex_unlock_nodebug(&mountpoint->mutex);
		/* Update header */
		seq++;
		header->seq_number = htons(seq);
		ts += 160;
		header->timestamp = htonl(ts);
		header->markerbit = 0;
	}
	JANUS_LOG(LOG_VERB, "[%s] Leaving filesource (live) thread\n", name);
	g_free(name);
	g_free(buf);
	fclose(audio);
	g_thread_unref(g_thread_self());
	return NULL;
}

/* FIXME Test thread to relay RTP frames coming from gstreamer/ffmpeg/others */
static void *janus_streaming_relay_thread(void *data) {
	JANUS_LOG(LOG_VERB, "Starting streaming relay thread\n");
	janus_streaming_mountpoint *mountpoint = (janus_streaming_mountpoint *)data;
	if(!mountpoint) {
		JANUS_LOG(LOG_ERR, "Invalid mountpoint!\n");
		g_thread_unref(g_thread_self());
		return NULL;
	}
	if(mountpoint->streaming_source != janus_streaming_source_rtp) {
		JANUS_LOG(LOG_ERR, "[%s] Not an RTP source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	janus_streaming_rtp_source *source = mountpoint->source;
	if(source == NULL) {
		JANUS_LOG(LOG_ERR, "[%s] Invalid RTP source mountpoint!\n", mountpoint->name);
		g_thread_unref(g_thread_self());
		return NULL;
	}
	int audio_fd = source->audio_fd;
	int video_fd[3] = {source->video_fd[0], source->video_fd[1], source->video_fd[2]};
	int data_fd = source->data_fd;
	char *name = g_strdup(mountpoint->name ? mountpoint->name : "??");
	/* Needed to fix seq and ts */
	uint32_t ssrc = 0, a_last_ssrc = 0, v_last_ssrc[3] = {0, 0, 0};
	/* File descriptors */
	socklen_t addrlen;
	struct sockaddr_in remote;
	int resfd = 0, bytes = 0;
	struct pollfd fds[5];
	char buffer[1500];
	memset(buffer, 0, 1500);
#ifdef HAVE_LIBCURL
	/* In case this is an RTSP restreamer, we may have to send keep-alives from time to time */
	gint64 now = janus_get_monotonic_time(), before = now, ka_timeout = 0;
	if(source->rtsp) {
		source->reconnect_timer = now;
		ka_timeout = ((gint64)source->ka_timeout*G_USEC_PER_SEC)/2;
	}
#endif
	/* Loop */
	int num = 0;
	janus_streaming_rtp_relay_packet packet;
	while(!g_atomic_int_get(&stopping) && !mountpoint->destroyed) {
#ifdef HAVE_LIBCURL
		/* Let's check regularly if the RTSP server seems to be gone */
		if(source->rtsp) {
			if(source->reconnecting) {
				/* We're still reconnecting, wait some more */
				g_usleep(250000);
				continue;
			}
			now = janus_get_monotonic_time();
			if(!source->reconnecting && (now - source->reconnect_timer > 5*G_USEC_PER_SEC)) {
				/* 5 seconds passed and no media? Assume the RTSP server has gone and schedule a reconnect */
				JANUS_LOG(LOG_WARN, "[%s] %"SCNi64"s passed with no media, trying to reconnect the RTSP stream\n",
					name, (now - source->reconnect_timer)/G_USEC_PER_SEC);
				audio_fd = -1;
				video_fd[0] = -1;
				video_fd[1] = -1;
				video_fd[2] = -1;
				data_fd = -1;
				source->reconnect_timer = now;
				source->reconnecting = TRUE;
				/* Let's clean up the source first */
				curl_easy_cleanup(source->curl);
				source->curl = NULL;
				if(source->curldata)
					g_free(source->curldata->buffer);
				g_free(source->curldata);
				source->curldata = NULL;
				if(source->audio_fd > -1) {
					close(source->audio_fd);
				}
				source->audio_fd = -1;
				if(source->video_fd[0] > -1) {
					close(source->video_fd[0]);
				}
				source->video_fd[0] = -1;
				if(source->video_fd[1] > -1) {
					close(source->video_fd[1]);
				}
				source->video_fd[1] = -1;
				if(source->video_fd[2] > -1) {
					close(source->video_fd[2]);
				}
				source->video_fd[2] = -1;
				if(source->data_fd > -1) {
					close(source->data_fd);
				}
				source->data_fd = -1;
				if(source->audio_rtcp_fd > -1) {
					close(source->audio_rtcp_fd);
				}
				source->audio_rtcp_fd = -1;
				if(source->video_rtcp_fd > -1) {
					close(source->video_rtcp_fd);
				}
				source->video_rtcp_fd = -1;
				/* Now let's try to reconnect */
				if(janus_streaming_rtsp_connect_to_server(mountpoint) < 0) {
					/* Reconnection failed? Let's try again later */
					JANUS_LOG(LOG_WARN, "[%s] Reconnection of the RTSP stream failed, trying again in a few seconds...\n", name);
				} else {
					/* We're connected, let's send a PLAY */
					if(janus_streaming_rtsp_play(source) < 0) {
						/* Error trying to play? Let's try again later */
						JANUS_LOG(LOG_WARN, "[%s] RTSP PLAY failed, trying again in a few seconds...\n", name);
					} else {
						/* Everything should be back to normal, let's update the file descriptors */
						JANUS_LOG(LOG_INFO, "[%s] Reconnected to the RTSP server, streaming again\n", name);
						audio_fd = source->audio_fd;
						video_fd[0] = source->video_fd[0];
						data_fd = source->data_fd;
						ka_timeout = ((gint64)source->ka_timeout*G_USEC_PER_SEC)/2;
					}
				}
				source->reconnect_timer = janus_get_monotonic_time();
				source->reconnecting = FALSE;
				continue;
			}
		}
		if(audio_fd < 0 && video_fd[0] < 0 && video_fd[1] < 0 && video_fd[2] < 0 && data_fd < 0) {
			/* No socket, we may be in the process of reconnecting, or waiting to reconnect */
			g_usleep(5000000);
			continue;
		}
		/* We may also need to occasionally send a GET_PARAMETER request as a keep-alive */
		if(ka_timeout > 0) {
			/* Let's be conservative and send a GET_PARAMETER when half of the timeout has passed */
			now = janus_get_monotonic_time();
			if(now-before > ka_timeout && source->curldata) {
				JANUS_LOG(LOG_VERB, "[%s] %"SCNi64"s passed, sending GET_PARAMETER\n", name, (now-before)/G_USEC_PER_SEC);
				before = now;
				/* Send an RTSP GET_PARAMETER */
				janus_mutex_lock(&source->rtsp_mutex);
				g_free(source->curldata->buffer);
				source->curldata->buffer = g_malloc0(1);
				source->curldata->size = 0;
				curl_easy_setopt(source->curl, CURLOPT_RTSP_STREAM_URI, source->rtsp_url);
				curl_easy_setopt(source->curl, CURLOPT_RTSP_REQUEST, (long)CURL_RTSPREQ_GET_PARAMETER);
				resfd = curl_easy_perform(source->curl);
				if(resfd != CURLE_OK) {
					JANUS_LOG(LOG_ERR, "[%s] Couldn't send GET_PARAMETER request: %s\n", name, curl_easy_strerror(resfd));
				}
				janus_mutex_unlock(&source->rtsp_mutex);
			}
		}
#endif
		/* Prepare poll */
		num = 0;
		if(audio_fd != -1) {
			fds[num].fd = audio_fd;
			fds[num].events = POLLIN;
			fds[num].revents = 0;
			num++;
		}
		if(video_fd[0] != -1) {
			fds[num].fd = video_fd[0];
			fds[num].events = POLLIN;
			fds[num].revents = 0;
			num++;
		}
		if(video_fd[1] != -1) {
			fds[num].fd = video_fd[1];
			fds[num].events = POLLIN;
			fds[num].revents = 0;
			num++;
		}
		if(video_fd[2] != -1) {
			fds[num].fd = video_fd[2];
			fds[num].events = POLLIN;
			fds[num].revents = 0;
			num++;
		}
		if(data_fd != -1) {
			fds[num].fd = data_fd;
			fds[num].events = POLLIN;
			fds[num].revents = 0;
			num++;
		}
		/* Wait for some data */
		resfd = poll(fds, num, 1000);
		if(resfd < 0) {
			if(errno == EINTR) {
				JANUS_LOG(LOG_HUGE, "[%s] Got an EINTR (%s), ignoring...\n", name, strerror(errno));
				continue;
			}
			JANUS_LOG(LOG_ERR, "[%s] Error polling... %d (%s)\n", name, errno, strerror(errno));
			mountpoint->enabled = FALSE;
			break;
		} else if(resfd == 0) {
			/* No data, keep going */
			continue;
		}
		int i = 0;
		for(i=0; i<num; i++) {
			if(fds[i].revents & (POLLERR | POLLHUP)) {
				/* Socket error? */
				JANUS_LOG(LOG_ERR, "[%s] Error polling: %s... %d (%s)\n", name,
					fds[i].revents & POLLERR ? "POLLERR" : "POLLHUP", errno, strerror(errno));
				mountpoint->enabled = FALSE;
				break;
			} else if(fds[i].revents & POLLIN) {
				/* Got an RTP or data packet */
				if(audio_fd != -1 && fds[i].fd == audio_fd) {
					/* Got something audio (RTP) */
					if(mountpoint->active == FALSE)
						mountpoint->active = TRUE;
					source->last_received_audio = janus_get_monotonic_time();
#ifdef HAVE_LIBCURL
					source->reconnect_timer = janus_get_monotonic_time();
#endif
					addrlen = sizeof(remote);
					bytes = recvfrom(audio_fd, buffer, 1500, 0, (struct sockaddr*)&remote, &addrlen);
					if(bytes < 0) {
						/* Failed to read? */
						continue;
					}
					//~ JANUS_LOG(LOG_VERB, "************************\nGot %d bytes on the audio channel...\n", bytes);
					/* If paused, ignore this packet */
					if(!mountpoint->enabled)
						continue;
					janus_rtp_header *rtp = (janus_rtp_header *)buffer;
					//~ JANUS_LOG(LOG_VERB, " ... parsed RTP packet (ssrc=%u, pt=%u, seq=%u, ts=%u)...\n",
						//~ ntohl(rtp->ssrc), rtp->type, ntohs(rtp->seq_number), ntohl(rtp->timestamp));
					/* Relay on all sessions */
					packet.data = rtp;
					packet.length = bytes;
					packet.is_rtp = TRUE;
					packet.is_video = FALSE;
					packet.is_keyframe = FALSE;
					/* Do we have a new stream? */
					if(ntohl(packet.data->ssrc) != a_last_ssrc) {
						a_last_ssrc = ntohl(packet.data->ssrc);
						JANUS_LOG(LOG_INFO, "[%s] New audio stream! (ssrc=%u)\n", name, a_last_ssrc);
					}
					packet.data->type = mountpoint->codecs.audio_pt;
					/* Is there a recorder? */
					janus_rtp_header_update(packet.data, &source->context[0], FALSE, 960);
					ssrc = ntohl(packet.data->ssrc);
					packet.data->ssrc = ntohl((uint32_t)mountpoint->id);
					janus_recorder_save_frame(source->arc, buffer, bytes);
					packet.data->ssrc = ntohl(ssrc);
					/* Backup the actual timestamp and sequence number set by the restreamer, in case switching is involved */
					packet.timestamp = ntohl(packet.data->timestamp);
					packet.seq_number = ntohs(packet.data->seq_number);
					/* Go! */
					janus_mutex_lock(&mountpoint->mutex);
					g_list_foreach(mountpoint->listeners, janus_streaming_relay_rtp_packet, &packet);
					janus_mutex_unlock(&mountpoint->mutex);
					continue;
				} else if((video_fd[0] != -1 && fds[i].fd == video_fd[0]) ||
						(video_fd[1] != -1 && fds[i].fd == video_fd[1]) ||
						(video_fd[2] != -1 && fds[i].fd == video_fd[2])) {
					/* Got something video (RTP) */
					int index = -1;
					if(fds[i].fd == video_fd[0])
						index = 0;
					else if(fds[i].fd == video_fd[1])
						index = 1;
					else if(fds[i].fd == video_fd[2])
						index = 2;
					if(mountpoint->active == FALSE)
						mountpoint->active = TRUE;
					source->last_received_video = janus_get_monotonic_time();
#ifdef HAVE_LIBCURL
					source->reconnect_timer = janus_get_monotonic_time();
#endif
					addrlen = sizeof(remote);
					bytes = recvfrom(fds[i].fd, buffer, 1500, 0, (struct sockaddr*)&remote, &addrlen);
					if(bytes < 0) {
						/* Failed to read? */
						continue;
					}
					//~ JANUS_LOG(LOG_VERB, "************************\nGot %d bytes on the video channel...\n", bytes);
					janus_rtp_header *rtp = (janus_rtp_header *)buffer;
					/* First of all, let's check if this is (part of) a keyframe that we may need to save it for future reference */
					if(source->keyframe.enabled) {
						if(source->keyframe.temp_ts > 0 && ntohl(rtp->timestamp) != source->keyframe.temp_ts) {
							/* We received the last part of the keyframe, get rid of the old one and use this from now on */
							JANUS_LOG(LOG_HUGE, "[%s] ... ... last part of keyframe received! ts=%"SCNu32", %d packets\n",
								name, source->keyframe.temp_ts, g_list_length(source->keyframe.temp_keyframe));
							source->keyframe.temp_ts = 0;
							janus_mutex_lock(&source->keyframe.mutex);
							GList *temp = NULL;
							while(source->keyframe.latest_keyframe) {
								temp = g_list_first(source->keyframe.latest_keyframe);
								source->keyframe.latest_keyframe = g_list_remove_link(source->keyframe.latest_keyframe, temp);
								janus_streaming_rtp_relay_packet *pkt = (janus_streaming_rtp_relay_packet *)temp->data;
								g_free(pkt->data);
								g_free(pkt);
								g_list_free(temp);
							}
							source->keyframe.latest_keyframe = source->keyframe.temp_keyframe;
							source->keyframe.temp_keyframe = NULL;
							janus_mutex_unlock(&source->keyframe.mutex);
						} else if(ntohl(rtp->timestamp) == source->keyframe.temp_ts) {
							/* Part of the keyframe we're currently saving, store */
							janus_mutex_lock(&source->keyframe.mutex);
							JANUS_LOG(LOG_HUGE, "[%s] ... other part of keyframe received! ts=%"SCNu32"\n", name, source->keyframe.temp_ts);
							janus_streaming_rtp_relay_packet *pkt = g_malloc0(sizeof(janus_streaming_rtp_relay_packet));
							pkt->data = g_malloc0(bytes);
							memcpy(pkt->data, buffer, bytes);
							pkt->data->ssrc = htons(1);
							pkt->data->type = mountpoint->codecs.video_pt;
							packet.is_rtp = TRUE;
							packet.is_video = TRUE;
							packet.is_keyframe = TRUE;
							pkt->length = bytes;
							pkt->timestamp = source->keyframe.temp_ts;
							pkt->seq_number = ntohs(rtp->seq_number);
							source->keyframe.temp_keyframe = g_list_append(source->keyframe.temp_keyframe, pkt);
							janus_mutex_unlock(&source->keyframe.mutex);
						} else {
							gboolean kf = FALSE;
							/* Parse RTP header first */
							janus_rtp_header *header = (janus_rtp_header *)buffer;
							guint32 timestamp = ntohl(header->timestamp);
							guint16 seq = ntohs(header->seq_number);
							JANUS_LOG(LOG_HUGE, "Checking if packet (size=%d, seq=%"SCNu16", ts=%"SCNu32") is a key frame...\n",
								bytes, seq, timestamp);
							int plen = 0;
							char *payload = janus_rtp_payload(buffer, bytes, &plen);
							if(payload) {
								switch(mountpoint->codecs.video_codec) {
									case JANUS_STREAMING_VP8:
										kf = janus_vp8_is_keyframe(payload, plen);
										break;
									case JANUS_STREAMING_VP9:
										kf = janus_vp9_is_keyframe(payload, plen);
										break;
									case JANUS_STREAMING_H264:
										kf = janus_h264_is_keyframe(payload, plen);
										break;
									default:
										break;
								}
								if(kf) {
									/* New keyframe, start saving it */
									source->keyframe.temp_ts = ntohl(rtp->timestamp);
									JANUS_LOG(LOG_HUGE, "[%s] New keyframe received! ts=%"SCNu32"\n", name, source->keyframe.temp_ts);
									janus_mutex_lock(&source->keyframe.mutex);
									janus_streaming_rtp_relay_packet *pkt = g_malloc0(sizeof(janus_streaming_rtp_relay_packet));
									pkt->data = g_malloc0(bytes);
									memcpy(pkt->data, buffer, bytes);
									pkt->data->ssrc = htons(1);
									pkt->data->type = mountpoint->codecs.video_pt;
									packet.is_rtp = TRUE;
									packet.is_video = TRUE;
									packet.is_keyframe = TRUE;
									pkt->length = bytes;
									pkt->timestamp = source->keyframe.temp_ts;
									pkt->seq_number = ntohs(rtp->seq_number);
									source->keyframe.temp_keyframe = g_list_append(source->keyframe.temp_keyframe, pkt);
									janus_mutex_unlock(&source->keyframe.mutex);
								}
							}
						}
					}
					/* If paused, ignore this packet */
					if(!mountpoint->enabled)
						continue;
					//~ JANUS_LOG(LOG_VERB, " ... parsed RTP packet (ssrc=%u, pt=%u, seq=%u, ts=%u)...\n",
						//~ ntohl(rtp->ssrc), rtp->type, ntohs(rtp->seq_number), ntohl(rtp->timestamp));
					/* Relay on all sessions */
					packet.data = rtp;
					packet.length = bytes;
					packet.is_rtp = TRUE;
					packet.is_video = TRUE;
					packet.is_keyframe = FALSE;
					packet.simulcast = source->simulcast;
					packet.substream = index;
					packet.codec = mountpoint->codecs.video_codec;
					/* Do we have a new stream? */
					if(ntohl(packet.data->ssrc) != v_last_ssrc[index]) {
						v_last_ssrc[index] = ntohl(packet.data->ssrc);
						JANUS_LOG(LOG_INFO, "[%s] New video stream! (ssrc=%u, index %d)\n", name, v_last_ssrc[index], index);
					}
					packet.data->type = mountpoint->codecs.video_pt;
					/* Is there a recorder? (FIXME notice we only record the first substream, if simulcasting) */
					janus_rtp_header_update(packet.data, &source->context[index], TRUE, 4500);
					if(index == 0) {
						ssrc = ntohl(packet.data->ssrc);
						packet.data->ssrc = ntohl((uint32_t)mountpoint->id);
						janus_recorder_save_frame(source->vrc, buffer, bytes);
						packet.data->ssrc = ntohl(ssrc);
					}
					/* Backup the actual timestamp and sequence number set by the restreamer, in case switching is involved */
					packet.timestamp = ntohl(packet.data->timestamp);
					packet.seq_number = ntohs(packet.data->seq_number);
					/* Go! */
					janus_mutex_lock(&mountpoint->mutex);
					g_list_foreach(mountpoint->listeners, janus_streaming_relay_rtp_packet, &packet);
					janus_mutex_unlock(&mountpoint->mutex);
					continue;
				} else if(data_fd != -1 && fds[i].fd == data_fd) {
					/* Got something data (text) */
					if(mountpoint->active == FALSE)
						mountpoint->active = TRUE;
					source->last_received_data = janus_get_monotonic_time();
#ifdef HAVE_LIBCURL
					source->reconnect_timer = janus_get_monotonic_time();
#endif
					addrlen = sizeof(remote);
					bytes = recvfrom(data_fd, buffer, 1500, 0, (struct sockaddr*)&remote, &addrlen);
					if(bytes < 0) {
						/* Failed to read? */
						continue;
					}
					/* Get a string out of the data */
					char *text = g_malloc0(bytes+1);
					memcpy(text, buffer, bytes);
					*(text+bytes) = '\0';
					/* Relay on all sessions */
					packet.data = (janus_rtp_header *)text;
					packet.length = bytes+1;
					packet.is_rtp = FALSE;
					/* Is there a recorder? */
					janus_recorder_save_frame(source->drc, text, strlen(text));
					/* Are we keeping track of the last message being relayed? */
					if(source->buffermsg) {
						janus_mutex_lock(&source->buffermsg_mutex);
						janus_streaming_rtp_relay_packet *pkt = g_malloc0(sizeof(janus_streaming_rtp_relay_packet));
						pkt->data = g_malloc0(bytes+1);
						memcpy(pkt->data, text, bytes+1);
						packet.is_rtp = FALSE;
						pkt->length = bytes+1;
						janus_mutex_unlock(&source->buffermsg_mutex);
					}
					/* Go! */
					janus_mutex_lock(&mountpoint->mutex);
					g_list_foreach(mountpoint->listeners, janus_streaming_relay_rtp_packet, &packet);
					janus_mutex_unlock(&mountpoint->mutex);
					packet.data = NULL;
					g_free(text);
					continue;
				}
			}
		}
	}

	/* Notify users this mountpoint is done */
	janus_mutex_lock(&mountpoint->mutex);
	GList *viewer = g_list_first(mountpoint->listeners);
	/* Prepare JSON event */
	json_t *event = json_object();
	json_object_set_new(event, "streaming", json_string("event"));
	json_t *result = json_object();
	json_object_set_new(result, "status", json_string("stopped"));
	json_object_set_new(event, "result", result);
	while(viewer) {
		janus_streaming_session *session = (janus_streaming_session *)viewer->data;
		if(session != NULL) {
			session->stopping = TRUE;
			session->started = FALSE;
			session->paused = FALSE;
			session->mountpoint = NULL;
			/* Tell the core to tear down the PeerConnection, hangup_media will do the rest */
			gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
			gateway->close_pc(session->handle);
		}
		mountpoint->listeners = g_list_remove_all(mountpoint->listeners, session);
		viewer = g_list_first(mountpoint->listeners);
	}
	json_decref(event);
	janus_mutex_unlock(&mountpoint->mutex);

	JANUS_LOG(LOG_VERB, "[%s] Leaving streaming relay thread\n", name);
	g_free(name);
	g_thread_unref(g_thread_self());
	return NULL;
}

static void janus_streaming_relay_rtp_packet(gpointer data, gpointer user_data) {
	janus_streaming_rtp_relay_packet *packet = (janus_streaming_rtp_relay_packet *)user_data;
	if(!packet || !packet->data || packet->length < 1) {
		JANUS_LOG(LOG_ERR, "Invalid packet...\n");
		return;
	}
	janus_streaming_session *session = (janus_streaming_session *)data;
	if(!session || !session->handle) {
		//~ JANUS_LOG(LOG_ERR, "Invalid session...\n");
		return;
	}
	if(!packet->is_keyframe && (!session->started || session->paused)) {
		//~ JANUS_LOG(LOG_ERR, "Streaming not started yet for this session...\n");
		return;
	}

	if(packet->is_rtp) {
		/* Make sure there hasn't been a publisher switch by checking the SSRC */
		if(packet->is_video) {
			if(!session->video)
				return;
			if(packet->simulcast) {
				/* Handle simulcast: don't relay if it's not the substream we wanted to handle */
				int plen = 0;
				char *payload = janus_rtp_payload((char *)packet->data, packet->length, &plen);
				if(payload == NULL)
					return;
				gboolean switched = FALSE;
				if(session->substream != session->substream_target) {
					/* There has been a change: let's wait for a keyframe on the target */
					int step = (session->substream < 1 && session->substream_target == 2);
					if(packet->substream == session->substream_target || (step && packet->substream == step)) {
						if(janus_vp8_is_keyframe(payload, plen)) {
							JANUS_LOG(LOG_VERB, "Received keyframe on substream %d, switching (was %d)\n",
								packet->substream, session->substream);
							session->substream = packet->substream;
							switched = TRUE;
							/* Notify the viewer */
							json_t *event = json_object();
							json_object_set_new(event, "streaming", json_string("event"));
							json_t *result = json_object();
							json_object_set_new(result, "substream", json_integer(session->substream));
							json_object_set_new(event, "result", result);
							gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
							json_decref(event);
						//~ } else {
							//~ JANUS_LOG(LOG_WARN, "Not a keyframe on SSRC %"SCNu32" yet, waiting before switching\n", ssrc);
						}
					}
				}
				/* If we haven't received our desired substream yet, let's drop temporarily */
				if(session->last_relayed == 0) {
					/* Let's start slow */
					session->last_relayed = janus_get_monotonic_time();
				} else {
					/* Check if 250ms went by with no packet relayed */
					gint64 now = janus_get_monotonic_time();
					if(now-session->last_relayed >= 250000) {
						session->last_relayed = now;
						int substream = session->substream-1;
						if(substream < 0)
							substream = 0;
						if(session->substream != substream) {
							JANUS_LOG(LOG_WARN, "No packet received on substream %d for a while, falling back to %d\n",
								session->substream, substream);
							session->substream = substream;
							/* Notify the viewer */
							json_t *event = json_object();
							json_object_set_new(event, "streaming", json_string("event"));
							json_t *result = json_object();
							json_object_set_new(result, "substream", json_integer(session->substream));
							json_object_set_new(event, "result", result);
							gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
							json_decref(event);
						}
					}
				}
				if(packet->substream != session->substream) {
					JANUS_LOG(LOG_HUGE, "Dropping packet (it's from substream %d, but we're only relaying substream %d now\n",
						packet->substream, session->substream);
					return;
				}
				session->last_relayed = janus_get_monotonic_time();
				char vp8pd[6];
				if(packet->codec == JANUS_STREAMING_VP8) {
					/* Check if there's any temporal scalability to take into account */
					uint16_t picid = 0;
					uint8_t tlzi = 0;
					uint8_t tid = 0;
					uint8_t ybit = 0;
					uint8_t keyidx = 0;
					if(janus_vp8_parse_descriptor(payload, plen, &picid, &tlzi, &tid, &ybit, &keyidx) == 0) {
						//~ JANUS_LOG(LOG_WARN, "%"SCNu16", %u, %u, %u, %u\n", picid, tlzi, tid, ybit, keyidx);
						if(session->templayer != session->templayer_target) {
							/* FIXME We should be smarter in deciding when to switch */
							session->templayer = session->templayer_target;
								/* Notify the viewer */
								json_t *event = json_object();
								json_object_set_new(event, "streaming", json_string("event"));
								json_t *result = json_object();
								json_object_set_new(result, "temporal", json_integer(session->templayer));
								json_object_set_new(event, "result", result);
								gateway->push_event(session->handle, &janus_streaming_plugin, NULL, event, NULL);
								json_decref(event);
						}
						if(tid > session->templayer) {
							JANUS_LOG(LOG_HUGE, "Dropping packet (it's temporal layer %d, but we're capping at %d)\n",
								tid, session->templayer);
							/* We increase the base sequence number, or there will be gaps when delivering later */
							session->context.v_base_seq++;
							return;
						}
					}
					/* If we got here, update the RTP header and send the packet */
					janus_rtp_header_update(packet->data, &session->context, TRUE, 4500);
					memcpy(vp8pd, payload, sizeof(vp8pd));
					janus_vp8_simulcast_descriptor_update(payload, plen, &session->simulcast_context, switched);
				}
				/* Send the packet */
				if(gateway != NULL)
					gateway->relay_rtp(session->handle, packet->is_video, (char *)packet->data, packet->length);
				/* Restore the timestamp and sequence number to what the publisher set them to */
				packet->data->timestamp = htonl(packet->timestamp);
				packet->data->seq_number = htons(packet->seq_number);
				if(packet->codec == JANUS_STREAMING_VP8) {
					/* Restore the original payload descriptor as well, as it will be needed by the next viewer */
					memcpy(payload, vp8pd, sizeof(vp8pd));
				}
			} else {
				/* Fix sequence number and timestamp (switching may be involved) */
				janus_rtp_header_update(packet->data, &session->context, TRUE, 4500);
				if(gateway != NULL)
					gateway->relay_rtp(session->handle, packet->is_video, (char *)packet->data, packet->length);
				/* Restore the timestamp and sequence number to what the publisher set them to */
				packet->data->timestamp = htonl(packet->timestamp);
				packet->data->seq_number = htons(packet->seq_number);
			}
		} else {
			if(!session->audio)
				return;
			/* Fix sequence number and timestamp (switching may be involved) */
			janus_rtp_header_update(packet->data, &session->context, FALSE, 960);
			if(gateway != NULL)
				gateway->relay_rtp(session->handle, packet->is_video, (char *)packet->data, packet->length);
			/* Restore the timestamp and sequence number to what the publisher set them to */
			packet->data->timestamp = htonl(packet->timestamp);
			packet->data->seq_number = htons(packet->seq_number);
		}
	} else {
		/* We're broadcasting a data channel message */
		if(!session->data)
			return;
		char *text = (char *)packet->data;
		if(gateway != NULL && text != NULL)
			gateway->relay_data(session->handle, text, strlen(text));
	}

	return;
}
