/*************************************************************************
* Copyright (C) 2021 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

#ifndef IFMA_ECPRECOMP4_P384_H
#define IFMA_ECPRECOMP4_P384_H

#include "owndefs.h"

#if (_IPP32E >= _IPP32E_K1)

#include "gfpec/pcpgfpecstuff.h"

#include "gfpec/ecnist/ifma_defs.h"
#include "gfpec/ecnist/ifma_ecpoint_p384.h"

#define BASE_POINT_WIN_SIZE (4)
#define BASE_POINT_N_ENTRY (1 << ((BASE_POINT_WIN_SIZE)-1))

#define OPERAND_BITSIZE (384)
#define LEN52_P384 (NUMBER_OF_DIGITS(OPERAND_BITSIZE, DIGIT_SIZE))

/* P384 affine point */
typedef struct {
   BNU_CHUNK_T X[LEN52_P384];
   BNU_CHUNK_T Y[LEN52_P384];
} P384_POINT_AFFINE_IFMA_MEM;

extern const __ALIGN64 P384_POINT_AFFINE_IFMA_MEM ifma_ec_nistp384r1_bp_precomp[97][BASE_POINT_N_ENTRY];

#if !defined(_DISABLE_ECP_384R1_HARDCODED_BP_TBL_)

const __ALIGN64 P384_POINT_AFFINE_IFMA_MEM ifma_ec_nistp384r1_bp_precomp[][BASE_POINT_N_ENTRY] = {
    {
        /* digit=0 [{1,2,3,..,}]*([2^0]*G)  */
        {{0x000607664d3aadc2, 0x000fa3dd07565fc8, 0x000e1e26a4ee117b, 0x0003afc541b4d6e6, 0x000459a30eff879c, 0x0004ede2b6454868, 0x000513812ff72361, 0x00000000000299e1}, {0x000af93c2b78abc2, 0x0006e23043dad1f8, 0x000d385481a72d55, 0x000e7562e83b050c, 0x000968f4ffd98bad, 0x00069a840c6c3521, 0x0005e9dd80022639, 0x000000000005a15c}},
        {{0x000a271bdb93b776, 0x00066c8229e549ca, 0x000a0046a4ddbf0b, 0x000e6f0ff9d48a26, 0x0005f0687f503504, 0x000e4b506da82149, 0x0000c39c90a4fd2d, 0x0000000000042746}, {0x000777e3e34947f7, 0x000cf42ea84624df, 0x000c322ca0a5f414, 0x000f18bdc588259c, 0x00015172bad915e4, 0x000b0e68409f1fe4, 0x0000c2070d430900, 0x00000000000123df}},
        {{0x0008420bd283fe68, 0x0000405e4dbe5ef5, 0x0007d2a868c2d376, 0x00034e9a170ccf19, 0x0002d51c6c3e6b20, 0x0003aa4703a48d73, 0x0003ace36f7e2d26, 0x00000000000e7c1c}, {0x000a7b5b465465fc, 0x000697e28482179f, 0x00092befa3c13549, 0x00063c04ef67446d, 0x000ad2e1d0b41326, 0x00002b33968012d5, 0x0002aff6db68b151, 0x0000000000098329}},
        {{0x0007a84fcfeee6dd, 0x000c00aae84771bc, 0x00004833a9bdf308, 0x000153b0aecac470, 0x0004736400ad2e4f, 0x00085d979078358d, 0x000228fb40f647d6, 0x0000000000034179}, {0x00059b3d50946875, 0x000f354f3e8e74aa, 0x0007e02066cc9331, 0x00061a34c542ad23, 0x00030418c6938e3e, 0x00020017d147162d, 0x000319e607b9e338, 0x00000000000303df}},
        {{0x0006ca2ee1bb26b1, 0x00017bb595eb9873, 0x000340e77dae425b, 0x000b1b5293c703ca, 0x0005aacc05e67f1e, 0x000e8e4c660db2cf, 0x000ffbc676b987e2, 0x000000000001d178}, {0x0002304b4db1c9d6, 0x0003c2b694ba022c, 0x0000733804c0a50f, 0x0001b3101c35b997, 0x000f982c041180b6, 0x000de236d4b237fa, 0x0004a3e6c5944024, 0x00000000000e209e}},
        {{0x0003f94fc482e189, 0x000c37eb5c930b8d, 0x000fa7363cfe5622, 0x000930f580d57f38, 0x00061bdf6015ec52, 0x00002d33b2a33f66, 0x000c404f0f6a962b, 0x00000000000f0430}, {0x000a60b1c9962152, 0x000203f62b16dde1, 0x000d30e7f024d36f, 0x000bffcb79e33b13, 0x00061b546f058bd4, 0x00021559a93c9e5f, 0x000eba586d8ededf, 0x00000000000af2a9}},
        {{0x000c82aa932c9f81, 0x00032df13b9d0ea3, 0x000012e0e11a7414, 0x000dcf8211faa95e, 0x0001659753ed731e, 0x000b2df555f4215d, 0x00025bf893db589d, 0x000000000001c890}, {0x000c8f68d90a16b6, 0x0002f0996b21f9df, 0x000c5d608c816e33, 0x000d76f50130844e, 0x000401fff78065aa, 0x0003b07060ff37c0, 0x000b3ef57f238e69, 0x00000000000af6c9}},
        {{0x0006da365db4f184, 0x000fe23f60a057fe, 0x000be85f6a0c5049, 0x0002e7193d7e30ff, 0x00064f3ddb371c5c, 0x000b664231d9aebd, 0x0009b11c7b5fe116, 0x00000000000349cd}, {0x0008ec6d3c0c6dd7, 0x0005e0d2cfe83aa5, 0x000f7a290df3f1cc, 0x00054bf7d8686e4b, 0x0003a42dbba27017, 0x0008ecf0ee992326, 0x000f617008d943c4, 0x000000000000d27e}},
    },
    {
        /* digit=1 [{1,2,3,..,}]*([2^4]*G)  */
        {{0x000b56e7a10896aa, 0x00082da6e8a7edb2, 0x000a339205afd669, 0x00065517917652b3, 0x000a2887d5ff37cf, 0x000bdc3fa317b63e, 0x000aa137065f5313, 0x00000000000435ab}, {0x000e15a5659db481, 0x0008e9b21615f8a0, 0x0000a5926b88aaa4, 0x00071dc154d41105, 0x000bc88ee1489148, 0x0002d1967b333bdf, 0x00051351c305c6a7, 0x0000000000081ef2}},
        {{0x0007c0045dd2a4a0, 0x00075ad852b872e3, 0x000b4af19266ca1e, 0x000b9fcc651b1dd6, 0x000612e8dc871896, 0x00031cfb0ba8953d, 0x000d793a9865baa6, 0x00000000000626b3}, {0x000b7328c510ad93, 0x000901148bc71a36, 0x0008838d56b7b5d9, 0x0005f9e9448fd096, 0x00000a2377b67731, 0x0005b4ff04bcb06f, 0x00099f73b42725b3, 0x00000000000aabca}},
        {{0x00077d8466e4794b, 0x000763ccf806a4c7, 0x00041b06944ed785, 0x0004e06ea52bef99, 0x00053a7d2f3c4f50, 0x0003a1bc940d01ff, 0x00040062e5c5d3e7, 0x00000000000ae7d7}, {0x0000b071271e42a1, 0x000923d30625e38e, 0x000c9ea33ece8520, 0x000a10d04bab9856, 0x00009da2a2ca5c3c, 0x000c9462c2605ca0, 0x00058348eab00eb5, 0x000000000002624a}},
        {{0x000c57a24f5d5ab6, 0x000eded4de4f83f2, 0x0004d9f2c578d7f4, 0x0003d30f8a0580de, 0x000dca57b7bde04b, 0x000e44d56a309199, 0x000e5cfc8e87cf3b, 0x00000000000d1b30}, {0x000dc1d0888b708b, 0x0008cb213c69fa81, 0x00085d35b9791d2c, 0x000bbbf1090fede9, 0x000c301fe259fe51, 0x000cd3fe86d97cab, 0x000a513ee127895e, 0x000000000009404c}},
        {{0x0004d8911e8568cc, 0x000c5194924b48e8, 0x00026b2f852cc83a, 0x0006428b12136094, 0x000351fea1dc4906, 0x00015ace6dd2ec6d, 0x00024620fe8c27a7, 0x00000000000a4463}, {0x0003c328530abb42, 0x000b900c213bba9c, 0x000bf43a5f2c2e1e, 0x000903c6484641de, 0x000a1378e68fbc7a, 0x000cd8ae42413063, 0x0006f9b960b5efee, 0x000000000000614a}},
        {{0x0001dedb8bf3dccc, 0x000a0dc529384912, 0x000bc9fafda07c1d, 0x000597e52ce08f71, 0x000581998af2ee21, 0x00041eb4226de4c6, 0x000b96cb4aa43c97, 0x0000000000039c18}, {0x000a9ce2b257fb6c, 0x000a566e1d5da261, 0x0002d61f72303077, 0x000396f305ee4f10, 0x0000c831254b8545, 0x0005680b8f9d19ea, 0x0004a4cee0842f5d, 0x000000000005a443}},
        {{0x00007168b4a67147, 0x000a12e206547853, 0x0001c6a852120cb6, 0x00009d5504c8129c, 0x000c9710b70b2b56, 0x000296a52fb25e37, 0x000dce83f2fd2cd8, 0x0000000000062f45}, {0x0000b128f82bb944, 0x000a8a818b9bda93, 0x000ed2d611039805, 0x000e43a2ec76a180, 0x0007caa846883e7e, 0x000182141473e687, 0x00004db9a19eb57c, 0x0000000000045ed2}},
        {{0x0001e64ba6661cc4, 0x000cee484fd9edbf, 0x0005b5c2b4988114, 0x000449e7c1c3984b, 0x000118eb5195c0dd, 0x0007a16d2f313389, 0x00020a0336aab877, 0x00000000000c2417}, {0x0001e21e239dcab2, 0x00034678db970845, 0x000627331787ffae, 0x000a0e4a022c7a44, 0x000434a02a6b5d85, 0x000791ce3b01f1e0, 0x000c5b2657eedda5, 0x00000000000a277a}},
    },
    {
        /* digit=2 [{1,2,3,..,}]*([2^8]*G)  */
        {{0x000fa21fa335ab82, 0x000a49a7a5b41c7a, 0x000300862e13765b, 0x000438e3d9f0e627, 0x0009e328c2e27539, 0x000cbf891013c671, 0x000d287f4a706ccc, 0x00000000000735a2}, {0x000a7119424dd00b, 0x0004246694eeffb4, 0x000059afb703b483, 0x000ed8b423d47e45, 0x000bf44c91809d54, 0x000e9b3848075a8c, 0x000c75d4f5b184ab, 0x0000000000041abd}},
        {{0x00093e732cc6e06c, 0x000c65e2cb07faab, 0x0006c10c7767a2e6, 0x000c53fd4de1f262, 0x000c838f7169a296, 0x0008a6ce7d408060, 0x00067168e19d7b2e, 0x0000000000094b58}, {0x000136755dca2adc, 0x0000293d02a07640, 0x000ed9dfab92ca5c, 0x00069f51aa3bc4ef, 0x0000dd09b1426aa0, 0x0002e59450e44fbf, 0x0006ace264f34383, 0x000000000001fc16}},
        {{0x0001b41eba2511e6, 0x0003e9ee4f521f6d, 0x0005af7a840c9880, 0x000396db7edb07d1, 0x000c2e8290630d5f, 0x0003495da09b3457, 0x0009b8f1d28188f8, 0x00000000000cce55}, {0x000f6b035c499b66, 0x0005617cbaf577ca, 0x0007eb3582ad9848, 0x00008995145b8fd9, 0x00081a33b1a72982, 0x0005149e992cb5da, 0x0004c0ca49fe334c, 0x000000000001772b}},
        {{0x000b80038e0f9767, 0x0006756ad758212d, 0x00066af19dfc2941, 0x000c6ffe2c8b0369, 0x0007fcd7336b85f2, 0x000a46acd55c6d35, 0x000ac7b1ecc56d22, 0x0000000000036277}, {0x000330b02f145871, 0x000c1a4ed11e8d27, 0x000297add7ae640e, 0x000ba45266158ab0, 0x000d89e0dff05fda, 0x0006b02d06f0b27f, 0x0006e132ef7ae2eb, 0x00000000000cc1b4}},
        {{0x0008162061985fbc, 0x00005c112733b3ba, 0x00062ae17de90bd5, 0x0008e01810097859, 0x0002bfe16c4fbb7d, 0x000d9f8107640a3e, 0x0005d74e34813ec1, 0x000000000008d260}, {0x00078cdfc58ed763, 0x000f72a544cd81e6, 0x000e167259300b75, 0x00057bacec18a7f0, 0x000511b882d69e61, 0x000f86563a555fc9, 0x00096e4305a4dd04, 0x000000000001d0fd}},
        {{0x00008da90a96090a, 0x00032f04145e8229, 0x000a916fb6ff9132, 0x000ba4e12aa299fb, 0x000991b3b5179ffc, 0x00081c747cc5ec24, 0x0003eb9edcd4616f, 0x0000000000077a88}, {0x000a4909883002a4, 0x0008b9b0bab581a7, 0x000e659d0317cb87, 0x000d81e438a9d43f, 0x000e25ca8b3cfe8c, 0x000bc720cf40e2b5, 0x0006a34254030067, 0x000000000006b244}},
        {{0x00099b58a43c6d42, 0x0005180e1cd16205, 0x000e96620312fe6d, 0x00019d509ddce071, 0x000e70c4b03267a0, 0x0003ba57e52573e3, 0x0004f716d253e14d, 0x0000000000016250}, {0x0003e944594baca0, 0x00013a237bbf8f9b, 0x000a642b05f4171f, 0x000531a1f384daed, 0x0003981251654b13, 0x0002dccc139067f3, 0x0007b5e98fb14167, 0x00000000000a75e2}},
        {{0x0009f542630002ea, 0x00044e65245ce93e, 0x00012350ea59da7e, 0x000c121bad2c8070, 0x0002060fcf245677, 0x00078cccac52dec3, 0x0006fb78d070675d, 0x000000000001bc8b}, {0x000ac9684403d046, 0x000b5c5cb86bea72, 0x00053d522dc955a1, 0x000cdf2c92a70d83, 0x0001f53cd2a1fbb7, 0x0004f11395a9ff1f, 0x0009f1fdbe6b7a98, 0x00000000000a470a}},
    },
    {
        /* digit=3 [{1,2,3,..,}]*([2^12]*G)  */
        {{0x0006766eb19e084c, 0x00028eb06571b5db, 0x000430cbda13e4c9, 0x000966726eed225a, 0x00046100e387a185, 0x0006298d18d9e56e, 0x000ad0470506b9dc, 0x00000000000f3350}, {0x0009595e79f27f3a, 0x0006683eb62a798b, 0x000ae3d2069c14b3, 0x000e880e1bd4a82e, 0x000fcaf3b3fcb089, 0x000ffd65cd4d1e70, 0x000ebbd0b1ec6395, 0x000000000009b184}},
        {{0x000d72326a677bdc, 0x0001bd4277730e1a, 0x0004e8c2adc8ef98, 0x00046b099f1867d4, 0x0002602dd4cc6b07, 0x0000811201ec73d7, 0x000f2d27fae51538, 0x000000000002f8b2}, {0x000e28e4b1971c05, 0x0001bb924af64246, 0x0005d0fd898e9387, 0x000e9ae068565acc, 0x0005a9a4f1464e88, 0x00093f7348a3dbd2, 0x0003bcdb4a3e483f, 0x000000000008f1d3}},
        {{0x000da9f02128c46a, 0x00049d1de964bafd, 0x0007f571d595c1ce, 0x00055af0de9eb074, 0x0005a60289bfbc4b, 0x000392c619f11b99, 0x0004fc3e59000c52, 0x000000000005ccff}, {0x000c017748720be7, 0x00064b28b306ba1a, 0x0007e101bd3e41b0, 0x000542ce3f824faa, 0x00022f52b71c59b0, 0x000c6d26370f097b, 0x000e5b4483b72604, 0x0000000000034d93}},
        {{0x000b2e0b9f0415b8, 0x000c7721bb8359b7, 0x000a5f46c16031df, 0x000a789348538714, 0x0007af598c4f9cc9, 0x0006f27c878b604a, 0x000ba5d370375e47, 0x000000000000b15c}, {0x00021b9613cec089, 0x000662bcfd9a4c03, 0x000e3ea0c45f94ee, 0x0006464a211b19f3, 0x00019990b504b05a, 0x0004951d3ce059d4, 0x0007b0011c5f87d9, 0x0000000000000d9c}},
        {{0x0002533a1c8fbed1, 0x000ce64e84c28804, 0x000338cbe4f167c9, 0x000d9ed9fbf23cc9, 0x000f5b93118bb77e, 0x0006255cf155fd45, 0x0008941e9f6d7d9c, 0x00000000000c4f64}, {0x00008205c725e2b1, 0x000154bc3a502a87, 0x00030c3fbf39b6ac, 0x0005548d3c862428, 0x0004030f713cc7df, 0x000785cbf9dbfc08, 0x000637f3623326ad, 0x00000000000dd3ad}},
        {{0x00053a3eba12bfa4, 0x000ced8b8b37a274, 0x000ff25533a7ef36, 0x000684bd17d58a93, 0x0002032fda408ac7, 0x0004b49645f9557e, 0x0001097fe128e6ed, 0x00000000000ed02e}, {0x000f765f56c35dab, 0x0008c0052d88eb68, 0x000256931b154329, 0x00010798446a4f6a, 0x0007c99ad35fbf46, 0x0000906073bc4391, 0x0008aada18234dda, 0x000000000005164f}},
        {{0x000f715095892612, 0x0004e02c16c7865e, 0x0009e82bb73222b5, 0x000bbc0795486af0, 0x00070427332d3abe, 0x0005d3cabad858cc, 0x00019c9a1d4b6aa3, 0x00000000000e208c}, {0x000d5b54420318c5, 0x0000afcc14276eea, 0x0008e6c4a86b5358, 0x0007cb4e7706b5bf, 0x000e479e2c750027, 0x0007ad688c01ed42, 0x000626ff1759604d, 0x00000000000c045f}},
        {{0x000e0b227c3a04b7, 0x00029f365419f1ee, 0x00001db5dec2705b, 0x000c165c41880aa9, 0x0007f9712fbd8a91, 0x000c556783eb27a9, 0x0009cfa6587aec76, 0x0000000000002cd7}, {0x000e78bc85d2b5fc, 0x000fefc878f9c549, 0x000d411713959cf3, 0x00084d8caf6df5e8, 0x0002aabcde7509a1, 0x000de597ad32bf23, 0x000858f601d0de03, 0x00000000000c5da2}},
    },
    {
        /* digit=4 [{1,2,3,..,}]*([2^16]*G)  */
        {{0x000f068a28ea9470, 0x0008bb6029961859, 0x0007d86ade910602, 0x000693b4df3e5b1b, 0x0008a0c3e35782db, 0x000b2f577b513148, 0x000cc3bfb01ff3cc, 0x0000000000027a72}, {0x0000fdf0e7cd346e, 0x000f626170927fbd, 0x000aa1bbda6cc535, 0x0006a634c872d772, 0x0000b14d9c9f0bec, 0x0006c7778a0a7cc9, 0x000a4c8a32d2c44f, 0x000000000003b889}},
        {{0x000f462aea173d82, 0x000a4860ef793767, 0x000a7a5856850902, 0x00083662ee7f523d, 0x000f54122af0322c, 0x000bb2d8058ccd95, 0x0005777454880c2b, 0x0000000000086d8d}, {0x000038487c4c8fb9, 0x00042d5a3057c6dd, 0x000955643c37ff31, 0x000c99ec3c512f97, 0x0006556d891e26aa, 0x0009f6112c3eac03, 0x0007e9866c3aa7bf, 0x00000000000c144f}},
        {{0x00000b161de71555, 0x000aead0d24c7983, 0x0006a55d94bbb854, 0x00034ff7655aa29d, 0x00057a5e217ea551, 0x00021b295a3d1038, 0x00036dfbb9eeb53a, 0x00000000000c84a1}, {0x000aac3258d9db81, 0x000087579398db29, 0x000fa470f6fa27aa, 0x0002e1e464522581, 0x0005479d8f2c99b3, 0x0000b80ef99d5495, 0x00050bc2a8a6a193, 0x00000000000656c8}},
        {{0x000f81f2532800c4, 0x00045171898aa3c4, 0x000ea2712f9cc33b, 0x000835ffdb2c1bad, 0x0001591f5aafbc0e, 0x000272c6a4ee3028, 0x00068afd71de3bcc, 0x000000000006e93f}, {0x000145dbf5847f9b, 0x000bc35ee08038de, 0x0001b04c30c2d081, 0x0007ff5957b2ff76, 0x000e5ec029c049f3, 0x000324c12315d8e7, 0x00057833230602ef, 0x00000000000966b2}},
        {{0x0003e43e11c6c113, 0x0002e86283e21e81, 0x0009c3e50b313030, 0x0009b1bb9784a9a5, 0x000ea0c0acb57d02, 0x0007c682b7c7798b, 0x0002b041241c716d, 0x000000000001d33c}, {0x00079a15b39e351d, 0x000dc5d469ca181c, 0x000ce825406e72f2, 0x0004cc2a13cf4ce5, 0x00069e3ce2793d05, 0x0004beafc13bd216, 0x00087e01bc70e68a, 0x000000000008aba0}},
        {{0x0009cf16a3c4418b, 0x0005884aa863e012, 0x00089c47322b55de, 0x0003206b5c399b2c, 0x00073cc109bd553f, 0x000384088775b921, 0x0003cf25c01263fe, 0x00000000000d5f74}, {0x00057c2efedc75c2, 0x000933d69705ce0b, 0x000359bbe99d9a50, 0x000ab1a2626cebe5, 0x000285b1afe80198, 0x0007e6efdaf8320f, 0x000bb6b9c0968ce1, 0x0000000000090215}},
        {{0x00066543cf4fd691, 0x0000d3ee52d8e909, 0x00094816ee49cd7e, 0x00095c61881a757d, 0x0009c13e370735ce, 0x000d2d3f60a8cf9d, 0x000c0de71258d548, 0x000000000000bbe7}, {0x000476d4cb00031d, 0x000bfbd6496e1309, 0x000c1c69b8768cb6, 0x000501358cfdfb53, 0x000b59275b4acbe8, 0x000f722ba655c902, 0x000aad7e0ff05b20, 0x0000000000042b17}},
        {{0x000b76411fcb09e7, 0x00066da643272cd3, 0x0006802b1cc8eac2, 0x0005a7c35b43943a, 0x00084606bbf22386, 0x00059fb9a6ac0158, 0x0001a59660ab7215, 0x000000000003ce2f}, {0x00083d9ad8b4f172, 0x0006e62af29aaa08, 0x00060fa06813a370, 0x00029b744c110388, 0x0001d36c2571ee9f, 0x000b552b7b2a19cf, 0x0003d4b87d88e265, 0x00000000000beb25}},
    },
    {
        /* digit=5 [{1,2,3,..,}]*([2^20]*G)  */
        {{0x0002b4a5d3a4b643, 0x0000231bdb4829ee, 0x0006d713dc69c388, 0x00042469b6fc64eb, 0x000a15786b8c18c0, 0x00063da4a0dcc15f, 0x0000fb81acdb068e, 0x00000000000dada7}, {0x0008c1ca45ab6321, 0x0009146361afe98f, 0x0001d88fcfcf648c, 0x000b61b615694e72, 0x0001872951d2f389, 0x0003badc77036def, 0x0008d340fd3bdad9, 0x00000000000380a6}},
        {{0x000d66b2c4fdf4c0, 0x0007ac5cf8997090, 0x000a08d2a2626f49, 0x000681e307b254af, 0x000775cd94d78139, 0x000684954cc87fb5, 0x00099190e7027478, 0x0000000000095ceb}, {0x0004649153ee63bb, 0x0006891bbc0ab337, 0x0001f845e221f84c, 0x0003d704b93d45fb, 0x0004f541da1f1cb8, 0x0007ffd10e229902, 0x000b9a3eef7ce14b, 0x00000000000b3fc1}},
        {{0x00067cf1182bf349, 0x0007f23c1f744697, 0x000288faa5d1b184, 0x0002c9d5afd1bfd3, 0x0004f89d76fea4f8, 0x000702f80a3d1e9a, 0x00089d964f705075, 0x00000000000f56c0}, {0x000478c2e0092e1f, 0x00012e205fa8ede0, 0x0002998cd8ea4a27, 0x0001004356d43961, 0x0001fdbbdfde331a, 0x000a00e7158b7890, 0x000a30a52a15979b, 0x0000000000055497}},
        {{0x000d05d101bb2988, 0x00097e5004e8001a, 0x000e96c63ff7fdd5, 0x00050d1bd869f77c, 0x000de7ebea2c405f, 0x0007620baecffa54, 0x000ff43354dc22d8, 0x00000000000b1c01}, {0x000d9286dd26e577, 0x000c2d9370076043, 0x00025722a20b989f, 0x00076273291e5c62, 0x0007f0a7ca55c0de, 0x000592a305cfebd8, 0x000ce4de1162809e, 0x00000000000a78eb}},
        {{0x000153343d6d3c05, 0x000a15562a856338, 0x00041dfd1ca25266, 0x000317409c75b8cc, 0x000124923f80c19f, 0x0005b291e21c7cc3, 0x000b05e63fe47f8f, 0x000000000000dc08}, {0x000a81ce4831b7d3, 0x0001558ae455ea5d, 0x0000f05c04530b31, 0x000c97f3ee3989cc, 0x0005f38759ae42c7, 0x000f46899c1b03af, 0x000c7d1a673c75bc, 0x000000000008d508}},
        {{0x000fc69642676f12, 0x0008d1e9b23b9bca, 0x000626ac6d6d75ba, 0x00000fe59b7721d0, 0x000c9e2f4cebd4cc, 0x0000af70ed5c36f9, 0x000799deca06bac9, 0x00000000000416ee}, {0x000affe525098c8a, 0x000df0d7afe1b4a6, 0x000083fa6f5ecd29, 0x0003d6ee6eaed183, 0x0002496087e011e4, 0x000a3a66e5baf860, 0x000677f833634fb1, 0x0000000000079398}},
        {{0x000c39f67e66a95e, 0x0005b76512d1b46e, 0x0009e5603da596ca, 0x0003aa8025a5f25a, 0x00095cbd2a40d1c7, 0x0008d61c62aba192, 0x0000f3b53cadc3c8, 0x0000000000009829}, {0x000894e8b1d3bb8b, 0x0003a72800ecafd7, 0x0003c62dea4f99fb, 0x00092a7282ba9d23, 0x0002bd5f1bb214bf, 0x0007c6c969062967, 0x000601362f68eba9, 0x000000000007ea9d}},
        {{0x0004d04ff62d3721, 0x00071e141e762de4, 0x000fa0d592d3e0eb, 0x000cde496131447a, 0x0005cb6c2ef746e6, 0x0002f9fd80458a5d, 0x000457b774763453, 0x0000000000016544}, {0x000adb5cae252cf8, 0x0005abfacb4de24e, 0x000e9db72a61c26c, 0x0003220f22d92e51, 0x0006557232589b54, 0x000ddb8bcaa4590f, 0x0007bdc7b6730e01, 0x0000000000069e1e}},
    },
    {
        /* digit=6 [{1,2,3,..,}]*([2^24]*G)  */
        {{0x000e98d9a0583230, 0x000f77d27d71f312, 0x000823b17edc1a45, 0x000afeb6075b2d00, 0x0006d93a06f7418c, 0x000d001b9c0d691e, 0x0007b9c16a95259d, 0x00000000000026f1}, {0x000b72219cfa1dea, 0x000984c1041afdb4, 0x00056e257be49c48, 0x000efd62c1758e9c, 0x0007e6c3229a8d08, 0x0002d89249cc6f20, 0x000fe7ec69e90208, 0x00000000000f331d}},
        {{0x00061c722c01a99c, 0x000518ec4335f7fe, 0x000df3425d366c49, 0x0006de001141ab62, 0x000c2eca98a13ff9, 0x000fdf648b21acc2, 0x000e8b6154849010, 0x00000000000d1403}, {0x00097be7041b8df0, 0x0008ff1a35f306b8, 0x000b8cedd3e80c1c, 0x0006c077fc9a752f, 0x000e420d48a089c2, 0x000fe2e738d2535f, 0x000f3980ec5ddd52, 0x0000000000071704}},
        {{0x000259172095dfcf, 0x000a020aa15d95a2, 0x000d85bd292d185e, 0x00005caef579e8f8, 0x000b325981bfe2f2, 0x000438be8ad27e38, 0x0000d9087b8284c3, 0x0000000000042236}, {0x00091bc7ac277af8, 0x000bb87cdf5accaa, 0x000de0f7da8c4a28, 0x00040c1891046669, 0x000c9c1578e8a712, 0x00050ffa2eb5a175, 0x000a28bd66910ad1, 0x0000000000011459}},
        {{0x000f920077501dce, 0x00091498808ed4f0, 0x000dc6c59ac5d089, 0x00025f176c6b964a, 0x000bac474261796e, 0x000a460c11aced64, 0x000e48a62470fc29, 0x000000000005e751}, {0x000842d2c145f36f, 0x0007acc00053aac5, 0x000ca1b81e5b854d, 0x000cc2e3f9ca178e, 0x000a0b80d1b0ddac, 0x000642225ad33f34, 0x00061b6a76df9364, 0x00000000000778e7}},
        {{0x0009eb69fa5f1bc6, 0x0008ed30302342c1, 0x000e3ef7d69039f0, 0x0009b575c4630f26, 0x00008d098d745364, 0x0007f5cbc60197fc, 0x000efc9c295d5464, 0x00000000000c0813}, {0x0001c5999be2ce7e, 0x000e7f6e08007370, 0x000b6019bc473a63, 0x000e08e11d9b388e, 0x000a5db61c657af3, 0x000b4dc4d073ec38, 0x00082a9b480cb89d, 0x00000000000372fa}},
        {{0x0003179049382c1e, 0x00069dffb03ae77d, 0x0002e9528cdd6bd1, 0x0002521b19fe0db8, 0x000f3d5c7fee4c26, 0x000e68e1e0ec1e54, 0x0008a62856510b05, 0x00000000000dc80b}, {0x000ac17897e6fc5d, 0x0000680a509308c8, 0x0000b4dde3197e47, 0x00012ee28235c538, 0x0008301f9653ca61, 0x0001fcdf8d28a0eb, 0x0005f322b11e26e1, 0x00000000000e4d73}},
        {{0x0002a6a91ebadb85, 0x0006346a2a08bc3b, 0x00020e574ba891b2, 0x00056a2b3df9fbc8, 0x000eb121d51228c5, 0x0003bc86c81d3161, 0x0000d14e27ce0b12, 0x00000000000bfb24}, {0x00072db1b86f039c, 0x0005986edb71958e, 0x00011b5a99c9a865, 0x000c6d8067f5870b, 0x00033fe8e5322f6b, 0x000a7997ff558b88, 0x000b9be9433e2321, 0x0000000000087e53}},
        {{0x00084d7bdfada95f, 0x0008b9c66a32b0d1, 0x0004ace9a2f6c763, 0x000fab721e716f0c, 0x000b96ed74e68c6c, 0x000110c8332c8fcc, 0x000efe475890dd0b, 0x000000000005cb4e}, {0x0003947e05207b63, 0x000d29d7b89a68e0, 0x00001e2e33262bf2, 0x000b55bca7a7d527, 0x000655d04585c3f0, 0x00057acc5a6e56a0, 0x00039e818e221c42, 0x00000000000fcb8d}},
    },
    {
        /* digit=7 [{1,2,3,..,}]*([2^28]*G)  */
        {{0x0007398749666d45, 0x0009c0a74da828f4, 0x00001ff782080bc1, 0x00026bb57c2f5ad1, 0x0002845d45e4896c, 0x0009a7d36981e2f7, 0x000e8fca152b877e, 0x000000000007b582}, {0x000b1649b1810a70, 0x000c3ea3b9bd9987, 0x000d4cd2bb2df2fb, 0x000fc5d5748a6550, 0x0007622665eed346, 0x000f16e277ac2f21, 0x000dc8bb5efe7fb6, 0x00000000000644c9}},
        {{0x000db9a336c7d7d8, 0x0003598d0898164f, 0x00065860354f4784, 0x00018287cfc13dbd, 0x000c8655a658651b, 0x000c91b712d606e4, 0x000090ba64d3c563, 0x00000000000b82a5}, {0x000726397fcaaf5f, 0x0006c2d1dff024ae, 0x00092238373e43a8, 0x000ee6b0ea1fe022, 0x000cd5c1273c1ac2, 0x000e603e7c100b60, 0x000dfb4496084cea, 0x0000000000077c2f}},
        {{0x00064d07c56a20fe, 0x00000a93fec079c4, 0x000155e36a436889, 0x00045a5cce5662fd, 0x000f83a9a4a9c00b, 0x000bbeb632e8a0a7, 0x00080f6e0cdebbc0, 0x0000000000063ccd}, {0x0008f36be2f62c1e, 0x00061fc10fa07d22, 0x000b3e653f03a3be, 0x0009cc66bf53af92, 0x0000f10bb6c9fda6, 0x0007625e1474b744, 0x0003cbcda9db3b1e, 0x000000000001dc7f}},
        {{0x0006fec7b896d97b, 0x0007de8e32259b22, 0x00051ccb0af3cd54, 0x000a4219f42edba4, 0x000d0d411d4df147, 0x000014bb46d4bc00, 0x00066fa1a13a2770, 0x00000000000fa101}, {0x000b6039e0c4cc34, 0x000d8b2a1dfa62b6, 0x000ae98992614f2f, 0x000a3a2f88c7359e, 0x0008347726a08409, 0x000507bb9071f383, 0x000167d18a551c27, 0x00000000000b359b}},
        {{0x000fae4c55d4b2c3, 0x0000aeaaaf45fd46, 0x000aa7e37459675f, 0x0009b673fe123f1e, 0x000dd8fd0129989b, 0x0004982a4e2ca56d, 0x000ec777d6d0cd62, 0x0000000000071e1f}, {0x0001c6bdd9bc3a7f, 0x000043e9a049f5c5, 0x0006deb929a38a20, 0x00008e24fed8f86f, 0x000ce199e8dbac2b, 0x0009cc964a1d1357, 0x00063b7cf06ec8e9, 0x00000000000d85ec}},
        {{0x000ba68a3fc0bcb2, 0x0004e7d111c66c1b, 0x000d9aa66fbcd347, 0x000730c6db857e9e, 0x0009f4b46d124cd8, 0x0008472dc3c9c03e, 0x0001bbd42f0242a7, 0x0000000000026084}, {0x000aac1b65a94c0d, 0x000ea6332b11a21d, 0x000acbe9385d6783, 0x00028eee7e8944ac, 0x00005ab28372402f, 0x0005e1ff33d1bab5, 0x0007296944e82cad, 0x00000000000e8c75}},
        {{0x00058e168fe9a81d, 0x00043a151dcbb9f9, 0x0002eed94828803a, 0x000fc00604d46e1d, 0x000572f3e28c947a, 0x000b1cd1dc3c9d57, 0x000a45ce4c1cbd14, 0x00000000000f80de}, {0x000d8f65d998669e, 0x0003c50920f39bce, 0x000b6be78ee5193f, 0x0008ba13f798e332, 0x0006c5edde471997, 0x000714a1e1294aaa, 0x00003c280002c2be, 0x00000000000f2126}},
        {{0x000493dde1b54616, 0x0002ea44f6ef79f3, 0x000c2b67fffeca1c, 0x0004ed80eaf66728, 0x0008181514a2cb0e, 0x0002927ea2bf485f, 0x00064574670e180a, 0x0000000000012c14}, {0x000339b9a314b3a8, 0x000724068c073875, 0x0004212e0016a517, 0x000651d698b28177, 0x00096da14fa8391b, 0x000a578b1f310d16, 0x000ad7a089be6bd8, 0x0000000000044389}},
    },
    {
        /* digit=8 [{1,2,3,..,}]*([2^32]*G)  */
        {{0x00024e7304503422, 0x000f0ba86aec16bc, 0x0007f0cf87c57f69, 0x000ff0789df2f808, 0x000d97a773d58978, 0x0003f35f685750cf, 0x0008c9806bb730fc, 0x00000000000fed86}, {0x0006b0ff06192aaf, 0x000eadc0fcde080e, 0x00055bc2901e7a1e, 0x0007d028d3ad6cd9, 0x000997293550fefb, 0x0005cfbba5c652b5, 0x000d2232e12942ed, 0x0000000000098800}},
        {{0x000418b23a7be4e3, 0x000cb162cdf33f48, 0x000c8d04be100c6b, 0x000d114454eb977c, 0x0008dea38a674478, 0x00035728a8ce403d, 0x000504d459633b74, 0x00000000000a63b0}, {0x00010a5f9fdcafdf, 0x000c40c74066a938, 0x000e6c61b766c71e, 0x000b588a99690ede, 0x000c3ad775623398, 0x000bb60ee4949517, 0x000becf9824f09cb, 0x0000000000085660}},
        {{0x00083bb80ede991f, 0x000a02daddb11952, 0x000f09f6c4b181d6, 0x000e82721a6aa89b, 0x0007467506deb73c, 0x0008d8daa1091958, 0x000dfd0927724c42, 0x000000000007c17a}, {0x0009a9bb30e43182, 0x0003a8518dab18e9, 0x000594c3465b3913, 0x000c37f89e7a3983, 0x0008e273f6f35943, 0x000143d228e63f5e, 0x00028ec6d0352b83, 0x00000000000ebd16}},
        {{0x000731dadf48f7e1, 0x000a14074ee26b83, 0x00088243bc9a29c8, 0x000d53972cecb4c8, 0x00079a7dd9c4aa01, 0x000c787cf4b0cf12, 0x00053f3e3e3f165b, 0x00000000000942de}, {0x000bfa5d149fa2b1, 0x000010cc6971941d, 0x0007bdd5c6a1acca, 0x000c1e292314a097, 0x000614a1adcb9fed, 0x00062b86a7547d22, 0x000b7d405561a486, 0x00000000000f5480}},
        {{0x000be69f3af05d97, 0x00082c4e59f2ff48, 0x000865d4a01ec6bd, 0x0000d824464bbbbd, 0x000016f9540dfbc8, 0x000595b5d3bacfa2, 0x00080d1954efb613, 0x000000000007a5cd}, {0x0009c6dbd9d7e6b6, 0x000ac926a54cf784, 0x000f3366624e7b07, 0x000167ccb5c8d4c7, 0x000ff9a21ce20677, 0x0002df8cdc994d22, 0x0009083a25ff6b42, 0x00000000000f68e9}},
        {{0x000a093607905265, 0x000ede544b89ab7b, 0x00017731e314dedd, 0x0000da69a73104b6, 0x00067274b105a6a9, 0x000c61bb65c26021, 0x0005068f9705cf60, 0x000000000003c4d1}, {0x000ed5677f9dc5c6, 0x00020ab5a27accb8, 0x000e0bb2ed27cc25, 0x00036d15a36afae3, 0x00095c455916e68f, 0x000b5d1fa79004b1, 0x00048916ffe6249b, 0x0000000000049338}},
        {{0x000a7603914a9a59, 0x0000b941be86e102, 0x0001a6f35b551149, 0x00095b469d75ec8a, 0x0003db0d4374658f, 0x000fc77053fa79d5, 0x00012885da635c6a, 0x000000000008e7e8}, {0x000f4285c3e56baf, 0x00002558cfa8eed1, 0x0000effdf411ca89, 0x00098b96a32e8849, 0x000e3c45ce1a104f, 0x00085de0268237ef, 0x000de35c820dc22d, 0x000000000007459e}},
        {{0x000a78ec870a31bb, 0x00036923d0b44369, 0x0005db7ea085ecd1, 0x000be009cf5b87e5, 0x000d1d61103d1656, 0x00065239313a37d0, 0x000ed81d705880fb, 0x00000000000ed712}, {0x000ff1a5976c303e, 0x0006f15ad02e6160, 0x00077114865ad858, 0x000376cba2b3ffe8, 0x000f9745443c56aa, 0x000903660c3be2b2, 0x00092d47c8a870eb, 0x000000000006c6c1}},
    },
    {
        /* digit=9 [{1,2,3,..,}]*([2^36]*G)  */
        {{0x000b550138d02bd3, 0x00038148bd39cbc2, 0x000f6b4c6038c07a, 0x000fbe2ce5484157, 0x000c87fdde9ff397, 0x000e9c179441e5c2, 0x000c716366b49ffe, 0x000000000002938d}, {0x0008a64bcbf3adf9, 0x000d026d450f9f8a, 0x00015da756f71781, 0x000bf4d298fd8771, 0x0007544768b65f68, 0x000491267e86df04, 0x00071a40b69a32f8, 0x00000000000f917c}},
        {{0x0007f3a58f523dab, 0x000a7a66c70349a5, 0x000f8ae356d6f09f, 0x0003e96b5ab54115, 0x000c7c57d123dee3, 0x000d6ad37d068929, 0x0001780839a208f1, 0x00000000000123f8}, {0x000f3c2b5c9dfc15, 0x000f4b3e5e52449d, 0x00055ba373af8955, 0x0000ab7389f2dd3e, 0x0005890bba6f513a, 0x00066bf093197a14, 0x00072261add75b6f, 0x000000000004eef1}},
        {{0x000fbdf154b15bac, 0x00063810b6ab3193, 0x0006da8c3809a3ef, 0x00038dd898977511, 0x000e7a336c9a3cf8, 0x0006f89c03e391e4, 0x000e227014833717, 0x00000000000bc4f1}, {0x000e7d4400e0ab41, 0x0006b32b104f92b2, 0x000d1a7a3b67e3fa, 0x0000437bf178ac12, 0x0005c99370d5b831, 0x0002b93a8722299f, 0x0007190a493cf033, 0x00000000000a420f}},
        {{0x00046acf9a0ee15c, 0x0006a21feb7fb87b, 0x000579369777bef5, 0x000557624b04e704, 0x0006342cb0ad03a6, 0x0004f64262531f18, 0x0003ea088c4d54a2, 0x0000000000006a87}, {0x000f1f11e0fca837, 0x000d5dbe0253ef23, 0x000bcfdbd73eb554, 0x000368173e65902b, 0x0002ccbfa504eaaf, 0x0000e163e71f1fab, 0x000f3bb7b845224b, 0x000000000003c779}},
        {{0x000cf36036019ecd, 0x00029cd7b3c4286b, 0x0005e1ca08cbdeb9, 0x000bcbd24ef5c386, 0x000ce579c309eb66, 0x000f6c9007edcc21, 0x000c2c7b19d49116, 0x00000000000b6317}, {0x000aad793c4e52a3, 0x000e7554ba553558, 0x0002315e3b514170, 0x000e33bffda4032e, 0x00082306675c3d1f, 0x0000c91e75dfec47, 0x000879be59305e00, 0x0000000000025a6b}},
        {{0x0002c6fd041a12fc, 0x0006aa35802f5d21, 0x0000c3d459456256, 0x000991d472b9d211, 0x0006a2f8e875261a, 0x0009b6d63d81a1ed, 0x000758942f213a69, 0x000000000000ae57}, {0x00067bfe08ea2ebd, 0x0007061191c82b48, 0x0000611a48f73652, 0x0003e86525112224, 0x000d30dabb91abe2, 0x000d2742466dd967, 0x00005077650c597e, 0x00000000000ab25a}},
        {{0x000abb01ee5e0194, 0x000bca624ab366b4, 0x0009dc413b0af513, 0x0009c4273aa694c3, 0x0009779288abe822, 0x000575e0e0cc3102, 0x00003ef8eff30f57, 0x000000000007d528}, {0x00093a51fb5fbbe1, 0x0002f32d87e548f1, 0x0004001c13dfb44c, 0x000b8dd16c6e6274, 0x000c2c140452aa2c, 0x0003031b1add098a, 0x000543d25f285d2d, 0x0000000000075b59}},
        {{0x00032a5061a42b94, 0x000dc520b0bb8a42, 0x000466f1305a432b, 0x000c73a73c239760, 0x0009783aabba85c1, 0x0004631556e4dec4, 0x00017b69f0c69bb0, 0x000000000009c97b}, {0x000fbbef3e8375b4, 0x000b155af24a9074, 0x000991d9ad3481f7, 0x000283d708671c48, 0x00035fd9001a4034, 0x0002eaf3b200ddab, 0x0006c4e45f28e434, 0x000000000001ba93}},
    },
    {
        /* digit=10 [{1,2,3,..,}]*([2^40]*G)  */
        {{0x000e8393cd68c757, 0x000b2b083ba6a1e9, 0x0004638d474c7417, 0x0007a21fc82dc041, 0x000a9d3679d89536, 0x0009724c0227be26, 0x000c0fc70f6d6c7e, 0x00000000000f9ebe}, {0x00075bdec21bc5d4, 0x000b029dde03dcdd, 0x000a669d8fc534ff, 0x00090c90f602f4cb, 0x000849722bc4daf5, 0x0009b22b617c5288, 0x000b90a8df99f008, 0x00000000000e59b9}},
        {{0x00015e6442d15d01, 0x000dc6f5775290ef, 0x000cdd79298e58a8, 0x000842778b96c6d8, 0x00022f59350519a1, 0x0007209d6a674f99, 0x000fff5abeeec46b, 0x0000000000047cf5}, {0x0009d3497d146805, 0x000ede24509b7378, 0x000ed2fba1e0b34e, 0x000af595761e8e3f, 0x0008d420a2887f7d, 0x0000ff696ed5cfcb, 0x000c8f365b29eb7a, 0x0000000000099a1a}},
        {{0x000785db50fa1164, 0x000694936c6a0393, 0x0005ce545ed4b2d3, 0x000e8b45714f2c6a, 0x00023f5ac25a03ae, 0x000b33794139bd69, 0x000ba96a2e42bab5, 0x000000000003ff7b}, {0x00034248c56f7e51, 0x00088b61d8643327, 0x0008d647e582cbe4, 0x0000e1472eb77fae, 0x00013b99c6356211, 0x00074c9f23d5b96f, 0x000250956ecf64f6, 0x00000000000ef2ba}},
        {{0x000a8baf84131eb9, 0x000019ee1ec3a29b, 0x000d9f684960ce84, 0x000737588102ac15, 0x0009c08527f432b9, 0x000e3dfbedd296cf, 0x000c4fb74f8145fa, 0x000000000006cd7c}, {0x000debad8e4205ae, 0x00062a0f2fe7a043, 0x00094ce3fc7d23aa, 0x000f2d40eeb90a7f, 0x0000be4de6846e7d, 0x000dd06bce2f46e2, 0x0009f6cd28feba3f, 0x00000000000e6d6d}},
        {{0x0002283f4c1e03dc, 0x000bc246ffcb6b34, 0x000a382150ba305c, 0x0003ae2250e66766, 0x0000924ce5fab4b4, 0x000d8c77695c1b5f, 0x0009d02555795beb, 0x00000000000acd9d}, {0x000cf0d26acc1b8f, 0x00088e1d74aa6321, 0x00035822f91490d5, 0x000df2795af56df1, 0x000fb331b6f4df74, 0x00059e13724b10c5, 0x0007f2b0a6df9a65, 0x00000000000c0663}},
        {{0x000c55f77d493f59, 0x000089ad73168775, 0x000791ccc3015317, 0x0006c2d30b3a5f4a, 0x0007c89723d59e94, 0x00031f6077bc4ced, 0x00034179f514a1bd, 0x000000000003a274}, {0x0007950f4645c0c2, 0x000e07eb010278e1, 0x000a3d29cb5ab91d, 0x000760f35be21cba, 0x000b7c793331718d, 0x00030d29eba58160, 0x00003afbc0ce1f8d, 0x00000000000c6f4b}},
        {{0x000986e6462b5c9b, 0x0000cbd8c0867ee8, 0x000db80962770b4f, 0x00012de024593896, 0x000fdef840b687ed, 0x0000b56e13f7d98f, 0x000e8771eee0cb5f, 0x00000000000d8d9b}, {0x000f5c1c38b9eff3, 0x000c1e6b50b5a5f4, 0x000fada267894657, 0x0001bd17cb1f9925, 0x000d4ff11827418b, 0x000042c63607818e, 0x000ae3e630d93a9b, 0x000000000008c779}},
        {{0x0000de60ecec558f, 0x000bb35d474260aa, 0x0007deb342712d19, 0x00015e22e91bf5f3, 0x000cc08b6b1abd6a, 0x000b97de8e366a84, 0x000f29759c122f55, 0x0000000000008a03}, {0x0005b54173576b1f, 0x00000dcc9fca2774, 0x00073c06ae128d8b, 0x00039029b59cd052, 0x00006f5e5bd4deae, 0x000099f4df532ede, 0x00005284fbeeb936, 0x00000000000088d2}},
    },
    {
        /* digit=11 [{1,2,3,..,}]*([2^44]*G)  */
        {{0x000b3d633d0721cb, 0x000732ba8c78fe5a, 0x00016e2c1c57f816, 0x000f36a2fc2451f3, 0x0008bb91e1e36842, 0x000ead762fc5c955, 0x000556035d1dfcc3, 0x00000000000031e5}, {0x000b4359fe8646d5, 0x000383af0cc803c6, 0x00070f15b8bb97ea, 0x000de0a6ade1d137, 0x000d93b2dcb580c3, 0x000a2214de8c3a5b, 0x00048de3adbd7c90, 0x0000000000011929}},
        {{0x000e8783f9b6f97e, 0x0009b65026296c0d, 0x00086ba77888a60e, 0x00063a460c8bbf8b, 0x00078b2a71206237, 0x0005497e7fa8f5ad, 0x000618fd744bdf08, 0x000000000002ba35}, {0x0004df87b45c7eff, 0x000870cebfe9d444, 0x000c034f12ddb3df, 0x00017a3fcf19627f, 0x0007c2f112616558, 0x000f2c85030ab44f, 0x0000c3bb001c9ddd, 0x0000000000007326}},
        {{0x000e365e9f55b0a4, 0x000aeb08fb116bd9, 0x0002cf623c1f798f, 0x000fc7b6f9549671, 0x000f76bd243c73ae, 0x00009d5a8c0fb886, 0x000049871eacc5ce, 0x00000000000e773f}, {0x0001eb3732cb8726, 0x000aa92945c840e9, 0x000022c04533de34, 0x000bc1d0509d7400, 0x00010f1af1754762, 0x0008c160f15cf97f, 0x000f0c1f85569532, 0x000000000003b439}},
        {{0x0004f7c9bedca76f, 0x0006dfa7d1236235, 0x000a7e4930642e7b, 0x0007288beb1282c5, 0x000a07fee8a99ea2, 0x00070fee91c069ef, 0x0006fa5749c7b558, 0x00000000000afcec}, {0x00048441716f41a1, 0x00064a3f8f1b0daf, 0x000b8af2f805e4cc, 0x00029a9b59dc06f1, 0x000b98a92c387533, 0x0002b4662fa8e5f5, 0x0006c66b6f46fd3c, 0x00000000000ec04c}},
        {{0x00054b9f6efe8494, 0x0005eaa16c27a15a, 0x000106292d7b104e, 0x000d193aae87c9d3, 0x0009916d634e7ae2, 0x000a65b4b125ab45, 0x000e2202ded714cf, 0x000000000004e212}, {0x0009494225bd1826, 0x000c097c48a1862b, 0x000bf9e4c3ff8573, 0x000b77b2652f5018, 0x000d078efd386fe8, 0x000cb82991daa602, 0x00062635885364db, 0x00000000000b8240}},
        {{0x000f5a3697f1c244, 0x0000e0430af76c1b, 0x000f0e87f66ce63d, 0x000905f12e919108, 0x000012db9e14e1a7, 0x000baeeac1c689b7, 0x0003196bdd3dc90a, 0x00000000000504f0}, {0x000e18cdf6373284, 0x0003c874afd60b16, 0x000a978150da10ac, 0x000eee1ebf4aab2c, 0x000c1aa49fe60d33, 0x000217cfda3631ca, 0x000e770d8340fbf2, 0x00000000000423b7}},
        {{0x000b3813851ecc4b, 0x0001df8c07372826, 0x000ea9f99e2d35f1, 0x000faf1a6305a291, 0x0007f3e0f93d2b97, 0x000aeb8c15bc61f6, 0x00024b7238583cd7, 0x0000000000039f5f}, {0x0002b746db300ac6, 0x000a11cc8b467be6, 0x000e46954d17b55e, 0x0005f95ba2641ae4, 0x0002ce9d565b1b9a, 0x000eedc6287a0c36, 0x00003d07fb51b2e1, 0x00000000000a9739}},
        {{0x000d77fe5e566bbd, 0x0001978a53b5a370, 0x00081dca6fe505a1, 0x000f427019a6f8e7, 0x0006dc3ad0ba3520, 0x000745b7cde6fcad, 0x0002dcfec96e4f79, 0x00000000000b133f}, {0x000924a225ecf745, 0x0000c50088a2b006, 0x000c145291ebead7, 0x00032ff23ae4b9d3, 0x000e85246712f213, 0x0000b515e8cbc659, 0x0008b727fa9c8df5, 0x00000000000494ac}},
    },
    {
        /* digit=12 [{1,2,3,..,}]*([2^48]*G)  */
        {{0x000ff6bf222c5c9c, 0x000322986475308d, 0x000309c5ef927cbf, 0x000d6b4216ab9acb, 0x0007be12d76a038c, 0x000347bdb3df9b7b, 0x00048913f4d9785f, 0x0000000000013e94}, {0x000466717b5c75f3, 0x0000a5f8e796eab2, 0x000d6af2aad3919a, 0x0005d8ad10740b88, 0x000b5337dee6254b, 0x000f02247c38b8ef, 0x000c4cf688c2e194, 0x000000000006c25b}},
        {{0x000272cd3b35df41, 0x000d936c9dbbec27, 0x00026ae70fa619c9, 0x0008db696a8f9f19, 0x00056b01e6bc1ab3, 0x000fc4adae031d23, 0x0004e410466ae68a, 0x00000000000ed9c4}, {0x0005ea962547af52, 0x000cb61272c12a27, 0x000f929706a5a2ac, 0x0007a910ecc49eb8, 0x000ccbe84d5cf4c4, 0x000e497d7eb95dfe, 0x000ce443f3b71c8e, 0x000000000004c6fe}},
        {{0x0002d9d94d551889, 0x0006182e5d818574, 0x000101531df0c231, 0x00044261daa2e22b, 0x0000e46f32576b02, 0x00069db38b86a358, 0x00027eacf145bd76, 0x000000000004df27}, {0x000d2ba752047cd9, 0x000c203d9391e25b, 0x0007c9592434b2d4, 0x0007845ec38fa9ac, 0x000a265ad6bbefb7, 0x00054a1b2dd40660, 0x000499a22d988618, 0x00000000000737ea}},
        {{0x000ef1248ca55f15, 0x00028e323ed0c422, 0x0001736a7d35b006, 0x0002f06e8d68e4e9, 0x000ad0742e5d9c09, 0x000d3df92d8f5555, 0x000eabe2d175bf00, 0x000000000004f71a}, {0x000a6a143a42cf09, 0x000c6d1762d7229e, 0x000840a2cbe90735, 0x000cb4c6281f2a74, 0x00003603e53a2caa, 0x000fecf29635ba47, 0x00036194a9811d49, 0x00000000000466bf}},
        {{0x0009fc85048451fa, 0x000fd4737236d065, 0x0005b89cfa755eca, 0x00070306da6e06f0, 0x0006f3838f569da9, 0x00043188730279bd, 0x0005d0fb328c8b94, 0x00000000000be90e}, {0x000859016f87df1b, 0x000843334a6711d3, 0x00078d74e5890358, 0x0007b6e38904b738, 0x000296b588a53493, 0x000577ae391e227c, 0x000c7da599b21544, 0x00000000000214a0}},
        {{0x000fcc62fe159c27, 0x000c9e63fbb0fb71, 0x0007ab3cd12c8947, 0x00030677afd4bf85, 0x000dfd37120d5cea, 0x000d718a74494e39, 0x0005fb8c572c7249, 0x000000000005fa30}, {0x0005abf2e0c1181d, 0x00074751c217ee1b, 0x0005917c5a26a520, 0x0006e6efe7a64872, 0x0000f53b0e479a99, 0x0005fd5931a4f6d1, 0x0009ee651390ecd1, 0x00000000000739ee}},
        {{0x000e27c9677a2151, 0x000ef0b6d37446aa, 0x0007f13e8f2bd87e, 0x000c94fa109847d5, 0x000044944c7712bb, 0x0005a31b874c0d53, 0x0005920d280b18eb, 0x000000000007ef42}, {0x000c07ddca373d80, 0x0004ef11030c77be, 0x000075bee798eeea, 0x0002d013d22f1b04, 0x000cd93ee54dd5e0, 0x00041d4b1b6d66c9, 0x0006ed80b4154faa, 0x00000000000acf8f}},
        {{0x0004ca485f1804e7, 0x000ad0f05710ab2a, 0x00002b0d41da0420, 0x000a67a46b8d0e2d, 0x000c698b78cc137d, 0x0008a9393454b89f, 0x000e69f2a6e1de25, 0x0000000000016488}, {0x000b96b954a8287c, 0x0003d7c6c5501c10, 0x000fb63222050457, 0x000e30e92f152478, 0x000327e70a0a4b9d, 0x00014936309d4ca9, 0x0001379c8b16340d, 0x00000000000ce642}},
    },
    {
        /* digit=13 [{1,2,3,..,}]*([2^52]*G)  */
        {{0x0003c432161be476, 0x000c0f8a8499b505, 0x000715248b87d78c, 0x0001b1d515e1328e, 0x000ba941e788b85e, 0x0005dd8d888a2636, 0x000350a045241d2b, 0x00000000000332f0}, {0x0008eeabc026bdc0, 0x000f796c4b204e16, 0x000ce54b1f342310, 0x0003fc6d00b602a1, 0x000e89aa3b796fc3, 0x000d4dd0007a914e, 0x00095635353eb7a4, 0x00000000000673e8}},
        {{0x00027e0f6ecb7465, 0x00040f36e83987c1, 0x0002e0c806d929c2, 0x00007464efc5b0d5, 0x000ad316c43436ab, 0x000ccf839b211e59, 0x000a072515ec9f16, 0x0000000000003dc6}, {0x000ec4a69e8d5661, 0x00017842b727527c, 0x00065c4526d40261, 0x000711ccef5255e9, 0x00075108cb92967d, 0x0001b9740cd3bfb0, 0x000308e50c0d8aec, 0x000000000001a9e6}},
        {{0x0002078cea733c1a, 0x0005b283eec25eca, 0x00036d44d991d5b4, 0x00083b827ad302e8, 0x0002bde3fdd0269a, 0x00030b3a6225f2f1, 0x000043046fcf3801, 0x00000000000c3ed9}, {0x00066dedc2439ae4, 0x000eff870f14cae6, 0x000680b39cf67cb8, 0x000d5f4847be7732, 0x0003d0ed73a0f3e1, 0x000b3babba949822, 0x000f706933ccf014, 0x0000000000037f08}},
        {{0x000ba839c8cf3524, 0x000ed1afa6aa5579, 0x000f9ef0d2ddddd9, 0x000920b5d36da502, 0x0009291e774f07fb, 0x0000d87a8144d51f, 0x0001a026c2c134f4, 0x00000000000a932f}, {0x0003544f31a7b78f, 0x0009935bb2a42294, 0x000ea47969f6664d, 0x000cabfaa1838ed8, 0x000fe2855f1f5c40, 0x000525934ea0c05f, 0x000fd4931ebb02fd, 0x0000000000016246}},
        {{0x000bd623cb7fe067, 0x00038ecfabd26775, 0x0008c3832c0a3527, 0x00064ccfe2691ca7, 0x00058347566acf4e, 0x000b8c733e3889e2, 0x000f5748da354885, 0x000000000009d9c9}, {0x000592cc5e5c9fb4, 0x0000e8c26a8d609c, 0x0000459f168ec210, 0x000a70a3f8db9f92, 0x0000e758213e181c, 0x000b2653e25aa645, 0x00003aa4898f9169, 0x00000000000ccd83}},
        {{0x000fac468f3d59db, 0x00099517d13b0e90, 0x000cf2490366b8de, 0x00026866d752aaab, 0x0001cab026676e2e, 0x0003163c395da9fa, 0x0003c9be8d91ad42, 0x00000000000a6995}, {0x0007a57ea4ee9030, 0x0006ef728b1d231f, 0x00013aa7ae93d8f6, 0x00000a82e75d9c30, 0x000e3ad09def97a9, 0x0008a2be8136c6b1, 0x0004474bab14a6dd, 0x0000000000025cf1}},
        {{0x000e1d30e97eb41c, 0x000b411b59f3da92, 0x00020acfcec12d54, 0x000f0a1936595900, 0x000cca0b1b0e5cad, 0x000274a5e8fef04f, 0x00027ebcb4d9fb0c, 0x00000000000ba784}, {0x0008e71ae4477c5e, 0x000fbc49f0bc478a, 0x000ac96d890c62e2, 0x0009e583f796b820, 0x000b17964262200e, 0x000db00395bbea92, 0x0002ba86b3c15756, 0x00000000000ead48}},
        {{0x0007642e08638534, 0x000b5cd92906c650, 0x000ae6db49a06b5d, 0x00029781fdc19156, 0x000c269d611e0d69, 0x00065b00a45d01a8, 0x000388e7bd1e7096, 0x000000000009bcaa}, {0x0007591cdd6ae97d, 0x000ed8f189e87506, 0x000ccf1d10959a9b, 0x000bf16c633b1123, 0x0009d6f1dac8ca65, 0x000c3381dc6adc9c, 0x000d120df2c293a3, 0x000000000008388c}},
    },
    {
        /* digit=14 [{1,2,3,..,}]*([2^56]*G)  */
        {{0x000653d60a9d872c, 0x0003bffdbd0eb2dc, 0x00061fddaf39f568, 0x000cead35e7a3a29, 0x000c67833028c11a, 0x0004c79764998cf8, 0x000a1c8f3a346b84, 0x00000000000db9b1}, {0x000642a471b701b9, 0x000628735dabc47d, 0x0005300f39a216a2, 0x000dd49d267a01b0, 0x0003ffa20d117c0a, 0x000ab2d4a2b46c91, 0x000080f2acef26d8, 0x000000000003d926}},
        {{0x000ba70a22083784, 0x00084e9d2a98a2f2, 0x00091072e4da23c2, 0x000325dceaf86ae5, 0x00088f161525f399, 0x000211b9d03b17c8, 0x000a48d8ac35c984, 0x000000000009050c}, {0x000bbfa19d5ef891, 0x0006ba818c44b2c9, 0x0005e1b560830da0, 0x000af35f8715b052, 0x00099d8829a9633a, 0x000a820f15463e1b, 0x00075db18df52d84, 0x00000000000d0966}},
        {{0x000ae853a945611a, 0x000bed54e2c7c031, 0x0009b2bf77ae12b1, 0x00005e8e60f7f5a6, 0x0008427483adcb41, 0x000bff383705db30, 0x0006e9f73ba98bf2, 0x000000000000220e}, {0x000231b48f25af77, 0x000e8c01c46b84a3, 0x0004ae472f3bd7a2, 0x000dc0bfa3403e6d, 0x0007d2202896f738, 0x000882e5e098bc68, 0x000b13ec0c217a5f, 0x00000000000a1f4e}},
        {{0x0009208204f520a2, 0x000ed2615c78254f, 0x0002b7a2ee7b484b, 0x0001d771c84a04b5, 0x000fcb9f9e349c3b, 0x0004f7846b6fc203, 0x000248bed65464c6, 0x00000000000eb004}, {0x00040455a574f8cd, 0x0003f5c017726a4e, 0x000b0a7d5066828d, 0x0006666aceb0e3ac, 0x0008fc046f0ab78c, 0x000aa959518d3c18, 0x0004e87f3e2f301b, 0x000000000008a284}},
        {{0x0008a96d207a2eb2, 0x000e3c899eba3614, 0x0002ec9689146ad2, 0x0008629da55568ed, 0x00082b1dc1d9607c, 0x000c001ff6b54722, 0x000b8523232311c9, 0x00000000000488f8}, {0x0000aca655c2cd82, 0x000723867ac9a850, 0x00092fe2557b4773, 0x000c647a74488fff, 0x000fbe0876407398, 0x00019a571f6ed920, 0x000aeb72beddc763, 0x000000000006a244}},
        {{0x0006fe658431a6d6, 0x00064b7da79c5a1c, 0x000fc6b2b6576354, 0x000b7b54aa36d18e, 0x0008ed0e30913481, 0x00093074c6efeaf8, 0x000654eb017bddc9, 0x00000000000529dd}, {0x00089f1ff5fdf666, 0x000fcf5177230c70, 0x000373317732e646, 0x00082d34ca267426, 0x0005adcd1650194d, 0x0007758b7eaeffe1, 0x0008194dcec3d9af, 0x000000000004cc2c}},
        {{0x0003e55601cd21ee, 0x0004794bdc7f4a7b, 0x00080eb7c8f212fa, 0x000dab0d654cb574, 0x00037a49195627e2, 0x000d5d0991d4e1e5, 0x000de7a1ef570c31, 0x00000000000295f3}, {0x000c78902e5817a8, 0x000198681b00d89c, 0x00061b3376c1d033, 0x000e90c6a1b57484, 0x000c1222e5544324, 0x000d53dd044f9324, 0x000ef0e30ba10fff, 0x00000000000b48ee}},
        {{0x0004a14fee68295a, 0x00041a349bb65da9, 0x0006f09eba200d68, 0x000d891c18d37516, 0x000dae8d2bfba6e1, 0x000b330789985aa4, 0x000e948baec9ae31, 0x0000000000098750}, {0x000e6cd5311f8630, 0x0009c0d834bf8a5b, 0x000c536623c88198, 0x000faaa0c51d098d, 0x0002b887a10f0c22, 0x0008bed323240404, 0x00066231f6a61424, 0x000000000001ce0d}},
    },
    {
        /* digit=15 [{1,2,3,..,}]*([2^60]*G)  */
        {{0x000c2532e44da930, 0x0009eac6dbdf6097, 0x0009dd0474f3e9a9, 0x0004de3dc28e1b18, 0x0002a66477111669, 0x000b08b4b2d039cf, 0x0008cea13bbe6fc5, 0x0000000000010ebd}, {0x0006e345ee3db6d1, 0x000c2cd4720862b2, 0x000fcd0516567c14, 0x0006303929bfa29c, 0x0003818249bfb3f6, 0x0007f7cd97c378e4, 0x000a1676068c8084, 0x00000000000987eb}},
        {{0x000e55f593b72e2d, 0x0008c1238e0971aa, 0x00081a4c08ea4523, 0x00054e7f74c5c514, 0x00028805813b4512, 0x0005e7a6e238b16d, 0x0000bc23987b2892, 0x000000000002a1a1}, {0x000c0e8fa1a83cf0, 0x00010944c784c643, 0x000f1939fa2e366e, 0x000ab2aead457171, 0x000631c042056bf1, 0x0000b8881d5f8f0e, 0x0005a8ac062526a1, 0x00000000000fe550}},
        {{0x000d9597e54c99bb, 0x00076cbbd8575ded, 0x000019092c8277a0, 0x00056202e6b72a58, 0x000e7c024443e5cb, 0x0005368f35738ea0, 0x00044f8ed06170b5, 0x000000000001aeed}, {0x0001a7ce730f30e4, 0x0003a3d90a6b26fd, 0x000480ba5c428a59, 0x00093dcc5612aec9, 0x000f99c1bacc0890, 0x00043eaadc272ef6, 0x00089147db3b43dc, 0x000000000000832d}},
        {{0x0001abaa498687bf, 0x0001bac92ee14ee5, 0x0002ef494748554e, 0x000ca876e7a32661, 0x000a8456c9af29a7, 0x00054326f0f2b7e7, 0x000ec87471824e97, 0x00000000000364d2}, {0x0007a1e32547416e, 0x000f386d8aacd172, 0x000735a9921c3b5f, 0x000fc881d79f7eb2, 0x00040040547d9805, 0x000b4e90b377ac3a, 0x00081bd39215d461, 0x000000000004c5fd}},
        {{0x000bf8e11ac3f3fa, 0x000df9ffe5562f1b, 0x0000905bb11344f5, 0x000f981cbefa3c77, 0x0003667bfd643039, 0x00040e3df2ab0688, 0x0009ca9007a25743, 0x000000000005a3a4}, {0x000a64031de3bcfa, 0x000fd9c7be629ab9, 0x0003fbe0cdc1be9c, 0x000ff39e7380dbb1, 0x00059facc3b8ffe7, 0x000ae422c620bb9b, 0x000c432ddcb8cd31, 0x00000000000d12c3}},
        {{0x00072db65199d9d3, 0x000d703621d34a54, 0x0008bb1d8b92a619, 0x00000f66ca2933b1, 0x0002de1494a2886d, 0x000dcdf82d0a2238, 0x000832b8656c4d0b, 0x00000000000c4a58}, {0x0005f2f5aceb0154, 0x0002fa982c1aa34d, 0x0006b9fce5077d2d, 0x0008de431390c6e9, 0x00018c4fe595fc26, 0x00090d82fea4160e, 0x0009076c427a6367, 0x00000000000fc519}},
        {{0x000775c688e983e3, 0x0003e8c0749464df, 0x000f192d78daad34, 0x000e96904bb2048e, 0x0004d6dc8b606cda, 0x000438bbc6dec959, 0x0005a58d26586954, 0x000000000001b0e9}, {0x0001e04544fa17f3, 0x0005e8189f1141bd, 0x000c131e8abc17dc, 0x00075a227f8ec1ab, 0x0000607e37397757, 0x000793e4e7a12524, 0x000af4afae84e74f, 0x000000000005bf5b}},
        {{0x00039549b6d366af, 0x0000aa10292b850d, 0x000fff80cff6798a, 0x000f18f73ad7a1fd, 0x000c5897b11f7a36, 0x000664c618b2c7b0, 0x00022cf7b9a272cb, 0x000000000008f81e}, {0x000e957ffad5963b, 0x000663b99b210935, 0x000ea3abc7ab4283, 0x000175d001db74bf, 0x00067159cb8a3af1, 0x000de4601526f084, 0x000eac6a3c6e1feb, 0x000000000008c232}},
    },
    {
        /* digit=16 [{1,2,3,..,}]*([2^64]*G)  */
        {{0x0004c667c92f7888, 0x000aaa54768d9e88, 0x000d1397d0aa7f52, 0x000f203faef6863d, 0x0004bd7471b3774d, 0x0007de2e9f795a03, 0x000cfff0958718b4, 0x00000000000e1160}, {0x0005c7ba07a0ffd6, 0x0005186ded97af9a, 0x0008fa18cb4fab4b, 0x0008920424e84590, 0x0004eecf8b2b0558, 0x00068a6fa3745591, 0x00019fe7d3df1fb9, 0x00000000000bad07}},
        {{0x000b4e8b4136f0e4, 0x000ae2566021f579, 0x000cbf2ef6760188, 0x00047a5d9f51b6a3, 0x000df8efa64634e0, 0x0001f584f0b50d91, 0x00091cc2bbcbb297, 0x000000000000907c}, {0x000fd43610d5f812, 0x0000ca7ebeb0dd65, 0x0008a7b6eef1e9c1, 0x0009a073f5c962ab, 0x00053ff74a2fc04d, 0x000f0749d95c155e, 0x0005d0923c65a53d, 0x0000000000027ae3}},
        {{0x000cd3b33afd62e7, 0x0003c4d37c266037, 0x00042b261375e38f, 0x000a2e928ca9d674, 0x000a79beb236566d, 0x000f801e7a9771c1, 0x000358af6b97a976, 0x0000000000071259}, {0x00004ab4fe03d3c3, 0x000bcc23a5e31cf5, 0x000c506466ff69e3, 0x000233b1911ccf3a, 0x000cfa3b3ace3f3a, 0x0004c5f5c93e4664, 0x0003c04bdc14832d, 0x000000000006abf1}},
        {{0x000832aefb763bd9, 0x000f0d5469c7af17, 0x0000d7bc962f1b04, 0x000ca21a16caa7b0, 0x0002e6cc7f39f881, 0x000378723221de18, 0x00066010d61ab531, 0x00000000000520c9}, {0x000ed27c50cc42c5, 0x000ac145214ccbc9, 0x000ac441f327ce66, 0x00059e30cb1fe6a2, 0x000a3e299fe79fce, 0x000af78cf2e6e77f, 0x0001593a0cf652d5, 0x000000000003e852}},
        {{0x00087e649fa97c56, 0x00029ebbe18b74d8, 0x000417a0e476f2ee, 0x0005a2a0b24f97ba, 0x000262fc61243ddf, 0x00003b73af9a400c, 0x000fde9ad9b0bc6b, 0x00000000000153c8}, {0x0006f3a6ac904b01, 0x000f41b6477d9542, 0x0008ea414bce433e, 0x0007128953069569, 0x000db9e775794428, 0x00005e0d14b5db01, 0x0002f5237edf0bde, 0x0000000000085533}},
        {{0x000e220415482512, 0x000190791dfa1ca2, 0x0007078d88a1aeff, 0x0005b77ccedf4f34, 0x000d5d965c0549f2, 0x0009e672705170cc, 0x00017637d9521bd4, 0x0000000000086a00}, {0x000e53f005b29758, 0x000b5dab44493664, 0x0005df02ede5bef3, 0x0000c3e4c2506ba1, 0x00065ec6f1324366, 0x000ac7691b2fb261, 0x000eb5ccc4221a99, 0x00000000000a576d}},
        {{0x0006b1405a6d6a62, 0x0001e3a17f86cedc, 0x000ffc614b14d0e0, 0x000a3f0927935e2a, 0x000ac04d4fb0a86e, 0x0007694212f43742, 0x00012f32a30bce38, 0x00000000000d8d39}, {0x00091bf3c81523e6, 0x000ed34154b18f9f, 0x0001725c2ac49b1c, 0x000be0f9f7a13c1b, 0x000a531736e526cf, 0x000cfa2c250d50b3, 0x000e5f81849773b2, 0x000000000003ba8a}},
        {{0x0003089126b94126, 0x000854a87307686d, 0x000fced9f497ff98, 0x000042f427ea198a, 0x0001ed259219c1d3, 0x0004850cadc59211, 0x000e8d0abdbd1623, 0x0000000000043b6f}, {0x0004352ff19d9aba, 0x00069d3c79d16450, 0x00090120dc8bc8d6, 0x000535d5a98bf664, 0x000dab2cdb2ba736, 0x000af89eeacb3b7d, 0x00053237d3743ada, 0x000000000000b348}},
    },
    {
        /* digit=17 [{1,2,3,..,}]*([2^68]*G)  */
        {{0x0006e0a86f5827e3, 0x000e1a68295d9bd5, 0x00032e6b7dcbdf31, 0x000a0cffe0c3df09, 0x000b3cd00a1a8deb, 0x0000f90885b4d037, 0x000ef7e9edc429aa, 0x000000000005847d}, {0x00025d7642f87bf0, 0x000f49739d03ced7, 0x000f63949ff1cd98, 0x00034ff1759060ab, 0x0009a7e94dbbdc3d, 0x000e8b7f3029e9aa, 0x0006f42a3cdfa0f7, 0x00000000000bbd8f}},
        {{0x000178a4d4ad7d87, 0x00040cd8c2ec862e, 0x0003f7b9dc665542, 0x00086d4db83e1ac5, 0x0002bb549bb5e123, 0x00007c0f19a79203, 0x00008eaf81cba628, 0x00000000000abf20}, {0x000001ed540ad4e0, 0x0000a9d7a72302a9, 0x0007110b90df5c50, 0x00024daec9005170, 0x00037193eb3047c3, 0x00015c655408cd0d, 0x000c228f0bcce2d3, 0x00000000000869d6}},
        {{0x000e5354c88a38dc, 0x000717192a26df13, 0x00059c532d5dc6ab, 0x000195620aeab120, 0x00057c2328498938, 0x00079bc39ebe39dd, 0x0000925787970e4f, 0x00000000000143e4}, {0x0006e01b286241f3, 0x0001e5f60303562b, 0x000cf202635ec6b7, 0x000b3a2f1856e619, 0x000bbf77d65c7ec4, 0x00011058fbef7dc4, 0x000945b444d50670, 0x00000000000c33cb}},
        {{0x0000e30b66c01c80, 0x000354b5e753d742, 0x000c1ca06540fdb4, 0x000d75f42fa905c4, 0x000ac33e462b4034, 0x0008de3bb89b85ed, 0x00039131f413c305, 0x000000000006ba75}, {0x00046bc0a659bd1e, 0x000573f50020190e, 0x0006ea7059f1f5a4, 0x000aa10efa2adddf, 0x000e1a6566aa7297, 0x000b4f2143e8ccdb, 0x000ae74dfda07fd7, 0x000000000007cb1c}},
        {{0x0008cb34259001d3, 0x0001d49d1cfcc46a, 0x0008f80188e0d2b9, 0x000d285d2ab0a996, 0x00071df5b6cb8cd5, 0x00003fadd9c5cff6, 0x000c0019f4095a30, 0x00000000000aa463}, {0x0002c2d4a3652c8f, 0x00087f86673b013d, 0x000814cd905b85f4, 0x00075b44cc2cd5ed, 0x0000b0342517b376, 0x000817cd771d5262, 0x0006183af1f31657, 0x0000000000079199}},
        {{0x000066ce8afe9a22, 0x000815600fedffd0, 0x000fa276d4e1e61e, 0x00062674cb08e4e2, 0x00052e9ab52d8f12, 0x0005686adfab0c1f, 0x000e4e090b1ac94c, 0x00000000000bb10d}, {0x000c301d650cbaa8, 0x0003b3383f48546f, 0x00045006d2b6b386, 0x00092fe500058f01, 0x0007f4ce508ac5ab, 0x0007a5166f03c7c9, 0x000d66525ac2cea7, 0x00000000000996ef}},
        {{0x0002ee0f55870dbe, 0x000f522daed9eb3c, 0x0008bd7a619de4ba, 0x0007daa46ec27562, 0x0004df0f7a076593, 0x0006036c14c58eeb, 0x000aa5eb972393fa, 0x000000000004d17f}, {0x0005a44cb91701e0, 0x000a3de29c31e831, 0x0004699fe7072908, 0x000952e2c563ddbf, 0x000e5d6b0b57ac00, 0x000c4728203a767f, 0x000d56e858fdf3a6, 0x00000000000a5213}},
        {{0x000b5ed7b705b7ba, 0x0002631796c22d0c, 0x00099ed65d60e408, 0x000cb716cb0c301b, 0x0007d1702ca94ff0, 0x000a02a90dd0cdbd, 0x000625470a26c8e1, 0x00000000000d7054}, {0x000218ab60af89bd, 0x000e0fadc2e8673f, 0x0000890003c99982, 0x0005ebb2951a8fe7, 0x000364197a76042b, 0x0009c1b5de4af9d0, 0x0002b1dd2c6bb324, 0x00000000000f0193}},
    },
    {
        /* digit=18 [{1,2,3,..,}]*([2^72]*G)  */
        {{0x0004ea16a709d85d, 0x000329ded9b0ccd4, 0x0002e9bda4d583bd, 0x000aaf5393937290, 0x0002a438413e94f3, 0x0003c7de32213686, 0x000540449286da37, 0x0000000000029aa1}, {0x0001d592acb2cf64, 0x000ef1bd13e4055d, 0x0004681cce9d2c6b, 0x000558f6bcd0cb2e, 0x000e9e610369d43b, 0x0006e0651f5757c0, 0x0000aa15c80b23c1, 0x0000000000000182}},
        {{0x0004d40f93bd737d, 0x0008c10a8aabc8f9, 0x000d5b1177a72237, 0x000df1a076945a2e, 0x000c03861f02a009, 0x000866a3cf869152, 0x000cbbc405226e9f, 0x000000000008b41e}, {0x000cb1b314f36d3a, 0x000bab5c9ed101e9, 0x0005d83bc9ae2498, 0x000f22e57589279a, 0x000d7fe2d6aa1a9e, 0x00073dbfa83ef607, 0x000066f2da845434, 0x0000000000057879}},
        {{0x000f91047abd3166, 0x000593df98ea92d5, 0x0002ec06cefe6ff8, 0x000d1b5ea6d69771, 0x00097e25ffbcc9cb, 0x000a6280e3f9f231, 0x000713bca8e0567d, 0x0000000000035cda}, {0x000fe70b79caf906, 0x0003946ef321cfa3, 0x000b4a657c54ec43, 0x0003b4b51de40452, 0x000cccd9ba8d37b7, 0x000f7045b5bccf3c, 0x000dcca2ca080dd0, 0x0000000000068cf4}},
        {{0x000d265e2cd7626e, 0x0007e3499b44b9a9, 0x000ad7706fd7669e, 0x0000406c1f517d80, 0x000b82a174e83014, 0x0002c9b3ad4b3c8e, 0x0006f4c8835b13af, 0x0000000000044371}, {0x000fa90f05b33562, 0x000c2336c4f4b966, 0x000385e7d9f805d4, 0x000a2dc828c34d3a, 0x000c34d0d1f76ffb, 0x000a4fbce257a345, 0x00049fbc1eff056c, 0x0000000000007683}},
        {{0x000db1540add72ad, 0x0009e5ebdc79cc93, 0x000baaf845cf7653, 0x000efe79c8b7f7df, 0x000ac469aa5793f6, 0x00033c455b2f1e40, 0x00012a882f0e9434, 0x00000000000aea07}, {0x00030f4a7dec8786, 0x000e9b13578c31bc, 0x0001af154c9f03fd, 0x000a2d854e09133c, 0x000cc6ad853283ac, 0x00004f75cae03943, 0x000c37ce0a619171, 0x00000000000f9838}},
        {{0x0004bc7eb2d5ec5c, 0x0007e9db114e2be8, 0x000c421a1f0f05da, 0x000016bd2f3080f0, 0x000fb9b7203704df, 0x0004fe2ab0cb3f7c, 0x000ec1adb877c781, 0x0000000000037761}, {0x00081bf9c70c3c38, 0x0003cb5d6068d8ec, 0x0006b75a387ed168, 0x000aa5457d36d422, 0x0007423cc8cb39b4, 0x000a6566dec1de46, 0x000f55280e02dafd, 0x000000000003ca55}},
        {{0x000e1cebdad8aedb, 0x0006b340d77c218d, 0x000b7391085bcda5, 0x0005b028b74d32db, 0x000a3cc1117e3e91, 0x0006c5939ae7101c, 0x000caa3c36152b52, 0x00000000000d3ec0}, {0x0001039f162ada74, 0x0004534de058e1ef, 0x0005d23a486e5201, 0x00030efc214ac0e4, 0x000cc7100acb27c0, 0x0009097b72a216d1, 0x000aaf3b73004330, 0x000000000007afd8}},
        {{0x000af1b588df6219, 0x0003a5d7afe99881, 0x000b20bd69a8d3f2, 0x000067dc65234936, 0x000527557f5de39a, 0x0000e9f9d3458527, 0x0002d8756b8d8693, 0x00000000000040d5}, {0x0003845ac0bf2beb, 0x000a9606eb60a92a, 0x000238ad785ef768, 0x0005748e068d2949, 0x00022fce158da621, 0x000183c8f73fbdca, 0x000c989cfee07dcb, 0x00000000000a5e03}},
    },
    {
        /* digit=19 [{1,2,3,..,}]*([2^76]*G)  */
        {{0x0000e52cf2e9b4ad, 0x0003f86230933fe6, 0x000fe37b1c8aab75, 0x00086ba6f595ed40, 0x000de2284a5801e1, 0x0009e5e25d3291c3, 0x000f5303dee2311b, 0x0000000000015cc9}, {0x000f15883981ad15, 0x000444fba15675de, 0x0008ff144b8a465b, 0x0001eb92532d015f, 0x00084e7455de8690, 0x000f811c94396fd7, 0x000b372fbcea8fbc, 0x00000000000ae952}},
        {{0x0007195bc501819f, 0x000f06bd2c54c87b, 0x000c8e0d99fb53f8, 0x0002654e3eee748a, 0x00039598d90727f5, 0x000cb371256f2058, 0x000c5b5f91c2d027, 0x00000000000a3e33}, {0x0002b15fc69c25d9, 0x0003e248490a9884, 0x000516ef1dee14b6, 0x0008b3d74f7d38ca, 0x00002fb602142013, 0x000033d4eae791f6, 0x000f176b4fb300bd, 0x00000000000bdfd1}},
        {{0x000558e7c27dd125, 0x0004d9bd2b68cfc0, 0x0001783c86b4626f, 0x000f35989586da1d, 0x000afd0944ca67e7, 0x000569b5abc04c8a, 0x00046eb8db5424f0, 0x000000000000cc55}, {0x0008ec9be17ec8dc, 0x0009e8f0ef9decbe, 0x000cdc094579a394, 0x000eb644e7329b8f, 0x0007e6896b042ec1, 0x00018b6bf3e0e7f3, 0x000fa6ad2bff11c1, 0x0000000000040f93}},
        {{0x000fa586c442030a, 0x000a30c3d2055f9d, 0x000061ff065411bd, 0x000a98b1fbf56cae, 0x000f396e31cd1b68, 0x00082c03b56fd85f, 0x000917d2ca584443, 0x000000000007d3ef}, {0x000572f96fe58801, 0x000c74129a730d28, 0x000ce4a000ce9071, 0x0004180653f3b241, 0x00049ee066bd9e85, 0x0002ea5fee65a1a1, 0x000e2e3420084e36, 0x00000000000c7911}},
        {{0x00079bb68a21bf45, 0x000b04c3f4f487b8, 0x000f5bdc807ffd5c, 0x0009bad1844be875, 0x000e272afc140fbd, 0x00040a7386b3b80e, 0x000a37d727b0b6aa, 0x000000000004d809}, {0x0001886f07e4e9c3, 0x0004c09fb0640689, 0x000e568c8f4da447, 0x000ff8ad73a211ba, 0x0001d9a6b97da7e5, 0x0009e8f56ccdbf62, 0x000a0938448e2cc8, 0x000000000002126d}},
        {{0x000bfb3f1befeb61, 0x0004c43f0ae67505, 0x0007b68cdb6c3beb, 0x0006c25dbb4492ec, 0x000884bff97fd5be, 0x0002da2e27efc9ee, 0x000c3254fbd9d706, 0x00000000000c6d2a}, {0x000d04af3852eaac, 0x0003e14cf6dc0935, 0x0006b6f8f3aa240c, 0x0006543d9a16800c, 0x0009beb1e28dd056, 0x000e658339cc2ade, 0x000cb2cbfeb45038, 0x00000000000d0f5a}},
        {{0x00014927529fad7a, 0x000714f10299a19b, 0x0007bc14eebc4178, 0x00018bc0cbede41f, 0x000e6b03ac7d024b, 0x0005213be7883c01, 0x000540a7b99e741c, 0x0000000000085a97}, {0x0007979358dabc76, 0x000060b6fe59caa3, 0x000a32d2efcf0218, 0x000969d6ad1891a4, 0x0003ee7761dccaa0, 0x000b12539c47afe1, 0x0005f4cae681fcdf, 0x0000000000002ea6}},
        {{0x00033b53cfd1d032, 0x000865e930d64f49, 0x000e6882e13322aa, 0x0004f7efa08bd4a8, 0x0006a6e10f64cfac, 0x0007ac162277e845, 0x00068b41be306740, 0x0000000000052a9b}, {0x000cf81e872685ed, 0x000e28d8d216a150, 0x000406cac9d54222, 0x0004102c60d54c43, 0x000028815187ad9a, 0x0003b8e59b09420f, 0x000ba1881179c60b, 0x000000000005b09a}},
    },
    {
        /* digit=20 [{1,2,3,..,}]*([2^80]*G)  */
        {{0x0002bb0ebd1ffa8d, 0x0000c22313ded9b3, 0x000c76c37673c0ae, 0x000eeaea76ec3812, 0x00060275ad3643d6, 0x000095bdaa165513, 0x0004b0e5c1b65adf, 0x00000000000367c4}, {0x000f3876ae1976b5, 0x0007de2b419c1bd2, 0x0005d5344cebbb0b, 0x000e51fa2baff060, 0x00006269b5d0b5fb, 0x000e8d667a0594d7, 0x0006e70b07b70523, 0x0000000000063e01}},
        {{0x00082243f0134f99, 0x000d9022aa09c335, 0x0001e1b9a8966dc5, 0x00044c5b38b417ba, 0x000b436451ec3173, 0x000b1c87629a74cd, 0x000ffdd898eb6ca8, 0x00000000000b74ee}, {0x000187e3675bea14, 0x000410d9eab63bb0, 0x000dd59e68360fb6, 0x00094ce14dbff2ff, 0x00092706528037c8, 0x00029a0cd3704d34, 0x000c4f9ee435d8ba, 0x0000000000009c11}},
        {{0x000414827bc8962c, 0x000552fde4893802, 0x00000bdd0aabfbb8, 0x0008a5b09456ed5c, 0x000925797c6c157b, 0x000c14c0673606a7, 0x0001caa9d0f47c33, 0x00000000000faf97}, {0x00082453cb3879aa, 0x000ec6e763c509a5, 0x0006f45dc5dfe238, 0x0007ecbc49e9efce, 0x0007b38f2b59d95b, 0x000c397db0c31792, 0x000bc797912b53b3, 0x0000000000045c7a}},
        {{0x0002600fe807b0d9, 0x000578b3269a1894, 0x000cd6a11a8caa23, 0x000c714e428865d4, 0x000ffdaba094bc5a, 0x000d657f2531dd17, 0x000bbf86d2405718, 0x000000000002f99d}, {0x00063a9cad22b4ec, 0x000d1c428a80ebb2, 0x000111bbf67dd9f1, 0x000691922a5d2566, 0x000318a18586a752, 0x000f633b3bfe6392, 0x00029828a0c772fc, 0x000000000003f3c5}},
        {{0x000cdc4baa08bf36, 0x00052cd81bdce7a2, 0x0009dbe6198554f1, 0x000228bf8ebe39d1, 0x000eb2cdd8524cb0, 0x000e9b3af496a9dc, 0x000f8a0f115a1ad0, 0x00000000000d8eee}, {0x000de9efc567fd4a, 0x0009b5232a514417, 0x000762e720496fa2, 0x00018b08994b4e8e, 0x000809097a52289d, 0x0004621d0b4e346a, 0x000ab8c641510f32, 0x0000000000095a41}},
        {{0x000c5913c30a847d, 0x00004c8c9b0adae6, 0x000c6088d1ec2259, 0x0007946e62c508f6, 0x0006e9fe2321b03b, 0x000c1d1366042592, 0x000c9c73b10bba80, 0x000000000009d218}, {0x000e72b1483512ee, 0x0006d6ed0c8eed83, 0x0009abfdad8b62ec, 0x000096ab9e96167e, 0x000cc473e3773078, 0x0003a3bc8b28f0e6, 0x000796b44e499568, 0x00000000000d3523}},
        {{0x000ec81c53fdc5cd, 0x00095261578c7817, 0x000dba0eec8c348b, 0x00066b414c803900, 0x0005faa7cc399932, 0x0009b7e27bacad75, 0x000533418c4defd6, 0x00000000000ae751}, {0x000b26e534337f55, 0x000f36f3bd3298fb, 0x000bd59fe7197151, 0x0004f73dc4c54ecd, 0x000d44d3d74eb716, 0x0009ea511ca66290, 0x000c49f77c62424c, 0x000000000001f714}},
        {{0x000343bd88e025df, 0x000b893a62e7c716, 0x0003b75e6fa4a448, 0x0003cbb10cc95456, 0x00062f37154c265e, 0x000f2e08b28cceb0, 0x0000916b79b0713b, 0x000000000003ab39}, {0x00046ae52480b7d8, 0x000a750a8d04cede, 0x000f3c798fdbfa72, 0x000336ac21cc62f0, 0x00036fc004503db4, 0x0004b2d954747f0a, 0x0007a7067cbf1c54, 0x00000000000fd2be}},
    },
    {
        /* digit=21 [{1,2,3,..,}]*([2^84]*G)  */
        {{0x000f8f632d3f21bf, 0x000880233e423a7e, 0x000fc3a5113e93d2, 0x0009cef6349e35be, 0x0003ded2ec542ca0, 0x000dded3d3b70afc, 0x000ee0c813474d52, 0x0000000000012f00}, {0x0002c304534f52c2, 0x00002a0908f763fa, 0x000cccb3aba1eb57, 0x00010c134b6a8e10, 0x00056e7c163e84ad, 0x00057e0c36f37fd9, 0x0001a27caae8c8d0, 0x0000000000055372}},
        {{0x00086fa76217d20b, 0x00015a354c1de108, 0x000a55c4648be905, 0x00010999d532bbe4, 0x000f520117516766, 0x00059e35eb7c2f4c, 0x000b7b6945d34ff6, 0x00000000000a1303}, {0x000495f21c8085bb, 0x00028a200ddba08f, 0x0003caf84b1e53dc, 0x000445faa82b46fc, 0x000cfccedc1b3018, 0x00053e29fc3e6a3a, 0x000547e86ae87033, 0x00000000000fd7e8}},
        {{0x000683184a93e477, 0x000a4728c8e14128, 0x0003d8bece755c7e, 0x000c2da4796df734, 0x000e00a55efc3016, 0x0006b1cb8f6d79e4, 0x0005a94fced26952, 0x0000000000091f16}, {0x00031165825c7f6e, 0x000cf528b857275f, 0x000d8cb6dce0ced4, 0x0003520e07bb990b, 0x000f4ea8ae6b4f90, 0x000b735bb07ddb97, 0x000d4a29c01e70b3, 0x000000000005000d}},
        {{0x000f814f4e7d6929, 0x0005e306b63e3665, 0x00098d89bb61e0d2, 0x00068898bdeddea9, 0x000d67a329ac36bb, 0x00092ccce9331655, 0x0009b414c7fe26ba, 0x00000000000be051}, {0x00042824d37928be, 0x000830dc39dbc42d, 0x00083d595f84e0c5, 0x0008818fa8682bba, 0x000db60aad97c7f7, 0x0002c64cc43396ea, 0x00043e751784d7ff, 0x00000000000866af}},
        {{0x0007b94e1c1e3bf9, 0x000a53742b61de85, 0x0004d0debb63955e, 0x0006787abce34016, 0x000ea4cb41b4f52c, 0x000ca817e4749711, 0x0000487ce0dfb03f, 0x0000000000096e85}, {0x000c4c6d7efe5c7c, 0x000b6cd65250dc1f, 0x0007ec2aec96d815, 0x0003cb2b4dd561a2, 0x000abb5379f57352, 0x000a847ba15ba773, 0x0002decc5e62d6c0, 0x0000000000004d85}},
        {{0x0003450b4a8e849f, 0x00036f0a69e68100, 0x000d9356a0504fb6, 0x0003ab7cd14af1d4, 0x000d558d555d773a, 0x000e63dd6c0b88ed, 0x000db3eb12d197d2, 0x00000000000bcd9c}, {0x000ed15d1fefd8b7, 0x000fa06432985766, 0x00052d6c3d3a4a47, 0x000f6164171b9bc6, 0x0009bbfcf90abd28, 0x00036084d3778163, 0x000a751f93898f3a, 0x00000000000dd00c}},
        {{0x0000e82da9d478f7, 0x000ae3b8d8e48b47, 0x0000534e09405090, 0x000430c2aad8c684, 0x0006b82c16731d72, 0x0000fb41de8c5d50, 0x00034f989978868a, 0x000000000000f812}, {0x000d9518cbfe70fc, 0x000f1a0c941ce78c, 0x000c323a8cf0aba1, 0x000e296101f89af8, 0x0003ca6a3227aa7d, 0x00064ec0c4cadd8e, 0x000db43b4db51f27, 0x0000000000009256}},
        {{0x000e28a122d0f32b, 0x00022b9207dbc085, 0x000dce5b9cc6b932, 0x0004e5534ba150b8, 0x0001dfec99724992, 0x0006f870d2935ebf, 0x00085c34bf5e94b7, 0x000000000004c203}, {0x000bdbb9ed40b2b9, 0x000c3c78ac1719f8, 0x0008866adbfa0f7d, 0x0007ee4cfb4b8bb6, 0x0002cd8724b676c6, 0x0000bc1efecfaefb, 0x0001b5e9bc3d6734, 0x00000000000ca554}},
    },
    {
        /* digit=22 [{1,2,3,..,}]*([2^88]*G)  */
        {{0x0004ccdc054842bd, 0x000704f293478100, 0x0000ca74cad5c267, 0x000c94f64d55f6f2, 0x000d725c51251bf7, 0x000b6a0914489b57, 0x0000fa3aa4d43ab1, 0x00000000000cf7a6}, {0x0007fef943134dbd, 0x00038aa918a4ce7d, 0x000eee4d56ef907f, 0x00000a16812a03d0, 0x00070b0a1e4bed0e, 0x00028386bb09acf2, 0x000321c73a41f7ac, 0x00000000000f4cd1}},
        {{0x000b2804db21ede5, 0x000fd10a53b8f9a2, 0x00080087f23a6e5f, 0x000513fabef2b974, 0x000cd3f802fd740d, 0x0000627afff6cc47, 0x0002dc1be6825beb, 0x000000000005886c}, {0x0002ebc44567c313, 0x000afb6cfb2c6b07, 0x000e1dddbc404a5d, 0x00016f53a5a485ba, 0x0002e7b2223e7a1e, 0x000a13a9b16b60b9, 0x0001a2332f33d836, 0x00000000000876cd}},
        {{0x000bcac1cf1b2eda, 0x00095802f495c1f2, 0x000e55ffc912d44b, 0x000b294e9b3f0f6d, 0x000033cbdcd1eaf3, 0x00083fce582cc760, 0x0004cc1a5642278e, 0x00000000000ffa0e}, {0x000dada72c71a95f, 0x000ed7d93e9766ea, 0x00015d2d9c24e57f, 0x000dd79eafe5f361, 0x0000fcaafabbcb53, 0x000053efecbbed8f, 0x000b7a570472705f, 0x000000000001ebfe}},
        {{0x0005f3669cd44c3c, 0x00025a896d28dae1, 0x0002e7c6820a84bb, 0x000fe8c06fb15cd6, 0x00061cdffdad1bba, 0x000119b3a2daab29, 0x0005a3984defc8e1, 0x00000000000de2a2}, {0x0000f914a19c745f, 0x000b3a7f54ecdb35, 0x00041f75def22ce4, 0x0000efc6a3b94437, 0x00076785163b6913, 0x0002b1d0a5d17f12, 0x00093cc019911b17, 0x000000000007e3e0}},
        {{0x00049fc81e69d5e0, 0x00085d307d4279b4, 0x00039a33441f2443, 0x00078371531d6263, 0x0008b42639ecfff7, 0x00099fb76fdd61b0, 0x000f0eca9462e7be, 0x000000000007c3f2}, {0x0008f656f3641bfa, 0x000f85c07066bcc1, 0x0008bd883c4e6617, 0x0001387d93ff4937, 0x000b864eed19b6d3, 0x000fad6382fd7d67, 0x000fe667b707ff5a, 0x00000000000d0206}},
        {{0x000bb568a2a2f3e3, 0x000a9235455295a9, 0x0008072539522964, 0x0006d10c75720aab, 0x0009f6a0df88ac83, 0x0005a9c96ea2c538, 0x0007bedfbf31519e, 0x000000000008cad4}, {0x000d6b001091ce3f, 0x00036dbd5de982e7, 0x000e935fa43e76d4, 0x000fef6324ba9235, 0x000e8f14a2f5a2bd, 0x000880ade8b0f981, 0x000d4a590de1c45f, 0x00000000000107f3}},
        {{0x00091e8b5ebad75d, 0x00009b03d47621b9, 0x000da50c79b05bb6, 0x000ca72548723c56, 0x0007c944e3849da9, 0x000947973a12e40d, 0x0001ea403dbcf2cc, 0x00000000000c6ef4}, {0x000f1cf0966fef48, 0x000d74ccc2856b9a, 0x00011bbeae1e30d2, 0x0007f7e43ee1576e, 0x000a0a894961701a, 0x000396f6e7f061c1, 0x000ca956059fb88f, 0x000000000001a91f}},
        {{0x0000a27b9e3336b7, 0x0000be185bdc20d5, 0x0008a8bb1575f81e, 0x0003397c3fe5695a, 0x000f61f6a04f1f6f, 0x00090a92b6c9f3a5, 0x0004543c0f9d4bb3, 0x000000000004793b}, {0x00055e696b3d9202, 0x000d991ad0c33a93, 0x000c4d967cf46a4f, 0x000d90565d53c83a, 0x00034176007dfa24, 0x000d1623cd63e583, 0x000345741089fd2c, 0x000000000002685d}},
    },
    {
        /* digit=23 [{1,2,3,..,}]*([2^92]*G)  */
        {{0x000883b867d2fbb5, 0x000271d51015761f, 0x000afd0328ae7e6f, 0x0007e2fc6dde9e2b, 0x000937d5bb1f2ea0, 0x000cc6e28ceed9ad, 0x000fc43121994b98, 0x0000000000067ad8}, {0x000d53a60b81c5b7, 0x000ca34707fb3d11, 0x0005e1e4ff2b5765, 0x0009b8d81ef6cb57, 0x000b1223a9a1fe93, 0x00074cf375c51e8a, 0x0008e18f6d7993db, 0x0000000000097280}},
        {{0x00067952d0358e02, 0x0005b0db3322730d, 0x000098edfd5f000a, 0x00085380e8a38fae, 0x0005769360d86efd, 0x000f27ef5c1812ea, 0x0002e8505723dc76, 0x00000000000f35af}, {0x000764970d5cb557, 0x000849f7bb7fecc3, 0x00052323da5066c3, 0x0007549773eaf7f9, 0x000489d51bda0118, 0x0000ccde605bbbba, 0x00087d52cabb0664, 0x00000000000e7ff3}},
        {{0x000fa071b96c7c7e, 0x0009c70187dfb0a2, 0x0000f19decc5bd43, 0x0008b816a6c0d616, 0x000d6a27e97b3018, 0x000a389aaeaeda67, 0x0002734a5192826b, 0x00000000000a2bf1}, {0x000832bf68c7b455, 0x0000c8ff10656bdb, 0x0008bc56c7023546, 0x000a3e11dc7aed54, 0x000b38853d61e34b, 0x000cbf658c457048, 0x000edde89825db1c, 0x000000000006b255}},
        {{0x00067336ad6a8789, 0x000d0c377b3730db, 0x00063e31dca6f13f, 0x000f2adf39b68951, 0x0006bd6da22ed9c2, 0x00009678e1bbff10, 0x000e783f7e5b8e7e, 0x000000000007ed3e}, {0x000a0e3ab40df97f, 0x00071689e6b30fb8, 0x0000b670f8712d68, 0x000277edda78c55a, 0x000cefb65cb717d3, 0x000a30f0ce686cac, 0x00071e758aa1ab19, 0x00000000000b11f0}},
        {{0x0007d3d2024290f7, 0x0000c0bef2d25159, 0x000b908d7c494e4d, 0x00017cedd4af7aeb, 0x000bc8e88ca42c24, 0x00020d15452b5b3b, 0x000d4bc00328c44f, 0x0000000000030af5}, {0x00053687c2073cc9, 0x000dee8bacbeb2cf, 0x000bf613b57931f9, 0x0001da61d9e70314, 0x000c0e70a331b67d, 0x000e325772ddefcb, 0x0009797d09bc6d61, 0x000000000005b52e}},
        {{0x0000d7a5438e4dd1, 0x000b5d40f36e5e3e, 0x000702a395a7941d, 0x00094d7cc51dbaf1, 0x000f0edf8bf9e340, 0x000b5e16520f93f0, 0x0003a8534b75aa23, 0x00000000000dc1b7}, {0x000196b771cc1d7e, 0x000b09dd48c40ca1, 0x000664851f0b3766, 0x000f8a2a9ffa0db3, 0x0004d94ed9bd4212, 0x00069cb198d5236c, 0x000dfc81bec489f1, 0x000000000006104e}},
        {{0x00076550d0174653, 0x0008d9cb21b56311, 0x000fe59ad5e35eee, 0x000a67100d7b34c9, 0x0001a6c1b0c3e3f3, 0x0003c3e31d4bcb3d, 0x000be86ebd26fb9f, 0x000000000002dc79}, {0x000582f4b4dfea09, 0x0002ae7cf9a0a1f3, 0x000c37b50dcbdab8, 0x000320e69e1f08bd, 0x000b946f83d78b0e, 0x000a399ad2a8caf9, 0x0000866daff0ba98, 0x0000000000038ed6}},
        {{0x000af81ab2009a09, 0x000fc16f3708ae33, 0x000f381b02290f16, 0x00015b227bb71a8b, 0x0004162f8b5597a6, 0x0001e3fb9a8b9a86, 0x000ff680ee8362d9, 0x00000000000f83a4}, {0x0005ffc830c362b9, 0x0009507a7873a90a, 0x0008c2637b53ccc0, 0x000683d860d60ba4, 0x00071f87d32a427f, 0x00050319dea99592, 0x0009ceb2ac69faa0, 0x00000000000d2d0b}},
    },
    {
        /* digit=24 [{1,2,3,..,}]*([2^96]*G)  */
        {{0x000fbfb66a502f43, 0x0008324480c57881, 0x000d6a55d7a45e41, 0x00002c3273e2bc82, 0x00053ef1ed4c7350, 0x0004948e88c42e9c, 0x000632028babf67f, 0x000000000008a978}, {0x0008edce917c4930, 0x0001bf5f13a4625c, 0x000e9dd4dc6a263a, 0x0006bcc37232768a, 0x0001576e8c1830f3, 0x000bcb766c5317b3, 0x0004dcef1d57a69b, 0x00000000000b3e3d}},
        {{0x000aa97e11eb2f75, 0x0004f53ebbaaea28, 0x000f59d2921161ee, 0x000f25d340986b44, 0x000e3365312a3c15, 0x000641f96c5925d2, 0x0008e4c35d3ee251, 0x0000000000098412}, {0x00032040e2f19c59, 0x00029b41a21a75a1, 0x0005c4225a472860, 0x000f4c0faf5663dd, 0x000a0a62c9dc4ffc, 0x0002d60c5889d285, 0x000be50908665acb, 0x000000000000a131}},
        {{0x0006f87a21caa381, 0x000a6318feb4c006, 0x000ed4b6b2c3b939, 0x000786d5b2386621, 0x0005a722c5911e4a, 0x00006cb0188e0430, 0x00036438eb062af4, 0x00000000000e7216}, {0x0003e9f2fb9e0c73, 0x00093450c1d28f54, 0x00023904331c181f, 0x000a53a6f6c06813, 0x000a25eac032fe46, 0x000c497224553199, 0x000a2941fa659aff, 0x00000000000bbcb7}},
        {{0x000a3dcccc791733, 0x000a8c0de60f43c9, 0x0004454fef6e3630, 0x0002aa8e734e2e0a, 0x0003ef773db6f412, 0x0002d71c508f40f6, 0x0008250a7e9484ce, 0x0000000000078a3f}, {0x0005a99063a64539, 0x000ef0cac7a5b50d, 0x000064e61e079dfa, 0x0009a90d3e181458, 0x000a0aadf689ecf1, 0x000f0acf70d1b062, 0x0002b942299f1ff3, 0x000000000005ac25}},
        {{0x000b4c356075ce0d, 0x000a8a4bfe150fb5, 0x000907dbaf6f1c40, 0x0008260a0ee708f9, 0x0004abc9d5f541e2, 0x000aae99eff4fdff, 0x0004d96ed92241ff, 0x00000000000c04fe}, {0x000731763369357f, 0x000a8f10a173e2a4, 0x00065723921ac826, 0x000823f8ed02e85d, 0x00042e522dc1d2a0, 0x000c939fdffa78cf, 0x000bba07041e4600, 0x000000000003a9a8}},
        {{0x0008012ba5de4e7d, 0x000c7a1fc20f7144, 0x000540a292e6e434, 0x00081c28e2e8e16b, 0x000a4c4562342e55, 0x000e31ad4912c80f, 0x0008117fd61fb9e9, 0x000000000001c19e}, {0x000de815bbe3ce38, 0x0003b6a7ab2cb6d7, 0x000aeaa95e1796a8, 0x0006a85ab06af4e7, 0x00044866378d33f7, 0x000826f8cda387e5, 0x000cb6bb71deb856, 0x00000000000eb64e}},
        {{0x000d7e72d8af89da, 0x000ba0e93593424a, 0x000584230569d130, 0x0003dba8e15d8645, 0x0001cab64644ab77, 0x000fff0619cea4a5, 0x0006a4516754d72e, 0x00000000000cd3a3}, {0x000ee6af7eb6909f, 0x000c6c7d352abc75, 0x00007221eaa0649d, 0x000ed979199938f5, 0x000acd6f34958a8a, 0x0005f93eeeb06789, 0x000851c51d776677, 0x000000000000a8af}},
        {{0x000b089742823cef, 0x000c544bfd166199, 0x0008c58b08cf9fc3, 0x000cf81c39366838, 0x00068b08680fe50f, 0x00032aab0c9b4eb0, 0x000ddaf90882c528, 0x00000000000ecbf5}, {0x0004bc1447794cdc, 0x000f8fd4d51c6fd3, 0x000cc355ce034b68, 0x00053dbfb187c34f, 0x0006f8ed037fb729, 0x0007bad45166f7f2, 0x000808d2b1077a09, 0x00000000000790dd}},
    },
    {
        /* digit=25 [{1,2,3,..,}]*([2^100]*G)  */
        {{0x000273ed4a6f3c1a, 0x00092a05c751c876, 0x000bdb5d554c8320, 0x000e26cf4c98fee4, 0x0009e7b3a7c79c56, 0x000cc16466084f8b, 0x0000bca0b042f6de, 0x00000000000fbf3a}, {0x000078eb34f9801e, 0x0006694524c2c1cd, 0x0008ad2d5e6f66b7, 0x0002d6e7ce1f9f62, 0x0007fc0c34fa1879, 0x0000a16938ebc2fd, 0x000019032a9f4178, 0x0000000000056a60}},
        {{0x00009be19d72cca6, 0x0009ea4b957d1c55, 0x000c88712c99979c, 0x0005e80a271d9e14, 0x00044b9367e51157, 0x00079779f0fff68d, 0x000f68f55ba67322, 0x00000000000d4d68}, {0x000ebae305d14369, 0x000b401474ab1533, 0x0007e0884a0d1ea6, 0x0003a8336111d25d, 0x000fab531cef6390, 0x0002ed86737342a5, 0x0007e1d3c93fe770, 0x000000000006279f}},
        {{0x000ed609fa3c6148, 0x000beb398290bf69, 0x000b5366c9b47f2a, 0x000aa2956560b89f, 0x000c50e6647b3e71, 0x000be42da80817aa, 0x000a4e35c833ada0, 0x00000000000f1bab}, {0x000a4dc6e615148d, 0x0009ac57be644e77, 0x000de3f8f9ac6cc1, 0x00077310defd9095, 0x000e899cba568300, 0x000effb5a92fe9bc, 0x0004bf48450ec4bc, 0x00000000000f2f5f}},
        {{0x000e1a6a67c12c1b, 0x000e7dccf68bc63e, 0x000af5107f8ecb57, 0x000e199254dd3e41, 0x0006ddce5c84e43a, 0x0000b4258e8526e2, 0x0004c452bad815ea, 0x0000000000009457}, {0x000fc838b5157d7f, 0x000e1a5362fcad0f, 0x0003a5c9b577c868, 0x00012b5eeffc2cdf, 0x000b14de4e5b0f93, 0x000233fa55475657, 0x000ec2746e67d4f0, 0x0000000000035471}},
        {{0x000afd6c233d63fa, 0x0009cf55edabb6d7, 0x000b353b7e2a9194, 0x000822b9bf171614, 0x0003afe808dcf53e, 0x000272f554a5b3b9, 0x0009a1590bbbded7, 0x00000000000eaea7}, {0x000101d093edb837, 0x000b5d7042bea6a3, 0x0006a766d7a462c5, 0x000e69031078aa66, 0x00087e37bc8bd8d1, 0x0000089759e837f1, 0x0008679558ff4f85, 0x00000000000421fe}},
        {{0x00076617121c3ac1, 0x000a13503d9370d0, 0x0006e1d369d53e18, 0x00001ef5b52e0f07, 0x0002f955b5fa67f1, 0x0000e0569b6b5b4c, 0x0003efb5f03d5388, 0x00000000000c9939}, {0x000c81fd72768469, 0x000a690755bd5e4b, 0x000324dbfd474da1, 0x000fd519e9ce792e, 0x000396ccfa14326f, 0x0004a22de1b772d7, 0x000342652710f487, 0x00000000000db210}},
        {{0x0008597293bd01e8, 0x000dfaa6489def01, 0x000dc630321a7c29, 0x00043a95fa2efed7, 0x0003720e1a8c4d89, 0x000a4f8a3c9baae5, 0x00006bc055444b95, 0x00000000000a7c12}, {0x00017016dee8b7c9, 0x000dfd97765b5926, 0x00050b911df827d2, 0x000c35af1503b16b, 0x000fa9fa21f8115c, 0x000fed1f11dd5359, 0x000679197c32996d, 0x00000000000f77f2}},
        {{0x0004d7575fc73ce2, 0x000ce58d6998db75, 0x000d11d5d661c62a, 0x0002a0165739fcf5, 0x000822f7a073420b, 0x000042f005c5db30, 0x0003d9016c547805, 0x00000000000a1241}, {0x0008aac8eaa9ff4f, 0x000fe8ceb2d6ffe8, 0x000f806ab6efe1ae, 0x000d012339c146c9, 0x000628ac05552638, 0x000d46e6302fbb7b, 0x00088e7fcb0c8162, 0x00000000000066d0}},
    },
    {
        /* digit=26 [{1,2,3,..,}]*([2^104]*G)  */
        {{0x00029831f2b8ed10, 0x0004917e3d0b77e8, 0x00062b9d59b96ed5, 0x0000261d0bb1849e, 0x000ec71a3bbef8ef, 0x0002a88ae9b804ce, 0x000b9e00b7d17154, 0x00000000000e9097}, {0x0001f74d24b8094d, 0x0000cdbad9f12075, 0x000917cb364517bb, 0x000834fb83debf54, 0x000d637e449ba8a4, 0x000b82664c3ee226, 0x0000e8e3070d93ca, 0x00000000000b3732}},
        {{0x000cd133b578c566, 0x0004bb506b7b86f8, 0x000a189da5b723c8, 0x00029ad3789281ee, 0x0001f9af456164b7, 0x000f9186069ae8f8, 0x0000fb007befe156, 0x00000000000edc25}, {0x0003b5e8dc3a9219, 0x000ab1c16d7598b7, 0x0000fdec987e0e87, 0x0007da936881f89e, 0x000103a5b1ea5dc9, 0x000153a0faddb603, 0x000ee27eb515b13e, 0x000000000002b4a1}},
        {{0x000bcceb06f924be, 0x000d4e6b739c7071, 0x0004a4fc7ff778b8, 0x0003acc1b4b45493, 0x0008f6bad44a14e3, 0x000020972d6704b2, 0x00033543cd7e5ddf, 0x00000000000e0d6c}, {0x0008ad0a64850139, 0x000b67d395c3ef6a, 0x00040e40560ce9c8, 0x0009f21efe4a5757, 0x0004f55ea727b626, 0x000450c823f97b37, 0x0005d5b689b4de42, 0x000000000004f5ae}},
        {{0x000bd2a72ae0f683, 0x000d7e7365f9e264, 0x0009dec60e7c539c, 0x00089ac8e611fc3c, 0x0005699c227dbfb3, 0x000f2ef17ce778fd, 0x000ed4dc1f47b63f, 0x0000000000002672}, {0x00013b4caed73e7f, 0x000a551c63805fa1, 0x0002b72b6fe7e133, 0x0003d13e00e12a38, 0x0000bf8df325375b, 0x000f5f21a91f9f75, 0x000584f1ea0e421d, 0x0000000000032c60}},
        {{0x000c6a3d385d54a6, 0x000a26d720050d7c, 0x00008151ff431078, 0x000aee6f1c89ce3d, 0x000b57b2b79d5d1d, 0x000c7ce823f7644b, 0x000b0bb1063f92fc, 0x00000000000c15a9}, {0x0004b892e4ed0e8c, 0x0007865661a5290d, 0x000a266171fca5c3, 0x000dc5d42ecede66, 0x000c8eb0689e0661, 0x000d7ff77be586ca, 0x000c3bc86641a02d, 0x00000000000b9ef4}},
        {{0x0006b862a25bd60b, 0x0005903b07fb53e1, 0x0000603f2bb26df9, 0x0006449356376454, 0x0002a55bcadf272e, 0x00071c6af2b6ad1c, 0x00080686c527765c, 0x00000000000c1678}, {0x000fb64422cb7cb4, 0x000660cae829453e, 0x000b68619a56abb9, 0x000f8337513dfb65, 0x000e1b351dda504d, 0x000af7cd04260ee8, 0x0000fda473c5d9db, 0x0000000000071e39}},
        {{0x000c98e3f8d5e928, 0x000c217544e9789d, 0x00047053e368c072, 0x0000e3937af8cf8c, 0x0009a2f308e2e146, 0x0005060e98b4aa2f, 0x000a7dafa2bdbc3a, 0x00000000000f84dd}, {0x0004961384232a8d, 0x000d53b256b55aeb, 0x00063be03fa9b44c, 0x00010d13c8e52f06, 0x000b38865772c405, 0x000416f1062c1542, 0x000eb54755de7ebf, 0x0000000000097aa2}},
        {{0x0009115b2961da7d, 0x000f01d9aa9f9fd6, 0x0004c642e6501980, 0x000d6a17f167b479, 0x00085bd9153d7ebe, 0x000682ba6324c13f, 0x0006e2e9ea5b734c, 0x000000000007e3ff}, {0x000ac4814edd792c, 0x000361ed2b04fb2c, 0x0001100e4a13806f, 0x000a5d3d68b8ec70, 0x000d1c6ce0d2afce, 0x00019410c21dc058, 0x0005340043de75e7, 0x00000000000e15cf}},
    },
    {
        /* digit=27 [{1,2,3,..,}]*([2^108]*G)  */
        {{0x000e0e09fdbcad95, 0x0004ef7aaa73197d, 0x000fdf5aa02c7c2e, 0x00017372e0c286f0, 0x00025da3472da1a4, 0x00076f2a23b66850, 0x0007d4bc0d116b75, 0x00000000000a36a2}, {0x000b11735059e67b, 0x0001ecdb3f4744cf, 0x00033adc74dab02a, 0x00011b969d4f1723, 0x00066cca4ef7c70d, 0x000c6afdfdf96d13, 0x0004da570693db2f, 0x0000000000056750}},
        {{0x00080567673bd755, 0x000df02c6b607d3c, 0x00025bebf3a2d95d, 0x000b01d4f57eb0c9, 0x0004d750e515af5a, 0x00089ad6859935b9, 0x000eff32721b408f, 0x00000000000a7e3c}, {0x0009a7a317218bf0, 0x00010bd462fd8d81, 0x000035ae54257b2d, 0x000da619263fa61b, 0x000eb6488eb34162, 0x0005680c42ed1f2d, 0x0007b0b0bd37a872, 0x0000000000029ec2}},
        {{0x000d73368a1e2f2d, 0x00047170f8a745d8, 0x00056c4ca9228944, 0x000bd224b08921af, 0x000de6caca3a5f65, 0x000cb9d5b1017ae3, 0x00099b613f36b626, 0x00000000000ed19d}, {0x00092b03ebebeebf, 0x000c752c915a4d7d, 0x00088621f044eee8, 0x000f7c206a51a132, 0x00066e0197ce2cd7, 0x00071b04cfed60db, 0x000b134e5b2bb7d9, 0x000000000002f45a}},
        {{0x0009fb6e89b64981, 0x00034fb09e755c21, 0x0005804661454c99, 0x0005b01cdb1f4f66, 0x0008f1dd1cf0451c, 0x000687e41b02f906, 0x000bd3d4765e7c0c, 0x00000000000d1967}, {0x000f135de5248a08, 0x000f406a0e4ec0b1, 0x000b70c2868ad046, 0x000d01ac651d8073, 0x00030618a96cc1e9, 0x0004ad8ce5e6ebd8, 0x00090f9ce1aacec5, 0x000000000009953f}},
        {{0x000f6e431dc5b814, 0x00044fa4b66978a6, 0x00032f1c66e89ca4, 0x000ba60986c85001, 0x0004dae014e110aa, 0x000bc0e4649bdfff, 0x000d5fe2a810bd0a, 0x0000000000062d1d}, {0x0007dc411c936920, 0x0002c15d2f2c7072, 0x00075b1f6234ac4c, 0x000b6f94c545f5b7, 0x0002f99241e840c6, 0x0007082935875b6b, 0x000344fa3e88a9d6, 0x00000000000ad6d9}},
        {{0x000edb2b4857e570, 0x000fba8d7901c33b, 0x000ae7d92e8b3979, 0x000d7c504e4bce1a, 0x000e5ab0696fb905, 0x000db9cd5dfbf9c4, 0x0000fe0f4e6e2ab6, 0x00000000000482e4}, {0x000aa1f98e714cdb, 0x000225ccd9b418dc, 0x00058c553b3cc2f6, 0x0000dbe59065754a, 0x000a461c1620cebe, 0x000e541a74d052d3, 0x0002d8396bac422f, 0x0000000000070bf3}},
        {{0x000a0c1fea68e228, 0x0005003c88a847c6, 0x0008f73e5d2fc7c6, 0x0003d393870d90fd, 0x0004f2fdb976d90a, 0x000312a302fd78d1, 0x000f2f9472a6066a, 0x000000000003484a}, {0x000f5efbe46efdf7, 0x0008c7cf25950c82, 0x0006be05e61c118c, 0x0009379e563a2bde, 0x0004b46e93d85d47, 0x00037285ce87b50a, 0x0007e69128fc11f1, 0x000000000007c67d}},
        {{0x0006a0e99182cb3e, 0x00059da2b072529a, 0x0000a5cae364c3e2, 0x000e35a5b132756b, 0x0004fadb6907c721, 0x00083f546e5695e8, 0x0008aebc5a3bf4c2, 0x00000000000dde12}, {0x000842d7630a9d87, 0x0000d179c7fa6028, 0x00016ae518569923, 0x00083bea16bad558, 0x000e9137ecd8af7e, 0x00030d1ab6f41231, 0x000ee8ac87543d8f, 0x00000000000b1ee0}},
    },
    {
        /* digit=28 [{1,2,3,..,}]*([2^112]*G)  */
        {{0x00021880433d0094, 0x000fe359cbfa01b1, 0x000ccfcddd1c5f17, 0x000e90f630cbcb5a, 0x00006ddbf2382fd5, 0x000f753e62b0f613, 0x0000165070970a3a, 0x0000000000041727}, {0x000724192d1373c9, 0x0003e2eb15b3b70b, 0x000786ed962decdc, 0x00074aee930a75c6, 0x0006849e77270d43, 0x0006a19ff3cd3604, 0x000c049ac3117e33, 0x00000000000433dd}},
        {{0x000b49aee05ae3b9, 0x0005d985dcb4303d, 0x000d13447c8994a2, 0x0007f3055fbf1d6c, 0x000f43fcd98a4059, 0x000ec99cab6d166b, 0x0003705932570915, 0x00000000000c5bdd}, {0x00050b2df8b6890e, 0x000dac18f782613c, 0x000d7472d5468bfb, 0x000a89df478e56c7, 0x000864d290535d50, 0x000cf7e0e242c2f4, 0x000d40cda12c6161, 0x00000000000ac8d1}},
        {{0x0001c7858f5f5373, 0x000027a9a9dd7275, 0x000baba450a60383, 0x000267b6726c3253, 0x000d7f841afb1550, 0x00004643012cfb2f, 0x0006cba41b19052f, 0x00000000000b7d78}, {0x0007d7e12176321a, 0x000667b61b39c512, 0x00091b7379479d0c, 0x000efb84a3489052, 0x000cf2e864b965d6, 0x00096cdb2f377862, 0x0000f63c94000dab, 0x000000000008efef}},
        {{0x0001ea0f696fd259, 0x000ff8c558000585, 0x0007b878d3ed6d3e, 0x000d75ceb7a35cc0, 0x000cf0a93110acf0, 0x000c50c850fbdce4, 0x000d6f82b2d13a9c, 0x00000000000cef70}, {0x000ce46086dea462, 0x0001e1cc9be2dbe9, 0x00043757ada5e365, 0x000479f195a532b9, 0x000a1ab3605c52bd, 0x00024c1e0769fe6c, 0x000d7aba6a63e48a, 0x0000000000099da5}},
        {{0x000ad5fa3c1cbaf9, 0x00066df2a6cec962, 0x0003dc76a0f4995b, 0x0007e4c0de8a6aa7, 0x0004be7b8f5d7c8a, 0x000da3e7e5c806b9, 0x0008e5ccca1c714f, 0x00000000000cff78}, {0x00056778e2279cec, 0x0006e17a081a8743, 0x0007133643f614b8, 0x00090549e4cfac8a, 0x00029d53a2000d7c, 0x000977a8b6fb74c6, 0x0007c465b9ed6d5f, 0x000000000000ba2e}},
        {{0x000a6aa57649a837, 0x0000f8435e295b6f, 0x0009b668270d6cf9, 0x0006e6fece5cb161, 0x00095336f7ffe2fe, 0x0008ee2676f249b0, 0x00017b5801feb990, 0x00000000000fc8f2}, {0x00006b2a4f1b5cde, 0x00053fc1adc3e51d, 0x000e0bc8cae40bdd, 0x0005404cc1d9b726, 0x0001a1f2c5a44afb, 0x00008f19575cabd6, 0x000f492bd797e1cd, 0x00000000000cbea0}},
        {{0x0008ce5f4471b983, 0x000d8e9759aa9a33, 0x0001c65f7f2a0258, 0x00059aa2c80bb617, 0x000230c9b328e49a, 0x0004ebffcdb89e21, 0x0008e4f42b9adb00, 0x0000000000047967}, {0x000c42c02e645b6d, 0x000f20dde5b0e690, 0x000d9a21c0cf9036, 0x0007a5cd8c8285c5, 0x00071d1562069f8d, 0x000e126bcd7cde59, 0x00060bdbe765d7a8, 0x00000000000e3f4e}},
        {{0x000bcd16afac5b0a, 0x00041509abccab5b, 0x000ba67813ffa4bc, 0x00071bd640bc2f7a, 0x00078f546c681ae3, 0x0008b8cbfa155c29, 0x00048b8858cadcc0, 0x000000000001d969}, {0x000da5c95714aa4d, 0x00010919cb2262bc, 0x00085c89980779aa, 0x0000f1ab4abe5612, 0x00069cb75bd125dd, 0x000a778e4c6bbe63, 0x0004b3bb367cf947, 0x00000000000bde52}},
    },
    {
        /* digit=29 [{1,2,3,..,}]*([2^116]*G)  */
        {{0x000545d771f3d008, 0x00076c6307398a47, 0x00054915c4cf9c72, 0x00026aae3a993222, 0x0006da9308d18e01, 0x00068e5050fbfdf7, 0x0008bd2163ed6b61, 0x000000000007500d}, {0x000f06ed5b023878, 0x000d3f571595fffb, 0x000ccfd2dfb01965, 0x0005f7e53d84c661, 0x0009025029da6b22, 0x000d89aed9c03126, 0x00015054c1edfa17, 0x000000000006b435}},
        {{0x0007c3fb4d5bcba6, 0x0005b009ef80a60c, 0x00017966ca95c6c2, 0x00055ed685add960, 0x000672c7dac8ab44, 0x00072f16818f323f, 0x000a99e7009790e3, 0x00000000000067be}, {0x000c5bcf19664dda, 0x00043f15bdbcc383, 0x0007e49d4e4912df, 0x0002c304bdbec36f, 0x000d1b3ac6c72fd6, 0x0002938898b3ea93, 0x000f8bce1c8e5c9c, 0x0000000000011564}},
        {{0x00098a86761cd6cf, 0x0008477faac22471, 0x0009e917079e4be8, 0x0001dd45fbd35ab8, 0x000a45d6d40e31ad, 0x000749ef67e0de9d, 0x000cf2a16c5f1939, 0x0000000000085691}, {0x00046f38fcf548ca, 0x000e7156536ac506, 0x000b8e250c84b9f9, 0x0001eacd35088fde, 0x000538fb995c165a, 0x00093b3a5ce8a733, 0x000e7934d0d33132, 0x000000000008b570}},
        {{0x0002b3fc481df360, 0x0006d6a3a232752d, 0x0000509b75e73d18, 0x0000c3322a82d404, 0x000a14cbc8351703, 0x000da272724bf18e, 0x000d4a319dc4cab2, 0x000000000006d020}, {0x000398eafd6e92bb, 0x00066aeebdd89825, 0x000a73a882a68cbb, 0x0002978595f361eb, 0x000c1ae8fa3cfc8a, 0x000341575e60dd96, 0x000e4819c2109aa5, 0x0000000000006a0e}},
        {{0x000065dd44a81443, 0x00083443deaaabf2, 0x000c83e66800298b, 0x00018ded51d6acd9, 0x00017d4ec9bef889, 0x000c48a6948c9b2f, 0x000f5478b41104df, 0x0000000000078827}, {0x0003f535562e2f60, 0x0004674ed7948c07, 0x0001ccc08940c103, 0x000449b0d9fe252c, 0x0006c14e825a6fe6, 0x000690f0031743d0, 0x000748d9d4ad8c08, 0x000000000000e65f}},
        {{0x000eaf196980dd99, 0x0009692501ff821f, 0x00072b4a2cc10b2a, 0x0009a05e53038473, 0x000c2543d1a995b9, 0x00080877130a72fd, 0x0001456648126b11, 0x00000000000eefb8}, {0x00041f6a86207b43, 0x000d8d9a87f836b7, 0x00016f9bd17dd926, 0x000e118c40122e4c, 0x000f853b291e3341, 0x000565bade513567, 0x000f77d4f1bbb73b, 0x00000000000a7658}},
        {{0x0007b4a08d152bb9, 0x0007cbab37a630ed, 0x00077618410cb7bb, 0x000086c1d4a8f5be, 0x00046581428269e1, 0x00076cc7d3a87384, 0x0003a0b642dc1626, 0x0000000000097e42}, {0x0008ca2aba5729e1, 0x0005b66ac1b365bb, 0x000e67e449b0ae7e, 0x000debbadcc68242, 0x0004b1536fa18f47, 0x0009e546144e9bf6, 0x000b901cfdd50493, 0x0000000000043bb3}},
        {{0x0005a0a5974046e0, 0x000112e32f89636a, 0x000495f55b511bd6, 0x000b1ad8f8328866, 0x0008adcbd933278d, 0x00064f863271d907, 0x0003652208fae34a, 0x0000000000039c89}, {0x0000e167567c2b6c, 0x00048cb46f2725d5, 0x00086f7986cb61ca, 0x000d316edd504188, 0x0004cabe36ed3421, 0x0002efcdab1d8a06, 0x0003eedb13e56036, 0x00000000000c71eb}},
    },
    {
        /* digit=30 [{1,2,3,..,}]*([2^120]*G)  */
        {{0x0000e8a3d453ef19, 0x000752af8ed809c8, 0x000017d0798a8632, 0x000726f782193578, 0x0001b87254c44c0e, 0x000e7691a8c1962a, 0x0002ee30796a71c9, 0x00000000000a75a1}, {0x000c339094f215d4, 0x0003e535f42c17eb, 0x000b753210a29b6e, 0x000ef35e39a7a591, 0x0002d459b91ab7f1, 0x000fd429da2789ae, 0x000f57eadbca7f02, 0x0000000000065290}},
        {{0x000ffd02d3a50b87, 0x00027c085500177e, 0x00023ee786608759, 0x0001d964318e861c, 0x000604fe9b85dda7, 0x0005e7e2001d3d39, 0x00081cda4bc065e2, 0x00000000000e076c}, {0x000171ac92e482e8, 0x000095b9f82189f0, 0x000cf8881039863b, 0x00083e4d8dd159bf, 0x000720b18043f526, 0x000a0d8f5ca9c888, 0x0005c473c040fa08, 0x0000000000017952}},
        {{0x00023e634e793b49, 0x0000c37ed2be4ce8, 0x000e92823d3628c0, 0x000ad8ae77c2f00c, 0x000a44de16a8b061, 0x000e490ffe87e1a9, 0x0003eddf4f57c87e, 0x0000000000000599}, {0x00036b9f1b67eda0, 0x00020e1036387a16, 0x000cdc81b1b14889, 0x00020d15ab42f920, 0x000dac0ff03359cb, 0x000c1e7f4a738a18, 0x0006e0da501a2e2a, 0x0000000000084d8a}},
        {{0x0000efaf35b1418f, 0x000173a8289046f2, 0x0004fa8840c897aa, 0x000898ae5fa19d2f, 0x00065e1c5fa4c574, 0x000390fc20b13dbb, 0x000187f11343ba7c, 0x00000000000d7d32}, {0x000b8b2ed2cc734d, 0x00011c92cb1bab11, 0x000e6307a3aa4fd9, 0x000dfe5672f2af7d, 0x000430932441da69, 0x000af02d69c62d7b, 0x000c22165672ad94, 0x00000000000cde81}},
        {{0x0006bff8295a2913, 0x000d91d27efcde72, 0x000bdb7b59692a46, 0x000fb83caa26d18c, 0x0003b82babbe99a3, 0x00083dd60d27e613, 0x000cb861030dfdd7, 0x0000000000073c78}, {0x000a3caf05842de3, 0x0002d8707a2cf633, 0x000f47297bf43775, 0x000d412a2b176b8c, 0x000fc72021017c3f, 0x000a6d536d5b52e2, 0x000ec024a63030f0, 0x0000000000004648}},
        {{0x00024dd131928645, 0x00063d45e3501026, 0x000cd2a97f7d8b99, 0x000a088e302483ae, 0x00082c2485c01532, 0x0009cdbe63475e8b, 0x00073808f9ea5696, 0x00000000000253cd}, {0x0002a544f2d5917a, 0x000581cf323a3b9c, 0x00023a0e49f58c76, 0x000e84fc06f3d0ad, 0x000e39efe6d99a31, 0x000ac4decd326b86, 0x00058277e3e1df14, 0x00000000000f3e0c}},
        {{0x000611e8121168e1, 0x0008967477cdc1e4, 0x0003ef00563660fb, 0x00060b9917eec666, 0x0004d66c5c1d31fd, 0x0009508baacd95bd, 0x0002432e0551f3bf, 0x00000000000688cb}, {0x000033e51fe09e4b, 0x000448c039740256, 0x0004cddb41207f6c, 0x00025e144db62afe, 0x000bc4a030da1918, 0x0000815043ee8aca, 0x000cd08ab154a894, 0x00000000000486c9}},
        {{0x0009bbd8fc757c25, 0x00003e92b56bf065, 0x0003d138d6f390b1, 0x000d0ef50f5c483a, 0x000611fa89fe7754, 0x0004ea7d8850a9ef, 0x000a2e97d74b1bba, 0x00000000000aab7b}, {0x00042e268ab251b7, 0x000af06f30ab067f, 0x000bbd0bcb9d997c, 0x0000bca7a2e053e3, 0x0008dcf0e14c4758, 0x000f553108579559, 0x000e18c3596781d6, 0x000000000004c9b7}},
    },
    {
        /* digit=31 [{1,2,3,..,}]*([2^124]*G)  */
        {{0x00056db726934b05, 0x000d28db8b78ca20, 0x000efbe1df76bc8b, 0x0000f022dde2e3c8, 0x00038cbf406c67e5, 0x000f7ff602e2461f, 0x00068832182781e1, 0x00000000000e30e2}, {0x00008c8748c4086f, 0x0000895adc204b38, 0x00054339345edf7d, 0x000f060b73417379, 0x000fd128d46cd5ca, 0x000fc1e04ed93187, 0x0000a13f2819cb20, 0x000000000002b7f7}},
        {{0x000424a4cdc9ef0a, 0x0009fd74d09c0410, 0x000c23c8eb2570ac, 0x000cfdce132b9412, 0x000c843d3c66db1c, 0x000cb3ef4a0e5309, 0x0001771fbd03a5f9, 0x0000000000000ea5}, {0x0001bb0b6df68f60, 0x000fa1ebf5a155d8, 0x00046a120ff8039f, 0x000c37aa0d34d161, 0x00050fca43af3256, 0x000841bdeee40efa, 0x000a0bc299bbd4b9, 0x000000000003bef4}},
        {{0x0001fdb0e87d4c69, 0x0008f5bbf8bfdcec, 0x0002c8b1b68f641d, 0x000718bfd3fb74be, 0x00015a085196abd2, 0x000dba2ea03c0150, 0x0008bc147474dc4d, 0x00000000000eaa2c}, {0x000a67e6cbd2f408, 0x000e9bc3dc5c2f94, 0x00045e933f00eed5, 0x000e98325d341410, 0x000406fcbc8e0276, 0x00024ebab9372694, 0x00050b8c4b7f2ca3, 0x000000000007dc3a}},
        {{0x000547a274927d68, 0x00075e01295fd654, 0x0003c43252dc8b6c, 0x000689f9eaf8eb82, 0x0005c622acc52ec9, 0x000757f2a327b96c, 0x00006eae918f81b2, 0x000000000004fd66}, {0x000f52e65907ef2f, 0x000a109bb7fcdbf4, 0x0007ab72ec41ee7a, 0x0006f125fb475125, 0x00018e399520df2a, 0x000b1c3a2be88faa, 0x00081f0166d57e3f, 0x00000000000e525f}},
        {{0x0005132af5b85725, 0x0006e0c64cec623a, 0x000d73b377e49971, 0x00043ae5a8027b17, 0x0007d0a7406ac27a, 0x0002bfeed1c5729c, 0x000357445fc7d34e, 0x00000000000809bc}, {0x000fe2c3577b04e1, 0x000a30598626b6a7, 0x00059236250da683, 0x0009b314294839a7, 0x00093ab9bc662151, 0x000ab597466282da, 0x00012e84c422c25c, 0x0000000000071c0f}},
        {{0x00044dd5af953087, 0x000b3f24fab9aa7e, 0x0007e53a0eea7298, 0x00070e733ac4fb68, 0x0002c8a6976575e3, 0x00030aa0ed8d6164, 0x000637f057f10a5e, 0x000000000004f1e8}, {0x000514c4bb67bcdc, 0x0005e887f06b2b4b, 0x00032e66a34587bd, 0x000f7eb6e6483a19, 0x000f475d4bfd4319, 0x000811fc744f1aaf, 0x000f8ba8ee73b50f, 0x0000000000082916}},
        {{0x000b0ecbb3371c13, 0x0009596495c1e57a, 0x0002216cb93b15bb, 0x000c858fe6e84ffb, 0x000b0b189be26806, 0x00058a7685c215e8, 0x000202ac61577e80, 0x0000000000088e0d}, {0x0004f2a99a53c49a, 0x000de70322e164ce, 0x000973bc42c42efc, 0x0001f014d701ab8e, 0x0004a61df25f8863, 0x000f6dbeb889f10d, 0x000168affec5a9bb, 0x00000000000934ee}},
        {{0x0006f789ccd0fed7, 0x0003cdefe3a7abd6, 0x000569da64526056, 0x000c26a73d1f9b54, 0x0008237ae8b77366, 0x00047825935d5d71, 0x000ee33efb20feb4, 0x00000000000aa545}, {0x000d70972e2560bb, 0x00081750edd05bef, 0x000da581b51c4635, 0x0000f370a9e29dc7, 0x0006bfbec7f616e6, 0x00047e1439cf0a13, 0x0001a9b430a34b2e, 0x00000000000c6cdd}},
    },
    {
        /* digit=32 [{1,2,3,..,}]*([2^128]*G)  */
        {{0x00056f0266d7e788, 0x00065f186d6bc61b, 0x000e03ac4fb95d1d, 0x000432edfe64dc3e, 0x0002d795ea57c9e7, 0x00045af2bd9fc483, 0x000517f4ffdb81c8, 0x000000000002b670}, {0x0004d8950582c931, 0x0007aa7c1be04d23, 0x00045b4daadb356b, 0x0008aba03f2ef6dc, 0x000420dc701d62ff, 0x000f1011960ecaac, 0x000fd09f4b559f4f, 0x000000000003cd54}},
        {{0x000973c51356e0d2, 0x000aa31954a56667, 0x00047c018911cdb5, 0x0003ec5e37ef1d2d, 0x000575bc3b668c43, 0x0009cd98bcd0f203, 0x0002a33723f14524, 0x00000000000ea3b4}, {0x000998df93db2ed6, 0x0000ff607fee05d9, 0x0005662b349502ed, 0x0005d74cea5417fc, 0x000a10fca22bd47b, 0x000635e8b6891940, 0x000fef5cea41332b, 0x00000000000b5934}},
        {{0x0000bd5aa7cff3d9, 0x0003c365a4a426e6, 0x00069eea0aae5d6c, 0x000c8b4b7205a704, 0x000357df815ca330, 0x000a926744858eab, 0x0005d76e522afaf1, 0x00000000000f4136}, {0x0001a59021cca096, 0x0002a00dd3461384, 0x0008629f956ac671, 0x000103f7c082301c, 0x00076b092b71f427, 0x000037a3314a2a18, 0x000e14210f632130, 0x0000000000039340}},
        {{0x00025fc00f86f9f5, 0x000b5b1f4fead323, 0x000e624d9ef69f8a, 0x0007efe2e28aa2a7, 0x0009b3151d6748b4, 0x0006a346070dad2e, 0x000af787a8178b43, 0x00000000000801f7}, {0x000701583b4bfabc, 0x00034fab462e36e3, 0x000ddd2177064fe4, 0x0006b3fe66812831, 0x000a6a3fdc1db469, 0x000e37dac3bd9910, 0x000fb34409128449, 0x00000000000cf605}},
        {{0x00081f556f82acd9, 0x0006f59470e4953f, 0x000963a40f813b02, 0x0003c1e0fb30ecd3, 0x0003b62892f14761, 0x000eefa161fbeffa, 0x000719ddfeef49f8, 0x00000000000daeff}, {0x0004929822ef7d6f, 0x000dcff1872cd89c, 0x0009edd2dba5c5c9, 0x0003a387695218a2, 0x000720802b8852a6, 0x000f4a473a0d413a, 0x000a4a233f1f3118, 0x00000000000c9d7d}},
        {{0x0006ae71b9b2b784, 0x000ca86737912907, 0x000e250552a02dec, 0x000185ee92c712de, 0x0006c201e3272efe, 0x000f6b0788b908f8, 0x000bf33ab5528894, 0x00000000000ce0a5}, {0x000abf1c0844ab25, 0x000185eab37ac7f5, 0x000c6bca8c37680c, 0x000edff304f1cf97, 0x0008f6fd3e90f3a3, 0x00016e5ed8992ecd, 0x0005c4ff24d7c69f, 0x00000000000def9a}},
        {{0x00054a31e739d50e, 0x0001e2edaaaa9c57, 0x000574cc3b6825a1, 0x000fc42bcb161908, 0x00046709d8513542, 0x000af4ce06c04ec2, 0x000e2e5fe9768ea9, 0x00000000000eb6eb}, {0x0005126ebc807472, 0x000c87b7da4abcae, 0x000d07139021766d, 0x0001c51fd563c816, 0x000926c775e17f69, 0x000b58889ea990d6, 0x000c2c2cd96f1321, 0x000000000000f1aa}},
        {{0x000e697116bc35a7, 0x0002057edf71ec8d, 0x00063f77e7793641, 0x000ba1aa6934160d, 0x000c5e639a54b37a, 0x000bce957d45b3d2, 0x0007f362c6b9ad70, 0x000000000005d7e8}, {0x000a8127f71a285b, 0x00017b02169816f9, 0x000c7a7939056bf2, 0x000692b478e4b92a, 0x0009be3fe25a15aa, 0x000bdd4a8dd67cf2, 0x00027af1ef75b006, 0x0000000000041df6}},
    },
    {
        /* digit=33 [{1,2,3,..,}]*([2^132]*G)  */
        {{0x000455d88f7fc60e, 0x00091a58a27c4fad, 0x000ee65aa7e47d3c, 0x000826600079a263, 0x000f2606caf52431, 0x000cb28c8113f6fc, 0x000aa6f6ff2be316, 0x000000000003c17c}, {0x000c05f668a9dc76, 0x000798ea577e0148, 0x000c912137590c65, 0x000eff4592a57ef2, 0x000c5e3f67b24b28, 0x00000890276e1f87, 0x000b7e40d7a676fb, 0x000000000004b6e6}},
        {{0x0007f95d338f5fb6, 0x0000076e27519dec, 0x000fefb9477deeec, 0x0002dfb71fba6625, 0x00085e0886af5832, 0x000ee544c228aec0, 0x0006398d83b6276a, 0x00000000000e29f9}, {0x0004808391599977, 0x0006d1ba4cfe02e0, 0x0000441e3d4b3e92, 0x000ed58cee95d92c, 0x000abe5355407e0e, 0x000a9a1f42d29282, 0x000c82866638b607, 0x000000000006ab8d}},
        {{0x000c43ee7094e067, 0x0000b599ece9688c, 0x00025bcbe937baac, 0x0005ba86373ccef5, 0x0003c64f72612afc, 0x0004ce1abe455683, 0x000a77f29ad54041, 0x000000000003ee82}, {0x000e0b1548890be5, 0x00094fa26eaa0940, 0x000c96b0a3e57442, 0x000ade54e1c94bd7, 0x000e20b0ddbe9570, 0x0006eff6ff4a86d8, 0x0007a4967ab653dc, 0x00000000000e8bab}},
        {{0x000de87dcf2f3d8e, 0x00056a16c79cee01, 0x000db47cfbc9e301, 0x0006aa0c2f47e69c, 0x000f2c9b4914b8bd, 0x0007242a8277d590, 0x000512b6d1c81009, 0x0000000000045f75}, {0x00044d25d22dbf16, 0x000f02176162aa27, 0x0001167bcefd598c, 0x000c0d1a503aee8c, 0x00057af4dd78fcf8, 0x000b43be45d1f94a, 0x00071b8f0bc1a62a, 0x00000000000ba9e0}},
        {{0x000adf4d6bef5a3d, 0x000a19a2dc376622, 0x000b08bb2e394780, 0x000e5cf7c0765207, 0x000c4063c3538e70, 0x000baec46c0dbb4a, 0x0001ad3f4550566a, 0x00000000000f9356}, {0x000e15a92e3f32ae, 0x00055d0bc6d91fc6, 0x00092d1d7bd54520, 0x00033424b80c021f, 0x0007af5e458c62a7, 0x000bfde586e1d546, 0x00052c6922574f70, 0x00000000000d19f0}},
        {{0x00077ca19b6937db, 0x000c3bf87f30d9c2, 0x000f3d29bdf61e62, 0x00032bd50ec0ba46, 0x000ae60d1e72d5fd, 0x0004ff2d37de8fe0, 0x0007e452d77fe0e1, 0x0000000000055ca4}, {0x000fd784e6c9781f, 0x0000339e3d19e6d1, 0x00045fa26fa441a0, 0x0000c7afe92c6d07, 0x000869a829043bd7, 0x0006d1d9ea3fbc06, 0x00024be263f00ed9, 0x0000000000075c0c}},
        {{0x0002619f4e7dcc16, 0x00090a1d1a8f0893, 0x000f4151f0e91898, 0x000c39b84ee41fc5, 0x0002e4a019099db2, 0x000a66c12ad292a0, 0x0000f6ee6b26ddfc, 0x00000000000f04f7}, {0x000e18e3a8161ac5, 0x0007f01e66788b92, 0x00099db080ad62ab, 0x000e4542595d5a6c, 0x000bdce965cc3e7c, 0x000ec7931f894a64, 0x0005be12e46952d9, 0x00000000000fb65b}},
        {{0x000d91fc53fa82b7, 0x0007d705bfe3760e, 0x0006eb6118c41a96, 0x00077a9bc4567977, 0x000617081a2f811e, 0x000e1640e4f52c4e, 0x0001d787405d819f, 0x0000000000070771}, {0x00085581b6dc5925, 0x00062dc6fff82580, 0x000af03b55b43c90, 0x00066bea9cf3eae3, 0x000e87139b8b0ecb, 0x00042b4d77418372, 0x00032e6aaccf295e, 0x0000000000031fc9}},
    },
    {
        /* digit=34 [{1,2,3,..,}]*([2^136]*G)  */
        {{0x000f65dda78f831c, 0x00073fa3b4c8e10f, 0x0006b8117d5cb12e, 0x00039562e8c4d8cc, 0x0005cf89935b06aa, 0x000fa071e4981c3e, 0x0003bebdbd0c4745, 0x0000000000049607}, {0x00004bb91c157448, 0x0007009a7298688d, 0x000867eb798cb22b, 0x000b5b3988f2781c, 0x000d73b1719d9a64, 0x000d2e8076ac9440, 0x00042b58ad54c7e3, 0x000000000008f067}},
        {{0x000489635dd868b5, 0x000ef339521774bb, 0x0001f606d4b5774d, 0x000d61e0a285bd5e, 0x000e71171b1c1084, 0x0009b29f93935a84, 0x0009bd8ac2433cf2, 0x000000000006dd1e}, {0x000f576ac5f0cc26, 0x000f788da0477c71, 0x0007313f812b64cc, 0x000a25f9d5b19e1d, 0x000b6a27fa79a792, 0x000a16a8e9ee015c, 0x000e67ea3bf8b57b, 0x00000000000c15fd}},
        {{0x000f3b091fcd53e3, 0x000c537f50e43695, 0x00003782e79d52fb, 0x000af85e1d111511, 0x000f6785ae1c3916, 0x0006ada8cf56e852, 0x0005b2bf8c72adae, 0x00000000000e1328}, {0x000cbe0d0bda153b, 0x000e35327920cb17, 0x000e7daa3650306d, 0x00028573caf37928, 0x000b550ffe1ca713, 0x0005d4e4fb15ab34, 0x000f9d818980666a, 0x000000000002f854}},
        {{0x0005d9265089916f, 0x000013b623150146, 0x0007ede4f0fb2f49, 0x000c31c56471100e, 0x000b1bae8d2a4bc2, 0x000f1f76a4a0e73b, 0x0006cabfc0770a8d, 0x00000000000a7bb1}, {0x00039cb13d9c4b7d, 0x000f42afe5b1cb58, 0x0008a2ce9b2dade4, 0x000a333a1af2a824, 0x0005dafed6cd97ca, 0x000c3b2e393cb92c, 0x00095609553e7e92, 0x0000000000061af2}},
        {{0x000ea55edca2a058, 0x0003f5559dd3109c, 0x000f7ba60d37c6c8, 0x0000fabcaf57d0dc, 0x000578742ab60d84, 0x0009c1c8e9625866, 0x000167c85482ad40, 0x00000000000adaa6}, {0x000208d39ae67d2a, 0x000dcec26ad9971a, 0x00067a35ccc54689, 0x000a7fc2539986bb, 0x0006a23dee41340a, 0x0001a9837d767487, 0x000da9a948a9292e, 0x0000000000071438}},
        {{0x000778a60c6bd7ad, 0x0007f39038863eb8, 0x000e85efa03ef35f, 0x000682ef3d8310f4, 0x0008f412315338aa, 0x00026310bb52d41e, 0x0006924fbef3dd70, 0x00000000000f6ff5}, {0x000ba50eddf2b7fe, 0x000b63831c6b1cbc, 0x0003bc6684c6c5e0, 0x0006abc6516bba59, 0x0005446d35a876d2, 0x00012a8d0c237f9a, 0x00015bb2b16c0ff0, 0x000000000000ee03}},
        {{0x00014182dba86bd2, 0x0009c817d77b02b0, 0x0006420950654aca, 0x0004107da68b7691, 0x000dbebc4c4dd3ac, 0x0003d39e96904bcd, 0x000950b0d2103ca5, 0x0000000000030a5e}, {0x00028ff31a9f909b, 0x000c7b092568034b, 0x000262a60542e8eb, 0x000ab34c15855ae5, 0x00063017194f2389, 0x00046b838c14dfd9, 0x0006fc420e071911, 0x000000000008fb4b}},
        {{0x0009c78614d38ab4, 0x000d813722ab0651, 0x00088626a03970e0, 0x000f467f76fdaf74, 0x000912ddfd9ad3d0, 0x000bdefd37ce072f, 0x000315ce918a5747, 0x00000000000750e5}, {0x000fa00e65975639, 0x000cd08bbb20dcda, 0x000822e7b86b49be, 0x0005c21ca865ba6a, 0x00002e6e8e6fc8fb, 0x000608b60ee6e41e, 0x0005cdd00ae6214c, 0x000000000006ff68}},
    },
    {
        /* digit=35 [{1,2,3,..,}]*([2^140]*G)  */
        {{0x000cd00e20fc1fe8, 0x0002bbbccce39826, 0x0009c6cbc7ade77c, 0x00035a5d2252fdaf, 0x000954e7dd499eae, 0x0005100fda8f4f20, 0x000727a56d72a629, 0x0000000000056767}, {0x000898f026420cbb, 0x000adbf60b247e57, 0x000b35db15577b1e, 0x0007ad4b93dab9cc, 0x000022d71f39c2a6, 0x000304db218cd0ed, 0x00082104380c425f, 0x000000000006729c}},
        {{0x0000345995afd46f, 0x0009dca07923b790, 0x000129149d2f3565, 0x00079e83cb025114, 0x0005beb383ef41e3, 0x00076dd0dfabac00, 0x00012724d12d9a10, 0x00000000000b208f}, {0x000d58cd6475c579, 0x000359cc38a604c8, 0x000857e410d47fbc, 0x00060d98ac219eff, 0x000faf284c806f63, 0x0009e366a1edaab3, 0x0007269c3b528101, 0x00000000000bc9e9}},
        {{0x000a386526319283, 0x0008d26bd07d697c, 0x0006e6a9c305333a, 0x000c5466798b96c3, 0x00081d3f3859d831, 0x0001b9ec71d6b410, 0x0001a9501d38ec99, 0x00000000000d7843}, {0x0008f62f7d623e0d, 0x000dd8b85baf6942, 0x0003f90bead64135, 0x000c1107acdcf58c, 0x000c848d5842efc7, 0x000c7fdacd9af415, 0x00019b6b5a06bc0a, 0x00000000000a3443}},
        {{0x000517cc17c08a8e, 0x000a28410d82975c, 0x0000ae8bc362b8a4, 0x0000d0d18c253486, 0x00007eb035a3ae46, 0x0001144145d0279a, 0x000f7987e7c1289a, 0x00000000000c0744}, {0x000ccad1ad801112, 0x00072b7b2b4f054d, 0x0007f502703051c0, 0x0007395b51ee6864, 0x000bf6122422124a, 0x000f1a7fff2937c4, 0x00032eb4ec133207, 0x00000000000f8860}},
        {{0x000daf15d19f8632, 0x000794c0d78053af, 0x000ade8ac0a0ca73, 0x000c453e4b57e236, 0x0005f4172b285217, 0x000f999f8b4669e3, 0x000d41509a3cd049, 0x0000000000087c69}, {0x000ec5ba18211916, 0x00032e14d01e8346, 0x000c499a14031eb7, 0x000bff270dc2bf04, 0x000bc01864000e17, 0x000a4dd3446560b7, 0x000d06e28c7b9c49, 0x000000000000f325}},
        {{0x000161555383d319, 0x000d0dd4eb3d0283, 0x000dca801a8bb250, 0x000285ddc1973c7e, 0x000aa0046b981200, 0x00019a7fcfd342be, 0x000fe5d191734912, 0x00000000000b7caf}, {0x000a5ee9f805422f, 0x00057ea5a0c4d360, 0x0004c8206c9d4abd, 0x00016dc6046eeeb1, 0x000ecc8e64b15b2a, 0x0007fadda1755e1f, 0x000ec251e4946e9e, 0x00000000000fcbf4}},
        {{0x00024bd4d62de244, 0x0001d46d7088801b, 0x0002ae4b99b01f02, 0x0008fca4fbea2725, 0x0005bd80a9dfe494, 0x000871c717d6c776, 0x0005701b470ea6f3, 0x000000000008330a}, {0x0004db08d904e89f, 0x0009bdaecddc0ed5, 0x000f26cdb2e08463, 0x000cd84caeeef145, 0x000fc685a35656c7, 0x000d93ddcc60c910, 0x000895e68bc5c59d, 0x00000000000feb64}},
        {{0x0001bb5474d7445b, 0x0008af877e8c6483, 0x000948d44b23fa45, 0x000269f2b004cfe9, 0x000795450f8b19ff, 0x000a7d4588adc5d7, 0x000fdc688ce8bce2, 0x0000000000097bd7}, {0x00043a2684d27187, 0x0007c1b9f4ad5bd1, 0x000b255e8d74e0bd, 0x0001f7e71d83d86e, 0x000d25ffc219abee, 0x00073f553e693c76, 0x000a551c9a84afc8, 0x0000000000066d77}},
    },
    {
        /* digit=36 [{1,2,3,..,}]*([2^144]*G)  */
        {{0x000e4a9a609d4f93, 0x000f05584cbb3289, 0x0002a9b59e61225b, 0x000d8267df2d43de, 0x00000109e8014126, 0x000172f1cdd5bbbf, 0x0000d985b92ee338, 0x00000000000f3143}, {0x000ac2d50dd03701, 0x000b11e059a07dd0, 0x000eb68a6d1ce296, 0x0000751560f20e77, 0x000e7aaf3a9ad622, 0x000bae14ea59489a, 0x0002497b70e2f664, 0x0000000000076d08}},
        {{0x0001e7720f69ad57, 0x0004baff47822226, 0x00011f4e0e8fa5c8, 0x0005f8d2ef696aa0, 0x000b4fa94fcabeb8, 0x000d41a438ce5baa, 0x0007720a32f96200, 0x0000000000074f6e}, {0x0005b3ae79f59da9, 0x000a8be0221aef2d, 0x0005793443a4452f, 0x00085d6a7e49f3a4, 0x0009cb5d3c6378ff, 0x000eb65f56300658, 0x000bdfe8e4383596, 0x00000000000ff8ad}},
        {{0x000ec83ab52a5d83, 0x0006cf6fd26cb934, 0x000ee1af72ab19a2, 0x000788be372817f8, 0x000ae8c90c31694a, 0x000aa0ce585f3dc1, 0x0002238c90c6bf15, 0x00000000000f01bb}, {0x00043ef688d7f41f, 0x00075bdae01dd56c, 0x000cf79ad5ecb5f0, 0x000ff4ec2548f251, 0x0008d12802a750dd, 0x0004d924054a4986, 0x000d1acd922fb640, 0x00000000000957f6}},
        {{0x0006a17488184bb0, 0x0004d082ea61c88d, 0x0004b5a68e9c5821, 0x000df962e54f5d00, 0x0006e39dc6ab7d33, 0x000179b13340b0d3, 0x00088cd97a8b848d, 0x00000000000288d3}, {0x000bcf17110adf56, 0x000462237e507d00, 0x000fa28fb932260c, 0x0008bcb42be74594, 0x0004de2176b6645f, 0x000e2f09cce2b0f5, 0x000001af570a09d1, 0x0000000000057fdc}},
        {{0x00009414087acee6, 0x000fc7dd81467f3f, 0x0007997c546fe735, 0x0004e832d2e9fbfb, 0x00061c9d9c7ca2ae, 0x000f27f140f127f6, 0x00029ef06c5a57e6, 0x000000000001f303}, {0x000b431a1206d910, 0x00037a1b2b82f82b, 0x0007312610c2e220, 0x000cb8b039f12bbb, 0x000902b92d6b1cf0, 0x000f649e606ce433, 0x0006eccb5e826869, 0x00000000000e9309}},
        {{0x000f672480aedb22, 0x000c79d21c740037, 0x0001a34fe6cb5f73, 0x000b1a3aae5d701b, 0x000891978d1557f8, 0x000b85b1e477e4f8, 0x000fb42913ffb40f, 0x0000000000058489}, {0x00047efe2cde596c, 0x000a60e1ab266dc3, 0x00073cc3adaf7198, 0x000ccb657ab8d871, 0x0006547f64bdf578, 0x0003e497e339fd79, 0x000706904693943f, 0x000000000001d864}},
        {{0x0009da8454e1979d, 0x00030da92e482d7a, 0x000b666249139a91, 0x000ebb3639c78714, 0x0009d77aac3c8763, 0x0002769a43bf7177, 0x000eda92a5420d3f, 0x00000000000d092e}, {0x000352bd3a0254a9, 0x000b5d7bd8d8f8e8, 0x0009446f6266c164, 0x000fa3ed6b0cf872, 0x00040490a771a5ea, 0x000ff37225255afc, 0x000cdbd40bb0fd15, 0x0000000000079294}},
        {{0x0005908307814aa6, 0x00009158bfe27b53, 0x000df35ce707d624, 0x000153b8e71aca19, 0x000171b11643f07b, 0x0004e2f905e67698, 0x00010977b7dd7dd0, 0x00000000000f0dcf}, {0x000bd23edf0138e0, 0x00071d3bab2a41f1, 0x000facf3129d6b1f, 0x00099afc6f53cda2, 0x000f628039e454fe, 0x00083aa16071f2a4, 0x0006f9f1f44181b1, 0x000000000005010f}},
    },
    {
        /* digit=37 [{1,2,3,..,}]*([2^148]*G)  */
        {{0x000b28d4d03ca18c, 0x000a0b1e36500cc3, 0x000e44ebafd13e2f, 0x0001d2ca4bfdcedc, 0x000fb29cdcfebf45, 0x000514871d210892, 0x0001e35b12bcd894, 0x000000000001809b}, {0x000c86d0fbcc5e1a, 0x000ce09b243105c8, 0x00037f6b6be14182, 0x000a16b5de334b63, 0x00020b116076cef0, 0x000ae2bf4fe0bd1e, 0x00093754e4b48289, 0x0000000000068c8a}},
        {{0x0007e2c503fdc14e, 0x00049309b1eb33af, 0x000b5e3acfac1d05, 0x000b81ebf8a894ca, 0x000e0d29329387c2, 0x000bf371427a40bc, 0x0008957f4c315be4, 0x0000000000001236}, {0x0002d22bfd1d81d3, 0x000f4319c88f7e2d, 0x000189361ce75d22, 0x000a05df0dd13811, 0x00024acba9fafcc2, 0x00027313ec8d55b0, 0x00094a871358de59, 0x0000000000017ce2}},
        {{0x0002c99cf8ff48b8, 0x0002410777188c8a, 0x00068c821fc35883, 0x000b515120380e77, 0x000fe13577d1261a, 0x000862db1453c858, 0x000e1cb1f6bb58f3, 0x00000000000b9529}, {0x000338bb0bed7b45, 0x0008003f63a416f7, 0x00081208dbc793e6, 0x000048b756e5af2e, 0x00031c984bef8423, 0x000a00f3e4d978ed, 0x0009e7a06242995b, 0x000000000004f4d1}},
        {{0x000e08308825a639, 0x000e8d3797831773, 0x000f79843c567224, 0x000a6a611a4e33a2, 0x00005328043a2ff7, 0x000e9f7dd904f86f, 0x000904e29d31c012, 0x00000000000c0a51}, {0x0002dae695c951e1, 0x00024070c2696563, 0x00060638255bc0fb, 0x0000d0c12c8a1f65, 0x0001ab352a835b97, 0x0005eb0c7572aaaa, 0x00059963df45a90c, 0x00000000000d4977}},
        {{0x000fe53cf0bc7d5d, 0x00073e35daffd3d8, 0x00005ab0f149d24a, 0x0005a40f009a48b1, 0x000d308a81c693f0, 0x000885426a4a801f, 0x000f0575e5dc467e, 0x000000000004629f}, {0x000c6c457ceba67d, 0x000c6356b879b5ee, 0x00084a5c7d7e4e3e, 0x000856eec2dd55b5, 0x000880f0c80411c9, 0x00001b21720f0443, 0x0007b8c0b5e3e218, 0x000000000006ea09}},
        {{0x00042a359a9c02d9, 0x0006601c2df8ca11, 0x0001ea46897d0b3b, 0x000341c8360fa6b2, 0x0002c52bb2d6e198, 0x0002efba5e67f809, 0x00032af944dc63a0, 0x000000000002c123}, {0x000d5d58228e0e7d, 0x000b239684f6c863, 0x000f4f910b494aa3, 0x0008eb646594725d, 0x0005793c32ddb7fb, 0x000f94b55bb4f5f0, 0x0002773ef3c33845, 0x0000000000009eac}},
        {{0x000bc0df5aa0ebff, 0x00046efd26dca17d, 0x000d31eadaba6e9d, 0x0008a89e1830a96b, 0x00013e039a029f10, 0x0006fbb3b7e8e368, 0x0002f11747b3e925, 0x00000000000abb3b}, {0x00023cb577b95e94, 0x000e2cac5818c280, 0x000b36c4a5c24e15, 0x000fd4d7a5485367, 0x0009aa3645074081, 0x0001a5f81fe2d8e7, 0x000db4e86ce00ea8, 0x0000000000077a9b}},
        {{0x000a3bf39d563e4f, 0x000f02c5b0e421f4, 0x000bae31a917643d, 0x00085959aa907285, 0x000af658699bace4, 0x0003b18e632be886, 0x000667ce75d6c6da, 0x0000000000069caf}, {0x000af371b713c401, 0x000f0c17c66ce4f4, 0x0002f4e783050dba, 0x0000041623db4f0b, 0x0002c74762e1ceb8, 0x00071c52fe75615b, 0x0002ecade8a54386, 0x00000000000cacaf}},
    },
    {
        /* digit=38 [{1,2,3,..,}]*([2^152]*G)  */
        {{0x000ea5d38989c046, 0x000c5a933aaf71af, 0x00084b51a5d47afc, 0x000dff8854de4972, 0x000b247bec1525a9, 0x00061e58da8b31d9, 0x000707468a25c846, 0x00000000000786a0}, {0x000d126f8d197fbf, 0x000be282db8ca2e4, 0x0000d8a3ccd2e3a9, 0x0000faaeaeda06f0, 0x0009add94b47a2c4, 0x0000690766963292, 0x00092cc72354f6b0, 0x000000000007878e}},
        {{0x000456ff7fbc201b, 0x00083854b0583e19, 0x0005b9f9d05986b8, 0x00093894c32fc71b, 0x000c9ec8f90dec82, 0x000c7b9c0882fad4, 0x0005e52d39990dc6, 0x00000000000d71a2}, {0x0003262dabc3b450, 0x000866b852e64a5a, 0x000281968ae95022, 0x00011545857f0497, 0x000c1ccb9bc83700, 0x0002bb853746621f, 0x0004ed97e44e6361, 0x00000000000758da}},
        {{0x000e66dca895aedb, 0x0007e58775856e71, 0x0003edbdf1471e8b, 0x0002e3da62265d35, 0x000672d98b0886a1, 0x0001e9c858ec4278, 0x0004ceb9da8016f6, 0x0000000000080099}, {0x0007a34c46f751da, 0x0004f63d0878c9c9, 0x00077928ee65f2b7, 0x0009e126a1c1efae, 0x000eebd0497a780b, 0x00065231eeed68d0, 0x000127bfeee3d292, 0x0000000000080e03}},
        {{0x000369b381ff008e, 0x00068d25f6507829, 0x000435b503d33f46, 0x000031108b9b08bf, 0x000f3fe92e910b36, 0x000189ddc16477af, 0x00038f6e5f6cb103, 0x00000000000c698a}, {0x000953f049518733, 0x000e102a092187fe, 0x000b4e74068daf16, 0x0005eac8fd6b76cb, 0x000611ef96b455a1, 0x000e433b51e37ec3, 0x0004c3d1b3b3fc30, 0x0000000000051d17}},
        {{0x000c3efd55e9f108, 0x0004ee67813dd55d, 0x000e8b95d557829a, 0x0007b634c8cf1647, 0x000fa3556b4674ea, 0x000db03dff1bde0b, 0x0006b45343f260c1, 0x00000000000e0a1d}, {0x000557bdaa85b25c, 0x000b5af56bed0543, 0x0009694c640e2d2a, 0x000c5892c72fa801, 0x0006a49486e504e0, 0x000f78943812e259, 0x000431d5e0bddb2e, 0x00000000000acdba}},
        {{0x0005b2c1b41a7e30, 0x000e8b6e95494a98, 0x000e156b8fa7f1c3, 0x00012ee2183ce113, 0x0000cc01d1434741, 0x000d0d25f4180ec4, 0x00016ddc5f8f7b8d, 0x00000000000974a6}, {0x00054a8b6ee62d01, 0x00072b9a7f0a96a9, 0x0007a0f1f81abc8f, 0x0000b82bc5671a8c, 0x00000466ffaf50eb, 0x000fdc348fa58667, 0x000299a75ff5aab9, 0x000000000007f784}},
        {{0x00064348b9a55592, 0x0008f24b18ccf351, 0x00046d73b67eefc8, 0x000977c532d340c4, 0x000191002448043a, 0x000960397a2de526, 0x00034e1e11027870, 0x000000000000164e}, {0x00056330f30da4d6, 0x00014cc5f57288f2, 0x000974f0c19f8ace, 0x00020963266aaedd, 0x0002c3ccd59f3b15, 0x000a6dd5dfaea30f, 0x00035a1da2e1fbc9, 0x00000000000ceda8}},
        {{0x0007996f48325ebf, 0x0004f7914213d709, 0x000270acf84435ed, 0x000e5a126a34238c, 0x0000701ce8c76eda, 0x000656e02e566bf5, 0x000fbf3562e87555, 0x00000000000b4e8e}, {0x0008d6657de7885e, 0x0004a5f10a503e86, 0x0006a10baa0b8f13, 0x000fc25cc2f2e415, 0x000caf5718c149d1, 0x000d6b890e973bb1, 0x000f129b8825dad2, 0x00000000000e2f00}},
    },
    {
        /* digit=39 [{1,2,3,..,}]*([2^156]*G)  */
        {{0x0009cd8cd3edcb0c, 0x00022e37211bdab0, 0x000bfe0383f52218, 0x0007e26a1b9f8b57, 0x000d7d7f72d5fdcd, 0x00049c9205641e45, 0x0002a15377c1bec4, 0x00000000000efc7b}, {0x0007344da1d40eaf, 0x000b6c657a9ff3f7, 0x000acc6777a25729, 0x000d1bcd020eaa96, 0x000a3f6860c76bfc, 0x000c7c80617534b0, 0x00056b8ce5722284, 0x000000000002bd74}},
        {{0x00070f4fca1b2907, 0x000d2aed02b6a844, 0x00048854f708389d, 0x0006ec4654e7c314, 0x0003e7034bfd8222, 0x000d6b555008ac00, 0x000d44e343c5407f, 0x000000000001b429}, {0x0003fda90bb8f0f9, 0x000bce0702a33908, 0x0003c08ba27edf85, 0x00072f6d46524015, 0x000d35f600437e6d, 0x0003e8cc4d92655e, 0x000b40a0dbaac627, 0x000000000003c3ff}},
        {{0x00098d832fcb2cab, 0x00078bc06ecab0e1, 0x0004e7cd0ece1448, 0x0006fa0453c94bf2, 0x0003ed1a6731a6fc, 0x000c3f1fb5460f94, 0x000d4eeb11656a4e, 0x00000000000ff1e7}, {0x000efd2eb43b2558, 0x0009059526466dba, 0x0007bc3cfc024713, 0x000588824fd2ce63, 0x00039f2c29257bdf, 0x000e97f3013df0c8, 0x0006411bf621e6de, 0x00000000000ebea6}},
        {{0x0003b015d907e90a, 0x0004e5906a35b43f, 0x000f32388b4d1260, 0x000ac9136f847648, 0x000bbe0f1dd365c8, 0x000d26c21f73b3de, 0x000ae740358868a2, 0x0000000000076792}, {0x000fc16ec80c7bf1, 0x0008d3ecea1669e7, 0x000b1ce4e42f6130, 0x00091090e0062443, 0x0006dd94681b6db8, 0x0005a3de2d29106e, 0x0006ccb40b8694a8, 0x00000000000936b8}},
        {{0x000fa93eb46c6ed1, 0x0009f28d33e792c1, 0x000af8e666ab1b38, 0x0009f3bce683c5c2, 0x00098371fe755a74, 0x000712c1d717629d, 0x0001aa5e828fc057, 0x000000000007e4c6}, {0x00082c4505e4cd17, 0x000035d927bad55e, 0x000bbbc997dd1436, 0x00099a398591dc25, 0x000a4836664c560a, 0x000d79298c885fe8, 0x0001d7d18acd4226, 0x00000000000185df}},
        {{0x000755a507c76d63, 0x000c69d8a925b591, 0x000ef5ac5d730610, 0x000ca6ddfb534b8b, 0x000c6dd78a324f53, 0x000a146d54e64874, 0x000201336e5b46c2, 0x0000000000098395}, {0x0001a5ef82624226, 0x0004ca4c095220d8, 0x000ae0a7c3b4840e, 0x0002f64c36286ed0, 0x0003c3ebb08c0ff9, 0x000f00c3057b1b90, 0x00036ec6bdc9b665, 0x000000000009a46d}},
        {{0x000c3639d049078f, 0x00048d67dc92ab51, 0x000e52783edc1242, 0x0009baa8b87c0b05, 0x000052760ef4b6f0, 0x00047bf855a8a903, 0x000e742e2ae75610, 0x0000000000085bd4}, {0x00097e1bd11078ed, 0x00093ced11ff7661, 0x000e3244d9aa20e1, 0x00088ff24b3e912a, 0x000a1afd219683d4, 0x0000d21286e166fd, 0x00039c66a912114f, 0x0000000000005c9f}},
        {{0x00070dd4c4f3a35e, 0x0006d913f92069a9, 0x000aaa1c8b2107ab, 0x0004e2c787c35959, 0x000fe7b7b7ddefab, 0x000e28e6aa55d465, 0x0007bc921aaad834, 0x0000000000012d6a}, {0x000d81dd493823d1, 0x00072109803c417d, 0x00095b2d5e30421d, 0x0008bdb99c5bb670, 0x0006bc7c2da71a8c, 0x000927eef1cd1c2b, 0x00041050189c975f, 0x00000000000229f9}},
    },
    {
        /* digit=40 [{1,2,3,..,}]*([2^160]*G)  */
        {{0x00086c2a1c05c1ca, 0x000a911a8fde5d4c, 0x0008768c091692e8, 0x000c275c74dfe82d, 0x000c38373a506818, 0x000e5f88f2b0294a, 0x00083a584c4061e8, 0x0000000000073423}, {0x0003f61270e03ada, 0x0002d263895b3203, 0x00007b3b0d15f74f, 0x00059da84da7f0d6, 0x000a924a09f21443, 0x0009ad83576e3095, 0x0002af612986e3d6, 0x0000000000039212}},
        {{0x000c342a3198c068, 0x0005d3ee0a31c35b, 0x0003fc2cd3c80f60, 0x0004f4ff0fbfa963, 0x000efafea65a90a6, 0x000f4b9f9513f054, 0x000590796ba7479c, 0x00000000000932a9}, {0x00037547e06b6b4e, 0x0005d93af8a64743, 0x000922627b827de1, 0x00071e9c1a46c909, 0x000f4c9bca7223bc, 0x0008cea30000643a, 0x000ec2b6d967c784, 0x0000000000073312}},
        {{0x000ecb2626cc0655, 0x000eafeb707ee40d, 0x0001edb8dca9d02f, 0x00016459a0d32bae, 0x000a1ffb926626de, 0x000578b4a9a2ff38, 0x0004434e5ad96d8f, 0x00000000000883e7}, {0x00041045bc252635, 0x0002d1d8eecaa72c, 0x000edd696d444309, 0x0002faed758ae41f, 0x00004a85d867e49b, 0x0008cba866a9a229, 0x0009822fb89dcee6, 0x000000000007f09a}},
        {{0x0009f15eb1bbb5c7, 0x000d952ec99556bc, 0x00015795c8a4dc35, 0x000337c6dcc7816c, 0x0009791f7cf1e881, 0x000885a42e4e7b6d, 0x000e41faa717aa59, 0x00000000000f9c01}, {0x000171c4ffe4a6bc, 0x0000ccf208d57a05, 0x00042714fd20944a, 0x000871b264ce04b2, 0x0006cd42026a7261, 0x0009f99fe66be4a8, 0x000e2bc5397b7782, 0x0000000000024578}},
        {{0x000b87e03bf622cc, 0x0002fadc79436506, 0x0000be1fdf9a8888, 0x000aaa40406c1296, 0x000c8b658d3cef9f, 0x000435baff4e6388, 0x000c997262beb41e, 0x00000000000fdaea}, {0x000ba84e0b8f458e, 0x00045d359f7d8428, 0x00016b951ae31b5d, 0x000f03229498ba51, 0x000a85bf35adb18c, 0x000aa4bc8c67388c, 0x00027a8a7a6aa262, 0x0000000000072f46}},
        {{0x0001e4a772f0f152, 0x000520733667a853, 0x000ae90cf08b2488, 0x0009b27a7bc26604, 0x0008eb4f0c7a7ca2, 0x000f276309fefa69, 0x0001337b6f351301, 0x00000000000fcfb1}, {0x00022648d86fc4cc, 0x000979928d9cf841, 0x0009da8838c5ffbb, 0x0001acbe041bef47, 0x000af8d998547fbd, 0x0005a0dc57d25159, 0x0009332ec3a7d8db, 0x0000000000087e3e}},
        {{0x00088fc716a5b4c1, 0x000423fb812eeafe, 0x000a6b9f65779527, 0x00041838fbefbe46, 0x0004a4b24a05e572, 0x0000c0a432b7d49f, 0x0001db23b138d071, 0x00000000000a85ec}, {0x00095de9d8768d19, 0x0006d7cc1f3d657e, 0x000b6e219733e2a8, 0x00023128f56981d2, 0x000eb4080a011bda, 0x000999b7f68663ea, 0x00080c8dd7ba7e9f, 0x00000000000b6865}},
        {{0x000b2ea58bc5b2ba, 0x00008ba8418f2c05, 0x000ab2d74f8719f3, 0x00038d7bce1e82e0, 0x000eb397915b4e64, 0x0004cf6599e489b3, 0x000affa9baea9ffb, 0x0000000000042ef4}, {0x0001f2cfd470aeb9, 0x000eeae3fb532a9f, 0x0001d5334d1ddf3c, 0x0008e28defe047ca, 0x0009d24e60fe8972, 0x0000f710c63a8c67, 0x000d3cadacbf9247, 0x000000000005e198}},
    },
    {
        /* digit=41 [{1,2,3,..,}]*([2^164]*G)  */
        {{0x0008597cdef10944, 0x000f4294d29542ec, 0x00058a58394c4343, 0x0007f9158f95038b, 0x000164420c94fa1a, 0x0003caea8b137981, 0x0003402b686e1e09, 0x00000000000f9e1c}, {0x00006da6276d2e4f, 0x000cacd1beecf39b, 0x000bd69a9fc92254, 0x000c7192dfc2b165, 0x000dcee7e854cecb, 0x000635cc82cdb955, 0x0007d39fefc321f2, 0x000000000002936f}},
        {{0x000f165eb19500c6, 0x000f54962c020b57, 0x000116eb855d7c76, 0x0009a79dd189c401, 0x000d6ce517a77854, 0x000d6bb9747675cd, 0x0000295102294e32, 0x000000000006ed1a}, {0x000926ef8530f97d, 0x000353efb54e82c3, 0x0002eec5a292f22f, 0x000c2948967fb050, 0x0009be04e909955b, 0x0001efad6373615c, 0x00097af1fcf82011, 0x000000000006fd2e}},
        {{0x000445389cf3da63, 0x0002b85e1e4ab383, 0x0004d8f24163478b, 0x0004da1a717e0356, 0x00080f1bf3dfc0b8, 0x000a6fd41f1a63b2, 0x000cab47b74201ab, 0x000000000003518f}, {0x000a17bf41f156b8, 0x00031e12511fae1c, 0x0005a421fc7a58f4, 0x000fb10db5665c05, 0x0003b99f4d5a20e0, 0x000b7dbceb5448a3, 0x000b87cb1847ad46, 0x00000000000fdcad}},
        {{0x00040c75d27c479b, 0x000823f2b5bd3e20, 0x000298ef45cbe8a6, 0x000f2db94256fe1f, 0x0009318ddb03532e, 0x00024c8e1acbfc45, 0x0003db2375f9fd5f, 0x0000000000037078}, {0x0002335b7531e78b, 0x000251bd461e7e12, 0x000e223c32f4b08c, 0x000777d8845f315e, 0x000697fc92c7c9f8, 0x00092954081aed29, 0x000fe09f2f8d7949, 0x00000000000ff5eb}},
        {{0x0004fbb34790cc07, 0x000a4397049f47ff, 0x00000e7e84e96f9b, 0x0005ec1e7862c0cf, 0x000834350bf1ce6d, 0x000ec8d7417a98db, 0x000aa86ceccd8030, 0x00000000000bcab4}, {0x00032ba3b5c44605, 0x000c04378b1fdff3, 0x0002189e90f61242, 0x0000faa8d60f86df, 0x000a3573271abddf, 0x0008f8032987583d, 0x0003f6b0afe4ec4b, 0x00000000000b69d0}},
        {{0x000f8df0532bb051, 0x000c5bd9a66d6010, 0x000c7470f77e12e7, 0x00051dfef38e9b37, 0x0005754fb3fba751, 0x000069b72a3348ae, 0x000faf1218fa8f13, 0x000000000000835d}, {0x0007a4c220543f04, 0x000c02121a98ebed, 0x0005ec4ceabea5d8, 0x000a4be1a1eb6eea, 0x0002adeb3ae51b23, 0x000ea2b45b5c48b9, 0x0008eaeebd305ded, 0x00000000000c3719}},
        {{0x00055d2a529f419b, 0x00062f73471b59c9, 0x0003ecf010a8f9e8, 0x000497dae082cef5, 0x0005c3e563bc57cb, 0x000125f95dfcbf2b, 0x00001922149c0fd1, 0x00000000000425bc}, {0x000c9ac134356e8b, 0x0000e049476de7aa, 0x0008c62ed440f124, 0x000c62d1256424f6, 0x000541bc66fa56c5, 0x0006ada140a118ee, 0x0008c50d8e9ff829, 0x000000000000134f}},
        {{0x000b2c8bea306a01, 0x000b1d0960bd7257, 0x0002ca4efc9832f7, 0x000cc77a98509175, 0x0001e5f81554975c, 0x000cff2c8b41bd53, 0x000be3bdf8ab57c8, 0x00000000000f5822}, {0x000ad597c5610627, 0x000ae1f0ac0e6f99, 0x000855b71125b2a5, 0x0008b82e3cc86ac3, 0x000d2b2beaee22bd, 0x0002617ffc43bde8, 0x0000968168781e41, 0x00000000000b7ee7}},
    },
    {
        /* digit=42 [{1,2,3,..,}]*([2^168]*G)  */
        {{0x000956a7f54f949d, 0x00074086945790c5, 0x000771ce236fcb82, 0x000b0c6000575064, 0x00080b09adeb2c04, 0x00081474f468be6e, 0x000cce2f3bf32b6f, 0x00000000000a712c}, {0x000c24bf8b416a6b, 0x0004c292cee41c19, 0x0008e149d7276386, 0x000a66156f47b2ac, 0x0005840b5d1be54d, 0x0006e8cf62ca7683, 0x00053eb52adb6a8a, 0x00000000000dade1}},
        {{0x00026c98ae834331, 0x000b365f7d2c0d6e, 0x0004cfcca31f7dbe, 0x000634c986436f32, 0x000133356165e268, 0x000c7957d6334d8d, 0x000983f17164269e, 0x00000000000b8093}, {0x0008676a1037657d, 0x000d9edb1fe5cc2a, 0x0001cd1de0787da3, 0x0005f4da691b6657, 0x0001e1e2727e746e, 0x00027b0296117129, 0x000197bb4aa8f16f, 0x000000000007f42c}},
        {{0x000a6e4b3ca52862, 0x000cf2af8bdfa5a6, 0x000675c2d00d6d96, 0x000d5ced7046d2e5, 0x0007d545fd33d57f, 0x00061ffd75ea025f, 0x000242e2ccb6f431, 0x0000000000009406}, {0x000194e235777423, 0x00019f3536d60805, 0x0005fe57dd0b2a05, 0x000b06a5cc554450, 0x000f9a9e2a66fd15, 0x0003dfd0261b0feb, 0x00051fdc3c057665, 0x000000000000a8ab}},
        {{0x00014e15511a3745, 0x00067fe19901abc4, 0x0000c6f09a808e87, 0x000012556c4ce5cb, 0x0005938c89ab92fd, 0x000a587b123172d6, 0x000c50a71f8a33aa, 0x00000000000b55c9}, {0x0009dec34d6b29ab, 0x00056ec005f6a241, 0x000b67510d45fff0, 0x000d67f9e26361fb, 0x000321389c2598a4, 0x000ffbcee7f0a2b2, 0x000888d158820795, 0x000000000009f36a}},
        {{0x00065f3227de5d3c, 0x0001e5ffec1d7642, 0x000dceb00d947f3b, 0x000844c649c850b5, 0x000610fa337dfbe3, 0x00080773d450263e, 0x00066c44f7c8f402, 0x00000000000e8969}, {0x0009e0325576575b, 0x000a42587f475435, 0x0002c020daa3c5c5, 0x0002f667071543b1, 0x000e06577b749e90, 0x0000bee1303398aa, 0x000cec030926d691, 0x00000000000ffa92}},
        {{0x000298023770357c, 0x000dae4ee3345cdf, 0x000943bb20278bd9, 0x000ca667ce118490, 0x000dcb69c7ead817, 0x000680afeda222cd, 0x000ca874ac74709d, 0x00000000000d9596}, {0x0002187a0dd3835c, 0x00082dff57da7e9a, 0x0006d7aa1a2ea94a, 0x000ba28b7fbc1b01, 0x000c13d4ed0f71c0, 0x0003a2ef260faab1, 0x000dff6fbe3567ea, 0x000000000004e577}},
        {{0x000b9f236346df15, 0x000f89b758b50c99, 0x00075b638b06df34, 0x000b3c39ee88a784, 0x0001105e1ec04669, 0x00088ea133f36a67, 0x000aadd0f30e5d9e, 0x00000000000baecb}, {0x000d89dcc4fedb34, 0x000d3ec65be650a4, 0x00093dd7d659e073, 0x0005d29942ce52c4, 0x0007d3ce28d4f719, 0x0007fc9041220187, 0x00055e9c962aa1a9, 0x0000000000065c5a}},
        {{0x00095e9750ada30e, 0x00094eccb421e7ff, 0x000dfa5cb406ba75, 0x000da9e05a53972f, 0x00007bc99fead344, 0x000f77bf53a8035e, 0x00078ae0214485c0, 0x00000000000e54df}, {0x00053ac771ed9aaf, 0x00027def45af5dcd, 0x00024afc33d18821, 0x0007db1c337e0181, 0x0006b84671d5a9b1, 0x0007d696d026a4f6, 0x00022a606142343b, 0x0000000000081cfd}},
    },
    {
        /* digit=43 [{1,2,3,..,}]*([2^172]*G)  */
        {{0x000d2a53b2756567, 0x000f3feb984b33c0, 0x0004e95d9895327d, 0x000f97e3ca0a9a03, 0x000dacc000ac177e, 0x00040f51a76d4796, 0x000810bad0fa6eb1, 0x0000000000006ebc}, {0x000c76db6b103c54, 0x00004f89eb78f367, 0x000ad3bb031162c2, 0x0009cfbb25e9d1c3, 0x000fb4e7aef3e2b2, 0x00067e8459388ea8, 0x0006d7ee606c12e2, 0x000000000009f580}},
        {{0x0008f59540514451, 0x00028cf947c06046, 0x00062665be2e4fbd, 0x00015c05f1835ca3, 0x0002d8c79d90d001, 0x000b5f791df08415, 0x0008cdc0c3a6846d, 0x00000000000749b1}, {0x0000ce2ed12d25c3, 0x000d8d314b7d4a22, 0x000dfaf9b4508139, 0x0005af06855a9438, 0x000ee9d02ab997ed, 0x00038a5d0ea84ae9, 0x000eabb87e903432, 0x00000000000650a0}},
        {{0x000c8f034ea36274, 0x00088357540ab419, 0x0000f5e32c760e57, 0x00073ac14d7fed37, 0x000fd186ee3d33d2, 0x000a7b1d9fef6b9f, 0x000bc2a94c207101, 0x0000000000050bc8}, {0x00013f9560f76983, 0x000e5c68959c999a, 0x000056367228b52b, 0x000a4fce6861310f, 0x0000c828330f7a73, 0x000f74fdb19bcac9, 0x0002f473e3b66f7e, 0x0000000000052d8f}},
        {{0x0008b2d06a3248b0, 0x0008e58225208828, 0x00095368e614a61f, 0x000e45562fe86dbb, 0x00082dc5b2f11224, 0x000096555a8c8a47, 0x0004528a957b8dc3, 0x00000000000b1591}, {0x00063a60dfca11b8, 0x000161e563c57eb9, 0x00024a9e6e7bbb16, 0x0002de99462e7a31, 0x0001c2489214f8f6, 0x00074639c6b3dfe9, 0x0004b56c1d8fc421, 0x00000000000ef88d}},
        {{0x00071f1bee09bff3, 0x0004a6eda384edeb, 0x0001d457a27a9c42, 0x000e6af10e01b58c, 0x000ae1175185c441, 0x0000dfcc6ac962bc, 0x0001b472d34d676f, 0x00000000000a9412}, {0x000f38c043747e7d, 0x000ad77dea14649d, 0x00065b52576e79ef, 0x000c09dd5d17db20, 0x000db45a6e94625a, 0x0001f8caf579edea, 0x000b8ea659282a84, 0x00000000000c11a7}},
        {{0x000f9bf364d3e9eb, 0x000407d0850b3dae, 0x000e31cf311bd53b, 0x000f1f891cb75575, 0x00089d3b6afe68ce, 0x0003b46e427a6ba7, 0x000059f220f9c1f8, 0x00000000000f526e}, {0x0007512e159b9fa5, 0x00094edf3cccac66, 0x0003e3258b2c0e34, 0x000d9e98cc78d3e7, 0x000b5fa48469cc09, 0x000686001391bc40, 0x00073a22bcdcd522, 0x000000000007c599}},
        {{0x0002ae0971dbb084, 0x000bd9de8555bad1, 0x000c9834d42cb891, 0x0000285de654b80c, 0x0008802be80bf17f, 0x000d823bf33998d8, 0x0006b64095923d5a, 0x00000000000e848f}, {0x000fcca2a1ce91f8, 0x00015c6e95017db4, 0x00048a3538b7e8ae, 0x0001be7a70558759, 0x0008441da4770c3f, 0x00048d4c2b652671, 0x0008deaba3b06f9c, 0x00000000000086f1}},
        {{0x000d473962fbc397, 0x000b15a4a5ce3b56, 0x0003f34c786123b8, 0x000388e4efe9a313, 0x000aef9d074bd459, 0x000ab9cb03a45c11, 0x000566f68ab50b93, 0x00000000000ecd9a}, {0x0008303c4d104a42, 0x000f0bfb79b7f7f4, 0x000a0520b1df9755, 0x000aa21877390f14, 0x0005e314cfb54f31, 0x0008eab122de0c64, 0x0004f656d904f623, 0x00000000000ccb4d}},
    },
    {
        /* digit=44 [{1,2,3,..,}]*([2^176]*G)  */
        {{0x0007aa3a213de299, 0x000c93eb83a8707f, 0x000eb1f52edb04b6, 0x000c77a53abe4e9c, 0x000ebb9feb8257b8, 0x0006f922e0a14673, 0x00026cc0a6cbf7f2, 0x00000000000a32e5}, {0x000ec20f9c4de649, 0x00049ca417e66df1, 0x0009b8c741987bef, 0x0000e7a62e135de2, 0x000ea7ee82c72bae, 0x000776c74962bdca, 0x000feb573d7f6ae5, 0x000000000006ffbe}},
        {{0x000a70e677b6f831, 0x000b9cbc0bb18f0d, 0x00037297c3884bb7, 0x000b350d38064428, 0x000c397a7602f62c, 0x000987d82e4d2f1c, 0x0009faa54bd48e43, 0x000000000004f0a1}, {0x000a1b4997cc48fa, 0x00005fa0c9a44d6a, 0x00012729fa8ce7f0, 0x0002c5b9cbd8d343, 0x000813d979568e24, 0x000da55fa1671eb0, 0x000ce5ccae388f40, 0x00000000000f376d}},
        {{0x000f917fe9a26ae0, 0x0003ae54d92cd055, 0x000e0e9599c7454d, 0x0007fd2849f6143a, 0x000dacd0202d7c90, 0x00092d19abb11dd2, 0x000677a4913a701d, 0x00000000000cf610}, {0x000f3626f0aa0d9f, 0x000a3fff1e1d462d, 0x0009edea6495252f, 0x000f33a92cabf724, 0x000ce329d1adcafc, 0x0009a0e4cd571e13, 0x000f867626ad237a, 0x00000000000130d7}},
        {{0x000f8196d845d3e6, 0x0002f8a89daef137, 0x0004ece7e9ffa3ea, 0x00023775b80bb4b5, 0x0002a45b362648d9, 0x0005bfe910a587c5, 0x0007d7daff503cc7, 0x00000000000116d0}, {0x000600a28d48ec00, 0x000916b5a471517d, 0x0001f4eebe019105, 0x0006cfc595abf8dc, 0x0009ed0391ce8f07, 0x000fcabd3c9de4ce, 0x0009edaaaad03ae1, 0x0000000000087b19}},
        {{0x000b51557a9fd4ff, 0x00034b0641d0941f, 0x00049c97e60548cf, 0x0005b4b00d5ec6f7, 0x00015569d89ad12a, 0x000ac72089be1a11, 0x000df19c0566deff, 0x000000000007034e}, {0x00000817b5f1bb5e, 0x000e5636aeb6adf8, 0x0006c2164b0bbfac, 0x000f898734e9d301, 0x00095ccdc6bcf4e2, 0x0008a4f28daf7421, 0x0002c39d39249f60, 0x000000000008820e}},
        {{0x000d9387d2303055, 0x000ac620248118e0, 0x00080c838dc206ab, 0x000eb38d7f033fdf, 0x000232646d6f86b3, 0x000dae596ee3226b, 0x0008e58c4825f6f1, 0x00000000000a5bcb}, {0x0000edaef1eabcc8, 0x0006f904a53484a2, 0x0007fa1deed27103, 0x000329f21d45f8ad, 0x000a06605546af6a, 0x000a93d14f20ad88, 0x000174cf7a0e9619, 0x0000000000091c97}},
        {{0x000ae272638371ad, 0x0005559edd263abb, 0x000509e662e63add, 0x0000d304e7f07169, 0x000119b88200bf4a, 0x000da801e36fc0e4, 0x0002c791db560240, 0x000000000002c72c}, {0x0006387b3ebbf52c, 0x0000dfe960c2596c, 0x00093166fd89300b, 0x00080febccc1576b, 0x0005c3c88f475a64, 0x0007bd8ec72f4e5e, 0x00055e224e7e749a, 0x0000000000076314}},
        {{0x00073d61157ae785, 0x000ab42c9fad4fc2, 0x0000e5d3107f2d93, 0x000c7fdb149854b8, 0x00002fc359eb0cf4, 0x000ec86034a7d900, 0x000e149ff0c3ea29, 0x000000000009b24e}, {0x000512a0eb94d71b, 0x000e0638c80999de, 0x000012d24a63feaf, 0x000e0f8a07ea5482, 0x0002aecdec3f2fb1, 0x00074025b1e580d3, 0x000164bbf895730f, 0x00000000000dd529}},
    },
    {
        /* digit=45 [{1,2,3,..,}]*([2^180]*G)  */
        {{0x00051b4cfaf372e5, 0x000b9fa2db519927, 0x000d4edc529b8ffb, 0x000af1d605917201, 0x000e782a09939f88, 0x000c710d31c5fea5, 0x000365e0fb15884e, 0x00000000000d32ce}, {0x00054592dfab29b3, 0x000d7d9f896aa46a, 0x000ce02cec631ef0, 0x000992dca6e73436, 0x000a6fd0c08b1b76, 0x0000f5c2376338fc, 0x00035bc3ea4c65a0, 0x00000000000b316b}},
        {{0x0001e6b975664a84, 0x000aa14f962e38f4, 0x000c4a3248d485a4, 0x00011c079e777bf3, 0x000b4657c31b2c0e, 0x000494de3a1705ef, 0x00071802688fd23c, 0x00000000000412a8}, {0x000b58909626d6fa, 0x00004878fc7ace41, 0x000bfc58da0a4094, 0x0003c4c704b70c17, 0x00087323c0087c81, 0x000089f1a98553da, 0x0002914f63cec663, 0x000000000009655d}},
        {{0x000f1be887dbd011, 0x0008384e5541ec05, 0x0002bbc7edd0f227, 0x0003ad69beaad3d1, 0x0000f51c2714e0a4, 0x000ea5b0c97dd182, 0x0007caac2b4cb457, 0x00000000000d1176}, {0x000327be0b446dbc, 0x000e318ccf36cb48, 0x000fe4f5fd270bff, 0x000ce43d8d292a8d, 0x0008d58c4ee79811, 0x0005c65a772c5fc6, 0x0006695bc0f0bed3, 0x00000000000cbbf9}},
        {{0x00002ac6c24f6061, 0x0003810586ea68e3, 0x0007603f1b64b158, 0x000c4d7dfbfa352d, 0x00021e1c2291a42f, 0x00011760abf38c38, 0x0004d746e40ef60e, 0x00000000000dcb97}, {0x000c4f8f01f54098, 0x000f1755b105ba56, 0x0001f9ffa1baf4e7, 0x000d0b00945db608, 0x000cf2809e1ca630, 0x0006c95c5a160ac9, 0x00074f38fc1113dc, 0x000000000005d525}},
        {{0x0001da84872cc14a, 0x0009a7eba3da615c, 0x000558d3935c6438, 0x0007d982b93d8b2c, 0x000c7cad1f758c91, 0x000ff150aca8fc6a, 0x0007fa5f581f19a0, 0x00000000000e0832}, {0x0005d538d28d5c50, 0x00038c774f18716a, 0x00051c30fd1c0854, 0x0006e9b8ad72b112, 0x000b917986cfce03, 0x00025cf9b463f9eb, 0x000feabe51632813, 0x00000000000dd7e5}},
        {{0x000710a35bbc6ad8, 0x00005a4e9f29eaf6, 0x000a92c5e19e2d59, 0x00084e42993359de, 0x000f224d5aa30e21, 0x000132c484f96ce7, 0x000f5f0862e2003a, 0x000000000000f015}, {0x00066db2ab4fc1aa, 0x0007cab9d51492bc, 0x0009b538d3bdb7a7, 0x000e671c1fd96e3a, 0x000e71a703b24865, 0x0002add107baf4db, 0x000d3083dd6cf914, 0x0000000000098461}},
        {{0x000b2da1393d8e42, 0x00051714c1d1ba41, 0x000ef78ff03cd88b, 0x000ea3a6951ac80b, 0x0000ac00c8377f23, 0x00024cc1b5c59929, 0x00062bf6efa2b3bf, 0x000000000001e844}, {0x0006a668e721edeb, 0x000069bda627d119, 0x000d91ed1a995ffb, 0x0007089c3b94ec3c, 0x000e3031699ad1ee, 0x0002b2453f75dba6, 0x000ed48ff75f7924, 0x00000000000289bf}},
        {{0x000ac44be8741dd1, 0x0002fcda68a7d811, 0x000fb56aeb52d290, 0x000e2dadce20b92a, 0x000cfc69dca6483b, 0x0004f98917de1601, 0x000a564bec17aaac, 0x000000000008d479}, {0x0009255137ea7d35, 0x00025c623cb8d743, 0x0009f513ea4e4bb1, 0x000b7c030dcde621, 0x00073a1733fdda9b, 0x0002fac31f84ea32, 0x000449d7afb6c3e8, 0x00000000000d3897}},
    },
    {
        /* digit=46 [{1,2,3,..,}]*([2^184]*G)  */
        {{0x0006c80cb0000ec5, 0x0007da189f30f16a, 0x000d675bfc196669, 0x0009ec37d8da76e8, 0x0006c1ea7c10307c, 0x0008c62d4b3e1d00, 0x000b3ac15e20b3b8, 0x000000000000bff3}, {0x000b01f1748ea86b, 0x000e29e330eb12b6, 0x000af2a26953e630, 0x0003cb002e1eb2af, 0x000bf525e4d4157d, 0x000a3dff1638f297, 0x000051a20f833234, 0x0000000000045a9c}},
        {{0x000b35a3033b3940, 0x00007fe9fdde8b8e, 0x0004a1bfc8bd5420, 0x00049acddde6f6e9, 0x00097e54356ca653, 0x00009f73cc53c29c, 0x000277ee15ad9457, 0x00000000000e5429}, {0x000bdd741f2769e6, 0x000d6f52035cdb19, 0x000b835933b3195b, 0x000b0ceca319bd4b, 0x000b6951fd8d26e0, 0x000c34d6f4e7eb67, 0x000b59ac3a6f4395, 0x0000000000000f60}},
        {{0x000d7763c7959780, 0x000c02c47010c514, 0x000f6a495cc56b87, 0x000be509d930f6e7, 0x000d5f56cf045c8a, 0x0002f1fc16bcf875, 0x0007101f6456c006, 0x000000000005304d}, {0x0001668b6ef47661, 0x0006dd76452e46c9, 0x0004266da10fc06f, 0x0009fc89021bde74, 0x000f4babfae0b5ae, 0x000a61fd6505c6b9, 0x00025a99d943c17e, 0x0000000000059bf1}},
        {{0x0003de8a392493e6, 0x0009cbd10b8bfc25, 0x00082fa94d1d5f3e, 0x000be5ec0c907818, 0x00071ad167ce9a18, 0x0008c1c563677f4c, 0x000275ccb254e2a6, 0x00000000000e2c4d}, {0x000250541623e5c3, 0x0003f44958cb1bf9, 0x000db2f9dd62ce34, 0x000a767a2ffdbd52, 0x0009b0b6c22d7445, 0x000c92b2e0e789c9, 0x0007823ff8b6565d, 0x00000000000eca98}},
        {{0x0000ad4fec1eb621, 0x000c36618fcad673, 0x0002540e2f8dc71a, 0x00039947d7ce5530, 0x00095257f24b90ad, 0x00098768cbf8c458, 0x0008305a94992020, 0x00000000000bb6d8}, {0x0007503283325be4, 0x0007dfae8f1616ec, 0x000fc3e0aeb8a2a8, 0x0001ea9139ea0507, 0x00012e39e1d8a72b, 0x0009c7bc282229a9, 0x0008254153bf3e47, 0x00000000000c5541}},
        {{0x000605d76cba7718, 0x000a6fadf9be90b6, 0x000490316f096fe8, 0x00024ec2f9953940, 0x000861203303cbad, 0x00089a4be6236a26, 0x000a82e4bafc3365, 0x000000000005e23f}, {0x000642137e1da447, 0x0005c8ccbc576c76, 0x0003f63011c9e098, 0x0002d5841df8dd26, 0x000914d31fcde6bc, 0x00026010bec24e1d, 0x0002f3acaaf13efc, 0x00000000000e01b9}},
        {{0x0005639fc34e5a27, 0x000c7c52789bb2f8, 0x0006d4ce23fe7231, 0x000f95aacafbbfb9, 0x000dc7f6eb6d8b6d, 0x000b4c9d737afdcb, 0x00045357775bdd6c, 0x0000000000011007}, {0x000bddf07c5f1b9e, 0x000e3903e5ba1399, 0x000d7d2fc919a9a5, 0x00018932d7ac9e4f, 0x000543ce66a8046d, 0x000956410e2fe9d1, 0x000f7244b5beb4d4, 0x00000000000bb147}},
        {{0x0006e03c443812f7, 0x0005c6f6a6104456, 0x00098182647d3e84, 0x0009a6ab51989e5d, 0x000f68f5b16842fd, 0x000f7fe671b60ce0, 0x0005a897324d8756, 0x0000000000067681}, {0x00061f269664533a, 0x0002a265ee993d1f, 0x000a90ce6a02c969, 0x000232334b4adb0c, 0x000e1e5a8d18e909, 0x0004f6456ddcd233, 0x000d9b3dc5b27c5f, 0x000000000007f421}},
    },
    {
        /* digit=47 [{1,2,3,..,}]*([2^188]*G)  */
        {{0x000b170252588bc8, 0x00099bfc40c30a63, 0x0001ef23d6587c46, 0x000b54dc027511d3, 0x0004cf3484ce4fd7, 0x0009beea5f479928, 0x000280655ab81106, 0x000000000007392e}, {0x000148f913baced2, 0x00084522e7b403b5, 0x000493599cdac0b8, 0x000b95877f3913fd, 0x00067dd525149cf1, 0x00008cbca3e06b92, 0x000529992e920e29, 0x000000000006d6ed}},
        {{0x00097dff92ad4838, 0x0001e8a46c9112d1, 0x0000711ed277a798, 0x00064f15cf8e4ca2, 0x000a01cb3488d4ba, 0x000bd7ded01b3908, 0x000ae169b1fa5d38, 0x0000000000018b2e}, {0x000f3694f52a1f22, 0x00019619324bdbe0, 0x000d1d925851b48d, 0x000f42e925b3f6ab, 0x00095f7e4d11a397, 0x0000d9f0132169a4, 0x0002e5c0fa9a548d, 0x0000000000089d8f}},
        {{0x0000115baf0f7c2d, 0x00040240239ae483, 0x000c7482d351827f, 0x0007151f58ec53e4, 0x000cc080d59ff9ab, 0x000f782f4d397862, 0x000d5873eee88536, 0x000000000006c1d8}, {0x000c16042c611b2f, 0x000f258bed3e5b15, 0x00036e097964eba0, 0x000c8ab8a482af89, 0x0009e746a8549044, 0x000a0548e0065858, 0x000a492538d1b926, 0x0000000000007cdd}},
        {{0x000178ddffec9175, 0x0004c9c0d5230baa, 0x0004bce21493d0f7, 0x00090dd985154559, 0x000c8dbfb46a67df, 0x000757ce3223e8b9, 0x000296f39529a36d, 0x00000000000b4648}, {0x00000562605aa919, 0x000dc330749e8973, 0x000a3ffdafa653e5, 0x0000b3494083aa87, 0x000321e321c68c32, 0x0003e78921161f5a, 0x000ccc0980deedde, 0x000000000006ad76}},
        {{0x0003725fda5a777d, 0x00086af9a69e965a, 0x000a3534516a8b8a, 0x000a77f3e52375ce, 0x0009a019a5932dff, 0x000238091ac65569, 0x000085d402f5c4df, 0x00000000000d3518}, {0x0004a37271e8fafd, 0x00007dcee2db4b54, 0x0006c1d813edf12e, 0x0005b6121bc49990, 0x000e68b9808d9cb1, 0x000d6ac5b40b34f9, 0x000b8a98de63590b, 0x0000000000039766}},
        {{0x00069c8c3d7c7657, 0x0005171191261c8a, 0x000244a0eba69bbd, 0x000344bdda57f44c, 0x000ac4f0cfd2ad4b, 0x000543efd674b758, 0x000d063bc058a077, 0x0000000000056618}, {0x000a82ca14a01b7a, 0x0009d95107c74391, 0x000a3c4cfae47f34, 0x000af35e3f1d63cf, 0x000643ab87265dbe, 0x00056c6fd012029c, 0x000e304f588a4ea2, 0x000000000003e5d2}},
        {{0x00081d2046b48f0f, 0x00043847622b5217, 0x000c2a7014a5d0be, 0x0009da7b82c435cd, 0x00025b73e01114da, 0x0008b37b399c8c43, 0x000ddab978fe55ec, 0x00000000000337d6}, {0x00034bf111412925, 0x00071e0d4ffda16d, 0x0003fc3275d2e3f4, 0x00062872913cddbb, 0x0004f67405be2a7d, 0x000a31060229afd0, 0x0009d6e372202e49, 0x000000000009fc21}},
        {{0x0001a0d511e4c022, 0x0009173fa3508062, 0x000e92c1603f0953, 0x000549f58493d985, 0x000adc79f602f64a, 0x000512b84d9ceae0, 0x0001516569e37bd1, 0x00000000000151c9}, {0x000c6addaeefed36, 0x0004c075678c2066, 0x00015cc88eb8c3b8, 0x000dca3a57fb96a6, 0x0000223dc3ce6334, 0x00011e2770ed9082, 0x0008e274f9c3aebd, 0x0000000000079c0b}},
    },
    {
        /* digit=48 [{1,2,3,..,}]*([2^192]*G)  */
        {{0x000688b6fc0935b1, 0x000f5378205dd339, 0x0000b901357b7bc3, 0x000c06c682e00f2c, 0x0003114d5423dbce, 0x00052463ef2a145c, 0x000b0aa01d98747a, 0x000000000007d717}, {0x000d2bf78a72f39e, 0x000d29653bc4f4a5, 0x00051b32471fd3a0, 0x00043dcaf8e3f402, 0x0000e86fe16ef779, 0x0009ffdcf70774a1, 0x0005c96b62e6f1bf, 0x0000000000058874}},
        {{0x000b3ac410563249, 0x000bc2a5f8ecef60, 0x000af14f01d834e5, 0x0001cedc59c4301d, 0x00010111d9989de3, 0x0000d5b951e0f40b, 0x000ab8d29d229f96, 0x00000000000d1dab}, {0x00033bacd39b8f1d, 0x000bd7b225cc8ccc, 0x0003c9f7b44c8f47, 0x00052a1f5fb06b38, 0x000f842b9081009b, 0x0002725128a575d3, 0x000cb7fddb48afe9, 0x000000000000b452}},
        {{0x000dcfd459bff4dd, 0x00050ae10069e26c, 0x000e9f25bee973af, 0x000caf27ebaad0bb, 0x00073dd6119cbbe4, 0x000fceefe5907bf3, 0x000c7e0a723dff9d, 0x00000000000f7cff}, {0x0002a3a44c0ca01e, 0x000d17bc95fa21e6, 0x000c0e71f388ad82, 0x0007ecd27b3335bd, 0x00027b8d7d49316a, 0x00019058fbf08e67, 0x0009ea4b209f93c6, 0x0000000000059d8f}},
        {{0x000cfbdc0726f5f2, 0x000ba167ec88a4a8, 0x00009c64d249271a, 0x000e2443877e6342, 0x000603462cb310f2, 0x0003afcee6321bf2, 0x000dcd1dbd10ee9d, 0x000000000002ca17}, {0x000667ac9826886c, 0x000d509465265738, 0x000151279a7a2541, 0x000b0f95e1c59136, 0x0001757d3a630043, 0x000def1e0a09b94d, 0x000d41533956529f, 0x00000000000d4fed}},
        {{0x000c29ae93761a8a, 0x0008a3459097559c, 0x000f79e8fee087bc, 0x0009a286ec406ef0, 0x0006fee5454dcc93, 0x000257f708d21427, 0x00085e66a0e1a56b, 0x00000000000006fb}, {0x0006c4387ea9f222, 0x0000ac44f9df22bc, 0x000c5644721083d5, 0x000a8224fca1a819, 0x0008f3ed85bf5894, 0x000899b5b8586e41, 0x000371d494dfb202, 0x00000000000ecb8e}},
        {{0x00094e8a6ed08358, 0x000cf690c0cfe457, 0x00092e98638a5e98, 0x0008042204d98a6b, 0x0005bad2eb082250, 0x0005823eec87a97f, 0x0003f6d307c59ed2, 0x000000000000df8b}, {0x0002a9b6fd1bc660, 0x000e9280ae343343, 0x00077184e86c10ae, 0x0000e5a24d04e396, 0x0007309830fcea93, 0x000afeebc0269d9a, 0x00002d41dc8f0ae4, 0x00000000000f14ee}},
        {{0x000ef795d8c64486, 0x00026efca7f7acdf, 0x000411f7c32b0a9c, 0x000b87fe57d08e0a, 0x000a7d9a1967c9ea, 0x000c2749248c01c2, 0x00063911c97ed97d, 0x000000000001cf1f}, {0x000b6334379af438, 0x000d1541f8c0d49b, 0x0006f782375b1fd6, 0x000935ba6be190e1, 0x000764494b4e9806, 0x0003c00b6ec6c5de, 0x000f15f04e2d4cb8, 0x00000000000c0b84}},
        {{0x000c9d81a70917bd, 0x000702e75a26e455, 0x0005bf4870175e47, 0x000d057ee7d8e4ba, 0x000d8994f44953df, 0x000538367b959110, 0x00029cb4a16596bb, 0x00000000000ef82f}, {0x000ea0c85d7d05e6, 0x000305a7642ffe63, 0x000b9d7a5d2d391f, 0x000b23803a4184c4, 0x00020f7fcd62c7ea, 0x000a8a0c660c67ef, 0x0002b041e05799df, 0x0000000000004d35}},
    },
    {
        /* digit=49 [{1,2,3,..,}]*([2^196]*G)  */
        {{0x000ffb708a3b5e85, 0x00030c97c01eab92, 0x0007510b2d7953a7, 0x000ce807fa7d3c2a, 0x000e81060874dba1, 0x0007eeead69e6f96, 0x000f3d6e3e0df74d, 0x00000000000b0c87}, {0x000317c5146f214f, 0x00028c55ae3dbb43, 0x00014b4be1d3dc49, 0x0008c591de7860a7, 0x00066e546731a600, 0x0001e45d48202f8b, 0x00015a652f2d07aa, 0x000000000006df54}},
        {{0x0007d6371007dea2, 0x00049041c706cbe5, 0x000dcf6b55c23258, 0x000cd27839e9d5ae, 0x0003bf3c4c067dc4, 0x000b7bd22dfc9db8, 0x0002da85b8094138, 0x00000000000d0f4c}, {0x0000b16d9a334a33, 0x00092d7b340062c0, 0x0002bb5502deaa2f, 0x000b2c2752366864, 0x00010113a85fa340, 0x000b327045ddd055, 0x0002ff7dfc7ab29c, 0x00000000000dabf2}},
        {{0x000a8373e5c690f8, 0x000ca2fbce9bdf20, 0x00049076d1995e9c, 0x00045939f4cbaf1b, 0x00092574a3bd48ea, 0x00092c39a56c5400, 0x00034384a39630e7, 0x00000000000eef81}, {0x000361503c11fa79, 0x00095f996760edb5, 0x000c1bbc8ea81e13, 0x00012e6966d70279, 0x00052e6f7c63a0ca, 0x000d13ead92a6d5d, 0x00068146809d269b, 0x0000000000067aac}},
        {{0x000cf4e4cd35d7a3, 0x000a13fc9b3cedea, 0x000ac33c871e844f, 0x000a58afe1ad536c, 0x0008cb39149f2003, 0x000edf470cec4be3, 0x0005194d578c99bd, 0x000000000003a356}, {0x0001980c5865f55f, 0x000607a762f2732a, 0x0003ce874c8a141c, 0x000817f270c508e9, 0x00049fdb29c8dc0e, 0x0002711d35a7be20, 0x0000c2fa1a0be3cb, 0x000000000003786a}},
        {{0x00065cdb1cddc024, 0x000c41d0af6b5128, 0x0006106e0f532684, 0x0001951b1ea8fc4c, 0x0004b1fae4826764, 0x00023477bc0b9006, 0x0009ce7012642f66, 0x000000000005bf01}, {0x00001d44438309f3, 0x0007fca1f46757f4, 0x0004d56451db59bb, 0x0004ef2d3868de95, 0x0001044e0c189c03, 0x0005e38c30533d92, 0x000053ba6cf14ecb, 0x00000000000509d7}},
        {{0x000af6aee4d4a85d, 0x000c0e164268de02, 0x0001633ba7cb9816, 0x000979478ab17f45, 0x000e0179ed0e734f, 0x000a2686746d468d, 0x00085d7e68f006df, 0x00000000000e3d04}, {0x00073699ad94d8f6, 0x000c30913a1d74ab, 0x000e2aa1b6d33ea2, 0x000a79e49eadd08e, 0x00017dd8f954eeb1, 0x000bb26d0433f5e4, 0x000e2970a6281430, 0x00000000000ff5e8}},
        {{0x000235f9cfa08aab, 0x000eb6a352b56ce9, 0x0002152b2e478d04, 0x0007c7a240e6dc62, 0x0002d313b4a9ee1f, 0x00001a40585d5be6, 0x000d5a1522c5d25c, 0x00000000000960af}, {0x000459bf66d63a1d, 0x000e4a3cb77f327b, 0x0001a15093d3f2d2, 0x0000c7d3b93fa9e1, 0x00013c0383ad8409, 0x000f7a220c77f1ee, 0x000bfc461c93b776, 0x0000000000004ac0}},
        {{0x000fd3f75cd14c88, 0x0002a3c7d6b63ea7, 0x0002b345f341120e, 0x0005d20aa0eaa1ec, 0x000fc0eab4908ed1, 0x000d9f260e944ad2, 0x000ba371525aa1f6, 0x0000000000016146}, {0x000bd29ab6e83fdb, 0x00068f94019075db, 0x0002a02a4fd970a1, 0x0001c37cab1060af, 0x0000c8cac96f6a4e, 0x0002466ec357fe4d, 0x000e7097a8b8ab6a, 0x000000000009c01b}},
    },
    {
        /* digit=50 [{1,2,3,..,}]*([2^200]*G)  */
        {{0x0007eae876f30205, 0x000645b0b5d68b38, 0x0002f6471178cf56, 0x0000a4a404a3458c, 0x00059f467b6072ad, 0x0006348091de8e25, 0x000178a4b3570590, 0x00000000000706f0}, {0x0007cba07f8d2545, 0x0006d588d21aac4c, 0x0001bb1a8ee3a06e, 0x000e73d241bcd915, 0x00022facc7ccf4e7, 0x00025d2a0b8d8a1d, 0x000608483c35a71d, 0x000000000001ef9f}},
        {{0x0009cd91f152b14c, 0x00034a704015f319, 0x000a64fabfbdef40, 0x000301f2ccb94180, 0x00046f00d8aa697f, 0x00038a0173ee8776, 0x0005432b5afaf881, 0x00000000000832d7}, {0x0002183eafee3abb, 0x000d627c27ce1884, 0x000735007191c91b, 0x0005ac75b752008f, 0x0001e84fe5f192dc, 0x0005929cecf382e0, 0x000ffa90e034197d, 0x0000000000015ca3}},
        {{0x000596896506329d, 0x00058cb51f038efe, 0x00073c05f41ddada, 0x000fafab41fe1a74, 0x000da719f25493c8, 0x0004f5cde6297701, 0x0005426e9165bc64, 0x000000000000c11c}, {0x000368f61fe7d95a, 0x00098a2564809894, 0x000e829acda88407, 0x000592622b1d1be2, 0x00026ecdd041286a, 0x0009f952486a3d75, 0x000b0f4b867e0a64, 0x000000000000629c}},
        {{0x000259f3facaa9bd, 0x0001d11dd860d21a, 0x000b8c19c604b970, 0x000aff635c019302, 0x0001e3a4a900d4f8, 0x00078c8ba96a727b, 0x0007c41426daffde, 0x000000000008d152}, {0x0001e6f4fd354295, 0x0004a0c0d5233cfd, 0x00066c04a38eba93, 0x000bee43d914fb41, 0x0008f3ba26a64828, 0x0004eb26f8324ea3, 0x0007bf027590f3a9, 0x00000000000acd95}},
        {{0x000a71b96f713d9b, 0x00013f4f668435ae, 0x0008fef0f35f5919, 0x000e86e7365712f9, 0x00088a822bc0f607, 0x0001299b3d588229, 0x000b1a2cfbd63ac6, 0x0000000000067167}, {0x0006f5a47be411d6, 0x000b0750f673f622, 0x00032c38df6a058a, 0x0005bd169123c758, 0x0006eab99b375e6d, 0x000aec6a36a93d1b, 0x0008186ef4f7e68c, 0x00000000000cf3ed}},
        {{0x0006410726f50135, 0x000fd959353be170, 0x000b4de98d5dc91d, 0x00026f799d7a4f4a, 0x000e52fe4b656a48, 0x000038573ab146ec, 0x000e8494fc21d735, 0x00000000000f4d56}, {0x0006901ebf8c490f, 0x00093e15ca04c71d, 0x000ef178dcf47997, 0x00079244f21a9114, 0x0009dcc76132ef7e, 0x000e890482eecb7e, 0x0002c55b484745db, 0x000000000006e43a}},
        {{0x000b8d876ab51a4b, 0x0001af92b3072f8e, 0x000d8f5d67f2d2e3, 0x000d5edc578e3a39, 0x00029587fa22e51b, 0x0002eba85efda70d, 0x000530cfec17089b, 0x00000000000af7ba}, {0x0004893a5eb2bed8, 0x000bb5ac155ae396, 0x0009a3394a2b6335, 0x00086c2c38718a82, 0x0003d63745b7280e, 0x0008a79aa9d12de7, 0x000bf70e8ea855bf, 0x00000000000bd705}},
        {{0x000260c123f30563, 0x000c53ede2484b68, 0x000620a80e97a435, 0x0009e93962a667bd, 0x000b130f2cea5606, 0x000366a66c931266, 0x0003b14bd6a6fca7, 0x00000000000aa5ac}, {0x0004e3f2adddce7d, 0x00044a025d0ef29e, 0x00075ab6560ff06a, 0x000927f2b3057f30, 0x000a1499f8844809, 0x000b9a653b001c10, 0x000d05309d141c30, 0x00000000000bf659}},
    },
    {
        /* digit=51 [{1,2,3,..,}]*([2^204]*G)  */
        {{0x000d68f9d41abcc8, 0x00016a6c328ffdb0, 0x000797038aa63e5d, 0x0007d39063de7eb8, 0x000710daf9bd691e, 0x0008b5d7a998df4e, 0x0004b8c7085b9e71, 0x0000000000016b3d}, {0x00019da01ecaa2a1, 0x000494dfce693daf, 0x0007011a8e84696a, 0x00004bf4491fb345, 0x00014552451c2c19, 0x0005e5e407c1bf11, 0x0003726562cc2c3c, 0x00000000000fe0e4}},
        {{0x00073ecab0b13cfe, 0x0002484c3630b425, 0x0005d7cee5256fab, 0x000125ff61af001c, 0x000fbfea35255abd, 0x000e0cb6e69bca56, 0x0005a6384af19900, 0x00000000000d0047}, {0x0001438f80a7fcba, 0x00090edafde48dd9, 0x000a30b2135b9aaa, 0x000f97a6c8ffcca5, 0x0003e5a9cc5cf14c, 0x000104e054d6cec2, 0x0007c1b0d678f88a, 0x000000000009fb52}},
        {{0x000cf7bdfab400e3, 0x0009e8618ffa6a37, 0x00041539f0cbda9a, 0x0000d744f61edff9, 0x0009eb7a476f5b1b, 0x0002ee99b33df67d, 0x0002cc7f1a767ea9, 0x000000000002223a}, {0x0004324c9cd0a9f1, 0x000f616f376f2586, 0x000c0794e16b9222, 0x000516ca58765df0, 0x00062260ed6b8dc1, 0x0004b29ba8934082, 0x000bba060eb0afcf, 0x000000000001d252}},
        {{0x000bb25431b5857e, 0x000160d6cadbf906, 0x000790df51943fb8, 0x0003c734ab19507e, 0x0005660086b33b43, 0x000eb0f434fc5340, 0x00058d770ae903a1, 0x00000000000f6b5b}, {0x000dc8bc8bb4bdd2, 0x000d6e206e14147d, 0x00079b5341d6e69b, 0x000ca2f47449e081, 0x000639fef8e1cfbf, 0x0006fc80cebaef25, 0x00061b959e37b8c1, 0x00000000000e911d}},
        {{0x0006b0541df5b61f, 0x000b0014a0907c72, 0x00060742ec5c6420, 0x0007c4dc999acc04, 0x00075e7ab1e3dbbb, 0x0008fb11e01b7710, 0x0005c2a33fefbfcd, 0x00000000000c0b8a}, {0x0004467ba7747f4b, 0x000ec774dc2669d0, 0x0007562d1fca7010, 0x000dc694d9c84626, 0x000a1e772c5f5ac4, 0x00083fe91bf6e002, 0x000dce4922120e0b, 0x000000000003efba}},
        {{0x000263fbc0d157fd, 0x0005bc483d4b1827, 0x00037dfadf4ae121, 0x000df6fcbab1fb10, 0x0000f6cdfc0c5165, 0x0004320fceb28437, 0x000e80ab565c0099, 0x0000000000062133}, {0x0007401414422436, 0x000ee7850cda5472, 0x0007bd0ba094b0ec, 0x000805cc2c82eddf, 0x0006cf244539d14b, 0x000dbe92dcb5468b, 0x000f1e97d43ee825, 0x0000000000089fb8}},
        {{0x000a494639f26e1a, 0x000b5421afbe0092, 0x0004dc6db28bf654, 0x000cf4db1a2705ad, 0x0006d128bcd556ca, 0x000191ad86a3a413, 0x000d242411c4b866, 0x0000000000015b45}, {0x000845fec1268b55, 0x000a74cc82459052, 0x000da0992b42bc3f, 0x000bb1c69f6298e1, 0x00031933fdb59c88, 0x0001308c3fe567b0, 0x000dfa8a6aea7188, 0x00000000000a2f0a}},
        {{0x0006a4d1fcc1ad65, 0x000d2d7bf2ac5a3e, 0x0005e362c26e5671, 0x000abaa9afa97632, 0x0004b62b36ab162f, 0x0006a84e97b7f166, 0x00043d77b9f79729, 0x000000000007dbd8}, {0x000519c3add29e33, 0x000fcc6e9c1e11fa, 0x000ac380a63f4305, 0x000a93d3bfa90c04, 0x0009d050e46afa7f, 0x000c5625655846fe, 0x000a65473b9a0d35, 0x000000000002b656}},
    },
    {
        /* digit=52 [{1,2,3,..,}]*([2^208]*G)  */
        {{0x000f0e1b25d4e4dc, 0x000a6372798f002b, 0x000537a488c42515, 0x000f9a98a1e25677, 0x000ce70391c85e64, 0x000f024585870254, 0x000bed2def81a341, 0x000000000001d087}, {0x0002a1629bafa8b1, 0x0006d3557d07cb43, 0x0009543a3877e0bb, 0x000c5675f73ba510, 0x000b9c7c670608c8, 0x0009850725309050, 0x000e962ab67da3bd, 0x00000000000e5df4}},
        {{0x00057ab93a62b1b3, 0x0004b7be81fb0ec2, 0x000d385405273506, 0x00040e27a8d16791, 0x000ad520811ebb3d, 0x000f65d231806c67, 0x0003d7add4bb6686, 0x00000000000e20e2}, {0x0008a96c64700a7f, 0x00088208b470000d, 0x000907fb1a1c5c32, 0x00064f8121c37e26, 0x0001a598efbbbd39, 0x000eef966d35ef30, 0x000e46bd76a276c5, 0x000000000000af64}},
        {{0x0007b8ba2901e630, 0x000573f40494a69d, 0x0001d7e86c246f17, 0x0003360c9e634b1f, 0x00096ab166bbacc3, 0x000fdb67e6cf72ff, 0x000736477d8f2db9, 0x00000000000cb644}, {0x0000821b2e82caf8, 0x0007549454e1ad4f, 0x000486f6c48cff7d, 0x0005f0be8e7b06ec, 0x00047dc40b498042, 0x000ed620b862df52, 0x000a648ca7d7c812, 0x000000000003c45b}},
        {{0x000d7620f273aa67, 0x000e1169474a1e10, 0x000056ab42590c74, 0x000de922ede425c9, 0x000a6df8a8908589, 0x000a8b8e350e03fe, 0x00091a0d8a5c1c4b, 0x000000000003fffb}, {0x00026981fafa18b5, 0x000f721cf05437b0, 0x0007e513859293da, 0x0007eaed0962c826, 0x0004213f6004c323, 0x000148b6b43d6ac3, 0x00080a45e619b2d4, 0x00000000000ea5fb}},
        {{0x00014768f5f99aa5, 0x00067314ea217285, 0x000017c8fd29a716, 0x000fb46a63ea8fc5, 0x000890d84e5b0902, 0x000e49b8a925a6dc, 0x000be5e2e74f9c14, 0x0000000000007d45}, {0x000cae18673b6270, 0x000667b768d075ba, 0x00089b2a5deeff6b, 0x000223360d5b216a, 0x00080a7386f475db, 0x000c47746b132b67, 0x00031d7f933fd580, 0x00000000000fbaa6}},
        {{0x000ee5b308cc45e3, 0x0002faba967ac481, 0x000d29b2fe96bd68, 0x000c601ef5f681db, 0x000ffddc580fe033, 0x0007572d85c34f77, 0x0004d0d30f5b66f3, 0x00000000000da20d}, {0x000030e8bcc549d9, 0x0002cdb2310273e6, 0x000ec784d2efa81a, 0x000a33f7899cd7f5, 0x000fda29c3821cce, 0x000e14ecf0f4e0a7, 0x000839c6d7c5f32b, 0x00000000000d9caa}},
        {{0x0003fe55b28c2fec, 0x0007ba884edf1601, 0x000775572e4af6c1, 0x00004152d7852a27, 0x0007f26efb4c66d0, 0x00022f8cb34732d7, 0x000ff518b3ef8e29, 0x0000000000018bcc}, {0x000ec4cab3e21461, 0x00004a219cb1deb6, 0x000868a49e96a154, 0x00099e1d90760ec0, 0x00078a94df2ef0af, 0x000058f89e6fe194, 0x000d9764b5dfcd04, 0x0000000000023d21}},
        {{0x000d7944d758c20d, 0x000209a8580a957a, 0x000f955204f37a28, 0x0000970be07f7827, 0x0000712f7b7cb4da, 0x0006a7b970ac2a26, 0x000f62c9b8ee8443, 0x0000000000011fb2}, {0x000ff5f68230c1a3, 0x0002a5daabed96f3, 0x0003bdf181469c85, 0x000b5b7d96cfb8e8, 0x000344d9e84382d6, 0x000e7da3c4d7d0d9, 0x000253fa2a9ea991, 0x00000000000d531b}},
    },
    {
        /* digit=53 [{1,2,3,..,}]*([2^212]*G)  */
        {{0x0003b2a8a65e5b7e, 0x000424cc41f278dc, 0x000bf1d7ec4af5a5, 0x00066640fa1ca255, 0x000b91e5edaf7053, 0x000d3de14eeb40f3, 0x000c43cdf98235f1, 0x00000000000ff018}, {0x000927ebce051283, 0x00074aa3228e6dee, 0x00043f750dae9462, 0x0000425650b2dab8, 0x00026d875f1790e9, 0x000e8a46ee4a8cad, 0x000fb5c212029c9c, 0x000000000005ed7c}},
        {{0x0007539a8f390740, 0x0008eadb5966f40b, 0x000e0b7342eb902f, 0x00073e244693a961, 0x00055982bbd3a76e, 0x0002ca13214da743, 0x000e7646e982cd5d, 0x0000000000024938}, {0x000cf856b36cb844, 0x00029749206b2571, 0x0006030c0c47215a, 0x000d1567025bb7cf, 0x000e19555c9ebee3, 0x0001639bae23f0e1, 0x000bd00dec383775, 0x0000000000005d43}},
        {{0x0002c235491635a5, 0x000e4e4e52e86121, 0x000459dc25e36e9d, 0x00051bfb49f2b393, 0x000b3f8097cf73b9, 0x0008fbf057b6cb7d, 0x0002119dfb8d0b32, 0x00000000000ebce6}, {0x000c890c36814c6b, 0x00007a31a15235a7, 0x0009c26d4a535440, 0x000834e6b5638766, 0x000d10ee5a281d22, 0x000aee4eafd91b30, 0x000a763d7a282d59, 0x0000000000073300}},
        {{0x000f6efbfb5bea3a, 0x000f878b5c14b0f6, 0x0005485b973e6dbd, 0x0002ab209e1759fd, 0x000db4b2f68cffa2, 0x0005be7f45a86263, 0x000f6d71e77c516b, 0x0000000000019844}, {0x000dc7fe7c7337ad, 0x0009d2519d0058c1, 0x000edd3b9e6ca5d6, 0x00074a685b3c2a9b, 0x000fc294f4492c6d, 0x00069fb469306f68, 0x000886552e77c22c, 0x0000000000010bb9}},
        {{0x0001ae09a4f32c66, 0x0007beba7daac862, 0x000767fe0f73dc31, 0x00018f885bdbc832, 0x000094d43909985c, 0x0009e108b86555ff, 0x000313b0b1b2b653, 0x00000000000c0bf1}, {0x0003b62754d457e4, 0x00021bd4777c10d3, 0x0007d7f58d2fb40a, 0x00057374a27f1ddc, 0x000eeaaa58ab85bc, 0x00076fac29a8ae24, 0x000377161cb2f5e8, 0x000000000006636e}},
        {{0x0001f89428b5c457, 0x0007f1674b959a73, 0x000b96ebf7106c2e, 0x000a32dc67c36488, 0x000368d720a63962, 0x00057a5b24949617, 0x000c0f4e81df85a9, 0x0000000000053123}, {0x0003624d70103a1a, 0x0003f5091dd340e1, 0x0000c9fe10861f33, 0x00020f52c119dbe8, 0x0006c94d609a5e77, 0x000dccd1fd584ae7, 0x000c6e476c63ba86, 0x0000000000032508}},
        {{0x000df9bca60288d5, 0x000016bbf77cab44, 0x0000fa9d18796041, 0x000eb1a2b9febf8d, 0x0001a25330ce357f, 0x00091799874240a8, 0x000f5c7a9ab575b4, 0x00000000000eda3e}, {0x000c7149276e2420, 0x00036360410d2e37, 0x0006d4d0d5e12db0, 0x000b466cc381b581, 0x0008247a49047bae, 0x000c58130024a98b, 0x0006d26e70b4c3e3, 0x00000000000ae8a5}},
        {{0x000d9a7dd453995c, 0x000393313a9d4705, 0x000fd95bba01fcaa, 0x000ef915e4dd5cea, 0x0003c565dd67c0fd, 0x000ed05ac902a2a9, 0x000ae9d8eba4dc7f, 0x00000000000e157d}, {0x00019071237f3ae4, 0x00006d655d0b3ced, 0x000513db82a990cd, 0x000525a0652872b6, 0x000fe68c0ddb5b7e, 0x0001cb31caf7968e, 0x00071e2ec02930f5, 0x00000000000f2be0}},
    },
    {
        /* digit=54 [{1,2,3,..,}]*([2^216]*G)  */
        {{0x0003b3ac56ccd2a3, 0x000649b23ab4e3e0, 0x000d023509576972, 0x0009e51e798edf99, 0x0009307675c7dbe9, 0x0008c0fb63854744, 0x00037223ffaf5562, 0x000000000001698c}, {0x000420dd9073adb8, 0x000d039f45a56f2d, 0x00011e9a2cdfa00e, 0x000079e4af138fd7, 0x000a2ee4ecc02a89, 0x000bbf92fb86371e, 0x000c51076d256a06, 0x00000000000ae3c4}},
        {{0x000340cb6908d50e, 0x00092ba2e95430b3, 0x000660e7e985a29e, 0x000b95145bdc19ee, 0x000e382e77bdf94d, 0x00020b29a951d00a, 0x0001f19940a5fbb2, 0x0000000000058fc9}, {0x000d804932dbc0b5, 0x000be682e42eaaa2, 0x000400a2efd4aee0, 0x000810016294d055, 0x00032e326d68be15, 0x000e64fceaea13fe, 0x000a8ac0dfe1ef15, 0x00000000000b8237}},
        {{0x0004480f8fce3f16, 0x000a7e59b80017bf, 0x0006c7396aa46dc8, 0x000172a5af5b47f5, 0x0000160d7e8d8799, 0x000f9a549f72c978, 0x00044a1d1ce972b2, 0x000000000004857b}, {0x000d15fb2758caea, 0x000542545bdd6f77, 0x000984fe91e9b1e2, 0x000343a4e23c0644, 0x00091d1fd9cd5a60, 0x00070b5b3986779f, 0x0005a35bd5611b35, 0x00000000000f9d76}},
        {{0x000b72123cb1cd13, 0x000e76ee65a0886b, 0x00081e4e332045b1, 0x000cc382876e523b, 0x0006d3bf53aac4a2, 0x0007f290cab7aba2, 0x0005bd5bf00871db, 0x000000000001ee6d}, {0x000bcea869ddbc32, 0x000334cafb21874b, 0x000bcaaf9d600f4d, 0x000785520b281cd0, 0x000e64d9c65ea1fe, 0x000a0e67be457198, 0x00068aa3d1a6d0c5, 0x0000000000090cc4}},
        {{0x000450a44e08b4d9, 0x00014cb0a365753f, 0x000b82633a02b2b6, 0x000210997ed887af, 0x000f30d9b2970b85, 0x000fb9c745fec3e1, 0x0007854ce4149f10, 0x000000000008cbff}, {0x000dc4bd785a06f1, 0x0009f81b0d7b3b6a, 0x000116390fc1ac37, 0x00021de2b841eb88, 0x000ad83c22e6aec2, 0x000affe9162c7d86, 0x00081d5504dcf885, 0x00000000000f4454}},
        {{0x000578651af84c0a, 0x000e0d4ee3f7f52b, 0x000cec289c787837, 0x000ee1363ebab5bb, 0x0007005ec2374c0b, 0x0002fb00670e32d7, 0x000899302fc73dc5, 0x000000000008f159}, {0x000ba114d96a8216, 0x0009d42a5478e2d1, 0x000e66d84b639b08, 0x0004970c8378f0e8, 0x00058e2c86c5042c, 0x000c7c76770c1957, 0x0003a861a95e6884, 0x00000000000d6fb4}},
        {{0x0000a2299e18ff96, 0x0001ceaf237a8503, 0x0006d80455ecbada, 0x000fa473f251ad61, 0x0006d828578e5fbf, 0x000e118adc40570f, 0x0005485cc65c0dd4, 0x000000000005da48}, {0x00073e60bf0732eb, 0x0000fe27fc2e7307, 0x0009067267d2e6a8, 0x0002fa55e27fb12d, 0x000810003fae35a3, 0x0009800c17fcfd72, 0x0002e6c74b50a3f4, 0x00000000000dbafb}},
        {{0x0002a6bfc8996b96, 0x000bd0c62fd2c8ba, 0x000a840806b7cf85, 0x000933dcef3f9e43, 0x000d9889ffa276b0, 0x0003c88d251b1ec2, 0x00052f9e84b2ba9a, 0x000000000001913e}, {0x000a4507b899f92f, 0x000e6bafc5e94164, 0x0002238654296051, 0x000cc41bed171099, 0x00036c7a41c84e9b, 0x0005369cd0db5b73, 0x000934d4be07a779, 0x000000000007bd3a}},
    },
    {
        /* digit=55 [{1,2,3,..,}]*([2^220]*G)  */
        {{0x0006d08f59277dc6, 0x0008a3f2eff5384f, 0x00049c170a3dfb6a, 0x000b18a0dde190dc, 0x000da26b0409af10, 0x000b1d944f491b98, 0x00054166080782a2, 0x0000000000097e8c}, {0x000baebab71369f0, 0x000d1fdbfc5f5495, 0x000a70804cb1f0f5, 0x000263857645ef4f, 0x0006a02583638e5d, 0x0005d250331bcfda, 0x000285f5330ab8d3, 0x00000000000c7cab}},
        {{0x0004a7ee3780eead, 0x0001ef16938f4dd4, 0x0005af2b9dcbcc11, 0x00095530b6490d71, 0x0001a28a296a2d50, 0x00026415c8432fef, 0x0008656f254dd08d, 0x00000000000d50c2}, {0x0005457026e64224, 0x0003c4f5bc4553f7, 0x0006183dc27db1b2, 0x000dd6e65e593411, 0x0002a56dc2eabab8, 0x000a90e05676baca, 0x000da038eea06bea, 0x00000000000174ba}},
        {{0x00021d43da6aa865, 0x0005de6a19dcb664, 0x0006a4c857b63184, 0x0009b9fc6455613f, 0x0004a7390d0eb4d8, 0x000ea135a6cb0fe4, 0x000982ade197a459, 0x0000000000020680}, {0x000776554c3cb5c6, 0x000b803db9be90e0, 0x000e56e339783849, 0x000e8d4753c196c6, 0x00000b7c6ff544de, 0x000a1b14259adcc7, 0x0004f2c6260ec24c, 0x0000000000046cbd}},
        {{0x000c69e90d279f7b, 0x00051a35e411c1f8, 0x000aa4eec7d05943, 0x000859e89a66f2be, 0x000e0def8ecd7c7c, 0x0004947b79908c37, 0x000ce88274124e34, 0x00000000000568b0}, {0x000eb0436a41e1a9, 0x00070e52919611c1, 0x000a98c568a44a8e, 0x00039e156bd3a7e1, 0x0006268f856260fb, 0x000fd8293e56a34d, 0x000fcbb3a1fe1613, 0x0000000000067537}},
        {{0x000d9811d879790e, 0x000da8a15d6fdcb9, 0x0006c38fcc4a52b8, 0x000e38c55bbe733d, 0x00051ff94a9d7a7e, 0x000585ab5eff146a, 0x000a5de572d13edd, 0x0000000000006491}, {0x000a7bbc541e4f72, 0x000a29c84e1c4d63, 0x000bbb62e6c0b1d5, 0x000cd9b385f06c82, 0x00077a7759c3db12, 0x0003bd7060c93eb9, 0x000d50f1f5b0fe68, 0x0000000000085ec3}},
        {{0x000e87011f7cd75e, 0x0006e89e48d8ba73, 0x0007fdc53e3e2631, 0x00033d7302c0daa2, 0x000a048eefe360f0, 0x000a7224415e4578, 0x0009cdfd6dec89b0, 0x0000000000030948}, {0x0003345fd128739e, 0x000ad7cdcc2a0188, 0x0002b63966c3b413, 0x0000a455812b560a, 0x00052ca31d8ca630, 0x0003a5b5a8fa5c41, 0x0004e036aa3c234f, 0x00000000000c86cf}},
        {{0x0004ff5664ce36b2, 0x0005e9a0e15351cb, 0x00019cbdd0d2f66a, 0x00059eafb29777cc, 0x000ae354cafdc170, 0x00007c3717e40e5f, 0x0009459cf594054d, 0x00000000000a71c3}, {0x0007429ea783b1e9, 0x0003469309e95af4, 0x0004f55088c266f7, 0x0004070e25823b6e, 0x000d0bc27359f216, 0x000925094ead851b, 0x000f4e3d21bfe8b0, 0x0000000000034a97}},
        {{0x000a4c18541d03ec, 0x0004ad927282fbf3, 0x0005c034c274cf2b, 0x000207f450db7135, 0x000423e16d9558b9, 0x0000e349cae95338, 0x0002bc4f10c6d4e6, 0x00000000000feb12}, {0x000eced76985b33a, 0x0002f22548cd1c2d, 0x0005b37b87399908, 0x000f912b6167b3cc, 0x00027902d2baa1c6, 0x000de34ba6967bab, 0x00025eebbe0b0836, 0x000000000004b796}},
    },
    {
        /* digit=56 [{1,2,3,..,}]*([2^224]*G)  */
        {{0x000e99ecf706c6bf, 0x0005c9e857f32800, 0x0001e880e21c15d7, 0x0008d68ff4f65674, 0x0005ac339148f853, 0x000dfc12f35380f1, 0x00093efef0bfdd5d, 0x000000000001387d}, {0x0009274bbe5eb9e6, 0x000aa618ce77c94f, 0x000ef0d12ae1c332, 0x000f06e00dc0da6a, 0x000e07603cc724ea, 0x0006963c7049113b, 0x0003005cf489088f, 0x00000000000ede4a}},
        {{0x000abae3c29bb132, 0x000af77e486f79a6, 0x000ea167f51170e1, 0x00028ab7df36628c, 0x00016704dcd6322b, 0x0009a35672d14d13, 0x0003b6d359977af2, 0x00000000000ec96d}, {0x00053212afaa74ef, 0x000f0fd6e400a371, 0x0003860e13fc28c5, 0x0001c7d9b8533afb, 0x00028de66eb862d8, 0x0006784eeefa638c, 0x0002237405a9d7e8, 0x00000000000a6c22}},
        {{0x000fc1e6b9032350, 0x000a46909994e4c6, 0x0006261c6638f0ac, 0x000ca05884aacaa5, 0x000996995a981505, 0x0002c000ee4b6530, 0x000b0930e00a5ed0, 0x00000000000236e7}, {0x0005ec99c1d0db26, 0x0002ce696f09d532, 0x0002f7914e3f9268, 0x000a7b401e1e2a4e, 0x00069d2d025aa9ad, 0x0004ffeb19630acb, 0x0004f69fab2c6ed8, 0x00000000000ab758}},
        {{0x000c87e27d06e6af, 0x00073f9b2dba43cc, 0x000cbbdd7e7ab099, 0x000a4f33b8104eed, 0x0000e4e1896e7692, 0x000d2aa365da885b, 0x000bcac2a30fec73, 0x0000000000086f60}, {0x000adfed330d989a, 0x00086bc8bf16d541, 0x000f4b7104707db4, 0x000e2f37e35610a9, 0x000482f9d71c8e79, 0x000e62733981139f, 0x000061f8997ec424, 0x00000000000a3518}},
        {{0x000efb4736bf4182, 0x0003f6cf0e6ef64b, 0x000b24ffed39dfee, 0x0007783856111dce, 0x0000c0e9f2b00277, 0x0007fe5073a1d36f, 0x0008f1fcf4f6365e, 0x000000000007fa7c}, {0x000ce2543c17ec02, 0x0005509a02de874d, 0x000cd3e25ee5e59f, 0x000a9654b7f4e35d, 0x0009805b58bd7211, 0x00057860ca6b2ba7, 0x000d58418302c209, 0x00000000000f99f9}},
        {{0x000634f7fb73c6b6, 0x0002e4e6ef40fcf1, 0x000c701a714f0702, 0x0003403fd41d144d, 0x0007e0774c37a4f0, 0x000c7484a3a50717, 0x00066b078e8c568e, 0x00000000000fbb3f}, {0x0000fb0d6daae4e9, 0x0002c169c9474ce5, 0x00027d6aef77ce07, 0x000968508303114b, 0x0008fad0def23e8e, 0x000c1c8da7a9797b, 0x0007210ad14404ef, 0x0000000000021ced}},
        {{0x000169a6e51baf05, 0x00088fde0d1b3e6e, 0x0008e5407b7aa0be, 0x000ad79c9eb9de48, 0x000b0ffbcdac16d3, 0x00020287c2707ec8, 0x00055ad7e6750fa2, 0x000000000009c2e1}, {0x000dcbd856a04522, 0x000e43018c309307, 0x000def4e0648d266, 0x00023aecf15a4af5, 0x000cc1b8cca01aa3, 0x00043a969f085d69, 0x00043047a3eaccb1, 0x00000000000f3a98}},
        {{0x000270a279eabd20, 0x000d5c7e9ddef0ef, 0x00060c66b8938b7d, 0x000746db239bb82a, 0x000b28ea13416bc0, 0x0001309b0c811a8e, 0x000b345f714ca71d, 0x00000000000d4eb9}, {0x000f50441ed062cb, 0x00091e0e5afdcc03, 0x0009d20438aad877, 0x0001e7b843343663, 0x000a0eed1116670e, 0x0009bbd50c8c38f9, 0x00095af914fae261, 0x0000000000051c19}},
    },
    {
        /* digit=57 [{1,2,3,..,}]*([2^228]*G)  */
        {{0x0001b38ab493e121, 0x0005bde849cd1240, 0x000576b3d2c358dc, 0x0009e3dabe92fbab, 0x00043324900a3fbd, 0x00020904e785414d, 0x000ba8daead1abde, 0x00000000000aa5f1}, {0x0002d0438c4bd099, 0x0002fd60a4f2ce26, 0x000593174efc1656, 0x000c78934efa243c, 0x000f216a8d8c163d, 0x00001617b3067dcc, 0x00051e116b6534a9, 0x00000000000bbabe}},
        {{0x0004b6e85f0076cc, 0x000c1929454f6549, 0x000021b9b8ac3fe0, 0x000a7c5ee25c0b0a, 0x000f2e752295f5b0, 0x000acac687d3372f, 0x000e3cd6dadc7d6e, 0x00000000000a96a8}, {0x0006465e062c14dc, 0x00030ea831db66b4, 0x0000548165c8c6c9, 0x00017e572e3c00c5, 0x000d2a5fb6ba5ff8, 0x000392476b022e25, 0x0005a0b611c5bcbb, 0x0000000000019048}},
        {{0x0000dddcc280d252, 0x000d5efda99d90b9, 0x0002988f9d0202d2, 0x0006cd1ad14ac705, 0x00031f4138808b9e, 0x000dd7fb91239ee3, 0x000b12d98e93d993, 0x00000000000894fc}, {0x000d9440883321ae, 0x000d433a92019c9f, 0x000ee20fd3f674ff, 0x00051280d0a320b4, 0x000b4b607b538450, 0x000228c2ec20551d, 0x00025c6e63c766ea, 0x00000000000ac48f}},
        {{0x000ea4ad3f5b0bfd, 0x000140372678d84b, 0x0008ab3dd6009aeb, 0x000bca4b79594c43, 0x000baf3b75cfebae, 0x0001e09c6e587850, 0x0004cd534183ac2c, 0x00000000000c0820}, {0x00012c542116a023, 0x000a7dac2cf06c18, 0x000f5e79e9f15f10, 0x00009f490b0f6c27, 0x0006c2c62207f6f5, 0x000ff18873ffc3cd, 0x000c6fbb21eb1132, 0x00000000000e62cc}},
        {{0x000a11bec64a35cd, 0x0004f1ca74a30c77, 0x000de3a654c55d5a, 0x00049038d6b4b005, 0x00002f7906ee3709, 0x000452bbb12ba86e, 0x000039aa76f77adc, 0x00000000000f9837}, {0x0000782bfd430ed7, 0x0001841fe306f87d, 0x000ce68ff3cdd73a, 0x00024ccaa7d44b2c, 0x0005d86900f9cffa, 0x0003ecfa022bae39, 0x000980e7a138782e, 0x0000000000086d28}},
        {{0x000489915b0c1e42, 0x0000991b8b685879, 0x0005ba3b38e17597, 0x000d69ea7d9931a2, 0x000c7632a26bcdb0, 0x000ba170f3c8441d, 0x0008adf11a365c62, 0x0000000000034e74}, {0x000d3f6d2f87f536, 0x0009d523ca2d7db2, 0x000b5fe1b40ff204, 0x000c7771d07308bd, 0x0007c291c2ef71af, 0x000b40d1773588d7, 0x00015629baa3b0aa, 0x00000000000eea9f}},
        {{0x000a95a8a249bd36, 0x000f90ae9143a26d, 0x000709bd167b8510, 0x000f7a7f3b4b882d, 0x000de22d9b9923df, 0x0004f02b1b178e73, 0x000c2fa83861afaa, 0x00000000000b9064}, {0x000de7d4573c6d34, 0x00022142eb294574, 0x0006f55c30205aad, 0x000717fe649a8b70, 0x000cad53c9bbd589, 0x000ecd8f7a925c47, 0x000142c339b11a09, 0x000000000001bbf1}},
        {{0x000abd60f6eb49c8, 0x0008c406a7e201fa, 0x0007246dc14c8322, 0x0002eff020748efc, 0x000af39b58dbd440, 0x0008cfb047827442, 0x00078f77e3f2768d, 0x00000000000e45a9}, {0x000ab42dc779cb3d, 0x000229db0829881a, 0x0005eb7284cde06b, 0x000ce47b82775f69, 0x000f63910016e434, 0x00047792fe84995e, 0x000eb9a35e8b971e, 0x000000000007c6aa}},
    },
    {
        /* digit=58 [{1,2,3,..,}]*([2^232]*G)  */
        {{0x00097b7667d86ea7, 0x0001b1fa064cf475, 0x00026db64fb0c148, 0x00002a1fa9d94539, 0x000bdc6bd7eada81, 0x0002f6044786aeca, 0x000208caf91e3bca, 0x000000000008573f}, {0x00036746e95246de, 0x0006cd309fce8dbb, 0x0001300c9068d932, 0x0001ae0f3d530575, 0x00000d1fd61e5779, 0x0005ebfa626b053f, 0x00097991c962c004, 0x00000000000076ed}},
        {{0x000013c3d6e02921, 0x000c449f2499410e, 0x000e2ab53501cdc4, 0x0009103d5e91bc0f, 0x000bcb404f68897b, 0x000f7fc0263db1ef, 0x0000af70efd9d842, 0x00000000000c6a23}, {0x000a0390300406c6, 0x000308d5b199ee1a, 0x0009868ea4fe37e1, 0x0003d34504dd889f, 0x0001f823686fde58, 0x000fc73dc0375600, 0x000f42f19ab86f95, 0x0000000000093f12}},
        {{0x000b8a18e1e24b49, 0x000143f896ca9186, 0x000d7ce3f5b4c07a, 0x0001632600e4e2b2, 0x000f69e702d5d074, 0x000e0df2b87c7db5, 0x000fba1f5a3b8053, 0x00000000000f443d}, {0x0005af3bc7c98ae2, 0x000deeb90e22f972, 0x00070953899b58fe, 0x0008299a5aa335b5, 0x000d4197b32afb1e, 0x00055918ed78504c, 0x000c720e7a79cc67, 0x00000000000883b1}},
        {{0x0004af5925d66db3, 0x0008dbf66baa58e7, 0x0001c386a0ca25fc, 0x00032abeaaa466ce, 0x0007e5f2733b80bc, 0x000531afa605b789, 0x000369e9e7e3a1a2, 0x000000000003deb8}, {0x000174c1d570e84f, 0x0008d36212ea2dd8, 0x0008479c4475fe18, 0x0004b31444e9ea02, 0x000169530c1befa5, 0x00079bdd19c2229a, 0x000cc368feb9854b, 0x000000000008b984}},
        {{0x000682b315cae64a, 0x000981df8d98c41e, 0x0009fabd7bca288f, 0x00064f91703e1431, 0x000e9de0ab4ca54d, 0x000dc4e0fa12198b, 0x00091f160e06241d, 0x0000000000066958}, {0x000b5c1e9cafc463, 0x0006f808565e66f7, 0x000a9467f76914c8, 0x000cd934bac17690, 0x000ca5be965f6682, 0x000a38e9062e7ac2, 0x00016ed33f6f8ad7, 0x000000000005b8b5}},
        {{0x000509d8741fea0a, 0x000c1f041e421362, 0x000c0d8899228fbc, 0x00093751128ed62b, 0x000acec6eae64fea, 0x000a65dae041a1c1, 0x0008e81f32359415, 0x000000000000154e}, {0x000a9eb331e84371, 0x000ec309e9f286a4, 0x000765936bc21528, 0x0004dd5692420c27, 0x000ec4fa1a6a7bb5, 0x000779e77193a41a, 0x000b8c35f5f3b43d, 0x00000000000046eb}},
        {{0x000b5769d7b9ec0d, 0x000aa3a5a366dc99, 0x0005d073ce2a60d9, 0x000e4aafe5355bee, 0x0003ced676663e16, 0x0003440036d8bac0, 0x00020c403eb33ed9, 0x00000000000333ab}, {0x00094bfc36e2db30, 0x000739fce19869f6, 0x000435b17be8c513, 0x0009611921a58e5d, 0x000620d5c61a8e68, 0x000c81b4e8f5f115, 0x000779b17f612fad, 0x00000000000e562f}},
        {{0x000d75f385d1b0f1, 0x000f0d6a4d25bfe8, 0x0007b705bfa0d54e, 0x00059731cdedfc0f, 0x000603d6502f9420, 0x0005ce8c80c4e385, 0x0000b7981a4fc5e1, 0x000000000007aca3}, {0x000ea22cf2bcfc17, 0x0009ef2037ef684f, 0x0007e5010cdb37dd, 0x000a23c71f3e4e4b, 0x0009f47b504b9c98, 0x0003233aaa73c8b8, 0x000f68b9e33f5402, 0x00000000000f92c9}},
    },
    {
        /* digit=59 [{1,2,3,..,}]*([2^236]*G)  */
        {{0x000a9e3a73909533, 0x00090ba03ba3b07a, 0x00090d7a3c9c5a5a, 0x0008cfe4f0f60b35, 0x000e6fcccd96f96b, 0x0009dd17ab908d77, 0x0000487208ef7de7, 0x000000000003ec3d}, {0x0006af6a704d4f0e, 0x000d09c5ad2d9a11, 0x000e77d5943d9764, 0x0009449470eb938b, 0x000bee7d772fac99, 0x000b7ad09faaf27f, 0x0000a9fbe402abd0, 0x0000000000057db0}},
        {{0x000ca62ea2a4a457, 0x0001b10c082a59d1, 0x000bdd4313beafb9, 0x000935b4cb291a7f, 0x000313e9ce08785d, 0x000f6f1c4fc2ae15, 0x00024c3146fabf4d, 0x00000000000c87dd}, {0x000f74ecc24bd4cf, 0x0000385fdd8765b9, 0x000dc418405d512e, 0x00005013e7e0297e, 0x000fb92df904c81d, 0x0005ddccbb56ddd1, 0x000f1d4612df9f29, 0x000000000000e27c}},
        {{0x0002069dae7548ac, 0x0002986d9b05f69f, 0x000025ad33463063, 0x0008e7c27d9d64d3, 0x000aba04e6ad9b6e, 0x0008abbbe79bd66b, 0x000d0477e4d0b082, 0x00000000000cc540}, {0x000bdbdb8d90e2e4, 0x00067f5d8a46ef90, 0x00078bfd47c637af, 0x000b852563fe6b52, 0x000997cdd04fb93b, 0x0007de47d06bb3ae, 0x00061c07f011d48b, 0x00000000000de78f}},
        {{0x000eb7904785958b, 0x0007d7830460f8a0, 0x000cf49e72cbbaa0, 0x000e2aa307d9c790, 0x000b5aad8b6c73be, 0x000848db51b02af1, 0x0004765e31882703, 0x00000000000f230f}, {0x000b79128735694b, 0x000bd4ea6535cc84, 0x00008e33135971e8, 0x0007b332fd33cac2, 0x0001c566914f4c19, 0x000952d3b24c7b0b, 0x0005f2956ce3c371, 0x00000000000fabae}},
        {{0x0004121555388dc8, 0x0008b8a95e5c1a8a, 0x000cd5dbd7b0133e, 0x000df6acda40bc0b, 0x00058234471d1859, 0x0003c0100f9097aa, 0x00067caddc0c9b78, 0x00000000000feb70}, {0x000ce59ade26052d, 0x000a6e3805fd0a27, 0x000626ac8acaae6d, 0x00099921943a0a1d, 0x00091dfe627ea459, 0x00006515fb47f061, 0x00073e313d4f09da, 0x000000000001f54a}},
        {{0x000f6dd7137bd615, 0x0007e0db87bc3c53, 0x0002eda89127238c, 0x000e9c4a3fd2a60d, 0x0001c9f017e5ea81, 0x000c75768190c9c7, 0x000de621864180bc, 0x0000000000043dbc}, {0x000fb5dd8276da18, 0x000de5b090c70dac, 0x000cd9057894e345, 0x000d268a918bf24b, 0x0008ae204f49fef3, 0x000d7c356e10c52c, 0x000a17f4be898d86, 0x000000000002fdb5}},
        {{0x0009bfc4aad5cc44, 0x000a20079c69c96d, 0x0008a5713957941d, 0x00086660139685a2, 0x0004946fbeddb8d1, 0x0001aace408590ca, 0x0001b67fecd9b964, 0x000000000002936a}, {0x000267114a49f247, 0x00065925b6235aee, 0x00055e3538cd2015, 0x000663d8c6fb34ac, 0x000c2fc1c10d971e, 0x00042b822543146d, 0x00024d6e4053c706, 0x00000000000492e5}},
        {{0x000d356fb82e9b9f, 0x000beca9872e0a14, 0x000f54683a031c30, 0x0007cb84e05b2811, 0x0006fa234d4596b1, 0x00035a7d89798714, 0x000384ba78949ebb, 0x000000000006fc59}, {0x0001361f78e02fa8, 0x00081a303e549f81, 0x00064be08532a2fa, 0x0002de8ee7220467, 0x000563e27035e57e, 0x000d2fe6fa05c106, 0x000aadaa38e86602, 0x00000000000ee2f6}},
    },
    {
        /* digit=60 [{1,2,3,..,}]*([2^240]*G)  */
        {{0x00044971cf281b0a, 0x00052c0426b768f1, 0x000ef3f4445c186e, 0x00012e3172c0d3e8, 0x0006ee75473731d3, 0x000a7ee615f49fde, 0x0005fb895530f06d, 0x00000000000e8b3a}, {0x0006345afdc270e9, 0x00019fd14973443f, 0x000f6896912e434e, 0x000ae07653908d03, 0x0006ba02a278e2ba, 0x00021b8f8c3d0143, 0x000297a0d0222e7b, 0x00000000000d7ec1}},
        {{0x000653659b1a252c, 0x000514f120aa7478, 0x000c72dfe03d7757, 0x000ac5ecfe5f7a92, 0x0009bbf3cec6c96b, 0x000361cd5d4e73d5, 0x00044ced8d233560, 0x00000000000562f4}, {0x00045d3e2b7ac684, 0x00022bd37d3cf9b9, 0x000cb601f2d0a968, 0x000535a3d2f41ee1, 0x000ee8b1743e7e35, 0x0005a27650353b52, 0x0008b831d89dfd7b, 0x000000000008d9ea}},
        {{0x000718fb55d90569, 0x000b306a67bd2493, 0x0001471031374c3f, 0x0005d5197bc62d32, 0x000924c51874ee0b, 0x000a1a0d552b1703, 0x0000acfed1f42382, 0x00000000000db627}, {0x0006189cf7edbc97, 0x0006c36be4a9b658, 0x000680236e8f5c91, 0x00036d3b8f8074cb, 0x0009718545c6c174, 0x000757d213bb4d39, 0x0003668e1ea3555c, 0x000000000008c474}},
        {{0x00063be615177c6f, 0x0002773457010af5, 0x0001ce08b2f26f1c, 0x0000e8c9c25fe5be, 0x000182dd0485705b, 0x000ac280540f36ea, 0x000b923d55bc8527, 0x00000000000ad921}, {0x000b6da293461f09, 0x0009551586cd4c76, 0x00086171a05efa67, 0x000605e84f0abcb8, 0x0003772dd0dabb4e, 0x0004e1d41354ef8e, 0x0004917f1a8f795e, 0x00000000000de5d8}},
        {{0x000beb4ebddc46f4, 0x00073ec72c64fb0c, 0x0005bac2d9d9096a, 0x00022001819efb1b, 0x00068cdde8703c5e, 0x000a87aeedf5ab6d, 0x000f1975a44e9d92, 0x00000000000bcf77}, {0x0009407ed3c226cf, 0x0005e8191efbc92d, 0x00064a74c9c1339d, 0x000e58265cf242d2, 0x000180b1d17be62b, 0x000de59a9ae99a3b, 0x000ce248cbb44692, 0x000000000002dcb3}},
        {{0x000a48783de6cfb4, 0x0006bf3899558552, 0x0009d51bfff43e77, 0x0004fd32df8d1a75, 0x000376d3fbbf0b1c, 0x000fd52bcf16bcc2, 0x0001f0d5888916f4, 0x00000000000d5cde}, {0x000f03d1ac917a2c, 0x000ae764ffffd280, 0x000af8be538ef59b, 0x0004762ccd57b860, 0x00032935106234f6, 0x000c642f32233a4c, 0x000f34df076095d9, 0x0000000000059f0d}},
        {{0x00010c66eff8425e, 0x000379580cdfaafe, 0x000d1f7ccb185b5d, 0x0005f77c327f3e8d, 0x000c35353c5f5d3d, 0x000258eb105d5339, 0x000f79c56fb5fe5c, 0x00000000000edce1}, {0x0005bd6f7b6e122d, 0x0007cab7aa141541, 0x0008987b379beb7f, 0x0001491458d9e533, 0x000caa7f0f31e124, 0x000fda7abdd2448c, 0x000a4dec58d3c7f0, 0x00000000000c91bb}},
        {{0x0002f037fabc6138, 0x000b73bd258d77ca, 0x0006aa4d0ec1d1f3, 0x0002512e3f966a14, 0x0007709d0c2d5b43, 0x000658259338bfca, 0x000023d142cc1049, 0x00000000000636b8}, {0x0007458ca547abc1, 0x000cda9ef9400a80, 0x000ad926836a9402, 0x00063c55cb644887, 0x00011cea475bfd2f, 0x00067a25fbae949b, 0x000a6aa45446031e, 0x00000000000dc6a7}},
    },
    {
        /* digit=61 [{1,2,3,..,}]*([2^244]*G)  */
        {{0x00085fa16820f665, 0x0009fd699ea2d24e, 0x0004f1772a862ed3, 0x0004bad66a8b35ba, 0x00024233fccb4660, 0x000d3cf0c0c779b6, 0x0007af578458acbf, 0x0000000000096bf5}, {0x0008a325d9d68d07, 0x00045a9724244e54, 0x0005b4b1e20150f9, 0x000a5b8c6be8c159, 0x000c774d62c40980, 0x000bde24b6230e3e, 0x000204da1467d84f, 0x00000000000cc862}},
        {{0x000b4d1a75edabf3, 0x0007567c51633fd8, 0x00020dc66cdc521c, 0x000c8dc9ee450d03, 0x0008b41a3e2f77fc, 0x000bf06898dd2b31, 0x000464df6a935e93, 0x00000000000a92e5}, {0x0001e3ee6beb3c9c, 0x000e449afcd9ef46, 0x00031a4b44405106, 0x0008ad2c7ea7810a, 0x000e550822b2cdbd, 0x000606adcfe61571, 0x000110744a4f9386, 0x00000000000d9d4e}},
        {{0x00006ff4ac15d783, 0x0007ef1084276ccf, 0x0009d3b1212d957e, 0x000dcf5bfb4283a4, 0x000db74017eb3752, 0x00078fcf8f6b2214, 0x00039afc1cdf7245, 0x0000000000012265}, {0x000c5dc1b7858cd2, 0x000cdcc0796680d4, 0x000e05b222bc5975, 0x0003a9a504cf7d65, 0x0003c93ed5932027, 0x000303f1b0c7b7e5, 0x0006a4aaf9c36866, 0x00000000000cb013}},
        {{0x000bfa5cdf24bf96, 0x000411fef389c07d, 0x00022753fd218088, 0x0000a1437f04a344, 0x0009d0169369bd77, 0x000377cc3c7438e2, 0x000a4f6b265742e2, 0x00000000000c369f}, {0x000bb384dc3d9a84, 0x00060dfdbcf462e9, 0x000d3f52e65bb342, 0x000a0b82a9c483da, 0x00042de432285574, 0x000d1fabe0563fe9, 0x00096658ca0e8aea, 0x0000000000066023}},
        {{0x000afbbacd3ede36, 0x00007746325d090f, 0x00094f8b4a38ccef, 0x0001c2866c3931a7, 0x000a783a73e7d9f2, 0x000d82d13c12880e, 0x00010e382e1ce28b, 0x00000000000ac023}, {0x000fda6b09a40144, 0x000b69802d06233d, 0x000de8140422422b, 0x000367efd4cf75a0, 0x0000e8f2f6ed38b4, 0x000e72ff4765cdee, 0x00070ae0b4d72b35, 0x00000000000947d4}},
        {{0x000a35bb9d72eb2a, 0x000c5383bda07268, 0x00038c9d09f99c2f, 0x000717d369f39c03, 0x00011a5a39006f3c, 0x000ec6c2b1bb593d, 0x000202d0f07ecc2a, 0x000000000004240b}, {0x00083c5449860db8, 0x0001935342f6c7b8, 0x0009a1541ab519cd, 0x000eb09ccb6a888b, 0x000785aa42c5fcd6, 0x0004e5895abb7a6f, 0x000582952f8824aa, 0x000000000005c406}},
        {{0x0001d7b0f8433a5d, 0x0004359d6e052cda, 0x000fea341e325461, 0x000d07d7907cc890, 0x000dc4ce5d800459, 0x00004f40267d720d, 0x0002a83262028eed, 0x00000000000d7881}, {0x00055f59d844fe29, 0x000fcf735fd6cf7f, 0x0001c0c0179cc733, 0x00006e8e19a43f29, 0x000f19592b76328c, 0x000b836f7b97ef65, 0x000a9981325f3db8, 0x00000000000d6e6c}},
        {{0x000a67318a4b19fa, 0x000a63667a71faf0, 0x0007be6235b29837, 0x000535efc62f7919, 0x000b389faf7fe084, 0x00071b7f65bc1652, 0x00070340cf51683a, 0x00000000000d4f39}, {0x0002b576e30c499d, 0x00099823e7549478, 0x00032769bfa306a4, 0x000ee027225b31ad, 0x0006fc282f165639, 0x0009f61ae7533bc8, 0x000803710009d2c6, 0x00000000000bf65e}},
    },
    {
        /* digit=62 [{1,2,3,..,}]*([2^248]*G)  */
        {{0x0006c5f4c042c4cd, 0x0001ceb29e44bf59, 0x0004c11cc5ce653f, 0x00004943d2bf689a, 0x000a47428dd2d09c, 0x000aafac83ab7799, 0x0006e0dc558d6be9, 0x0000000000087f9f}, {0x00056bc34f65dad0, 0x000c793842bcd3a9, 0x000241a2ffbfced6, 0x00052687e6d47b5b, 0x000a4c37eeee1645, 0x000c412ceab304b7, 0x0002c8dbb3d4e13f, 0x00000000000a726a}},
        {{0x00059e97675084b3, 0x0008d79d88dffddc, 0x00002dc16c994a53, 0x000000fce7d606f0, 0x000e2fa27fd3b528, 0x0009436afc773557, 0x0004c755b53dd3e1, 0x00000000000e10b6}, {0x00077a15a41de95f, 0x000bfe5832664b2b, 0x000d15e689d49c17, 0x000bf537af3e3dd9, 0x000cf47d298c7b93, 0x00012136994fafa2, 0x000ff694c2ff9a21, 0x0000000000033468}},
        {{0x0001b740495480b0, 0x0003f91e8c991baf, 0x000b4c043871430f, 0x0003f6dd095f5b94, 0x000cddf3cad27c5e, 0x00057aacfed7522f, 0x000f5180bb87056c, 0x000000000000cc5f}, {0x0006cac5a2f35aa0, 0x000e0964def7e61e, 0x000e006a84529a7b, 0x000192584de66a22, 0x000e075f07c5cc75, 0x000eade939acaf7f, 0x00058f6505c2f81e, 0x000000000000e3cf}},
        {{0x0001fb7f00850a1a, 0x0001dbe45d81a1aa, 0x0000ee7897985bd0, 0x000a516b078ea895, 0x0001b446476463c5, 0x0001e52329f3efe4, 0x00022084580e2410, 0x00000000000c8ae8}, {0x000539dde4d08a27, 0x00077bff4077d088, 0x000c1e715cd7b849, 0x000eef207e1c03eb, 0x00096e654d584df4, 0x0006f15964ab3d03, 0x000cd5b20056cd08, 0x000000000008acd7}},
        {{0x0008f4a5a8f26e12, 0x000538caa623c979, 0x000e7d0ec3691999, 0x000f856ee285bfe4, 0x0006d0674ecd089a, 0x0006f661277b461f, 0x0009f8baa9d9b38d, 0x0000000000059066}, {0x000460fe25d6fd2a, 0x000d2b164340c38e, 0x0003981c9e27c186, 0x0003b9176e4346f2, 0x00017caad5fc73c0, 0x00087803d6a678eb, 0x000a94103ff0790a, 0x00000000000f7430}},
        {{0x000dd3174fcc39a0, 0x00094517075af0fd, 0x000fd65c623f7ba2, 0x000e9493b86f6398, 0x00017f296bf4320c, 0x0001082765115657, 0x0009000919607a96, 0x000000000002f035}, {0x0004b29394d28cac, 0x0008f5d13de7c5ae, 0x000ea9b2719ceec5, 0x0008089c0f58697a, 0x00030ca7ca20f1b2, 0x0001e1be52adcd1a, 0x000ba096c07f42c2, 0x00000000000baa0a}},
        {{0x000dc0265a01c54c, 0x000233027c169336, 0x0003342d9c6b202a, 0x0006cd31b8b0179e, 0x000a6dd5a4a7e6eb, 0x000c82f110d2d27e, 0x00002241682c0007, 0x0000000000081075}, {0x000f1169b3430f67, 0x00019f3903f514b2, 0x00091dfa21d2d176, 0x0005eed470bd3b32, 0x0007e85a62931b62, 0x0005ad640a46398a, 0x0003a58ff6ef8c80, 0x0000000000095bfa}},
        {{0x0006e424a5742fd1, 0x0003a8cd6c9f7239, 0x000180484ef81e04, 0x000b0f589b8ad2c9, 0x00070d9c999d9c0d, 0x000f84ab4692a8db, 0x00006ca407fd03c9, 0x00000000000cc9a9}, {0x000b5d757bfbcaf2, 0x000014813a7654ce, 0x0008615f305ee56c, 0x000f2934fa0a23f1, 0x0007cb1f8d7aca6b, 0x000d10c531dfa3af, 0x000c1328036f5498, 0x000000000000e012}},
    },
    {
        /* digit=63 [{1,2,3,..,}]*([2^252]*G)  */
        {{0x00034e0d2aae29e9, 0x00091a53f1a10241, 0x0004dd23936a5886, 0x000ed8532976d137, 0x00065ee692d130ef, 0x0009e7cc1cf5ada1, 0x0004723ceda69d25, 0x000000000009baab}, {0x00095b8627836d36, 0x000237ee5baef4bf, 0x000e9b1caaa769fb, 0x0008ddacfdaff633, 0x0009cec076939e5c, 0x000fb509d575c8db, 0x0006979afc8ae0fd, 0x0000000000085651}},
        {{0x0000da11d13ab853, 0x00042a7342f9446b, 0x00039f5ef3dbc527, 0x0008bf07471c3856, 0x000130827f3450e4, 0x000779c23729e5ab, 0x000d5817199191ea, 0x000000000009fa9f}, {0x00077a9b9dc5fca3, 0x000827d5799369d5, 0x0003bee46a6f5cc0, 0x0007d51e85417260, 0x000dec720355253f, 0x000a16268107a793, 0x00031b1c14d0566a, 0x000000000008bbb2}},
        {{0x00016132445f3e21, 0x0009ef5689c5dce5, 0x0006397102e2e73d, 0x00079d012eaa5340, 0x0006d9271e941af8, 0x000e63c34dba775e, 0x00024b60f8507310, 0x00000000000a686a}, {0x0006624c82f5c532, 0x000dded5ca8f992c, 0x000b2edebcb9407c, 0x00039426d90a7d51, 0x000bccd37e76e2ee, 0x000412e0fe5b3e4c, 0x0000489013bd08f1, 0x00000000000ce999}},
        {{0x00009cf56d1d974c, 0x0005561d0e01b86e, 0x00046be97f98120d, 0x000e224b362902dd, 0x00029e0a5d16d4a4, 0x000e580e945923c1, 0x0009544fc2bdd495, 0x00000000000a8c3d}, {0x0004daa7a4671003, 0x000a1aeff11352c5, 0x0005a1c5b6c0120c, 0x0007d3eab8f9f31a, 0x00066d566c158090, 0x0002e6fa7af2c121, 0x00003a0082daba95, 0x000000000002df9e}},
        {{0x000fc79dfc540dc4, 0x0004091b379c535c, 0x00090bc691de574f, 0x0004208176b3b828, 0x0008aaba5cd3d0f4, 0x000794f9fd18c211, 0x0008944aed5c9770, 0x00000000000f4c33}, {0x000946afeb1a61f9, 0x000b9b8e67989163, 0x0000523ed92edf12, 0x000ef39ab5573985, 0x0004b71031051b7e, 0x000ac89e4175e393, 0x000db8943abf4a82, 0x0000000000057ffc}},
        {{0x00066f02c7c0f325, 0x000f899c2b4cc9e8, 0x000f2b3e2c2a45ff, 0x000b66b5a352b7b1, 0x00043ec757ed067b, 0x000d36adc3c7d6fc, 0x0003a42f69291cb9, 0x000000000007c914}, {0x0004d853fbc3268f, 0x00006768c3c3f0a6, 0x000dd7feada4c10c, 0x0004b025299ea1bb, 0x00091a2d4005d86e, 0x00017fa60be46b7e, 0x000fbeaf865a1693, 0x00000000000cdce2}},
        {{0x000297083b5951f8, 0x0002b66fcba529da, 0x000a6f9544c17490, 0x000b97241b4305fd, 0x000e25b6af89a371, 0x0005f9e8cb2d858c, 0x000ef8bab07d5327, 0x00000000000d525b}, {0x00073848e93e22c2, 0x00084ebf79e3bb83, 0x0007a079aa9da5d4, 0x0008b66cba1a65a3, 0x00038b1eb8b67dbd, 0x000a6436a88ea401, 0x000e33210ac1b38f, 0x000000000003df40}},
        {{0x0000b48bcae58bd8, 0x000c45f79f0de54f, 0x000db929ac4cc6a4, 0x000a68e11571c5c9, 0x000e2474faf7d631, 0x0002a924a6d072b4, 0x000d7a5e043a6d86, 0x00000000000b0fc8}, {0x000fb0fbc0056e2d, 0x0008bf54bb7f2f12, 0x00058e2455c41895, 0x0009f4c5909ec050, 0x0005abd164608145, 0x000b1c69ed28cda2, 0x0006f4bb676d018d, 0x00000000000ac503}},
    },
    {
        /* digit=64 [{1,2,3,..,}]*([2^256]*G)  */
        {{0x0005d534448865a4, 0x00079dbe04c30004, 0x000de820bf917110, 0x000b54670911457d, 0x000b3f7a1757dbf8, 0x0007bd62c9e683b5, 0x00097e7b89a08a9c, 0x000000000000b3fc}, {0x000ba720a0047774, 0x000980ac9abfb9ba, 0x00007bd7b6be79d8, 0x000c335d9deed984, 0x000f72603e080aa6, 0x000783bb2e270580, 0x0009ae70857a946c, 0x00000000000caa92}},
        {{0x00008763d38fe0f6, 0x00001f071c986bc1, 0x000413c627ede21a, 0x000b12a61483bc2d, 0x00001caf6dd6845e, 0x000c21b9217471d8, 0x00036d7b603616dd, 0x00000000000b9925}, {0x000a996617818c3e, 0x0007f433f065211f, 0x000b56653ac9464b, 0x0007386a9adf49b1, 0x00092cb2944fdf61, 0x000d9c0764bfeb39, 0x000576bf288427b3, 0x00000000000965b4}},
        {{0x000b333ccbe322d0, 0x000fbce23a199d59, 0x000074dc302e3830, 0x00000121d15a0063, 0x0005f4355a1fc720, 0x0007f780715fde73, 0x0006913849761f60, 0x0000000000018204}, {0x00074bfa03274297, 0x0002575756e4ea63, 0x0009809c9584a1de, 0x00032763d4ce3dc3, 0x000ef1d66e006312, 0x0008829e6a769d2f, 0x0006ef8624ddbac2, 0x00000000000d2df0}},
        {{0x000f606b2db32f32, 0x000b59ec11596f9e, 0x0004a5f37de5fafc, 0x00094e131fca111c, 0x000896076f45acf7, 0x000732588438b917, 0x0000daad71035b51, 0x00000000000a5d91}, {0x000d85cad07f3bba, 0x000a7efaad0e82d9, 0x000f829c6ffe7ff7, 0x0002426474ce3273, 0x000a2aa270da9940, 0x0008fa6ebd53b687, 0x00041196c8bb78d7, 0x000000000006e699}},
        {{0x00044c83de3d28f8, 0x0008b2dfc39e611a, 0x00099024ec940ff4, 0x0002123313aa4788, 0x000b9c6acbd11cf1, 0x000d54177785b025, 0x0001eab4aeb5b8e4, 0x00000000000d943c}, {0x00041945f056ffdc, 0x000097892db846ab, 0x0000f6f8e04921ff, 0x00013c61cbd3232b, 0x000700b6c34ebdee, 0x0003de32b873e0a7, 0x000302b279505ae2, 0x00000000000c5a03}},
        {{0x000395d621a49e0a, 0x000c7430b669fc05, 0x00093c4a448f25c5, 0x000730c4c33493e3, 0x000fca00ed3901ad, 0x0004a55168c882c1, 0x00027dab2e9c89d2, 0x00000000000f5d03}, {0x00020231a1107ea0, 0x000f946488bd7b79, 0x000d5a9d2b183f55, 0x0000e90aecfac7d9, 0x000fac6a9e8cf9f6, 0x0008bcda84afd1f5, 0x00099857a20bf8c9, 0x000000000008b46b}},
        {{0x0005f4b3119ad50d, 0x0006c1f2f6c1efc1, 0x000be8ee71be11e0, 0x000e4a6569c090bd, 0x000adb3986d020dc, 0x000f553eae4f7401, 0x000bff1389799485, 0x0000000000009026}, {0x0003665df6023a4c, 0x000b388067aa6ec6, 0x00069e9d017576cb, 0x0005019459bf26a1, 0x00058b6e76f68416, 0x0000d2434ec125bd, 0x0000b0d636f9321f, 0x000000000008e05e}},
        {{0x00021074c1c07346, 0x00068fe1f11ac76d, 0x000a3e93fcce998e, 0x000015a3c5babf98, 0x000d5929ea0a99d0, 0x000fb7b7e7e795b4, 0x0002e8c9cf68331d, 0x00000000000a6499}, {0x000716aaedb25dc3, 0x0000d0b7e0dd7aad, 0x00064fd90a09f648, 0x0003cf034f02b979, 0x000a5662f0c86402, 0x000058ccaf7dd410, 0x0001eda3bb6088a1, 0x000000000004e780}},
    },
    {
        /* digit=65 [{1,2,3,..,}]*([2^260]*G)  */
        {{0x0006b5199cfd1ff1, 0x0009d140a0f9f409, 0x0004971fd020b4cb, 0x0007c2304a1f742e, 0x0001195ba34e78ff, 0x0007f78b4c0568f5, 0x000a53e97183603b, 0x00000000000f9efa}, {0x0009e6e1cb2c1b4d, 0x000bae924d2c4efd, 0x0004b415d4c5ceca, 0x000e73f6ee37e106, 0x000a5e5a1dde4b12, 0x000cd64161836fdf, 0x0004d87f1b92259d, 0x0000000000067754}},
        {{0x000befa999003a02, 0x0000e279fd119411, 0x000606c204c4310e, 0x0007da4da44105e5, 0x000a28223fe1d8f5, 0x0006f2d2eb1814b7, 0x000e06cf2fd241e0, 0x0000000000001dfd}, {0x0001563b8cdc2810, 0x000a4876a31af711, 0x000bd72037bc4e78, 0x000d608493a6a0aa, 0x0008a88c03e75117, 0x000916897dcec808, 0x000c57eae1d352ea, 0x000000000006e8b2}},
        {{0x0001a9c4fdcf93d1, 0x000650254486c9e5, 0x0006796f28c8ff02, 0x00058ba000f54926, 0x000934009fb6fb58, 0x0002bde301315bdd, 0x000358b18a8e0ac4, 0x00000000000f1070}, {0x000b8de4d767059a, 0x0004ebeb1db7458b, 0x000305d015a22913, 0x00014d4e4122b217, 0x0002aabccc7b1522, 0x000d07571d3e673f, 0x000f1794c50f64ae, 0x00000000000e66b4}},
        {{0x0004bef0d3847d2e, 0x000aa09f8bb05816, 0x000388d5b381065f, 0x000c7b6076e13ec8, 0x00035d5ac07f26eb, 0x000ab69e6bda0b55, 0x0001fabcb8132248, 0x000000000001c0f2}, {0x0006ceb771ee0889, 0x0007b7a466528564, 0x000e55b024527048, 0x00011864c1d7cb8d, 0x000f2d08130185dd, 0x0005ea0f0096f849, 0x0009b2b4f503dd8a, 0x00000000000d1bf6}},
        {{0x000f0c6218b7c4e9, 0x0008540336b12c41, 0x000f74c446fc6c56, 0x000048d9c841f0d0, 0x00051d617f50c337, 0x00017d3794ce6d02, 0x00024300fef21981, 0x00000000000f95be}, {0x000f7d33fe9f8bcc, 0x000ec98de119d3a3, 0x000e778f8a8c16b8, 0x0005b720f9678bed, 0x000d334ace309412, 0x000e86e04fc5b57c, 0x00003909486527b7, 0x000000000006e552}},
        {{0x0001aa0f2e0127d7, 0x0007b4ab7a22bd4a, 0x00047f417a4172fd, 0x0009f95078de336c, 0x0006f786924520f5, 0x000f1d96ffd28fa3, 0x000f0f1de42581cb, 0x00000000000366e1}, {0x000e183b180aaf06, 0x000febc65fa9d0a4, 0x000739e25cf57814, 0x0004f3a4a0822a93, 0x000bfd05a0aa0638, 0x0005ee3ce1c81332, 0x000d00bfb12361d9, 0x0000000000042016}},
        {{0x000f0b32c63695dc, 0x000143b75c45b10b, 0x00037d16d187f9f5, 0x000227df8e0a114c, 0x000e73ddba245450, 0x000f246455e69f1a, 0x0007412007b80e57, 0x000000000003d159}, {0x0007528fbc79a394, 0x000ffcbe54d2888b, 0x0004e47907e4e2fb, 0x00070594cc39f745, 0x00032b1b8da9e19f, 0x000f68d2680f00ab, 0x000829b765aaf973, 0x00000000000e993f}},
        {{0x0000b5faa7527283, 0x000530a159f61ee6, 0x0008e1f9dcf0213d, 0x000b8ee1661a2405, 0x000ec95c41b36d1a, 0x00051eb56cae7f40, 0x000250bc3d20407c, 0x00000000000e8754}, {0x000ae69a0a837ff0, 0x0000281561056151, 0x00032df2869a92d9, 0x000c1bf6af2a00cc, 0x00042d0a56c0abdb, 0x000c8959ee9a9425, 0x0002f34774a7b77b, 0x0000000000019cef}},
    },
    {
        /* digit=66 [{1,2,3,..,}]*([2^264]*G)  */
        {{0x00035e2ab65ec55c, 0x000b3114a25c78e3, 0x00036712a123ad50, 0x00068411e7bd7df9, 0x000ad6da54dbc49b, 0x0000da3215b0359f, 0x00087ea6e6e5f93f, 0x00000000000d640a}, {0x0003244eb630ddc0, 0x000f7c1a4f6cdf83, 0x0008137a92bebef0, 0x000eb8e3c0a631d4, 0x000db756445ff44c, 0x000c8880e0205b11, 0x0000d304844e845b, 0x00000000000b85e0}},
        {{0x0005c0fc3837219a, 0x000cee76894c3764, 0x000faaa07cdf021d, 0x000fcfbe55cd1e6b, 0x00023a7ad5b9566d, 0x00087a4ef5421a9f, 0x000423a005838a46, 0x0000000000023a2c}, {0x000326b6922fa665, 0x000ed4b780011f85, 0x00024bae2459585b, 0x00069e937ced8f3d, 0x000d8c1e1eaa2b5f, 0x00089cc4d306b621, 0x0007748acbbe71d3, 0x00000000000f4ab7}},
        {{0x000e8b5d217d418f, 0x0002ee557e7c707f, 0x000dffc8ed3b58d7, 0x000efa4b42b3ab4b, 0x000d8e53b32db9b0, 0x000b1bfbd9d71dc4, 0x000d3fe3a93e4e11, 0x000000000009901b}, {0x0009bf805b79f71c, 0x000b35f3a5304e02, 0x0007d4e3dff75554, 0x0007a7bef469e919, 0x000278f160dcf415, 0x0009189a9f03282f, 0x000f2197e4b7f4c8, 0x00000000000a967e}},
        {{0x00083fb3cef3edcd, 0x000642df6f242046, 0x00054e87770387bc, 0x000232b372bcc88c, 0x00086e428cc59e80, 0x000a1b76326ba13b, 0x0005f32526ef1f13, 0x00000000000dc97c}, {0x0007320a0cda3969, 0x000344954867fb10, 0x0004f1baa6436a30, 0x000fa69be143027a, 0x0003cf54f28b7e39, 0x00097c2da1232946, 0x000099cf0991dc75, 0x0000000000052e07}},
        {{0x0004ba256e3a80b8, 0x0005fbdfea74874f, 0x00090a65af244c4b, 0x0005675aaba39901, 0x0006a12be348d5a6, 0x00018ac5d2250648, 0x0006f9766dd428e8, 0x00000000000fff9b}, {0x000ff89f1b07248b, 0x000174b3b10f3ab0, 0x000ac70bef37c1ca, 0x00011ed5c9e36bef, 0x000364cdfcdd2a61, 0x000462f54a99a302, 0x000b5fdbfbbe7a59, 0x00000000000cc266}},
        {{0x00081f8725c117c2, 0x00037dc9daeefc8d, 0x0009cda216a5b4ef, 0x0002271ca53d53df, 0x00054d1aa50b206d, 0x000001e99b054633, 0x000aa452bca91088, 0x0000000000017fb7}, {0x0007b7c6cf80a17f, 0x0007afff3af9472e, 0x0007c0765e5ecd82, 0x00004be24753989d, 0x0006e90950c6aac0, 0x0008efe5b4a5de2b, 0x0009f9b46af0ab73, 0x00000000000db445}},
        {{0x000c5196b2aa222f, 0x000d15bdb6d3f8f0, 0x000389884e011601, 0x00008a96ddd9e23e, 0x0008467e7577ab50, 0x0007555edac1e974, 0x000d57e74e35b601, 0x000000000009d04f}, {0x00080785cb147efa, 0x000908585c6fc59b, 0x00035ae9e9b63afd, 0x000d1e80b684dd21, 0x0009edbaddcc2739, 0x000c57660e1ba788, 0x00089a464f42059d, 0x00000000000a7cf2}},
        {{0x0002dd131270b84a, 0x000dd412f64a3e09, 0x000b9c94f3cfd9dd, 0x000cdc2c2e964e0f, 0x000d0011cb01fbe6, 0x000228f23e9a3b1f, 0x0009a16c30762dcf, 0x00000000000be919}, {0x000e7d10046b4ea1, 0x000b4732711dfdf5, 0x0008160cd8dd88e3, 0x0008ba0c6eceba7a, 0x0001f3c3d31d8ee0, 0x000716948b171153, 0x000add65060b633c, 0x000000000002ff2c}},
    },
    {
        /* digit=67 [{1,2,3,..,}]*([2^268]*G)  */
        {{0x00000725d23401df, 0x000096f178dbeb92, 0x000498bead595449, 0x000b4b459b46611d, 0x0007c72b4a58a6e8, 0x0005985adafce826, 0x00073321142175a4, 0x00000000000e649d}, {0x000052db6dbc2445, 0x0003a6bf42fe0182, 0x000384cfd9aea017, 0x000a58acd0291983, 0x0009215492e1b0c8, 0x000b5a5f0a73ff32, 0x00049b895c545eb1, 0x000000000006fcaf}},
        {{0x000a5a0bc1e856f0, 0x000f28baef5de481, 0x000a8b075c84a181, 0x00058d754c8267b1, 0x0005ccf703932685, 0x0008ee021f924f79, 0x000a4be3763f30f6, 0x000000000005c01b}, {0x0006ee7749647d88, 0x0008acc01a3e928c, 0x000c22c3ac36bfb8, 0x000512e6c45e3401, 0x00084ef433f61ab3, 0x00021f5afa978fe4, 0x000a4023e2ea018e, 0x0000000000011524}},
        {{0x000ecc04426e192d, 0x000c692a7fbcd69c, 0x000df4111d9bb7a6, 0x000db2d02a8feb62, 0x0002728cecbe8e45, 0x000837662176c0cb, 0x00082c33fcfbb0d1, 0x00000000000d6f28}, {0x000d967a59df021f, 0x000bc75a0f344b04, 0x0001f4de7fbff391, 0x00090cefa0453b03, 0x000ff54c96fbf4ea, 0x000f77afc5108858, 0x0002fc86b46b5731, 0x000000000006bf7e}},
        {{0x000215c3f055a1c5, 0x0002fe1d42d93e03, 0x000c50c3e0aa63b9, 0x00098e783686967e, 0x000474a30dbf5f66, 0x000d20230bb5e159, 0x000f8bef86bb5bcf, 0x00000000000403b8}, {0x0001c2443b169c63, 0x000fbf249aac89ec, 0x000e2f941f78369d, 0x000960cefca9a90a, 0x00088f449c4b3b80, 0x000cfe09ef28e338, 0x000d260cdfc61bf0, 0x000000000008b22c}},
        {{0x0001733d4c0dd30d, 0x000af7013d596371, 0x00017f590b1fdd3f, 0x000898e35915f523, 0x00039fd967d4bba7, 0x00073558f9fbe5f1, 0x000bea0aba8344d1, 0x00000000000ffeb6}, {0x0009792191976aee, 0x0004e928f68cbbf7, 0x0007210e163722d8, 0x000c4e06abb0ac9a, 0x000a6895762709b5, 0x0004045a401ef3f8, 0x000ac355dbe9f79f, 0x00000000000ef178}},
        {{0x00094c375a95d7b3, 0x000dd7f89c2c3b20, 0x000a45fd88318202, 0x000e48b12d9f811e, 0x000dc7a43dd0be70, 0x00018ca3a703ac1b, 0x000012a4309a2c21, 0x0000000000001943}, {0x0009ce1f49259e54, 0x000e6d9597d8737f, 0x0001541e69df2d37, 0x000760df1561aaa4, 0x000e2d91ee39fcf8, 0x000bbcb6de1e0306, 0x00074699979031e0, 0x00000000000cfcc8}},
        {{0x0005490cc3cc0b27, 0x0002a4073f24615c, 0x0003e0a87e36ac6d, 0x00006c18fcec6a28, 0x0005d75a73f10abb, 0x0008c94883a7d129, 0x00074165fd8700f1, 0x00000000000732f7}, {0x00049117079dd0b3, 0x00098d15d8c801e0, 0x0009200404155fcf, 0x00007c7120e12665, 0x0001f42cc9fd9816, 0x000a888cf0e486b1, 0x000249e606c16c01, 0x000000000001ea23}},
        {{0x00014f7d981a63eb, 0x000cb923e948b882, 0x0000fdde16bb34e4, 0x000011a6df27debf, 0x0001d57f4ca2345a, 0x0004196e9ba6784c, 0x00026df01b370311, 0x000000000001aab4}, {0x000a2240acace9dc, 0x000d82ffdc977a4c, 0x000a7e87839c540c, 0x000c01216f09c1f8, 0x000e5b9cc0b65ca1, 0x000150569612021d, 0x0003ea910e10e95e, 0x00000000000e2ab9}},
    },
    {
        /* digit=68 [{1,2,3,..,}]*([2^272]*G)  */
        {{0x0008093fcd51c0da, 0x000aaac58c9dfd06, 0x00005aab38065215, 0x0004afc2cc4252aa, 0x000397c2bdf932ef, 0x00049e19a08bd48b, 0x000496ac8a7803a8, 0x0000000000075c31}, {0x000fef0d3072e592, 0x00057733dc7dec06, 0x000d0f9063e4be72, 0x0005b0847be21a8a, 0x00055ebf426f6d9c, 0x0004e9e5bfab0fdc, 0x000089d60748caf9, 0x0000000000069366}},
        {{0x0000e5d6509e91ef, 0x00048b06c17a3a05, 0x0001e37973be9551, 0x000ae990038aeb31, 0x000b58b402ca2440, 0x00077732a83bf711, 0x000932b8763e00b5, 0x00000000000f651a}, {0x000cf91f09ef177f, 0x00070be02ab9cf6b, 0x0005ba59eb97ec90, 0x00072f64283f8100, 0x00072365da5e7ed2, 0x000dc6beb098cf05, 0x000d72d90e6f1805, 0x00000000000cf7b9}},
        {{0x000bed14c2e5c1d5, 0x0009f89b2edba20f, 0x000638f30da8440c, 0x000d9ce8943fce4f, 0x00045a2ff9961dde, 0x0006e87feaaf07ff, 0x00008c69e60f92c9, 0x00000000000b9ee8}, {0x000d99ec41a8cc2a, 0x0005279d6d8c67c9, 0x000b28385a21a71a, 0x0005267350b7fc9c, 0x000d8a2abab0c8a6, 0x0004faf7ce9fcb46, 0x000c8a57c3bfc62c, 0x00000000000a8207}},
        {{0x000f000e371891b6, 0x000b56f101762b79, 0x0004acb33b163eb0, 0x000e2aaac1dc8274, 0x0009f835a1b62c58, 0x0009915a451410e8, 0x00048d981333a762, 0x000000000000c141}, {0x0009810640a6c340, 0x000184d20b3d37fe, 0x000e4eb008d53e3a, 0x000b68c2a2645f81, 0x0007add082eaa664, 0x0005ff2067a6bc85, 0x000fcce7467dc63e, 0x0000000000004e2d}},
        {{0x0000cfcf5d6b5960, 0x000cc57fe04dac14, 0x000998e82d998b8f, 0x000bd40e0a341ea4, 0x000b9c24fdba5128, 0x000c1dea81adf3cb, 0x000ef2b1ce8520f4, 0x00000000000db22e}, {0x000911256ee6f617, 0x000b2e9f05f28713, 0x0000043d1376f349, 0x00052d3f63bfade4, 0x00006cbcc395a2ad, 0x000714bd81b43575, 0x000ca6fd1d7e0666, 0x00000000000a40ea}},
        {{0x0004ca398c06d768, 0x000c23d0efae2846, 0x000878540a1ede95, 0x0005a253f00eeef2, 0x000cd86161cea5cb, 0x0004a389ac2f9258, 0x000159d2b0865f5c, 0x000000000009b1f2}, {0x0008e174cdba2f0b, 0x000245a758a60599, 0x00087073a8d80dcb, 0x0005a4949d301c92, 0x00061c4bb89b8865, 0x00060259c129647e, 0x0007d106f0665ec3, 0x0000000000006619}},
        {{0x000cae6e407127c8, 0x000070262c90bff7, 0x000ca8e9829543e7, 0x000f8fb1577634a8, 0x0004597668ad6514, 0x000ff6ecf4678d63, 0x000dfc90213b637c, 0x000000000007518c}, {0x0005a2999ad1c0e5, 0x00071d006db8f897, 0x0004562511312669, 0x0003aeadf6356731, 0x000f1a366456acc7, 0x000e73aeaaeaa4fa, 0x000a480df51aea1c, 0x000000000009a8e4}},
        {{0x0001074c67a33fda, 0x00086a23545689bd, 0x0009b9de484075b5, 0x000247b132c568a3, 0x000dcc4760bbab0c, 0x000ae821b129a5c9, 0x000f8303eba46726, 0x000000000003df1c}, {0x000f0c5101b30f29, 0x0009f010813cfb68, 0x000cc25f999a807c, 0x00020b31fe4b6007, 0x000398dd32399073, 0x000abb34cda3bfeb, 0x0005f123f1ed1641, 0x00000000000946f8}},
    },
    {
        /* digit=69 [{1,2,3,..,}]*([2^276]*G)  */
        {{0x000261be9524972c, 0x0005728524830014, 0x000a9e4b0c851ae7, 0x000fa9900e4f78b6, 0x000b54a3f7a33a66, 0x00065a79afd8a65b, 0x00069a505d3f6319, 0x0000000000018914}, {0x000675265cae6514, 0x0008278367cbbd6a, 0x000c906352414281, 0x000b3f6f5af173f1, 0x000a5ca36597e41b, 0x00099f79557cb03c, 0x000ef127f86cb87b, 0x00000000000ad4dc}},
        {{0x00092527cce78878, 0x000b138a49bf4890, 0x00076c4f31c0490e, 0x0009eca01b2f7c2e, 0x00005be8359f5f70, 0x000646b3fc479a65, 0x00011b6b6a22bfd6, 0x00000000000cc5b7}, {0x000c9aa1253a31c2, 0x000d18e7dbc638f0, 0x00085601e946c9b1, 0x0007e04030271eb4, 0x0008e22fb3c72d3a, 0x000e0d46cb08b597, 0x000915860d62789b, 0x000000000001d49a}},
        {{0x000e171e53bdf97e, 0x0008556d65155b11, 0x00013206465d7fcc, 0x000049f10dfaeae2, 0x000ae30b70b8ef9e, 0x000c6a56219750b3, 0x00002611ed860015, 0x0000000000012097}, {0x000714d4a4467bbb, 0x000e279559d04c7c, 0x000363f176594d50, 0x0007323ae8fb3c53, 0x0009a4111f88d6fd, 0x00010a683834639c, 0x000e9afc69a029a3, 0x00000000000255f2}},
        {{0x0006e6e51e1dd6f5, 0x0009766797f569dc, 0x0007480d93f86f88, 0x000f4a531adb034d, 0x000e5d185cfc18ee, 0x00053c27bcf1cb5e, 0x000569f596a59bbd, 0x0000000000069002}, {0x000a4105949da385, 0x00053d9433e78e1e, 0x00022ac847a15a5d, 0x000d7ed65d68ece1, 0x0009fb2aded8233c, 0x0009b750e201bbe0, 0x000f25987f4975ac, 0x00000000000337f7}},
        {{0x0009f8de8126d12d, 0x000626fb6b7bb82e, 0x000f3dc00034e80a, 0x00089d8316ff5318, 0x00079894d65f98ad, 0x0002801704800ee1, 0x000537034ea3c448, 0x00000000000c30be}, {0x00066423e58a8102, 0x0009243c2d27d6b9, 0x000c30559bd3f060, 0x00040105eaef29b7, 0x00017b678fc76545, 0x0003012701d8f07a, 0x00036554bc6f737b, 0x00000000000e9473}},
        {{0x0005ee6b65e3f046, 0x0002a3561b3fe3d4, 0x0003adfe7b57a7b4, 0x000831347fb79d76, 0x0001c48002ed4374, 0x00044dcb0a7f497e, 0x000e0686221cce2c, 0x0000000000043785}, {0x000bdab4ad1119ad, 0x000279ee9061f323, 0x000458b83a9d33ec, 0x000e85b76f7f52c7, 0x00090fd6c65f1d8d, 0x0001f8a3939a713c, 0x00040af74ca06771, 0x000000000008ebc6}},
        {{0x000c37bd5d97c8e0, 0x000841942c2ff6fd, 0x000f5c9ed6475a4f, 0x00052b5771a5972c, 0x0004363c2048f3c1, 0x000db8da24cfb1e7, 0x00043dcd2ce8249d, 0x00000000000a5ee4}, {0x00044b387eb3689d, 0x000606fa84b3cea2, 0x00069e100b1b94b7, 0x00071368e13869e1, 0x000c96f0f8a12e45, 0x000d93dc039ac6a0, 0x000ec9e24458ac42, 0x000000000005f60b}},
        {{0x0001937d698b9991, 0x000c7514cc1dc470, 0x0007f6d12f9a74ce, 0x00022ac2fc9230fb, 0x000d92e23e21f0e4, 0x0006d1ac0aa50a1a, 0x00049370ac46d867, 0x00000000000f9f54}, {0x0006e32302426e7c, 0x0002059a6b86a840, 0x0004338952e5938c, 0x000cff13192968cc, 0x00040b342dd8d3d9, 0x0001da44113e700b, 0x000760fd5dc7bb3b, 0x000000000002c883}},
    },
    {
        /* digit=70 [{1,2,3,..,}]*([2^280]*G)  */
        {{0x00067944f366244c, 0x000d59e9af31428e, 0x0009cbe9f6c4942e, 0x000bd92582864947, 0x0009b2e09205204a, 0x00056b1017995988, 0x000f260c727a3dfe, 0x000000000008b657}, {0x000fac647b56af7e, 0x0004cde35d954514, 0x000e44c1af2c53f1, 0x000261f36019feec, 0x000966511d8ffa03, 0x0001148b9afda42d, 0x0003e1f63211dd82, 0x000000000006f13a}},
        {{0x000fe40d0b19b4c9, 0x0006a3997900ec16, 0x000abf1980c5bfc2, 0x0005b0a661f3c579, 0x0008aab1292b71bf, 0x000d993f244b1338, 0x000966cbe80cb9d4, 0x000000000002203f}, {0x000fa9a95e9148b0, 0x00053bc82a651a1e, 0x0005a004b90c2827, 0x000c7e7502f0003e, 0x000056a38f73a388, 0x00087d9b09c704ef, 0x0001f393cde8a734, 0x00000000000ec11a}},
        {{0x0004e6a4ec3efae0, 0x000e047fe47986cd, 0x0007fd532d6e5353, 0x000fa8b9a8671744, 0x0000514bf471b76c, 0x000dcf1fc9bc87bf, 0x000fa837392dce71, 0x000000000003ad1e}, {0x0001e59290757ebf, 0x000cebbea5a33841, 0x00075b7402f28c87, 0x0004ecd9b1665aa2, 0x0006e329df225b3d, 0x000aa808de7b0df2, 0x000faf18fb438e0b, 0x00000000000dd516}},
        {{0x000d8d8f4a6b2b66, 0x0002a63f27a255cd, 0x0003341f52773d72, 0x0007cd965f9ce38b, 0x0000635ba3005d31, 0x000343662162c92e, 0x000ac85259f64ffe, 0x000000000008e614}, {0x000a1c59580d5fe2, 0x000e397fb55baa90, 0x000365cc03ff132a, 0x0007325780618255, 0x00086c1e6306a57e, 0x00067b14c892f6b3, 0x0008c5f12e4c0723, 0x00000000000c83a4}},
        {{0x00040eada4657ebb, 0x000e31e6f9a95314, 0x000a04269b290326, 0x00056256e8b41991, 0x000b7a4a53f9365d, 0x0002b9b16e1b9c53, 0x00070155dc1d50e3, 0x00000000000bb6d5}, {0x000b5b1121311bd1, 0x000984c270249c99, 0x000f7b0846375c38, 0x0005b610689902bd, 0x000fa4cfc5a819a9, 0x000e5b424dd5706b, 0x000a87d33bdb7314, 0x00000000000e69eb}},
        {{0x000fd0482abbdf6b, 0x0004f6897401cbf6, 0x000652cc2d40814f, 0x000b32939e2b43ba, 0x0001bf809331e511, 0x00058f8f43952286, 0x000727815f6c1dc9, 0x00000000000c213e}, {0x0009a038bd421fe7, 0x0008ceb09ae59065, 0x000d25d81924ad76, 0x00015a2a20f86c7b, 0x00069aa308078f48, 0x000758878748a176, 0x0003d7f1f46a211b, 0x000000000006f6fc}},
        {{0x0005626ca6bcb860, 0x000ac75ba2f9acbe, 0x000c234135494b93, 0x0009e0b610d080fd, 0x000333a99c2f3c33, 0x000580aff614ac29, 0x000a3aa5e23af2aa, 0x00000000000e1fdb}, {0x000ca5f5b322ee30, 0x00065aa603365011, 0x0001b05f57ab1134, 0x000524456330a336, 0x000b9e025aa3a2b2, 0x000a0d5cfc396b5b, 0x00083babbf77faf8, 0x00000000000ea31c}},
        {{0x00047a49edb97340, 0x0002d300b04831fb, 0x0008ca060cb164c9, 0x000c5420cc10a74f, 0x0004eff9d01017d9, 0x000aa12857f637e8, 0x000114d732aa1e27, 0x00000000000e2a77}, {0x0008fc5f031be351, 0x000d262acf1585bd, 0x0001b0dfbfbd395e, 0x000c5d22e6eb2db2, 0x00028d18263d88be, 0x000bbd6acaec2720, 0x0004c04b687353e4, 0x000000000000ff7e}},
    },
    {
        /* digit=71 [{1,2,3,..,}]*([2^284]*G)  */
        {{0x0006ecc79d0fef75, 0x000dfa6e19d09c24, 0x000097d1dea89821, 0x00080950ee527a58, 0x000b7c4278a6ef73, 0x0000798ff78b7174, 0x0000cac133d867e7, 0x000000000009e923}, {0x000f7954ffb5e165, 0x000877431eeef1ad, 0x000728c869881eba, 0x000306a8d6af3f83, 0x0000bbec076af7a4, 0x000257ca3633bb5a, 0x0006636d07623e7a, 0x0000000000021c00}},
        {{0x000f8db3d9e25bfd, 0x000ed35ae891ae18, 0x00094f42c82c0dd3, 0x000b3cae63745c4f, 0x0005f218d139c2da, 0x000f3b0e255b8c18, 0x0003fa6ab039c889, 0x00000000000c644c}, {0x0008fe1e4d75d65d, 0x000d9c8496f1ff0b, 0x00010bc25fdaddb7, 0x0008392921149191, 0x000fef73f3455c67, 0x000fcfb22e962e52, 0x0009113818baf354, 0x000000000009d4bc}},
        {{0x000af695e08e48a8, 0x000683a35938698b, 0x0003114a51ff94f2, 0x0000f82899d204cf, 0x0009158780b5fcce, 0x0004edd4f4ca4158, 0x000c34c28f1bb73c, 0x000000000007a6ae}, {0x000b281e555c9d40, 0x000676d4783df7ee, 0x000d1f8a08acdccd, 0x000c2bcdf6d7f923, 0x00000f356ca3b7b8, 0x000b8e964f0a8d47, 0x0000e1229894bfde, 0x00000000000d3aa7}},
        {{0x000d000279a1a838, 0x000b77f1f985b888, 0x000a43ce9b20f9f6, 0x000930a5188fcf70, 0x000ba3a82904b40a, 0x0001510273fad79a, 0x00017b7bee8d22f3, 0x00000000000c2c3e}, {0x0003fdcf0848373a, 0x000041fbe06e92b8, 0x000ae48de4a30594, 0x00090ec2d9cf7a56, 0x000a9bd062c5fff2, 0x000d353815d0deda, 0x000eda080344abd7, 0x00000000000f5cf9}},
        {{0x00012d81fb28e752, 0x00018093526223ac, 0x000d23e13369699e, 0x000d26ee94c9abed, 0x0003a20b52679a62, 0x0003d5fa4425b86b, 0x000fc68164f9c8e7, 0x000000000003b12c}, {0x0006c5fe565db2c2, 0x000be6cc34bad21b, 0x0002901d6245306f, 0x000f50970e4963cb, 0x000cf0813b31f80f, 0x000f1ab8ee403ccc, 0x000b92a72ab5f584, 0x0000000000070f15}},
        {{0x000e213ed814f6f4, 0x00034170098ddc99, 0x000e9c62ee04edbc, 0x000b04b8660d7749, 0x000b5152ad1e35b1, 0x0008e92aa8b4b6c4, 0x000f4e0b0bd1c24a, 0x0000000000088172}, {0x0007445379f53f7e, 0x000d513a76898c7b, 0x000f4e73336bcd42, 0x0004f973dc17e72a, 0x00060a7124b24b39, 0x0008a372c1b0cd16, 0x00003b7549880cce, 0x00000000000e7126}},
        {{0x00055aa45f74a004, 0x0008dc14a3abb979, 0x0004a411c66e9f88, 0x000f1828a4be08cc, 0x0009deca5a89e49b, 0x000fbb36be6e7bf8, 0x000e57a7326cf88e, 0x00000000000c7f6a}, {0x000da2212d19f21a, 0x0005dceced37b724, 0x00062d313930a591, 0x0005b3fb62a62e4c, 0x000793e5c0e1a7e6, 0x000b8d77382cb79a, 0x0002f014c9a99461, 0x000000000000b05f}},
        {{0x000a2dd7deb1dde9, 0x000a831d31f799b5, 0x0008e986349103b0, 0x000362e52f04d685, 0x000921c7b6511793, 0x0006f47f65808e70, 0x00040533fe9123ed, 0x0000000000058f71}, {0x000d5dbe8f1956b4, 0x00061821d30451f3, 0x000e7b4e166f399a, 0x000db145b2048aa2, 0x000ed7811330932d, 0x0008fd85f17d989f, 0x000434032ae4cb12, 0x000000000007d1ef}},
    },
    {
        /* digit=72 [{1,2,3,..,}]*([2^288]*G)  */
        {{0x000acb4203cc71b2, 0x000a9070d34e112c, 0x00047b523d821ba6, 0x000f9b95eb8ad292, 0x0006a6a45dda269a, 0x000df41e0dde0a03, 0x000f2aa18b528e63, 0x00000000000a260d}, {0x000c22d6a6aefb01, 0x000d424a67709f9b, 0x0002a9971343d570, 0x000e2a1968444330, 0x000d131a6cf073c1, 0x000e794543660558, 0x000dcdb0289c832e, 0x00000000000c6d8d}},
        {{0x000d86ee6f475d5b, 0x000d30ed1082f1a9, 0x000a6711ccf28680, 0x000ef0eece773534, 0x00064bf0c426a35f, 0x00011b2fb76adaa4, 0x0004d3fbab929aa0, 0x000000000009d8cc}, {0x000bb954fd395b2c, 0x000dabe6d7f2076f, 0x000a4cbdfe911e02, 0x000cba5106ceaac2, 0x000a45258697c7e2, 0x0005945b0fe94528, 0x000f7a7109b17d07, 0x00000000000cae17}},
        {{0x000ce6be22a12c0b, 0x0008a6855029961b, 0x000a3c801158be2e, 0x000d2b4a8e92e70b, 0x000f0cc9fca9f601, 0x000bc2e96fd02799, 0x0009b889f99ca013, 0x00000000000ff9db}, {0x0006b1bd37f91ceb, 0x000616ae70849faf, 0x0005d65bd4da44b0, 0x00092a4e0945e8f9, 0x0002ecea36f6394d, 0x00018a069ddab675, 0x000860650b74d60d, 0x00000000000ad650}},
        {{0x000bf5b4a4380ef4, 0x0001cfc1faedb8cc, 0x000e6ba03d77f4c0, 0x00085456416defe8, 0x0000c58653e7f004, 0x000f6f99e75de777, 0x000c07dd2dcbebe2, 0x00000000000d159a}, {0x000c9af1998a9b17, 0x000a045dd713c4a8, 0x000e1c3678ddf502, 0x000756b6b962b38d, 0x000bbffd3f0c3f73, 0x00008ebae3cb9f53, 0x000dbad723c40b7f, 0x000000000000c3cd}},
        {{0x00003107748ef382, 0x00058a804b2f0156, 0x000b319b886fb1eb, 0x000f4fd353970be7, 0x000712854a9131ba, 0x0009713983481c8b, 0x00094ab0424eecf3, 0x000000000000f4ed}, {0x000648ea129bd87a, 0x000a992b74ad013d, 0x000692f5d2444d08, 0x000b28070a5f0c93, 0x000b9f5101ca6741, 0x000d0f81e46c12cf, 0x000288e7014d7901, 0x00000000000d758c}},
        {{0x0000c39f81600214, 0x000686a565269228, 0x000c17ee858626b0, 0x00011a7d8c6ddb92, 0x000b2332347b0bda, 0x0009dff48b7ebfaf, 0x000c83cf791881e9, 0x00000000000dc357}, {0x00082704ffef2257, 0x000fe2a264f8831e, 0x000c9352584eb7a7, 0x0008260bb47d4188, 0x000c3d8170326d47, 0x00005f901544de4e, 0x00063cac7150a9dc, 0x0000000000075c09}},
        {{0x000bcfd07a0f22c4, 0x0007c58805ca858d, 0x000b1c2b2351e4ce, 0x000c9a1349cfed00, 0x00017800a980f9cb, 0x000847e9d67bc7b4, 0x000c48e7e6dbc0f6, 0x00000000000af379}, {0x00047d63263e04cc, 0x0003c80b6fc048b9, 0x000808eb22fe8f7f, 0x0001ed14e9dc24bc, 0x00073fab87e6d2d1, 0x000c630ae48d74f1, 0x00020a1feca5af50, 0x0000000000062d06}},
        {{0x000e56ad2bfbbdbc, 0x0008ff6377e3bcab, 0x000f30b92835c3f0, 0x000ac42c21c0cf1e, 0x0009857d8edb7f85, 0x000b77ed67a3e31c, 0x00010b2eb1076327, 0x0000000000038dae}, {0x00026ce2d0dbce6d, 0x000dced7c6fb0aad, 0x00066cec0d07ac1c, 0x00035504569e7309, 0x0007bedae3ab4dc4, 0x00027e463aa82fb9, 0x000a1106d37bef4f, 0x00000000000f0c35}},
    },
    {
        /* digit=73 [{1,2,3,..,}]*([2^292]*G)  */
        {{0x00011f784b3a0046, 0x000066edf5561b45, 0x000c8578d54224c1, 0x000ae56d8d66d29a, 0x000ae9d58eb27699, 0x000f9176d5f81602, 0x0003b1d0c04874fa, 0x0000000000052315}, {0x00067980492adf03, 0x000309690930fad0, 0x000146ea47ef3788, 0x000725873bcba90f, 0x000de902e3292d0e, 0x000bc27b957efe72, 0x000c093fcd598676, 0x0000000000093940}},
        {{0x000cdd973668c530, 0x00077c68b641e912, 0x00064e1ee190b62b, 0x0008cbad89ce6688, 0x00006c269d8eeda9, 0x000f3402315f84dc, 0x00034af0eb685ecb, 0x00000000000c0326}, {0x000bc909db075650, 0x000562e3d7fff094, 0x00012e59576050a3, 0x000abcfa3e050e0c, 0x0007c001dcd9f02c, 0x0006d2d52ccd2da6, 0x00041f50aa372306, 0x000000000007224a}},
        {{0x000ec1f1185b9762, 0x0007e4b1f8698c94, 0x000d0af36ed765ea, 0x00049f56536e156a, 0x000109c95a75ba1c, 0x0006d9af131c3163, 0x0008b7268a308f0a, 0x00000000000fa9ca}, {0x000f13b9ac944487, 0x000b1c3b1cbccb86, 0x000446bb4e60be0c, 0x000c7b334396850c, 0x000a048a69431b4e, 0x0005820bf42f21d5, 0x000a580bf948f55b, 0x00000000000b48dc}},
        {{0x000b86ea9a29c966, 0x000cd04418b58cb8, 0x0002d1612a6d4e73, 0x000e22c7c65a6da4, 0x00051dbeffe45f87, 0x0003cc05caac7106, 0x0002c4ff619d64bd, 0x00000000000e65c9}, {0x0002126bd4b5ea52, 0x000d123af2b21062, 0x000e0691b9e9f16b, 0x0007efac6478c561, 0x000af6cd140ec34a, 0x000a0db2e57cde95, 0x000b7664b74575f5, 0x0000000000075d7f}},
        {{0x00078c972345257c, 0x00058a9532cb266c, 0x0002b89ba64976de, 0x000067545b1733a4, 0x0004cc6e65dd74ea, 0x0006ff2aa7f37f02, 0x00079e6612b1425c, 0x0000000000030374}, {0x000f1ac7f7aa2f0c, 0x00097e512bca4fe9, 0x0003049963042b95, 0x000bd29a259d6515, 0x00077373034f5b97, 0x0005aa640215eb65, 0x0008b779da0fc488, 0x000000000007e435}},
        {{0x00041294c6ef7a53, 0x00023ce3cc51372d, 0x0009af7f675a386a, 0x000d981efc9f71b0, 0x000a20bea9baa9e6, 0x000ef7c1fc01ded6, 0x000a5cc4b7e6424f, 0x00000000000f9dec}, {0x000954a122b474b3, 0x0002aa0a43d0a3de, 0x000a0d55c109a592, 0x000a173b5d57bbf3, 0x00056299978197ce, 0x000ca2bb13c3f30f, 0x0002e069a7820a8e, 0x000000000000ff69}},
        {{0x000deeb8201c384c, 0x00029aca3f998176, 0x0001642e9c9fe922, 0x000dbd63d82750be, 0x0002faa3400be031, 0x0005fd54bfb2552a, 0x00030374e9b5d365, 0x0000000000035894}, {0x000839a12661f3f2, 0x000a15a575a4221a, 0x00056e3aabcc078b, 0x000ada596e1e1175, 0x000d0f305dc3ca6f, 0x0008d7af5126e9af, 0x00069391da8d7305, 0x00000000000d8353}},
        {{0x000cb9cfccb5f12c, 0x0002c28405061c1b, 0x000211e4805b5eb0, 0x000944405547f7c3, 0x000d3441e472789d, 0x00044ce877fa445e, 0x000d14feb198254a, 0x00000000000129ae}, {0x00073247d16dc8ed, 0x0008622b05de2de2, 0x000590692ea55dfa, 0x00052caa83f7f094, 0x0006cefaa757fba2, 0x00017d8060f6ad6e, 0x0007ade1ad187165, 0x000000000008e9d9}},
    },
    {
        /* digit=74 [{1,2,3,..,}]*([2^296]*G)  */
        {{0x000c5ac4350713f2, 0x00062cbb991946e6, 0x00025875bf6be408, 0x00024aab6ab6c041, 0x0000bc41f4b7d50b, 0x0005496cf419d281, 0x000ac0e2e88d7a29, 0x00000000000f2457}, {0x000a357ad05dad34, 0x0008e838e4a3533c, 0x00068ef45f1df4fb, 0x000ab1c7a5c5fdd4, 0x000dde34cb8d9fa3, 0x0009105fe324798c, 0x000249bfa5a9d088, 0x00000000000fd0df}},
        {{0x000b286ea52474ce, 0x000419974759bb2f, 0x00039b1e3f8aef8e, 0x000c58ec64941cb4, 0x000a01d055758afe, 0x000448db52d7c57f, 0x000440c07c74cc5c, 0x0000000000001bb1}, {0x0007977c86a9c43f, 0x0003893162d96796, 0x000ee52587841e45, 0x000c57e6af18dedc, 0x000f174a30894f28, 0x000dd626536faed8, 0x000d7b702dbd64a3, 0x000000000006adc0}},
        {{0x0002a70e7d991630, 0x000854981be688ec, 0x000de0f53a95f8f1, 0x000e05074cff00d2, 0x0001b52c7f9c3168, 0x0001c7986111150a, 0x0002ad0db2ed84a5, 0x00000000000e6127}, {0x000c16b3e4d1fe2a, 0x000897443d935440, 0x0002c51d6c8aaceb, 0x000ef4e866b5e41e, 0x0005b965a67071e0, 0x000e1aa7705c57a4, 0x000e3285bdb58c70, 0x000000000009df3c}},
        {{0x0009a11de31a0e3c, 0x000c6c41e4aed12a, 0x000342590d5103b5, 0x000ab7b8ed812d38, 0x00081afb58b8f5ea, 0x000efcc3a7801fbb, 0x000b91be5cdba671, 0x00000000000cd10c}, {0x000761b96524aa0d, 0x00013882e821e20b, 0x0008d1b11d20578f, 0x000de2bd8a0f0393, 0x000c857fd3a03afe, 0x00060f767e20a117, 0x000da1ebaa2aa430, 0x000000000002b9d1}},
        {{0x000d9fea4e493f3b, 0x000e070e478fb44f, 0x0000254b6a02edec, 0x0002b005be36ea75, 0x000f6717e06aaec8, 0x0008f197782618b8, 0x0000d12db1f6d400, 0x0000000000016b39}, {0x000ddfa794ec9644, 0x000bafe86fd4d041, 0x0006bc15de0d2bf6, 0x00023e70c0491593, 0x0005ed4562356cbb, 0x000d836efd05be7e, 0x000604601374069b, 0x0000000000032e5f}},
        {{0x0001ebab46405b61, 0x000f009db138cd2b, 0x0009fd7df23c53c0, 0x00047794feebab9a, 0x00017fdb1e710bb6, 0x000f1f65dfce4aee, 0x0006cd7d5c0c61a7, 0x000000000008d02d}, {0x000621234d968fad, 0x0005c7cac0485224, 0x0002b8dbf87d2555, 0x0009a59db0aa864d, 0x0008ff94f8b5b587, 0x000c97f8e75f391b, 0x000232a6c994ae49, 0x00000000000d690b}},
        {{0x000e6486f79bfa96, 0x000e507c60c742b4, 0x000d5aa8cb9b07e8, 0x0000a09dcba89982, 0x000a802ce50ca8b0, 0x0002144b0a77f1f3, 0x0000db6c4050df2a, 0x00000000000ab1b1}, {0x000a93fc6ce00720, 0x000e89025d46fd65, 0x000c1e4037fdb9f5, 0x0002d629ae2d5361, 0x0009eff0f6a9bcad, 0x0005fa5a07a15282, 0x00039ce87345cd92, 0x000000000000e840}},
        {{0x000cab548e246919, 0x000d15017e1454d5, 0x0005f67186d3e9b6, 0x000059dfa98234b5, 0x00099985af982541, 0x0005b7b1c4bf344d, 0x0001ad39737ed67b, 0x0000000000087c41}, {0x00069e3ff4c10920, 0x000022fad8cecc3b, 0x0008e45c000ca718, 0x000d856404043f8d, 0x0005863927a11a53, 0x0008921216c1f07a, 0x000c8f38d3a4f4b6, 0x0000000000001976}},
    },
    {
        /* digit=75 [{1,2,3,..,}]*([2^300]*G)  */
        {{0x000dd4906efe62a1, 0x00063bcc8842253e, 0x0007a247757fdab1, 0x000a1ef2571d6b1e, 0x0001739a956e745d, 0x000bffc16f01c292, 0x00072ceffd8065e4, 0x00000000000f36fe}, {0x000478592361b14b, 0x0003ff49e90d1ebe, 0x0001338e12b047f8, 0x000b583b7240d91d, 0x000c03f8387fcb78, 0x00000f2d92e1a7c3, 0x000a8416566c2232, 0x00000000000aaf80}},
        {{0x000f32d335de93bd, 0x0002d7a1a522aee8, 0x000b425f9e81403f, 0x0004abe3d4ac4abe, 0x000337f996e841bb, 0x000727761c52950b, 0x0007d201d991e679, 0x00000000000978fd}, {0x0006e2e2907ff054, 0x0007486bad5e5c68, 0x00010431aa78725c, 0x000eb7705757453a, 0x00093939df7758ff, 0x00078ea0fbb18612, 0x0001d3bfdd394a6a, 0x000000000006e33e}},
        {{0x000c3ccc7d56f63d, 0x00076db1cffd2381, 0x0002131b488cf4cf, 0x0001e5aa3c1db85a, 0x0001ac55d391a418, 0x000d270d0e5183e5, 0x0005462793d5efef, 0x00000000000c0631}, {0x000c80c1d49264dc, 0x00008ecb40a58378, 0x000aa60ec18fee5f, 0x00014ab6c5830fab, 0x000ca03f201ea89b, 0x0008c54f64dc477d, 0x00066e5604f5dac8, 0x000000000002acfc}},
        {{0x0003e9c742a5516d, 0x0009da7f29531cb5, 0x0007fe865f9eba30, 0x0004ce9efae348cc, 0x000cac52758dd2f5, 0x0002961a45acb931, 0x000fe1287b3e18d2, 0x00000000000748f8}, {0x000f0554bb131f76, 0x0001d1b62b340cc5, 0x000b08ad2a8e1d6a, 0x0008dadbe8da8486, 0x000954f18276bad7, 0x0004bfdad85fa710, 0x000237e2cc9d287c, 0x000000000002c75f}},
        {{0x0004d2c2f1f927d3, 0x00033283dd7ef2a5, 0x000a91f40054b9d9, 0x000c095870282aba, 0x0008f4e36324ba08, 0x0004cecf2b02b00d, 0x00024e53aaddc060, 0x0000000000084ddd}, {0x000a889fa3214142, 0x000ca34bc00a03f8, 0x0000c263c00349e0, 0x000eaaacef68f74d, 0x00023c0293515228, 0x00042e3b740a790d, 0x0001e0e0e9af5c84, 0x00000000000a9f14}},
        {{0x00087ba6916bcaca, 0x0005bcdcdd7d6721, 0x000b58247a460e0a, 0x000c569c00ce5c40, 0x000fc589d0c75923, 0x000fa6fe6099c17f, 0x0008c40335eeea89, 0x00000000000a4e86}, {0x000a7415516e07c0, 0x00024b7037325a9f, 0x000944c467e9c1bd, 0x00032efd5e26d28a, 0x0008de919cbedf31, 0x000bbb18d7ed994b, 0x00050812df67cd6f, 0x000000000006baff}},
        {{0x000c3f58ee594ee5, 0x0008b72f4949e0b4, 0x0001e5bef192dfaf, 0x0004726092a58066, 0x00010f5cdb7271b0, 0x0004bfc49ad5c3c1, 0x000be7d4951f9e55, 0x000000000006131e}, {0x0009f77a3840f92f, 0x0009237da59f303d, 0x000ff0c06b034503, 0x000962a4d89fa660, 0x000e71d824f5d020, 0x000050e312610c61, 0x000bfb9c917da73c, 0x00000000000e6e7e}},
        {{0x0003a9eb85c489f9, 0x0003af2ec9ef5122, 0x000d827990bab108, 0x000b2387d2ac6e41, 0x00020318a3b790e2, 0x00084a7a0cf682b1, 0x000d057fe76089ea, 0x0000000000016546}, {0x000bc6c0a0421345, 0x00041f987118eca7, 0x00079f0cde4a8da9, 0x0007774888a7e8d0, 0x000a63ec5831544c, 0x0002fe985bd2647b, 0x0003e8b479cea3fd, 0x0000000000028d16}},
    },
    {
        /* digit=76 [{1,2,3,..,}]*([2^304]*G)  */
        {{0x0005b08a8a2a82b4, 0x000fa04baa17598e, 0x0006a0661405c62f, 0x000031dc6cd61096, 0x000030bf87a5c4a6, 0x000e497ccba2534d, 0x0006657ebc6e2131, 0x00000000000851fd}, {0x00018d257f3e221b, 0x000e69ecae010ee2, 0x0009788f9b15d390, 0x000a9c22fa527837, 0x0001236d442e39e8, 0x000c289ce74c9380, 0x0009cf8b21ba23b1, 0x0000000000025c76}},
        {{0x000aafc618669c84, 0x00037761e10e2b1a, 0x000c03aea1a1a99e, 0x0000ff43dbc6fcaf, 0x0006cbe0ba8aa087, 0x000f1ff6a9765128, 0x0005b647d46075c0, 0x000000000003abeb}, {0x000f7f59840e13b0, 0x000af1234c80e297, 0x0000d9f885a37dab, 0x0002d41d82ab780a, 0x000fde426e50399a, 0x0006dde778a03afa, 0x0005eca2bcb109ca, 0x000000000000618f}},
        {{0x000adbd12143606f, 0x0003e0d27500f7d7, 0x000cde21714cb370, 0x00078f8610763e40, 0x000f47ef08feee75, 0x0007d04c06e56078, 0x000aa3a8d03a7f04, 0x0000000000027cc8}, {0x000677d27af9cf4f, 0x000056b08eb37c00, 0x000a4a2b5d3fdd3b, 0x000869ae3f10ba99, 0x0003e500e8466f9b, 0x000171b1a3bfb983, 0x0000456d975557b9, 0x0000000000018fa0}},
        {{0x000e348941e625d9, 0x000fc15d47e52aa7, 0x000dc7038ccf5369, 0x0006a3070cddb11e, 0x00032b5a8db75122, 0x000f37d6563a253d, 0x000b91e8be4d1fbe, 0x00000000000c0920}, {0x00045423a83becc9, 0x0008308b713a8319, 0x000ebb9d576c8cba, 0x0005877917ee393e, 0x00091528cd12a897, 0x000daf6aa294792e, 0x0009b94512dc8c37, 0x00000000000197a9}},
        {{0x000e5d8cfc9c3b49, 0x000e0cfebe026cd6, 0x00010748e4a52d81, 0x0009d16725ebd099, 0x00007a584e8d00f8, 0x0001cab5894c6f2b, 0x0003c8095ac9cc91, 0x000000000009c407}, {0x000462695d0ed83f, 0x00056265919b06a1, 0x0003cfa033d1cd2a, 0x00097f1815a70aa4, 0x00001c5fd35a6db5, 0x000d52a8bdd9e4a1, 0x000fab38b8e35112, 0x000000000002ef20}},
        {{0x000003aa77fd8f2e, 0x000b43b21dc1edb9, 0x000008b1e2380339, 0x000e5d30da87d664, 0x000f0cd999a0f536, 0x000e08f5d4ce10ed, 0x0007e68949181450, 0x0000000000052664}, {0x000bcdd23ee4cc2b, 0x000a3cb207ee8e61, 0x0002ac9dd73c4f6c, 0x00040d225e0cf154, 0x0001067fa0527d49, 0x0008b2e21b688b31, 0x000eab89308326a9, 0x0000000000072311}},
        {{0x000c6c4de58ef59c, 0x000db2a6280408d6, 0x000f4e9cd796cbfa, 0x000daa3c647fcba8, 0x000f4952d3a44a2c, 0x000e2e2c3e1a9e91, 0x00040ebc1b3d8011, 0x0000000000018e43}, {0x00017ee00926dad8, 0x000cceb8696ff13d, 0x000436379cf7b867, 0x000d6ce6c1e905c6, 0x00091ed2e8cba471, 0x0007c8c914e13d60, 0x00084e52951509c7, 0x00000000000e2348}},
        {{0x000a3b1514eda4ba, 0x000ac37be5d3f53e, 0x000cab6203ce338b, 0x00024b59d1569e7a, 0x0009b4e293ab165a, 0x000b7c0a4254aaf5, 0x000b183c751fbd6f, 0x000000000005018c}, {0x0000fe7bbd1e6c72, 0x000eaacb3b8965e4, 0x000a2f2ab6db69d7, 0x0007258621df6d82, 0x0003185f0e2a245d, 0x000ddbd812af0e28, 0x00002a1e7edc818f, 0x00000000000c538d}},
    },
    {
        /* digit=77 [{1,2,3,..,}]*([2^308]*G)  */
        {{0x00042da90721f8f9, 0x0009f6858b6747d2, 0x00082ecd0813dd9f, 0x000674e29d9194d6, 0x0006dcb30a51324e, 0x0008aa5492dc6fdf, 0x000a923f3ecb7752, 0x0000000000023ffa}, {0x000c7344981b01f4, 0x00090d8efcbd58f8, 0x000d0144b46e312e, 0x000b0b74cf4a1708, 0x000fbbf335fd04f2, 0x000928c36bc0bb49, 0x000d45c6b6e5bd5a, 0x00000000000d893d}},
        {{0x00016a2799bedc34, 0x00022fb083c2a5e5, 0x0000296a1d9c8cea, 0x00030e747a81b965, 0x000658a987376e41, 0x0002829c9cd704dc, 0x000d63842ae32959, 0x00000000000be20d}, {0x000f531daa9bdad7, 0x00053abee8e552a9, 0x00078232300aac8e, 0x00039b2467112ba1, 0x000fbe341038781f, 0x000a2f8be6f06058, 0x00066e5effdca512, 0x00000000000af344}},
        {{0x000afa3ee32c886c, 0x0006d3933b4c0742, 0x000439c40f3f936d, 0x0003c0f0b7e019e4, 0x000fa48715d652fd, 0x00039be537f8e5dd, 0x000cdc0879c5defa, 0x0000000000056f01}, {0x0005bfa2d160b7fa, 0x0004671b2dcba45c, 0x000e0bb6fffdcb60, 0x0004846640b2d753, 0x0000049a2cbe6af9, 0x0002ab0f178a1071, 0x00043841a1ce67ac, 0x00000000000a76d8}},
        {{0x000a317691072fc1, 0x00025ab86639e0b1, 0x0006a85f7a13b8bd, 0x0004418d763f0bbd, 0x000211a9802cb573, 0x000269b8ad2cc332, 0x000355e41d7db881, 0x000000000002ecfa}, {0x0004e57bc1e94af4, 0x0006159d6912442e, 0x000356e842631b03, 0x0008b6b8a7397395, 0x00091cc83ae38447, 0x0007ccd97e86ad7d, 0x0004c6bc9b208479, 0x0000000000069494}},
        {{0x0006d3d1d84bf486, 0x000a4bbb6725a843, 0x000b1260c5ec574a, 0x000dd0664bcb2d64, 0x000a780d5c1bab65, 0x000a61e6e755e848, 0x000b4b84e6f686a3, 0x00000000000bd100}, {0x000cfe4a11d7ea4f, 0x000a3b34fdf84517, 0x000d9afd2967c440, 0x000c881f9a125f0d, 0x00064aeadf8e3c2c, 0x0006bf0c2b1fcce5, 0x0001bebdfc54c0c4, 0x0000000000009d28}},
        {{0x000af84d84069bc1, 0x00010e1aa844a91f, 0x000628bd439b9e84, 0x000aba65ddcb5d25, 0x000a38009c6a3025, 0x000261ee5ac8cdab, 0x000430b9ee07b46e, 0x0000000000006ab8}, {0x0003196ba81697a7, 0x000207a36b461d0a, 0x0005a884d3b0fe10, 0x000f469341275a1d, 0x000f1c7cf858ae4e, 0x00092af3df7cb393, 0x00017683c00eafd1, 0x00000000000a1e15}},
        {{0x000475d9329b61bc, 0x00078687d0980c72, 0x000d109af3ad73e4, 0x00007fabbcde5b3c, 0x00016ed103a56414, 0x000a6ee2212ec9d2, 0x000c11f858c7851f, 0x00000000000818f2}, {0x000247f6c002433a, 0x000a04720d42a19d, 0x000f64099b42a8aa, 0x000f25533723e4ef, 0x0003d3260b6299af, 0x000391c33d6c67ce, 0x00038163a6fc9d86, 0x0000000000031de2}},
        {{0x00092324c0c2dff4, 0x0004f585edee2c89, 0x000820f42406ab00, 0x000924fd85ad3203, 0x00019124bffa3bd7, 0x000c628682e7d8f9, 0x000f5d885397c044, 0x000000000007fda9}, {0x00035c11900b0d2c, 0x000e6d302582e314, 0x0003e1f066fcf73e, 0x0006b5b63653c06e, 0x000d9c70dad4b021, 0x000bba93d8f08a1a, 0x0002b7ccf014aaff, 0x00000000000a33f1}},
    },
    {
        /* digit=78 [{1,2,3,..,}]*([2^312]*G)  */
        {{0x000cb198f774dfa5, 0x000dd1e347819af2, 0x00020411ce747fbb, 0x0005ccc28af85bb9, 0x000b3ae9ba11fc18, 0x000bdb313c923d78, 0x000b877d494d7ea7, 0x00000000000af8f8}, {0x0004fe8c8323652c, 0x000bdc178ccc093c, 0x00006a94572bacfb, 0x00047cc80a50372b, 0x0000e6f901976087, 0x000c2f2168147aa9, 0x000b514683d4c978, 0x00000000000552f3}},
        {{0x000f8017d10c6b96, 0x0002229d81537326, 0x0007f1980aaaed23, 0x000ce6e8783d1fe4, 0x000df0e3c126aa4a, 0x000ca4b719c14ebd, 0x000f350eb1c08d6e, 0x000000000001c91d}, {0x00013bd5d85a7c75, 0x00064db8119bb0fc, 0x000fdf5036f03e4b, 0x000916835436f82c, 0x000b9b18d212053c, 0x000bc3497eb41ccb, 0x000d8193b3fb43b1, 0x000000000008c1ce}},
        {{0x0007ad7a8ff88fe5, 0x00031521f4af6aed, 0x00033eaf4c7af196, 0x0009ab7ae0377807, 0x000db65f73ffad09, 0x0006059b9541cadc, 0x000f8253430463a8, 0x00000000000043e9}, {0x00061f938516beb2, 0x0001bd515f4c75f1, 0x000d0a9d767932f1, 0x00053567ef1b9007, 0x000f1295b5fe7bed, 0x0009176278782b45, 0x000d9fab54ebaa03, 0x000000000008787e}},
        {{0x0006f626b572390e, 0x000905255508b5c5, 0x000fa6a60258070c, 0x0005eea3095205c4, 0x00002c470ef7d976, 0x00040bf0afea2ed7, 0x000a16f5e8a0fc9c, 0x00000000000e4137}, {0x00055209e40da33c, 0x00003b917537047e, 0x00003ab2893cdb40, 0x000bce9023854871, 0x0008a8bc3a94bea6, 0x000c450444620840, 0x00015164a87a2a3a, 0x000000000007df70}},
        {{0x000cf2f10b0520cd, 0x00004c38d07270c5, 0x000e4dc64fc6cd1a, 0x0001d8f300e953df, 0x00040a7dce63abde, 0x000425847b98b23d, 0x00061b63e51543fc, 0x0000000000043a21}, {0x000d54e8ecf42bd9, 0x000ee2650a1d5920, 0x00086c043a07e5a3, 0x000b7048bd9bd78b, 0x000df89e86bdc15d, 0x000579426bad1dba, 0x0005f8d7d4845024, 0x00000000000287bf}},
        {{0x0006ca98107f85ef, 0x000615fb05303f18, 0x00091aa7ab0bf32f, 0x0000fa47d92132d0, 0x0008994564bf7a0a, 0x00029e066e993faf, 0x000574bf30252517, 0x00000000000d4260}, {0x000990cf5caa4f19, 0x00009f1eecc3d948, 0x000f8dbc962efd6e, 0x0001cf1d1dd88736, 0x0003b6f1aae115fd, 0x0009344b97ce882a, 0x0009ca7efb7ec6da, 0x00000000000cc5b5}},
        {{0x0003c28364de2038, 0x000c48bcbf6cdce6, 0x000e355114722287, 0x000febf92ebe1bcf, 0x00019af24c2e4ee8, 0x000a861d57847829, 0x0007ec618760b094, 0x000000000006bae2}, {0x000de113e64c1ae9, 0x000fc77c174b9e18, 0x00055d1ba670f4b5, 0x0004f6a58a0062c2, 0x00047781acd1db9e, 0x0006d86b03480c9b, 0x000ef52d7c628ef5, 0x00000000000b3e2e}},
        {{0x000ade0c1eab3d7e, 0x0000fee9e25d91a6, 0x00060f30210199eb, 0x0001157d2d91300a, 0x000fffcbe50fd2fc, 0x0001d376ee7597e2, 0x0005624be3814fe1, 0x00000000000dbf14}, {0x000860a341c4b797, 0x0005638a107c07ee, 0x00045c1dbdbe39f3, 0x0004e09caebd481c, 0x000bf9ace60d1801, 0x0001cb8e7b28c331, 0x00065ee8fba04f21, 0x00000000000e42dc}},
    },
    {
        /* digit=79 [{1,2,3,..,}]*([2^316]*G)  */
        {{0x00060f98795275e8, 0x0009d009dc2f44a9, 0x000392d1d1132d95, 0x000faa2ec5d91fec, 0x0000f160afcbc89b, 0x0007566b5461be9a, 0x0009a2fdd084bb7d, 0x00000000000e4809}, {0x000f69da99c8ac01, 0x000bd1b461fc964c, 0x00012ac0cf0d2b27, 0x000332d9f09a206f, 0x00036941d6ef3246, 0x00087f5f6bde4c0c, 0x00040044ef03fdf3, 0x0000000000057b63}},
        {{0x000b480bf51dac8f, 0x000ea7e485158a07, 0x0002c9e9ee8b46dc, 0x0000dafa6fbabfcb, 0x00038f09fff720de, 0x00006994f04f0158, 0x000d335a0562976a, 0x000000000006e3cc}, {0x000d89d62ca77de2, 0x00045c087ef9c0e4, 0x000f0235509703e4, 0x0004814d916ee937, 0x0008d23d140b4fc0, 0x00062028f3369c97, 0x000adbb0fcd70f16, 0x00000000000e1e28}},
        {{0x0003a6684cfed7ee, 0x00094e3ccfe8dea6, 0x000d721274c869f9, 0x000e2cbb605c4274, 0x0005b0d809923f13, 0x0009913815ee1239, 0x000b6e4dcaea94b9, 0x000000000008c4c4}, {0x000ec88cc6307d53, 0x0004bd7aad54ab94, 0x000859bdd379b119, 0x0001eee68048fd9e, 0x000fd57225fc6eb3, 0x0008dfd3f693842d, 0x000cf13e62cd85a8, 0x00000000000d5462}},
        {{0x000df1c3e7de2ba1, 0x000ff99b5f14f615, 0x000c00e1f8e35232, 0x0003900ff19bf251, 0x0000260925ede749, 0x000ea1eafc82e5b8, 0x00021e3398d340c0, 0x0000000000012871}, {0x000896ddafdbd954, 0x00046ea6dfb0ad7a, 0x00039aa9f60646cc, 0x00078bf1201299a2, 0x000d32e5cebce266, 0x000181e32b5369e4, 0x000add2304eac86d, 0x000000000003d364}},
        {{0x000f357cfbf02e91, 0x000a2f55f868d743, 0x000049d92d19de08, 0x000ff9bbb5ba194e, 0x000957724703bec0, 0x000e0e4fbf7325c4, 0x000d7c5ab8f06323, 0x00000000000ecb0f}, {0x000a0f97049bea9a, 0x00069cc72e8da9e4, 0x000c7a0fc237673e, 0x00046e64252763c4, 0x000eb8d81de3bfa6, 0x000772f007a769ef, 0x000a5ef5ec70031e, 0x000000000002729a}},
        {{0x000cc742f063ab99, 0x0003b51f19bb30d3, 0x000878f139d45b8c, 0x000de2f987112cb9, 0x0008759de2cb5f7c, 0x00049bd98dc51d94, 0x0001fd9d05c410ed, 0x0000000000036434}, {0x00020d69ae77fca8, 0x000a9d7869d68233, 0x00007a8daf0bfe29, 0x0008180c25910747, 0x000a6ebc4ebc7f25, 0x000da1d687b90fc7, 0x0004935565390f3d, 0x00000000000a44e4}},
        {{0x000116432be8131b, 0x000ebb4a2742da3f, 0x0008bb7a8fda7aaf, 0x000adccdd7fcf4fc, 0x000337d4fcc6b15f, 0x0001c7ab70b1fa2b, 0x000cc9242c7b26f0, 0x0000000000050574}, {0x000069eaa2593652, 0x000b76ee50f421b3, 0x000a5035e1a2b7ad, 0x0003af18420501c6, 0x000b1b6e7d2a04d1, 0x000516f80c22e092, 0x0005b5a393be7ee9, 0x00000000000b80bb}},
        {{0x0003f2ad59508de2, 0x000bd9e41b9b6b62, 0x0006d3eb906556a9, 0x0000c8cf272346c8, 0x0005def5749367e5, 0x0002676fcc98a5ef, 0x000659c2dea8afa0, 0x000000000006ace4}, {0x000f1f5cd12db7d7, 0x00090c2b707aa093, 0x000d4cda421d1577, 0x0000fa2a778c20c3, 0x000d57af166b5b58, 0x00078ce62272c3a1, 0x000087e47a64a9ca, 0x0000000000071d35}},
    },
    {
        /* digit=80 [{1,2,3,..,}]*([2^320]*G)  */
        {{0x000e3ee1ae74a706, 0x00069c20fb9106bc, 0x0005cf0e923045cf, 0x000bc6a5cc0aa89b, 0x000b4fb9f01e12bc, 0x000ae1b895279c6a, 0x00003cbc8e178911, 0x00000000000c2900}, {0x0006736dff18cbd8, 0x000579949d6eb7ac, 0x000931ae3b95e19f, 0x0006ee2dd2275c0b, 0x0008c419f1d568de, 0x0002b5a40a9e4fff, 0x00096dbc759ca4a5, 0x0000000000063b09}},
        {{0x000eafd50b503ff0, 0x00058f682715078f, 0x000b7ec2b450662f, 0x0008f55c51916c40, 0x000b61ee081fa46c, 0x00064872a98c3524, 0x000faa5ab7f2c2ad, 0x000000000007e555}, {0x000425ebf2c1673e, 0x0003c802e0d22c20, 0x0004e9f77f229a7f, 0x00019087af20d270, 0x00016be50784b56d, 0x00072b8518d619e2, 0x00076e10fdbb7213, 0x0000000000049355}},
        {{0x000df2627e9f5583, 0x000e073c57fde593, 0x000f9da9fd5339d4, 0x00071f1cfb9821f2, 0x0009cf6a1d68d25c, 0x000d24e649427e20, 0x0008b5bf3c3916ac, 0x00000000000bda7b}, {0x0001a88dfb057584, 0x000e2e7c5f7c7d49, 0x00071c4b5c378478, 0x000c472e66b277de, 0x0007b2816f1e818d, 0x000383c6d4413fd4, 0x00053740f262af48, 0x000000000004f190}},
        {{0x000aa901e2868cf5, 0x00009d97a9b136f1, 0x000f9d37eb7a3b6c, 0x00060c2548188194, 0x00061d59be44c571, 0x000f2b2474dbdeab, 0x000b33b93a9dad81, 0x00000000000ecbca}, {0x000a766b5fedf0a6, 0x000ce5e1ce82823b, 0x000997bad589b344, 0x000273a2d82b0df8, 0x0006ac8ff52fe716, 0x000ea7ccdb017f5f, 0x00074b7563e79970, 0x000000000003f0e6}},
        {{0x000fda8d824654de, 0x00033487edc06864, 0x00024946c59b50d3, 0x0002bea10fdba0ce, 0x00052c3dbf337fc6, 0x0007d2a46c836bbc, 0x000f714fdfb5c34f, 0x00000000000dca0d}, {0x00024bebc7614440, 0x0007730a0767544c, 0x00067fffd43f5d03, 0x000d49dfacc13dc4, 0x000aae1a261ad307, 0x000d98650148d673, 0x000fdee008f95b52, 0x000000000009f558}},
        {{0x000015df21e067c3, 0x0003dd084564b8db, 0x000d0b4c4a73b501, 0x0004bf63bc7206d7, 0x000ffc3d577224bd, 0x000c4924fdd37b4e, 0x00096e8156d6f635, 0x000000000001d1d3}, {0x000ca9d7511fd3e1, 0x0009d977b3b3982e, 0x000fc4c09646cd37, 0x000f9c4ae811d70c, 0x000ca06c7f0de581, 0x00049d87fe5441d5, 0x0009bb0275c92b19, 0x0000000000078046}},
        {{0x000cdb7d87948ad3, 0x0009117cd6af6151, 0x000e60b294f18979, 0x0005b1d788224330, 0x00031b9745257e49, 0x000f3ce21fed338e, 0x000896527743d386, 0x00000000000b515c}, {0x000e2d7dc921023b, 0x000d19fde7038e85, 0x000c1b9c9ba2d99f, 0x000600b36ff74dd6, 0x00048302c3c95d46, 0x00044b5eb8ea74b0, 0x000402d560f570f7, 0x00000000000fe762}},
        {{0x000cf97b63daba80, 0x000eea7f8501424f, 0x000f397f945052f2, 0x0006b8a2ad5053d9, 0x0003fbcebe43c658, 0x000054f1688c09a4, 0x00047fde7f2a4a20, 0x00000000000bbbb1}, {0x00041659e06114fc, 0x00075087e48f29cd, 0x0009df479d6378c0, 0x00002ff7f27cac87, 0x000c37b1f6e18ae8, 0x0006d1deab01a131, 0x000f803f0d4c2e9a, 0x00000000000e7cee}},
    },
    {
        /* digit=81 [{1,2,3,..,}]*([2^324]*G)  */
        {{0x000643c3f46ade04, 0x0007aaa03b473d97, 0x00091126d04a500a, 0x000ef86f37b57bd6, 0x000b2a6371d2b286, 0x00077ccbd95b797e, 0x000a6da489ef8940, 0x000000000008e99c}, {0x0007980fdcec94f4, 0x000845cf9e236463, 0x0000b4681fa85d18, 0x00023b6ba7ba4781, 0x000757de30bf3295, 0x00036ca01d406ab8, 0x000ac5aa10e4a215, 0x00000000000dfa7a}},
        {{0x000146df48d94f2a, 0x000a9762a5eebd68, 0x000728ae94bf3482, 0x00034aeaa4f964b4, 0x000ed6526f010fd5, 0x000d71d72cf46648, 0x0008eabc62a54a18, 0x000000000004f848}, {0x000d9dd392598cb3, 0x000a262c40de70e6, 0x00099451ce04b75f, 0x000d7466ab190976, 0x000b2dd1ad3d3a51, 0x000f88078f07bea4, 0x00035d2a0ec4672e, 0x00000000000664e4}},
        {{0x0002f7cfce391389, 0x000d017c563bd4e7, 0x0006d61076868226, 0x00049ac4cd1bfc15, 0x0000d4e53d3b591d, 0x000079a6ed446551, 0x000eb4ca95ccf6ad, 0x00000000000b5ea6}, {0x00099522fba1cf25, 0x000cd600ccc9ba8e, 0x000c729701bdaf25, 0x000edc569c6c6e81, 0x000a586abcdc5f27, 0x000dbc5b6b3fc0f4, 0x000fe0dbfdf3f985, 0x000000000004ea80}},
        {{0x000e7847d76787ae, 0x00083ee6e7005dda, 0x000646f5e7ea8d16, 0x000df7dfc45111c8, 0x0001875c23c8373a, 0x000039d490c7535b, 0x000a062a7e04f897, 0x000000000009afd1}, {0x000cbb06099a2183, 0x000fb049dd385881, 0x0008a6ac539cc7cc, 0x00057b9925498c29, 0x0002da25daa3dce0, 0x000decb7d7b9c9bd, 0x0006ef6d0b70a3d4, 0x0000000000003df7}},
        {{0x000405f3ffe95388, 0x0004eff54e8bb8ae, 0x000ad0d572466c0c, 0x0005bc4ed1fae2ed, 0x0009ea983db1b226, 0x000851454a5e23c2, 0x0005064af5f55909, 0x0000000000036eb4}, {0x00062710e23e547f, 0x000844c72b2d0dfe, 0x0003b7abb313e4c9, 0x0008ad4e19c9c269, 0x00053e8b8004b1aa, 0x000c747fdb9cfea9, 0x000e4107782d788a, 0x00000000000ce9d9}},
        {{0x000aeca3f9d9a2f2, 0x000f8779905b5deb, 0x00064a28a88ba6a4, 0x000f0aa679096dc3, 0x0000f558913cce9a, 0x00024a2e2fdf04f1, 0x000b3505300cb06f, 0x000000000005d581}, {0x000679a2a74abdaf, 0x000aa855c8de8c32, 0x0000f6e26bb556cb, 0x0002a4197740d667, 0x000744c9360d6761, 0x0005cc8807588a52, 0x000188928ac91032, 0x00000000000acdd3}},
        {{0x000ac6fa45a00390, 0x000933774af3674c, 0x000bcfd34179ae6d, 0x000205ec11875399, 0x00018799c22ede45, 0x000b0842ef3b1144, 0x000d64fd53f681e8, 0x0000000000066d36}, {0x0007de25fd86d3c2, 0x0009ca2c3391c37b, 0x000760d08855c7c1, 0x000fa4a6ab8ab4bf, 0x000bd841a6234df1, 0x0000b189a2b0cd34, 0x000da752441d79b1, 0x00000000000fa025}},
        {{0x0009e3af5a1f4313, 0x0000c3c6c56185fb, 0x00083406c1a4afdd, 0x0009c90ae2fb830d, 0x0003ec31a504b325, 0x0006230bdf7cc6e4, 0x000e3a83bb13c612, 0x00000000000ff1da}, {0x000c261cfaa4903f, 0x000010cc6c1a44f0, 0x00029f13aa21ef15, 0x000c75fe604a58e6, 0x00012491f872d454, 0x000e116a5ceadfd9, 0x0004816c5575da9c, 0x00000000000b24a4}},
    },
    {
        /* digit=82 [{1,2,3,..,}]*([2^328]*G)  */
        {{0x00087efd2e620105, 0x00045c5ad791e85f, 0x000885fe6de88f7b, 0x0009e4998544bc47, 0x0005d273c360673b, 0x000a75a8c934d8f5, 0x000c977fb48d0768, 0x000000000005e0fa}, {0x000ee7923fcfedee, 0x0000dbe01655fb13, 0x0001ed409cde6661, 0x00030de533251391, 0x0001de9f3fc574f8, 0x000e01fa5a724033, 0x0004b496b25206a7, 0x0000000000007eda}},
        {{0x000a5e409b41a55f, 0x000283e6e9380f21, 0x000bd6398225ced2, 0x000c8b4b843ec071, 0x0002af8a23f1f2d7, 0x0007923350055d88, 0x000e74e848010ed3, 0x0000000000055e08}, {0x000839aa213ff6af, 0x0003d956ea8e93ca, 0x000299a04653a35e, 0x0009271cca1b5491, 0x0001f1d0b7f15f7b, 0x0009963fbd41d228, 0x000cd8b187047b5c, 0x0000000000002536}},
        {{0x000a16fe10008718, 0x000fd7138a03f4b5, 0x0008b9afc3d67514, 0x00041fd2a7cbcc17, 0x0005f6073ecba29c, 0x00004fa086014349, 0x00053298b46d3805, 0x00000000000401f9}, {0x000b314f178aa1e6, 0x0009808cc4abc63d, 0x0005190f1545e910, 0x000c8faa75d9af1b, 0x000fc2e5592d6eb2, 0x000468e666f50198, 0x000fc3773b9d55c0, 0x000000000001a2a9}},
        {{0x0004cdf21adad97a, 0x00008e51a956ff23, 0x0002a26288680976, 0x0000fa2f974d2cb8, 0x00074c6a78dec616, 0x000c89d11e00474f, 0x000133ee916e510a, 0x00000000000f826f}, {0x0002a0e3be0ad710, 0x000103fc65d3ecf2, 0x00089ef785c9b19a, 0x000a82e0b9196b29, 0x0000a572b6f55237, 0x000bed2bc4975e68, 0x00025e41ea8b92b9, 0x0000000000098268}},
        {{0x000bc901bf39d0d3, 0x000fae352fb9c164, 0x0000b33959aa5988, 0x000ce6fafcb7de1c, 0x0004f694cad0f9ff, 0x00061d6110e574cf, 0x0005d8adb4671cb7, 0x00000000000c8aa0}, {0x000b2a78a9127846, 0x000a987cd7872df6, 0x00010a813ed5d7a6, 0x000531ec2cb43e23, 0x0004b9e6b89f1c41, 0x000e5d4d262fc310, 0x0001dd7f406a0dc4, 0x00000000000df3f6}},
        {{0x0005300bc3a99814, 0x0008c06f721d246f, 0x0003df2a36c6f8d6, 0x0006b7a5a44ffdab, 0x000915764390c5ca, 0x000747d7dc53e328, 0x0001227f16917459, 0x0000000000061f79}, {0x000965ef8eb0e108, 0x000ae099f4e3a08f, 0x0009b651d265b539, 0x00043576b72f3285, 0x000cc33695bd270a, 0x0002249029d6bb26, 0x00051f60f9202126, 0x00000000000e1b0a}},
        {{0x000bf8353cd33783, 0x00086265d4eff002, 0x00010092c6e845e0, 0x0005c996be6ce254, 0x00096b6056eec399, 0x000a007266fd43e5, 0x000fb1d328c28f28, 0x00000000000f719a}, {0x000f1e2a3462b4c4, 0x000fb75dcabd2f96, 0x0006c939c482a7c5, 0x000a522d764ab7e8, 0x000dcf8a487267c6, 0x000810a449458c96, 0x000786af1af61b78, 0x0000000000073902}},
        {{0x000b7c441c7f3df5, 0x000a88a390dca834, 0x0004f92ea9440c60, 0x000ea69bc5e82f15, 0x000d4628d6f177bb, 0x00068e24071f02e2, 0x0000ee6260790cf9, 0x0000000000066527}, {0x0001df8e5c25d23a, 0x000f233639544be6, 0x000c37cf5bac2371, 0x0000f867807ed557, 0x00019dfa452c6906, 0x000d27c0d1b0ac01, 0x00090cbbdb25fe06, 0x00000000000a60ef}},
    },
    {
        /* digit=83 [{1,2,3,..,}]*([2^332]*G)  */
        {{0x000957058557b81a, 0x000dc1877dfd042b, 0x000fe0adb79e32df, 0x0009536c78e295f6, 0x000374a04cc59d08, 0x0007f75edcbb5d55, 0x00078ce366600682, 0x0000000000061e73}, {0x000dd46741146ae1, 0x0000a74170170ac0, 0x00052de550198353, 0x00030f8ac1c4a0fc, 0x000d3cd92824d75a, 0x0005e86d4d7e7be0, 0x0005ef5ea0abdb1b, 0x00000000000b3b61}},
        {{0x0003f4e13d621f8b, 0x0005a618ca06b4bf, 0x0002813dfa928efe, 0x00072cca3b6ce7e1, 0x000e5f14bb579813, 0x00073abbbb44a68e, 0x000c0afc1167e963, 0x00000000000767d4}, {0x000277deb88a4f9a, 0x00077c6dcdfeb3e6, 0x000664e1fd3b5c4b, 0x00001aabfcbf2f89, 0x000a67aa82774775, 0x000aab9cd0be7228, 0x000b4a5de7b8331d, 0x00000000000262fe}},
        {{0x000cce85ebd04b7f, 0x0007f7a9623b5682, 0x000f8d813f260bd0, 0x000dec9e0f8e530f, 0x00091d65dcd4f8a5, 0x00039541c88bcbe2, 0x00097a09fe380367, 0x00000000000e716a}, {0x000184fc8c1d72cf, 0x0003bbd8e34de924, 0x000a19b51363e270, 0x000d905351941206, 0x0006f7a3c4830183, 0x0003557396345099, 0x0002f718283cea01, 0x000000000009aaa7}},
        {{0x000a56eb624d0cf2, 0x0006a7b7da0287f4, 0x00087c61d1ede60c, 0x000d83c51f3dbd9c, 0x000fa91d085ac056, 0x00035262de2f4d46, 0x00029e31759c9ceb, 0x000000000000c9dd}, {0x0009a41b82bb0c12, 0x000e8108cf9ba936, 0x0004fc5060b7103d, 0x00016b273d372b32, 0x0007d5415e53f698, 0x0006b8b0e126575a, 0x000a5e546bb4c130, 0x000000000004d54e}},
        {{0x0002c932a7c5abec, 0x000a881dd074d9bd, 0x000f76c734b204ea, 0x000e1f0352b58f62, 0x00029c390eefb5af, 0x0006a8d236d2d948, 0x000be5d073f9a0f3, 0x00000000000466eb}, {0x000118cb3dd8d87a, 0x0004d968c04a5c76, 0x00007c8586d1cdac, 0x00096bb15c0970b9, 0x000370ab8dda98e1, 0x000039fa92af28ea, 0x000ff6b912fbda7a, 0x00000000000cbd02}},
        {{0x000e45ea08b7c4d8, 0x0001d5bcf28785c2, 0x000fc63d7c5e8cf8, 0x0005de6707a7138f, 0x000cfb2041a48868, 0x000db3060a77646d, 0x0006908608695289, 0x00000000000e27a1}, {0x0000f0b051870cd1, 0x000794148d936eb2, 0x000a4e50dd50fda5, 0x0002d4db0391d14a, 0x000462b0373f9d72, 0x000a9992c5683112, 0x000b8487b0363be9, 0x000000000008853d}},
        {{0x0002c578f8f683b4, 0x000576c8e5a2524a, 0x0007522f4370ef9a, 0x0004aad50ad6f3a6, 0x00051f107e269b01, 0x00051bf732a3933c, 0x0006df4b04395cc9, 0x00000000000b0ef5}, {0x00017d2e75e44b21, 0x000cd28c9fcdca32, 0x00061a5012b72649, 0x000f58e53ace23e2, 0x000efe688f02b546, 0x00091406dee5fec4, 0x0004d9d8b43f6d2e, 0x0000000000026176}},
        {{0x0002d2ad73a4790a, 0x00034daa7fcc985a, 0x0000276056234a54, 0x00018c59894edceb, 0x000da9ac29a7fbf9, 0x000372f2c470c45e, 0x000d3fdfd44f6fbc, 0x00000000000a1e38}, {0x00009dd8fe96aa5f, 0x0003923d2400aaad, 0x000c441c1c12013d, 0x000488148baa4ac3, 0x000ebc26e6b0c76a, 0x000ef566273227e4, 0x0002bbe732edcfe8, 0x000000000000c566}},
    },
    {
        /* digit=84 [{1,2,3,..,}]*([2^336]*G)  */
        {{0x000d5d97020dd4a2, 0x0005087c7dd7d118, 0x000d6556f04f9110, 0x00048789fff00158, 0x000b909eaf1abcbc, 0x000443a788ffcef8, 0x000fed9905f4eef1, 0x00000000000e15cb}, {0x000831894b4ea50c, 0x0006cca2969ec57b, 0x000ae2b1ba048b54, 0x000fab787ea6e3a2, 0x000ff98c4fe567f7, 0x0009cb91e0d0ee10, 0x000ab4c646879ac1, 0x000000000007d1d7}},
        {{0x00092725c84d0435, 0x00013cfbcbc49b56, 0x000166c387130162, 0x000e05622322f91f, 0x000a7662ae23735d, 0x0002be7a6d2590bd, 0x00017263d468fed8, 0x00000000000695ea}, {0x0005986e2611645f, 0x00050c50f4940fa9, 0x0003729def8820a9, 0x000cd041fd257785, 0x00044a3d0808680c, 0x000684cba25ddac4, 0x00088841d8b738c4, 0x0000000000053963}},
        {{0x0003a8028963f3e3, 0x00076b05cb83426c, 0x00074bc4cf79f053, 0x000b5aaefa9e4f0b, 0x000d9bd075ecf023, 0x000a41bfbb8061e5, 0x000d967d2ba50f1a, 0x0000000000090865}, {0x0004ae64b7cf3e16, 0x000d97f221a788f4, 0x00000b032e3ccff2, 0x000eb6af6ab15418, 0x0004dc87cd93085f, 0x00039dbf6fd14102, 0x0003643e196fdb32, 0x00000000000dbcd5}},
        {{0x000d4ecde2fce0e7, 0x000891466c9f4fd3, 0x000b63a0ee692739, 0x0008c75b58655519, 0x0004e65862cc291e, 0x0009971671ddb715, 0x000c19285153bc2b, 0x00000000000954b6}, {0x00039e1aad688b1d, 0x000e0985d4505697, 0x000559cd349baa64, 0x00098dafa5fe5e6a, 0x0006f90f231d39e0, 0x000afd0b53fc2604, 0x000c18de5d5ced32, 0x0000000000060c09}},
        {{0x000b1d0586854e54, 0x000cf3720b17188f, 0x0005eebb9c78a9a7, 0x0006e315b54eefca, 0x000472b0edaa03c4, 0x00077476e387e466, 0x000259fc59b03dee, 0x00000000000607a7}, {0x000e6b356394b5a7, 0x0008b1478bceddaf, 0x000e3cdb4ff0323d, 0x0002f72cf272ee0a, 0x00038bd885cd7127, 0x0006ba31c19e3a4a, 0x000b369af6415b37, 0x00000000000807b2}},
        {{0x0008b39051ba2cdd, 0x00030dbfcfa7593b, 0x00015a46addd3783, 0x000851d67a19b610, 0x000431ae1a67cc6c, 0x000d1e135ace88bf, 0x00090a74554a6193, 0x0000000000042118}, {0x000bd87496a1468e, 0x000277cb3268907e, 0x000047feb42207f2, 0x0002c21766e72f3c, 0x000a880642e30a67, 0x0004d2b3624bb718, 0x0004ab425dc41d18, 0x0000000000089102}},
        {{0x000caa1ac9c2c636, 0x000673180789b5d1, 0x000d386a756090c3, 0x000b895e99e922ea, 0x00003d0bbe807951, 0x000fe79c0987ea25, 0x000d2a6d2f49f0e2, 0x00000000000c2b18}, {0x000ed8226be989ea, 0x000d8707798f36cd, 0x000ebf8dc541b02a, 0x0001e32bc4479412, 0x000938a845f346cf, 0x0005e5d924dfc145, 0x000c3f210083fe45, 0x00000000000a1fed}},
        {{0x000faea0d2bc54ec, 0x000a972fc8198080, 0x00029736c6c3c3d0, 0x00098bbd99e30373, 0x000a0efddfca3691, 0x0009147416b68390, 0x0008078c35f3e4f0, 0x00000000000ca7d7}, {0x0005a76575b0ee6c, 0x000e9564d991dc4d, 0x0003cf033490be81, 0x0000dab5d635893d, 0x0006f944e49a51f3, 0x000233bc00427e34, 0x00093e1e0bf1b56a, 0x00000000000617bf}},
    },
    {
        /* digit=85 [{1,2,3,..,}]*([2^340]*G)  */
        {{0x000129009060318c, 0x00056d23658c8428, 0x0004596f83a71a5a, 0x00010ec082210f1b, 0x0001bfc364906dab, 0x0004f2d14fb1add9, 0x000443eda8b02f9a, 0x0000000000056560}, {0x0004169ffc0d0413, 0x000986c01479d686, 0x000d5173dbdf44ef, 0x0003c1d718883983, 0x000e13c34fd24dcf, 0x0000c15ac87a9c04, 0x0005d2fe0e08ec51, 0x00000000000c0f49}},
        {{0x00099e97b82e73d1, 0x0005db097b2c50cb, 0x00071fe923d57f3e, 0x00066e9420996453, 0x000eff7290736382, 0x0008c394043d059d, 0x000925eebb5fe114, 0x00000000000daa8e}, {0x000995963e49eb67, 0x00007f43c78d7e6f, 0x000a34a7c03b2d6b, 0x000a7d66b6363576, 0x000be096a954cdd5, 0x000afa2fc8b70a2e, 0x00095a011d5419d0, 0x0000000000004fb0}},
        {{0x00033bb61172fb0c, 0x00008eb05c51603f, 0x0000d435c61a818f, 0x00073cd489576f54, 0x0000535a8d57cfd5, 0x0001436c4e653538, 0x00022d1394731467, 0x000000000005f0a1}, {0x0007992fb047bfc7, 0x00099aecc7ec110e, 0x000e8ab91f3e896f, 0x000f6523d4221aba, 0x000f2851996bdf96, 0x000cfb67efef5649, 0x00005246fb9f26d5, 0x00000000000ef5c4}},
        {{0x00016ebee78befa2, 0x0003188e7ac8d2f0, 0x000b37c50c491499, 0x000d52918be419b3, 0x0004c057621c3b96, 0x000e57e597e75e35, 0x000463cb1b709f5c, 0x00000000000844f2}, {0x00053157dde1a349, 0x0006d53608198549, 0x0000450e27f360c3, 0x000d348e04114157, 0x00004d73eeae6dd0, 0x0000ed85e5b28e95, 0x000b580843923269, 0x000000000006da14}},
        {{0x000ce0f0a89b6284, 0x000735cbdc4424ba, 0x0003b0cbdca1bfb5, 0x0006886c118d02b1, 0x000675aaf27658a2, 0x0005f6ec187ff74e, 0x000d32b133be95f6, 0x000000000004b1b8}, {0x000fda7aaf38358a, 0x0001bdd8a8ef25a8, 0x000eba5b65bc2a4f, 0x000a6db26623fb02, 0x0000bae15453d525, 0x000f89fc9f2368d4, 0x000a3e55d6a84d84, 0x0000000000086021}},
        {{0x0009f5565cd739fd, 0x000f210520b26d6e, 0x0004ac2940ef5ddb, 0x0008ef88948f78bf, 0x000550ad452e1246, 0x000c4847040a9d27, 0x000ace5e382347c1, 0x0000000000033e73}, {0x000c8b4278956db1, 0x000437d83d02e97a, 0x000af99c625baa1b, 0x000d448885679e6a, 0x0005fc95919c6716, 0x000c2194a0e12bdd, 0x000d2d7da4420492, 0x0000000000015ffc}},
        {{0x000c11f07976d090, 0x00025bd048bdc85d, 0x0003c8a142cbe0e8, 0x000a758a985ac100, 0x000cf2c7ace940aa, 0x000ec347d6039bcd, 0x000ab7712092cc6a, 0x000000000006b5ac}, {0x0003db66ec59bab1, 0x000551ebcf80c829, 0x000fffebfb9d4dbf, 0x0003d1ad0b610f09, 0x000498c28ac73fdd, 0x00059750fdd3f6f9, 0x0001ac650b77943a, 0x000000000002d399}},
        {{0x0001cfefd0dbceda, 0x000826cbd2756691, 0x000e925943cf3919, 0x000aa0be4c58c7c2, 0x000f0e488784177e, 0x000916b0f603551e, 0x0002e2c8eba131df, 0x0000000000015973}, {0x00008eec1292bc0e, 0x0000355acca849df, 0x000e7c404ec832b8, 0x0002c703bd3b202e, 0x00056ddd8eba162a, 0x000b1d93d4c5e5d2, 0x000dcf66e7844a77, 0x00000000000110b9}},
    },
    {
        /* digit=86 [{1,2,3,..,}]*([2^344]*G)  */
        {{0x0003ab751954f075, 0x000f91b66faabe09, 0x0001714e51539902, 0x000f3a0a675f7c8e, 0x000f30313a711a82, 0x000aea9e682884a2, 0x00005d7ac5d7b058, 0x00000000000cd5ff}, {0x000d5d715b7b74ff, 0x000287c29638d05f, 0x0006736db974b38e, 0x0003c47a17ae3a7c, 0x00077009e38ae85e, 0x000f9c52e91b107c, 0x0008a0f3b777d8f1, 0x0000000000011b68}},
        {{0x00072d048b012b70, 0x0009a4ae3d232353, 0x000ebed55756fa98, 0x000c769ec62fd6b0, 0x000f62a4720cba73, 0x000c1f491d586ba7, 0x0005716497cd140c, 0x000000000007b2ac}, {0x0007af894008277c, 0x000b9a65eabb5e68, 0x000cb737865439ae, 0x0001b84231457d7c, 0x0005901645c525b3, 0x000f7b656cab62f7, 0x00095c2377d74db2, 0x000000000002d33c}},
        {{0x000e54e4ebfecf4f, 0x000b310105dfc241, 0x0000ded90576b5a5, 0x00068324e80fc085, 0x000287dc7da6122e, 0x0004728cf3b26e76, 0x0001297bc183de6d, 0x000000000008bcdc}, {0x000bafda0190b71b, 0x0005d8b995cec24c, 0x00099629f2c641a0, 0x0009e6b4da5b77d0, 0x00096caf9161612e, 0x000f45eb68ec048a, 0x0000e9e3628ee2c7, 0x00000000000d8565}},
        {{0x000dc37cf3c258c7, 0x0005aaae2f447f93, 0x000c6f7663c30e3d, 0x000b2f482c1f372f, 0x000d351a5f7f3262, 0x0006c75a85521feb, 0x000fb8f8ec919091, 0x000000000002728d}, {0x000f483180d24d43, 0x0005f5dd4ff4f0a1, 0x0000b042bcddd9b8, 0x0005b98ba8b777b7, 0x0006fb7f8409318b, 0x000fd31dbd971d42, 0x000347ed1465e8b0, 0x0000000000000f66}},
        {{0x000f5311360afd40, 0x000bacea0374a33b, 0x000feb5ded889fab, 0x00002361bb01f474, 0x000a8c328b8bf6ce, 0x00053d6302a5b28b, 0x0005d86991b1d8b2, 0x0000000000085945}, {0x0008eeb3e3866a93, 0x00024f5c6e141989, 0x000231ed9c304f2e, 0x000032f67e76ece0, 0x0002338980594eb4, 0x000001b765bf14aa, 0x0001340804a7c00e, 0x00000000000734cd}},
        {{0x00036d45a9f2195d, 0x0007e5ba5288b70f, 0x000e413923c56371, 0x000a997602f3c65a, 0x000fe08c0223c1a9, 0x0007e6f1dc5c7512, 0x00059748a19c3c36, 0x00000000000fb241}, {0x0001d161a9f86145, 0x000879f674a5f0bc, 0x0001526754b42988, 0x0008c4303d6f13ed, 0x000d917433fb5aeb, 0x000f534900fed575, 0x000616e4a5ef9a59, 0x00000000000f315e}},
        {{0x0008596b7b07e015, 0x0009c7059c585ee9, 0x000cfabffa6395f2, 0x000d9318d9633cd0, 0x000f37bda14896dc, 0x0008964dcb2abc44, 0x00076ce31adb3feb, 0x00000000000e3168}, {0x000ed4ab82ecfd95, 0x000363173288028b, 0x000f24a3fd6c4552, 0x00034c91d6f69add, 0x00022c34e118b5b3, 0x0005984ede613e56, 0x0003a18bbdbf5c5f, 0x00000000000cf4f6}},
        {{0x0008f4401a1165b4, 0x000b4315ad7a4644, 0x000672c1b06e4df0, 0x00097a8baa564733, 0x000446edcfcbe12f, 0x0001a968ef263db4, 0x00087547cf91d53c, 0x00000000000c15db}, {0x0007f5c515f16ba5, 0x000345d35e53a1e8, 0x000a90f359724b01, 0x00011ea246da3d37, 0x000653f068205d3d, 0x00010c00fc1ddfcc, 0x0008c78169d71166, 0x000000000004bddc}},
    },
    {
        /* digit=87 [{1,2,3,..,}]*([2^348]*G)  */
        {{0x0005418902e018d8, 0x000328002f4583b6, 0x00029b160f5eca39, 0x000d112fb93f735b, 0x0002196e3084a8ce, 0x000e8c74427cb629, 0x0005ff72395bdd77, 0x00000000000ee71f}, {0x00030cc165e06d5c, 0x000ca7cfc14d95b5, 0x000ac1b9673d9545, 0x0000129d213738ef, 0x0001bc0b5ea366e5, 0x000a067007a905d9, 0x00082192cb630afc, 0x00000000000bf3a0}},
        {{0x0009c76f27cbedd6, 0x00086e96c4aebbe2, 0x00087447d6551831, 0x000d2f632f9151d3, 0x0004302b99e2f86a, 0x000fd317105daf87, 0x000c624299dbfa14, 0x00000000000812cf}, {0x000ccd383b8a542c, 0x0001b42e615367ce, 0x000e792323a5de78, 0x0006c70548fffa38, 0x00077b6db825c34c, 0x000f2989e1fbed77, 0x0003ee850bded44e, 0x0000000000081546}},
        {{0x0008bd3d7d2dac19, 0x000ce352e14371a1, 0x000574a96d5757aa, 0x000d9395a7b7719e, 0x000b8544328b64d5, 0x000f9c5934d5197b, 0x00045e5220626522, 0x0000000000074d6a}, {0x000ed277c567a2e7, 0x00003f52c9eeb86b, 0x00037cef0ab9cbea, 0x000b3bfc9ed39349, 0x0007a14c3d70e606, 0x000db5876fc5046d, 0x0007a181cd053e5e, 0x00000000000a3034}},
        {{0x00026a12e217941d, 0x0008a9decf2164e6, 0x000546598e5e9913, 0x0009ce8aeb36b93a, 0x000158fb4dc8d564, 0x0002d60cfab9f77d, 0x00061966b11fb6b5, 0x00000000000eaaa4}, {0x00098700891e3d12, 0x000dbf3522a998ce, 0x00034cf7624bc215, 0x00097c625e387237, 0x00072d4595ee2679, 0x000e5456ced5047b, 0x0006feaaa41e5f55, 0x0000000000078cfc}},
        {{0x000abf51b7538110, 0x000353fd75579f7b, 0x000019e5c13ce4a6, 0x00076854d208bb77, 0x000ac1c9512f4c82, 0x00081e9f3941aeb9, 0x0003396ce0bed5a1, 0x000000000003744a}, {0x000bd7c923f4230a, 0x0006180eaff7a041, 0x000dbb73984381c7, 0x0007c1b0f8e7fd8b, 0x000aaf499630f16e, 0x00080a16856bdddd, 0x00012be8c112df11, 0x000000000002987f}},
        {{0x000b02027989cfc0, 0x000a60ce9ab12f61, 0x000973d1a5ee6cd8, 0x000b3b69a1753a9f, 0x00002bae5685f031, 0x000d3c06632160ba, 0x000f5cdfde9ae80c, 0x00000000000a180c}, {0x000f0a3aff152330, 0x00023e2c194158b5, 0x000e1481b10c0c49, 0x000e7d12ea20322e, 0x00007968c7ff67ea, 0x00027bab93eb507b, 0x0006eaeb300ff9ce, 0x0000000000097575}},
        {{0x000b71b2ec924484, 0x00007325de655ef5, 0x00027d8f4ae5da5e, 0x00026ad89e4c34bb, 0x0000ec5a615fa909, 0x000c770ac8e61adb, 0x000f8a5a2233d43f, 0x00000000000cbb23}, {0x0003b01804f61225, 0x000a4a7e6c5861cf, 0x0004fcae81492249, 0x00040be697e7dc09, 0x000eb6f29135b0f3, 0x000e89899783a7d0, 0x000f55412469b007, 0x000000000001bfaa}},
        {{0x0003c3de6d71b673, 0x000cca438634e69c, 0x000f00cb40863203, 0x000dedfac40dd56f, 0x000e4956a5de2ab4, 0x00013f84d758e95e, 0x00016cfc11e39451, 0x000000000006059f}, {0x0008f87a862c83c5, 0x00038fd8e2236750, 0x000df4ebd7b9092c, 0x000b1538ea13b455, 0x0003013d382702ae, 0x0009ca201de1275d, 0x000351470a7b7e65, 0x000000000005c77b}},
    },
    {
        /* digit=88 [{1,2,3,..,}]*([2^352]*G)  */
        {{0x000bd58c2fd385c4, 0x000d8281f6e58982, 0x000c3afee2ff7056, 0x000a41afd89abd8e, 0x00007984feefe29f, 0x000d20a64fcb0b0b, 0x000cd50b0928a6d9, 0x000000000006979c}, {0x0008ab27cda5c7bd, 0x0001cce9c34c7521, 0x0006dc0b027875db, 0x000635250946a5c3, 0x0006f39f53b9d464, 0x000bc8b64b09a97c, 0x0000d61d47bc20dc, 0x00000000000d458b}},
        {{0x0004b16a9cc79c48, 0x00075763e36638c9, 0x000ff772bf788245, 0x00011a8c66b40e9f, 0x0008f384b70862ab, 0x0001978760624469, 0x000837e7cdf3bd69, 0x00000000000064f8}, {0x000fbc5f9be69c3f, 0x0007895900a21f89, 0x00053a9326cbd6b2, 0x000fbbddd3e6e471, 0x0006baa2e2a03f65, 0x00085282484f52e0, 0x00032c1dc462a8e3, 0x000000000007e4c0}},
        {{0x0008c150b5ec2626, 0x00094cc580ea6853, 0x000526b13f535e43, 0x000604fb23480cfc, 0x000a344146898665, 0x00010a94595787cb, 0x000c78425d7c6f4d, 0x00000000000b2f1c}, {0x000940b59f5f9db6, 0x000f455da8884e6f, 0x000468b788890b3d, 0x00081d7d99e417f5, 0x000abf28fba2c648, 0x0004eff801eeba5a, 0x000fb720feb7b350, 0x0000000000050deb}},
        {{0x0002dafb4e000df5, 0x0009720ebf79c9aa, 0x000a041b02faa426, 0x00007e78d630d2b3, 0x0007fa605dcb016c, 0x000d0470520021d5, 0x0007e66190f3e942, 0x000000000008974b}, {0x000bb7ed0e0135bf, 0x0003b6710da6c4cb, 0x00011bda556d9709, 0x0004b8ba583089a0, 0x0004ea7dbcdd192c, 0x000df1097171ab8c, 0x0000ed715f60818c, 0x00000000000205d1}},
        {{0x000c0f09863d151a, 0x0004b4a6226f970c, 0x0004a88f8872d167, 0x0002e60a1193cac8, 0x000543dda270b44e, 0x000d647382ce6393, 0x0006751a9e8a2138, 0x000000000009d843}, {0x000abb28a3b6891b, 0x0008a98a1222e3ef, 0x000341bb8ccbdd0d, 0x0005be5555088026, 0x00017b38f0648047, 0x000e249f5c39ccd9, 0x000b74ea31304de8, 0x000000000004d42d}},
        {{0x000330dc4e7217de, 0x0009c39a689bbd9a, 0x00001ce7a86200c3, 0x000108a8d29457b6, 0x000014ed4b4dcf33, 0x00015625f312612e, 0x00063bcbb21f3451, 0x00000000000303a3}, {0x0009f7756d9dff06, 0x00004aeb0c8c0639, 0x000d964999e9958d, 0x0002604683b37dcb, 0x0007862b08477a02, 0x000d69390fa4ced7, 0x0003145a49bdc136, 0x00000000000c0215}},
        {{0x000155abc265f191, 0x000f0544874b6521, 0x0007a036ebfb6d29, 0x000afd631411ba1e, 0x0000466d6415b303, 0x000d4c2de30047aa, 0x0007b13a1b676594, 0x000000000002860c}, {0x0008a20d9745d768, 0x00046fe140a72d14, 0x0007d03c948ac2bf, 0x0006df9f46f144f8, 0x000bc0defbc46c9f, 0x00024c075f7e7b95, 0x000c39fbc9a96978, 0x00000000000d7773}},
        {{0x000f0ec4d363b0b7, 0x000bdd2db4d34f56, 0x0005740dc154bacd, 0x00065cc723a57c02, 0x0007d8ded62c4475, 0x0002f1a9ed98c359, 0x0003b20aeac6d9de, 0x000000000000a98d}, {0x0004a99c9e88ef97, 0x00034d9708f06642, 0x0001bd570d037e70, 0x00032cf49cda0113, 0x000a858467e24993, 0x000748e8e546df74, 0x0000738b84a55093, 0x0000000000006f09}},
    },
    {
        /* digit=89 [{1,2,3,..,}]*([2^356]*G)  */
        {{0x000f37fe6b03f897, 0x00052c0a40cabe70, 0x00091f1b94fec55b, 0x000898c340187426, 0x000636d9e57a8625, 0x0008aa21169c9f3f, 0x00016869d9d7f337, 0x00000000000e7dca}, {0x000214154225907a, 0x000ba24d49aebb77, 0x000fbddce2036f3c, 0x0001d01576f533f5, 0x00099ef82ece667c, 0x00007c14d372eee1, 0x0007f6577723c0c2, 0x000000000000eed3}},
        {{0x000850ebff25b818, 0x00013c61db976bce, 0x000a1c9cf36381eb, 0x000b8ca060b1adcb, 0x000188e2c178ce89, 0x0003bdd3c41db448, 0x00042cba6339f392, 0x00000000000d29db}, {0x000bbd1437baf649, 0x000f53521116eaf1, 0x0005d6c8cce9ea0c, 0x000c984bd79fe5d8, 0x000e9d45eee49357, 0x00088e01f2eda73b, 0x0008731a50c59f62, 0x0000000000075018}},
        {{0x00038b1bc1ced6c1, 0x0004df323953ab33, 0x000a2512b2fa2401, 0x000dfd9bd32cb8d7, 0x00037ecf35382937, 0x00039941507c4e56, 0x000a06b573960eb6, 0x00000000000dd1a4}, {0x000a33b92253abc1, 0x00050c625f400562, 0x000bec6e4c3f3a23, 0x0001e5b6220f24df, 0x000827d01c6f66c5, 0x00012372d9f97d75, 0x00004860b1572404, 0x0000000000004d55}},
        {{0x000a3b570044f6d8, 0x0005fc552c6c6093, 0x000f9e99da8c0559, 0x000375c19610b0cd, 0x000bb051dad5c9ed, 0x000556643f0e7b4e, 0x0001d87267a304c6, 0x00000000000c96dc}, {0x000cd649696f60dc, 0x000dbf0ed5f9a8b8, 0x00051f0009075842, 0x00066f4af5a4b4c2, 0x000d20644cf2ef5d, 0x000ae23c00b5bed3, 0x000b66f7e4543a75, 0x0000000000041325}},
        {{0x0006798502ee1353, 0x0006c04ef7ad5a7b, 0x000c6878548d78d6, 0x0008a6a47591d88b, 0x00046902edb49ae3, 0x0002b143a27d9125, 0x000df65dae3d8381, 0x0000000000093a2f}, {0x00091ef1ecb7e486, 0x0000807a00388858, 0x000fb3b7ca4398f1, 0x000c17172b1bcba8, 0x00032d0033c73fd6, 0x00016c28aa8d70f9, 0x000ad79ea9eea329, 0x00000000000423ed}},
        {{0x000a6b031af68d65, 0x000b59949b33d112, 0x00063134f5d066ef, 0x00071e788e17f2bf, 0x0002da9c088188ce, 0x000d8c9e9d851baf, 0x0005b6e869c5ba86, 0x00000000000a0142}, {0x0009ab9cc7c38dc7, 0x000eceae8b1c5d3f, 0x000b0c8e79b3cc01, 0x0002910c374bb97b, 0x0007054874831494, 0x000bc6ee13e13c45, 0x00047be0e6fad435, 0x00000000000b54d2}},
        {{0x000e68e47f9ea217, 0x0003c9c85e50a3a1, 0x000b0543d8520966, 0x000ce81807f33dba, 0x000a3db81e06cb78, 0x000638d709d337d8, 0x000babe223eae472, 0x000000000006bf2e}, {0x0006763908bcca20, 0x000662804b59c92a, 0x000900c614fc9957, 0x000ca1e7bc6949c1, 0x00008f051155321e, 0x000539f40bf2906c, 0x000808b802a3289c, 0x00000000000073cd}},
        {{0x000fc3dc90267bca, 0x000956d34bdf61b1, 0x000bd7a38090ba35, 0x00051135a3bfdd10, 0x0005f1296bdd61d0, 0x0002e9a7c4abab57, 0x00003d72da68494d, 0x000000000008d11f}, {0x000319b6e5281755, 0x0002247811121597, 0x000526a929004138, 0x0003ec67521898f7, 0x000996c1da75ec29, 0x000f2c7b3f0cf026, 0x000af8b0dbd4c380, 0x0000000000034e4b}},
    },
    {
        /* digit=90 [{1,2,3,..,}]*([2^360]*G)  */
        {{0x000c4e9ab57885e4, 0x000f3c32730e8279, 0x00033a83277668a3, 0x000fdbf3f5cd8478, 0x00067272d4dd2bac, 0x000b7577645f3641, 0x000be2dd813a795e, 0x0000000000050997}, {0x0005df7d4526b8ac, 0x00040da054e2b966, 0x00068890ccefdc5e, 0x0008f31026116a38, 0x0000dcd85e914c42, 0x0003adc2d372af9a, 0x0009e6fda6e90367, 0x00000000000a8db5}},
        {{0x00029349f348fe26, 0x0004251a033a4db8, 0x000e0f2d8a80c6e4, 0x000ce49a42a266f1, 0x000b82c4d11e92ee, 0x000bae08a87e037d, 0x00078c875be1416f, 0x00000000000c5394}, {0x000064a55c345b79, 0x0001651f8b907a36, 0x0000a11c59759db6, 0x000bd666c51b9c32, 0x0004ee565de82c74, 0x00082c8c3635d3d2, 0x00093c6bd6566389, 0x00000000000daf6a}},
        {{0x0007ac96ab52d7dc, 0x000dd68fe359d36e, 0x000f3dea99f83698, 0x00052c3fc704935e, 0x000952e4e22d0ec6, 0x0003e3ada1eeff1f, 0x000871ba6777cb08, 0x000000000008befc}, {0x0007324fc4458b0a, 0x000e34ede689e853, 0x000cf3cdc3eb9d08, 0x00080517ab83c288, 0x00052c8c1f48e0f0, 0x000241aac1f3d5f2, 0x0007e057991607af, 0x00000000000b8a33}},
        {{0x000ca2047d1f8f40, 0x00004179c59cea3e, 0x000880cb97b10e4b, 0x0005cfdba0cf3857, 0x000b3a8fbafadda7, 0x000384e0b94081a5, 0x000a45b91de90a33, 0x0000000000027aa2}, {0x00058a31cd3d8717, 0x000220747bcc094c, 0x0000191551c0619d, 0x000e3e9722484f1a, 0x000da62d1dda9c1c, 0x000963bce13a7ee5, 0x00021b77fd79343a, 0x0000000000083d2f}},
        {{0x0009d92935034280, 0x0008b3dbb3fa61cd, 0x000a3f5ecd8961d8, 0x0006574793ce8041, 0x000783dbce4f35ac, 0x0003cf6b0b2697dc, 0x000164e35c5bf2e1, 0x00000000000b0c4a}, {0x0008df1996e7f4a8, 0x00038aa49090842d, 0x0003b655f6623523, 0x000e96c54d504e4c, 0x0001b73310a3f646, 0x000bde5dad74e754, 0x000ef7ead7159618, 0x00000000000aa09a}},
        {{0x0005ac9c05f620d2, 0x000df609deb16279, 0x000a25447b2b8795, 0x0003132b378305ce, 0x000869275f5e4a9f, 0x000b089b5f101799, 0x0000c514be746761, 0x000000000000c81b}, {0x000e2dd3d7fa135b, 0x00030ca90a9bf94b, 0x0006097de4edabc4, 0x000ac9620c71989d, 0x000a390aedd01b25, 0x000d8cd39b971b61, 0x00011a995214d779, 0x0000000000065c6e}},
        {{0x0003855cea5ad9ec, 0x000d4a8393a23731, 0x0006e8588fe37236, 0x0004e0255a202691, 0x0002a446bcc6d882, 0x00051499b9dd9a27, 0x000535c929cded53, 0x00000000000cfe76}, {0x000bc0f428a70f0d, 0x0009626f3d7d9b38, 0x000a6acd906c92a5, 0x000cc06759e6ddb7, 0x0004e302b89191fd, 0x00007e1f7ac2e190, 0x000b4aad25c645c2, 0x00000000000ab3de}},
        {{0x000338669b2cb8c7, 0x00071a13f19b3741, 0x00031ac1789cc21d, 0x0000c997044a6f4f, 0x0008dca1075c00ec, 0x00020ab0caf5665f, 0x000effded5ca3f06, 0x00000000000a896d}, {0x000378285debab1d, 0x000a5032ab2b2a9f, 0x000438bbbc4fc5ea, 0x000b6f725133304f, 0x0001977a45672286, 0x000abae4f0d16d53, 0x00003b00a66036f1, 0x0000000000095f01}},
    },
    {
        /* digit=91 [{1,2,3,..,}]*([2^364]*G)  */
        {{0x000925dc8a89e47c, 0x00066f2cdae310b9, 0x0002591b317d8ec6, 0x000f25ad750b29bd, 0x0005dba15f3e9d14, 0x0009fe2dd931b9c6, 0x00043334db169ebf, 0x0000000000052663}, {0x000c577396a26bac, 0x000968859a4f76bc, 0x0005596f973fb63a, 0x0008d6a67ac4db88, 0x0002be25d7557463, 0x000db10ee5c8ea7b, 0x000cd38a0f0a8738, 0x00000000000e890b}},
        {{0x000bf56a7de9f2ff, 0x00022883533aee6b, 0x0008916bc1e75ddf, 0x0005fb546d0fcf42, 0x000a4ab05da78f0b, 0x0007b66147e6b0a0, 0x000cf2cd91a869c6, 0x000000000006c6f7}, {0x000593da575680e9, 0x000d7c072a1067a9, 0x0006a216ef99f7a9, 0x000192c8f0ce9469, 0x000fd11494417378, 0x000ccf45adf2e7b9, 0x0005684fa17cb614, 0x0000000000045f03}},
        {{0x000725d8721dd338, 0x000930f4ca77fc15, 0x000b6681d170fa2e, 0x000625b4ef805afc, 0x000fc015c5864873, 0x0001a8c93bddba22, 0x000f427091b50aa4, 0x000000000004c005}, {0x00044f31df11a4ec, 0x0009ff43d3745415, 0x00007d532af636b7, 0x000a10c82770690b, 0x000b77513f003efc, 0x000c3af9bd0c25d7, 0x000adac0015cc12d, 0x0000000000094c6c}},
        {{0x000e8ca221c6ffc6, 0x0008d8b70e94f6d7, 0x000aee261a6327ec, 0x00019ba4d3ef22f4, 0x000a1999f948e222, 0x000fe73027f8a712, 0x000af025575e96ab, 0x00000000000564a1}, {0x000b25cff3d9db0b, 0x000b1045e9c658aa, 0x0003561802f8c969, 0x000b825c3db161be, 0x000aba33a906dd23, 0x0000e41f205fb173, 0x000d1d9a8b5ecb87, 0x00000000000ccc30}},
        {{0x0002aea737bc7de5, 0x000c2bed7d94c6a0, 0x00033bc161f07397, 0x0006af6106834bff, 0x00044c4f74486541, 0x0001dfc06dade8a1, 0x0007bc62227a1d88, 0x000000000008dc2b}, {0x000d41c50320c146, 0x0001c4f1154170fa, 0x0005645ed0c5caa2, 0x00053aba0f1b9617, 0x000d48b0ec8e98cc, 0x00072d0342f68011, 0x000e384bbc34679c, 0x000000000009c576}},
        {{0x000919b377dae55c, 0x000a5d9243926709, 0x000605401369e5f3, 0x000feee0e35f201a, 0x0003196f23f3a1bc, 0x000c1d068d25be5f, 0x00050b298c67f2ff, 0x000000000000c3d9}, {0x000d303412559962, 0x000fa5e15ab9975c, 0x000533c7cf964ba3, 0x0002ec947a7e2696, 0x00022ac22e49a9f8, 0x00006f090f02af9a, 0x000cb0dfb3103fa7, 0x000000000003cf8d}},
        {{0x000298e2791b8ead, 0x000259f5f44f9ed5, 0x000ce416b16e73bb, 0x000f258f9627f9cc, 0x000938fd51bdd7e8, 0x0006a4a7922499dc, 0x000250a9d7497c84, 0x00000000000026ae}, {0x000e3b88b9c7c3db, 0x0009084d47d19214, 0x000f52469be04308, 0x000138806a52e316, 0x00027f08953f9b2f, 0x000fcac083fc8da0, 0x0003fc22d074d292, 0x00000000000701d7}},
        {{0x000b913b1d418177, 0x000cbb7b856256b6, 0x000f3717023b8633, 0x000ddd5c91b8ffd9, 0x000155d38511b4eb, 0x0000da525ef815ae, 0x000bbd98587d551b, 0x0000000000034a9e}, {0x0005ef8ce15a684b, 0x000b8844811fa3d0, 0x000d70ffbd583fff, 0x0008bb4f623789be, 0x000e404980584b26, 0x0002b435ab26ae5b, 0x000eb3b47a5ded3c, 0x000000000008fcec}},
    },
    {
        /* digit=92 [{1,2,3,..,}]*([2^368]*G)  */
        {{0x00069e40601a7dfa, 0x000cb70765682efa, 0x0003c3be2bc6c542, 0x0005ef0a6db6169b, 0x00012032d5992a93, 0x0006f13160029276, 0x000f51b5babb2d3a, 0x00000000000db26a}, {0x00039e460e4f3b3d, 0x0000200c3401304d, 0x0003ff9e293a0443, 0x000e2ed0f9b36565, 0x000934031d18a1bb, 0x000fe1224e17a5d4, 0x000cf5e3661047f8, 0x00000000000623c6}},
        {{0x000ad46943acea12, 0x000a859367582797, 0x00066c45ce5e5faf, 0x000351af6a27741d, 0x00087f929e0d5d96, 0x0003f1afbeab94db, 0x0001dd865ba01104, 0x000000000005fb63}, {0x00040d1f1e090717, 0x00038192e065294f, 0x0002fb37089941d8, 0x000228db7cae5f66, 0x000d6a828b037bcc, 0x000f301aa02eaecc, 0x000a077c48a2ea91, 0x00000000000f5eb1}},
        {{0x0000bb651b864105, 0x000154dd3bd8408c, 0x0001daf9340de0f6, 0x0006998ba0668966, 0x000cad713bbd1ad4, 0x000ca4fa5b7c679a, 0x000a1cef9389aff7, 0x00000000000d68b6}, {0x0009051030865994, 0x000093acfff601ae, 0x000275b28bea6bfe, 0x0006441c66ad734b, 0x00042fd3eb165e2f, 0x0004e211749f144c, 0x0001a3bd7f22082e, 0x0000000000041791}},
        {{0x0002b76266c69181, 0x0004edf0ae8df2d5, 0x000906801183c79d, 0x00017ca1dd286917, 0x0009709b678ede37, 0x0000acdc60c1db87, 0x00003a0288959a40, 0x000000000005ca2d}, {0x000fd72975fb9eb5, 0x00084e52534b365b, 0x000f103241b3e4c6, 0x00034f873eece315, 0x000d642a67991a9a, 0x000a7e8a80fb0c7b, 0x000238375cc0cfea, 0x00000000000d00ec}},
        {{0x0005a38c81bac32b, 0x000bdb67d88fb498, 0x000506df3f19aea7, 0x000433ede7910791, 0x000a20ec085cc26e, 0x0000eefb30ed8fc0, 0x0008c22684a901b3, 0x00000000000e855e}, {0x0003801582a535c0, 0x000a23d31d6c91c3, 0x00014e8637446682, 0x000380e755f1af17, 0x000a6c985ecfd938, 0x00065084e6a8e434, 0x000ec653bec9c1cb, 0x0000000000070309}},
        {{0x000a18c1c3a6f1f6, 0x000f49a2461c2364, 0x0005b210149e5bc7, 0x000bfea0ae15fd0a, 0x00027ac4d98f2265, 0x000831902d817b35, 0x0001fe9788aa4511, 0x00000000000d4616}, {0x0004b65395df8642, 0x0003d2a7cac2f471, 0x0003af0e90c12844, 0x00048d211070e2f3, 0x000ab769926571a6, 0x000aea1556050d8b, 0x000411cf8f24a040, 0x000000000003522b}},
        {{0x000a6d728de1a709, 0x0002e45d4ed63b83, 0x000803bc165cd24c, 0x00029022b267f598, 0x000fc8446afe76be, 0x0004d6f50791c22a, 0x00077ed6ce8b8859, 0x00000000000e3f1f}, {0x0000a258fb32c514, 0x0006d1a72b16f0c7, 0x0009dc6ac4083dba, 0x000743e1cad2e785, 0x00041e0e640d6a1c, 0x00074648529ff0a5, 0x0001fb4cd519be4b, 0x000000000004584f}},
        {{0x00051eecfb6439ff, 0x000619ca8cc9cbbc, 0x0007f10adb0b792f, 0x0005d53059b2bbb3, 0x0000730e5dc37211, 0x0001d89988fe7ac3, 0x0007a54e67ef6984, 0x00000000000c8ed3}, {0x00004a045edf3ac0, 0x0005e48164b3dc80, 0x0006b3cd6d953b04, 0x0004643beed38ce9, 0x000fa3e30b3db7fd, 0x000ddd484993cfa9, 0x00075ddcc5e7af01, 0x0000000000068401}},
    },
    {
        /* digit=93 [{1,2,3,..,}]*([2^372]*G)  */
        {{0x0007c9efed4b7222, 0x000ab79768e9e648, 0x000c0eb72891fdba, 0x000821bbe0c67d8d, 0x000a6909fdfcc194, 0x000ccdda7537a6a4, 0x00026cae6d705195, 0x0000000000092b39}, {0x0004e2be194fc116, 0x000ab2c5dd6af51d, 0x000bd2ad4ffd8821, 0x0007388397a3bd3a, 0x000a8baf59c2dd3c, 0x0000ec9589d176c6, 0x000315d5c6219e38, 0x0000000000054e8e}},
        {{0x00045f27ccc6232a, 0x000560683349d797, 0x0004a272721dca0a, 0x000181050663b9f6, 0x00099733dac647c7, 0x0002137c5f327d14, 0x0001c703f55e4a29, 0x00000000000da59e}, {0x000b58d5c8ca43e3, 0x000c7baa8b4dce4f, 0x000e015119c73eca, 0x0008c1db0322dfa0, 0x0007fd69954f99a8, 0x000a29889394440a, 0x00064a9060473da5, 0x000000000002f048}},
        {{0x000df3b38c2a0a79, 0x0000190762c1ecbf, 0x000c4019f47729e5, 0x000849f5c47c645a, 0x00056d72e7487420, 0x000e11773b082319, 0x00050eaeac683b8e, 0x00000000000fb5c5}, {0x000722990680a985, 0x0003e6d986d69658, 0x000da73ac5c10442, 0x000136fd94d52235, 0x0009ed5b01c6d13e, 0x0008fcdcc1aa7541, 0x0005ee7c2014b140, 0x00000000000cffde}},
        {{0x0009c28bf944531a, 0x000bbc7d0abb1f4a, 0x000c2054201bd117, 0x000defca80767df6, 0x000871820908eb67, 0x00074d19a6fa174d, 0x00081a3215df4856, 0x00000000000960a0}, {0x000602066fc6cc5c, 0x0006993ed71807f6, 0x0002dfba1c3883c8, 0x000d56efe787e592, 0x00098fe8b96f0909, 0x000ab68115ae74fc, 0x000e0c8fc342c435, 0x000000000000b991}},
        {{0x00047f2399bf97b6, 0x00021c600dd569b8, 0x000917d84fb94235, 0x0000a27de43c1d70, 0x000e616bdb162d7b, 0x00029795321bf3a9, 0x0008b233ce767c96, 0x000000000009e439}, {0x000800714c3c1216, 0x000934069cced410, 0x000f1cffed0b3d13, 0x000bf89c53942369, 0x0009acb24e389981, 0x000da473507d9c07, 0x000cd9d2e5e904a2, 0x000000000002f1b6}},
        {{0x000ee38595e6d6e3, 0x0003bfeecc6e7a5f, 0x00087ab2bdd62b28, 0x0004e20e81879cac, 0x00089c4a27124845, 0x0000749c3d0ea9cb, 0x000f93ba8b6f3354, 0x00000000000e971d}, {0x0002b66fa9a77846, 0x000b64c2a1b129de, 0x0005071339da6099, 0x0000b53435739873, 0x000a57040c00e6c6, 0x00091a7b4b9f56a6, 0x0002466de6489f62, 0x0000000000053bc3}},
        {{0x000554330abff5cf, 0x000ae828a9dbe5c5, 0x000263a5672326a0, 0x000428089faa5816, 0x00002dba2a1d1558, 0x000197ae62da0922, 0x000abba225f631df, 0x000000000008f4eb}, {0x0005dcc2ee38818c, 0x0001bbbadc9b2cb5, 0x000ab544fb33427b, 0x0006cc2759413f76, 0x000121dd1222072a, 0x000ac2c737b39ea4, 0x000296d6baf9e196, 0x0000000000075d46}},
        {{0x000b2448cce42a48, 0x0001555671128874, 0x0000a30717d307ff, 0x000207ce99ca16c1, 0x00057c7a26b97e21, 0x000c3ab00785fa10, 0x00059f33c28658c2, 0x00000000000d79cd}, {0x0004e804722ed471, 0x0006f9a674db4469, 0x000c9ef186db33b8, 0x000722166bf2c6d5, 0x000abd4c3791990e, 0x000696ee1ba29aff, 0x0006a5d98510cf20, 0x00000000000f93d2}},
    },
    {
        /* digit=94 [{1,2,3,..,}]*([2^376]*G)  */
        {{0x000bbb4af21bdc49, 0x00065e7cf5babca3, 0x0003f75a29c95b64, 0x000531d04fa84740, 0x00045780efda9c1a, 0x00062cb5a399e848, 0x000f3b6e9c12be4f, 0x0000000000094a5d}, {0x000aeeb34b3f3b9c, 0x0006d3c15fe11c69, 0x0009e05f576f256d, 0x0006da4be294c912, 0x000aa99b83983ef7, 0x0001b9f6a5cf21d5, 0x0007bf50679cf875, 0x0000000000054d0b}},
        {{0x0009b2dd4ee443f9, 0x000fcc84f9171b4d, 0x000ea8b580b8da51, 0x000753ba05068eb1, 0x0004077777efcdc4, 0x0004cf44aa177ad1, 0x00044e0b7f54eb7e, 0x0000000000018601}, {0x000becbcecff7dd7, 0x0002b1279ed4c55b, 0x000e10b5af306b99, 0x00058fab8cdb6cf3, 0x0004a0c7614aebc9, 0x0002c6d8ebc4f93f, 0x00038ac5bd6cde7b, 0x000000000008d65f}},
        {{0x0000f0922779f009, 0x000bb4da7d63a943, 0x00025a4f402efdea, 0x000001668841d1ef, 0x0005b10366f61e8c, 0x000b121e90688f48, 0x000c1e38ee34b005, 0x0000000000014e0d}, {0x000a8eaa6c54d224, 0x000e683c1d44e6fa, 0x000050a1a6ad5a9a, 0x000274453715349d, 0x000dc5ca18f66b73, 0x000ef86e35065bdc, 0x000f5d677313cf2d, 0x0000000000033ebf}},
        {{0x000a3669bab3aa8c, 0x0001fffec66743ba, 0x000af6b2298cee7c, 0x0001365ee61bc95f, 0x000e8948b6e23181, 0x0001644be358bbd6, 0x0007a71cd9c342e3, 0x0000000000060a8a}, {0x000569b8813054fe, 0x0000ae065519f224, 0x000901053fc569ee, 0x000c03b8517ae8b7, 0x000173750d6e8957, 0x000aee6d7c96a040, 0x0001144eb2e364b2, 0x00000000000ed099}},
        {{0x0006635997173f0f, 0x0003b71714896339, 0x00025444ec94256f, 0x0005f33fe56d0905, 0x000638a1b2efb078, 0x0000fc15add43e0f, 0x000d362df35cfb36, 0x0000000000090b3e}, {0x000659873829fac7, 0x0000bd4970bdd7ef, 0x000e315a3dccca14, 0x0008a3e4a43c4f3e, 0x000fd9cb4808c99e, 0x000c3d0948013609, 0x000223f3fef0eaec, 0x000000000008642d}},
        {{0x0005a811304eaea0, 0x00018ffe3104b37f, 0x000808bc57214341, 0x000ed42cca2f1506, 0x0002c421c69415aa, 0x000b87b368618aa0, 0x000d5f4d6a091dfa, 0x000000000000e786}, {0x0009341357580c61, 0x000ee0be97746d4b, 0x000cd6467b049086, 0x000823b7a70d4224, 0x000809de860f6094, 0x000cf60205928791, 0x00086e7f7fca496d, 0x000000000005f460}},
        {{0x0007768861c39ce8, 0x00007b509f7028f0, 0x000fc09189281be1, 0x000832a7f07f3291, 0x000d8fbdcb83853d, 0x000a4650d074fdec, 0x00072080ed8de9d6, 0x00000000000a5d68}, {0x00079ea50cb0600f, 0x0009ae66666d224c, 0x0008a45c66bc5d53, 0x00099510da620fbe, 0x000fdf866f777e2d, 0x000045beb901145f, 0x000008442a37e5db, 0x00000000000beb1b}},
        {{0x00048a2dbd606668, 0x000c9bb39a179bed, 0x0009439710e05c24, 0x000abe3b91f14833, 0x000fbd39a2bc6be6, 0x0001d813ef972804, 0x000d4a06737e54f9, 0x000000000001a87c}, {0x000d0c4b5d0bc923, 0x000b2bf88b2e5982, 0x00052c33a491bb9f, 0x000aedfa58af0431, 0x0006ac0511b07a93, 0x0007e2c198853cdf, 0x000d8a9d7f416d06, 0x00000000000f9671}},
    },
    {
        /* digit=95 [{1,2,3,..,}]*([2^380]*G)  */
        {{0x000f6957a24c76d1, 0x000222e7d0a157df, 0x0001697f0073308f, 0x000ddc63eb317f9f, 0x00015adf71d715f1, 0x000858bc3f027507, 0x00071a2ce33c2eee, 0x00000000000da73b}, {0x00035cb76cad67cc, 0x000de9ef6a709ffc, 0x000c7c7ed1727cfd, 0x000f5a67502de655, 0x0001a47019739f5b, 0x000ea7b24d11122a, 0x0002e182cfaac2a2, 0x000000000000a458}},
        {{0x0007ab134ebd334b, 0x0005196a1e74f032, 0x00014e69e2323c83, 0x0009c4fbe35109cc, 0x00004521d7e61244, 0x000d002931fad8b3, 0x000229c85eb23d57, 0x0000000000035f68}, {0x0000d113aed4dbde, 0x0000c7c5b86677f2, 0x000189b64d0338ba, 0x0000456757cd5717, 0x000e3a1c866b067b, 0x000ba88c0eaab81b, 0x0007dc72a6af75bb, 0x000000000000ef56}},
        {{0x000ac045af51c5bb, 0x00010597a26ff058, 0x00059bcfafa87a4b, 0x000aff65c27f5f6d, 0x0005d8d544e60b06, 0x000275c32ce5ab66, 0x000e7c92f031be99, 0x00000000000f42e0}, {0x000c8a905adb28dd, 0x000175bb28b05c35, 0x00031ae347df4e7d, 0x000d299f93fb7dcb, 0x00086e2ccad9e73f, 0x0004d4f57fcd999a, 0x000c9bb8c8a6df33, 0x000000000009a2ac}},
        {{0x00030bea7fcf0ecc, 0x0006a3afdee26fd5, 0x000cc01988c78a70, 0x0008ee3ba4a33026, 0x000a2a883b7f340c, 0x000d7412b1a6ea51, 0x000b6e64f27976d5, 0x0000000000091251}, {0x0008bb3d89325d7a, 0x000d25f3f8f40ad2, 0x000216e9116c139b, 0x00073d6ad61b2cbc, 0x0005d542676dde3b, 0x0007d712bf08398d, 0x000023d373931e8e, 0x000000000007f96e}},
        {{0x0004e4e1b942add8, 0x000e15adf3d9957d, 0x0005ba50de0860ba, 0x000ce33a82d3736d, 0x000e718c8aa3f9fb, 0x0006ccf69f307823, 0x00065b860578cf41, 0x000000000001ef84}, {0x0009d5f6cc7a9ceb, 0x000939ee0cf97011, 0x0002f3cbdb7c8fc5, 0x0009a8dc09da90cb, 0x000dcbf3ccb8f99b, 0x000a626321f521c2, 0x0009da6bf6694891, 0x00000000000854fe}},
        {{0x0000917c5016c853, 0x00049e44e3f85fb2, 0x0000e1793eed8bac, 0x000254501c4e171e, 0x000cce52defb1004, 0x00063100ac12df68, 0x00035fb1fae2fbf3, 0x0000000000035732}, {0x0007edc8c1da40d3, 0x000c58d4deb655bb, 0x000bfb14f6d9d28f, 0x00085835a4e118f4, 0x0001308772e93249, 0x0004e48765abfa96, 0x0007b241c7611ff5, 0x000000000001f586}},
        {{0x000782cfada548c3, 0x0001e463031709b7, 0x0005afd4d0d5166c, 0x00097fff172c7d05, 0x000735b193cfd8e4, 0x0009df4bc7f7009d, 0x000e376b9fd3f7f1, 0x00000000000b46f1}, {0x0004c800fedd3a70, 0x0000987c6eaa8c8c, 0x000dff8047ad562a, 0x0001042539eab96e, 0x000779b8f5cc2a12, 0x0007840dc29a6d5d, 0x00030f33803a10b7, 0x0000000000011a6a}},
        {{0x0000457a60ad991e, 0x0005e3c90a59f250, 0x0005eeda9345d42a, 0x000885c1a0d58e29, 0x000d6816b0099aca, 0x000e4718f77b9b6d, 0x000c458e5c12139e, 0x000000000004e4ea}, {0x0005f9c85e9bfc5c, 0x00086c00c3568eed, 0x000b5b4b0eba61f9, 0x000ffc75eaf3eab0, 0x000d42a87e32cba1, 0x0007a882f5f99092, 0x0005122e7b49c76f, 0x0000000000029040}},
    },
    {
        /* digit=96 [{1,2,3,..,}]*([2^384]*G)  */
        {{0x000dbfc10083fe9c, 0x000bfcf17f9dc084, 0x0005e1e364b063e9, 0x00054ffe437b29cb, 0x0009e27ee9e2a694, 0x0003af03b6628d78, 0x000d6256e3b975ee, 0x000000000002f532}, {0x000e1d6aa4b057a7, 0x00043cae15213418, 0x000003dc9b2b9ddc, 0x000a959fa4d82608, 0x00055ae17566902b, 0x000c82eb4bad700d, 0x000ea716b2c5dc14, 0x0000000000036708}},
        {{0x000e41b4a9c3409a, 0x000849cbd62cd8bb, 0x00047e8c2d4de38c, 0x000bbe98f886eca3, 0x00049b432990b7f3, 0x00030035684860e8, 0x00029ed19a3bafa4, 0x000000000000e209}, {0x000ed539231869f6, 0x00098f15ff660294, 0x00050e66a84ebdc7, 0x000de8d2955612f1, 0x0006ec97a4a53ec6, 0x0000d15fe2d95b4e, 0x00024d868731f0f9, 0x00000000000aefae}},
        {{0x000253f48e5fab58, 0x0006889f97859df6, 0x000dee3f7bba228a, 0x0001d4ea62a5c827, 0x000ea3045da6826e, 0x000d3d237ecba3e4, 0x000358beed1da058, 0x00000000000b8b41}, {0x000293271a9d4293, 0x000d2730e7ac94c3, 0x000f438703e7096b, 0x0004cace8c1c6462, 0x0001055d39de9ba9, 0x000b71db3ec7c382, 0x000a11c89705a82f, 0x00000000000781b6}},
        {{0x000606caa5686427, 0x000cdc1951fb0886, 0x0000daff401d6319, 0x00009bb2754f1d38, 0x000e0e0e2bf19831, 0x0005a3da40f7db33, 0x000197348c4e1937, 0x00000000000840a6}, {0x0008d7f45b2f9769, 0x000d11687a735d22, 0x000434d1e1dee1ff, 0x000773c0aa9e79d1, 0x0009f56de9654a25, 0x000dda7af656f6e2, 0x00063fdf84666df0, 0x000000000003f9e4}},
        {{0x000a9c9951735edd, 0x0006afe9f90d3ad9, 0x000f97a0b45a07d7, 0x0007d6ec36e3b706, 0x000f580c8e6dd513, 0x000bab45c3ea55f4, 0x000d5819398b33e1, 0x00000000000aee76}, {0x000d9c115c469d59, 0x00071e72c8214ce4, 0x0006b3ef7ac6abe3, 0x0001ac198f9553cb, 0x000a1c98ad467fa5, 0x0000ebbb2019ac4e, 0x0000a68942e16e01, 0x00000000000bed3c}},
        {{0x000f13cdd62e69a0, 0x000461a814bf695a, 0x000e4323f4986584, 0x0001bfa2a4dd16f8, 0x000a76fb9f2b33c9, 0x000b1f1114af4d9c, 0x000b45a23635ef75, 0x00000000000a0891}, {0x000bef0dcdd6b903, 0x0005f5d1b87ba5f6, 0x0005d7ac1d6bdae8, 0x000cb4543d4ef435, 0x00064aa784ea70bd, 0x00070e4220ed12d3, 0x000483009cd34901, 0x00000000000280cf}},
        {{0x000bddf9aa596a1d, 0x0008952c04b79c0e, 0x00099a2770f3fa4a, 0x00044bf1911b3184, 0x0006bb897d318407, 0x000e9de5dd13f080, 0x00052b376dc8b81d, 0x00000000000c996b}, {0x00047f465159b51f, 0x00041d91e47b224a, 0x0009b71ad19e642b, 0x000c167eface7572, 0x0001d4805ed6a441, 0x0003fd6654eb9588, 0x0005778fc93daf3f, 0x00000000000cc570}},
        {{0x0000b16f46a35f8c, 0x00065a630c20c4e5, 0x0001f4362772ed03, 0x000aca10c0dec6cd, 0x000ba9e2f55428c8, 0x000bbb1705d34bb5, 0x000f6e8e81b4f732, 0x000000000008363b}, {0x000ca7950547e910, 0x000969603fe028be, 0x00047954fea1ddef, 0x000bb8efc191d12e, 0x0005dba97347c0da, 0x000656aaaf0e463b, 0x000cf0b7f7c207a8, 0x000000000003f08d}},
    }};

#endif /* #if !defined(_DISABLE_ECP_384R1_HARDCODED_BP_TBL_) */

IPP_OWN_DEFN(const cpPrecompAP *, gfpec_precom_nistP384r1_radix52_fun, (void))
{
   static cpPrecompAP t = {
      /* w */ 4,
      /* select function */ p384r1_select_ap_w4_ifma,
      /* precomputed data */ (BNU_CHUNK_T *)ifma_ec_nistp384r1_bp_precomp
   };
   return &t;
}

#endif // (_IPP32E >= _IPP32E_K1)

#endif /* #ifndef IFMA_ECPRECOMP4_P384_H */
