/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "fileviewer.h"
#include "ifpguiview.h"
#include "ifp_gui.h"

#include <QtGui>
#include <sys/stat.h>
#include <unistd.h>
#include "errno.h"
#include "qglobal.h"
#ifdef BSD4
#include "libgen.h"
#endif

//#include "gui_user.h"
#include "backend/ifpinterface.h"
#include "createdirdlg.h"
#include "backend/iointerface.h"

static const int autoopenTime = 1500;


QPixmap *folderLocked = 0;
QPixmap *folderClosed = 0;
QPixmap *folderOpen = 0;
QPixmap *fileNormal = 0;
QPixmap *fileMusic = 0;

/*****************************************************************************
 *
 * Class Directory
 *
 *****************************************************************************/

DirectoryItem::DirectoryItem(DirectoryItem * parent, const QStringList& dirname, IOInterface *Intf)
        : QTreeWidgetItem(parent, dirname, IO_INTF_DIR  + 1000), file(dirname[0]),
        showDirsOnly(parent->showDirsOnly)
{
    QString fullFileName;
    parent_dir = parent;
    ioIntf = Intf;

    if ((dirname[0].isNull()) || (dirname[0].isEmpty()))
        throw "Exception";

    setChildIndicatorPolicy(QTreeWidgetItem::ShowIndicator);

    fullFileName = ioIntf->ProtocolType + fullName();
    readable = ioIntf->get_isReadable(fullFileName);

    if (!readable)
        setIcon(0, QIcon(QPixmap(*folderLocked)));
    else
        setIcon(0, QIcon(QPixmap(*folderClosed)));

}


DirectoryItem::DirectoryItem(QTreeWidget * parent, const QStringList& dirname, IOInterface *Intf)
        : QTreeWidgetItem(parent, dirname, IO_INTF_DIR + 1000), file(dirname[0]),
        showDirsOnly(((DirectoryView*)parent)->showDirsOnly())
{
    QString fullFileName;
    parent_dir = 0;
    ioIntf = Intf;

    setChildIndicatorPolicy(QTreeWidgetItem::ShowIndicator);

    fullFileName = ioIntf->ProtocolType + fullName();
    readable = Intf->get_isReadable(fullFileName);

    if (!readable)
        setIcon(0, QIcon(QPixmap(*folderLocked)));
    else
        setIcon(0, QIcon(QPixmap(*folderClosed)));

}

void DirectoryItem::setOpen(bool opn)
{
    int ret;
    QString dirName, filenamestr, sizestr, filetype;
    QList<FileInfos> file_info_list;
    QList<FileInfos>::iterator fi_iter;

    if (opn)
        setIcon(0, QIcon(QPixmap(*folderOpen)));
    else
        setIcon(0, QIcon(QPixmap(*folderClosed)));

    if (opn && !childCount())
    {
        dirName = ioIntf->ProtocolType + fullName();
        ret = ioIntf->exec_GetDirectoryListing(dirName, &file_info_list);
        if (ret != IO_INTF_OK)
        {
            setIcon(0, QIcon(QPixmap(*folderClosed)));
            return;
        }

        if (file_info_list.isEmpty())
        {
            readable = FALSE;
            return;
        }

        for (fi_iter= file_info_list.begin(); fi_iter != file_info_list.end(); ++fi_iter)
        {
            sizestr = " ";
            filenamestr = fi_iter->FileName();
            if (fi_iter->FileType() == IO_INTF_FILE)
            {
                // Add file
                sizestr = QString("%1").arg(fi_iter->FileSize(), 12);
                filetype = "File";
                QStringList  tmp;
                tmp << filenamestr << filetype << sizestr;
                new FileItem(this, tmp, ioIntf);
            }
            else if (fi_iter->FileType() == IO_INTF_DIR)
            {
                // Add directory
                filetype = "Directory";
                QStringList tmp;
                tmp << filenamestr << "Directory";
                new DirectoryItem(this, tmp, ioIntf);
            }
            else if (fi_iter->FileType() == IO_INTF_SYMLINK)
            {
                // Add directory
                filetype = "Symbolic Link";
            }
            else if (fi_iter->FileType() == IO_INTF_SPECIAL)
            {
                // Add directory
                QStringList tmp;
                tmp << filenamestr << "Special" << sizestr;
                new FileItem(this, tmp, ioIntf);
            }
        }
    }
}


QString DirectoryItem::fullName()
{
    QString lFileName = "";

    lFileName = file.fileName();
    if (parent_dir)
    {
        lFileName = parent_dir->fullName() + lFileName + "/";
    }
    return lFileName;
}

QString DirectoryItem::baseName()
{
    QString lFileName = "";

    lFileName = file.fileName();
    return lFileName;
}

void DirectoryItem::setDirName(const QString& newName)
{
    file.setFileName(newName);
}


/*****************************************************************************
 *
 * Class DirectoryView
 *
 *****************************************************************************/
DirectoryView::DirectoryView(QWidget *parent, bool sdo)
        : QTreeWidget(parent), dirsOnly(sdo), oldCurrent(0),
        dropItem(0), mousePressed(FALSE)
{
    autoopen_timer = new QTimer(this);
    if (!folderLocked)
    {
        folderLocked = new QPixmap(":/images/folder-locked.png");
        folderClosed = new QPixmap(":/images/folder.png");
        folderOpen = new QPixmap(":/images/folder-open.png");
        fileNormal = new QPixmap(":/images/file.png");
        fileMusic = new QPixmap(":/images/audio.png");
    }

    createActions();
    connect(this, SIGNAL(itemChanged(QTreeWidgetItem *, int)),
            this, SLOT(fvItemRenamed(QTreeWidgetItem *, int)));
    connect(this, SIGNAL(itemExpanded(QTreeWidgetItem *)),
            this, SLOT(setItemExpanded(QTreeWidgetItem *)));

    setItemsExpandable(TRUE);
    //setSelectionMode(QListView::Extended);
    //setAcceptDrops(TRUE);
    viewport()->setAcceptDrops(TRUE);
    setSelectionMode(QAbstractItemView::ExtendedSelection);
    setSelectionBehavior(QAbstractItemView::SelectRows);


    connect(autoopen_timer, SIGNAL(timeout()),
            this, SLOT(openFolder()));
    msgflag = 0;
    OldRenamePath = "";
}

void DirectoryView::openFolder()
{
    autoopen_timer->stop();
    if (dropItem && !dropItem->isExpanded())
    {
        dropItem->setExpanded(TRUE);
        //dropItem->repaint();
    }
}

void DirectoryView::setItemExpanded ( QTreeWidgetItem * item )
{
    if (item->type() == IO_INTF_DIR + 1000)
        ((DirectoryItem*)item)->setOpen(true);
    //else
    //    ((DirectoryItem*)item->parent())->setOpen(true);

}

void DirectoryView::keyPressEvent(QKeyEvent *kpevent)
{
    if (kpevent->key() == Qt::Key_Delete)
    {
        fvDeleteFile();
        kpevent->accept();
    }
    else if (kpevent->key() == Qt::Key_F2)
    {
        fvRenameFile();
        kpevent->accept();
    }
    else if (kpevent->key() == Qt::Key_F5)
    {
        fvRefreshFileItem();
    }
}

void DirectoryView::dragEnterEvent(QDragEnterEvent *dr_event)
{

    if (!dr_event->mimeData()->hasUrls())
    {
        dr_event->ignore();
        return;
    }
    dr_event->acceptProposedAction();
    oldCurrent = currentItem();

    QTreeWidgetItem *itm = itemAt(dr_event->pos());
    if (itm)
    {
        if (itm != dropItem)
        {
            autoopen_timer->stop();
        }
        dropItem = itm;
        autoopen_timer->start(autoopenTime);
    }
}


void DirectoryView::dragMoveEvent(QDragMoveEvent *dr_event)
{
    QWidget * srcWidget, *destWidget;

    srcWidget = dr_event->source();
    destWidget = (QWidget*)(this);

    if ((srcWidget == NULL) || (srcWidget == destWidget))
    {
        dr_event->ignore();
        return ;
    }

    if (!dr_event->mimeData()->hasUrls())
    {
        dr_event->ignore();
        return ;
    }
    dr_event->acceptProposedAction();

    QPoint vp = ((QDragMoveEvent*)dr_event)->pos();
    QTreeWidgetItem *itm = itemAt(vp);
    if (itm)
    {
        clearSelection();
        setItemSelected(itm, TRUE);
        setCurrentItem(itm);

        if (itm != dropItem)
        {
            autoopen_timer->stop();
        }
        dropItem = itm;
        autoopen_timer->start(autoopenTime);
    }
}

void DirectoryView::dragLeaveEvent(QDragLeaveEvent *)
{
    autoopen_timer->stop();
    dropItem = 0;
}

void DirectoryView::dropEvent(QDropEvent *dr_event)
{
    autoopen_timer->stop();

    QWidget *srcWidget, *destWidget;
    QString filename, Message, destDir;
    //ifp_progress_info pginfo;
    QList<QUrl> lst;
    QTreeWidgetItem *item;
    FileInfos fi;
    QList<FileInfos> list;

    if (!dr_event->mimeData()->hasUrls())
    {
        dr_event->ignore();
        return ;
    }

    srcWidget = dr_event->source();
    destWidget = (QWidget*)(this);

    if ((srcWidget == NULL) || (srcWidget == destWidget))
    {
        dr_event->ignore();
        return ;
    }

    lst = dr_event->mimeData()->urls();
    if (lst.count() <= 0)
    {
        dr_event->ignore();
        return;
    }
    dr_event->accept();
    QApplication::setOverrideCursor(QCursor(Qt::ArrowCursor));

    //-------------------------------------------------------
    // Copy from ifp device to local file system
    //-------------------------------------------------------
    item = itemAt(dr_event->pos());
    destDir = baseDirectory(item);

    for (int cnt = 0; cnt < lst.count(); ++cnt)
    {
        filename = lst.at(cnt).toString();
        backslash2slash(filename);
        fi = FileInfos(filename, -1, 0, NULL);
        list.append(fi);
    }
    MainIfpGuiWindow->enableWindow(false);
    ioIntf->exec_TransferDownload(destDir, &list);
    MainIfpGuiWindow->enableWindow(true);

    setDir(destDir);
    fvRefreshFileItem();
    QApplication::restoreOverrideCursor();

    return;
}

QString DirectoryView::fullPath(QTreeWidgetItem* item)
{
    QString fullpath = item->text(0);
    while((item = item->parent()))
    {
        if (item->parent())
            fullpath = item->text(0) + "/" + fullpath;
        else
            fullpath = item->text(0) + fullpath;
    }
#ifdef Q_WS_WIN
    if (fullpath.length() > 2 && fullpath[ 1 ] != ':')
    {
        QDir dir(fullpath);
        fullpath = dir.currentDirPath().left(2) + fullpath;
    }
#endif

    return fullpath;
}

QString DirectoryView::baseDirectory(QTreeWidgetItem* item)
{
    QString baseDir;
    QTreeWidgetItem *parentItem;

    baseDir = "/";

    if (item == NULL)
        return(baseDir);
    //how to get the directory?? // are we in a directory??
    if (item->type() == IO_INTF_FILE + 1000)
    {
        parentItem = item->parent();
        if (parentItem)
            baseDir = fullPath(parentItem) + "/";
    }
    else
    {
        baseDir = fullPath(item) + "/";
    }

    return baseDir;
}

QString DirectoryView::currentDirectory(void)
{
    QTreeWidgetItem *item = currentItem();
    if ((!item) || ( !item->isSelected() ))
    {
        setCurrentItem(findItems("/", Qt::MatchExactly).first());
        //return("/");
    }

    return(baseDirectory(item));
}

void DirectoryView::mousePressEvent(QMouseEvent* evnt)
{
    QTreeWidget::mousePressEvent(evnt);
    QPoint pnt(evnt->pos());
    QTreeWidgetItem *itm = itemAt(pnt);

    // Ignore all clicks except button 1
    if (evnt->button() != Qt::LeftButton)
        return;

    if (itm)
    {
        presspos = evnt->pos();
        mousePressed = TRUE;
        //// if the user clicked into the root decoration of the item, don't try to start a drag!
        //if (pnt.x() > header()->cellPos(header()->mapToActual(0)) +
        //      treeStepSize() * (itm->depth() + (rootIsDecorated() ? 1 : 0)) + itemMargin() ||
        //      pnt.x() < header()->cellPos(header()->mapToActual(0)))
        //   {
        //   presspos = evnt->pos();
        //   mousePressed = TRUE;
        //   }
    }
}

void DirectoryView::unselectAllChildren(QTreeWidgetItem *parentItem)
{
    QTreeWidgetItem *current;
    int cnt;

    for(cnt = 0; cnt < parentItem->childCount(); cnt++)
    {
        current = parentItem->child(cnt);
        if(current != NULL)
        {
            current->setSelected(false);
            if (current->childCount() > 0)
                unselectAllChildren(current);
        }
    }
}

int DirectoryView::getListOfSelectedItems(QList<FileInfos> *file_info_list)
{
    QString filename;
    FileInfos fi;
    QList<QTreeWidgetItem *> selected;

    //--------------------------------------------------------
    // make a list of all selected files
    //--------------------------------------------------------
    selected = this->selectedItems();
    for (int i = 0; i < selected.size(); ++i)
    {
        filename = fullPath(selected.at(i));
        if (selected.at(i)->type() == IO_INTF_FILE + 1000)
        {
            fi = FileInfos(QString(filename), IO_INTF_FILE, 0, (void*)selected.at(i));
            file_info_list->append(fi);
        }
        else
        {
            // do not include files or directories below this one
            unselectAllChildren(selected.at(i));
            fi = FileInfos(QString(filename), IO_INTF_DIR, 0, (void*)selected.at(i));
            file_info_list->append(fi);
        }
    }
    return(IO_INTF_OK);
}

void DirectoryView::mouseMoveEvent(QMouseEvent* evnt)
{
    if (mousePressed && (presspos - evnt->pos()).manhattanLength() > QApplication::startDragDistance())
    {
        QList<QUrl> source;
        QString filename;
        QDrag *drag = new QDrag(this);
        QMimeData *mimeData = new QMimeData;

        mousePressed = FALSE;

        QTreeWidgetItemIterator it(this);
        while(*it)
        {
            filename = fullPath(*it);
            if ((*it)->isSelected())
            {
                if ((*it)->type() == IO_INTF_DIR + 1000)
                {
                    unselectAllChildren(*it);
                }
                source << QUrl(filename);
            }
            ++it;
        }
        mimeData->setUrls(source);
        drag->setMimeData(mimeData);
        drag->start(Qt::CopyAction);
    }
}

void DirectoryView::mouseReleaseEvent(QMouseEvent *)
{
    mousePressed = FALSE;
}

void DirectoryView::setDir(const QString &s)
{
    QTreeWidgetItem *item;
    QTreeWidgetItemIterator it(this);

    ++it;
    for ( ; (*it); ++it)
    {
        (*it)->setExpanded(FALSE);
    }

    QStringList lst(s.split("/", QString::SkipEmptyParts));
    item = topLevelItem(0);
    QStringList::Iterator it2 = lst.begin();
    for ( ; it2 != lst.end(); ++it2)
    {
        while(item)
        {
            if (item->text(0) == *it2)
            {
                item->setExpanded(TRUE);
                break;
            }
            item = itemBelow(item);
        }
    }

    if (item)
    {
        setUpdatesEnabled(FALSE);
        clearSelection();
        setCurrentItem(item);
        item->setSelected(TRUE);
        scrollToItem(item);
        setUpdatesEnabled(TRUE);
    }
}

void DirectoryView::contextMenuEvent(QContextMenuEvent *event)
{
    QList<QAction *> actions;
    QTreeWidgetItem *item = currentItem();

    if (item != NULL)
    {
        actions.append(dvRenameFileAct);
        actions.append(dvDeleteFileAct);

        if (item->type() == IO_INTF_DIR + 1000)
            actions.append(dvCreateDirectoryAct);

        actions.append(dvRefreshFileItemAct);
    }
    if (actions.count() > 0)
        QMenu::exec(actions, event->globalPos());
}

void DirectoryView::createActions()
{
    //QAction *fmTransferSelectedFilesAct;

    dvRefreshFileItemAct = new QAction(QIcon(":/images/reload.png"), tr("Refresh"), this);
    dvRefreshFileItemAct->setStatusTip(tr("Refresh the current view or selected directory"));
    dvRefreshFileItemAct->setWhatsThis(tr("<p>Refresh the current view or selected directory.</p>"));
    connect(dvRefreshFileItemAct, SIGNAL(triggered()), this, SLOT(fvRefreshFileItem()));

    dvCreateDirectoryAct = new QAction(QIcon(":/images/newfolder.png"), tr("New Directory"), this);
    dvCreateDirectoryAct->setStatusTip( tr("Create a new directory"));
    dvCreateDirectoryAct->setWhatsThis( tr("<p>Create a new subdirectory at the current level or below the current selected directory.</p>"));
    connect(dvCreateDirectoryAct, SIGNAL(triggered()), this, SLOT(fvCreateDirectory()));

    //dvPropertiesAct = new QAction(QIcon(":/images/properties.png"), tr("Properties"), this);
    //dvPropertiesAct->setStatusTip( tr("Show the properties for the selected file or directory"));
    //dvPropertiesAct->setWhatsThis( tr("<p>Select a file from the list and press this button to show it's properties.</p>"));
    //connect(dvPropertiesAct, SIGNAL(triggered()), this, SLOT(propertiesClick()));

    dvRenameFileAct = new QAction(QIcon(":/images/rename.png"), tr("Rename"), this);
    dvRenameFileAct->setStatusTip( tr("Rename Selected File or Directory"));
    dvRenameFileAct->setWhatsThis( tr("<p>Select a file from the list and press this button to rename it.</p>"));
    connect(dvRenameFileAct, SIGNAL(triggered()), this, SLOT(fvRenameFile()));

    dvDeleteFileAct = new QAction(QIcon(":/images/delete.png"), tr("Delete"), this);
    dvDeleteFileAct->setStatusTip( tr("Delete Selected File or Directory"));
    dvDeleteFileAct->setWhatsThis( tr("<p>Select a file from the list and press this button to delete it.</p>"));
    connect(dvDeleteFileAct, SIGNAL(triggered()), this, SLOT(fvDeleteFile()));
}

void DirectoryView::fvUpOneLevel(void)
{
    QString baseDir;
    QTreeWidgetItem *parentItem;
    QTreeWidgetItem *item = currentItem();

    if (!item)
    {
        setDir("/");
        return;
    }

    if (item->type() == IO_INTF_FILE + 1000)
    {
        parentItem = item->parent();
        if (!parentItem)
        {
            baseDir = "/";
        }
        else
        {
            parentItem = item->parent();
            if (!parentItem)
                baseDir = "/";
            else
                baseDir = fullPath(parentItem) + "/";
        }
    }
    else
    {
        parentItem = item->parent();
        if (!parentItem)
        {
            baseDir = "/";
        }
        else
        {
            baseDir = fullPath(parentItem) + "/";
        }
    }
    setDir(baseDir);
}

void DirectoryView::fvCreateDirectory(void)
{
    QString BaseNewDirPath, newDirPath, err;
    int ret;
    QTreeWidgetItem *item;

    item = currentItem();
    BaseNewDirPath = "/";

    if (item)
    {
        BaseNewDirPath = fullPath(item);
    }

    ret = ioIntf->exec_Mkdir(BaseNewDirPath, newDirPath);
    if (ret == IO_INTF_USER_CANCEL)
    {
        // do nothing
    }
    else if (ret != IO_INTF_OK)
    {
        err = ioIntf->get_LastError();
        showInfoMsgBox(tr("Make Directory Status"), err);
    }
    else
    {
        setDir(BaseNewDirPath);
        fvRefreshFileItem();
        backslash2slash(newDirPath);
        setDir(newDirPath);
    }
}

void DirectoryView::fvRenameFile(void)
{
    QTreeWidgetItem *item = currentItem();

    if (item)
    {
        OldRenamePath = fullPath(item);
        OldFileName = item->text(0);
        this->openPersistentEditor(item);
    }
}

void DirectoryView::fvDeleteFile(void)
{
    int ret;
    QList<FileInfos> file_info_list;

    //--------------------------------------------------------
    // make a list of all selected files then call exec_Delete
    //--------------------------------------------------------
    ret = getListOfSelectedItems(&file_info_list);

    ret = ioIntf->exec_Delete(&file_info_list);
    for (int i = 0; i < file_info_list.size(); ++i)
    {
        if (file_info_list.at(i).Processed())
            file_info_list.removeAt(i);
    }

    QList<QTreeWidgetItem *> selected;
    int idx;

    selected = this->selectedItems();
    for (int i = 0; i < selected.size(); ++i)
    {
        idx = selected[i]->parent()->indexOfChild(selected[i]);
        selected[i]->parent()->takeChild(idx);
    }

}

void DirectoryView::clearItem(QTreeWidgetItem* item)
{
    QTreeWidgetItem *childItem;

    setUpdatesEnabled(FALSE);
    while(item->child(0))
    {
        childItem = item->takeChild(0);
        delete(childItem);
    }
    setUpdatesEnabled(TRUE);
}

void DirectoryView::fvItemRenamed(QTreeWidgetItem * item, int col)
{
    QString newName, baseName, err, temp;
    int ret;

    if (!item)
        return;

    this->closePersistentEditor(item);

    if ((col != 0) || (OldRenamePath == ""))
        return;

    // rename item
    newName = fullPath(item);
    if (OldRenamePath != newName)
    {
        ret = ioIntf->exec_Rename(OldRenamePath, newName);
        if (ret != 0)
        {
            //baseName = basename(OldRenamePath);
            item->setText(0, OldFileName);

            err = ioIntf->get_LastError();
            showInfoMsgBox(tr("Rename Status"), err);
        }
        if (item->type() == IO_INTF_DIR + 1000)
        {
            ((DirectoryItem*)item)->setDirName(newName);
        }
    }
    OldRenamePath = "";
    OldFileName = "";
}

void DirectoryView::fvProperties(void)
{
    QString fileName;
    QTreeWidgetItem *item = currentItem();

    if (item)
    {
        fileName = fullPath(item);
        ioIntf->exec_ShowProperties(fileName);
    }
}

void DirectoryView::fvRefreshFileItem(void)
{
    QTreeWidgetItem *dirItem;
    QTreeWidgetItem *item = currentItem();
    if ((!item) || ( !item->isSelected() ))
    {
        setCurrentItem(findItems("/", Qt::MatchExactly).first());
        item = currentItem();

        if ((!item) || ( !item->isSelected() ))
            return;
    }

    QString setFocusName = fullPath(item);

    if (setFocusName == "/")
        setFocusName = "//";

    if (item)
    {
        if (item->type() == IO_INTF_FILE + 1000)
        {
            dirItem = item->parent();
        }
        else
        {
            dirItem = item;
        }
    }
    else
    {
        return;
    }

    clearItem(dirItem);
    setUpdatesEnabled(FALSE);
    ((DirectoryItem*)dirItem)->setOpen(TRUE);
    setUpdatesEnabled(TRUE);
    setDir(setFocusName);
}

void DirectoryView::fvTransferSelectedFiles(QString& destDir)
{
    int ret;
    QList<FileInfos> list;

    //----------------------------------------------------------------
    // make a list of all selected files then call exec_TransferUpload
    //----------------------------------------------------------------
    ret = getListOfSelectedItems(&list);

    MainIfpGuiWindow->enableWindow(false);
    ret = ioIntf->exec_TransferUpload(destDir, &list);
    MainIfpGuiWindow->enableWindow(true);
}



fileManagerView::fileManagerView(QWidget *parent, QString title, bool sort, int ioIntfType)
        : QWidget(parent)
{
    QStringList lst;
    QVBoxLayout *vLayout = new QVBoxLayout(this);
    QFrame *frame = new QFrame();

    frame->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
    vLayout->addWidget(frame);

    top_layout = new QGridLayout();
    top_layout->setVerticalSpacing(0);
    frame->setLayout(top_layout);

    createActions();
    createToolBars();

    internal_fileview = new DirectoryView(this, FALSE);
    top_layout->addWidget(internal_fileview);
    setFocusProxy(internal_fileview);
    switch (ioIntfType)
    {
    case IO_INTF_TYPE_LOCAL :
        internal_fileview->ioIntf = new FileInterface();
        break;
    case IO_INTF_TYPE_IFP :
        internal_fileview->ioIntf = new iFPInterface();
        break;
    default: // IO_INTF_TYPE_LOCAL
        internal_fileview->ioIntf = new FileInterface();
    }
    internal_fileview->setColumnCount(3);
    lst.clear();
    lst << tr("Name") << tr("Type") << tr("Size");
    internal_fileview->setHeaderLabels(lst);

    internal_fileview->setWindowTitle(title);
    internal_fileview->setAllColumnsShowFocus(TRUE);
    internal_fileview->setSortingEnabled(sort);
    if (sort)
    {
        internal_fileview->sortItems(1, Qt::AscendingOrder);
    }
    //internal_fileview->setColumnWidth(0, internal_fileview->width() - 200);
    internal_fileview->setColumnWidth(0, 340);
    internal_fileview->setColumnWidth(1, 60);
    internal_fileview->setColumnWidth(2, 100);

    // add the root directory and set it to open
    lst.clear();
    lst << "/";
    DirectoryItem *fileview_root = new DirectoryItem(internal_fileview, lst, internal_fileview->ioIntf);
    fileview_root->setExpanded(TRUE);
}

fileManagerView::~fileManagerView()
{
}

void fileManagerView::createActions()
{
    //QAction *fmTransferSelectedFilesAct;

    fmUpOneLevelAct = new QAction(QIcon(":/images/updirectory.png"), tr("Up 1 Level"), this);
    fmUpOneLevelAct->setStatusTip(tr("Go Up a Directory"));
    fmUpOneLevelAct->setWhatsThis(tr("<p>Press this button to go up a directory from the current selected directory.</p>"));
    connect(fmUpOneLevelAct, SIGNAL(triggered()), this, SLOT(upOneLevelClick()));

    fmRefreshFileItemAct = new QAction(QIcon(":/images/reload.png"), tr("Refresh"), this);
    fmRefreshFileItemAct->setStatusTip(tr("Refresh the current view or selected directory"));
    fmRefreshFileItemAct->setWhatsThis(tr("<p>Refresh the current view or selected directory.</p>"));
    connect(fmRefreshFileItemAct, SIGNAL(triggered()), this, SLOT(refreshClick()));

    fmCreateDirectoryAct = new QAction(QIcon(":/images/newfolder.png"), tr("New Directory"), this);
    fmCreateDirectoryAct->setStatusTip( tr("Create a new directory"));
    fmCreateDirectoryAct->setWhatsThis( tr("<p>Create a new subdirectory at the current level or below the current selected directory.</p>"));
    connect(fmCreateDirectoryAct, SIGNAL(triggered()), this, SLOT(createDirectoryClick()));

    fmPropertiesAct = new QAction(QIcon(":/images/properties.png"), tr("Properties"), this);
    fmPropertiesAct->setStatusTip( tr("Show the properties for the selected file or directory"));
    fmPropertiesAct->setWhatsThis( tr("<p>Select a file from the list and press this button to show it's properties.</p>"));
    connect(fmPropertiesAct, SIGNAL(triggered()), this, SLOT(propertiesClick()));

    fmRenameFileAct = new QAction(QIcon(":/images/rename.png"), tr("Rename"), this);
    fmRenameFileAct->setStatusTip( tr("Rename Selected File or Directory"));
    fmRenameFileAct->setWhatsThis( tr("<p>Select a file from the list and press this button to rename it.</p>"));
    connect(fmRenameFileAct, SIGNAL(triggered()), this, SLOT(renameClick()));

    fmDeleteFileAct = new QAction(QIcon(":/images/delete.png"), tr("Delete"), this);
    fmDeleteFileAct->setStatusTip( tr("Delete Selected File or Directory"));
    fmDeleteFileAct->setWhatsThis( tr("<p>Select a file from the list and press this button to delete it.</p>"));
    connect(fmDeleteFileAct, SIGNAL(triggered()), this, SLOT(deleteClick()));
}

void fileManagerView::createToolBars()
{
    fileToolBar = new QToolBar(tr("File"), this);

    top_layout->addWidget(fileToolBar);
    fileToolBar->addAction(fmUpOneLevelAct);
    fileToolBar->addAction(fmRefreshFileItemAct);
    fileToolBar->addAction(fmCreateDirectoryAct);
    fileToolBar->addWidget(new QLabel("      "));
    //fileToolBar->addSeparator();
    fileToolBar->addAction(fmPropertiesAct);
    fileToolBar->addAction(fmRenameFileAct);
    fileToolBar->addWidget(new QLabel("      "));
    //fileToolBar->addSeparator();
    fileToolBar->addAction(fmDeleteFileAct);
}

void fileManagerView::showItemContextMenu(const QPoint &position)
{
    QList<QAction *> actions;
    QTreeWidgetItem *item = internal_fileview->itemAt(position);

    if (item != NULL)
    {
        actions.append(fmRenameFileAct);
        actions.append(fmDeleteFileAct);

        if (item->type() == IO_INTF_DIR + 1000)
            actions.append(fmCreateDirectoryAct);

        actions.append(fmRefreshFileItemAct);
    }
    if (actions.count() > 0)
        QMenu::exec(actions, internal_fileview->mapToGlobal(position));
}

void fileManagerView::setDir(const QString &dirName)
{
    internal_fileview->setDir(dirName);
}

void fileManagerView::fmCreateDirectory(void)
{
    internal_fileview->fvCreateDirectory();
}

void fileManagerView::fmRenameFile(void)
{
    internal_fileview->fvRenameFile();
}

void fileManagerView::fmDeleteFile(void)
{
    internal_fileview->fvDeleteFile();
}

void fileManagerView::fmRefreshFileItem(void)
{
    internal_fileview->fvRefreshFileItem();
}

void fileManagerView::fmTransferSelectedFiles(QString& destDir)
{
    internal_fileview->fvTransferSelectedFiles(destDir);
}

int fileManagerView::fmUploadFileList(QString& destination, QList<FileInfos> *file_info_list)
{
    int ret;
    MainIfpGuiWindow->enableWindow(false);
    ret = internal_fileview->ioIntf->exec_TransferUpload(destination, file_info_list);
    MainIfpGuiWindow->enableWindow(true);
    return(ret);
}

void fileManagerView::upOneLevelClick(void)
{
    internal_fileview->fvUpOneLevel();
}

void fileManagerView::refreshClick(void)
{
    internal_fileview->fvRefreshFileItem();
}

void fileManagerView::createDirectoryClick(void)
{
    internal_fileview->fvCreateDirectory();
}

void fileManagerView::propertiesClick(void)
{
    internal_fileview->fvProperties();
}

void fileManagerView::renameClick(void)
{
    internal_fileview->fvRenameFile();
}

void fileManagerView::deleteClick(void)
{
    internal_fileview->fvDeleteFile();
}

QString fileManagerView::currentDirectory(void)
{
    return(internal_fileview->currentDirectory());
}

bool fileManagerView::isProcessing()
{
    return(internal_fileview->ioIntf->isProcessing());
}
