/* Ergo, version 3.3, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2013 Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Kohn−Sham Density Functional Theory Electronic Structure Calculations 
 * with Linearly Scaling Computational Time and Memory Usage,
 * Elias Rudberg, Emanuel H. Rubensson, and Pawel Salek,
 * J. Chem. Theory Comput. 7, 340 (2011),
 * <http://dx.doi.org/10.1021/ct100611z>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

#include "integrals_hermite.h"
#include "boysfunction.h"
#include <cmath>
#include <stdio.h>


int
get_related_integrals_hermite(const IntegralInfo & integralInfo,
			      const JK::ExchWeights & paramsCAM,
			      int n1max, int noOfMonomials_1,
			      int n2max, int noOfMonomials_2,
			      ergo_real dx0, 
			      ergo_real dx1, 
			      ergo_real dx2, 
			      ergo_real alpha0,
			      ergo_real resultPreFactor,
			      ergo_real* primitiveIntegralList)
{
  
  int Nmax = n1max + n2max;
  ergo_real R_12_squared = dx0*dx0 + dx1*dx1 + dx2*dx2;
  
  ergo_real BoysList[Nmax+1];

  
  /* Get Boys function values and store them in BoysList.
     NOTE: If CAM params are used, the values in BoysList are
     not simply Boys function values but are modified 
     according to the CAM params.  */

  if(paramsCAM.computeRangeSeparatedExchange)
    {
      ergo_real BoysFunctionList_std[Nmax+1];
      ergo_real BoysFunctionList_mod[Nmax+1];
      ergo_real mu = paramsCAM.mu;
      ergo_real v1_squared = mu * mu / (mu * mu + alpha0);
      ergo_real v1 = std::sqrt(v1_squared);
      /* Prepare BoysFunctionList_std */
      /* Use downward recursion to get Boys function values */
      ergo_real arg1 =  alpha0 * R_12_squared;
      ergo_real expMinusArg1 = std::exp(-arg1);
      BoysFunctionList_std[Nmax] = integralInfo.BoysFunction(Nmax, arg1);
      for(int i = Nmax-1; i >= 0; i--)
	BoysFunctionList_std[i] = (2*arg1*BoysFunctionList_std[i+1] + expMinusArg1) / (2*i+1);
      /* Prepare BoysFunctionList_mod */
      /* Use downward recursion to get Boys function values */
      ergo_real arg2 =  alpha0 * R_12_squared * v1_squared;
      ergo_real expMinusArg2 = std::exp(-arg2);
      BoysFunctionList_mod[Nmax] = integralInfo.BoysFunction(Nmax, arg2);
      for(int i = Nmax-1; i >= 0; i--)
	BoysFunctionList_mod[i] = (2*arg2*BoysFunctionList_mod[i+1] + expMinusArg2) / (2*i+1);
      // rescale
      for(int i = 0; i <= Nmax; i++)
	BoysFunctionList_mod[i] *= v1 * std::pow(v1_squared, i);  /* TODO: avoid using pow() here! */
      // add BoysFunctionList_std and BoysFunctionList_mod using weights given by cam_param_alpha and cam_param_beta
      for(int i = 0; i <= Nmax; i++)
	BoysList[i] = paramsCAM.alpha * BoysFunctionList_std[i] + paramsCAM.beta * BoysFunctionList_mod[i];
    }
  else
    {
      /* Compute all Boys function values needed */
      /* Use downward recursion to get Boys function values */
      ergo_real arg = alpha0 * R_12_squared;
      BoysList[Nmax] = integralInfo.BoysFunction(Nmax, arg);
      if(Nmax > 0)
	{
	  ergo_real expMinusArg = std::exp(-arg); 
	  for(int i = Nmax-1; i >= 0; i--)
	    BoysList[i] = (2*arg*BoysList[i+1] + expMinusArg) / (2*i+1);
	}
    }



#if 1
  if(n1max == 0)
    {
      if(n2max == 0)
	{
	  primitiveIntegralList[0] = resultPreFactor * BoysList[0];
	  return 0;
	} // END IF (n2max == 0)
      if(n2max == 1)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  
	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  primitiveIntegralList[0] = R0000;

	  primitiveIntegralList[1] = R0001;
	  primitiveIntegralList[2] = R0010;
	  primitiveIntegralList[3] = R0100;

	  return 0;
	} // END IF (n2max == 1)
      if(n2max == 2)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  primitiveIntegralList[0] = R0000;

	  primitiveIntegralList[1] = R0001;
	  primitiveIntegralList[2] = R0010;
	  primitiveIntegralList[3] = R0100;

	  primitiveIntegralList[4] = R0002;
	  primitiveIntegralList[5] = R0011;
	  primitiveIntegralList[6] = R0020;
	  primitiveIntegralList[7] = R0101;
	  primitiveIntegralList[8] = R0110;
	  primitiveIntegralList[9] = R0200;

	  return 0;
	} // END IF (n2max == 1)
    }
  if(n1max == 1)
    {
      if(n2max == 0)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  primitiveIntegralList[0] = R0000;

	  primitiveIntegralList[1] = -R0001;
	  primitiveIntegralList[2] = -R0010;
	  primitiveIntegralList[3] = -R0100;

	  return 0;
	} // END IF (n2max == 0)
      if(n2max == 1)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  // i1 = 0 (000)
	  primitiveIntegralList[ 0] =  R0000; // i2 000
	  primitiveIntegralList[ 1] =  R0001; // i2 001
	  primitiveIntegralList[ 2] =  R0010; // i2 010
	  primitiveIntegralList[ 3] =  R0100; // i2 100
	  // i1 = 1 (001)
	  primitiveIntegralList[ 4] = -R0001; // i2 000
	  primitiveIntegralList[ 5] = -R0002; // i2 001
	  primitiveIntegralList[ 6] = -R0011; // i2 010
	  primitiveIntegralList[ 7] = -R0101; // i2 100
	  // i1 = 2 (010)
	  primitiveIntegralList[ 8] = -R0010; // i2 000
	  primitiveIntegralList[ 9] = -R0011; // i2 001
	  primitiveIntegralList[10] = -R0020; // i2 010
	  primitiveIntegralList[11] = -R0110; // i2 100
	  // i1 = 3 (100)
	  primitiveIntegralList[12] = -R0100; // i2 000
	  primitiveIntegralList[13] = -R0101; // i2 001
	  primitiveIntegralList[14] = -R0110; // i2 010
	  primitiveIntegralList[15] = -R0200; // i2 100

	  return 0;
	} // END IF (n2max == 1)
      if(n2max == 2)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor; 
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;
	  ergo_real R3000 = -8*alpha0*alpha0*alpha0*BoysList[3] * resultPreFactor;

	  ergo_real R2100 = dx0 * R3000;
	  ergo_real R2010 = dx1 * R3000;
	  ergo_real R2001 = dx2 * R3000;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R1200 = R2000 + dx0 * R2100;
	  ergo_real R1020 = R2000 + dx1 * R2010;
	  ergo_real R1002 = R2000 + dx2 * R2001;

	  ergo_real R1110 = dx0 * R2010;
	  ergo_real R1101 = dx0 * R2001;
	  ergo_real R1011 = dx1 * R2001;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  ergo_real R0111 =             dx0 * R1011;
	  ergo_real R0300 = 2 * R1100 + dx0 * R1200;
	  ergo_real R0030 = 2 * R1010 + dx1 * R1020;
	  ergo_real R0003 = 2 * R1001 + dx2 * R1002;
	  ergo_real R0210 =     R1010 + dx0 * R1110;
	  ergo_real R0201 =     R1001 + dx0 * R1101;
	  ergo_real R0120 =     R1100 + dx1 * R1110;
	  ergo_real R0021 =     R1001 + dx1 * R1011;
	  ergo_real R0102 =     R1100 + dx2 * R1101;
	  ergo_real R0012 =     R1010 + dx2 * R1011;

	  // i1 = 0 (000)
	  primitiveIntegralList[ 0] =  R0000; // i2 000
	  primitiveIntegralList[ 1] =  R0001; // i2 001
	  primitiveIntegralList[ 2] =  R0010; // i2 010
	  primitiveIntegralList[ 3] =  R0100; // i2 100
	  primitiveIntegralList[ 4] =  R0002; // i2 002
	  primitiveIntegralList[ 5] =  R0011; // i2 011
	  primitiveIntegralList[ 6] =  R0020; // i2 020
	  primitiveIntegralList[ 7] =  R0101; // i2 101
	  primitiveIntegralList[ 8] =  R0110; // i2 110
	  primitiveIntegralList[ 9] =  R0200; // i2 200
	  // i1 = 1 (001)
	  primitiveIntegralList[10] = -R0001; // i2 000
	  primitiveIntegralList[11] = -R0002; // i2 001
	  primitiveIntegralList[12] = -R0011; // i2 010
	  primitiveIntegralList[13] = -R0101; // i2 100
	  primitiveIntegralList[14] = -R0003; // i2 002
	  primitiveIntegralList[15] = -R0012; // i2 011
	  primitiveIntegralList[16] = -R0021; // i2 020
	  primitiveIntegralList[17] = -R0102; // i2 101
	  primitiveIntegralList[18] = -R0111; // i2 110
	  primitiveIntegralList[19] = -R0201; // i2 200
	  // i1 = 2 (010)
	  primitiveIntegralList[20] = -R0010; // i2 000
	  primitiveIntegralList[21] = -R0011; // i2 001
	  primitiveIntegralList[22] = -R0020; // i2 010
	  primitiveIntegralList[23] = -R0110; // i2 100
	  primitiveIntegralList[24] = -R0012; // i2 002
	  primitiveIntegralList[25] = -R0021; // i2 011
	  primitiveIntegralList[26] = -R0030; // i2 020
	  primitiveIntegralList[27] = -R0111; // i2 101
	  primitiveIntegralList[28] = -R0120; // i2 110
	  primitiveIntegralList[29] = -R0210; // i2 200
	  // i1 = 3 (100)
	  primitiveIntegralList[30] = -R0100; // i2 000
	  primitiveIntegralList[31] = -R0101; // i2 001
	  primitiveIntegralList[32] = -R0110; // i2 010
	  primitiveIntegralList[33] = -R0200; // i2 100
	  primitiveIntegralList[34] = -R0102; // i2 002
	  primitiveIntegralList[35] = -R0111; // i2 011
	  primitiveIntegralList[36] = -R0120; // i2 020
	  primitiveIntegralList[37] = -R0201; // i2 101
	  primitiveIntegralList[38] = -R0210; // i2 110
	  primitiveIntegralList[39] = -R0300; // i2 200

	  return 0;
	} // END IF (n2max == 2)
    }
  if(n1max == 2)
    {
      if(n2max == 0)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  // i1 = 0 (000)
	  primitiveIntegralList[0] =  R0000;
	  // i1 = 1 (001)
	  primitiveIntegralList[1] = -R0001;
	  // i1 = 2 (010)
	  primitiveIntegralList[2] = -R0010;
	  // i1 = 3 (100)
	  primitiveIntegralList[3] = -R0100;
	  // i1 = 4 (002)
	  primitiveIntegralList[4] =  R0002;
	  // i1 = 5 (011)
	  primitiveIntegralList[5] =  R0011;
	  // i1 = 6 (020)
	  primitiveIntegralList[6] =  R0020;
	  // i1 = 7 (101)
	  primitiveIntegralList[7] =  R0101;
	  // i1 = 8 (110)
	  primitiveIntegralList[8] =  R0110;
	  // i1 = 9 (200)
	  primitiveIntegralList[9] =  R0200;

	  return 0;
	} // END IF (n2max == 0)
      if(n2max == 1)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;
	  ergo_real R3000 = -8*alpha0*alpha0*alpha0*BoysList[3] * resultPreFactor;

	  ergo_real R2100 = dx0 * R3000;
	  ergo_real R2010 = dx1 * R3000;
	  ergo_real R2001 = dx2 * R3000;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R1200 = R2000 + dx0 * R2100;
	  ergo_real R1020 = R2000 + dx1 * R2010;
	  ergo_real R1002 = R2000 + dx2 * R2001;

	  ergo_real R1110 = dx0 * R2010;
	  ergo_real R1101 = dx0 * R2001;
	  ergo_real R1011 = dx1 * R2001;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  ergo_real R0111 =             dx0 * R1011;
	  ergo_real R0300 = 2 * R1100 + dx0 * R1200;
	  ergo_real R0030 = 2 * R1010 + dx1 * R1020;
	  ergo_real R0003 = 2 * R1001 + dx2 * R1002;
	  ergo_real R0210 =     R1010 + dx0 * R1110;
	  ergo_real R0201 =     R1001 + dx0 * R1101;
	  ergo_real R0120 =     R1100 + dx1 * R1110;
	  ergo_real R0021 =     R1001 + dx1 * R1011;
	  ergo_real R0102 =     R1100 + dx2 * R1101;
	  ergo_real R0012 =     R1010 + dx2 * R1011;

	  // i1 = 0 (000)
	  primitiveIntegralList[ 0] =  R0000; // i2 000
	  primitiveIntegralList[ 1] =  R0001; // i2 001
	  primitiveIntegralList[ 2] =  R0010; // i2 010
	  primitiveIntegralList[ 3] =  R0100; // i2 100
	  // i1 = 1 (001)
	  primitiveIntegralList[ 4] = -R0001; // i2 000
	  primitiveIntegralList[ 5] = -R0002; // i2 001
	  primitiveIntegralList[ 6] = -R0011; // i2 010
	  primitiveIntegralList[ 7] = -R0101; // i2 100
	  // i1 = 2 (010)
	  primitiveIntegralList[ 8] = -R0010; // i2 000
	  primitiveIntegralList[ 9] = -R0011; // i2 001
	  primitiveIntegralList[10] = -R0020; // i2 010
	  primitiveIntegralList[11] = -R0110; // i2 100
	  // i1 = 3 (100)
	  primitiveIntegralList[12] = -R0100; // i2 000
	  primitiveIntegralList[13] = -R0101; // i2 001
	  primitiveIntegralList[14] = -R0110; // i2 010
	  primitiveIntegralList[15] = -R0200; // i2 100
	  // i1 = 4 (002)
	  primitiveIntegralList[16] =  R0002; // i2 000
	  primitiveIntegralList[17] =  R0003; // i2 001
	  primitiveIntegralList[18] =  R0012; // i2 010
	  primitiveIntegralList[19] =  R0102; // i2 100
	  // i1 = 5 (011)
	  primitiveIntegralList[20] =  R0011; // i2 000
	  primitiveIntegralList[21] =  R0012; // i2 001
	  primitiveIntegralList[22] =  R0021; // i2 010
	  primitiveIntegralList[23] =  R0111; // i2 100
	  // i1 = 6 (020)
	  primitiveIntegralList[24] =  R0020; // i2 000
	  primitiveIntegralList[25] =  R0021; // i2 001
	  primitiveIntegralList[26] =  R0030; // i2 010
	  primitiveIntegralList[27] =  R0120; // i2 100
	  // i1 = 7 (101)
	  primitiveIntegralList[28] =  R0101; // i2 000
	  primitiveIntegralList[29] =  R0102; // i2 001
	  primitiveIntegralList[30] =  R0111; // i2 010
	  primitiveIntegralList[31] =  R0201; // i2 100
	  // i1 = 8 (110)
	  primitiveIntegralList[32] =  R0110; // i2 000
	  primitiveIntegralList[33] =  R0111; // i2 001
	  primitiveIntegralList[34] =  R0120; // i2 010
	  primitiveIntegralList[35] =  R0210; // i2 100
	  // i1 = 9 (200)
	  primitiveIntegralList[36] =  R0200; // i2 000
	  primitiveIntegralList[37] =  R0201; // i2 001
	  primitiveIntegralList[38] =  R0210; // i2 010
	  primitiveIntegralList[39] =  R0300; // i2 100

	  return 0;
	} // END IF (n2max == 0)
      if(n2max == 2)
	{
	  ergo_real R0000 = BoysList[0] * resultPreFactor;
	  ergo_real R1000 = -2*alpha0*BoysList[1] * resultPreFactor;
	  ergo_real R2000 =  4*alpha0*alpha0*BoysList[2] * resultPreFactor;
	  ergo_real R3000 = -8*alpha0*alpha0*alpha0*BoysList[3] * resultPreFactor;
	  ergo_real R4000 = 16*alpha0*alpha0*alpha0*alpha0*BoysList[4] * resultPreFactor;

	  ergo_real R3100 = dx0 * R4000;
	  ergo_real R3010 = dx1 * R4000;
	  ergo_real R3001 = dx2 * R4000;

	  ergo_real R2100 = dx0 * R3000;
	  ergo_real R2010 = dx1 * R3000;
	  ergo_real R2001 = dx2 * R3000;

	  ergo_real R2200 = R3000 + dx0 * R3100;
	  ergo_real R2020 = R3000 + dx1 * R3010;
	  ergo_real R2002 = R3000 + dx2 * R3001;

	  ergo_real R2110 = dx0 * R3010;
	  ergo_real R2101 = dx0 * R3001;
	  ergo_real R2011 = dx1 * R3001;

	  ergo_real R1100 = dx0 * R2000;
	  ergo_real R1010 = dx1 * R2000;
	  ergo_real R1001 = dx2 * R2000;

	  ergo_real R1200 = R2000 + dx0 * R2100;
	  ergo_real R1020 = R2000 + dx1 * R2010;
	  ergo_real R1002 = R2000 + dx2 * R2001;

	  ergo_real R1110 = dx0 * R2010;
	  ergo_real R1101 = dx0 * R2001;
	  ergo_real R1011 = dx1 * R2001;

	  ergo_real R1111 =             dx0 * R2011;
	  ergo_real R1300 = 2 * R2100 + dx0 * R2200;
	  ergo_real R1030 = 2 * R2010 + dx1 * R2020;
	  ergo_real R1003 = 2 * R2001 + dx2 * R2002;
	  ergo_real R1210 =     R2010 + dx0 * R2110;
	  ergo_real R1201 =     R2001 + dx0 * R2101;
	  ergo_real R1120 =     R2100 + dx1 * R2110;
	  ergo_real R1021 =     R2001 + dx1 * R2011;
	  ergo_real R1102 =     R2100 + dx2 * R2101;
	  ergo_real R1012 =     R2010 + dx2 * R2011;

	  ergo_real R0100 = dx0 * R1000;
	  ergo_real R0010 = dx1 * R1000;
	  ergo_real R0001 = dx2 * R1000;

	  ergo_real R0200 = R1000 + dx0 * R1100;
	  ergo_real R0020 = R1000 + dx1 * R1010;
	  ergo_real R0002 = R1000 + dx2 * R1001;

	  ergo_real R0110 = dx0 * R1010;
	  ergo_real R0101 = dx0 * R1001;
	  ergo_real R0011 = dx1 * R1001;

	  ergo_real R0111 =             dx0 * R1011;
	  ergo_real R0300 = 2 * R1100 + dx0 * R1200;
	  ergo_real R0030 = 2 * R1010 + dx1 * R1020;
	  ergo_real R0003 = 2 * R1001 + dx2 * R1002;
	  ergo_real R0210 =     R1010 + dx0 * R1110;
	  ergo_real R0201 =     R1001 + dx0 * R1101;
	  ergo_real R0120 =     R1100 + dx1 * R1110;
	  ergo_real R0021 =     R1001 + dx1 * R1011;
	  ergo_real R0102 =     R1100 + dx2 * R1101;
	  ergo_real R0012 =     R1010 + dx2 * R1011;

	  ergo_real R0400 = 3 * R1200 + dx0 * R1300;
	  ergo_real R0040 = 3 * R1020 + dx1 * R1030;
	  ergo_real R0004 = 3 * R1002 + dx2 * R1003;
	  ergo_real R0310 = 2 * R1110 + dx0 * R1210;
	  ergo_real R0301 = 2 * R1101 + dx0 * R1201;
	  ergo_real R0130 = 2 * R1110 + dx1 * R1120;
	  ergo_real R0031 = 2 * R1011 + dx1 * R1021;
	  ergo_real R0103 = 2 * R1101 + dx2 * R1102;
	  ergo_real R0013 = 2 * R1011 + dx2 * R1012;
	  ergo_real R0220 =     R1020 + dx0 * R1120;
	  ergo_real R0202 =     R1002 + dx0 * R1102;
	  ergo_real R0022 =     R1002 + dx1 * R1012;
	  ergo_real R0211 =     R1011 + dx0 * R1111;
	  ergo_real R0121 =     R1101 + dx1 * R1111;
	  ergo_real R0112 =     R1110 + dx2 * R1111;

	  // i1 = 0 (000)
	  primitiveIntegralList[ 0] =  R0000; // i2 000
	  primitiveIntegralList[ 1] =  R0001; // i2 001
	  primitiveIntegralList[ 2] =  R0010; // i2 010
	  primitiveIntegralList[ 3] =  R0100; // i2 100
	  primitiveIntegralList[ 4] =  R0002; // i2 002
	  primitiveIntegralList[ 5] =  R0011; // i2 011
	  primitiveIntegralList[ 6] =  R0020; // i2 020
	  primitiveIntegralList[ 7] =  R0101; // i2 101
	  primitiveIntegralList[ 8] =  R0110; // i2 110
	  primitiveIntegralList[ 9] =  R0200; // i2 200
	  // i1 = 1 (001)
	  primitiveIntegralList[10] = -R0001; // i2 000
	  primitiveIntegralList[11] = -R0002; // i2 001
	  primitiveIntegralList[12] = -R0011; // i2 010
	  primitiveIntegralList[13] = -R0101; // i2 100
	  primitiveIntegralList[14] = -R0003; // i2 002
	  primitiveIntegralList[15] = -R0012; // i2 011
	  primitiveIntegralList[16] = -R0021; // i2 020
	  primitiveIntegralList[17] = -R0102; // i2 101
	  primitiveIntegralList[18] = -R0111; // i2 110
	  primitiveIntegralList[19] = -R0201; // i2 200
	  // i1 = 2 (010)
	  primitiveIntegralList[20] = -R0010; // i2 000
	  primitiveIntegralList[21] = -R0011; // i2 001
	  primitiveIntegralList[22] = -R0020; // i2 010
	  primitiveIntegralList[23] = -R0110; // i2 100
	  primitiveIntegralList[24] = -R0012; // i2 002
	  primitiveIntegralList[25] = -R0021; // i2 011
	  primitiveIntegralList[26] = -R0030; // i2 020
	  primitiveIntegralList[27] = -R0111; // i2 101
	  primitiveIntegralList[28] = -R0120; // i2 110
	  primitiveIntegralList[29] = -R0210; // i2 200
	  // i1 = 3 (100)
	  primitiveIntegralList[30] = -R0100; // i2 000
	  primitiveIntegralList[31] = -R0101; // i2 001
	  primitiveIntegralList[32] = -R0110; // i2 010
	  primitiveIntegralList[33] = -R0200; // i2 100
	  primitiveIntegralList[34] = -R0102; // i2 002
	  primitiveIntegralList[35] = -R0111; // i2 011
	  primitiveIntegralList[36] = -R0120; // i2 020
	  primitiveIntegralList[37] = -R0201; // i2 101
	  primitiveIntegralList[38] = -R0210; // i2 110
	  primitiveIntegralList[39] = -R0300; // i2 200
	  // i1 = 4 (002)
	  primitiveIntegralList[40] =  R0002; // i2 000
	  primitiveIntegralList[41] =  R0003; // i2 001
	  primitiveIntegralList[42] =  R0012; // i2 010
	  primitiveIntegralList[43] =  R0102; // i2 100
	  primitiveIntegralList[44] =  R0004; // i2 002
	  primitiveIntegralList[45] =  R0013; // i2 011
	  primitiveIntegralList[46] =  R0022; // i2 020
	  primitiveIntegralList[47] =  R0103; // i2 101
	  primitiveIntegralList[48] =  R0112; // i2 110
	  primitiveIntegralList[49] =  R0202; // i2 200
	  // i1 = 5 (011)
	  primitiveIntegralList[50] =  R0011; // i2 000
	  primitiveIntegralList[51] =  R0012; // i2 001
	  primitiveIntegralList[52] =  R0021; // i2 010
	  primitiveIntegralList[53] =  R0111; // i2 100
	  primitiveIntegralList[54] =  R0013; // i2 002
	  primitiveIntegralList[55] =  R0022; // i2 011
	  primitiveIntegralList[56] =  R0031; // i2 020
	  primitiveIntegralList[57] =  R0112; // i2 101
	  primitiveIntegralList[58] =  R0121; // i2 110
	  primitiveIntegralList[59] =  R0211; // i2 200
	  // i1 = 6 (020)
	  primitiveIntegralList[60] =  R0020; // i2 000
	  primitiveIntegralList[61] =  R0021; // i2 001
	  primitiveIntegralList[62] =  R0030; // i2 010
	  primitiveIntegralList[63] =  R0120; // i2 100
	  primitiveIntegralList[64] =  R0022; // i2 002
	  primitiveIntegralList[65] =  R0031; // i2 011
	  primitiveIntegralList[66] =  R0040; // i2 020
	  primitiveIntegralList[67] =  R0121; // i2 101
	  primitiveIntegralList[68] =  R0130; // i2 110
	  primitiveIntegralList[69] =  R0220; // i2 200
	  // i1 = 7 (101)
	  primitiveIntegralList[70] =  R0101; // i2 000
	  primitiveIntegralList[71] =  R0102; // i2 001
	  primitiveIntegralList[72] =  R0111; // i2 010
	  primitiveIntegralList[73] =  R0201; // i2 100
	  primitiveIntegralList[74] =  R0103; // i2 002
	  primitiveIntegralList[75] =  R0112; // i2 011
	  primitiveIntegralList[76] =  R0121; // i2 020
	  primitiveIntegralList[77] =  R0202; // i2 101
	  primitiveIntegralList[78] =  R0211; // i2 110
	  primitiveIntegralList[79] =  R0301; // i2 200
	  // i1 = 8 (110)
	  primitiveIntegralList[80] =  R0110; // i2 000
	  primitiveIntegralList[81] =  R0111; // i2 001
	  primitiveIntegralList[82] =  R0120; // i2 010
	  primitiveIntegralList[83] =  R0210; // i2 100
	  primitiveIntegralList[84] =  R0112; // i2 002
	  primitiveIntegralList[85] =  R0121; // i2 011
	  primitiveIntegralList[86] =  R0130; // i2 020
	  primitiveIntegralList[87] =  R0211; // i2 101
	  primitiveIntegralList[88] =  R0220; // i2 110
	  primitiveIntegralList[89] =  R0310; // i2 200
	  // i1 = 9 (200)
	  primitiveIntegralList[90] =  R0200; // i2 000
	  primitiveIntegralList[91] =  R0201; // i2 001
	  primitiveIntegralList[92] =  R0210; // i2 010
	  primitiveIntegralList[93] =  R0300; // i2 100
	  primitiveIntegralList[94] =  R0202; // i2 002
	  primitiveIntegralList[95] =  R0211; // i2 011
	  primitiveIntegralList[96] =  R0220; // i2 020
	  primitiveIntegralList[97] =  R0301; // i2 101
	  primitiveIntegralList[98] =  R0310; // i2 110
	  primitiveIntegralList[99] =  R0400; // i2 200

	  return 0;
	}
    }
#endif

  ergo_real R[Nmax+1][Nmax+1][Nmax+1][Nmax+1];

  int n;
  ergo_real factor = 1;
  for(n = 0; n <= Nmax; n++)
    {
      R[n][0][0][0] = factor * BoysList[n];
      factor *= -2*alpha0;
    }

  int minus1topowList[Nmax+1];
  int ifactor = 1;
  for(n = 0; n <= Nmax; n++)
    {
      minus1topowList[n] = ifactor;
      ifactor *= -1;
    }
  
  // Use recurrences to get remaining R values
  for(n = Nmax - 1; n >= 0; n--)
    {
      int nn = Nmax - n;
      int noOfMonomials = integralInfo.monomial_info.no_of_monomials_list[nn];
      int i;
      for(i = 1; i < noOfMonomials; i++)
	{
	  int ix = integralInfo.monomial_info.monomial_list[i].ix;
	  int iy = integralInfo.monomial_info.monomial_list[i].iy;
	  int iz = integralInfo.monomial_info.monomial_list[i].iz;
	  if(ix > 0)
	    {
	      ergo_real Rval = dx0 * R[n+1][ix-1][iy][iz];
	      if(ix > 1)
		Rval += (ix - 1) * R[n+1][ix-2][iy][iz];
	      R[n][ix][iy][iz] = Rval;
	    }
	  else if(iy > 0)
	    {
	      ergo_real Rval = dx1 * R[n+1][ix][iy-1][iz];
	      if(iy > 1)
		Rval += (iy - 1) * R[n+1][ix][iy-2][iz];
	      R[n][ix][iy][iz] = Rval;
	    }
	  else if(iz > 0)
	    {
	      ergo_real Rval = dx2 * R[n+1][ix][iy][iz-1];
	      if(iz > 1)
		Rval += (iz - 1) * R[n+1][ix][iy][iz-2];
	      R[n][ix][iy][iz] = Rval;
	    }
	} // END FOR i
    } // END FOR n

  int i1, i2;
  for(i1 = 0; i1 < noOfMonomials_1; i1++)
    {
      int ix1 = integralInfo.monomial_info.monomial_list[i1].ix;
      int iy1 = integralInfo.monomial_info.monomial_list[i1].iy;
      int iz1 = integralInfo.monomial_info.monomial_list[i1].iz;
      int n1 = ix1+iy1+iz1;
      ergo_real prefactor = minus1topowList[n1] * resultPreFactor;
      for(i2 = 0; i2 < noOfMonomials_2; i2++)
	{
	  int ix2 = integralInfo.monomial_info.monomial_list[i2].ix;
	  int iy2 = integralInfo.monomial_info.monomial_list[i2].iy;
	  int iz2 = integralInfo.monomial_info.monomial_list[i2].iz;
	  primitiveIntegralList[i1*noOfMonomials_2+i2] = prefactor * R[0][ix1+ix2][iy1+iy2][iz1+iz2];
	} // END FOR i2  
    }
  
  return 0;
}


