package v2

import (
	"net/url"
	"testing"
)

type urlBuilderTestCase struct {
	description  string
	expectedPath string
	build        func() (string, error)
}

// TestURLBuilder tests the various url building functions, ensuring they are
// returning the expected values.
func TestURLBuilder(t *testing.T) {
	var (
		urlBuilder *URLBuilder
		err        error
	)

	testCases := []urlBuilderTestCase{
		{
			description:  "test base url",
			expectedPath: "/v2/",
			build: func() (string, error) {
				return urlBuilder.BuildBaseURL()
			},
		},
		{
			description:  "test tags url",
			expectedPath: "/v2/foo/bar/tags/list",
			build: func() (string, error) {
				return urlBuilder.BuildTagsURL("foo/bar")
			},
		},
		{
			description:  "test manifest url",
			expectedPath: "/v2/foo/bar/manifests/tag",
			build: func() (string, error) {
				return urlBuilder.BuildManifestURL("foo/bar", "tag")
			},
		},
		{
			description:  "build blob url",
			expectedPath: "/v2/foo/bar/blobs/tarsum.v1+sha256:abcdef0123456789",
			build: func() (string, error) {
				return urlBuilder.BuildBlobURL("foo/bar", "tarsum.v1+sha256:abcdef0123456789")
			},
		},
		{
			description:  "build blob upload url",
			expectedPath: "/v2/foo/bar/blobs/uploads/",
			build: func() (string, error) {
				return urlBuilder.BuildBlobUploadURL("foo/bar")
			},
		},
		{
			description:  "build blob upload url with digest and size",
			expectedPath: "/v2/foo/bar/blobs/uploads/?digest=tarsum.v1%2Bsha256%3Aabcdef0123456789&size=10000",
			build: func() (string, error) {
				return urlBuilder.BuildBlobUploadURL("foo/bar", url.Values{
					"size":   []string{"10000"},
					"digest": []string{"tarsum.v1+sha256:abcdef0123456789"},
				})
			},
		},
		{
			description:  "build blob upload chunk url",
			expectedPath: "/v2/foo/bar/blobs/uploads/uuid-part",
			build: func() (string, error) {
				return urlBuilder.BuildBlobUploadChunkURL("foo/bar", "uuid-part")
			},
		},
		{
			description:  "build blob upload chunk url with digest and size",
			expectedPath: "/v2/foo/bar/blobs/uploads/uuid-part?digest=tarsum.v1%2Bsha256%3Aabcdef0123456789&size=10000",
			build: func() (string, error) {
				return urlBuilder.BuildBlobUploadChunkURL("foo/bar", "uuid-part", url.Values{
					"size":   []string{"10000"},
					"digest": []string{"tarsum.v1+sha256:abcdef0123456789"},
				})
			},
		},
	}

	roots := []string{
		"http://example.com",
		"https://example.com",
		"http://localhost:5000",
		"https://localhost:5443",
	}

	for _, root := range roots {
		urlBuilder, err = NewURLBuilderFromString(root)
		if err != nil {
			t.Fatalf("unexpected error creating urlbuilder: %v", err)
		}

		for _, testCase := range testCases {
			url, err := testCase.build()
			if err != nil {
				t.Fatalf("%s: error building url: %v", testCase.description, err)
			}

			expectedURL := root + testCase.expectedPath

			if url != expectedURL {
				t.Fatalf("%s: %q != %q", testCase.description, url, expectedURL)
			}
		}
	}
}
