/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2004-07-21
 * Description : a widget to display an image histogram.
 * 
 * Copyright (C) 2004-2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

#ifndef HISTOGRAMWIDGET_H
#define HISTOGRAMWIDGET_H

// Qt includes.

#include <QWidget>
#include <QMouseEvent>
#include <QPaintEvent>

// Local includes

class ImageHistogram;
class HistogramWidgetPriv;

class HistogramWidget : public QWidget
{
Q_OBJECT

public:

    enum HistogramType
    {
        ValueHistogram = 0,       // Luminosity.
        RedChannelHistogram,      // Red channel.
        GreenChannelHistogram,    // Green channel.
        BlueChannelHistogram,     // Blue channel.
        AlphaChannelHistogram,    // Alpha channel.
        ColorChannelsHistogram    // All color channels.
    };

    enum HistogramScale
    {
        LinScaleHistogram=0,      // Linear scale.
        LogScaleHistogram         // Logarithmic scale.
    };

    enum HistogramAllColorMode
    {
        RedColor=0,               // Red color to foreground in All Colors Channel mode.
        GreenColor,               // Green color to foreground in All Colors Channel mode.
        BlueColor                 // Blue color to foreground in All Colors Channel mode.
    };

    enum HistogramRenderingType
    {
        FullImageHistogram=0,     // Full image histogram rendering.
        ImageSelectionHistogram   // Image selection histogram rendering.
    };

public:

    /** Constructor without image data. Needed to use updateData() method after to create instance.*/
    HistogramWidget(QWidget *parent=0, bool selectMode=true,
                    bool blinkComputation=true,
                    bool statisticsVisible=false);

    /** Constructor with image data and without image selection data.*/
    HistogramWidget(uchar *i_data, uint i_w, uint i_h,         // Full image info.
                    bool i_sixteenBits,                        // 8 or 16 bits image.
                    bool has_alpha,
                    QWidget *parent=0, bool selectMode=true,
                    bool blinkComputation=true,
                    bool statisticsVisible=false);

    /** Constructor with image data and image selection data.*/
    HistogramWidget(uchar *i_data, uint i_w, uint i_h,         // Full image info.
                    uchar *s_data, uint s_w, uint s_h,         // Image selection info.
                    bool i_sixteenBits,                        // 8 or 16 bits image.
                    bool has_alpha,
                    QWidget *parent=0, bool selectMode=true,
                    bool blinkComputation=true,
                    bool statisticsVisible=false);

    void setup(bool selectMode=true,
               bool blinkComputation=true,
               bool statisticsVisible=false);

    ~HistogramWidget();

    /** Stop current histogram computations.*/
    void stopHistogramComputation(void);

    /** Update full image histogram data methods.*/
    void updateData(uchar *i_data, uint i_w, uint i_h,
                    bool i_sixteenBits,                        // 8 or 16 bits image.
                    bool has_alpha,
                    uchar *s_data=0, uint s_w=0, uint s_h=0, 
                    bool blinkComputation=true);

    /** Update image selection histogram data methods.*/
    void updateSelectionData(uchar *s_data, uint s_w, uint s_h,
                             bool i_sixteenBits,               // 8 or 16 bits image.
                            bool has_alpha,
                             bool blinkComputation=true);

    void setDataLoading();
    void setLoadingFailed();

    void setHistogramGuideByColor(QColor color);

    void reset(void);

public:

    int             m_channelType;         // Channel type to draw.
    int             m_scaleType;           // Scale to use for drawing.
    int             m_colorType;           // Color to use for drawing in All Colors Channel mode.
    void            setRenderingType(HistogramRenderingType type);

    ImageHistogram *m_imageHistogram;      // Full image.
    ImageHistogram *m_selectionHistogram;  // Image selection.

signals:

    void signalIntervalChanged( int min, int max );
    void signalMaximumValueChanged( int );
    void signalHistogramComputationDone( bool );
    void signalHistogramComputationFailed( void );

public slots:

    void slotMinValueChanged( int min );
    void slotMaxValueChanged( int max );

protected slots:

    void slotBlinkTimerDone( void );
    void slotCalculationStarted(const ImageHistogram *histogram);
    void slotCalculationFinished(const ImageHistogram *histogram, bool success);

protected:

    void paintEvent( QPaintEvent * );
    void mousePressEvent ( QMouseEvent * e );
    void mouseReleaseEvent ( QMouseEvent * e );
    void mouseMoveEvent ( QMouseEvent * e );

private:

    void notifyValuesChanged();
    void connectHistogram(const ImageHistogram *histogram);

private:

    HistogramWidgetPriv* d;
};

#endif /* HISTOGRAMWIDGET_H */
