/* $Id: TwoDLinearWCS.cc,v 1.5 2013-08-07 15:28:51 cgarcia Exp $
 *
 * This file is part of the VIMOS Pipeline
 * Copyright (C) 2002-2012 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-08-07 15:28:51 $
 * $Revision: 1.5 $
 * $Name: not supported by cvs2svn $
 */

#include <cmath>
#include "TwoDLinearWCS.h"


#include "vimoswcs.h"
#include "vimoswcscat.h"


TwoDLinearWCS::TwoDLinearWCS()
{
    std::string  proj = "TAN";
    wcs_ = vimoswcsxinit(0.,0.,0.,0.,0.,0,0,0.,0,0.,
                         const_cast<char *>(proj.c_str()));
}

TwoDLinearWCS::TwoDLinearWCS
(const double& centralRA, const double& centralDEC, const double& plateScale,
 const cpl_size& dimX, const cpl_size& dimY, 
 const double& rotAngle, bool flip,
 const double& epoch, const int equinox)
{
    std::string  proj = "TAN";
    wcs_ = vimoswcsxinit(centralRA, centralDEC, plateScale, dimX/2., dimY/2., 
            dimX, dimY, rotAngle, equinox, epoch,
            const_cast<char *>(proj.c_str()));
    if(!flip) vimoswcsdeltset(wcs_, wcs_->cdelt[0],wcs_->cdelt[1],rotAngle);
    else      vimoswcsdeltset(wcs_,-wcs_->cdelt[0],wcs_->cdelt[1],rotAngle);
}

TwoDLinearWCS::~TwoDLinearWCS()
{
    if(wcs_ != NULL)
        vimoswcsfree(wcs_);
}

void TwoDLinearWCS::toWorld
(const double& xPixel, const double& yPixel, double& ra, double& dec) const
{
    pix2vimoswcs(wcs_, xPixel, yPixel, &ra, &dec);
    ra /= 15.;
}

bool TwoDLinearWCS::toPixel
(const double& ra, const double& dec, double& xPixel, double& yPixel) const
{
    int off;
    vimoswcs2pix(wcs_, ra * 15., dec, &xPixel, &yPixel, &off);
    if(off) return false;
    return true;
}

double TwoDLinearWCS::epoch() const
{
    return (wcs_->epoch);
}

double TwoDLinearWCS::equinox() const
{
    return (wcs_->equinox);
}

double TwoDLinearWCS::plateScale() const
{
    double xPixSize=std::fabs(((*wcs_).xinc)*3600.);
    double yPixSize=std::fabs(((*wcs_).yinc)*3600.);
    return ((xPixSize + yPixSize) / 2.);
}

double TwoDLinearWCS::crpix1() const
{
    return wcs_->crpix[0];
}

double TwoDLinearWCS::crpix2() const
{
    return wcs_->crpix[1];
}

double TwoDLinearWCS::crval1() const
{
    return wcs_->crval[0];
}

double TwoDLinearWCS::crval2() const
{
    return wcs_->crval[1];
}

double TwoDLinearWCS::cdelt1() const
{
    return wcs_->cdelt[0];
}

double TwoDLinearWCS::cdelt2() const
{
    return wcs_->cdelt[1];
}

std::string TwoDLinearWCS::ctype1() const
{
    return wcs_->ctype[0];
}

std::string TwoDLinearWCS::ctype2() const
{
    return wcs_->ctype[1];
}

std::string TwoDLinearWCS::cunit1() const
{
    return wcs_->units[0];
}

std::string TwoDLinearWCS::cunit2() const
{
    return wcs_->units[1];
}

double * TwoDLinearWCS::cdMatrix() const
{
    return wcs_->cd;
}

double TwoDLinearWCS::orientation() const
{
    return wcs_->rot;
}
