#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <nss.h>
#include <p12.h>
#include <pk11pub.h>
#include <nspr.h>
#include <secder.h>
#include <secerr.h>
#include <secoid.h>

#include <cstring>
#include <stdio.h>
#include <iostream>
#include <unistd.h>

#include "nsscertextension.h"
#include "canlxx.h"

namespace AuthN {
namespace NSS {

#define PORT_ErrorToString(err) PR_ErrorToString((err), PR_LANGUAGE_I_DEFAULT)

#if NSS_VMAJOR <=3 && NSS_VMINOR <12
  // SEC_StringToOID does not appear on older nss version
  SECStatus SEC_StringToOID(PLArenaPool* pool, SECItem* to,
      const char* from, PRUint32 len);
#endif

  void PrintExtension(std::string& out, CERTCertExtension* ext);


//RFC 3820 and VOMS AC sequence
#define OIDT static const unsigned char
  /* RFC 3820 Proxy OID. (1 3 6 1 5 5 7 1 14)*/
  OIDT proxy[] = { 0x2B, 6, 1, 5, 5, 7, 1, 14 };
  OIDT anyLanguage[] = { 0x2B, 6, 1, 5, 5, 7, 21, 0 };//(1.3 .6.1.5.5.7.21.0)
  OIDT inheritAll[] = { 0x2B, 6, 1, 5, 5, 7, 21, 1 }; //(1.3.6.1.5.5.7.21.1)
  OIDT Independent[] = { 0x2B, 6, 1, 5, 5, 7, 21, 2 }; //(1.3.6.1.5.5.7.21.1)
  /* VOMS AC sequence OID. ()*/
  OIDT VOMS_acseq[] = { 0x2B, 6, 1, 4, 1, 0xBE, 0x45, 100, 100, 5 }; //(1.3.6.1.4.1.8005.100.100.5)
  // according to BER "Basic Encoding Ruls", 8005 is encoded as 0xBE 0x45

#define OI(x) { siDEROID, (unsigned char *)x, sizeof x }
#define ODN(oid,desc) { OI(oid), (SECOidTag)0, desc, CKM_INVALID_MECHANISM, INVALID_CERT_EXTENSION }
  static const SECOidData oids[] = {
    ODN(proxy,		"RFC 3820 proxy extension"),
    ODN(anyLanguage, 	"Any language"),
    ODN(inheritAll, 	"Inherit all"),
    ODN(Independent, 	"Independent"),
    ODN(VOMS_acseq,     "acseq"),
  };

  static const unsigned int numOids = (sizeof oids) / (sizeof oids[0]);
  SECOidTag tag_proxy, tag_anylang, tag_inheritall, tag_independent, tag_vomsacseq;

  SECStatus RegisterDynamicOids(Context& context) {

    SECStatus rv = SECSuccess;
    tag_proxy = SECOID_AddEntry(&oids[0]);
    if (tag_proxy == SEC_OID_UNKNOWN) {
      rv = SECFailure;
      context.Log(Context::LogError, "Failed to add RFC proxy OID");
    }
    else {
      context.LogFormat(Context::LogDebug, "Succeeded to add RFC proxy OID, tag %d is returned", tag_proxy);
    }

    tag_anylang = SECOID_AddEntry(&oids[1]);
    if (tag_anylang == SEC_OID_UNKNOWN) {
      rv = SECFailure;
      context.Log(Context::LogError, "Failed to add any language OID");
    }
    else {
      context.LogFormat(Context::LogDebug, "Succeeded to add any language OID, tag %d is returned", tag_anylang);
    }

    tag_inheritall = SECOID_AddEntry(&oids[2]);
    if (tag_inheritall == SEC_OID_UNKNOWN) {
      rv = SECFailure;
      context.Log(Context::LogError, "Failed to add any inheritall OID");
    }
    else {
      context.LogFormat(Context::LogDebug, "Succeeded to add inheritall OID, tag %d is returned", tag_inheritall); 
    }

    tag_independent = SECOID_AddEntry(&oids[3]);
    if (tag_independent == SEC_OID_UNKNOWN) {
      rv = SECFailure;
      context.Log(Context::LogError, "Failed to add independent OID");
    }
    else {
      context.LogFormat(Context::LogDebug, "Succeeded to add independent OID, tag %d is returned", tag_independent);
    }

    tag_vomsacseq = SECOID_AddEntry(&oids[4]);
    if (tag_vomsacseq == SEC_OID_UNKNOWN) {
      rv = SECFailure;
      context.Log(Context::LogError, "Failed to add VOMS AC sequence OID");
    }
    else {
      context.LogFormat(Context::LogDebug, "Succeeded to add VOMS AC sequence OID, tag %d is returned", tag_vomsacseq);
    }

    return rv;
  }


//-----------Handle the Proxy--------//
  //With pathlen and with policy
  struct ProxyPolicy1 {
    SECItem policylanguage;
    SECItem policy;
  };
  struct ProxyCertInfo1 {
    PLArenaPool *arena;
    SECItem pathlength;
    ProxyPolicy1 proxypolicy;
  };
  const SEC_ASN1Template ProxyPolicyTemplate1[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyPolicy1) },
    { SEC_ASN1_OBJECT_ID,
          offsetof(ProxyPolicy1, policylanguage), NULL, 0 },
    { SEC_ASN1_OCTET_STRING,
          offsetof(ProxyPolicy1, policy), NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyPolicyTemplate1)
  const SEC_ASN1Template ProxyCertInfoTemplate1[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyCertInfo1) },
    { SEC_ASN1_INTEGER,
          offsetof(ProxyCertInfo1, pathlength), NULL, 0 },
    { SEC_ASN1_INLINE,
          offsetof(ProxyCertInfo1, proxypolicy),
          ProxyPolicyTemplate1, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyCertInfoTemplate1)

  //With pathlen and without policy
  struct ProxyPolicy2 {
    SECItem policylanguage;
  };
  struct ProxyCertInfo2 {
    PLArenaPool *arena;
    SECItem pathlength;
    ProxyPolicy2 proxypolicy;
  };
  const SEC_ASN1Template ProxyPolicyTemplate2[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyPolicy2) },
    { SEC_ASN1_OBJECT_ID,
          offsetof(ProxyPolicy2, policylanguage), NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyPolicyTemplate2)
  const SEC_ASN1Template ProxyCertInfoTemplate2[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyCertInfo2) },
    { SEC_ASN1_INTEGER,
          offsetof(ProxyCertInfo2, pathlength), NULL, 0 },
    { SEC_ASN1_INLINE,
          offsetof(ProxyCertInfo2, proxypolicy),
          ProxyPolicyTemplate2, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyCertInfoTemplate2)

  //Without pathlen and with policy
  struct ProxyPolicy3 {
    SECItem policylanguage;
    SECItem policy;
  };
  struct ProxyCertInfo3 {
    PLArenaPool *arena;
    ProxyPolicy3 proxypolicy;
  };
  const SEC_ASN1Template ProxyPolicyTemplate3[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyPolicy3) },
    { SEC_ASN1_OBJECT_ID,
          offsetof(ProxyPolicy3, policylanguage), NULL, 0 },
    { SEC_ASN1_OCTET_STRING,
          offsetof(ProxyPolicy3, policy), NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyPolicyTemplate3)
  const SEC_ASN1Template ProxyCertInfoTemplate3[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyCertInfo3) },
    { SEC_ASN1_INLINE,
          offsetof(ProxyCertInfo3, proxypolicy),
          ProxyPolicyTemplate3, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyCertInfoTemplate3)

  //Without pathlen and without policy
  struct ProxyPolicy4 {
    SECItem policylanguage;
  };
  struct ProxyCertInfo4 {
    PLArenaPool *arena;
    ProxyPolicy4 proxypolicy;
  };
  const SEC_ASN1Template ProxyPolicyTemplate4[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyPolicy4) },
    { SEC_ASN1_OBJECT_ID,
          offsetof(ProxyPolicy4, policylanguage), NULL, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyPolicyTemplate4)
  const SEC_ASN1Template ProxyCertInfoTemplate4[] = {
    { SEC_ASN1_SEQUENCE,
          0, NULL, sizeof(ProxyCertInfo4) },
    { SEC_ASN1_INLINE,
          offsetof(ProxyCertInfo4, proxypolicy),
          ProxyPolicyTemplate4, 0 },
    { 0, 0, NULL, 0 }
  };
  SEC_ASN1_CHOOSER_IMPLEMENT(ProxyCertInfoTemplate4)

  SECStatus EncodeProxyCertInfoExtension1(PRArenaPool *arena, 
      ProxyCertInfo1* info, SECItem* dest) {
    SECStatus rv = SECSuccess;
    PORT_Assert(info != NULL && dest != NULL);
    if(info == NULL || dest == NULL) {
      return SECFailure;
    }
    if(SEC_ASN1EncodeItem(arena, dest, info, SEC_ASN1_GET(ProxyCertInfoTemplate1)) == NULL) {
      rv = SECFailure;
    }
    return(rv);
  }

  SECStatus EncodeProxyCertInfoExtension2(PRArenaPool *arena,
      ProxyCertInfo2* info, SECItem* dest) {
    SECStatus rv = SECSuccess;
    PORT_Assert(info != NULL && dest != NULL);
    if(info == NULL || dest == NULL) {
      return SECFailure;
    }
    if(SEC_ASN1EncodeItem(arena, dest, info, SEC_ASN1_GET(ProxyCertInfoTemplate2)) == NULL) {
      rv = SECFailure;
    }
    return(rv);
  }

  SECStatus EncodeProxyCertInfoExtension3(PRArenaPool *arena,
      ProxyCertInfo3* info, SECItem* dest) {
    SECStatus rv = SECSuccess;
    PORT_Assert(info != NULL && dest != NULL);
    if(info == NULL || dest == NULL) {
      return SECFailure;
    }
    if(SEC_ASN1EncodeItem(arena, dest, info, SEC_ASN1_GET(ProxyCertInfoTemplate3)) == NULL) {
      rv = SECFailure;
    }
    return(rv);
  }

  SECStatus EncodeProxyCertInfoExtension4(PRArenaPool *arena,
      ProxyCertInfo4* info, SECItem* dest) {
    SECStatus rv = SECSuccess;
    PORT_Assert(info != NULL && dest != NULL);
    if(info == NULL || dest == NULL) {
      return SECFailure;
    }
    if(SEC_ASN1EncodeItem(arena, dest, info, SEC_ASN1_GET(ProxyCertInfoTemplate4)) == NULL) {
      rv = SECFailure;
    }
    return(rv);
  }

  typedef SECStatus (* EXTEN_EXT_VALUE_ENCODER) (PRArenaPool *extHandleArena,
                                               void *value, SECItem *encodedValue);

  SECStatus EncodeAndAddExtensionValue(PRArenaPool *arena, void *extHandle,
      void *value, PRBool criticality, int extenType,
      EXTEN_EXT_VALUE_ENCODER EncodeValueFn) {
    SECItem encodedValue;
    SECStatus rv;

    encodedValue.data = NULL;
    encodedValue.len = 0;
    do {
      rv = (*EncodeValueFn)(arena, value, &encodedValue);
      if (rv != SECSuccess)
        break;
      rv = CERT_AddExtension(extHandle, extenType, &encodedValue,
                               criticality, PR_TRUE);
      if (rv != SECSuccess)
        break;
    } while (0);

    return (rv);
  }

  SECStatus AddProxyCertInfoExtension(Context& context, 
      void* extHandle, int pathlen, char* policylang, char* policy) {
    PRArenaPool *arena = NULL;
    SECStatus rv = SECSuccess;
    SECOidData* oid = NULL;
    SECItem policy_item;
    std::string pl_lang;
    SECOidTag tag;
    void* mark;

    pl_lang = policylang;
    if(pl_lang == "Any language") tag = tag_anylang;
    else if(pl_lang == "Inherit all") tag = tag_inheritall;
    else if(pl_lang == "Independent") tag = tag_independent;
    else { context.LogFormat(Context::LogError, "The policy language: %s is not supported", policylang); goto error; }
    oid = SECOID_FindOIDByTag(tag);

    if((policy != NULL) && (pathlen != -1)) {
      ProxyCertInfo1* proxy_certinfo = NULL;
      ProxyPolicy1* proxy_policy = NULL;
    
      arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
      if (!arena ) {
        context.Log(Context::LogError, "Failed to new arena");
        return SECFailure;
      }
      proxy_certinfo = PORT_ArenaZNew(arena, ProxyCertInfo1);
      if ( proxy_certinfo== NULL) {
       return SECFailure;
      }
      proxy_certinfo->arena = arena;
      if((pathlen != 0) && (SEC_ASN1EncodeInteger(arena, &proxy_certinfo->pathlength, pathlen) == NULL)) {
        context.Log(Context::LogError, "Failed to create pathlen"); goto error;
      }
      if (oid == NULL || SECITEM_CopyItem(arena, &proxy_certinfo->proxypolicy.policylanguage, &oid->oid) == SECFailure) {
        context.Log(Context::LogError, "Failed to create policy lang"); goto error;       
      }
      proxy_certinfo->proxypolicy.policy.len = PORT_Strlen(policy);
      proxy_certinfo->proxypolicy.policy.data = (unsigned char*)PORT_ArenaStrdup(arena, policy);

      rv = EncodeAndAddExtensionValue(arena, extHandle, proxy_certinfo, 
        PR_TRUE, tag_proxy, (EXTEN_EXT_VALUE_ENCODER)EncodeProxyCertInfoExtension1);
    }
    else if((policy == NULL) && (pathlen != -1)) {
      ProxyCertInfo2* proxy_certinfo = NULL;
      ProxyPolicy2* proxy_policy = NULL;

      arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
      if (!arena ) {
        context.Log(Context::LogError, "Failed to new arena");
        return SECFailure;
      }
      proxy_certinfo = PORT_ArenaZNew(arena, ProxyCertInfo2);
      if ( proxy_certinfo== NULL) {
       return SECFailure;
      }
      proxy_certinfo->arena = arena;
      if((pathlen != -1) && (SEC_ASN1EncodeInteger(arena, &proxy_certinfo->pathlength, pathlen) == NULL)) {
        context.Log(Context::LogError, "Failed to create pathlen"); goto error;
      }
      if (oid == NULL || SECITEM_CopyItem(arena, &proxy_certinfo->proxypolicy.policylanguage, &oid->oid) == SECFailure) {
        context.Log(Context::LogError, "Failed to create policy lang"); goto error;
      }

      rv = EncodeAndAddExtensionValue(arena, extHandle, proxy_certinfo, 
        PR_TRUE, tag_proxy, (EXTEN_EXT_VALUE_ENCODER)EncodeProxyCertInfoExtension2);
    }
    else if((policy != NULL) && (pathlen == -1)) {
      ProxyCertInfo3* proxy_certinfo = NULL;
      ProxyPolicy3* proxy_policy = NULL;

      arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
      if (!arena ) {
        context.Log(Context::LogError, "Failed to new arena");
        return SECFailure;
      }
      proxy_certinfo = PORT_ArenaZNew(arena, ProxyCertInfo3);
      if ( proxy_certinfo== NULL) {
       return SECFailure;
      }
      proxy_certinfo->arena = arena;
      if (oid == NULL || SECITEM_CopyItem(arena, &proxy_certinfo->proxypolicy.policylanguage, &oid->oid) == SECFailure) {
        context.Log(Context::LogError, "Failed to create policy lang"); goto error;
      }
      proxy_certinfo->proxypolicy.policy.len = PORT_Strlen(policy);
      proxy_certinfo->proxypolicy.policy.data = (unsigned char*)PORT_ArenaStrdup(arena, policy);

      rv = EncodeAndAddExtensionValue(arena, extHandle, proxy_certinfo, 
        PR_TRUE, tag_proxy, (EXTEN_EXT_VALUE_ENCODER)EncodeProxyCertInfoExtension3);
    }
    else {
      ProxyCertInfo4* proxy_certinfo = NULL;
      ProxyPolicy4* proxy_policy = NULL;

      arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);
      if (!arena ) {
        context.Log(Context::LogError, "Failed to new arena");
        return SECFailure;
      }
      proxy_certinfo = PORT_ArenaZNew(arena, ProxyCertInfo4);
      if ( proxy_certinfo== NULL) {
       return SECFailure;
      }
      proxy_certinfo->arena = arena;
      if (oid == NULL || SECITEM_CopyItem(arena, &proxy_certinfo->proxypolicy.policylanguage, &oid->oid) == SECFailure) {
        context.Log(Context::LogError, "Failed to create policy lang"); goto error;
      }

      rv = EncodeAndAddExtensionValue(arena, extHandle, proxy_certinfo, 
        PR_TRUE, tag_proxy, (EXTEN_EXT_VALUE_ENCODER)EncodeProxyCertInfoExtension4);
    }

error:
    if (arena)
	PORT_FreeArena(arena, PR_FALSE);
    return (rv);
  }

  // Add a binary into extension, specific for the VOMS AC sequence
  SECStatus AddVOMSACSeqExtension(void* extHandle, char* vomsacseq, int length) {
    SECStatus rv = SECFailure;
    SECOidData* oid = NULL;
    SECOidTag tag;

    tag = tag_vomsacseq;
    oid = SECOID_FindOIDByTag(tag);

    if(vomsacseq != NULL) {
      SECItem encodedValue;
      encodedValue.data = (unsigned char*)vomsacseq;
      encodedValue.len = length;
      rv = CERT_AddExtension(extHandle, tag, &encodedValue,
                               PR_FALSE, PR_TRUE);
    }

    return (rv);
  }

  static CERTCertExtension* GetCertExtension(CERTCertExtension** extensions, SECItem* oid) {
    CERTCertExtension** exts;
    CERTCertExtension* ext = NULL;
    SECComparison comp;
    exts = extensions;

    if (exts) {
      while (*exts) {
        ext = *exts;
        comp = SECITEM_CompareItem(oid, &ext->id);
        if (comp == SECEqual) break;
        exts++;
      }
      return (*exts ? ext : NULL);
    }
    return NULL;
  }

  static CERTCertExtension* GetCertExtension(CERTCertExtension** extensions, 
      const std::string& oid_str) {
    CERTCertExtension* ext = NULL;
    SECItem oid;
    SECStatus rv = SECFailure;
    PRArenaPool* arena = NULL;

    arena = PORT_NewArena(256);
    // if (SECITEM_AllocItem(NULL, &oid, 256) == NULL) return NULL;
    rv = SEC_StringToOID(arena, &oid, (char*)(oid_str.c_str()), oid_str.length());
    if(rv == SECSuccess) {
      ext = GetCertExtension(extensions, &oid);
    }

    PORT_FreeArena(arena, PR_FALSE);
    return ext;
  }

  static void OIDToString(SECItem* oid, std::string& oid_str) {
    char* oids = NULL;
    oids = CERT_GetOidString(oid);
    if(oids != NULL) { 
      oid_str.assign(oids);
      PR_smprintf_free(oids);
    }
  }

  SECStatus GetExtension(const std::string& cert_name,
      const std::string& oid_str, AuthN::Credentials::Extension& extension) {
    SECStatus rv;
    CERTCertificate* cert = NULL;
    CERTCertDBHandle* handle;
    CERTCertExtension* ext = NULL;

    handle = CERT_GetDefaultCertDB();
    cert = CERT_FindCertByNicknameOrEmailAddr(handle, (char*)(cert_name.c_str()));
    if(!cert) {
      return SECFailure;
    }
 
    ext = GetCertExtension(cert->extensions, oid_str);
    if(ext != NULL) {
      OIDToString(&(ext->id), extension.oid);
      extension.critical = (ext->critical.data && ext->critical.data[0] == 0xff) ? true : false;
      extension.value.assign((const char*)(ext->value.data));

      std::string log_out;
      PrintExtension(log_out, ext);
      std::cout<<log_out<<std::endl;
    }
   
    CERT_DestroyCertificate(cert); 
    return SECSuccess;
  }

  SECStatus GetExtension(const std::string& cert_name, int pos,
      AuthN::Credentials::Extension& extension) {
    SECStatus rv;
    CERTCertificate* cert = NULL;
    CERTCertDBHandle* handle;
    CERTCertExtension* ext = NULL;
    CERTCertExtension** exts = NULL;
    int index = 0;

    handle = CERT_GetDefaultCertDB();
    cert = CERT_FindCertByNicknameOrEmailAddr(handle, (char*)(cert_name.c_str()));
    if(!cert) {
      return SECFailure;
    }

    exts = cert->extensions;
    if (exts) {
      while (*exts) {
        ext = *exts;
        if(index++ == pos) break;
        exts++;
      }
      if(*exts == NULL) ext = NULL;
    }

    if(ext != NULL) {
      OIDToString(&(ext->id), extension.oid);
      extension.critical = (ext->critical.data && ext->critical.data[0] == 0xff) ? true : false;
      extension.value.assign((const char*)(ext->value.data));

      std::string log_out;
      PrintExtension(log_out, ext);
      std::cout<<log_out<<std::endl;
    }

    CERT_DestroyCertificate(cert);
    if(ext == NULL) return SECFailure;
    return SECSuccess;
  }




/*********************/
/* The following code is for print an certificate extension.
 * The code is basically some duplication of mozilla nss code:
 * mozilla/security/nss/cmd/lib/secutil.c
 * Due to the extension print code has not been exposed by mozilla nss,
 * we have to duplicated the piece of code here.
 * Therefore the original license (MPL 1.1/GPL 2.0/LGPL 2.1) applies here.
 */
/*********************/

  void PrintAny(std::string& out, SECItem* i, char* m, int level);

  SECOidTag PrintObjectID(std::string& out, SECItem* oid, char* m, int level);

  void PrintString(std::string& out, SECItem *si, char *m, int level);

  static PRBool wrapEnabled = PR_TRUE;

#define INDENT_MULT     4
  void Indent(std::string& out, int level) {
    int i;
    for(i = 0; i < level; i++) {
      out.append("    ");
    }
  }

  void PrintErrMsg(std::string& out, int level, char *progName, char *msg, ...) {
    char buffer[256];
    memset(buffer, 0, 256);
    va_list args;
    PRErrorCode err = PORT_GetError();
    const char * errString = PORT_ErrorToString(err);

    va_start(args, msg);

    Indent(out, level);
    out.append(progName).append(": ");
    vsprintf(buffer, msg, args);
    out.append(buffer);
    if (errString != NULL && PORT_Strlen(errString) > 0)
      out.append(": ").append(errString).append("\n");
    else {
      out.append(": error "); out.push_back((int)err);
      out.append("\n");
    }
    va_end(args);
  }

  std::string int2string(int in, char* format) {
    char stemp[sizeof(int)];
    snprintf(stemp, sizeof(int),  format, in);
    std::string ret; ret.append(stemp);
    return ret;
  }

  void PrintAsHex(std::string& out, SECItem* data, const char* m, int level) {
    unsigned i;
    int column;
    PRBool isString     = PR_TRUE;
    PRBool isWhiteSpace = PR_TRUE;
    PRBool printedHex   = PR_FALSE;
    unsigned int limit = 15;

    if(m) {
      Indent(out, level); out.append(m).append(":");
      level++;
      if (wrapEnabled)
        out.append("\n");
    }

    if(wrapEnabled) {
      Indent(out, level); column = level*INDENT_MULT;
    }
    if(!data->len) {
      out.append("(empty)\n");
      return;
    }
    /* take a pass to see if it's all printable. */
    for (i = 0; i < data->len; i++) {
      unsigned char val = data->data[i];
      if (!val || !isprint(val)) {
        isString = PR_FALSE;
        break;
      }
      if (isWhiteSpace && !isspace(val)) {
        isWhiteSpace = PR_FALSE;
      }
    }

    /* Short values, such as bit strings (which are printed with this
    ** function) often look like strings, but we want to see the bits.
    ** so this test assures that short values will be printed in hex,
    ** perhaps in addition to being printed as strings.
    ** The threshold size (4 bytes) is arbitrary.
    */
    if (!isString || data->len <= 4) {
      for (i = 0; i < data->len; i++) {
        if (i != data->len - 1) {
          char* format = (char*)"%02x";
          out.append(int2string(data->data[i], format)).append(":");
          column += 3;
        } else {
          char* format = (char*)"%02x";
          out.append(int2string(data->data[i], format));
          column += 2;
          break;
        }
        if (wrapEnabled && (column > 76 || (i % 16 == limit))) {
          out.append("\n");
          Indent(out, level);
          column = level*INDENT_MULT;
          limit = i % 16;
        }
      }
      printedHex = PR_TRUE;
    }
    if(isString && !isWhiteSpace) {
      if(printedHex != PR_FALSE) {
        out.append("\n");
        Indent(out, level); column = level*INDENT_MULT;
      }
      for (i = 0; i < data->len; i++) {
        unsigned char val = data->data[i];
        if (val) {
          out.push_back(val);
          column++;
        } else {
          column = 77;
        }
        if (wrapEnabled && column > 76) {
          out.append("\n");
          Indent(out, level); column = level*INDENT_MULT;
        }
      }
    }

    if (column != level*INDENT_MULT) {
      out.append("\n");
    }
  }





  static const char printable[257] = {
        "................"      /* 0x */
        "................"      /* 1x */
        " !\"#$%&'()*+,-./"     /* 2x */
        "0123456789:;<=>?"      /* 3x */
        "@ABCDEFGHIJKLMNO"      /* 4x */
        "PQRSTUVWXYZ[\\]^_"     /* 5x */
        "`abcdefghijklmno"      /* 6x */
        "pqrstuvwxyz{|}~."      /* 7x */
        "................"      /* 8x */
        "................"      /* 9x */
        "................"      /* ax */
        "................"      /* bx */
        "................"      /* cx */
        "................"      /* dx */
        "................"      /* ex */
        "................"      /* fx */
  };


  SECStatus StripTagAndLength(SECItem* i) {
    unsigned int start;
    if (!i || !i->data || i->len < 2) { /* must be at least tag and length */
      return SECFailure;
    }
    start = ((i->data[1] & 0x80) ? (i->data[1] & 0x7f) + 2 : 2);
    if (i->len < start) {
      return SECFailure;
    }
    i->data += start;
    i->len  -= start;
    return SECSuccess;
  }

  void PrintInteger(std::string& out, SECItem* i, char* m, int level) {
    int iv;

    if (!i || !i->len || !i->data) {
      Indent(out, level); 
      if (m) {
        out.append(m).append(": (null)\n");
      } else {
        out.append("(null)\n"); 
      }
    } 
    else if (i->len > 4) {
      PrintAsHex(out, i, m, level);
    } else {
      if (i->type == siUnsignedInteger && *i->data & 0x80) {
        /* Make sure i->data has zero in the highest bite 
         * if i->data is an unsigned integer */
        SECItem tmpI;
        char data[] = {0, 0, 0, 0, 0};

        PORT_Memcpy(data + 1, i->data, i->len);
        tmpI.len = i->len + 1;
        tmpI.data = (unsigned char*)data;
        iv = DER_GetInteger(&tmpI);
      } else {
        iv = DER_GetInteger(i);
      }
      Indent(out, level); 
      if (m) {
        char* format1 = (char*)"%d";
        char* format2 = (char*)"%x";
        out.append(m).append(": ").append(int2string(iv, format1)).
            append(" (0x").append(int2string(iv, format2)).append(")\n");
      } else {
        char* format1 = (char*)"%d";
        char* format2 = (char*)"%x";        
        out.append(int2string(iv, format1)).            
            append(" (0x").append(int2string(iv, format2)).append(")\n");
      }
    }
  }

  void PrintRawString(std::string& out, SECItem* si, 
      const char* m, int level) {
    int column;
    unsigned int i;

    if(m) { 
      Indent(out, level); out.append(m).append(": ");
      column = (level * INDENT_MULT) + strlen(m) + 2;
      level++;
    } else {
      Indent(out, level);
      column = level*INDENT_MULT;
    }
    out.append("\""); column++;

    for(i = 0; i < si->len; i++) {
      unsigned char val = si->data[i];
      if(wrapEnabled && column > 76) {
        out.append("\n");
        Indent(out, level); column = level*INDENT_MULT;
      }
 
      out.push_back(printable[val]); column++;
    }

    out.append("\""); column++;

    if (wrapEnabled && (column != level*INDENT_MULT || column > 76)) {
      out.append("\n");
    }
  }

  void PrintBoolean(std::string& out, SECItem *i, const char *m, int level) {
    int val = 0;
    
    if ( i->data && i->len ) {
      val = i->data[0];
    }

    if (!m) {
      m = "Boolean";
    }
    Indent(out, level); 
    out.append(m).append(": ").append(val ? "True" : "False").append("\n");
  }

/*
 * Format and print "time".  If the tag message "m" is not NULL,
 * do indent formatting based on "level" and add a newline afterward;
 * otherwise just print the formatted time string only.
 */
  void PrintTime(std::string& out, int64 time, char *m, int level) {
    PRExplodedTime printableTime; 
    char *timeString;

    /* Convert to local time */
    PR_ExplodeTime(time, PR_GMTParameters, &printableTime);

    timeString = (char*)PORT_Alloc(256);
    if (timeString == NULL) return;

    if (m != NULL) {
      Indent(out, level);
      out.append(m).append(": ");
    }

    if (PR_FormatTime(timeString, 256, "%a %b %d %H:%M:%S %Y", &printableTime)) {
      out.append(timeString);
    }

    if (m != NULL) out.append("\n");

    PORT_Free(timeString);
  }

/*
 * Format and print the UTC Time "t".  If the tag message "m" is not NULL,
 * do indent formatting based on "level" and add a newline afterward;
 * otherwise just print the formatted time string only.
 */
  void PrintUTCTime(std::string& out, SECItem *t, char *m, int level) {
    int64 time;
    SECStatus rv;

    rv = DER_UTCTimeToTime(&time, t);
    if (rv != SECSuccess)
	return;

    PrintTime(out, time, m, level);
  }

/*
 * Format and print the Generalized Time "t".  If the tag message "m"
 * is not NULL, * do indent formatting based on "level" and add a newline
 * afterward; otherwise just print the formatted time string only.
 */
  void PrintGeneralizedTime(std::string& out, SECItem *t, char *m, int level) {
    int64 time;
    SECStatus rv;

    rv = DER_GeneralizedTimeToTime(&time, t);
    if (rv != SECSuccess)
	return;

    PrintTime(out, time, m, level);
  }

/*
 * Format and print the UTC or Generalized Time "t".  If the tag message
 * "m" is not NULL, do indent formatting based on "level" and add a newline
 * afterward; otherwise just print the formatted time string only.
 */
  void PrintTimeChoice(std::string& out, SECItem *t, char *m, int level) {
    switch (t->type) {
      case siUTCTime:
        PrintUTCTime(out, t, m, level);
        break;

      case siGeneralizedTime:
        PrintGeneralizedTime(out, t, m, level);
        break;

      default:
        PORT_Assert(0);
        break;
    }
  }

  /* This prints a SET or SEQUENCE */
  static void PrintSet(std::string& out, SECItem *t, char *m, int level) {
    int            type        = t->data[0] & SEC_ASN1_TAGNUM_MASK;
    int            constructed = t->data[0] & SEC_ASN1_CONSTRUCTED;
    const char *   label;
    SECItem        my          = *t;

    if (!constructed) {
      PrintAsHex(out, t, m, level);
      return;
    }
    if (SECSuccess != StripTagAndLength(&my))
      return;

    Indent(out, level);
    if (m) {
      out.append(m).append(": ");
    }

    if (type == SEC_ASN1_SET)
      label = "Set ";
    else if (type == SEC_ASN1_SEQUENCE)
      label = "Sequence ";
    else
      label = "";
    out.append(label).append("{\n");

    while (my.len >= 2) {
      SECItem  tmp = my;

      if (tmp.data[1] & 0x80) {
	unsigned int i;
        unsigned int lenlen = tmp.data[1] & 0x7f;
        if (lenlen > sizeof tmp.len)
          break;
        tmp.len = 0;
        for (i=0; i < lenlen; i++) {
          tmp.len = (tmp.len << 8) | tmp.data[2+i];
        }
        tmp.len += lenlen + 2;
      } else {
        tmp.len = tmp.data[1] + 2;
      }
      if (tmp.len > my.len) {
        tmp.len = my.len;
      }
      my.data += tmp.len;
      my.len  -= tmp.len;
      PrintAny(out, &tmp, NULL, level + 1);
    }
    Indent(out, level); out.append("}\n");
  }

  static void PrintContextSpecific(std::string& out, SECItem *i, char *m, int level) {
    int type        = i->data[0] & SEC_ASN1_TAGNUM_MASK;
    int constructed = i->data[0] & SEC_ASN1_CONSTRUCTED;
    SECItem tmp;

    if (constructed) {
      char * m2;
      if (!m) 
        m2 = PR_smprintf("[%d]", type);
      else
        m2 = PR_smprintf("%s: [%d]", m, type);
      if (m2) {
        PrintSet(out, i, m2, level);
        PR_smprintf_free(m2);
      }
      return;
    }

    Indent(out, level);
    if (m) {
      out.append(m).append(": ");
    }

    char* format = (char*)"%d";
    out.append("[").append(int2string(type, format)).append("]\n");

    tmp = *i;
    if (SECSuccess == StripTagAndLength(&tmp))
      PrintAsHex(out, &tmp, m, level+1);
  }

  static void PrintOctetString(std::string& out, SECItem *i, char *m, int level) {
    SECItem tmp = *i;
    if (SECSuccess == StripTagAndLength(&tmp))
      PrintAsHex(out, &tmp, m, level);
  }

  static void PrintBitString(std::string& out, SECItem *i, char *m, int level) {
    int unused_bits;
    SECItem tmp = *i;

    if (SECSuccess != StripTagAndLength(&tmp) || tmp.len < 2)
    	return;

    unused_bits = *tmp.data++;
    tmp.len--;

    PrintAsHex(out, &tmp, m, level);
    if (unused_bits) {
      Indent(out, level + 1);
      char* format = (char*)"%d";
      out.append("(").append(int2string(unused_bits, format)).append(" least significant bits unused)\n");
    }
  }

  /* in a decoded bit string, the len member is a bit length. */
  static void PrintDecodedBitString(std::string& out, SECItem *i, char *m, int level) {
    int unused_bits;
    SECItem tmp = *i;

    unused_bits = (tmp.len & 0x7) ? 8 - (tmp.len & 7) : 0;
    DER_ConvertBitString(&tmp); /* convert length to byte length */

    PrintAsHex(out, &tmp, m, level);
    if (unused_bits) {
      Indent(out, level + 1);
      char* format = (char*)"%d";
      out.append("(").append(int2string(unused_bits, format)).append(" least significant bits unused)\n");
    }
  }


  /* Print a DER encoded Boolean */
  static void PrintEncodedBoolean(std::string& out, SECItem *i, char *m, int level) {
    SECItem my    = *i;
    if (SECSuccess == StripTagAndLength(&my))
	PrintBoolean(out, &my, m, level);
  }

  /* Print a DER encoded integer */
  static void PrintEncodedInteger(std::string& out, SECItem *i, char *m, int level) {
    SECItem my    = *i;
    if (SECSuccess == StripTagAndLength(&my))
      PrintInteger(out, &my, m, level);
  }

  /* Print a DER encoded OID */
  static void PrintEncodedObjectID(std::string& out, SECItem *i, char *m, int level) {
    SECItem my    = *i;
    if (SECSuccess == StripTagAndLength(&my))
      PrintObjectID(out, &my, m, level);
  }

  static void PrintBMPString(std::string& out, SECItem *i, char *m, int level) {
    unsigned char * s;
    unsigned char * d;
    int      len;
    SECItem  tmp = {siBuffer, NULL, 0}; 
    SECItem  my  = *i;

    if (SECSuccess != StripTagAndLength(&my))
      goto error;
    if (my.len % 2) 
      goto error;
    len = (int)(my.len / 2);
    tmp.data = (unsigned char *)PORT_Alloc(len);
    if (!tmp.data)
      goto error;
    tmp.len = len;
    for (s = my.data, d = tmp.data ; len > 0; len--) {
      PRUint32 bmpChar = (s[0] << 8) | s[1]; s += 2;
      if (!isprint(bmpChar))
        goto error;
      *d++ = (unsigned char)bmpChar;
    }
    PrintRawString(out, &tmp, m, level);
    PORT_Free(tmp.data);
    return;

  error:
    PrintAsHex(out, i, m, level);
    if (tmp.data) PORT_Free(tmp.data);
  }


  static void PrintUniversalString(std::string& out, SECItem* i, char* m, int level) {
    unsigned char* s;
    unsigned char* d;
    int len;
    SECItem tmp = {siBuffer, NULL, 0};
    SECItem my  = *i;

    if(SECSuccess != StripTagAndLength(&my)) goto error;
    if(my.len % 4) goto error;
    len = (int)(my.len / 4);
    tmp.data = (unsigned char *)PORT_Alloc(len);
    if(!tmp.data) goto error;
    tmp.len = len;
    for(s = my.data, d = tmp.data; len > 0; len--) {
      PRUint32 bmpChar = (s[0] << 24) | (s[1] << 16) | (s[2] << 8) | s[3];
      s += 4;
      if(!isprint(bmpChar)) goto error;
      *d++ = (unsigned char)bmpChar;
    }
    PrintRawString(out, &tmp, m, level);
    PORT_Free(tmp.data);
    return;

  error:
    PrintAsHex(out, i, m, level);
    if(tmp.data) PORT_Free(tmp.data);
  }

  static void PrintUniversal(std::string& out, SECItem* i, char* m, int level) {
    switch (i->data[0] & SEC_ASN1_TAGNUM_MASK) {
      case SEC_ASN1_ENUMERATED:
      case SEC_ASN1_INTEGER:
        PrintEncodedInteger(out, i, m, level);
        break;
      case SEC_ASN1_OBJECT_ID:
        PrintEncodedObjectID(out, i, m, level);
        break;
      case SEC_ASN1_BOOLEAN:
        PrintEncodedBoolean(out, i, m, level);
        break;
      case SEC_ASN1_UTF8_STRING:
      case SEC_ASN1_PRINTABLE_STRING:
      case SEC_ASN1_VISIBLE_STRING:
      case SEC_ASN1_IA5_STRING:
      case SEC_ASN1_T61_STRING:
        PrintString(out, i, m, level);
        break;
      case SEC_ASN1_GENERALIZED_TIME:
        PrintGeneralizedTime(out, i, m, level);
        break;
      case SEC_ASN1_UTC_TIME:
        PrintUTCTime(out, i, m, level);
        break;
      case SEC_ASN1_NULL:
        Indent(out, level);
        if (m && m[0])
          out.append(m).append(": NULL\n");
        else
          out.append("NULL\n");
        break;
      case SEC_ASN1_SET:
      case SEC_ASN1_SEQUENCE:
        PrintSet(out, i, m, level);
        break;
      case SEC_ASN1_OCTET_STRING:
        PrintOctetString(out, i, m, level);
        break;
      case SEC_ASN1_BIT_STRING:
        PrintBitString(out, i, m, level);
        break;
      case SEC_ASN1_BMP_STRING:
        PrintBMPString(out, i, m, level);
        break;
      case SEC_ASN1_UNIVERSAL_STRING:
        PrintUniversalString(out, i, m, level);
        break;
      default:
        PrintAsHex(out, i, m, level);
        break;
    }
  }


  void PrintAny(std::string& out, SECItem* i, char* m, int level) {
    if(i && i->len && i->data) {
      switch (i->data[0] & SEC_ASN1_CLASS_MASK) {
      case SEC_ASN1_CONTEXT_SPECIFIC:
        PrintContextSpecific(out, i, m, level);
        break;
      case SEC_ASN1_UNIVERSAL:
        PrintUniversal(out, i, m, level);
        break;
      default:
        PrintAsHex(out, i, m, level);
        break;
      }
    }
  }


/************/

  static SECStatus PrintX509InvalidDate(std::string& out, 
    SECItem* value, char* msg, int level) {
    SECItem decodedValue;
    SECStatus rv;
    int64 invalidTime;
    char* formattedTime = NULL;

    decodedValue.data = NULL;
    rv = SEC_ASN1DecodeItem (NULL, &decodedValue,
         SEC_ASN1_GET(SEC_GeneralizedTimeTemplate), value);
    if(rv == SECSuccess) {
      rv = DER_GeneralizedTimeToTime(&invalidTime, &decodedValue);
      if(rv == SECSuccess) {
        formattedTime = CERT_GenTime2FormattedAscii
                (invalidTime, (char*)"%a %b %d %H:%M:%S %Y");
        Indent(out, level +1);
        out.append(msg).append(": ").append(formattedTime).append("\n");
        PORT_Free (formattedTime);
      }
    }
    PORT_Free (decodedValue.data);
    return (rv);
  }

  void PrintString(std::string& out, SECItem *si, char *m, int level) {
    SECItem my = *si;
    if (SECSuccess != StripTagAndLength(&my) || !my.len)
      return;
    PrintRawString(out, &my, m, level);
  }




  static const char * const nsTypeBits[] = {
    "SSL Client",
    "SSL Server",
    "S/MIME",
    "Object Signing",
    "Reserved",
    "SSL CA",
    "S/MIME CA",
    "ObjectSigning CA"
  };

  static SECStatus PrintNSCertType(std::string& out, SECItem* value, char* msg, int level) {
    int     unused;
    int     NS_Type;
    int     i;
    int     found   = 0;
    SECItem my      = *value;

    if ((my.data[0] != SEC_ASN1_BIT_STRING) ||
      SECSuccess != StripTagAndLength(&my)) {
      PrintAny(out, value, (char*)"Data", level);
      return SECSuccess;
    }

    unused = (my.len == 2) ? (my.data[0] & 0x0f) : 0;
    NS_Type = my.data[1] & (0xff << unused);

    Indent(out, level);
    if (msg) {
      out.append(msg).append(": ");
    } else {
      out.append("Netscape Certificate Type: ");
    }
    for (i=0; i < 8; i++) {
      if ( (0x80 >> i) & NS_Type) {
        out.append((const char*)(found ? ',' : '<')).append(nsTypeBits[i]);
        found = 1;
      }
    }
    out.append(found ? ">\n" : "none\n");
    return SECSuccess;
  }

  static SECStatus PrintBasicConstraints(std::string& out, 
      SECItem* value, char* msg, int level) {
    CERTBasicConstraints constraints;
    SECStatus rv;

    Indent(out, level);
    if (msg) {
      out.append(msg).append(": ");
    }
    rv = CERT_DecodeBasicConstraintValue(&constraints,value);
    if (rv == SECSuccess && constraints.isCA) {
      if(constraints.pathLenConstraint >= 0) {
        out.append("Is a CA with a maximum path length of ")
           .append(int2string(constraints.pathLenConstraint, (char*)"%d")).append(".\n");
      } 
      else {
        out.append("Is a CA with no maximum path length.\n");
      }
    } 
    else {
      out.append("Is not a CA.\n");
    }
    return SECSuccess;
  }

  SECOidTag PrintObjectID(std::string& out, SECItem* oid, char* m, int level) {
    SECOidData* oiddata;
    char* oidString = NULL;

    oiddata = SECOID_FindOID(oid);
    if (oiddata != NULL) {
      const char *name = oiddata->desc;
      Indent(out, level);
      if (m != NULL)
        out.append(m).append(": ");
      out.append(name).append("\n");
      return oiddata->offset;
    }
    oidString = CERT_GetOidString(oid);
    if (oidString) {
      Indent(out, level);
      if (m != NULL)
        out.append(m).append(": ");
      out.append(oidString).append("\n");
      PR_smprintf_free(oidString);
      return SEC_OID_UNKNOWN;
    }
    PrintAsHex(out, oid, m, level);
    return SEC_OID_UNKNOWN;
  }


  SECStatus PrintExtKeyUsageExtension(std::string& out, 
      SECItem* value, char* msg, int level) {
    CERTOidSequence* os;
    SECItem** op;

    os = CERT_DecodeOidSequence(value);
    if((CERTOidSequence *)NULL == os ) {
      return SECFailure;
    }

    for( op = os->oids; *op; op++ ) {
      PrintObjectID(out, *op, msg, level + 1);
    }
    CERT_DestroyOidSequence(os);
    return SECSuccess;
  }

  static const char * const usageBits[] = {
    "Digital Signature",   /* 0x80 */
    "Non-Repudiation",     /* 0x40 */
    "Key Encipherment",    /* 0x20 */
    "Data Encipherment",   /* 0x10 */
    "Key Agreement",       /* 0x08 */
    "Certificate Signing", /* 0x04 */
    "CRL Signing",         /* 0x02 */
    "Encipher Only",       /* 0x01 */
    "Decipher Only",       /* 0x0080 */ 
    NULL
  };

  static void PrintX509KeyUsage(std::string& out, SECItem* value, char* msg, int level) {
    int     unused;
    int     usage;
    int     i;
    int     found   = 0;
    SECItem my      = *value;

    if ((my.data[0] != SEC_ASN1_BIT_STRING) ||
      SECSuccess != StripTagAndLength(&my)) {
      PrintAny(out, value, (char*)"Data", level);
      return;
    }

    unused = (my.len >= 2) ? (my.data[0] & 0x0f) : 0;
    usage  = (my.len == 2) ? (my.data[1] & (0xff << unused)) << 8
                           : (my.data[1] << 8) |
                             (my.data[2] & (0xff << unused));

    Indent(out, level);
    out.append("Usages: ");
    for(i=0; usageBits[i]; i++) {
      if( (0x8000 >> i) & usage) {
        if (found)
          Indent(out, level + 2);
        out.append(usageBits[i]).append("\n");
        found = 1;
      }
    }
    if(!found) {
      out.append("(none)\n");
    }
  }

  static void PrintName(std::string& out, CERTName* name, const char* msg, int level) {
    char* nameStr = NULL;
    char* str;
    SECItem my;

    if (!name) {
      PORT_SetError(SEC_ERROR_INVALID_ARGS);
      return;
    }
    if (!name->rdns || !name->rdns[0]) {
      str = (char*)"(empty)";
    } else {
      str = nameStr = CERT_NameToAscii(name);
    }
    if (!str) {
      str = (char*)"!Invalid AVA!";
    }
    my.data = (unsigned char *)str;
    my.len  = PORT_Strlen(str);
    PrintRawString(out, &my, msg, level);

    //Indent(out, level); 
    //out.append(msg).append(": ").append(str).append("\n");

    PORT_Free(nameStr);
  }

  static void PrintIPAddress(std::string& out, SECItem *value, char *msg, int level) {
    PRStatus   st;
    PRNetAddr  addr;
    char       addrBuf[80];

    memset(&addr, 0, sizeof addr);
    if (value->len == 4) {
      addr.inet.family = PR_AF_INET;
      memcpy(&addr.inet.ip, value->data, value->len);
    } 
    else if (value->len == 16) {
      addr.ipv6.family = PR_AF_INET6;
      memcpy(addr.ipv6.ip.pr_s6_addr, value->data, value->len);
      if (PR_IsNetAddrType(&addr, PR_IpAddrV4Mapped)) {
        /* convert to IPv4.  */
	addr.inet.family = PR_AF_INET;
	memcpy(&addr.inet.ip, &addr.ipv6.ip.pr_s6_addr[12], 4);
	memset(&addr.inet.pad[0], 0, sizeof addr.inet.pad);
      }
    } 
    else {
      goto error;
    }

    st = PR_NetAddrToString(&addr, addrBuf, sizeof addrBuf);
    if(st == PR_SUCCESS) {
      Indent(out, level);
      out.append(msg).append(": ").append(addrBuf).append("\n");
    } 
    else {
  error:
      PrintAsHex(out, value, msg, level);
    }
  }

  static void PrintGeneralName(std::string& out, 
      CERTGeneralName* gname, char* msg, int level) {
    char label[40];
    if(msg && msg[0]) {
      Indent(out, level++); out.append(msg).append(": \n");
    }
    switch (gname->type) {
    case certOtherName :
      PrintAny(out, &gname->name.OthName.name, (char*)"Other Name", level);
      PrintObjectID(out, &gname->name.OthName.oid,  (char*)"OID",      level+1);
      break;
    case certDirectoryName :
      PrintName(out, &gname->name.directoryName, (char*)"Directory Name", level);
      break;
    case certRFC822Name :
      PrintRawString(out, &gname->name.other, (char*)"RFC822 Name", level);
      break;
    case certDNSName :
      PrintRawString(out, &gname->name.other, (char*)"DNS name", level);
	break;
    case certURI :
      PrintRawString(out, &gname->name.other, (char*)"URI", level);
      break;
    case certIPAddress :
      PrintIPAddress(out, &gname->name.other, (char*)"IP Address", level);
      break;
    case certRegisterID :
      PrintObjectID( out, &gname->name.other, (char*)"Registered ID", level);
      break;
    case certX400Address :
      PrintAny(out, &gname->name.other, (char*)"X400 Address", level);
      break;
    case certEDIPartyName :
      PrintAny(out, &gname->name.other, (char*)"EDI Party", level);
      break;
    default:
      PR_snprintf(label, sizeof label, "unknown type [%d]", 
	                                (int)gname->type - 1);
      PrintAsHex(out, &gname->name.other, label, level);
      break;
    }
  }

  static void PrintGeneralNames(std::string& out, 
      CERTGeneralName* gname, char* msg, int level) {
    CERTGeneralName *name = gname;
    do { 
      PrintGeneralName(out, name, msg, level);
      name = CERT_GetNextGeneralName(name);
    } while (name && name != gname);
  }  


  static void PrintAuthKeyIDExtension(std::string& out, 
      SECItem* value, char* msg, int level) {
    CERTAuthKeyID *kid  = NULL;
    PLArenaPool   *pool = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if (!pool) {
      std::string error;
      PrintErrMsg(error, 0, (char*)"Error", (char*)"Allocating new ArenaPool");
      std::cerr<<error<<std::endl;
      return;
    }
    kid = CERT_DecodeAuthKeyID(pool, value);
    if (!kid) {
      PrintErrMsg(out, level, (char*)"Error", (char*)"Parsing extension");
      PrintAny(out, value, (char*)"Data", level);
    } else {
      int keyIDPresent  = (kid->keyID.data && kid->keyID.len);
      int issuerPresent = kid->authCertIssuer != NULL;
      int snPresent = (kid->authCertSerialNumber.data &&
	               kid->authCertSerialNumber.len);
      if(keyIDPresent)
        PrintAsHex(out, &kid->keyID, (char*)"Key ID", level);
      if (issuerPresent)
        PrintGeneralName(out, kid->authCertIssuer, (char*)"Issuer", level);
      if (snPresent)
	PrintInteger(out, &kid->authCertSerialNumber, 
	             (char*)"Serial Number", level);
    }
    PORT_FreeArena(pool, PR_FALSE);
  }


  static void PrintAltNameExtension(std::string& out, 
      SECItem* value, char* msg, int level) {
    CERTGeneralName * nameList;
    CERTGeneralName * current;
    PLArenaPool     * pool      = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if (!pool) {
      std::string error;
      PrintErrMsg(error, level, (char*)"Error", (char*)"Allocating new ArenaPool");
      std::cerr<<error<<std::endl;
      return;
    }
    nameList = current = CERT_DecodeAltNameExtension(pool, value);
    if (!current) {
      if (PORT_GetError() == SEC_ERROR_EXTENSION_NOT_FOUND) {
        /* Decoder found empty sequence, which is invalid. */
	PORT_SetError(SEC_ERROR_EXTENSION_VALUE_INVALID);
      }
      PrintErrMsg(out, level, (char*)"Error", (char*)"Parsing extension");
      PrintAny(out, value, (char*)"Data", level);
    } else {
      do {
        PrintGeneralName(out, current, msg, level);
	current = CERT_GetNextGeneralName(current);
      } while (current != nameList);
    }
    PORT_FreeArena(pool, PR_FALSE);
  }

  static void PrintRDN(std::string& out, CERTRDN* rdn, const char* msg, int level) {
    CERTName name;
    CERTRDN *rdns[2];

    name.arena = NULL;
    name.rdns  = rdns;
    rdns[0] = rdn;
    rdns[1] = NULL;
    PrintName(out, &name, msg, level);
  }

  static void PrintCRLDistPtsExtension(std::string& out, 
      SECItem* value, char* msg, int level) {
    CERTCrlDistributionPoints * dPoints;
    PLArenaPool* pool = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if (!pool) {
      std::string error;
      PrintErrMsg(error, level, (char*)"Error", (char*)"Allocating new ArenaPool");
      std::cerr<<error<<std::endl;
      return;
    }
    dPoints = CERT_DecodeCRLDistributionPoints(pool, value);
    if (dPoints && dPoints->distPoints && dPoints->distPoints[0]) {
      CRLDistributionPoint ** pPoints = dPoints->distPoints;
      CRLDistributionPoint *  pPoint;
      while (NULL != (pPoint = *pPoints++)) {
        Indent(out, level); 
        out.append("Distribution point:\n");
        if (pPoint->distPointType == generalName && 
          pPoint->distPoint.fullName != NULL) {
          PrintGeneralNames(out, pPoint->distPoint.fullName, NULL, level + 1);
	} 
        else if (pPoint->distPointType == relativeDistinguishedName &&
	         pPoint->distPoint.relativeName.avas) {
          PrintRDN(out, &pPoint->distPoint.relativeName, "RDN", level + 1);
	} 
        else if (pPoint->derDistPoint.data) {
          PrintAny(out, &pPoint->derDistPoint, (char*)"Point", level + 1);
	}
	if (pPoint->reasons.data) {
          PrintDecodedBitString(out, &pPoint->reasons, (char*)"Reasons", level + 1);
	}
	if (pPoint->crlIssuer) {
          PrintGeneralName(out, pPoint->crlIssuer, (char*)"CRL issuer", level + 1);
        }
      }
    } else {
      PrintErrMsg(out, level, (char*)"Error", (char*)"Parsing extension");
      PrintAny(out, value, (char*)"Data", level);
    }
    PORT_FreeArena(pool, PR_FALSE);
  }

  static void PrintNameConstraintSubtree(std::string& out, 
      CERTNameConstraint *value, char *msg, int level) {
    CERTNameConstraint *head = value;
    Indent(out, level); out.append(msg).append(" Subtree:\n");
    level++;
    do {
      PrintGeneralName(out, &value->name, NULL, level);
      if (value->min.data)
        PrintInteger(out, &value->min, (char*)"Minimum", level+1);
      if (value->max.data)
        PrintInteger(out, &value->max, (char*)"Maximum", level+1);
	value = CERT_GetNextNameConstraint(value);
    } while (value != head);
  }

  static void PrintNameConstraintsExtension(std::string& out, 
      SECItem *value, char *msg, int level) {
    CERTNameConstraints * cnstrnts;
    PLArenaPool * pool = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if (!pool) {
      std::string error;
      PrintErrMsg(error, level, (char*)"Error", (char*)"Allocating new ArenaPool");
      std::cerr<<error<<std::endl;
      return;
    }
    cnstrnts = CERT_DecodeNameConstraintsExtension(pool, value);
    if (!cnstrnts) {
      PrintErrMsg(out, level, (char*)"Error", (char*)"Parsing extension");
      PrintAny(out, value, (char*)"Raw", level);
    } else {
      if (cnstrnts->permited)
        PrintNameConstraintSubtree(out, cnstrnts->permited, (char*)"Permitted", level);
      if (cnstrnts->excluded)
        PrintNameConstraintSubtree(out, cnstrnts->excluded, (char*)"Excluded", level);
    }
    PORT_FreeArena(pool, PR_FALSE);
  }


  static void PrintAuthorityInfoAcess(std::string& out, 
      SECItem *value, char *msg, int level) {
    CERTAuthInfoAccess **infos = NULL;
    PLArenaPool * pool = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if (!pool) {
        std::string error;
        PrintErrMsg(error, level, (char*)"Error", (char*)"Allocating new ArenaPool");
        std::cerr<<error<<std::endl;
	return;
    }
    infos = CERT_DecodeAuthInfoAccessExtension(pool, value);
    if (!infos) {
      PrintErrMsg(out, level, (char*)"Error", (char*)"Parsing extension");
      PrintAny(out, value, (char*)"Raw", level);
    } else {
      CERTAuthInfoAccess *info;
      while (NULL != (info = *infos++)) {
        if (info->method.data) {
          PrintObjectID(out, &info->method, (char*)"Method", level);
        } else {
          Indent(out,level);
          out.append("Error: missing method\n");
        }
        if (info->location) {
          PrintGeneralName(out, info->location, (char*)"Location", level);
        } else {
          PrintAny(out, &info->derLocation, (char*)"Location", level);
	}
      }
    }
    PORT_FreeArena(pool, PR_FALSE);
  }

  void PrintPrivKeyUsagePeriodExtension(std::string& out, 
      SECItem *value, char *msg, int level) {
    CERTPrivKeyUsagePeriod * prd;
    PLArenaPool * arena = PORT_NewArena(DER_DEFAULT_CHUNKSIZE);

    if ( !arena ) {
      goto error;
    }
    prd = CERT_DecodePrivKeyUsagePeriodExtension(arena, value);
    if (!prd) {
      goto error;
    }
    if (prd->notBefore.data) {
      PrintGeneralizedTime(out, &prd->notBefore, (char*)"Not Before", level);
    }
    if (prd->notAfter.data) {
      PrintGeneralizedTime(out, &prd->notAfter, (char*)"Not After ", level);
    }
    if (!prd->notBefore.data && !prd->notAfter.data) {
      Indent(out, level);
      out.append("Error: notBefore or notAfter MUST be present.\n");
  error:
      PrintAny(out, value, msg, level);
    }
    if (arena) {
      PORT_FreeArena(arena, PR_FALSE);
    }
  }

  void PrintExtension(std::string& out, CERTCertExtension* ext) {
    SECItem* ext_value_item;
    SECItem* ext_id_item = NULL;
    SECItem* ext_critical_item = NULL;
    SECOidTag tag;
    SECStatus rv = SECFailure;
    int level = 0;

    ext_value_item = &ext->value;
    ext_id_item = &ext->id;
    ext_critical_item = &ext->critical;

    PrintObjectID(out, ext_id_item, (char*)"Name", level);
    if (ext_critical_item->len ) {
      PrintBoolean(out, ext_critical_item, (char*)"Critical", level);
    }

    tag = SECOID_FindOIDTag(ext_id_item);
    switch(tag) {
      case SEC_OID_X509_INVALID_DATE:
      case SEC_OID_NS_CERT_EXT_CERT_RENEWAL_TIME:
        PrintX509InvalidDate(out, ext_value_item, (char*)"Date", level);
        break;
      case SEC_OID_X509_CERTIFICATE_POLICIES:
        // PrintPolicy(out, ext_value_item, (char*)"Data", level);
        break;
      case SEC_OID_NS_CERT_EXT_BASE_URL:
      case SEC_OID_NS_CERT_EXT_REVOCATION_URL:
      case SEC_OID_NS_CERT_EXT_CA_REVOCATION_URL:
      case SEC_OID_NS_CERT_EXT_CA_CRL_URL:
      case SEC_OID_NS_CERT_EXT_CA_CERT_URL:
      case SEC_OID_NS_CERT_EXT_CERT_RENEWAL_URL:
      case SEC_OID_NS_CERT_EXT_CA_POLICY_URL:
      case SEC_OID_NS_CERT_EXT_HOMEPAGE_URL:
      case SEC_OID_NS_CERT_EXT_LOST_PASSWORD_URL:
      case SEC_OID_OCSP_RESPONDER:
        PrintString(out, ext_value_item, (char*)"URL", level);
        break;
      case SEC_OID_NS_CERT_EXT_COMMENT:
        PrintString(out, ext_value_item, (char*)"Comment", level);
        break;
      case SEC_OID_NS_CERT_EXT_SSL_SERVER_NAME:
        PrintString(out, ext_value_item, (char*)"ServerName", level);
        break;
      case SEC_OID_NS_CERT_EXT_CERT_TYPE:
        PrintNSCertType(out, ext_value_item, (char*)"Data", level);
        break;
      case SEC_OID_X509_BASIC_CONSTRAINTS:
        PrintBasicConstraints(out, ext_value_item, (char*)"Data", level);
        break;
      case SEC_OID_X509_EXT_KEY_USAGE:
        PrintExtKeyUsageExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_KEY_USAGE:
        PrintX509KeyUsage(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_AUTH_KEY_ID:
        PrintAuthKeyIDExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_SUBJECT_ALT_NAME:
        case SEC_OID_X509_ISSUER_ALT_NAME:
        PrintAltNameExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_CRL_DIST_POINTS:
        PrintCRLDistPtsExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_PRIVATE_KEY_USAGE_PERIOD:
        PrintPrivKeyUsagePeriodExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_NAME_CONSTRAINTS:
        PrintNameConstraintsExtension(out, ext_value_item, NULL, level);
        break;
      case SEC_OID_X509_AUTH_INFO_ACCESS:
        PrintAuthorityInfoAcess(out, ext_value_item, NULL, level);
        break;

      case SEC_OID_X509_CRL_NUMBER:
      case SEC_OID_X509_REASON_CODE:

      /* PKIX OIDs */
      case SEC_OID_PKIX_OCSP:
      case SEC_OID_PKIX_OCSP_BASIC_RESPONSE:
      case SEC_OID_PKIX_OCSP_NONCE:
      case SEC_OID_PKIX_OCSP_CRL:
      case SEC_OID_PKIX_OCSP_RESPONSE:
      case SEC_OID_PKIX_OCSP_NO_CHECK:
      case SEC_OID_PKIX_OCSP_ARCHIVE_CUTOFF:
      case SEC_OID_PKIX_OCSP_SERVICE_LOCATOR:
      case SEC_OID_PKIX_REGCTRL_REGTOKEN:
      case SEC_OID_PKIX_REGCTRL_AUTHENTICATOR:
      case SEC_OID_PKIX_REGCTRL_PKIPUBINFO:
      case SEC_OID_PKIX_REGCTRL_PKI_ARCH_OPTIONS:
      case SEC_OID_PKIX_REGCTRL_OLD_CERT_ID:
      case SEC_OID_PKIX_REGCTRL_PROTOCOL_ENC_KEY:
      case SEC_OID_PKIX_REGINFO_UTF8_PAIRS:
      case SEC_OID_PKIX_REGINFO_CERT_REQUEST:

      /* Netscape extension OIDs. */
      case SEC_OID_NS_CERT_EXT_NETSCAPE_OK:
      case SEC_OID_NS_CERT_EXT_ISSUER_LOGO:
      case SEC_OID_NS_CERT_EXT_SUBJECT_LOGO:
      case SEC_OID_NS_CERT_EXT_ENTITY_LOGO:
      case SEC_OID_NS_CERT_EXT_USER_PICTURE:

      /* x.509 v3 Extensions */
      case SEC_OID_X509_SUBJECT_DIRECTORY_ATTR:
      case SEC_OID_X509_SUBJECT_KEY_ID:
      case SEC_OID_X509_POLICY_MAPPINGS:
      case SEC_OID_X509_POLICY_CONSTRAINTS:

      default:
        PrintAny(out, ext_value_item, (char*)"Data", level);
        break;
    }

  }

#if NSS_VMAJOR <=3 && NSS_VMINOR <12 
  SECStatus SEC_StringToOID(PLArenaPool* pool, SECItem* to, 
      const char* from, PRUint32 len) {
    PRUint32 decimal_numbers = 0;
    PRUint32 result_bytes = 0;
    SECStatus rv;
    PRUint8 result[1024];

    static const PRUint32 max_decimal = (0xffffffff / 10);
    static const char OIDstring[] = {"OID."};

    if (!from || !to) {
      PORT_SetError(SEC_ERROR_INVALID_ARGS);
      return SECFailure;
    }
    if (!len) {
      len = PL_strlen(from);
    }
    if (len >= 4 && !PL_strncasecmp(from, OIDstring, 4)) {
      from += 4; /* skip leading "OID." if present */
      len  -= 4;
    }
    if (!len) {
bad_data:
      PORT_SetError(SEC_ERROR_BAD_DATA);
      return SECFailure;
    }
    do {
      PRUint32 decimal = 0;
      while (len > 0 && isdigit(*from)) {
        PRUint32 addend = (*from++ - '0');
	--len;
	if (decimal > max_decimal)  /* overflow */
          goto bad_data;
        decimal = (decimal * 10) + addend;
        if (decimal < addend)	/* overflow */
          goto bad_data;
      }
      if (len != 0 && *from != '.') {
        goto bad_data;
      }
      if (decimal_numbers == 0) {
        if (decimal > 2)
          goto bad_data;
        result[0] = decimal * 40;
        result_bytes = 1;
      } else if (decimal_numbers == 1) {
        if (decimal > 40)
          goto bad_data;
        result[0] += decimal;
      } else {
        /* encode the decimal number,  */
        PRUint8 * rp;
        PRUint32 num_bytes = 0;
        PRUint32 tmp = decimal;
        while (tmp) {
          num_bytes++;
          tmp >>= 7;
        }
        if (!num_bytes )
          ++num_bytes;  /* use one byte for a zero value */
        if (num_bytes + result_bytes > sizeof result)
          goto bad_data;
        tmp = num_bytes;
        rp = result + result_bytes - 1;
        rp[tmp] = (PRUint8)(decimal & 0x7f);
        decimal >>= 7;
        while (--tmp > 0) {
          rp[tmp] = (PRUint8)(decimal | 0x80);
          decimal >>= 7;
        }
        result_bytes += num_bytes;
      }
      ++decimal_numbers;
      if (len > 0) { /* skip trailing '.' */
        ++from;
        --len;
      }
    } while (len > 0);
    /* now result contains result_bytes of data */
    if (to->data && to->len >= result_bytes) {
      PORT_Memcpy(to->data, result, to->len = result_bytes);
      rv = SECSuccess;
    } else {
      SECItem result_item = {siBuffer, NULL, 0 };
      result_item.data = result;
      result_item.len  = result_bytes;
      rv = SECITEM_CopyItem(pool, to, &result_item);
    }
    return rv;
  }
#endif

}
}
