//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Detector/SphericalDetectorItem.cpp
//! @brief     Implements class SphericalDetectorItem
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/Model/Detector/SphericalDetectorItem.h"
#include "Base/Const/Units.h"
#include "Device/Detector/SphericalDetector.h"
#include "ResolutionFunctionItems.h"

namespace {
namespace Tag {

const QString AlphaAxis("AlphaAxis");
const QString PhiAxis("PhiAxis");
const QString BaseData("BaseData");

} // namespace Tag

void initResolutionFunction(ResolutionFunctionItem* newFunc, const ResolutionFunctionItem*)
{
    newFunc->setUnit(Unit::degree);
}
} // namespace

SphericalDetectorItem::SphericalDetectorItem()
{
    m_resolutionFunction.initWithInitializer("Resolution function", "Detector resolution function",
                                             initResolutionFunction);

    m_phiAxis.initMin("Min", "Lower edge of first phi-bin", -1.0, Unit::degree,
                      RealLimits::limited(-90, 90));
    m_phiAxis.initMax("Max", "Upper edge of last phi-bin", 1.0, Unit::degree,
                      RealLimits::limited(-90, 90));

    m_alphaAxis.initMin("Min", "Lower edge of first alpha-bin", 0.0, Unit::degree,
                        RealLimits::limited(-90, 90));
    m_alphaAxis.initMax("Max", "Upper edge of last alpha-bin", 2.0, Unit::degree,
                        RealLimits::limited(-90, 90));
}

void SphericalDetectorItem::writeTo(QXmlStreamWriter* w) const
{
    XML::writeAttribute(w, XML::Attrib::version, uint(1));

    // parameters from base class
    w->writeStartElement(Tag::BaseData);
    DetectorItem::writeTo(w);
    w->writeEndElement();

    // phi axis
    w->writeStartElement(Tag::PhiAxis);
    m_phiAxis.writeTo(w);
    w->writeEndElement();

    // alpha axis
    w->writeStartElement(Tag::AlphaAxis);
    m_alphaAxis.writeTo(w);
    w->writeEndElement();
}

void SphericalDetectorItem::readFrom(QXmlStreamReader* r)
{
    const uint version = XML::readUIntAttribute(r, XML::Attrib::version);
    Q_UNUSED(version)

    while (r->readNextStartElement()) {
        QString tag = r->name().toString();

        // parameters from base class
        if (tag == Tag::BaseData) {
            DetectorItem::readFrom(r);
            m_resolutionFunction->setUnit(Unit::degree);
            XML::gotoEndElementOfTag(r, tag);

            // phi axis
        } else if (tag == Tag::PhiAxis) {
            m_phiAxis.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

            // alpha axis
        } else if (tag == Tag::AlphaAxis) {
            m_alphaAxis.readFrom(r);
            XML::gotoEndElementOfTag(r, tag);

        } else
            r->skipCurrentElement();
    }
}

std::unique_ptr<IDetector> SphericalDetectorItem::createDomainDetector() const
{
    const int n_x = m_phiAxis.nbins();
    const double x_min = Units::deg2rad(m_phiAxis.min());
    const double x_max = Units::deg2rad(m_phiAxis.max());

    const int n_y = m_alphaAxis.nbins();
    const double y_min = Units::deg2rad(m_alphaAxis.min());
    const double y_max = Units::deg2rad(m_alphaAxis.max());

    return std::make_unique<SphericalDetector>(n_x, x_min, x_max, n_y, y_min, y_max);
}

int SphericalDetectorItem::xSize() const
{
    return m_phiAxis.nbins();
}

int SphericalDetectorItem::ySize() const
{
    return m_alphaAxis.nbins();
}

void SphericalDetectorItem::setXSize(size_t nx)
{
    m_phiAxis.setNbins(nx);
}

void SphericalDetectorItem::setYSize(size_t ny)
{
    m_alphaAxis.setNbins(ny);
}

double SphericalDetectorItem::axesToCoreUnitsFactor() const
{
    return Units::deg;
}
