/*----------------------------------------------------------------------------*/
/** \file
 * \brief Aqsis implementation of the Renderman Interface version 3.2
 *
 * The Renderman Interface is Copyright (c) 1988 Pixar. All Rights reserved.
 *
 * ===================================================================
 * C-compatible header. C++ constructs must be preprocessor-protected.
 * ===================================================================
 */
/*----------------------------------------------------------------------------*/

#ifndef RI_H_INCLUDED
#define RI_H_INCLUDED

#include <aqsis/config.h>

#include <float.h>

#include <aqsis/ri/ritypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * RenderMan Interface Standard Include File
 */

#define RI_FALSE    0
#define RI_TRUE     1
#define RI_INFINITY FLT_MAX
#define RI_EPSILON  FLT_EPSILON
#define RI_NULL     ((RtToken)0)

#define RI_FLOATMIN FLT_MIN
#define RI_FLOATMAX FLT_MAX

#define RI_PI       3.14159265359f
#define RI_PIO2     RI_PI/2

#define RI_SHADER_EXTENSION ".slx"

/* Extern Declarations for Predefined RI Data Structures */

AQSIS_RI_SHARE  extern  RtToken RI_FRAMEBUFFER, RI_FILE;
AQSIS_RI_SHARE  extern  RtToken RI_RGB, RI_RGBA, RI_RGBZ, RI_RGBAZ, RI_A,
                                RI_Z, RI_AZ;
AQSIS_RI_SHARE  extern  RtToken RI_MERGE, RI_ORIGIN;
AQSIS_RI_SHARE  extern  RtToken RI_PERSPECTIVE, RI_ORTHOGRAPHIC;
AQSIS_RI_SHARE  extern  RtToken RI_HIDDEN, RI_PAINT;
AQSIS_RI_SHARE  extern  RtToken RI_CONSTANT, RI_SMOOTH;
AQSIS_RI_SHARE  extern  RtToken RI_FLATNESS, RI_FOV;

AQSIS_RI_SHARE  extern  RtToken RI_AMBIENTLIGHT, RI_POINTLIGHT, RI_DISTANTLIGHT,
                                RI_SPOTLIGHT;
AQSIS_RI_SHARE  extern  RtToken RI_INTENSITY, RI_LIGHTCOLOR, RI_FROM, RI_TO,
                                RI_CONEANGLE, RI_CONEDELTAANGLE,
                                RI_BEAMDISTRIBUTION;
AQSIS_RI_SHARE  extern  RtToken RI_MATTE, RI_METAL, RI_PLASTIC, RI_SHINYMETAL,
                                RI_PAINTEDPLASTIC;
AQSIS_RI_SHARE  extern  RtToken RI_KA, RI_KD, RI_KS, RI_ROUGHNESS, RI_KR,
                                RI_TEXTURENAME, RI_SPECULARCOLOR;
AQSIS_RI_SHARE  extern  RtToken RI_DEPTHCUE, RI_FOG, RI_BUMPY;
AQSIS_RI_SHARE  extern  RtToken RI_MINDISTANCE, RI_MAXDISTANCE, RI_BACKGROUND,
                                RI_DISTANCE, RI_AMPLITUDE;

AQSIS_RI_SHARE  extern  RtToken RI_RASTER, RI_SCREEN, RI_CAMERA, RI_WORLD,
                                RI_OBJECT;
AQSIS_RI_SHARE  extern  RtToken RI_INSIDE, RI_OUTSIDE, RI_LH, RI_RH;
AQSIS_RI_SHARE  extern  RtToken RI_P, RI_PZ, RI_PW, RI_N, RI_NP, RI_CS, RI_OS,
                                RI_S, RI_T, RI_ST;
AQSIS_RI_SHARE  extern  RtToken RI_BILINEAR, RI_BICUBIC;
AQSIS_RI_SHARE  extern  RtToken RI_LINEAR, RI_CUBIC;
AQSIS_RI_SHARE  extern  RtToken RI_PRIMITIVE, RI_INTERSECTION, RI_UNION,
                                RI_DIFFERENCE;
AQSIS_RI_SHARE  extern  RtToken RI_WRAP, RI_NOWRAP, RI_PERIODIC, RI_NONPERIODIC,
                                RI_CLAMP, RI_BLACK;
AQSIS_RI_SHARE  extern  RtToken RI_IGNORE, RI_PRINT, RI_ABORT, RI_HANDLER;
AQSIS_RI_SHARE  extern  RtToken RI_IDENTIFIER, RI_NAME;
AQSIS_RI_SHARE  extern  RtToken RI_COMMENT, RI_STRUCTURE, RI_VERBATIM;
AQSIS_RI_SHARE  extern  RtToken RI_WIDTH, RI_CONSTANTWIDTH;

AQSIS_RI_SHARE  extern  RtToken RI_CURRENT, RI_SHADER, RI_EYE, RI_NDC;

AQSIS_RI_SHARE  extern  RtBasis RiBezierBasis, RiBSplineBasis,
                                RiCatmullRomBasis, RiHermiteBasis,
                                RiPowerBasis;

/* Spline basis steps */
#define RI_BEZIERSTEP       ((RtInt)3)
#define RI_BSPLINESTEP      ((RtInt)1)
#define RI_CATMULLROMSTEP   ((RtInt)1)
#define RI_HERMITESTEP      ((RtInt)2)
#define RI_POWERSTEP        ((RtInt)4)

/* Aqsis-specific "matte with alpha" argument to RiMatte. */
#define RI_MATTEALPHA 2

AQSIS_RI_SHARE  extern  RtInt   RiLastError;

/* Declarations of All of the RenderMan Interface Subroutines */

#define PARAMETERLIST   RtInt count, RtToken tokens[], RtPointer values[]

/*
-------------------------------------------------------------------------------
Autogenerated procedure declarations.  These are all generated from ri.xml

Sections correspond to the RISpec version 3.2, with some extensions.
-------------------------------------------------------------------------------


[[[cog
from codegenutils import *

riXml = parseXml(riXmlPath)

declTemplate = 'AQSIS_RI_SHARE %s %s(%s);'
def printDecl(returnType, procName, args):
    cog.outl(wrapDecl(declTemplate % (returnType, procName,
                                      ', '.join(args)), 80))

for p in riXml.findall('Procedures/'+'*'):
    if p.tag == 'Section':
        cog.outl(); cog.outl()
        cog.outl(commentBannerC(p.text))
    elif p.tag == 'SubSection':
        cog.outl(commentBannerC(p.text, fillchar='-', width = 50))
    elif p.tag == 'SubSubSection':
        cog.outl('/'+'* '+ p.text + ' *'+'/')
    else:
        args = [formalArgC(arg) for arg in cArgs(p)]
        returnType = p.findtext('ReturnType')
        procName = cName(p)
        if p.findall('Arguments/ParamList'):
            printDecl(returnType, procName, args + ['...'])
            printDecl(returnType, procName + 'V', args +
                      ['RtInt count, RtToken tokens[], RtPointer values[]'])
        else:
            printDecl(returnType, procName, args)

]]]*/


/* ======== Relationship to the RenderMan Shading Language ======== */
AQSIS_RI_SHARE RtToken RiDeclare(RtString name, RtString declaration);


/* ======================== Graphics State ======================== */
AQSIS_RI_SHARE RtVoid RiBegin(RtToken name);
AQSIS_RI_SHARE RtVoid RiEnd();
AQSIS_RI_SHARE RtContextHandle RiGetContext();
AQSIS_RI_SHARE RtVoid RiContext(RtContextHandle handle);
AQSIS_RI_SHARE RtVoid RiFrameBegin(RtInt number);
AQSIS_RI_SHARE RtVoid RiFrameEnd();
AQSIS_RI_SHARE RtVoid RiWorldBegin();
AQSIS_RI_SHARE RtVoid RiWorldEnd();
/* ------------- Conditional RIB -------------- */
AQSIS_RI_SHARE RtVoid RiIfBegin(RtString condition);
AQSIS_RI_SHARE RtVoid RiElseIf(RtString condition);
AQSIS_RI_SHARE RtVoid RiElse();
AQSIS_RI_SHARE RtVoid RiIfEnd();
/* ----------------- Options ------------------ */
/* Camera */
AQSIS_RI_SHARE RtVoid RiFormat(RtInt xresolution, RtInt yresolution,
                               RtFloat pixelaspectratio);
AQSIS_RI_SHARE RtVoid RiFrameAspectRatio(RtFloat frameratio);
AQSIS_RI_SHARE RtVoid RiScreenWindow(RtFloat left, RtFloat right,
                                     RtFloat bottom, RtFloat top);
AQSIS_RI_SHARE RtVoid RiCropWindow(RtFloat xmin, RtFloat xmax, RtFloat ymin,
                                   RtFloat ymax);
AQSIS_RI_SHARE RtVoid RiProjection(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiProjectionV(RtToken name, RtInt count, RtToken tokens[],
                                    RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiClipping(RtFloat cnear, RtFloat cfar);
AQSIS_RI_SHARE RtVoid RiClippingPlane(RtFloat x, RtFloat y, RtFloat z,
                                      RtFloat nx, RtFloat ny, RtFloat nz);
AQSIS_RI_SHARE RtVoid RiDepthOfField(RtFloat fstop, RtFloat focallength,
                                     RtFloat focaldistance);
AQSIS_RI_SHARE RtVoid RiShutter(RtFloat opentime, RtFloat closetime);
/* Displays */
AQSIS_RI_SHARE RtVoid RiPixelVariance(RtFloat variance);
AQSIS_RI_SHARE RtVoid RiPixelSamples(RtFloat xsamples, RtFloat ysamples);
AQSIS_RI_SHARE RtVoid RiPixelFilter(RtFilterFunc function, RtFloat xwidth,
                                    RtFloat ywidth);
AQSIS_RI_SHARE RtVoid RiExposure(RtFloat gain, RtFloat gamma);
AQSIS_RI_SHARE RtVoid RiImager(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiImagerV(RtToken name, RtInt count, RtToken tokens[],
                                RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiQuantize(RtToken type, RtInt one, RtInt min, RtInt max,
                                 RtFloat ditheramplitude);
AQSIS_RI_SHARE RtVoid RiDisplay(RtToken name, RtToken type, RtToken mode, ...);
AQSIS_RI_SHARE RtVoid RiDisplayV(RtToken name, RtToken type, RtToken mode,
                                 RtInt count, RtToken tokens[],
                                 RtPointer values[]);
AQSIS_RI_SHARE RtFloat RiGaussianFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                        RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiBoxFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                   RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiMitchellFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                        RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiTriangleFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                        RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiCatmullRomFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                          RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiSincFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                    RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiDiskFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                    RtFloat ywidth);
AQSIS_RI_SHARE RtFloat RiBesselFilter(RtFloat x, RtFloat y, RtFloat xwidth,
                                      RtFloat ywidth);
/* Additional options */
AQSIS_RI_SHARE RtVoid RiHider(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiHiderV(RtToken name, RtInt count, RtToken tokens[],
                               RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiColorSamples(RtInt N, RtFloat nRGB[], RtFloat RGBn[]);
AQSIS_RI_SHARE RtVoid RiRelativeDetail(RtFloat relativedetail);
/* Implementation-specific options */
AQSIS_RI_SHARE RtVoid RiOption(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiOptionV(RtToken name, RtInt count, RtToken tokens[],
                                RtPointer values[]);
/* ---------------- Attributes ---------------- */
AQSIS_RI_SHARE RtVoid RiAttributeBegin();
AQSIS_RI_SHARE RtVoid RiAttributeEnd();
/* Color and opacity */
AQSIS_RI_SHARE RtVoid RiColor(RtColor Cq);
AQSIS_RI_SHARE RtVoid RiOpacity(RtColor Os);
/* Texture coordinates */
AQSIS_RI_SHARE RtVoid RiTextureCoordinates(RtFloat s1, RtFloat t1, RtFloat s2,
                                           RtFloat t2, RtFloat s3, RtFloat t3,
                                           RtFloat s4, RtFloat t4);
/* Light sources */
AQSIS_RI_SHARE RtLightHandle RiLightSource(RtToken shadername, ...);
AQSIS_RI_SHARE RtLightHandle RiLightSourceV(RtToken shadername, RtInt count,
                                            RtToken tokens[],
                                            RtPointer values[]);
AQSIS_RI_SHARE RtLightHandle RiAreaLightSource(RtToken shadername, ...);
AQSIS_RI_SHARE RtLightHandle RiAreaLightSourceV(RtToken shadername, RtInt count,
                                                RtToken tokens[],
                                                RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiIlluminate(RtLightHandle light, RtBoolean onoff);
/* Surface shading */
AQSIS_RI_SHARE RtVoid RiSurface(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiSurfaceV(RtToken name, RtInt count, RtToken tokens[],
                                 RtPointer values[]);
/* Displacement shading */
AQSIS_RI_SHARE RtVoid RiDeformation(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiDeformationV(RtToken name, RtInt count,
                                     RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiDisplacement(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiDisplacementV(RtToken name, RtInt count,
                                      RtToken tokens[], RtPointer values[]);
/* Volume shading */
AQSIS_RI_SHARE RtVoid RiAtmosphere(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiAtmosphereV(RtToken name, RtInt count, RtToken tokens[],
                                    RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiInterior(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiInteriorV(RtToken name, RtInt count, RtToken tokens[],
                                  RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiExterior(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiExteriorV(RtToken name, RtInt count, RtToken tokens[],
                                  RtPointer values[]);
/* Layered Shaders */
AQSIS_RI_SHARE RtVoid RiShaderLayer(RtToken type, RtToken name,
                                    RtToken layername, ...);
AQSIS_RI_SHARE RtVoid RiShaderLayerV(RtToken type, RtToken name,
                                     RtToken layername, RtInt count,
                                     RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiConnectShaderLayers(RtToken type, RtToken layer1,
                                            RtToken variable1, RtToken layer2,
                                            RtToken variable2);
/* Shading rate */
AQSIS_RI_SHARE RtVoid RiShadingRate(RtFloat size);
/* Shading interpolation */
AQSIS_RI_SHARE RtVoid RiShadingInterpolation(RtToken type);
/* Matte objects */
AQSIS_RI_SHARE RtVoid RiMatte(RtBoolean onoff);
/* Bound */
AQSIS_RI_SHARE RtVoid RiBound(RtBound bound);
/* Detail */
AQSIS_RI_SHARE RtVoid RiDetail(RtBound bound);
AQSIS_RI_SHARE RtVoid RiDetailRange(RtFloat offlow, RtFloat onlow,
                                    RtFloat onhigh, RtFloat offhigh);
/* Geometric approximation */
AQSIS_RI_SHARE RtVoid RiGeometricApproximation(RtToken type, RtFloat value);
/* Orientation and sides */
AQSIS_RI_SHARE RtVoid RiOrientation(RtToken orientation);
AQSIS_RI_SHARE RtVoid RiReverseOrientation();
AQSIS_RI_SHARE RtVoid RiSides(RtInt nsides);
/* ------------- Transformations -------------- */
AQSIS_RI_SHARE RtVoid RiIdentity();
AQSIS_RI_SHARE RtVoid RiTransform(RtMatrix transform);
AQSIS_RI_SHARE RtVoid RiConcatTransform(RtMatrix transform);
AQSIS_RI_SHARE RtVoid RiPerspective(RtFloat fov);
AQSIS_RI_SHARE RtVoid RiTranslate(RtFloat dx, RtFloat dy, RtFloat dz);
AQSIS_RI_SHARE RtVoid RiRotate(RtFloat angle, RtFloat dx, RtFloat dy,
                               RtFloat dz);
AQSIS_RI_SHARE RtVoid RiScale(RtFloat sx, RtFloat sy, RtFloat sz);
AQSIS_RI_SHARE RtVoid RiSkew(RtFloat angle, RtFloat dx1, RtFloat dy1,
                             RtFloat dz1, RtFloat dx2, RtFloat dy2,
                             RtFloat dz2);
/* Named coordinate systems */
AQSIS_RI_SHARE RtVoid RiCoordinateSystem(RtToken space);
AQSIS_RI_SHARE RtVoid RiCoordSysTransform(RtToken space);
AQSIS_RI_SHARE RtPoint* RiTransformPoints(RtToken fromspace, RtToken tospace,
                                          RtInt npoints, RtPoint points[]);
/* Transformation stack */
AQSIS_RI_SHARE RtVoid RiTransformBegin();
AQSIS_RI_SHARE RtVoid RiTransformEnd();
/* ---------------- Resources ----------------- */
AQSIS_RI_SHARE RtVoid RiResource(RtToken handle, RtToken type, ...);
AQSIS_RI_SHARE RtVoid RiResourceV(RtToken handle, RtToken type, RtInt count,
                                  RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiResourceBegin();
AQSIS_RI_SHARE RtVoid RiResourceEnd();
/* ---- Implementation-specific Attributes ---- */
AQSIS_RI_SHARE RtVoid RiAttribute(RtToken name, ...);
AQSIS_RI_SHARE RtVoid RiAttributeV(RtToken name, RtInt count, RtToken tokens[],
                                   RtPointer values[]);


/* ===================== Geometric Primitives ===================== */
/* ----------------- Polygons ----------------- */
AQSIS_RI_SHARE RtVoid RiPolygon(RtInt nvertices, ...);
AQSIS_RI_SHARE RtVoid RiPolygonV(RtInt nvertices, RtInt count, RtToken tokens[],
                                 RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiGeneralPolygon(RtInt nloops, RtInt nverts[], ...);
AQSIS_RI_SHARE RtVoid RiGeneralPolygonV(RtInt nloops, RtInt nverts[],
                                        RtInt count, RtToken tokens[],
                                        RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiPointsPolygons(RtInt npolys, RtInt nverts[],
                                       RtInt verts[], ...);
AQSIS_RI_SHARE RtVoid RiPointsPolygonsV(RtInt npolys, RtInt nverts[],
                                        RtInt verts[], RtInt count,
                                        RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiPointsGeneralPolygons(RtInt npolys, RtInt nloops[],
                                              RtInt nverts[], RtInt verts[],
                                              ...);
AQSIS_RI_SHARE RtVoid RiPointsGeneralPolygonsV(RtInt npolys, RtInt nloops[],
                                               RtInt nverts[], RtInt verts[],
                                               RtInt count, RtToken tokens[],
                                               RtPointer values[]);
/* ----------------- Patches ------------------ */
AQSIS_RI_SHARE RtVoid RiBasis(RtBasis ubasis, RtInt ustep, RtBasis vbasis,
                              RtInt vstep);
AQSIS_RI_SHARE RtVoid RiPatch(RtToken type, ...);
AQSIS_RI_SHARE RtVoid RiPatchV(RtToken type, RtInt count, RtToken tokens[],
                               RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiPatchMesh(RtToken type, RtInt nu, RtToken uwrap,
                                  RtInt nv, RtToken vwrap, ...);
AQSIS_RI_SHARE RtVoid RiPatchMeshV(RtToken type, RtInt nu, RtToken uwrap,
                                   RtInt nv, RtToken vwrap, RtInt count,
                                   RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiNuPatch(RtInt nu, RtInt uorder, RtFloat uknot[],
                                RtFloat umin, RtFloat umax, RtInt nv,
                                RtInt vorder, RtFloat vknot[], RtFloat vmin,
                                RtFloat vmax, ...);
AQSIS_RI_SHARE RtVoid RiNuPatchV(RtInt nu, RtInt uorder, RtFloat uknot[],
                                 RtFloat umin, RtFloat umax, RtInt nv,
                                 RtInt vorder, RtFloat vknot[], RtFloat vmin,
                                 RtFloat vmax, RtInt count, RtToken tokens[],
                                 RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiTrimCurve(RtInt nloops, RtInt ncurves[], RtInt order[],
                                  RtFloat knot[], RtFloat min[], RtFloat max[],
                                  RtInt n[], RtFloat u[], RtFloat v[],
                                  RtFloat w[]);
/* ----------- Subdivision Surfaces ----------- */
AQSIS_RI_SHARE RtVoid RiSubdivisionMesh(RtToken scheme, RtInt nfaces,
                                        RtInt nvertices[], RtInt vertices[],
                                        RtInt ntags, RtToken tags[],
                                        RtInt nargs[], RtInt intargs[],
                                        RtFloat floatargs[], ...);
AQSIS_RI_SHARE RtVoid RiSubdivisionMeshV(RtToken scheme, RtInt nfaces,
                                         RtInt nvertices[], RtInt vertices[],
                                         RtInt ntags, RtToken tags[],
                                         RtInt nargs[], RtInt intargs[],
                                         RtFloat floatargs[], RtInt count,
                                         RtToken tokens[], RtPointer values[]);
/* ----------------- Quadrics ----------------- */
AQSIS_RI_SHARE RtVoid RiSphere(RtFloat radius, RtFloat zmin, RtFloat zmax,
                               RtFloat thetamax, ...);
AQSIS_RI_SHARE RtVoid RiSphereV(RtFloat radius, RtFloat zmin, RtFloat zmax,
                                RtFloat thetamax, RtInt count, RtToken tokens[],
                                RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiCone(RtFloat height, RtFloat radius, RtFloat thetamax,
                             ...);
AQSIS_RI_SHARE RtVoid RiConeV(RtFloat height, RtFloat radius, RtFloat thetamax,
                              RtInt count, RtToken tokens[],
                              RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiCylinder(RtFloat radius, RtFloat zmin, RtFloat zmax,
                                 RtFloat thetamax, ...);
AQSIS_RI_SHARE RtVoid RiCylinderV(RtFloat radius, RtFloat zmin, RtFloat zmax,
                                  RtFloat thetamax, RtInt count,
                                  RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiHyperboloid(RtPoint point1, RtPoint point2,
                                    RtFloat thetamax, ...);
AQSIS_RI_SHARE RtVoid RiHyperboloidV(RtPoint point1, RtPoint point2,
                                     RtFloat thetamax, RtInt count,
                                     RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiParaboloid(RtFloat rmax, RtFloat zmin, RtFloat zmax,
                                   RtFloat thetamax, ...);
AQSIS_RI_SHARE RtVoid RiParaboloidV(RtFloat rmax, RtFloat zmin, RtFloat zmax,
                                    RtFloat thetamax, RtInt count,
                                    RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiDisk(RtFloat height, RtFloat radius, RtFloat thetamax,
                             ...);
AQSIS_RI_SHARE RtVoid RiDiskV(RtFloat height, RtFloat radius, RtFloat thetamax,
                              RtInt count, RtToken tokens[],
                              RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiTorus(RtFloat majorrad, RtFloat minorrad,
                              RtFloat phimin, RtFloat phimax, RtFloat thetamax,
                              ...);
AQSIS_RI_SHARE RtVoid RiTorusV(RtFloat majorrad, RtFloat minorrad,
                               RtFloat phimin, RtFloat phimax, RtFloat thetamax,
                               RtInt count, RtToken tokens[],
                               RtPointer values[]);
/* ------- Points and Curve Primitives -------- */
AQSIS_RI_SHARE RtVoid RiPoints(RtInt npoints, ...);
AQSIS_RI_SHARE RtVoid RiPointsV(RtInt npoints, RtInt count, RtToken tokens[],
                                RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiCurves(RtToken type, RtInt ncurves, RtInt nvertices[],
                               RtToken wrap, ...);
AQSIS_RI_SHARE RtVoid RiCurvesV(RtToken type, RtInt ncurves, RtInt nvertices[],
                                RtToken wrap, RtInt count, RtToken tokens[],
                                RtPointer values[]);
/* --------- Blobby Implicit Surfaces --------- */
AQSIS_RI_SHARE RtVoid RiBlobby(RtInt nleaf, RtInt ncode, RtInt code[],
                               RtInt nfloats, RtFloat floats[], RtInt nstrings,
                               RtToken strings[], ...);
AQSIS_RI_SHARE RtVoid RiBlobbyV(RtInt nleaf, RtInt ncode, RtInt code[],
                                RtInt nfloats, RtFloat floats[], RtInt nstrings,
                                RtToken strings[], RtInt count,
                                RtToken tokens[], RtPointer values[]);
/* ---------- Procedural Primitives ----------- */
AQSIS_RI_SHARE RtVoid RiProcedural(RtPointer data, RtBound bound,
                                   RtProcSubdivFunc refineproc,
                                   RtProcFreeFunc freeproc);
AQSIS_RI_SHARE RtVoid RiProcFree(RtPointer data);
AQSIS_RI_SHARE RtVoid RiProcDelayedReadArchive(RtPointer data, RtFloat detail);
AQSIS_RI_SHARE RtVoid RiProcRunProgram(RtPointer data, RtFloat detail);
AQSIS_RI_SHARE RtVoid RiProcDynamicLoad(RtPointer data, RtFloat detail);
/*  Implementation-specific Geometric Primitives  */
AQSIS_RI_SHARE RtVoid RiGeometry(RtToken type, ...);
AQSIS_RI_SHARE RtVoid RiGeometryV(RtToken type, RtInt count, RtToken tokens[],
                                  RtPointer values[]);
/* ----- Soids and Spatial Set Operations ----- */
AQSIS_RI_SHARE RtVoid RiSolidBegin(RtToken type);
AQSIS_RI_SHARE RtVoid RiSolidEnd();
/* ------------ Retained Geometry ------------- */
AQSIS_RI_SHARE RtObjectHandle RiObjectBegin();
AQSIS_RI_SHARE RtVoid RiObjectEnd();
AQSIS_RI_SHARE RtVoid RiObjectInstance(RtObjectHandle handle);


/* ============================ Motion ============================ */
AQSIS_RI_SHARE RtVoid RiMotionBegin(RtInt N, ...);
AQSIS_RI_SHARE RtVoid RiMotionBeginV(RtInt N, RtFloat times[]);
AQSIS_RI_SHARE RtVoid RiMotionEnd();


/* ====================== External Resources ====================== */
/* ---------- Texture Map Utilities ----------- */
/* Making texture maps */
AQSIS_RI_SHARE RtVoid RiMakeTexture(RtString imagefile, RtString texturefile,
                                    RtToken swrap, RtToken twrap,
                                    RtFilterFunc filterfunc, RtFloat swidth,
                                    RtFloat twidth, ...);
AQSIS_RI_SHARE RtVoid RiMakeTextureV(RtString imagefile, RtString texturefile,
                                     RtToken swrap, RtToken twrap,
                                     RtFilterFunc filterfunc, RtFloat swidth,
                                     RtFloat twidth, RtInt count,
                                     RtToken tokens[], RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiMakeBump(RtString imagefile, RtString bumpfile,
                                 RtToken swrap, RtToken twrap,
                                 RtFilterFunc filterfunc, RtFloat swidth,
                                 RtFloat twidth, ...);
AQSIS_RI_SHARE RtVoid RiMakeBumpV(RtString imagefile, RtString bumpfile,
                                  RtToken swrap, RtToken twrap,
                                  RtFilterFunc filterfunc, RtFloat swidth,
                                  RtFloat twidth, RtInt count, RtToken tokens[],
                                  RtPointer values[]);
/* Making environment maps */
AQSIS_RI_SHARE RtVoid RiMakeLatLongEnvironment(RtString imagefile,
                                               RtString reflfile,
                                               RtFilterFunc filterfunc,
                                               RtFloat swidth, RtFloat twidth,
                                               ...);
AQSIS_RI_SHARE RtVoid RiMakeLatLongEnvironmentV(RtString imagefile,
                                                RtString reflfile,
                                                RtFilterFunc filterfunc,
                                                RtFloat swidth, RtFloat twidth,
                                                RtInt count, RtToken tokens[],
                                                RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiMakeCubeFaceEnvironment(RtString px, RtString nx,
                                                RtString py, RtString ny,
                                                RtString pz, RtString nz,
                                                RtString reflfile, RtFloat fov,
                                                RtFilterFunc filterfunc,
                                                RtFloat swidth, RtFloat twidth,
                                                ...);
AQSIS_RI_SHARE RtVoid RiMakeCubeFaceEnvironmentV(RtString px, RtString nx,
                                                 RtString py, RtString ny,
                                                 RtString pz, RtString nz,
                                                 RtString reflfile, RtFloat fov,
                                                 RtFilterFunc filterfunc,
                                                 RtFloat swidth, RtFloat twidth,
                                                 RtInt count, RtToken tokens[],
                                                 RtPointer values[]);
/* Making shadow maps */
AQSIS_RI_SHARE RtVoid RiMakeShadow(RtString picfile, RtString shadowfile, ...);
AQSIS_RI_SHARE RtVoid RiMakeShadowV(RtString picfile, RtString shadowfile,
                                    RtInt count, RtToken tokens[],
                                    RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiMakeOcclusion(RtInt npics, RtString picfiles[],
                                      RtString shadowfile, ...);
AQSIS_RI_SHARE RtVoid RiMakeOcclusionV(RtInt npics, RtString picfiles[],
                                       RtString shadowfile, RtInt count,
                                       RtToken tokens[], RtPointer values[]);
/* ------------------ Errors ------------------ */
AQSIS_RI_SHARE RtVoid RiErrorHandler(RtErrorFunc handler);
AQSIS_RI_SHARE RtVoid RiErrorIgnore(RtInt code, RtInt severity,
                                    RtString message);
AQSIS_RI_SHARE RtVoid RiErrorPrint(RtInt code, RtInt severity,
                                   RtString message);
AQSIS_RI_SHARE RtVoid RiErrorAbort(RtInt code, RtInt severity,
                                   RtString message);
/* -------------- Archive Files --------------- */
AQSIS_RI_SHARE RtVoid RiArchiveRecord(RtToken type, char * format, ...);
AQSIS_RI_SHARE RtVoid RiReadArchive(RtToken name, RtArchiveCallback callback,
                                    ...);
AQSIS_RI_SHARE RtVoid RiReadArchiveV(RtToken name, RtArchiveCallback callback,
                                     RtInt count, RtToken tokens[],
                                     RtPointer values[]);
AQSIS_RI_SHARE RtArchiveHandle RiArchiveBegin(RtToken name, ...);
AQSIS_RI_SHARE RtArchiveHandle RiArchiveBeginV(RtToken name, RtInt count,
                                               RtToken tokens[],
                                               RtPointer values[]);
AQSIS_RI_SHARE RtVoid RiArchiveEnd();
/*[[[end]]]*/

/* Specific to Aqsis */

AQSIS_RI_SHARE  RtBoolean   BasisFromName( RtBasis * b, const char * strName );
AQSIS_RI_SHARE  RtVoid  RiProgressHandler( RtProgressFunc handler );

#ifdef  __cplusplus
}
#endif

/*
  Error Codes
  
   1 - 10         System and File Errors
  11 - 20         Program Limitations
  21 - 40         State Errors
  41 - 60         Parameter and Protocol Errors
  61 - 80         Execution Errors
*/
#define RIE_NOERROR     ((RtInt)0)

#define RIE_NOMEM       ((RtInt)1)      /* Out of memory */
#define RIE_SYSTEM      ((RtInt)2)      /* Miscellaneous system error */
#define RIE_NOFILE      ((RtInt)3)      /* File nonexistent */
#define RIE_BADFILE     ((RtInt)4)      /* Bad file format */
#define RIE_VERSION     ((RtInt)5)      /* File version mismatch */
#define RIE_DISKFULL    ((RtInt)6)      /* Target disk is full */

#define RIE_INCAPABLE   ((RtInt)11)     /* Optional RI feature */
#define RIE_UNIMPLEMENT ((RtInt)12)     /* Unimplemented feature */
#define RIE_LIMIT       ((RtInt)13)     /* Arbitrary program limit */
#define RIE_BUG         ((RtInt)14)     /* Probably a bug in renderer */

#define RIE_NOTSTARTED  ((RtInt)23)     /* RiBegin not called */
#define RIE_NESTING     ((RtInt)24)     /* Bad begin-end nesting */
#define RIE_NOTOPTIONS  ((RtInt)25)     /* Invalid state for options */
#define RIE_NOTATTRIBS  ((RtInt)26)     /* Invalid state for attribs */
#define RIE_NOTPRIMS    ((RtInt)27)     /* Invalid state for primitives */
#define RIE_ILLSTATE    ((RtInt)28)     /* Other invalid state */
#define RIE_BADMOTION   ((RtInt)29)     /* Badly formed motion block */
#define RIE_BADSOLID    ((RtInt)30)     /* Badly formed solid block */

#define RIE_BADTOKEN    ((RtInt)41)     /* Invalid token for request */
#define RIE_RANGE       ((RtInt)42)     /* Parameter out of range */
#define RIE_CONSISTENCY ((RtInt)43)     /* Parameters inconsistent */
#define RIE_BADHANDLE   ((RtInt)44)     /* Bad object/light handle */
#define RIE_NOSHADER    ((RtInt)45)     /* Can't load requested shader */
#define RIE_MISSINGDATA ((RtInt)46)     /* Required parameters not provided */
#define RIE_SYNTAX      ((RtInt)47)     /* Declare type syntax error */

#define RIE_MATH        ((RtInt)61)     /* Zerodivide, noninvert matrix, etc. */

/* Error severity levels */
#define RIE_INFO        ((RtInt)0)      /* Rendering stats and other info */
#define RIE_WARNING     ((RtInt)1)      /* Something seems wrong, maybe okay */
#define RIE_ERROR       ((RtInt)2)      /* Problem. Results may be wrong */
#define RIE_SEVERE      ((RtInt)3)      /* So bad you should probably abort */


#endif 

/* vi: set et: */
