/*###############################################################################
# Linux Management Providers (LMP), EthernetPort provider package
# Copyright (C) 2007 Ilsoo Byun <widepis@etri.re.kr ,widepis@empal.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
###############################################################################*/

#include "OpenDRIM_EthernetPortAccess.h"
#include "Common/EthernetPort_Common.h"
#include "net_dev.h"

const string systemCreationClassName = "OpenDRIM_ComputerSystem";
const string creationClassName = "OpenDRIM_EthernetPort";
static string systemName;

int EthernetPort_OpenDRIM_EthernetPort_load(const CMPIBroker* broker, string& errorMessage) {
	_E_;
	CF_assert(CF_getSystemName(systemName, errorMessage));
	_L_;
	return OK;
}

int EthernetPort_OpenDRIM_EthernetPort_unload(string& errorMessage) {
	_E_;
	// Nothing
	_L_;
	return OK;
}

int EthernetPort_OpenDRIM_EthernetPort_retrieve(const CMPIBroker* broker, const CMPIContext* ctx, vector<OpenDRIM_EthernetPort>& result, const char** properties, string& errorMessage, const string& discriminant) {
	_E_;
	vector<string> eth_names;
	CF_assert(CF_getEthernetPortNames(eth_names, errorMessage));

	//Initialize an instance
	OpenDRIM_EthernetPort instance;
	instance.setSystemCreationClassName(systemCreationClassName);
	instance.setSystemName(systemName);
	instance.setCreationClassName(creationClassName);

	for (unsigned int i=0;i<eth_names.size();i++)
	{
		if (!CF_startsWith(eth_names[i], "eth")) continue;

		instance.setDeviceID(eth_names[i]);
		if (discriminant == "ei")
			CF_assert(EthernetPort_OpenDRIM_EthernetPort_populate(instance, errorMessage));
		result.push_back(instance);
	}
	_L_;
	return OK;
}

int EthernetPort_OpenDRIM_EthernetPort_getInstance(const CMPIBroker* broker, const CMPIContext* ctx, OpenDRIM_EthernetPort& instance, const char** properties, string& errorMessage) {
	_E_;
	vector<string> eth_names;
	vector<string>::size_type index;

	CF_assert(CF_getEthernetPortNames(eth_names, errorMessage));

	if (instance.SystemCreationClassName != systemCreationClassName ||
		instance.CreationClassName != creationClassName ||
		instance.SystemName != systemName ||
		!CF_foundInList(instance.DeviceID,eth_names,index))
	{
		errorMessage = "No instance";
		return NOT_FOUND;
	}

	CF_assert(EthernetPort_OpenDRIM_EthernetPort_populate(instance, errorMessage));

	_L_;
	return OK;
}

int EthernetPort_OpenDRIM_EthernetPort_setInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& newInstance, const OpenDRIM_EthernetPort& oldInstance, const char** properties, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_createInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_deleteInstance(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_RequestStateChange(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, const OpenDRIM_EthernetPort_RequestStateChange_In& in, OpenDRIM_EthernetPort_RequestStateChange_Out& out, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_SetPowerState(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, const OpenDRIM_EthernetPort_SetPowerState_In& in, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_Reset(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_EnableDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, const OpenDRIM_EthernetPort_EnableDevice_In& in, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_OnlineDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, const OpenDRIM_EthernetPort_OnlineDevice_In& in, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_QuiesceDevice(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, const OpenDRIM_EthernetPort_QuiesceDevice_In& in, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_SaveProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_RestoreProperties(const CMPIBroker* broker, const CMPIContext* ctx, const OpenDRIM_EthernetPort& instance, unsigned int& returnValue, string& errorMessage) {
	_E_;
	_L_;
	return NOT_SUPPORTED;
}

int EthernetPort_OpenDRIM_EthernetPort_populate(OpenDRIM_EthernetPort& instance, string& errorMessage) {
	_E_;

	/*
	 * Properties to fill from profile
	 * + Mandatory (from Ethernet):
	 * [X] PortType
	 * [X] NetworkAddresses
	 * [X] Capabilities
	 * [X] EnabledCapabilities
	 * [X] LinkTechnology
	 * [X] PermanentAddress
	 * + Mandatory (from Network):
	 * [X] SystemCreationClassName	[KEY]
	 * [X] CreationClassName		[KEY]
	 * [X] SystemName				[KEY]
	 * [X] DeviceID					[KEY]
	 * [X] Name
	 * [X] EnabledState
	 * [X] RequestedState
	 * [X] ElementName
	 * + Optional:
	 * [-] Speed
	 * [X] MaxSpeed
	 * [-] RequestedSpeed
	 * + Others:
	 * [X] FullDuplex
	 */

	string hwaddr, temp;
	string ethtool_str;
	string addr;
	vector<string> addresses;
	unsigned int enabledState;
	unsigned int index;

	instance.setName(instance.DeviceID);
	instance.setLinkTechnology(LT_ETHERNET);

	const char* mac = CN_getMacAddress(instance.DeviceID.c_str());
	if (mac != NULL)
	{
		addr=CF_removeChar(mac,":");
		instance.setPermanentAddress(addr);
		addresses.push_back(addr);
		instance.setNetworkAddresses(addresses);
		free((void*)mac);
	}

	CF_getEthEnabledState(instance.DeviceID,enabledState,errorMessage);
	instance.setEnabledState(enabledState);

	// Setting RequestedState as Not applicable,
	// as far as the RequestStateChange method is Not Supported.
	// (see: Network Port Profile)
	instance.setRequestedState(RS_NOT_APPLICABLE);

	index=instance.DeviceID.find_first_of("0123456789");
	if(index!=string::npos)
		instance.setElementName("Ethernet Interface " + instance.DeviceID.substr(index,instance.DeviceID.length()-index));
	else
		instance.setElementName("Ethernet Interface");

	CF_assert(CF_runCommand("ethtool eth0 | grep Speed | awk '{print $2}'", ethtool_str, temp, errorMessage));
	instance.setMaxSpeed(atoi(ethtool_str.c_str())*1024*1024);

	int speed = atoi(ethtool_str.c_str());
	switch(speed)
	{
		case 10:
			instance.setPortType(PT_10BASET);
			break;
		case 100:
			instance.setPortType(PT_100BASET);
			break;
		case 1000:
			instance.setPortType(PT_1000BASET);
			break;
		case 2500:
			instance.setPortType(PT_2500BASET);
			break;
		case 10000:
			instance.setPortType(PT_10GBASET);
			break;
		default:
			instance.setPortType(PT_UNKNOWN);
			break;
	}

	CF_assert(CF_runCommand("ethtool eth0 | grep Duplex | awk '{print $2}'", ethtool_str, temp, errorMessage));
	if (CF_trimText(ethtool_str) == "Full")
		instance.setFullDuplex(true);
	else
		instance.setFullDuplex(false);

	vector<unsigned short> capabilities;
	vector<unsigned short> enabledCapabilities;

	CF_assert(CF_getCapabilities(capabilities,enabledCapabilities,instance.DeviceID,errorMessage));
	instance.setCapabilities(capabilities);
	instance.setEnabledCapabilities(enabledCapabilities);
	_L_;
	return OK;
}
