import unittest

from gi.repository import GData, GLib, Unity

import unity_gdrive_daemon as gdrive


class ResultSet(Unity.ResultSet):
    def __init__(self):
        Unity.ResultSet.__init__(self)
        self.results = []

    def do_add_result(self, result):
        self.results.append(dict(uri=result.uri,
                                 icon=result.icon_hint,
                                 category=result.category,
                                 result_type=result.result_type,
                                 mimetype=result.mimetype,
                                 title=result.title,
                                 comment=result.comment,
                                 dnd_uri=result.dnd_uri,
                                 author=result.metadata['author'].get_string(),
                                 shared=result.metadata['shared'].get_boolean(),
                                 starred=result.metadata['starred'].get_boolean(),
                                 updated=result.metadata['updated'].get_int32()))


class FakeDocumentsEntry:
    """A fake GData.DocumentsEntry"""

    def get_resource_id(self):
        return 'spreadsheet:abcdefghij'

    def look_up_link(self, rel):
        assert rel == GData.LINK_ALTERNATE
        return GData.Link.new("http://example.com/view", rel)

    def get_updated(self):
        return 1000000000

    def get_title(self):
        return "Fake title"

    def get_authors(self):
        return [GData.Author.new('john.smith', '', 'john.smith@example.com')]

    def get_categories(self):
        return [GData.Category.new(
                'http://schemas.google.com/g/2005/labels#starred',
                'http://schemas.google.com/g/2005/labels', 'starred')]

    def get_content_uri(self):
        return "http://example.com/export"


class StubGDocsAccount(gdrive.GDocsAccount):
    def __init__(self):
        self._enabled = True

    def get_doc_list(self, search, filters, is_global):
        # It would be nice if we could use
        # GData.Parsable.new_from_xml() to generate the feed from XML,
        # but it segfaults.
        return [FakeDocumentsEntry()]


def fake_setup_accounts(self):
    self._gdocs_accounts.append(StubGDocsAccount())


class GDriveTests(unittest.TestCase):
    def setUp(self):
        super(GDriveTests, self).setUp()
        self._orig_setup_accounts = gdrive.GDriveScope.setup_accounts
        gdrive.GDriveScope.setup_accounts = fake_setup_accounts

    def tearDown(self):
        gdrive.GDriveScope.setup_accounts = self._orig_setup_accounts
        super(GDriveTests, self).tearDown()

    def perform_query(self, query):
        scope = gdrive.GDriveScope()
        filter_set = scope.get_filters()
        result_set = ResultSet()
        context = Unity.SearchContext.create(
            query, 0, filter_set, None, result_set, None)

        search = scope.create_search_for_query(context)
        search.run()
        return result_set.results

    def test_get_categories(self):
        scope = gdrive.GDriveScope()
        category_set = scope.get_categories()
        self.assertEqual(
            [cat.props.id for cat in category_set.get_categories()],
            ['global', 'recent', 'downloads', 'folders'])

    def test_get_filters(self):
        scope = gdrive.GDriveScope()
        filter_set = scope.get_filters()
        self.assertEqual(
            [filter.props.id for filter in filter_set.get_filters()],
            ['modified', 'type'])

    def test_search(self):
        results = self.perform_query("foo")
        self.assertEqual(len(results), 1)
        self.assertEqual(results[0], dict(
                uri='http://example.com/view',
                icon='x-office-spreadsheet',
                category=1,
                result_type=Unity.ResultType.PERSONAL,
                mimetype='text/html',
                title='Fake title',
                comment='spreadsheet',
                dnd_uri='http://example.com/export',
                author='john.smith',
                shared=False,
                starred=True,
                updated=1000000000))

    @unittest.skip("Can not set up metadata in ScopeResult")
    def test_preview(self):
        result = Unity.ScopeResult.create(
            uri='http://example.com/view',
            icon_hint='x-office-spreadsheet',
            category=1,
            result_type=Unity.ResultType.PERSONAL,
            mimetype='text/html',
            title='Fake title',
            comment='spreadsheet',
            dnd_uri='http://example.com/export',
            metadata={
                'author': GLib.Variant('s', 'john.smith'),
                'shared': GLib.Variant('b', False),
                'starred': GLib.Variant('b', True),
                'updated': GLib.Variant('i', 1000000000)})
        metadata = Unity.SearchMetadata()
        scope = gdrive.GDriveScope()
        previewer = scope.create_previewer(result, metadata)
        preview = previewer.run()
