/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * Copyright 2009 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: PSONMSolverImpl.java,v $
 * $Revision: 1.1 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

package com.sun.star.NLPSolver;

import com.sun.star.NLPSolver.dialogs.IEvolutionarySolverStatusDialog;
import com.sun.star.lang.IllegalArgumentException;
import com.sun.star.uno.XComponentContext;
import com.sun.star.lib.uno.helper.Factory;
import com.sun.star.lang.XSingleComponentFactory;
import com.sun.star.registry.XRegistryKey;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.adaptivebox.global.IUpdateCycleEngine;
import net.adaptivebox.knowledge.ILibEngine;
import net.adaptivebox.knowledge.Library;
import net.adaptivebox.knowledge.SearchPoint;
import net.adaptivebox.psonm.PSONMAgent;


public final class PSONMSolverImpl extends BaseEvolutionarySolver
   implements com.sun.star.lang.XServiceInfo
{
    private static final String m_implementationName = PSONMSolverImpl.class.getName();
    private static final String[] m_serviceNames = {
        "com.sun.star.sheet.Solver",
        "com.sun.star.beans.PropertySet"
    };

    public PSONMSolverImpl( XComponentContext context )
    {
        super(context, "PSO-NM Hybrid Evolutionary Algorithm");

        registerProperty(m_c1);
        registerProperty(m_c2);
        registerProperty(m_weight);
        registerProperty(m_clusters);
        registerProperty(m_kmIterations);
        registerProperty(m_kmConvergence);
        registerProperty(m_nmSteps);
    }

    public static XSingleComponentFactory __getComponentFactory( String sImplementationName ) {
        XSingleComponentFactory xFactory = null;

        if ( sImplementationName.equals( m_implementationName ) )
            xFactory = Factory.createComponentFactory(PSONMSolverImpl.class, m_serviceNames);
        return xFactory;
    }

    public static boolean __writeRegistryServiceInfo( XRegistryKey xRegistryKey ) {
        return Factory.writeRegistryServiceInfo(m_implementationName,
                                                m_serviceNames,
                                                xRegistryKey);
    }

    // com.sun.star.lang.XServiceInfo:
    public String getImplementationName() {
         return m_implementationName;
    }

    public boolean supportsService( String sService ) {
        int len = m_serviceNames.length;

        for( int i=0; i < len; i++) {
            if (sService.equals(m_serviceNames[i]))
                return true;
        }
        return false;
    }

    public String[] getSupportedServiceNames() {
        return m_serviceNames;
    }
    
    // com.sun.star.sheet.XSolver:
    private PSONMAgent[] m_agents;

    private PropertyInfo<Double> m_c1 = new PropertyInfo<Double>("C1", 1.494, "C1");
    private PropertyInfo<Double> m_c2 = new PropertyInfo<Double>("C2", 1.494, "C2");
    private PropertyInfo<Double> m_weight = new PropertyInfo<Double>("Weight", 0.729, "Weight");
    private PropertyInfo<Integer> m_clusters = new PropertyInfo<Integer>("KM.Clusters", 5, "Clusters");
    private PropertyInfo<Integer> m_kmIterations = new PropertyInfo<Integer>("KM.Iterations", 20, "Clustering Max Iterations");
    private PropertyInfo<Double> m_kmConvergence = new PropertyInfo<Double>("KM.Convergence", 0.1, "Clustering Convergence");
    private PropertyInfo<Integer> m_nmSteps = new PropertyInfo<Integer>("NM.Steps", 10, "Nelder Mead Steps");

    public void solve() {
        try {
            m_librarySize.setValue(m_swarmSize.getValue()); //DEPS' library is as large as the swarm
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(PSONMSolverImpl.class.getName()).log(Level.SEVERE, null, ex);
        }

        initializeSolve();

        try {
        //Init:
        m_agents = new PSONMAgent[m_swarmSize.getValue()];
        for (int i = 0; i < m_swarmSize.getValue(); i++) {
            m_agents[i] = new PSONMAgent();
            m_agents[i].setProblemEncoder(m_problemEncoder);
            m_agents[i].setBestPoint(m_library.getSelectedPoint(i));

            m_agents[i].setSpecComparator(m_specCompareEngine);
            if (m_agents[i] instanceof ILibEngine)
                ((ILibEngine)m_agents[i]).setLibrary(m_library);
        }

        KMCluster[] clusters = new KMCluster[m_clusters.getValue()];
        for (int i = 0; i < m_clusters.getValue(); i++)
            clusters[i] = new KMCluster(m_problemEncoder, m_specCompareEngine, m_maximize);

        //Learn:
        m_solverStatusDialog.setVisible(true);
        m_solverStatusDialog.setMaxIterations(m_learningCycles.getValue());
        m_solverStatusDialog.setMaxStagnation(m_required.getValue());
        do {
            m_toleratedCount = 0;
            m_toleratedMin = -1.0 * m_tolerance.getValue();
            m_toleratedMax = m_tolerance.getValue();
            for (int learningCycle = 1; learningCycle <= m_learningCycles.getValue() &&
                    m_toleratedCount < m_required.getValue() &&
                    m_solverStatusDialog.getUserState() != IEvolutionarySolverStatusDialog.CANCEL; learningCycle++) {

                m_library.refreshGbest(m_specCompareEngine);

                for (int i = 0; i < m_swarmSize.getValue(); i++)
                    m_agents[i].generatePoint();

                for (int i = 0; i < m_swarmSize.getValue(); i++)
                    m_agents[i].learn();

                //start kmeans
                int maxIteration = m_kmIterations.getValue();
                double moveDistance;
                do {
                    moveDistance = 0.0;

                    for (int k = 0; k < clusters.length; k++)
                        clusters[k].getNodes().clear();

                    for (int i = 0; i < m_swarmSize.getValue(); i++) {
                        SearchPoint point = m_agents[i].getTrailingPoint();
                        double minDist = Double.MAX_VALUE;
                        int minCluster = -1;
                        for (int k = 0; k < clusters.length; k++) {
                            double dist = clusters[k].getDistanceTo(point);
                            //System.out.println(k + ": " + dist);
                            if (dist < minDist) {
                                minDist = dist;
                                minCluster = k;
                            }
                        }
                        //System.out.println("min " + minCluster + ": " + minDist);
                        clusters[minCluster].getNodes().add(point);
                    }

                    for (int k = 0; k < clusters.length; k++)
                        moveDistance += clusters[k].updateCenter();

                } while (moveDistance > m_kmConvergence.getValue() && --maxIteration > 0);

                /*System.out.println();
                for (int i = 0; i < clusters.length; i++) {
                    System.out.print("Cluster " + i);
                    clusters[i].printCenter();
                    System.out.println();
                    for (int j = 0; j < clusters[i].getNodes().size(); j++) {
                        System.out.println(clusters[i].getNodes().get(j).getObjectiveValue());
                    }
                }*/
                //stop kmeans

                for (int k = 0; k < clusters.length; k++) {
                    for (int i = 0; i < m_nmSteps.getValue(); i++) {
                        clusters[k].performNelderMeadStep();
                    }
                }

                for (int i = 0; i < m_swarmSize.getValue(); i++) {
                    SearchPoint agentPoint = m_agents[i].getTrailingPoint();
                    boolean inRange = (agentPoint.getObjectiveValue() >= m_toleratedMin && agentPoint.getObjectiveValue() <= m_toleratedMax);
                    if (Library.replace(m_envCompareEngine, agentPoint, m_totalBestPoint)) {
                        m_solverStatusDialog.setBestSolution(m_totalBestPoint.getObjectiveValue(), m_totalBestPoint.isFeasible());
                        if (!inRange) {
                            m_toleratedMin = agentPoint.getObjectiveValue() - m_tolerance.getValue();
                            m_toleratedMax = agentPoint.getObjectiveValue() + m_tolerance.getValue();
                            m_toleratedCount = 0;
                        }
                    }
                }

                if (m_specCompareEngine instanceof IUpdateCycleEngine)
                    ((IUpdateCycleEngine)m_specCompareEngine).updateCycle(learningCycle);

                m_solverStatusDialog.setIteration(learningCycle);
                m_solverStatusDialog.setStagnation(m_toleratedCount);
            }
        } while (m_solverStatusDialog.waitForUser() == IEvolutionarySolverStatusDialog.CONTINUE);

        } catch (Exception ex) {
            Logger.getLogger(PSONMSolverImpl.class.getName()).log(Level.SEVERE, null, ex);
        }

        finalizeSolve();
    }

}
