// Copyright 2024 Michael Ficarra. All rights reserved.
// This code is governed by the BSD license found in the LICENSE file.

/*---
es6id: 20.2.2.32
description: >
  Math.sqrt returns the best possible Number for the square root, not an approximation.
info: |
  Math.sqrt ( x )

  Return 𝔽(the square root of ℝ(_n_)).
---*/

// 1000 random non-negative finite doubles, with progressively larger magnitude exponents, and their square roots
const examples = [
  [2.6364218206773287, 1.6237061990019404],
  [6.093135406384165, 2.4684277194976088],
  [0.5577847536359243, 0.7468498869491274],
  [18.826149681104784, 4.33891111698601],
  [51.0071358534849, 7.141928020743761],
  [0.09916483871835496, 0.31490449142296295],
  [0.05495461478727153, 0.23442400642270309],
  [0.025188429959113877, 0.15870863227661525],
  [0.00812479431693535, 0.09013764095501585],
  [1724.98994093245, 41.53299821747101],
  [0.0029825902870519953, 0.05461309629614489],
  [7267.933822887094, 85.25217781902755],
  [0.0008860138193399767, 0.02976598426627241],
  [25136.0059901805, 158.54338835214952],
  [0.00017917048388385395, 0.013385457925818375],
  [112728.51764979148, 335.7506778098765],
  [0.00004969153203848829, 0.007049222087470949],
  [0.00001941546662946677, 0.004406298517970244],
  [933382.27260725, 966.1171112278521],
  [0.000005294364184376967, 0.002300948540140993],
  [3130866.954533674, 1769.42560016907],
  [9.72044333465995e-7, 0.0009859230869930955],
  [16080744.632595163, 4010.0803773235225],
  [4.4205797932006607e-7, 0.0006648744086818698],
  [1.756023173880081e-7, 0.00041904930185839484],
  [111405072.76243196, 10554.860148880798],
  [3.0892296725681764e-8, 0.00017576204574845436],
  [1.8639944384971586e-8, 0.00013652818165115797],
  [642938223.9516982, 25356.226532189252],
  [6.7141406880308e-9, 0.00008193986019045187],
  [3768749884.450741, 61390.144847937445],
  [1.8258753524207704e-9, 0.00004273026272351681],
  [8682620259.345915, 93180.5787669615],
  [3.2719931235745735e-10, 0.000018088651479794103],
  [1.7288819479897516e-10, 0.000013148695555034163],
  [131195381256.19533, 362209.02978279727],
  [5.1535438181624584e-11, 0.000007178818717701721],
  [2.8633714112951604e-11, 0.000005351047945304883],
  [1075736798229.0712, 1037177.3224618204],
  [1769289622560.063, 1330146.4665818058],
  [2.584443592924671e-12, 0.0000016076204753997976],
  [1.6505683161263436e-12, 0.0000012847444555733034],
  [9895708558925.95, 3145744.515838174],
  [18685414632224.207, 4322662.909853625],
  [1.6121069579556452e-13, 4.0151051766493555e-7],
  [81368179193918.36, 9020431.20886792],
  [4.853388692504837e-14, 2.203040783214155e-7],
  [1.5241670622374688e-14, 1.2345716108178857e-7],
  [975630473715166.6, 31235084.019659154],
  [4.469656075594871e-15, 6.685548650331454e-8],
  [3075560259258901, 55457733.989578955],
  [7307054298634085, 85481309.64505683],
  [7.428035353751654e-16, 2.7254422308593617e-8],
  [2.3245002467788256e-16, 1.5246311838535985e-8],
  [38651061189498650, 196598731.4035842],
  [8.626254925032578e-17, 9.287763414855364e-9],
  [245261483562121020, 495238814.6764357],
  [2.359419496918105e-17, 4.857385610509119e-9],
  [1.2379295553316081e-17, 3.5184223102572665e-9],
  [4.115087182801168e-18, 2.0285677663812883e-9],
  [2.978665174163489e-18, 1.7258809849359512e-9],
  [8483294251614005000, 2912609526.114684],
  [4.523188214083293e-19, 6.725465198841856e-10],
  [24378912475850270000, 4937500630.46581],
  [1.9581825550381185e-19, 4.425135653331001e-10],
  [96034813220780460000, 9799735364.834116],
  [2.920538955165126e-20, 1.7089584416144022e-10],
  [1.750634688256351e-20, 1.3231155233978442e-10],
  [754994936573400200000, 27477171189.432877],
  [1.8632934465655627e21, 43165882900.336494],
  [4.2056655125676105e21, 64851102631.8567],
  [7.037373863164293e21, 83889056873.7323],
  [5.016081426480113e-22, 2.239661007045511e-11],
  [3.1360853670074786e-22, 1.7708995925821087e-11],
  [1.4460461763569206e-22, 1.2025166012812133e-11],
  [1.1138491594465553e23, 333743787874.2547],
  [3.32077706603068e-23, 5.762618385795367e-12],
  [4.50259352006827e23, 671013674977.5127],
  [1.2030527277267097e-23, 3.4685050493356783e-12],
  [3.9898456768622155e-24, 1.9974598060692525e-12],
  [2.099099422079296e-24, 1.448826912394747e-12],
  [1.2947577980725777e-24, 1.1378742452804605e-12],
  [4.523588897224216e-25, 6.725763077320086e-13],
  [2.6439027364270637e-25, 5.141889474139894e-13],
  [6.161406847244829e25, 7849462941657.11],
  [1.1262097160333162e26, 10612302841670.682],
  [2.8903531131865582e26, 17001038536473.465],
  [5.520139379052935e26, 23494976865391.75],
  [1.164692380222816e27, 34127589721848.45],
  [4.279137256127335e-27, 6.541511489042371e-14],
  [3.4013217683189986e27, 58320851916951.61],
  [8.3233105280907e-28, 2.88501482285459e-14],
  [4.892374771987226e-28, 2.211871328081095e-14],
  [2.644502650010035e-28, 1.6261926853881846e-14],
  [7.787526091150166e28, 279061392728377.34],
  [8.992269683411683e28, 299871133712661.4],
  [3.987026745804168e-29, 6.314290732777647e-15],
  [2.1265718826609917e-29, 4.611476859598226e-15],
  [9.318934400304718e29, 965346279855302.4],
  [3.478110095369478e-30, 1.8649691942146063e-15],
  [3.0336290098099214e-30, 1.7417316124506442e-15],
  [1.187505107289765e-30, 1.0897270792679079e-15],
  [6.121852606868157e-31, 7.824226867153174e-16],
  [3.330016415530198e31, 5770629441863511],
  [1.1972007699823333e-31, 3.4600589156578435e-16],
  [9.491784329862492e31, 9742578883366812],
  [3.2090013924080404e-32, 1.7913685808364622e-16],
  [5.953924050511395e32, 24400664028897644],
  [7.943918556616615e32, 28184957968066256],
  [1.5494270953537593e33, 39362762801329880],
  [2.709475968421315e-33, 5.20526269118218e-17],
  [5.895348228542645e33, 76781171054775180],
  [5.837257350750867e-34, 2.4160416699119384e-17],
  [3.637594352139908e34, 190724784759084400],
  [1.4131093499710667e-34, 1.1887427602181503e-17],
  [9.862394293586106e34, 314044491968671600],
  [3.1845200900520636e-35, 5.64315522562694e-18],
  [4.8782336780412935e35, 698443532294579600],
  [1.110815576633264e-35, 3.33289000213518e-18],
  [5.491170564095181e-36, 2.343324681749241e-18],
  [1.508770843596759e-36, 1.228320334276348e-18],
  [1.1699721732111957e-36, 1.0816525196250392e-18],
  [6.612965083691836e-37, 8.132013947166985e-19],
  [2.3376769318234387e-37, 4.834952876526759e-19],
  [8.442517993381293e37, 9188317579068158000],
  [1.3847550294555677e38, 11767561469801497000],
  [3.723438694436284e-38, 1.9296213862922134e-19],
  [1.882723158337299e-38, 1.3721235944102482e-19],
  [7.125389733783852e-39, 8.441202363279683e-20],
  [2.6849968016192544e39, 51816954769836240000],
  [2.906000639202325e39, 53907333816488500000],
  [9.295880368156956e-40, 3.0489146213295245e-20],
  [1.4927361896539553e40, 122177583445325800000],
  [2.3049623126773745e-40, 1.5182102333594564e-20],
  [6.612130780424898e40, 257140638181227560000],
  [7.787983839518702e-41, 8.824955433042539e-21],
  [3.362042923738955e-41, 5.7983126198394606e-21],
  [1.94114615866614e-41, 4.405844026592567e-21],
  [9.468800250495811e41, 973077604844331600000],
  [4.6047974907521363e-42, 2.1458791882937252e-21],
  [3.17089316421976e42, 1.780700189313114e21],
  [1.3918013247929126e-42, 1.1797462967913536e-21],
  [1.200207317308855e43, 3.464400838974692e21],
  [2.0319091746754344e-43, 4.507670323654376e-22],
  [7.4327616786354335e43, 8.621346576165137e21],
  [1.7268671846659664e44, 1.3141031864606243e22],
  [2.0484061765160574e44, 1.4312254107987523e22],
  [6.5456923007084346e44, 2.5584550613032926e22],
  [9.328456638771375e44, 3.0542522225204937e22],
  [3.760974008764618e-45, 6.132678051850283e-23],
  [1.4122172352163547e-45, 3.757947891092098e-23],
  [1.3300600995467618e-45, 3.646998902586566e-23],
  [4.429527186753373e-46, 2.1046441948114111e-23],
  [4.183050428910713e46, 2.045250700748129e23],
  [1.2337655524558335e-46, 1.1107499954786557e-23],
  [5.97546732989559e-47, 7.730114701539421e-24],
  [3.1005119225635093e-47, 5.568224063885638e-24],
  [5.6296556417084825e47, 7.50310311918241e23],
  [1.3620262837869847e48, 1.1670588176210249e24],
  [3.1130476432230247e-48, 1.7643830772321027e-24],
  [4.870459456344632e48, 2.2069117463878415e24],
  [9.858220813082454e48, 3.139780376568153e24],
  [5.556962829131254e-49, 7.454503893037587e-25],
  [2.589637950003785e49, 5.088848543633211e24],
  [8.499087188535236e49, 9.219049402479215e24],
  [1.141437146870446e50, 1.068380618913712e25],
  [3.9388099652794106e-50, 1.9846435360737734e-25],
  [1.710592494534373e-50, 1.3078962093890987e-25],
  [8.869444860439435e-51, 9.417773017247462e-26],
  [1.8157478720946456e51, 4.261159316541269e25],
  [2.3728034589514474e-51, 4.8711430475315003e-26],
  [1.1950714257640223e52, 1.0931932243496674e26],
  [1.4016634209157064e52, 1.1839186715799807e26],
  [2.8011285784075223e52, 1.6736572463941123e26],
  [8.153815667314413e52, 2.8554886915052584e26],
  [1.0637264599470274e53, 3.261481963689248e26],
  [2.1530551611602513e-53, 4.640102543220625e-27],
  [1.919404771488337e-53, 4.381101198886345e-27],
  [1.398778231958682e54, 1.182699552700804e27],
  [2.9296586655033626e54, 1.7116245690873226e27],
  [1.7224158129622894e-54, 1.3124084017417328e-27],
  [7.730559269102998e54, 2.7803883306299135e27],
  [3.2899966756320666e-55, 5.73584926199431e-28],
  [2.642441513471452e55, 5.140468376978359e27],
  [6.830315646013029e55, 8.264572370070353e27],
  [1.5249399316387194e56, 1.2348845823147682e28],
  [3.1014249013844634e56, 1.7610862844802533e28],
  [1.076148310797754e-56, 1.0373756845028486e-28],
  [9.62492113199095e-57, 9.810668240232645e-29],
  [4.1188883231917496e-57, 6.417856591722621e-29],
  [1.6307004169851293e-57, 4.038193181343767e-29],
  [1.1505235690700644e-57, 3.391936864197305e-29],
  [4.772620584318082e-58, 2.1846328259728412e-29],
  [2.1569523804262153e-58, 1.4686566584556838e-29],
  [1.1647751060992745e-58, 1.0792474721301295e-29],
  [7.5475260834017815e-59, 8.687649902822846e-30],
  [3.0845980569349766e-59, 5.5539157870235815e-30],
  [6.53471003867459e59, 8.083755339367088e29],
  [8.868322438087189e59, 9.417177091935347e29],
  [3.9544754004744846e-60, 1.9885862818782806e-30],
  [2.18763082689505e-60, 1.4790641726764427e-30],
  [6.432581762266233e-61, 8.020337749911929e-31],
  [5.1761509221555365e-61, 7.194547186693222e-31],
  [2.0987471033469038e-61, 4.581208468676037e-31],
  [1.4083531117605594e-61, 3.7528031013637783e-31],
  [2.0146258988532675e62, 1.4193751790324034e31],
  [3.688455065160669e-62, 1.920535098653672e-31],
  [6.448285106075812e62, 2.5393473779843142e31],
  [8.904845547242042e-63, 9.436548917502649e-32],
  [4.1290721987804467e-63, 6.425785709763785e-32],
  [1.4682044892502313e-63, 3.8317156591404733e-32],
  [1.0644526030301698e64, 1.031723123241003e32],
  [4.4912608966110374e-64, 2.119259516107227e-32],
  [2.504689501975298e-64, 1.5826210860390108e-32],
  [5.746518613638574e64, 2.397189732507332e32],
  [1.5276919882932144e65, 3.9085700560348335e32],
  [3.9087997083184305e65, 6.25203943391149e32],
  [7.512464344555431e65, 8.667447343108253e32],
  [1.6420921069710679e66, 1.2814414176898871e33],
  [1.9598989681860843e66, 1.3999639167443153e33],
  [4.324485009209941e66, 2.0795396147248413e33],
  [1.1478971583074518e-66, 1.0713996258667687e-33],
  [2.0059262296399578e67, 4.478756780223679e33],
  [1.94168179703729e-67, 4.406451857262587e-34],
  [6.834230205628381e67, 8.266940308015041e33],
  [6.158434038093654e-68, 2.481619237129994e-34],
  [3.6938289932502024e68, 1.9219336599503643e34],
  [1.264905179588931e-68, 1.124680034315952e-34],
  [5.918784590915353e-69, 7.693363757756001e-35],
  [3.0447177466729825e-69, 5.517896108729289e-35],
  [4.447622918769664e69, 6.6690500963553e34],
  [1.3357828040249557e70, 1.1557607036168672e35],
  [2.947814701158749e-70, 1.7169201207856902e-35],
  [5.479903009207335e70, 2.3409192658456498e35],
  [1.131553461291788e-70, 1.0637450170467489e-35],
  [4.790550608827023e-71, 6.921380360034422e-36],
  [2.54391755369521e-71, 5.043726354289267e-36],
  [7.495246645844719e71, 8.65750925257647e35],
  [5.147631150891665e-72, 2.2688391637336624e-36],
  [1.7842682854052505e72, 1.3357650562150704e36],
  [5.393978612912003e72, 2.3224940501348984e36],
  [1.0425513888847234e-72, 1.0210540577681102e-36],
  [1.977350004555512e73, 4.4467403843214323e36],
  [4.473107873130231e73, 6.688129688582774e36],
  [7.812063748904077e73, 8.838587980500096e36],
  [1.6975310044313772e74, 1.3028933204339398e37],
  [3.2300906331893816e74, 1.797245290212046e37],
  [5.559975934650521e74, 2.357960121514043e37],
  [1.7823914887221356e75, 4.22183785657637e37],
  [2.8066889548237318e75, 5.297819320082304e37],
  [1.4324209107939194e-75, 3.7847336904912073e-38],
  [5.75982981787146e-76, 2.3999645451280026e-38],
  [2.84709401114082e76, 1.6873334024847668e38],
  [4.5903215370624054e76, 2.1425035675728538e38],
  [7.998760572609657e76, 2.8282080143811305e38],
  [1.954729464018061e77, 4.421232253589559e38],
  [3.1001981692242718e-77, 5.567942321202934e-39],
  [5.446357966558231e77, 7.379944421578141e38],
  [1.4347678702696647e78, 1.1978179620750662e39],
  [3.191631668127788e-78, 1.786513830936606e-39],
  [1.5003212059742984e-78, 1.2248759961621823e-39],
  [9.461315908907407e-79, 9.726929581788595e-40],
  [2.0626210170940118e79, 4.541608764627367e39],
  [3.159471179501996e79, 5.620917344617331e39],
  [1.3398724719445527e-79, 3.6604268493504317e-40],
  [1.668072418684072e80, 1.2915387793961403e40],
  [2.981112023363782e80, 1.7265897090402751e40],
  [6.730415699386019e80, 2.59430447314613e40],
  [1.146987159199044e81, 3.386719886850762e40],
  [2.8388438747706336e-81, 5.328080212206488e-41],
  [1.885863173736937e-81, 4.342652615322733e-41],
  [8.168904869785256e81, 9.038199416800482e40],
  [4.345402328538356e-82, 2.0845628626976822e-41],
  [5.289561444406141e82, 2.2999046598513908e41],
  [9.392344048036107e82, 3.064693140925549e41],
  [1.2370130058650805e83, 3.517119568432498e41],
  [2.2766481688304407e-83, 4.771423444665586e-42],
  [6.272692440835879e83, 7.920033106519112e41],
  [1.1372198484272548e84, 1.0664051052143622e42],
  [3.091574311219671e84, 1.7582873232835614e42],
  [1.3746815221495492e-84, 1.172468132679754e-42],
  [1.4290753385479132e85, 3.780311281558588e42],
  [4.544362100405665e-85, 6.741188397015518e-43],
  [4.0560995194055724e85, 6.368751462732372e42],
  [9.69623186841726e85, 9.8469446370015e42],
  [2.1751510974085668e86, 1.4748393463047313e43],
  [3.3647408154590466e86, 1.8343229855887013e43],
  [1.1151391703566532e-86, 1.0560015011147728e-43],
  [7.056575060080249e-87, 8.400342290692832e-44],
  [2.142176308138171e-87, 4.628365054895919e-44],
  [5.114917830494906e87, 7.15186537240104e43],
  [8.165944247869666e-88, 2.8576116334921487e-44],
  [2.875018702470018e88, 1.6955880108298766e44],
  [2.1594319452794002e-88, 1.469500576821731e-44],
  [7.527119560918543e-89, 8.675897395035595e-45],
  [1.8374445944027442e89, 4.286542422982355e44],
  [1.6767324851823106e-89, 4.0947924064380976e-45],
  [7.417835243305932e89, 8.612685552895759e44],
  [1.3811213798299968e90, 1.1752112064773705e45],
  [3.6689238512159455e-90, 1.915443512927475e-45],
  [6.065346263910295e90, 2.4627923712546895e45],
  [1.3352594509051504e91, 3.6541202099891987e45],
  [2.3510251243523464e91, 4.8487370771700403e45],
  [1.4208669421331486e-91, 3.769438873536947e-46],
  [1.2351207183001675e92, 1.1113598509484529e46],
  [2.0458875580594228e92, 1.430345258341294e46],
  [2.501775831168855e-92, 1.5817002975181028e-46],
  [9.833334799593875e-93, 9.916317259746118e-47],
  [1.443674272924411e93, 3.79957138757046e46],
  [2.520868275101921e93, 5.020824907424996e46],
  [6.448541902391714e93, 8.030281378875658e46],
  [8.535272996692151e93, 9.2386541209703e46],
  [1.9719051461783734e94, 1.404245400981742e47],
  [4.792547889064161e94, 2.1891888655536694e47],
  [1.2199529487370906e95, 3.4927824849782594e47],
  [3.547959660180053e-95, 5.956475182673099e-48],
  [3.985044064860309e95, 6.312720542571411e47],
  [7.649929455714116e95, 8.746387514690918e47],
  [1.6129741510041275e96, 1.270029192973188e48],
  [3.0931163460970288e-96, 1.758725773421493e-48],
  [1.6813486411690822e-96, 1.2966682849399387e-48],
  [1.3180348897262367e97, 3.6304750236384175e48],
  [3.1039403407959814e97, 5.571301769601052e48],
  [1.5240410485097617e-97, 3.903896833306128e-49],
  [6.971288070729216e-98, 2.6403196910088777e-49],
  [5.735539804827279e-98, 2.394898704502401e-49],
  [2.8535834407877993e98, 1.6892552917743958e49],
  [9.871697492391928e-99, 9.935641646311489e-50],
  [2.0088993541036913e99, 4.482074691595056e49],
  [1.946317951646461e-99, 4.411709364460063e-50],
  [9.390677962280974e-100, 3.064421309526641e-50],
  [6.786330632385783e-100, 2.605058661985519e-50],
  [2.9404933734963053e100, 1.7147866845460123e50],
  [1.8531326172506335e-100, 1.3612981367983405e-50],
  [1.119927915234845e101, 3.3465324071863475e50],
  [3.884226801915178e-101, 6.232356538192578e-51],
  [2.7269202968149374e-101, 5.2219922412954017e-51],
  [8.254209708878875e101, 9.085268135216965e50],
  [6.972532168924607e-102, 2.6405552766273625e-51],
  [2.708617885299807e102, 1.6457879223338004e51],
  [5.236877514456369e102, 2.2884224947453146e51],
  [1.161433356069375e103, 3.4079808627241074e51],
  [3.003819106806815e-103, 5.480710817774292e-52],
  [4.439008636160038e103, 6.66258856313373e51],
  [9.002276106267622e-104, 3.0003793270631003e-52],
  [3.3960782967394955e-104, 1.842845163528259e-52],
  [5.4149724334913127e104, 2.3270093324890885e52],
  [9.152523378515453e104, 3.0253137653002956e52],
  [1.9679279323943353e105, 4.436133375355542e52],
  [2.888390108400342e105, 5.374374483044834e52],
  [1.2821795498726208e-105, 3.580753481981999e-53],
  [1.3003917169373775e106, 1.1403471914015386e53],
  [4.2557699836238334e-106, 2.0629517647351412e-53],
  [4.966486115455959e106, 2.2285614452951392e53],
  [7.561962584675896e-107, 8.695954567887241e-54],
  [1.8228493744993866e107, 4.2694840139054115e53],
  [5.309397876047218e107, 7.286561518334432e53],
  [8.787211807544319e107, 9.374012912058698e53],
  [2.171532430163008e108, 1.4736120351581714e54],
  [2.9516158424420328e-108, 1.7180267292571535e-54],
  [8.472579233576034e108, 2.9107695260147333e54],
  [1.2438387282560994e109, 3.526809788259213e54],
  [2.58582397072793e109, 5.0850997735815666e54],
  [7.142650627467428e109, 8.451420370249861e54],
  [8.209521882901238e-110, 2.86522632315516e-55],
  [3.345752216918009e-110, 1.8291397477825496e-55],
  [2.299685403238757e-110, 1.5164713657826703e-55],
  [6.295539155198952e110, 2.5090913006901425e55],
  [5.43310320934929e-111, 7.370958695684904e-56],
  [3.3217325941458126e111, 5.763447400771358e55],
  [9.080493504293825e-112, 3.013385721127288e-56],
  [4.972949414619484e-112, 2.2300110794835716e-56],
  [2.1721331872931473e-112, 1.4738158593573172e-56],
  [1.0727716873786853e-112, 1.0357469224567772e-56],
  [1.1873055200390316e113, 3.445729995282613e56],
  [2.3754165580184708e113, 4.8738245331756365e56],
  [1.609299549103162e-113, 4.011607594348134e-57],
  [8.969303837674172e-114, 2.9948796032018e-57],
  [5.132332826100722e-114, 2.265465255990637e-57],
  [3.724784734670678e114, 1.9299701382847035e57],
  [1.5646836489831724e-114, 1.2508731546336633e-57],
  [5.020232827545436e-115, 7.085360137315136e-58],
  [3.3172031575574013e115, 5.759516609540597e57],
  [1.6097392763493938e-115, 4.0121556255327306e-58],
  [1.1375940996735337e116, 1.0665805640801512e58],
  [3.2011962048050223e-116, 1.7891887001669283e-58],
  [2.4028349243070718e-116, 1.5501080363339427e-58],
  [1.230286517507037e117, 3.5075440375097745e58],
  [5.7063306788576066e-117, 7.554025866289847e-59],
  [3.81746944369302e117, 6.178567345018601e58],
  [1.3262966997768381e-117, 3.6418356632017843e-59],
  [6.215118914740789e-118, 2.4930140221709124e-59],
  [3.3962400369703565e-118, 1.8428890462994121e-59],
  [8.041976479629169e118, 2.8358378796449504e59],
  [9.604637524630492e118, 3.099134963926304e59],
  [3.2018333305125866e119, 5.658474468010425e59],
  [3.974461700498037e119, 6.304333192731835e59],
  [7.949516952470846e-120, 2.819488775021253e-60],
  [2.2910901954456233e120, 1.5136347628954693e60],
  [4.3047561134490487e120, 2.0747906191828246e60],
  [8.443162099771256e-121, 9.188668075282324e-61],
  [1.6803623278269906e121, 4.0992222772460026e60],
  [3.466137788532709e-121, 5.887391432997053e-61],
  [4.828680085021368e121, 6.94887047297715e60],
  [1.488585954642875e122, 1.2200762085389892e61],
  [2.085558231663234e122, 1.4441461946988726e61],
  [4.810979496264978e122, 2.1933945145059924e61],
  [9.271443477547573e-123, 9.62883351063231e-62],
  [2.430333473853147e123, 4.9298412488163825e61],
  [2.993319649043068e123, 5.471123878183592e61],
  [1.1123330255382685e-123, 3.3351657013381934e-62],
  [2.0685437908296826e124, 1.4382433002902125e62],
  [3.2908617460684498e-124, 1.8140732471618807e-62],
  [6.257641579642928e124, 2.501527849064033e62],
  [1.3046045199205295e125, 3.6119309516109654e62],
  [1.941505366402168e125, 4.40625165690995e62],
  [6.765208720831414e125, 8.225088887563108e62],
  [6.251642730647003e-126, 2.5003285245437254e-63],
  [3.0641830188726155e-126, 1.750480796487815e-63],
  [2.3196894914652613e-126, 1.5230526883418253e-63],
  [8.220375952756892e126, 2.8671198009076794e63],
  [2.09533707026623e127, 4.5774851941499824e63],
  [4.1547513406756876e127, 6.445736064000517e63],
  [1.436629465086068e-127, 3.7902895207174713e-64],
  [5.85938717595173e-128, 2.4206171064321038e-64],
  [2.471985659713248e-128, 1.5722549601490363e-64],
  [5.819519862953747e128, 2.4123681027060831e64],
  [9.218870861176231e128, 3.0362593534110736e64],
  [5.620988360142958e-129, 7.497325096421362e-65],
  [2.6361099341571754e-129, 5.1343061207500816e-65],
  [8.255185333945022e129, 9.085805046304385e64],
  [4.097861906979064e-130, 2.0243176398428837e-65],
  [4.412087305581445e130, 2.100496918726958e65],
  [6.23803731967102e130, 2.4976063179914924e65],
  [1.299767257814368e131, 3.605228505676676e65],
  [2.549273557804692e-131, 5.049033132991595e-66],
  [1.637101885409647e-131, 4.046111572126561e-66],
  [1.3221246389874849e132, 1.1498367879779655e66],
  [3.612840396282678e-132, 1.9007473257334018e-66],
  [1.70281157328564e-132, 1.3049182247503634e-66],
  [8.912245931850124e-133, 9.440469231902683e-67],
  [1.5427361520965407e133, 3.927768007528628e66],
  [4.3098017678179273e133, 6.564908047960708e66],
  [5.502730671085936e133, 7.418039276713177e66],
  [1.1386671763785346e134, 1.0670834908190336e67],
  [2.236312809103837e134, 1.495430643361248e67],
  [6.236366484285548e134, 2.4972718082510657e67],
  [5.743247779152324e-135, 7.578421853626469e-68],
  [3.7844822222579143e-135, 6.15181454715429e-68],
  [4.2707239060150784e135, 6.535077586391058e67],
  [8.865447213092549e-136, 2.9774900861451325e-68],
  [1.767098824964197e136, 1.329322694067997e68],
  [3.135905087942047e-136, 1.7708486914307633e-68],
  [5.117868324041611e136, 2.262270612469165e68],
  [8.382019173328589e-137, 9.155336789724663e-69],
  [2.6767187228501482e-137, 5.173701501681507e-69],
  [5.72089957920286e137, 7.563662855523678e68],
  [9.560405120961301e-138, 3.091990478795383e-69],
  [1.6090135834811053e138, 1.2684689919273176e69],
  [1.8358925481123047e-138, 1.3549511238831844e-69],
  [9.201073744514271e138, 3.0333271739979303e69],
  [1.3138614811165913e139, 3.624722721970042e69],
  [2.764863859467541e-139, 5.258197276127571e-70],
  [1.1469353514275982e-139, 3.3866433993374596e-70],
  [1.2456670985142225e140, 1.1160945741800837e70],
  [3.4128588296281774e-140, 1.8473924406114087e-70],
  [1.784835945015018e-140, 1.3359775241429093e-70],
  [9.357407295518402e140, 3.058987952823352e70],
  [5.048358544655811e-141, 7.105180183961425e-71],
  [4.4161437611876514e141, 6.645407257036736e70],
  [9.379659653552042e-142, 3.062623002191429e-71],
  [2.333480160468471e142, 1.527573291357397e71],
  [2.4739236141094746e-142, 1.5728711371595178e-71],
  [5.784299275618662e142, 2.4050570212821695e71],
  [5.233592582491059e-143, 7.23435731941066e-72],
  [3.9415846715456154e-143, 6.278204099538032e-72],
  [1.1868816058647257e-143, 3.445114810662666e-72],
  [6.577826324246065e-144, 2.564727339162209e-72],
  [3.0725285828553693e144, 1.7528629675064075e72],
  [4.0585639406710166e144, 2.0145877843050216e72],
  [7.10962244957678e144, 2.6663875280192824e72],
  [6.338440095951786e-145, 7.961432092250605e-73],
  [2.56909001448098e145, 5.0686191556290556e72],
  [1.110572217652152e-145, 3.3325248951090406e-73],
  [1.5958927229227292e146, 1.2632864769808665e73],
  [2.240196093605432e146, 1.4967284635515662e73],
  [1.6192150761553748e-146, 1.2724838215692076e-73],
  [5.479215508511587e-147, 7.402172322036003e-74],
  [2.0747066085163763e147, 4.554894739196918e73],
  [1.4523662643826972e-147, 3.810992343711408e-74],
  [9.021039579731393e-148, 3.0035045496438647e-74],
  [1.6374048701078094e148, 1.2796112183424343e74],
  [2.878063529808568e-148, 1.696485640908454e-74],
  [6.770526584989555e148, 2.602023555809892e74],
  [6.443503565916299e-149, 8.027143679987483e-75],
  [2.833320640745976e-149, 5.322894551600638e-75],
  [6.097769380912148e149, 7.808821537794386e74],
  [1.1834867852330407e150, 1.08788178826242e75],
  [2.462709399587796e-150, 1.5693022014856783e-75],
  [1.9879288113628234e-150, 1.4099392935026754e-75],
  [9.478052690900818e-151, 9.735529102673782e-76],
  [4.05005514259216e-151, 6.364004354643513e-76],
  [4.5346562200806216e151, 6.733985610380098e75],
  [8.142463934449162e151, 9.023560236652251e75],
  [1.1660827207891676e152, 1.0798531014861085e76],
  [2.435729824085869e-152, 1.5606824866339307e-76],
  [1.583997049543372e-152, 1.2585694456577961e-76],
  [6.60974084990108e-153, 8.130031272941747e-77],
  [3.273811472025356e-153, 5.721723055186572e-77],
  [1.2653205989779825e-153, 3.557134519494564e-77],
  [7.710625323033553e153, 8.781016639907677e76],
  [5.083379212603646e-154, 2.254635050868243e-77],
  [1.9920577485201303e-154, 1.4114027591442955e-77],
  [7.861241441904935e154, 2.803790548865042e77],
  [1.6728236347700093e155, 4.090016668389029e77],
  [3.251820444921185e155, 5.70247353779146e77],
  [4.440892237648901e155, 6.664001979027993e77],
  [1.1681842368440651e156, 1.0808257199216094e78],
  [1.7915485035054454e156, 1.3384873938537655e78],
  [6.646174087313228e156, 2.5780174722668633e78],
  [7.032837066076427e156, 2.6519496726137974e78],
  [2.4094941590204663e157, 4.908659856845315e78],
  [1.905174102120031e-157, 4.364830010573185e-79],
  [1.0178435770051598e-157, 3.190366087152319e-79],
  [1.7180009240411648e158, 1.3107253427172165e79],
  [2.6929244128629224e-158, 1.6410132275100413e-79],
  [8.485212227840385e158, 2.9129387614298355e79],
  [5.41881199646115e-159, 7.361258042251439e-80],
  [2.7472768415354795e159, 5.241447168040025e79],
  [4.799301551799175e159, 6.927699150366719e79],
  [1.1374786077016586e160, 1.066526421473776e80],
  [4.3816782112557046e-160, 2.0932458554254215e-80],
  [2.3517038127806393e-160, 1.5335265934377009e-80],
  [1.237208646167306e-160, 1.1122988115462976e-80],
  [5.7767171821779034e-161, 7.6004718157348e-81],
  [2.0612249668487447e-161, 4.540071548829099e-81],
  [6.711421209045612e161, 8.192326415033529e80],
  [7.808511908558749e-162, 2.7943714693216343e-81],
  [3.4473128106975573e162, 1.8566940541450435e81],
  [1.8369041700619853e-162, 1.3553243781700326e-81],
  [1.260738215622096e163, 3.550687561053628e81],
  [4.993183184923244e-163, 7.066245951651588e-82],
  [1.8680839626659697e-163, 4.322133689123891e-82],
  [1.1211916206658371e164, 1.0588633626043716e82],
  [1.8155531144088822e164, 1.3474246229043323e82],
  [1.963483009831839e-164, 1.4012433799421995e-82],
  [7.742158677595007e164, 2.7824734819212578e82],
  [8.524310818712796e-165, 9.232719436175236e-83],
  [3.6811311730063326e165, 6.067232625346034e82],
  [1.2546946077241453e-165, 3.542166861857506e-83],
  [1.0803027221133152e166, 1.0393761215812663e83],
  [1.9445301455466088e166, 1.3944641069409455e83],
  [2.1307013933368853e-166, 1.4596922255519776e-83],
  [6.228420706812873e166, 2.495680409590313e83],
  [2.2496095146137002e167, 4.743004864654579e83],
  [3.1351722345912756e-167, 5.599260874964905e-84],
  [5.193151860234628e167, 7.206352655979741e83],
  [5.13033154406953e-168, 2.2650235195400354e-84],
  [3.5533385843503496e168, 1.8850301282341218e84],
  [1.5884445082387458e-168, 1.2603350777625552e-84],
  [7.914791185532489e-169, 8.896511218186873e-85],
  [2.9993932318039227e169, 5.47667164599442e84],
  [1.6626831066640226e-169, 4.0776011411907645e-85],
  [8.355521887396055e-170, 2.8905919614148334e-85],
  [3.905302516244791e-170, 1.9761838265315275e-85],
  [2.0604906035449972e-170, 1.4354409091094614e-85],
  [5.560844553380822e170, 2.3581443029171946e85],
  [4.314951945149526e-171, 6.56882938212702e-86],
  [2.8204375498860285e-171, 5.310779180013069e-86],
  [1.5205372644544306e-171, 3.899406704172355e-86],
  [1.2639861342741105e172, 1.1242713792826493e86],
  [3.267362329351932e-172, 1.807584667270646e-86],
  [4.9318723152088004e172, 2.2207819152741678e86],
  [9.851995727572769e-173, 9.925722002742556e-87],
  [1.341334816546909e173, 3.662423810193065e86],
  [3.0549427174435646e173, 5.527153623198441e86],
  [7.276217164805913e173, 8.530074539419871e86],
  [1.163454185954185e174, 1.0786353350202213e87],
  [2.86877797094319e174, 1.6937467257365222e87],
  [7.223700701025096e174, 2.687694309445383e87],
  [9.652804348451466e174, 3.106896256467452e87],
  [1.881537005213637e175, 4.337668734716423e87],
  [4.5425564549011394e175, 6.739849000460722e87],
  [6.40966011070583e175, 8.006035292643814e87],
  [5.185011052471997e-176, 2.2770619342635363e-88],
  [2.749158746457103e-176, 1.6580587282895328e-88],
  [5.11714070126712e176, 2.2621097898349495e88],
  [1.2048028177393118e177, 3.4710269629308725e88],
  [2.221500865954068e177, 4.713280031945978e88],
  [1.0311112869226584e-177, 3.211092161434577e-89],
  [1.1094139445051383e178, 1.0532872089345519e89],
  [2.490487124239358e-178, 1.57812772747942e-89],
  [2.330135386790282e-178, 1.5264780990208416e-89],
  [6.750127466470132e178, 2.598100742171121e89],
  [5.9812615013650325e-179, 7.733861584852054e-90],
  [3.0495885975809073e-179, 5.522308029783297e-90],
  [7.559930703547839e179, 8.694786198376496e89],
  [1.8861963610062518e180, 1.3733886416474588e90],
  [4.123445043401793e180, 2.0306267612246702e90],
  [1.2995584883700009e-180, 1.1399817929993448e-90],
  [7.508262971805556e-181, 8.665023353578198e-91],
  [2.8581459679089006e-181, 5.34616308010605e-91],
  [2.010774000939296e-181, 4.4841654752465326e-91],
  [7.055998721396565e181, 8.399999238926492e90],
  [4.7712328879494584e-182, 2.1843151988551146e-91],
  [2.641251438357198e-182, 1.62519273883352e-91],
  [9.978590503605425e182, 3.1588907077652157e91],
  [1.805969017437007e183, 4.2496694194219475e91],
  [3.311401730637796e183, 5.754478022060555e91],
  [6.1836920680944595e183, 7.863645508346914e91],
  [1.4451878655158245e184, 1.2021596672305325e92],
  [3.012040084411346e-184, 1.735523000254202e-92],
  [1.601953307742924e-184, 1.2656829412388096e-92],
  [9.580089286745693e184, 3.095171931693891e92],
  [4.669187498473844e-185, 6.8331453215000805e-93],
  [4.144610141860627e185, 6.437864662961335e92],
  [9.492697652515747e185, 9.743047599450465e92],
  [6.05640687564016e-186, 2.460976813308114e-93],
  [3.3900197014146888e-186, 1.8412006141142493e-93],
  [1.1088203277756655e-186, 1.0530053787971197e-93],
  [5.446058191563198e-187, 7.379741317663647e-94],
  [3.4818280071502916e-187, 5.900701659252305e-94],
  [4.83733978937074e187, 6.955098697625174e93],
  [7.102790120358866e187, 8.427805242385984e93],
  [1.70224544101063e188, 1.3047012842067067e94],
  [3.587555480751102e188, 1.8940843383416436e94],
  [1.111578363088116e189, 3.3340341376298413e94],
  [4.831202505642164e-189, 6.950685222078586e-95],
  [4.39742526156247e189, 6.631308514586295e94],
  [6.600035403065669e189, 8.124060193687433e94],
  [1.3115415331164787e190, 1.1452255381000193e95],
  [4.082081721360552e-190, 2.0204162247815552e-95],
  [1.2912499941463614e-190, 1.1363318151606781e-95],
  [1.3706120747889848e191, 3.702177838501258e95],
  [4.8269484074059805e-191, 6.947624347506117e-96],
  [3.492057097322307e191, 5.909362992169551e95],
  [1.2855154679277422e-191, 3.5854085791269903e-96],
  [6.804598198429846e-192, 2.6085624773867015e-96],
  [3.044118876083912e-192, 1.7447403463220285e-96],
  [1.6633732162998678e-192, 1.2897182701271886e-96],
  [5.92061069636697e-193, 7.694550471838475e-97],
  [3.417479980242522e193, 5.845921638409569e96],
  [1.7967018876867518e-193, 4.2387520423902503e-97],
  [1.3497017505311573e194, 1.1617666506365025e97],
  [2.772095891650699e-194, 1.6649612282725082e-97],
  [1.929589035259489e-194, 1.3890964816237528e-97],
  [1.190348587338919e-194, 1.0910309745093945e-97],
  [6.688968337750512e-195, 8.178611335520544e-98],
  [3.0025634753608016e195, 5.479565197495876e97],
  [1.0389120965689685e-195, 3.2232159353182783e-98],
  [4.790853668250854e-196, 2.1888018796252104e-98],
  [2.0328636857920928e196, 1.425785287409045e98],
  [7.39313791205219e196, 2.7190325323637063e98],
  [1.0520367544930053e-196, 1.0256884295403772e-98],
  [3.7249037335101477e-197, 6.103198942775951e-99],
  [1.4647096086784373e-197, 3.827152477597982e-99],
  [8.710278890747314e197, 9.332887490346872e98],
  [1.9895606316593648e198, 1.4105178593904315e99],
  [3.337969037792304e-198, 1.8270109572173628e-99],
  [8.764480026910918e198, 2.9604864510601834e99],
  [4.882878087920101e-199, 6.987759360424557e-100],
  [2.9226901340764247e-199, 5.406191019633347e-100],
  [1.6113055482971697e-199, 4.014107059231442e-100],
  [1.394294883676362e200, 1.180802643830188e100],
  [3.43448576367021e-200, 1.8532365644110873e-100],
  [1.343973340598184e-200, 1.1592986416787454e-100],
  [1.0734811999880925e-200, 1.0360893783781844e-100],
  [1.8076042552476152e201, 4.2515929429422276e100],
  [3.885559777670769e201, 6.233425845929964e100],
  [1.4626690461917114e-201, 3.824485646713439e-101],
  [7.932134017217473e-202, 2.816404448444412e-101],
  [2.2191420551232144e202, 1.489678507303913e101],
  [7.464395046342848e202, 2.732104508678767e101],
  [5.794039047818522e-203, 7.61185854296999e-102],
  [2.828625409058378e-203, 5.318482310827383e-102],
  [2.3327529102932513e-203, 4.829858083104773e-102],
  [7.593036024779165e203, 8.713802857982939e101],
  [1.894697070646319e204, 1.376479956500028e102],
  [2.728232221942395e204, 1.651736123581002e102],
  [6.0915439225658e204, 2.4681053305249756e102],
  [4.206419733866126e-205, 6.485691739410782e-103],
  [3.5864263971120897e205, 5.988677981918956e102],
  [4.305257846248795e205, 6.561446369702944e102],
  [1.3002967826868136e206, 1.1403055654897128e103],
  [2.1373652233536096e206, 1.4619730583542262e103],
  [4.840419963714771e206, 2.2000954442284478e103],
  [1.0701181890744888e207, 3.2712660990425233e103],
  [5.158083681752355e-207, 7.181980006761613e-104],
  [3.1029727382874974e-207, 5.5704333209253094e-104],
  [1.274831055377559e-207, 3.570477636644093e-104],
  [1.3489686157315508e208, 1.1614510819365364e104],
  [2.6307645245628207e208, 1.621963169915649e104],
  [5.550183956658152e208, 2.3558828401807574e104],
  [6.286081390232014e-209, 7.928481185089622e-105],
  [1.9643885770481934e209, 4.432142345467024e104],
  [6.454330742797223e209, 8.03388495237343e104],
  [7.721811800325596e-210, 2.7788148193655502e-105],
  [1.8207023725207008e210, 1.3493340477882786e105],
  [3.218033715400385e210, 1.793887877042594e105],
  [1.5182930144642057e-210, 1.2321903320770724e-105],
  [1.658390863592661e211, 4.0723345437140364e105],
  [2.3535094105232726e211, 4.851298187622848e105],
  [1.742926489493158e-211, 4.174837109987835e-106],
  [1.3058146787525189e212, 1.1427224854497783e106],
  [3.52712070015826e-212, 1.878063018154146e-106],
  [5.9645421531910525e212, 2.4422412151937514e106],
  [7.663654818804923e-213, 8.754230302433745e-107],
  [3.720293489637861e-213, 6.099420865654264e-107],
  [2.8220309837206994e-213, 5.312279156558604e-107],
  [8.366565616612615e-214, 2.892501619120137e-107],
  [1.6998855564459006e214, 1.3037965932022911e107],
  [2.183583955369743e214, 1.477695488038636e107],
  [5.491510206105432e214, 2.3433971507419378e107],
  [1.2243259828922321e215, 3.499036985932318e107],
  [2.5079076905528487e215, 5.007901447265959e107],
  [1.5526912646129438e-215, 3.940420364140029e-108],
  [5.984306608120332e-216, 2.4462842451604704e-108],
  [4.720225715614302e-216, 2.1726080446353644e-108],
  [4.034622111402736e216, 2.0086368789312658e108],
  [1.1275867337969803e-216, 1.0618788696442643e-108],
  [5.963731360409445e-217, 7.722519899883357e-109],
  [3.0055039546043495e217, 5.48224767281117e108],
  [6.560742304146522e217, 8.099840926923517e108],
  [1.1135721656144034e218, 1.055259288333632e109],
  [2.6056709456206447e-218, 1.6142090774186114e-109],
  [5.366173850389301e218, 2.3165003454325884e109],
  [5.867216543541286e-219, 7.659775808430222e-110],
  [1.5526576580419578e219, 3.9403777205262416e109],
  [3.384048562478265e219, 5.817257569059724e109],
  [7.994346698689276e-220, 2.827427576205848e-110],
  [5.056795334717539e-220, 2.248731939275453e-110],
  [3.277143910889988e220, 1.8102883502055654e110],
  [8.323920791290883e220, 2.8851205852253184e110],
  [7.345976724953095e-221, 8.570867356897489e-111],
  [3.3397269591736347e221, 5.7790370817062894e110],
  [2.1951148131975832e-221, 4.685205239045119e-111],
  [9.83176320579265e221, 9.915524799924938e110],
  [4.188876382334804e-222, 2.046674469068006e-111],
  [1.5593712958067274e-222, 1.2487478912121242e-111],
  [8.651349400967479e-223, 9.30126303303346e-112],
  [1.9994770517608634e223, 4.471551242869597e111],
  [2.599110284403816e-223, 5.098147001022839e-112],
  [1.4067506928660039e-223, 3.7506675310749736e-112],
  [1.4481336677122486e224, 1.2033842560513448e112],
  [1.9628488748790776e224, 1.401017085862652e112],
  [1.613780409522561e-224, 1.2703465706343923e-112],
  [7.583018780057219e224, 2.753728160159826e112],
  [5.324383114632452e-225, 7.296837064531763e-113],
  [4.819180723659685e225, 6.9420319241989125e112],
  [1.0751674764144168e226, 1.0369028288197583e113],
  [1.4222458161267847e226, 1.1925794800040728e113],
  [2.4296291089302997e226, 1.5587267589062235e113],
  [9.10364186556454e-227, 9.541300679448552e-114],
  [7.664339137733834e-227, 8.754621144135156e-114],
  [3.078940893781277e-227, 5.548820499692955e-114],
  [1.7070782807027268e-227, 4.131680385391308e-114],
  [9.620445845911016e-228, 3.1016843562669328e-114],
  [2.739707225998974e228, 1.6552060977409955e114],
  [2.105544610440993e-228, 1.4510494858691046e-114],
  [1.0901656892505088e-228, 1.044109998635445e-114],
  [2.1783998058035627e229, 4.6673330776831886e114],
  [2.7334296403019473e229, 5.22822115092882e114],
  [7.906784117899939e229, 8.892009962826143e114],
  [1.2763298709108703e230, 1.1297477023259973e115],
  [3.7395189017675573e230, 1.9337835715941837e115],
  [4.662457923757396e230, 2.159272545038582e115],
  [8.065462200139888e230, 2.839975739357625e115],
  [2.7784108705151504e231, 5.271063337235809e115],
  [5.762773217325168e231, 7.591293181879599e115],
  [1.0807699610815717e-231, 3.287506594794255e-116],
  [5.1187539345449674e-232, 2.2624663388755573e-116],
  [2.2776012171256968e-232, 1.5091723616359056e-116],
  [5.760671142152797e232, 2.4001398172091554e116],
  [6.109982493590774e-233, 7.816637700182076e-117],
  [2.7776845455513406e233, 5.2703743183490685e116],
  [6.1461076150762305e233, 7.839711483898008e116],
  [6.684307446140772e-234, 2.5854027628477486e-117],
  [2.133225887264769e234, 1.460556704570134e117],
  [1.2587230854515603e-234, 1.1219282889077894e-117],
  [6.94302784909868e-235, 8.332483332775818e-118],
  [2.115442757878419e235, 4.5993942621593325e117],
  [2.6814031611038588e-235, 5.178226685945546e-118],
  [1.4644892248261278e-235, 3.8268645453244457e-118],
  [4.965498778069491e-236, 2.2283399152888436e-118],
  [4.0062724232429093e236, 2.001567491553285e118],
  [1.0070033023400693e-236, 1.003495541763923e-118],
  [8.49169358153587e236, 2.914051060214263e118],
  [2.4744875263587095e237, 4.974422103479669e118],
  [5.211096062451675e237, 7.218792185990448e118],
  [9.021973557613818e237, 9.498407002026086e118],
  [1.955698867623561e238, 1.398463037632229e119],
  [3.1870358496382564e238, 1.7852271143017787e119],
  [1.1767391497682072e-238, 1.0847760827784723e-119],
  [4.4017246051207265e-239, 6.634549423375129e-120],
  [3.269537309371882e239, 5.71798680426239e119],
  [7.930808198257907e239, 8.905508519033546e119],
  [1.4486518290238644e240, 1.2035995301693435e120],
  [2.8991415762931794e240, 1.702686576059487e120],
  [1.6817917282383117e-240, 1.2968391296680987e-120],
  [1.0670902139648442e-240, 1.0330005875917226e-120],
  [3.392635344432064e-241, 5.824633331319718e-121],
  [5.058549388122564e241, 7.112347986510899e120],
  [1.1134390902629017e-241, 3.336823474897798e-121],
  [6.7759154382504495e-242, 2.603058861848969e-121],
  [2.3950894906446155e-242, 1.547607666899016e-121],
  [1.7440131843329147e-242, 1.3206109133022167e-121],
  [1.2495592926661218e243, 3.5349105966998966e121],
  [3.782275124743448e-243, 6.150020426586767e-122],
  [1.8975316610917503e-243, 4.3560666444531706e-122],
  [9.071570348531914e243, 9.524479171341557e121],
  [2.4768707198353443e244, 1.573807713742484e122],
  [1.55066206748012e-244, 1.2452558241100982e-122],
  [9.772649676191643e244, 3.1261237461418002e122],
  [5.325183733004515e-245, 7.297385650357609e-123],
  [2.458620835671438e-245, 4.958448180299395e-123],
  [1.0286760303875382e-245, 3.207297975535697e-123],
  [1.027203508067744e246, 1.0135104873989929e123],
  [2.4430828477738524e246, 1.563036419209051e123],
  [4.0785834590031695e246, 2.019550311084913e123],
  [5.755379386024469e-247, 7.58642167693338e-124],
  [1.6868902308558232e247, 4.1071769268632965e123],
  [2.3567688475284907e-247, 4.854656370463816e-124],
  [9.859407940096723e247, 9.929455141193158e123],
  [1.8386051560036232e248, 1.3559517528303221e124],
  [2.1964082707846216e-248, 1.48202843116609e-124],
  [5.598771249322913e248, 2.3661722780311056e124],
  [8.670911616466196e-249, 9.311772987173922e-125],
  [2.5833259747972307e249, 5.082642988443346e124],
  [1.4071068008571067e-249, 3.7511422271850834e-125],
  [9.55443868847519e-250, 3.0910255075743373e-125],
  [2.5382793668679673e250, 1.59319784297744e125],
  [1.5554641250664774e-250, 1.247182474646945e-125],
  [8.61840455427426e-251, 9.283536262801078e-126],
  [4.059388356440167e-251, 6.371332950364599e-126],
  [3.7314447147512766e251, 6.1085552422412265e125],
  [1.5210973593336573e-251, 3.9001248176611696e-126],
  [9.756602885865253e251, 9.877551764412704e125],
  [2.2009398161443592e252, 1.483556475549333e126],
  [1.978437998183238e-252, 1.4065695852616883e-126],
  [1.0564351500100374e253, 3.2502848336877146e126],
  [3.6131821393046605e-253, 6.010975078391742e-127],
  [1.657707740565561e-253, 4.0714957209428095e-127],
  [6.959133905107306e-254, 2.6380170403367953e-127],
  [1.2833488995008763e254, 1.1328499015760545e127],
  [2.68978130196551e254, 1.6400552740580147e127],
  [6.644336716912607e254, 2.577661094269882e127],
  [1.2949675889582017e255, 3.5985658100946296e127],
  [2.907391109101237e255, 5.392022912693562e127],
  [1.49263741115478e-255, 3.863466592523844e-128],
  [8.603678754036722e-256, 2.9332028150192274e-128],
  [2.7210277398978328e256, 1.6495538002435182e128],
  [1.4399522364166417e-256, 1.1999800983419023e-128],
  [7.183134232594344e256, 2.680136980192308e128],
  [5.2719329627496335e-257, 7.260807780646472e-129],
  [4.340847884131169e257, 6.588511124777106e128],
  [1.4718196694482473e-257, 3.836430201956302e-129],
  [6.623656994793045e-258, 2.573646633629614e-129],
  [3.120658943705647e258, 1.766538690124178e129],
  [4.04000638809245e258, 2.0099767133209403e129],
  [8.023454180088804e258, 2.832570242745765e129],
  [4.004851762548488e-259, 6.328389813016016e-130],
  [5.801938865318406e259, 7.617045926944649e129],
  [8.689808547636416e259, 9.32191426029891e129],
  [2.1927090421968508e260, 1.4807798763478826e130],
  [3.303438678181491e260, 1.8175364310465667e130],
  [1.2874125055248046e-260, 1.1346420164636971e-130],
  [1.587189482040237e261, 3.9839546709773654e130],
  [2.9535217379352106e-261, 5.434631301141974e-131],
  [5.035444923640649e261, 7.09608689605803e130],
  [1.1592636133551025e262, 1.076691048237656e131],
  [3.098400364216125e262, 1.7602273615121784e131],
  [2.268479071515142e-262, 1.5061470949130906e-131],
  [9.677233455330007e262, 3.11082520488214e131],
  [2.1735445747919434e263, 4.6621288858116565e131],
  [2.8565577802327522e-263, 5.344677520891931e-132],
  [1.0849558012731932e-263, 3.2938667266196323e-132],
  [6.291151023577279e-264, 2.508216701877507e-132],
  [3.0168484729305057e264, 1.7369077329928916e132],
  [1.383677885909719e-264, 1.1762983830260581e-132],
  [9.412846028260774e264, 3.0680361843141246e132],
  [2.2645222175441895e265, 4.758699630722861e132],
  [4.7259317591760604e265, 6.874541264096144e132],
  [1.1701821045843852e-265, 3.420792458750436e-133],
  [4.778529516934361e-266, 2.1859847933904668e-133],
  [3.9044132595607796e266, 1.9759588203099728e133],
  [1.1997692490875455e-266, 1.0953397870467161e-133],
  [1.5527737081534268e267, 3.940524975372478e133],
  [3.2344371605411528e267, 5.687211232705493e133],
  [5.673793622551648e267, 7.532458843267349e133],
  [1.0432719139857164e268, 1.0214068307906094e134],
  [2.6396070789187234e268, 1.6246867633235409e134],
  [6.1645455350787286e268, 2.4828502844671744e134],
  [7.57335196315868e-269, 8.70250076883575e-135],
  [2.6411315435684497e269, 5.139194045342567e134],
  [4.366168809024183e269, 6.6076991525221415e134],
  [7.550788368437509e269, 8.689527241707405e134],
  [1.9822642738282828e270, 1.4079290727264221e135],
  [3.536521323110423e-270, 1.8805640970491866e-135],
  [1.6057482754260527e-270, 1.2671812322734474e-135],
  [1.262544295346429e271, 3.553229932535227e135],
  [2.3010769290058998e271, 4.7969541680173473e135],
  [3.7062783835457044e271, 6.087921142348762e135],
  [6.94237193723016e-272, 2.6348381235343777e-136],
  [5.510587017365919e-272, 2.3474639544337885e-136],
  [4.259902098903556e272, 2.0639530273006594e136],
  [7.895718756015192e-273, 8.88578570302885e-137],
  [1.810050052625588e273, 4.254468301239989e136],
  [2.416242194745764e-273, 4.91552865391482e-137],
  [7.060772991181245e273, 8.402840585885969e136],
  [7.625987987080336e-274, 2.7615191447969967e-137],
  [2.881851068004715e-274, 1.697601563384269e-137],
  [6.42722295033941e274, 2.5351968267452944e137],
  [7.056728511863852e274, 2.6564503593825826e137],
  [3.040091767513743e-275, 5.513702719147762e-138],
  [2.2283448684906393e-275, 4.720534788019933e-138],
  [1.0072841238605406e-275, 3.1737739740891136e-138],
  [1.6046782981249131e276, 1.2667589739666e138],
  [4.307671534205205e276, 2.075493082186786e138],
  [5.842932421290422e276, 2.4172158408570846e138],
  [5.705890005196658e-277, 7.553734179329227e-139],
  [3.4712871927451646e-277, 5.891763057646806e-139],
  [4.823430519741443e277, 6.945092166228929e138],
  [7.136419113470423e277, 8.447732899109928e138],
  [4.08963991290667e-278, 2.022285813851907e-139],
  [2.595217100722346e-278, 1.6109677528499278e-139],
  [7.93017940128612e278, 2.8160574215179135e139],
  [1.1381236275635708e279, 3.3736087911368307e139],
  [3.422488386655499e279, 5.850203745730143e139],
  [1.2079372735258513e-279, 3.475539200650528e-140],
  [1.1898249394367005e280, 1.090790969634742e140],
  [3.8411962793995745e-280, 1.9598970073449203e-140],
  [5.529110560059116e280, 2.3514060814880778e140],
  [6.582939661533738e-281, 8.113531698054639e-141],
  [2.800873919861212e-281, 5.292328334354561e-141],
  [2.43156480510695e-281, 4.9310899455464713e-141],
  [9.052455275105984e281, 9.514439171651676e140],
  [1.8875044651665902e282, 1.3738647914429536e141],
  [2.7174477010643608e-282, 1.6484682893717916e-141],
  [8.986807408557548e282, 2.9978004284070593e141],
  [8.055540602950735e-283, 8.975266348666615e-142],
  [3.3644682898381725e283, 5.800403684088007e141],
  [5.278122844502319e283, 7.265069059893594e141],
  [9.083181777796106e-284, 3.0138317434448966e-142],
  [2.9476442971654457e-284, 1.7168704951642235e-142],
  [1.7646609632018615e-284, 1.3284054212482956e-142],
  [1.1308465452883921e-284, 1.0634126881358864e-142],
  [6.086715970811502e-285, 7.801740812672196e-143],
  [3.538741008808386e285, 5.948731805022299e142],
  [1.003022139565505e-285, 3.1670524775657015e-143],
  [1.0806559149307645e286, 1.0395460138593023e143],
  [2.4479547408340165e-286, 1.5645941137668953e-143],
  [7.356097331454571e286, 2.71221262652001e143],
  [9.958473995393397e286, 3.1557049918193235e143],
  [2.9012650444171675e287, 5.386339243323955e143],
  [1.4262134993751524e-287, 3.7765241947790463e-144],
  [9.560856866769819e287, 9.777963421270209e143],
  [5.360153478949893e-288, 2.3152005267254698e-144],
  [2.985471533025146e-288, 1.7278517103690196e-144],
  [1.3678899290477521e-288, 1.1695682660912754e-144],
  [4.918892186481433e-289, 7.013481436833945e-145],
  [3.3538002972446483e-289, 5.7912004776597474e-145],
  [1.5019152605051456e-289, 3.875455148115052e-145],
  [1.2806299811228752e290, 1.1316492306023431e145],
  [1.925405008294064e290, 1.3875896397328947e145],
  [1.9668724315129337e-290, 1.4024522920630612e-145],
  [1.1746025155475246e291, 3.427247460495849e145],
  [6.215282046104887e-291, 7.883706010566913e-146],
  [3.064717337940106e-291, 5.53598892515159e-146],
  [1.1859264944087496e-291, 3.4437283493457344e-146],
  [1.5530285678267847e292, 1.2462056683496448e146],
  [2.566318941170117e292, 1.6019734520803138e146],
  [7.6196505926992395e292, 2.7603714591879187e146],
  [8.866804451319227e-293, 9.416371090456889e-147],
  [3.541785235884867e-293, 5.951289974354188e-147],
  [6.135161510116648e293, 7.832727181586658e146],
  [1.2461504700719871e294, 1.1163110991439559e147],
  [3.9085284121159114e-294, 1.9769998513191426e-147],
  [1.6225440935468405e-294, 1.2737912283992382e-147],
  [1.0547409643958778e-294, 1.027005824908446e-147],
  [5.353607023922659e-295, 7.316834714494143e-148],
  [2.7552376817847925e-295, 5.249035798872773e-148],
  [4.7000955374089e295, 6.855724277863646e147],
  [9.357622017220938e295, 9.673480251295775e147],
  [1.8104389901117752e296, 1.345525544206343e148],
  [1.6168227993124728e-296, 1.271543471263359e-148],
  [1.209584855930238e-296, 1.0998112819617e-148],
  [1.3948114861279186e297, 3.7347175075605365e148],
  [2.653442227639427e297, 5.151157372512925e148],
  [1.40998731625735e-297, 3.754979781912747e-149],
  [2.092459027978939e298, 1.446533452077393e149],
  [2.7568763180772853e298, 1.6603843886514006e149],
  [7.122994738351322e298, 2.6688939166537367e149],
  [9.68872464921942e298, 3.1126716256649077e149],
  [2.2939424654665187e299, 4.789511943263655e149],
  [6.636893002415853e299, 8.1467128355035645e149],
  [6.905542849360564e299, 8.30995959638828e149],
  [1.3415205933077406e300, 1.1582403003296598e150],
  [3.296046200672238e300, 1.8155016388514327e150],
  [5.821518575208046e300, 2.412782330673044e150],
  [5.621832682768831e-301, 7.497888157854071e-151],
];

for (let i = 0; i < examples.length; ++i) {
  const example = examples[i];
  assert.sameValue(Math.sqrt(example[0]), example[1], "Math.sqrt(" + example[0] + ") is " + example[1]);
}

reportCompare(0, 0);
