/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* EDITBOX.C --- Edit boxes. */

#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <graphics.h>
#include <string.h>
#include <ctype.h>
#include <mem.h>
#include "window.h"
#include "key.h"
#include "resource.h"

enum {

edit_x_margin = 1,
edit_y_margin = 2,
edit_cursor_margin = 0,
edit_border_width = 1,
edit_border_color = cBLACK,
edit_color = cBLACK,
edit_bg_color = cWHITE,
edit_disabled_color = cLIGHTGRAY,
edit_unedited_bg_color = cLIGHTGRAY,
edit_insert_cursor_color = cLIGHTRED,
edit_replace_cursor_color = cDARKGRAY,
edit_skip = 8,
edit_button = bLEFT,

};


/* Edit box that currently has the keyboard focus. */
static EDIT_BOX eb;

/* Keystroke handlers. */
LOCAL(void) insert_char(char);
LOCAL(void) replace_char(char);
LOCAL(void) clear_all(void);
LOCAL(void) left(void);
LOCAL(void) right(void);
LOCAL(void) del(void);
LOCAL(void) change_edit_mode(void);
LOCAL(void) bs(void);
LOCAL(void) home(void);
LOCAL(void) end(void);

/* Pointer to control key handler. */
TYPEDEF_LOCAL(void) (*CTRL_KEY_HANDLER)(void);

/* Entry in table that dispatches keystrokes to
   correct handler. */
struct ctrl_entry {
  int key;
  CTRL_KEY_HANDLER fun;
};

/* Keystroke handler table. */
LOCAL(void) (*default_key_action)(char) = insert_char;
static struct ctrl_entry ctrls[] = {
  ctrl('Y'),    clear_all,
  LEFT_ARROW,   left,
  RIGHT_ARROW,  right,
  DEL,          del,
  INS,          change_edit_mode,
  '\b',         bs,
  HOME,         home,
  END,          end,
};

#define N_CTRLS (sizeof ctrls/sizeof(struct ctrl_entry))
#define insert_mode_p   (default_key_action == insert_char)

/* Modify handler table entry for keystroke
   to get replace and insert edit modes. */
LOCAL(void) change_edit_mode(void)
{
  default_key_action = insert_mode_p ? replace_char : insert_char;
}

/* Draw the cursor at its current position. */
LOCAL(void) draw_cursor(int color)
{
  int x0, y0;

  if (!w_status_p(&eb->window, wsVISIBLE))
    return;
  push_graphics_state(&eb->window, 0);
  protect_cursor(&eb->window);
  setfillstyle(SOLID_FILL, color);
  x0 = eb->x_left + (eb->cursor_pos - eb->left_pos) * eb->ch_width;
  y0 = edit_y_margin + eb->ch_height + edit_cursor_margin;
  bar(x0, y0, x0 + eb->ch_width - 2, y0 + 1);
  unprotect_cursor();
  pop_graphics_state();
}

/* Keep `stack' of cursor off and on calls
   because ticks are working asynchronously
   with keystrokes and mouse button presses.  */
static c_level = 1;

/* Turn cursor off and count. */
LOCAL(void) cursor_off(void)
{
  if (++c_level > 0)
    draw_cursor(edit_bg_color);
}

/* Turn cursor on and count. */
LOCAL(void) cursor_on(void)
{
  if (--c_level <= 0)
    draw_cursor(insert_mode_p ? edit_insert_cursor_color : edit_replace_cursor_color);
}

/* Draw edit box title. */
LOCAL(void) draw_title(EDIT_BOX eb)
{
  if (!w_status_p(&eb->window, wsVISIBLE) || eb->title.str == NULL)
    return;
  push_graphics_state(&eb->window, 0);
  protect_cursor(&eb->window);
  setcolor(eb_status_p(eb, ebENABLED) ? edit_color : edit_disabled_color);
  settextjustify(LEFT_TEXT, BOTTOM_TEXT);
  out_hot_textxy(0, -edit_y_margin - 1, &eb->title);
  unprotect_cursor();
  pop_graphics_state();
}

/* Draw edit box text starting at position p, which must
   be >= p0.  If edit box not full, clear to end.  These
   don't use the globals because we may have to refresh
   a box that's not the one with the focus. */
LOCAL(void) _draw_text(EDIT_BOX eb, int p)
{
  char *chp;
  char chb[2];
  int i, x, d, n, y1;

  if (!w_status_p(&eb->window, wsVISIBLE))
    return;
  chb[1] = '\0';
  y1 = edit_y_margin + eb->ch_height - 1;
  d = p - eb->left_pos;
  n = min(eb->box_len - d, eb->str_len - p);
  push_graphics_state(&eb->window, 0);
  settextjustify(LEFT_TEXT, TOP_TEXT);
  setcolor(edit_color);
  setfillstyle(SOLID_FILL,
	       eb_status_p(eb, ebUNEDITED) ?
		 edit_unedited_bg_color : edit_bg_color);
  protect_cursor(&eb->window);
  for (chp = eb->buf + p, x = eb->x_left + d * eb->ch_width, i = 0;
       i < n;
       ++chp, x += eb->ch_width, ++i) {
    bar(x, edit_y_margin, x + eb->ch_width - 1, y1);
    chb[0] = *chp;
    outtextxy(x, edit_y_margin, chb);
  }
  setfillstyle(SOLID_FILL, edit_bg_color);
  if (d + n < eb->box_len)
    bar(eb->x_left + (d + n) * eb->ch_width, edit_y_margin,
	eb->x_left + eb->box_len * eb->ch_width - 1, y1);
  unprotect_cursor();
  pop_graphics_state();
}

/* Draw text in the box with the focus. */
#define draw_text(P)    _draw_text(eb, P)

/* Draw edit box character at position p,
   where:  left_pos <= p < left_pos + box_len. */
LOCAL(void) draw_char(int p)
{
  int x;
  char chb[2];

  push_graphics_state(&eb->window, 0);
  settextjustify(LEFT_TEXT, TOP_TEXT);
  setcolor(edit_color);
  setfillstyle(SOLID_FILL, edit_bg_color);
  protect_cursor(&eb->window);
  x = eb->x_left + (p - eb->left_pos) * eb->ch_width;
  bar(x, edit_y_margin, x + eb->ch_width - 1, edit_y_margin + eb->ch_height);
  chb[0] = eb->buf[p];
  chb[1] = '\0';
  outtextxy(x, edit_y_margin, chb);
  unprotect_cursor();
  pop_graphics_state();
}

/* Move cursor left 1 position.  If
   that runs off left edge of box,
   shift right by `skip' positions. */
LOCAL(void) move_cursor_left(int skip)
{
  if (eb->cursor_pos > 0) {
    if (eb->cursor_pos == eb->left_pos) {
      eb->left_pos = max(0, eb->left_pos - min(skip, eb->box_len - 1));
      draw_text(eb->left_pos);
    }
    --eb->cursor_pos;
  }
}

/* Symmetric with above, but have to
   worry about hitting end of string. */
LOCAL(void) move_cursor_right(int skip)
{
  if (eb->cursor_pos < eb->str_len) {
    if (eb->cursor_pos == eb->left_pos + eb->box_len - 1) {
      eb->left_pos = min(eb->left_pos + min(skip, eb->box_len - 1), eb->buf_len - eb->box_len + 1);
      draw_text(eb->left_pos);
    }
    ++eb->cursor_pos;
  }
}

/* Handle arrow keys. */
LOCAL(void) left(void) { move_cursor_left(edit_skip); }
LOCAL(void) right(void) { move_cursor_right(edit_skip); }

/* Insert char `key' at current cursor pos. */
LOCAL(void) insert_char(char key)
{
  if (eb->str_len < eb->buf_len) {
    int insert_pos = eb->cursor_pos;
    char *buf = eb->buf;
    movmem(buf+insert_pos, buf + insert_pos + 1, eb->str_len - insert_pos);
    ++eb->str_len;
    buf[insert_pos] = key;
    draw_text(insert_pos);
    right();
  }
}

/* Replace char at cursor with `key'.  Insert if
   cursor is at end of string. */
LOCAL(void) replace_char(char key)
{
  if (eb->cursor_pos < eb->str_len) {
    int replace_pos = eb->cursor_pos;
    eb->buf[replace_pos] = key;
    draw_char(replace_pos);
    right();
  }
  else insert_char(key);
}

/* Delete the char at the current cursor position. */
LOCAL(void) delete_char(void)
{
  int delete_pos = eb->cursor_pos;
  char *delete_ptr = eb->buf + delete_pos;
  --eb->str_len;
  movmem(delete_ptr + 1, delete_ptr, eb->str_len - delete_pos);
  draw_text(delete_pos);
}

/* Handle a delete keystroke. */
LOCAL(void) del(void)
{
  if (eb->cursor_pos < eb->str_len)
    delete_char();
}

/* Handle a backspace keystroke. */
LOCAL(void) bs(void)
{
  if (eb->cursor_pos > 0 && eb->cursor_pos <= eb->str_len) {
    left();
    delete_char();
  }
}

/* Handle an `End' keystroke. */
LOCAL(void) end(void)
{
  eb->left_pos = min(eb->buf_len - eb->box_len + 1,
		     max(0, eb->str_len - eb->box_len + 1));
  eb->cursor_pos = eb->str_len;
  draw_text(eb->left_pos);
}

/* Handle a `Home' keystroke. */
LOCAL(void) home(void)
{
  draw_text(eb->left_pos = eb->cursor_pos = 0);
}

/* Handle ^Y. */
LOCAL(void) clear_all(void)
{
  draw_text(eb->left_pos = eb->cursor_pos = eb->str_len = 0);
}

/* See if this is the first control key
   (or mouse click) since the call to
   open_edit.  If a control (or mouse
   click), just erase the lolite, else
   clear the edit buffer.  */
LOCAL(void) check_ctrl_key(void)
{
  if (eb_status_p(eb, ebUNEDITED)) {
    eb->status &= notbit(ebUNEDITED);
    draw_text(eb->left_pos);
  }
}

/* Same as above for printable keystrokes. */
#define check_printable_key()		\
{                                       \
  if (eb_status_p(eb, ebUNEDITED)) {    \
    eb->status &= notbit(ebUNEDITED);   \
    clear_all();                        \
  }					\
}

/* Disable arrow buttons if they
   can't do anything anyway. */
LOCAL(void) check_arrows(void)
{
  if (eb->cursor_pos > 0)
    enable_button(&eb->larrow);
  else
    disable_button(&eb->larrow);
  if (eb->cursor_pos < eb->str_len)
    enable_button(&eb->rarrow);
  else
    disable_button(&eb->rarrow);
}

/* Toggle cursor on timer tick. */
#pragma argsused
static void toggle_cursor(EVENT e)
{
  static int p = 0;

  if ((p = 1 - p) == 0)
    cursor_on();
  else
    cursor_off();
}

/* Move left on left arrow push. First
   moves cursor to left edge of box,
   rest scroll to left. */
static void larrow_press_action(ENV env)
{
  if (env == &eb->window) {
    cursor_off();
    check_ctrl_key();
    if (eb->cursor_pos > eb->left_pos)
      eb->cursor_pos = eb->left_pos;
    else
      move_cursor_left(1);
    check_arrows();
    cursor_on();
  }
}

/* Move right on right arrow push. Symmetric
   with above. */
static void rarrow_press_action(ENV env)
{
  if (env == &eb->window) {
    int right_pos = eb->left_pos + eb->box_len - 1;
    cursor_off();
    check_ctrl_key();
    if (eb->cursor_pos < right_pos)
      eb->cursor_pos = min(right_pos, eb->str_len);
    else
      move_cursor_right(1);
    check_arrows();
    cursor_on();
  }
}

/* Handle a map event by redrawing edited text. */
LOCAL(void) handle_map(EVENT e)
{
  EDIT_BOX map_eb = (EDIT_BOX)e->map.window;
  _draw_text(map_eb, map_eb->left_pos);
  draw_title(map_eb);
}

/* Move the cursor in response to a mouse
   click near a letter. */
LOCAL(void) handle_button_press(EVENT e)
{
  int p;

  if (e->mouse.button != edit_button)
    return;

  if (e->mouse.window == &eb->window)
    if (double_click_p(&eb->window))
      (*eb->key_actions[0].action.code)(eb->buf, eb->key_actions[0].action.env);
    else {
      check_ctrl_key();
      cursor_off();
      p = eb->left_pos + (e->mouse.x - eb->x_left) / eb->ch_width;
      if (p < eb->left_pos)
	p = eb->left_pos;
      if (p > eb->str_len)
	p = eb->str_len;
      if (p - eb->left_pos >= eb->box_len)
	p = eb->left_pos + eb->box_len - 1;
      eb->cursor_pos = p;
      check_arrows();
      cursor_on();
    }
  else
    set_focus(e->mouse.window);
}

/* Check for match with caller's handler table. */
LOCAL(int) call_key_action(EDIT_BOX eb, int key)
{
  KEY_NAME_ACTION a;
  int i, n = eb->n_key_actions;

  for (i = 0, a = eb->key_actions; i < n; ++i, ++a) 
    if (a->key == key) {
      (*a->action.code)(get_edit_text(eb), a->action.env);
      return 1;
    }
  return 0;
}

/* Dispatch key event to the right handler. */
LOCAL(void) handle_keystroke(EVENT e)
{
  struct ctrl_entry *p;
  int i, key = e->keystroke.key;

  if (call_key_action(eb, key))
    return;

  /* Check for match in edit handler table. */
  for (i = 0, p = ctrls; i < N_CTRLS; ++i, ++p)
    if (p->key == key) {
      cursor_off();
      check_ctrl_key();
      (*p->fun)();
      check_arrows();
      cursor_on();
      return;
    }

  /* Handle printable characters by insertion/replacement. */
  if (isprint(key)) {
    /* Call insert/replace action. */
    cursor_off();
    check_printable_key();
    (*default_key_action)(key);
    check_arrows();
    cursor_on();
    return;
  }

  /* We didn't handle key. */
  refuse_keystroke(e);
}

/* Allow an edit box to be opened. */
void enable_edit_box(EDIT_BOX eb)
{
  eb->status |= bit(ebENABLED);
  draw_title(eb);
  eb->window.event_mask |= bit(eBUTTON_PRESS)|bit(eGAIN_FOCUS)|bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY);
}

/* Close an edit box if it was open and disable it. */
void disable_edit_box(EDIT_BOX eb)
{
  if (!eb_status_p(eb, ebENABLED))
    return;
  eb->status &= notbit(ebENABLED);
  draw_title(eb);
  eb->window.event_mask &= notbit(eBUTTON_PRESS)&notbit(eGAIN_FOCUS)&notbit(eKEYSTROKE)&notbit(eVISIBLE_HOTKEY);
  unset_focus(&eb->window);
}

/* Turn on the current edit box. */
LOCAL(void) handle_gain_focus(EVENT e)
{
  eb = (EDIT_BOX)e->focus.window;
  eb->status |= bit(ebUNEDITED);
  home();
  global_event_handler[eTICK(tCURSOR)] = toggle_cursor;
  start_timer(tCURSOR, delta(.3), 1);
  cursor_on();
  check_arrows();
}

/* Turn off the current edit box. */
#pragma argsused
LOCAL(void) handle_lose_focus(EVENT e)
{
  EDIT_BOX tmp;

  disable_button(&eb->larrow);
  disable_button(&eb->rarrow);
  cursor_off();
  stop_timer(tCURSOR);
  global_event_handler[eTICK(tCURSOR)] = null_handler_code;
  eb->status &= notbit(ebUNEDITED);
  home();
  tmp = eb;
  eb = NULL;

  /* Must do this absolutely last because
     validator may set the focus back to an edit box! */
  if (w_status_p(&tmp->window, wsVISIBLE))
    call_key_action(tmp, VALIDATOR_PSEUDOKEY);
}

LOCAL(void) handle_hotkey(EVENT e)
{
  EDIT_BOX eb = (EDIT_BOX)e->hotkey.window;
  if (key_hot_p(e->hotkey.key, &eb->title))
    set_focus(&eb->window);
}

BeginDefDispatch(editbox)
  Dispatch(eMAP, handle_map)
  Dispatch(eBUTTON_PRESS, handle_button_press)
  Dispatch(eKEYSTROKE, handle_keystroke)
  Dispatch(eGAIN_FOCUS, handle_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_lose_focus)
  Dispatch(eVISIBLE_HOTKEY, handle_hotkey)
EndDefDispatch(editbox)

/* Make a new edit box.  Some trouble is taken
   to accomodate different font heights.  */
void open_edit_box(EDIT_BOX eb, int x, int y, WINDOW parent)
{
  unsigned mask;

  int chw = textwidth("W");
  int chh = textheight("Hy");

  int edit_width = edit_x_margin + eb->box_len * chw + edit_x_margin;
  int edit_height = (edit_y_margin + chh + edit_y_margin) | 1;
  int arrow_width = edit_height;
  int arrow_shadow_width = (arrow_width - larrow_width) >> 1;
  int x_rarrow = arrow_width + edit_border_width*2 + edit_width;

  assert(arrow_shadow_width >= 0);

  /* Make edit box window. */
  mask = bit(eMAP)|bit(eLOSE_FOCUS);
  if (eb_status_p(eb, ebENABLED))
    mask |= bit(eBUTTON_PRESS)|bit(eGAIN_FOCUS)|bit(eKEYSTROKE)|bit(eVISIBLE_HOTKEY);
  open_window(&eb->window, parent, x, y,
	      x_rarrow + arrow_width, edit_height,
	      edit_border_width, edit_border_color, edit_bg_color,
	      mask);
  SetDispatch(&eb->window, editbox);

  SetBitmapButtonDesc(&eb->larrow, larrow, 0, arrow_shadow_width, edit_border_width, Repeat, larrow_press_action, eb);
  SetBitmapButtonDesc(&eb->rarrow, rarrow, 0, arrow_shadow_width, edit_border_width, Repeat, rarrow_press_action, eb);

  open_button(&eb->larrow, 0, 0, &eb->window);
  map_window(&eb->larrow.window);
  open_button(&eb->rarrow, x_rarrow, 0, &eb->window);
  map_window(&eb->rarrow.window);

  eb->cursor_pos = eb->left_pos = eb->str_len = 0;
  eb->x_left = arrow_width + edit_border_width + edit_x_margin;
  eb->ch_width = chw;
  eb->ch_height = chh;
  eb->buf = malloc(eb->buf_len + 1);
}

/* Set the text buffer in an edit box.  Truncate if too long. */
void set_edit_text(EDIT_BOX set_eb, char *str)
{
  int len;
  if (str == NULL)
    str = "";
  len = min(strlen(str), set_eb->buf_len);
  memcpy(set_eb->buf, str, len);
  set_eb->buf[len] = '\0';
  set_eb->str_len = len;
  _draw_text(set_eb, set_eb->left_pos);
  if (set_eb == eb) {
    cursor_off();
    home();
    cursor_on();
  }
}

/* Return the text in an edit box as a perishable string.
   Caller should copy if edit is still open. */
char *get_edit_text(EDIT_BOX eb)
{
  eb->buf[eb->str_len] = '\0';
  return(eb->buf);
}
