/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* COMMAND.C -- Invertible (Undoable) commands. */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "env.h"

/*  ----- Commands -------------------------------------------------- */

/* Command to execute to undo the last one. 
   Everything is designed so this could be
   expanded to a vector for n-level undo. */
static COMMAND undo_command = NULL;

/* Allocate a command. */
COMMAND cmd_alloc(COMMAND_TYPE type, CANVAS canvas, unsigned n_obj, ... )
{
  size_t size;
  COMMAND cmd;
  va_list ap;

  size = sizeof(COMMAND_REC) + (n_obj - 1) * sizeof(index);
  if (binary_command_type_p(type))
    size += n_obj * sizeof(index);
  cmd = malloc(size);
  cmd->type = type;
  cmd->canvas = canvas;
  cmd->n_obj = n_obj;
  cmd->other_origin = INULL;
  cmd->cont_pos_from = canvas->cont_pos;
  set_cont_pos_invalid(&cmd->cont_pos_to);
  va_start(ap, n_obj);
  if (type == cMOVE) {
    cmd->move_d.x = va_arg(ap, CC);
    cmd->move_d.y = va_arg(ap, CC);
  }
  else if (type == cADD_AND_MOVE_CONT_POS) 
    cmd->cont_pos_to = va_arg(ap, CP_REC);
  va_end(ap);
  return cmd;
}

/* Resize a command to accomodate more/fewer objects. */
COMMAND cmd_realloc(COMMAND cmd, unsigned n_obj)
{
  unsigned size;
  int i;

  size = sizeof(COMMAND_REC) + (n_obj - 1) * sizeof(index);
  if (binary_command_type_p(cmd->type)) {
    
    /* Adjust size for double contents. */
    size += n_obj * sizeof(index);

    /* Request is smaller than old: shift down before chopping block. */
    if (n_obj < cmd->n_obj) {
      for (i = 0; i < n_obj; i++)
	cmd->obj[n_obj + i] = cmd->obj[cmd->n_obj + i];
    }
  }

  /* Grow or shrink the command. */
  cmd = realloc(cmd, size);

  if (binary_command_type_p(cmd->type) && n_obj > cmd->n_obj) {

    /* Binary request is larger than old: shift up after growing. */
    for (i = cmd->n_obj - 1; i >= 0; i--)
      cmd->obj[n_obj + i] = cmd->obj[cmd->n_obj + i];
  }

  cmd->n_obj = n_obj;
  return cmd;
}

/* Free storage for a command.  If it's
   an add command, free its objects too. */
void destroy_command(COMMAND cmd)
{
  if (cmd == NULL)
    return;
  if (bit(cmd->type) & (bit(cADD)|
			bit(cADD_AND_UNPICK)|
			bit(cADD_AND_MOVE_CONT_POS)|
			bit(cREPLACE)))
    destroy_objects(cmd->canvas, cmd->obj, cmd->n_obj);
  if (cmd->type == cDELETE && cmd->other_origin != INULL)
    destroy_objects(cmd->canvas, &cmd->other_origin, 1);
			     
  free(cmd);
}

/* Execute a command structure, invert 
   it, and make it the undo command. */
void exec_command(COMMAND cmd)
{
  assert(cmd->n_obj > 0);

  switch (cmd->type) {

    case cADD:
    case cADD_AND_MOVE_CONT_POS:
      add_objects(cmd->canvas, cmd->obj, cmd->n_obj, &cmd->other_origin);
      cmd->type = cDELETE;
      break;

    case cADD_AND_UNPICK:
      pick_objects(cmd->canvas, cmd->obj + cmd->n_obj, cmd->n_obj, 0);
      add_objects(cmd->canvas, cmd->obj, cmd->n_obj, &cmd->other_origin);
      assert(cmd->other_origin == INULL);
      cmd->type = cDELETE_AND_PICK;
      break;

    case cDELETE:
      delete_objects(cmd->canvas, cmd->obj, cmd->n_obj);
      if (cmd->other_origin != INULL)
	add_objects(cmd->canvas, &cmd->other_origin, 1, &cmd->other_origin);
      cmd->type = cADD;
      break;

    case cDELETE_AND_PICK:
      pick_objects(cmd->canvas, cmd->obj + cmd->n_obj, cmd->n_obj, 1);
      delete_objects(cmd->canvas, cmd->obj, cmd->n_obj);
      assert(cmd->other_origin == INULL);
      cmd->type = cADD_AND_UNPICK;
      break;

    case cMOVE:
      move_objects(cmd->canvas, cmd->obj, cmd->n_obj, cmd->move_d);
      set_cont_pos_invalid(&cmd->canvas->cont_pos);
      neg(&cmd->move_d.x);
      neg(&cmd->move_d.y);
      break;

    case cREPLACE: {
      int i, n_obj;
      index origin;

      n_obj = cmd->n_obj;
      delete_objects(cmd->canvas, cmd->obj + n_obj, n_obj);
      add_objects(cmd->canvas, cmd->obj, n_obj, &origin);
      assert(origin == INULL);
      for (i = 0; i < n_obj; ++i) 
	swap_indices(&cmd->obj[i], &cmd->obj[i + n_obj]);
    } break;

    case cMAKE_THICK:
      set_object_line_thickness(cmd->canvas, cmd->obj, cmd->n_obj, ltTHICK);
      cmd->type = cMAKE_THIN;
      break;

    case cMAKE_THIN:
      set_object_line_thickness(cmd->canvas, cmd->obj, cmd->n_obj, ltTHIN);
      cmd->type = cMAKE_THICK;
      break;

    default:
      assert(0);
  }
  cmd->canvas->cont_pos = cmd->cont_pos_to;
  swap_cp_recs(&cmd->cont_pos_from, &cmd->cont_pos_to);
  if (cmd != undo_command) {
    destroy_command(undo_command);
    undo_command = cmd;
  }
  enable_button(&tool_selector_buttons[tsUNDO]);
}

/* Execute the cached undo command, if there is one. */
void undo_last_command(void)
{
  if (undo_command != NULL) 
    exec_command(undo_command);
}

/* Dump the cached undo command. */
void clear_undo(void)
{
  destroy_command(undo_command);
  undo_command = NULL;
  disable_button(&tool_selector_buttons[tsUNDO]);
}
