## -*-Tcl-*- (install)
 # ###################################################################
 #  FILE: "version.tcl"
 #                                    created: 23/7/97 {12:24:48 am} 
 #                                last update: 5/3/98 {7:48:07 PM} 
 #                                    version: 1.0.2
 #  Author: Jonathan Guyer
 #  E-mail: <j-guyer@nwu.edu>
 #    mail: Northwestern University
 #          Evanston, IL 60208
 #          
 # ###################################################################
 ##

alpha::extension version 1.1 {
	aecoerce::register "hexd" "vers" aecoerce::hexd:vers
	aecoerce::register "TEXT" "vers" aecoerce::TEXT:vers
}

## 
 # Copyright (c) 1998  Jonathan	Guyer
 # 
 # This	program	is free	software; you can redistribute it and/or modify
 # it under	the	terms of the GNU General Public	License	as published by
 # the Free	Software Foundation; either	version	2 of the License, or
 # (at your	option)	any	later version.
 # 
 # This	program	is distributed in the hope that	it will	be useful,
 # but WITHOUT ANY WARRANTY; without even the implied warranty of
 # MERCHANTABILITY or FITNESS FOR A	PARTICULAR PURPOSE.	 See the
 # GNU General Public License for more details.
 # 
 # You should have received	a copy of the GNU General Public License
 # along with this program;	if not,	write to the Free Software
 # Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,	USA. 
 ##


## 
 # This	package	obtains	and	formats	the	'vers' resource	code of	the	specified
 # file.  If the file has no 'vers'	resource, the error	-1728 from the Finder
 # will	be thrown for you to catch as you see fit.	If the file's 'vers'
 # resource	is wrong, e.g. Alpha 6.52 seems	to think that it's version
 # 6.5.0b1,	there's	really nothing I can do	about that 8^).	 
 # 
 # NOTE: This package requires aeparse.tcl.
 # 
 # 
 # To check	if Alpha is	modern enough for your mode, you could call
 # 
 # proc	test {}	{
 #	   global HOME ALPHA
 #	   if {[file::version "$HOME:$ALPHA"] <	[file::stringToVersion "6.52"]}	{
 #		   error "A	newer version of Alpha is required to run this package"
 #	   } 
 # }
 # 
 # ...of course, since Alpha has the wrong 'vers' resource,	this test would	
 # fail, but you get the idea.
 ##

## 
 # -------------------------------------------------------------------------
 # 
 # "aecoerce::hexd:vers" --
 # 
 #  This should work with the reply under MacOS 7.x, where the result
 #  seems to be the entire version resource in hexadecimal form, 
 #  consisting of the encoded version number 
 #  followed by the version string, first as the number and then 
 #  as the full text.
 #                                 
 #            length    length
 #   __code__????||string||________________________text____________________
 #  07008000000003372E3018416C7068612056657273696F6E20372E3020A92031393937
 #    7008000       7 . 0   A l p h a   V e r s i o n   7 . 0      1 9 9 7
 # -------------------------------------------------------------------------
 ##
proc aecoerce::hexd:vers {value} {
	# Strip any leading 0's to enable ordinal comparisons
	if {[scan [string range $value 0 7] "%d" versionCode]} {
		# Skip over the code and 4 bytes of unknown
		# Grab the hex count; remember 2 digits per character
		set count [expr 2 * "0x[string range $value 12 13]"]
		# Clip off the stuff already read
		set value [string range $value 14 end]
		# Convert $count digits to text for the versionString
		set versionString [string range $value 0 [expr $count - 1]]
		set versionString [aecoerce::hexd:TEXT $versionString]
		# Ignore the length of the text
		# Just convert the remaining hex digits to text for the versionText
		set versionText [string range $value [expr $count + 2] end]
		set versionText [aecoerce::hexd:TEXT $versionText]
	} else {
		error "Can't coerce $value from 'hexd' to 'vers'" "" -1700
	}
	return [list $versionCode $versionString $versionText]
}

## 
 # -------------------------------------------------------------------------
 # 
 # "aecoerce::TEXT:vers" --
 # 
 #  This should work with the reply under MacOS 8.x, 
 #  where Apple has done us the "favor" of only returning the 
 #  version string, making ordinal comparisons a royal 
 #  pain-in-the-butt! 
 #
 #  We thus parse the string and hope we find something meaningful.
 #  The first instance of <number>.<number> is assumed to be
 #  the version (a string like "version 5" will _not_ be 
 #  recognized because simple integers are just as likely to be the 
 #  year). 
 #
 #  Pathological version parts like "fc" (final candidate) 
 #  and "p" (patch) 8^) are not recognized, as they have no analog 
 #  in the version resource.
 # -------------------------------------------------------------------------
 ##
proc aecoerce::TEXT:vers {value} {
	if {[regexp {[0-9]+\.[0-9]+(\.[0-9]+([dabf][0-9]*)?)?} \
	  		$value versionString]} {
		set versionCode [file::stringToVersion $versionString]
		set versionText $value
	} else {
		error "Can't coerce $value from 'TEXT' to 'vers'" "" -1700
	}
	return [list $versionCode $versionString $versionText]
}

## 
 # Obtains the 'vers' resource,	if any,	from $file.	
 # Supply either a -path or -creator, but not both.
 # If -creator option is supplied
 # the file creator is expected (of an application).
 # If -all option is supplied, returns the full list of
 # {versionCode versionString versionText}, otherwise
 # just versionCode is returned.
 ##
proc file::version {args} {
	set opts(-all) 0
	
	getOpts {creator path}
	
	if {[info exists opts(-creator)]} {
		if {[info exists opts(-path)]} {
			error "file::version error: -creator and -path options are incompatible."
		} else {
			set from "obj {want:type(file), seld:$opts(-creator), \
			  				form:fcrt, from:'null'()}"
		}
	} elseif {[info exists opts(-path)]} {
		set from "obj {want:type(file), seld:$opts(-path),\
						form:name, from:'null'()}"
	} else {
		error "file::version error: Either -creator or -path must be supplied."
	}
	
    set result [ \
      aecoerce::apply [ \
        aeparse::keywordValue ---- [ \
          aeparse::event [ \
            AEBuild -r 'MACS' core getd ---- [ \
              propertyObject vers $from \
            ] \
          ] \
        ] 1 \
      ] vers \
	]
	
	if {$opts(-all)} {
		return $result
	} else {
		return [lindex $result 0]
	}
}

## 
 # Converts	the	version	code, as returned by file::version,	into a 
 # human-readable string, e.g., file::versionToString "6506002" 
 # will return "6.5.0b2"
 # 
 # $version will be left-padded with 0's to 8 digits, thus
 # 
 #   file::versionToString 65
 #   
 # yields the rather meaningless "0.0.065", rather than "6.5.0".
 ##
proc file::versionToString {version {nosubsubversion 0}} {
	# Add leading 0's (trimmed to allow ordinal comparisons)
	set version [format "%08d" $version]
	
	# version
	scan [string range $version 0 1] "%d" vers
	
	# subversion
	append vers ".[string index $version 2]"
	
	# some pathological programs ;^) mush the subversion 
	# and subsubversion together
	if {!$nosubsubversion} {append vers "."}
	
	# subsubversion, or second digit of subversion
	append vers "[string index $version 3]"
	
	# releaseCode
	switch [string range $version 4 5] {
	  "20" {append vers "d"}
	  "40" {append vers "a"}
	  "60" {append vers "b"}
	}
	
	# nonRelease
	scan [string range $version 6 7] "%d" nonRelease
	
	if {$nonRelease} {
		if {[string range $version 4 5] == "80"} {
			append vers "f"
		}
		append vers $nonRelease
	}
	
	return $vers
}

## 
 # Converts	a version string into a version code, suitable for ordinal 
 # comparisons, e.g., file::stringToVersion "6.5.0b2" will return "6506002"
 # 
 # Versions may be formatted as "6.52" or "6.5.2", but not "6.52.1" 
 # ("6.52.0" slides by on a technicality).
 # 
 # All fields, including the version itself, are optional. 
 # file::stringToVersion "" yields 8000, which is version "0.0.0"
 ##
proc file::stringToVersion {vers} {
	regexp {^([0-9]*)(\.([0-9]+)(\.([0-9]+))?)?} $vers \
		whole version blah subversion blah subsubversion
	
	if {$version == ""} {set version 0}
	if {$subversion == ""} {set subversion 0}
	if {$subsubversion == ""} {set subsubversion 0}

	set chars [string range $vers [string length $whole] end]
	
	# This is to put versions like '6.52' in the 'correct' form
	if {[string length $subversion] == 2
	&&  !$subsubversion} {
		set subsubversion [string index $subversion 1]
		set subversion [string index $subversion 0]
	} 
		
	set releaseCode ""
	set nonRelease 0
	regexp {([a-zA-Z]+)([0-9]*)} $chars blah releaseCode nonRelease
	if {$nonRelease == ""} {
		set nonRelease 0
	} 
	
	if {([string length $version] > 2
     || [string length $subversion] > 2
     || [string length $subsubversion] > 1
     || [string length $releaseCode] > 1
     || [string length $nonRelease] > 2) 
 	|| ([string length $subversion] == 2
	 && [string length $subsubversion])} {
		error "\"$vers\" is not properly formatted"
	} 
	
	switch $releaseCode {
	  "d" {set releaseCode 20}
	  "a" {set releaseCode 40}
	  "b" {set releaseCode 60}
	  "f" -
	  ""  {set releaseCode 80}
	}
	
	scan [format "%01d%01d%01d%02d%02d" \
			$version $subversion $subsubversion $releaseCode $nonRelease] \
			"%d" version
	return $version
}
