/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines for loading TFM files.
 */

#include "types.h"
#include "font.h"
#include "tfm.h"
#include "tracedef.h"

#include "fileio.p"
#include "fio.p"
#include "font.p"
#include "strng.p"

/*
 *	The following routine loads data from a TFM into the font and
 *	character definitions:
 */

int Load_TFM (Font_Dir, Resolution, Font_Ptr)
char *Font_Dir;
struct Ratio *Resolution;
struct Font_Definition *Font_Ptr;
{
	auto   struct TFM_Memory_File *TFM_Ptr;
	auto   struct Char_Definition *Char_Ptr;
	auto   pointer TFM_File;
	auto   unsigned int Index, Wndex;
	static char Default_File[80];
	extern struct TFM_Memory_File *Read_TFM_File();
	extern char *Mem_Alloc();
	msgcode DVIOUT_NOTFMFILE, DVIOUT_BADTFMFILE;
/*
 *	Set up default file name; try to open the file:
 */
	stringcpy_m (Default_File, Font_Dir, sizeof (Default_File));
	stringcat_m (Default_File, ".TFM", sizeof (Default_File));
	if (trace_pixel)
		printf ("TFM file for font %s\n", Font_Ptr->Name_Ptr);
	if ((TFM_File = Open_File_M (Font_Ptr->Name_Ptr, Default_File, "r", 0)) == 0) {
		Message (DVIOUT_NOTFMFILE, Font_Ptr->Name_Ptr, 1);
		return (0);
	}
/*
 *	Read in and organize the TFM data:
 */
	TFM_Ptr = Read_TFM_File (TFM_File);
	Close_File_M (TFM_File);
	if (TFM_Ptr == 0) {
		Message (DVIOUT_BADTFMFILE, Font_Ptr->Name_Ptr, 1);
		return (0);
	}
/*
 *	Re-organize the information into the same data structures
 *	used for PXL files:
 */
	Font_Ptr->Checksum = TFM_Ptr->Header->TFM_Checksum;
	Font_Ptr->Font_Design_Size = TFM_Ptr->Header->Design_Size;
	Font_Ptr->Font_Coding[0] = '\0';
	Font_Ptr->Font_Family[0] = '\0';
	Font_Ptr->Font_Face = 0;
	if (TFM_Ptr->Header_Size >= 12)
		bcplcpy_m (Font_Ptr->Font_Coding, TFM_Ptr->Header->Font_Coding_Scheme,
			   sizeof (Font_Ptr->Font_Coding));
	if (TFM_Ptr->Header_Size >= 17)
		bcplcpy_m (Font_Ptr->Font_Family, TFM_Ptr->Header->Font_Identifier,
			   sizeof (Font_Ptr->Font_Family));
	if (TFM_Ptr->Header_Size >= 18)
		Font_Ptr->Font_Face = TFM_Ptr->Header->Face;
	Font_Ptr->Pixel_Id = ID_TFM << ID_V_TYPE;
/*
 *	For each character in the font, allocate an entry and fill in the
 *	width information:
 */
	for (Index = TFM_Ptr->Smallest_Char_Code; Index <= TFM_Ptr->Largest_Char_Code; Index++)
	if (Font_Ptr->Font_Directory[Index] == 0)
		;
	else if ((Wndex = TFM_Ptr->Char_Info[Index-TFM_Ptr->Smallest_Char_Code].Width_Index) == 0) {
		Font_Ptr->Font_Directory[Index] = 0;
		Undef_Char_Count++;
	} else {
		Char_Ptr = (struct Char_Definition *) Mem_Alloc (sizeof (struct Char_Definition));
		Char_Ptr->Character_Code = Index;
		Char_Ptr->DVI_Width = TFM_Ptr->Width[Wndex];
		Char_Ptr->Pixel_Width = 0;
		Char_Ptr->Pixel_Height = 0;
		Char_Ptr->X_Origin = 0;
		Char_Ptr->Y_Origin = 0;
		Char_Ptr->Pixel_Array = 0;
		Char_Ptr->Driver_Id = (unsigned long) Font_Ptr->Font_Directory[Index];
		Font_Ptr->Font_Directory[Index] = Char_Ptr;
	}
/*
 *	Check for any additional undefined characters:
 */
	Undef_Char_Count += Check_Undefined_Char_M (Font_Ptr, 0, TFM_Ptr->Smallest_Char_Code) +
			    Check_Undefined_Char_M (Font_Ptr, TFM_Ptr->Largest_Char_Code + 1,
						    Font_Ptr->Char_Dir_Count);
	Mem_Free (TFM_Ptr);
	return (1);
}

/*
 *	The following routine loads data from a TFM file into memory.
 */

struct TFM_Memory_File *Read_TFM_File (TFM_File)
pointer TFM_File;
{
	auto   struct TFM_Memory_File *TFM_Ptr;
	auto   unsigned long *Ptr;
	auto   unsigned int Index, Sum;
	static char *Face_Names[18] = {
		"MRR", "MIR", "BRR", "BIR", "LRR", "LIR",
		"MRC", "MIC", "BRC", "BIC", "LRC", "LIC",
		"MRE", "MIE", "BRE", "BIE", "LRE", "LIE"
	};
	static unsigned short Length_Array[12];
#define LF 0
#define LH 1
#define BC 2
#define EC 3
#define NW 4
#define NH 5
#define ND 6
#define NI 7
#define NL 8
#define NK 9
#define NE 10
#define NP 11
	static char Temp_Str[40];
	extern unsigned long *Read_N_Ints();
	extern char *Mem_Alloc();
/*
 *	Read in the first 12 16-bit words which contain the length
 *	of the various part of the file:
 */
	Sum = 0;
	for (Index = 0; Index < 12; Index++) {
		Length_Array[Index] = Read_Unsigned_Int_M (2, TFM_File);
		if (Index == BC)
			Sum -= Length_Array[Index] - 1;
		else if (Index != LF)
			Sum += Length_Array[Index];
	}
	if (trace_pixel) {
		printf ("  Number of words in TFM file: %u\n", Length_Array[LF]);
		printf ("  Number of words in header array: %u\n", Length_Array[LH]);
		printf ("  Starting character code: %u\n", Length_Array[BC]);
		printf ("  Ending character code: %u\n", Length_Array[EC]);
		printf ("  Number of words in width table: %u\n", Length_Array[NW]);
		printf ("  Number of words in height table: %u\n", Length_Array[NH]);
		printf ("  Number of words in depth table: %u\n", Length_Array[ND]);
		printf ("  Number of words in italic table: %u\n", Length_Array[NI]);
		printf ("  Number of words in lig/kern table: %u\n", Length_Array[NL]);
		printf ("  Number of words in kern table: %u\n", Length_Array[NK]);
		printf ("  Number of words in extensible recipe table: %u\n", Length_Array[NE]);
		printf ("  Number of words in param array: %u\n", Length_Array[NP]);
	}
/*
 *	Check lengths for consistency:
 */
	if (Length_Array[LH] < 2 ||
	    Length_Array[LF] != (Sum + 6) ||
	    Length_Array[BC] > Length_Array[EC] + 1)
		return (0);
/*
 *	Allocate memory for the file data:
 */
	TFM_Ptr = (struct TFM_Memory_File *)
		Mem_Alloc (sizeof (struct TFM_Memory_File) + Sum * 4);
	TFM_Ptr->Smallest_Char_Code = Length_Array[BC];
	TFM_Ptr->Largest_Char_Code = Length_Array[EC];
	TFM_Ptr->Header_Size = Length_Array[LH];
	Ptr = &TFM_Ptr->TFM_File_Data[0];
/*
 *	Read in the header:
 */
	TFM_Ptr->Header = (struct Header_Array *) Ptr;
	Ptr = Read_N_Ints (Ptr, 2, TFM_File);
	Index = Length_Array[LH] - 2;
	Read_Block_M (Ptr, Index * sizeof (*Ptr), TFM_File);
	Ptr = &Ptr[Index];
	if (trace_pixel) {
		printf ("  TFM Header:\n");
		printf ("    Checksum: %u\n", TFM_Ptr->Header->TFM_Checksum);
		printf ("    Design Size: %u\n", TFM_Ptr->Header->Design_Size);
		if (Length_Array[LH] >= 12)
			printf ("    Coding Scheme: %s\n",
				bcplcpy_m (Temp_Str, TFM_Ptr->Header->Font_Coding_Scheme,
					   sizeof (Temp_Str)));
		if (Length_Array[LH] >= 17)
			printf ("    Identifier: %s\n",
				bcplcpy_m (Temp_Str, TFM_Ptr->Header->Font_Identifier,
					   sizeof (Temp_Str)));
		if (Length_Array[LH] >= 18) {
			if (TFM_Ptr->Header->Face >= 18)
				sprintf (Temp_Str, "%03o", TFM_Ptr->Header->Face);
			else
				stringcpy_m (Temp_Str, Face_Names[TFM_Ptr->Header->Face], sizeof (Temp_Str));
			printf ("    Face: %s\n", Temp_Str);
			printf ("    Seven Bit Safe: %s\n",
				(TFM_Ptr->Header->Seven_Bit_Safe_Flag == 0) ? "false" : "true");
		}
	}
/*
 *	Read in the character data:
 */
	TFM_Ptr->Char_Info = (struct Char_Info_Word *) Ptr;
	Sum = Length_Array[EC] - Length_Array[BC] + 1;
	Read_Block_M (Ptr, Sum*4, TFM_File);
	Ptr = &Ptr[Sum];
/*
 *	Read in the Width, Height, Depth and Italic tables:
 */
	TFM_Ptr->Width = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[NW], TFM_File);
	TFM_Ptr->Height = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[NH], TFM_File);
	TFM_Ptr->Depth = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[ND], TFM_File);
	TFM_Ptr->Italic = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[NI], TFM_File);
/*
 *	Read in the ligature/kern data:
 */
	TFM_Ptr->Lig_Kern = (struct Lig_Kern_Command *) Ptr;
	Read_Block_M (Ptr, Length_Array[NL]*4, TFM_File);
	Ptr = &Ptr[Length_Array[NL]];
/*
 *	Read in the kern table:
 */
	TFM_Ptr->Kern = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[NK], TFM_File);
/*
 *	Read in the extensible recipe data:
 */
	TFM_Ptr->Exten = (struct Extensible_Recipe *) Ptr;
	Read_Block_M (Ptr, Length_Array[NE]*4, TFM_File);
	Ptr = &Ptr[Length_Array[NE]];
/*
 *	Finally, read in the param array:
 */
	TFM_Ptr->Param_Array_Size = Length_Array[NP];
	TFM_Ptr->Param_Array = (fix_word *) Ptr;
	Ptr = Read_N_Ints (Ptr, Length_Array[NP], TFM_File);
/*
 *	Do some final error checking:
 */
	if (TFM_Ptr->Width[0] != 0 || TFM_Ptr->Height[0] != 0 ||
	    TFM_Ptr->Depth[0] != 0 || TFM_Ptr->Italic[0] != 0) {
		Mem_Free (TFM_Ptr);
		return (0);
	}
	return (TFM_Ptr);
}

unsigned long *Read_N_Ints (Out_Ptr, N_Ints, File)
unsigned long *Out_Ptr;
char *File;
unsigned int N_Ints;
{
	auto   unsigned long *Ptr;
	auto   unsigned int Count;

	Ptr = Out_Ptr;
	for (Count = N_Ints; Count > 0; Count--)
		*Ptr++ = Read_Unsigned_Int_M (4, File);
	return (Ptr);
}
