/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This progam interactively queries the Apple LaserWriter
 *	for font metric information and creates a property list
 *	file suitable as input to PLtoTF to generate a TFM file
 *	for one or more native LaserWriter fonts.
 */

#include descrip
#include "types.h"
#include "iostatus.h"

#include "strng.p"

#define abs(x) ((x < 0) ? -x : x)
#define max(x,y) ((x > y) ? x : y)
#define min(x,y) ((x < y) ? x : y)

struct Kern_Entry {
	struct Kern_Entry *Link;
	double Kern;
	unsigned char Kern_Char;
};

struct Lig_Entry {
	struct Lig_Entry *Link;
	unsigned char Lig_Char;
	unsigned char New_Char;
};

struct Char_Definition {
	double Char_Width;
	double Char_Height;
	double Char_Depth;
	double Char_Italic;
	struct Kern_Entry *Kern_Ptr;
	struct Lig_Entry *Lig_Ptr;
} Char_Defs[256];
unsigned char Char_Vector[256];
unsigned char Symbol_Char_Vector[256];
char *Encoding_Vector[256];

struct Bounding_Box {
	double LLx, LLy, URx, URy;
};
struct Subs {
	char *Char_Name;
	int Encoding_Pos;
};

unsigned long Options;
#define MONITOR  0x00000001
#define GREEK    0x00000002
int Encoding_Scheme = 0;
#define TEX_ENCODING    0
#define TT_ENCODING     1
#define PS_TEXT         2
#define PS_SYMBOL       3
#define ASCII_ENCODING  4
#define EBCDIC_ENCODING 5
int Lig_Kern_Count = 0;

double Param_Array[7];
double Design_Size = 1.0;
double Scale_Factor;
double Stroke_Width;
double Slant_Angle;
double Extra_Italic_Correction;
long Checksum;
int Fixed_Pitch;
int Paint_Type;
int Font_Type;

unsigned short Channel;
char Dev_Name[16];
char Font_Name[40], Family_Name[40];
char Weight[12];

main ()
{
	auto   int Index, Jndex, Kndex;
	static double Char_Dimen[4], Current_Flat;
	static char Temp_Str[128];
	static struct Subs Greeks[] = {
		{ "/Gamma", 0000 }, { "/Delta", 0001 }, { "/Theta", 0002 }, { "/Lambda", 0003 },
		{ "/Xi", 0004 }, { "/Pi", 0005 }, { "/Sigma", 0006 }, { "/Upsilon1", 0007 },
		{ "/Phi", 0010 }, { "/Psi", 0011 },  { "/Omega", 0012 }
	}, Ebcdics[] = {
		{ "/logicalnot", 0x5F }, { "/logicalor", 0x4F }, { "/arrowvertex", 0xFA }
	}, Typewriters[] = {
		{ "/arrowup", 0013 }, { "/arrowdown", 0014 }
	};
	extern unsigned short Connect();
	extern double Query_Float(), tan(), atof();
	extern unsigned int Read_Line();
	extern int Query_Boolean(), Query_Int(), strcmp();
	extern char *Query_String(), *Read_String();
	globalvalue SS$_NORMAL;
/*
 *	Initialize:
 */
	for (Index = 0; Index < 256; Index++) {
		Char_Defs[Index].Char_Width = 0.0;
		Char_Defs[Index].Char_Height = 0.0;
		Char_Defs[Index].Char_Depth = 0.0;
		Char_Defs[Index].Char_Italic = 0.0;
		Char_Defs[Index].Kern_Ptr = 0;
		Char_Defs[Index].Lig_Ptr = 0;
		Encoding_Vector[Index] = 0;
	}
	Encoding_Vector[0040] = "/space";
	Encoding_Vector[0170] = "/x";
/*
 *	Obtain all inputs from the command line:
 */
	Process_Command_Options ();
/*
 *	Connect device and determine if the requested font exists:
 */
	if ((Channel = Connect (Dev_Name)) == 0) {
		sprintf (Temp_Str, "%%QUERYLW-F-CONNFAIL, unable to connect device %s", Dev_Name);
		To_Log (Temp_Str);
		Exit (4);
	}
	Read_Line (Temp_Str, sizeof (Temp_Str));
	sprintf (Temp_Str, "FontDirectory /%s known == flush", Font_Name);
	if (Query_Boolean (Temp_Str) == 0) {
		sprintf (Temp_Str, "%%QUERYLW-F-UNKFONT, font %s is invalid", Font_Name);
		To_Log (Temp_Str);
		Write_Line ("\004", 1);
		Read_Line (Temp_Str, 2);
		Disconnect (Channel);
		Exit (4);
	}
/*
 *	Obtain the Font Family Name, font version (used as the checksum),
 *	and whether the font is a fixed pitch font, the font type,
 *	paint type and weight:
 *
 *	The special logic for the font version is because some bimbo
 *	at Apple or Adobe Systems misspelled "version" as "verion" in
 *	the Courier-Bold font dictionary!!! (also couldn't spell
 *	"UnderlineThickness")
 */
	sprintf (Temp_Str, "/%s findfont", Font_Name);
	Output_Driver (Temp_Str);
	Paint_Type = Query_Int ("dup /PaintType get == flush");
	stringcpy_m (Family_Name, Query_String ("dup /FontInfo get /FamilyName get = flush"),
		     sizeof (Family_Name));
	stringcpy_m (Weight, Query_String ("dup /FontInfo get /Weight get = flush"),
		     sizeof (Weight));
	fold (Weight);
	if (Query_Boolean ("dup /FontInfo get /version known == flush") != 0)
		Output_Driver ("dup /FontInfo get /version get == flush");
	else if (Query_Boolean ("dup /FontInfo get /verion known == flush") != 0)
		Output_Driver ("dup /FontInfo get /verion get == flush");
	else
		Output_Driver ("(001.000) == flush");
	Checksum = (long) (1000.0 * atof (Read_String () + 1));
	Fixed_Pitch = Query_Boolean ("dup /FontInfo get /isFixedPitch get == flush");
	Font_Type = Query_Int ("dup /FontType get == flush");
	Slant_Angle = Query_Float ("dup /FontInfo get /ItalicAngle get neg == flush");
/*
 *	Set up some macros:
 */
	Scale_Factor = (Paint_Type == 0) ? 50.0 : 5.0;
	Extra_Italic_Correction = 0.5 * Scale_Factor / 16.0;
	Output_Driver ("/tmpary 4 array def");
	Output_Driver ("/qb { newpath 0 0 moveto false charpath flattenpath pathbbox tmpary astore == flush } def");
	Output_Driver ("/qw { stringwidth pop == flush } def");
	sprintf (Temp_Str, "/sf { %f 72.27 %f mul mul 72 div scalefont setfont } def",
		 Design_Size, Scale_Factor);
	Output_Driver (Temp_Str);
/*
 *	Get current settings of things; set up initial conditions:
 */
	Current_Flat = Query_Float ("currentflat == flush");
	Output_Driver ("initgraphics 1.0 setflat sf");
/*
 *	If the PaintType is not a filled font, obtain the stroke
 *	width for use in adjusting the character bounding boxes:
 */
	if (Paint_Type != 0)
		Stroke_Width = 0.5 * Query_Float ("currentfont /StrokeWidth get 0 \
currentfont /FontMatrix get transform pop == flush");
/*
 *	Determine the parameter values for the font: slant, space,
 *	space_stretch, space_shrink, x_height, quad and extra_space:
 */
	Param_Array[0] = tan (Slant_Angle * 0.017453293);
	Get_Char_Dimen (' ', Char_Dimen);
	Param_Array[1] = Char_Dimen[0];
	if (Fixed_Pitch == 0) {
		Param_Array[2] = Char_Dimen[0] / 2.0;
		Param_Array[3] = Char_Dimen[0] / 3.0;
		Param_Array[6] = Param_Array[3];
	} else {
		Param_Array[2] = 0.0;
		Param_Array[3] = 0.0;
		Param_Array[6] = Param_Array[1];
	}
	Get_Char_Dimen ('x', Char_Dimen);
	Param_Array[4] = Char_Dimen[1];
	Param_Array[5] = Design_Size;
/*
 *	Now re-encode the font to match the TeX cm font family;
 *	get the resulting character vector:
 */
	Build_Font ();
	Output_Driver ("nf sf");
	Get_Char_Vector (Char_Vector);
/*
 *	Obtain the dimensions of the individual characters:
 */
	for (Index = 0; Index < 256; Index++)
	if (Char_Vector[Index] != 0)
		Get_Char_Dimen (Index, &Char_Defs[Index].Char_Width);
/*
 *	Determine character kerning, if any:
 */
/*	Build_Kern_Table ();	*/
/*
 *	If specified, determine the font metric information for the
 *	eleven greek characters in "TEX TEXT" encoding. Additionally,
 *	if the encoding is EBCDIC, steal the logical or, logical not,
 *	and long vertical bar from appropriate Symbol characters. If
 *	the font uses TEX TYPEWRITER TEXT encoding, steal the up arrow
 *	and down arrow characters from the Symbol font.
 */
	if ((Options & GREEK) != 0 || Encoding_Scheme == EBCDIC_ENCODING || Encoding_Scheme == TT_ENCODING) {
		Output_Driver ("/Symbol findfont sf");
		Get_Char_Vector (Symbol_Char_Vector);
		if ((Options & GREEK) != 0) {
			for (Index = 0; Index < arraysize (Greeks); Index++) {
				for (Jndex = 0; Jndex < 256; Jndex++)
				if (strcmp (Encoding_Vector[Jndex], Greeks[Index].Char_Name) == 0) {
					Kndex = Greeks[Index].Encoding_Pos;
					Get_Char_Dimen (Jndex, &Char_Defs[Kndex].Char_Width);
					Char_Vector[Kndex] = 1;
					break;
				}
			}
		}
		if (Encoding_Scheme == EBCDIC_ENCODING) {
			for (Index = 0; Index < arraysize (Ebcdics); Index++) {
				for (Jndex = 0; Jndex < 256; Jndex++)
				if (strcmp (Encoding_Vector[Jndex], Ebcdics[Index].Char_Name) == 0) {
					Kndex = Ebcdics[Index].Encoding_Pos;
					Get_Char_Dimen (Jndex, &Char_Defs[Kndex].Char_Width);
					Char_Vector[Kndex] = 1;
					break;
				}
			}
		} else if (Encoding_Scheme == TT_ENCODING) {
			for (Index = 0; Index < arraysize (Typewriters); Index++) {
				for (Jndex = 0; Jndex < 256; Jndex++)
				if (strcmp (Encoding_Vector[Jndex], Typewriters[Index].Char_Name) == 0) {
					Kndex = Typewriters[Index].Encoding_Pos;
					Get_Char_Dimen (Jndex, &Char_Defs[Kndex].Char_Width);
					Char_Vector[Kndex] = 1;
					break;
				}
			}
		}
	}
/*
 *	Restore any parameters that have been altered:
 */
	sprintf (Temp_Str, "%f setflat", Current_Flat);
	Output_Driver (Temp_Str);
/*
 *	Send end-of-file to device:
 */
	Write_Line ("\004", 1);
	Read_Line (Temp_Str, 2);
	Disconnect (Channel);
/*
 *	Generate the Property List output:
 */
	Generate_PL ();
	Exit (SS$_NORMAL);
}

Process_Command_Options ()
{
	auto   char *Ptr1, *Ptr2;
	static char Buffer[80];
	extern int Check_Run_Option(), Get_Run_Option_Value(), strcmp();

	Get_Run_Option_Value ("FONT", Font_Name, sizeof (Font_Name));
	if (Check_Run_Option ("DEVICE", "", 0) == 0)
		stringcpy_m (Dev_Name, "LASERWRITER", sizeof (Dev_Name));
	else
		Get_Run_Option_Value ("DEVICE", Dev_Name, sizeof (Dev_Name));
	Options = 0;
	if (Check_Run_Option ("MONITOR", "", 0) > 0)
		Options |= MONITOR;
	if (Check_Run_Option ("GREEK", "", 0) > 0)
		Options |= GREEK;
	if (Check_Run_Option ("LIGATURE_CODE", "", 0) > 0)
		while (Get_Run_Option_Value ("LIGATURE_CODE", Buffer, sizeof (Buffer)) != 0) {
			Add_Ligature (Buffer);
			Lig_Kern_Count++;
		}
	if (Check_Run_Option ("KERN", "", 0) > 0)
		while (Get_Run_Option_Value ("KERN", Buffer, sizeof (Buffer)) != 0) {
			Add_Kern (Buffer);
			Lig_Kern_Count++;
		}
	if (Check_Run_Option ("ENCODING", "", 0) > 0) {
		if (Check_Run_Option ("ENCODING", "TEX", 0) > 0)
			Encoding_Scheme = TEX_ENCODING;
		else if (Check_Run_Option ("ENCODING", "TYPEWRITER", 0) > 0)
			Encoding_Scheme = TT_ENCODING;
		else if (Check_Run_Option ("ENCODING", "NATIVE", 0) > 0)
			Encoding_Scheme = (strcmp (Font_Name, "Symbol") == 0) ? PS_SYMBOL : PS_TEXT;
		else if (Check_Run_Option ("ENCODING", "ASCII", 0) > 0)
			Encoding_Scheme = ASCII_ENCODING;
		else if (Check_Run_Option ("ENCODING", "EBCDIC", 0) > 0)
			Encoding_Scheme = EBCDIC_ENCODING;
	}
}

/*
 *	Routine Add_ligature processes one ligature specification.
 *	A ligature specification has the form "a.b.c", where "a"
 *	is the first character of the ligature sequence, "b" is the
 *	second character of the ligature sequence, and "c" is the
 *	new composite character. Any or all of the three characters
 *	can be an ascii character or an octal code. An ascii character
 *	must be preceded by a "C" and an octal code must be preceded
 *	by an "O".
 *
 *	For example, the ligature code "Cf.Ci.O14" directs that an
 *	"f" followed by an "i" is to become a new character, coded
 *	at position '14 in the font.
 *
 *	The period cannot be represented using the "C" form. Other
 *	command line delimiters, such as left and right parentheses
 *	and quote characters also cannot be represented using the "C"
 *	form, unless enclosed in quotes. Additionally, if any of the
 *	three characters are lower case, the ligature code string
 *	must be enclosed in quotes in the DCL command. (It would
 *	be a good idea to always enclose these specifications in
 *	quotes).
 */

Add_Ligature (Lig_Spec)
char *Lig_Spec;
{
	auto   struct Lig_Entry *L_Ptr;
	auto   unsigned char c;
	static char Str1[20], Str2[20], Str3[20];
	extern unsigned char Parse_Code();
	extern char *Mem_Alloc();
	extern int sscanf();

	Str1[0] = '\0';
	Str2[0] = '\0';
	Str3[0] = '\0';
	sscanf (Lig_Spec, "%[^.].%[^.].%s", Str1, Str2, Str3);
	c = Parse_Code (Str1);
	for (L_Ptr = (struct Lig_Entry *) &Char_Defs[c].Lig_Ptr; L_Ptr->Link != 0; L_Ptr = L_Ptr->Link)
		;
	L_Ptr = L_Ptr->Link = Mem_Alloc (sizeof (struct Lig_Entry));
	L_Ptr->Lig_Char = Parse_Code (Str2);
	L_Ptr->New_Char = Parse_Code (Str3);
}

/*
 *	Add_Kern is similar to Add_Ligature, except that the third
 *	entry is a real number. The kern value is in design_size
 *	units, not points.
 */

Add_Kern (Kern_Spec)
char *Kern_Spec;
{
	auto   struct Kern_Entry *K_Ptr;
	auto   unsigned char c;
	static double Kern_Value;
	static char Str1[20], Str2[20];
	extern unsigned char Parse_Code();
	extern char *Mem_Alloc();
	extern int sscanf();

	Str1[0] = '\0';
	Str2[0] = '\0';
	Kern_Value = 0.0;
	sscanf (Kern_Spec, "%[^.].%[^.].%F", Str1, Str2, &Kern_Value);
	c = Parse_Code (Str1);
	for (K_Ptr = (struct Kern_Entry *) &Char_Defs[c].Kern_Ptr; K_Ptr->Link != 0; K_Ptr = K_Ptr->Link)
		;
	K_Ptr = K_Ptr->Link = Mem_Alloc (sizeof (struct Kern_Entry));
	K_Ptr->Kern_Char = Parse_Code (Str2);
	K_Ptr->Kern = Kern_Value;
}

unsigned char Parse_Code (Code)
char *Code;
{
	auto   char *Ptr;
	auto   char c;
	auto   unsigned char t;

	Ptr = Code;
	if ((c = *Ptr) == '\0') {
		To_Log ("%QUERYLW-E-NULLCODE, null character code");
		t = 0;
	} else if (c == 'C' || c == 'c') {
		if ((t = *++Ptr) == 0)
			To_Log ("%QUERYLW-E-MISSINGCODE, missing \"C\" character code");
	} else if (c == 'O' || c == 'o') {
		t = 0;
		while ((c = *++Ptr) >= '0' && c <= '7')
			t = (t << 3) + (c - '0');
	}
	return (t);
}

Get_Char_Vector (Ind_Vector)
unsigned char *Ind_Vector;
{
	auto   int Vector_Size, Index;
	extern int Query_Int(), strcmp();
	
	Vector_Size = Query_Int ("currentfont /Encoding get length == flush");
	Query_String_Array ("currentfont /Encoding get { == flush } forall", Encoding_Vector, Vector_Size);
	for (Index = 0; Index < Vector_Size; Index++)
		Ind_Vector[Index] = (strcmp (Encoding_Vector[Index], "/.notdef") == 0) ? 0 : 1;
}

Build_Kern_Table ()
{
	auto   struct Kern_Entry *K_Ptr, *N_Ptr;
	auto   double Width, Delta_Width;
	auto   int Index_1, Index_2;
	static char C1Str[6], C2Str[6], QStr[16];
	extern double Query_Float();
	extern char *Mem_Alloc();

	for (Index_1 = 0; Index_1 < 256; Index_1++) {
		K_Ptr = 0;
		if (Char_Vector[Index_1] != 0) {
			Build_Char_Code (Index_1, C1Str);
			for (Index_2 = 0; Index_2 < 256; Index_2++)
			if (Char_Vector[Index_2] != 0) {
				Width = Char_Defs[Index_1].Char_Width + Char_Defs[Index_2].Char_Width;
				Build_Char_Code (Index_2, C2Str);
				sprintf (QStr, "(%s%s) qw", C1Str, C2Str);
				Delta_Width = Query_Float (QStr) - Width;
				if (abs (Delta_Width) > 0.2) {
					N_Ptr = (struct Kern_Entry *) Mem_Alloc (sizeof (struct Kern_Entry));
					N_Ptr->Link = K_Ptr;
					K_Ptr = N_Ptr;
					K_Ptr->Kern = Delta_Width;
					K_Ptr->Kern_Char = Index_2;
				}
			}
		}
		Char_Defs[Index_1].Kern_Ptr = K_Ptr;
	}
}

Get_Char_Dimen (Char_Code, Dimen_Array)
unsigned char Char_Code;
double Dimen_Array[4];
{
	auto   double Width;
	static struct Bounding_Box Char_BBox;
	static char CStr[8], DStr[12];
	extern double Query_Float();
	extern int Query_Float_Vector(), strcmp();
/*
 *	Obtain the character bounding box for the specified
 *	character; adjust by the stroke width for non-filled
 *	fonts:
 */
	Build_Char_Code (Char_Code, CStr);
	sprintf (DStr, "(%s) qw", CStr);
	Width = Query_Float (DStr);
	sprintf (DStr, "(%s) qb", CStr);
	Query_Float_Vector (DStr, &Char_BBox, 4);
	if (Paint_Type != 0 && strcmp (Encoding_Vector[Char_Code], "/space") != 0) {
		Char_BBox.LLx -= Stroke_Width;
		Char_BBox.LLy -= Stroke_Width;
		Char_BBox.URx += Stroke_Width;
		Char_BBox.URy += Stroke_Width;
	}
	Char_BBox.URx += Extra_Italic_Correction;
	Dimen_Array[0] = Width / Scale_Factor;
	Dimen_Array[1] = (Char_BBox.URy <= 0.0) ? 0.0 : Char_BBox.URy / Scale_Factor;
	Dimen_Array[2] = (Char_BBox.LLy >= 0.0) ? 0.0 : -Char_BBox.LLy / Scale_Factor;
	Dimen_Array[3] = (Char_BBox.URx <= Width) ? 0.0 : (Char_BBox.URx - Width) / Scale_Factor;
}

Build_Char_Code (Char_Code, String)
unsigned char Char_Code;
char *String;
{
	if (Char_Code < '\040' || Char_Code > '\176' || Char_Code == '(' || Char_Code == ')' || Char_Code == '\\')
		sprintf (String, "\\%03o", Char_Code);
	else
		sprintf (String, "%c", Char_Code);
}

/*
 *	This next routine builds an almost-TeX-compatible font assembled
 *	from the Times-Roman and Symbol fonts in the LaserWriter. Missing
 *	are the characters ff, ffi, ffl (ligatures), dot-less j, and the
 *	slash used for crossing l's and L's. These are all pretty rare,
 *	so the impact is minimal, except possibly for the dot-less j that
 *	could be used in Math Mode.
 */

Build_Font ()
{
	auto   int Index;
	auto   char *Name_Ptr;
	static char New_Font_Name[40], Temp_Str[256];
	extern int Query_Int_Vector();
/*
 *	Set up a few definitions:
 */
	sprintf (New_Font_Name, "TeX-%s", Font_Name);
	sprintf (Temp_Str, "/ff /%s findfont def", Font_Name);
	Output_Driver (Temp_Str);
/*
 *	Create the font encoding vector:
 */
	Output_Driver ("/TeXEncoding 256 array def 0 1 255 { TeXEncoding exch /.notdef put } for");
	if (Encoding_Scheme == PS_TEXT || Encoding_Scheme == PS_SYMBOL)
		Generate_Native_Encoding ();
	else if (Encoding_Scheme == ASCII_ENCODING)
		Generate_Ascii_Encoding ();
	else if (Encoding_Scheme == EBCDIC_ENCODING)
		Generate_Ebcdic_Encoding ();
	else if (Encoding_Scheme == TT_ENCODING)
		Generate_Typewriter_Encoding ();
	else
		Generate_TeX_Encoding ();
/*
 *	Now, build a new font dictionary and put the Encoding vector,
 *	the CharStrings dictionary, the bounding box and the new name
 *	into it:
 */
	Output_Driver ("/nf ff length dict def");
	Output_Driver ("ff { 1 index /FID ne { nf 3 1 roll put } { pop pop } ifelse } forall");
	sprintf (Temp_Str, "nf /FontName /%s put", New_Font_Name);
	Output_Driver (Temp_Str);
	Output_Driver ("nf /Encoding TeXEncoding put");
/*
 *	Finally, define the new font:
 */
	Output_Driver ("nf /FontName get nf definefont");
}

Generate_TeX_Encoding ()
{
	auto   int Index;
	static char Temp_Str[64];
	static char *Names[128] = {
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, "fi", "fl", "lslash",
		"Lslash", "dotlessi", 0, "grave", "acute", "caron", "breve",
		"macron", "ring", "cedilla", "germandbls", "ae", "oe",
		"oslash", "AE", "OE", "Oslash", "space", "exclam",
		"quotedblright", "numbersign", "dollar", "percent", "ampersand",
		"quoteright", "parenleft", "parenright", "asterisk", "plus",
		"comma", "hyphen", "period", "slash", "zero", "one", "two",
		"three", "four", "five", "six", "seven", "eight", "nine",
		"colon", "semicolon", "exclamdown", "equal", "questiondown",
		"question", "at", "A", "B", "C", "D", "E", "F", "G", "H", "I",
		"J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U",
		"V", "W", "X", "Y", "Z", "bracketleft", "quotedblleft",
		"bracketright", "circumflex", "dotaccent", "quoteleft", "a",
		"b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m",
		"n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y",
		"z", "endash", "emdash", "hungarumlaut", "tilde", "dieresis"
	};
/*
 *	Define the font's encoding array. This contains the normal
 *	names (except in a different order):
 */
	Output_Driver ("TeXEncoding");
	for (Index = 0; Index < 128; Index++)
	if (Names[Index] != 0) {
		sprintf (Temp_Str, "dup %u /%s put", Index, Names[Index]);
		Output_Driver (Temp_Str);
	}
	Output_Driver ("pop");
}

Generate_Typewriter_Encoding ()
{
	auto   int Index;
	static char Temp_Str[64];
	static char *Names[128] = {
		0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, "quotesingle", "exclamdown",
		"questiondown", "dotlessi", 0, "grave", "acute", "caron", "breve",
		"macron", "ring", "cedilla", "germandbls", "ae", "oe",
		"oslash", "AE", "OE", "Oslash", "space", "exclam",
		"quotedbl", "numbersign", "dollar", "percent", "ampersand",
		"quoteright", "parenleft", "parenright", "asterisk", "plus",
		"comma", "hyphen", "period", "slash", "zero", "one", "two",
		"three", "four", "five", "six", "seven", "eight", "nine",
		"colon", "semicolon", "less", "equal", "greater", "question",
		"at", "A", "B", "C", "D", "E", "F", "G", "H", "I",
		"J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U",
		"V", "W", "X", "Y", "Z", "bracketleft", "backslash",
		"bracketright", "circumflex", "underscore", "quoteleft", "a",
		"b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m",
		"n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y",
		"z", "braceleft", "bar", "braceright", "tilde", "dieresis"
	};

	Output_Driver ("TeXEncoding");
	for (Index = 0; Index < 128; Index++)
	if (Names[Index] != 0) {
		sprintf (Temp_Str, "dup %u /%s put", Index, Names[Index]);
		Output_Driver (Temp_Str);
	}
	Output_Driver ("pop");
}

Generate_Native_Encoding ()
{
	Output_Driver ("ff /Encoding get TeXEncoding copy");
}

Generate_Ascii_Encoding ()
{
	auto   int Index;
	static char Temp_Str[64];
	static char *Ascii_Names[] = {
		"space", "exclam", "quotedbl", "numbersign", "dollar",
		"percent", "ampersand", "quoteright", "parenleft",
		"parenright", "asterisk", "plus", "comma", "hyphen",
		"period", "slash", "zero", "one", "two", "three", "four",
		"five", "six", "seven", "eight", "nine", "colon",
		"semicolon", "less", "equal", "greater", "question", "at",
		"A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K",
		"L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V",
		"W", "X", "Y", "Z", "bracketleft", "backslash",
		"bracketright", "asciicircum", "underscore", "quoteleft",
		"a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k",
		"l", "m", "n", "o", "p", "q", "r", "s", "t", "u", "v",
		"w", "x", "y", "z", "braceleft", "bar", "braceright",
		"asciitilde"
	};

	Output_Driver ("TeXEncoding");
	for (Index = 0; Index < 95; Index++) {
		sprintf (Temp_Str, "dup %u /%s put", Index+32, Ascii_Names[Index]);
		Output_Driver (Temp_Str);
	}
	Output_Driver ("pop");
}

/*
 *	Define the encoding vector for the EBCDIC character set. Not all
 *	characters within the set have exact representations in the
 *	LaserWriter fonts. The 'logical or' (0x4F), 'vertical line' (0x6A)
 *	and 'long vertical mark' (0xFA) are all represent by "bar". The
 *	'logical not' (0x5F) does not have any good substitute, and is replaced
 *	by a tilde. The 'SP', 'RSP' and 'NSP' characters are all replaced
 *	by "space". The 'SHY' character is replaced by "endash" (?). The
 *	peculiar 'hook' (0xCC), 'fork' (0xCE) and 'chair' (0xEC) characters
 *	are not represented.
 */

Generate_Ebcdic_Encoding ()
{
	auto   int Index;
	static char Temp_Str[64];
	static struct Subs Ebcdic_Subs[] = {
		{ "space", 0x40 }, { "space", 0x41 }, { "cent", 0x4A }, { "period", 0x4B }, { "less", 0x4C },
		{ "parenleft", 0x4D }, { "plus", 0x4E }, { "bar", 0x4F }, { "ampersand", 0x50 },
		{ "exclam", 0x5A }, { "dollar", 0x5B }, { "asterisk", 0x5C }, { "parenright", 0x5D },
		{ "semicolon", 0x5E }, { "asciitilde", 0x5F }, { "hyphen", 0x60 }, { "slash", 0x61 },
		{ "bar", 0x6A }, { "comma", 0x6B }, { "percent", 0x6C }, { "underscore", 0x6D },
		{ "greater", 0x6E }, { "question", 0x6F }, { "grave", 0x79 }, { "colon", 0x7A },
		{ "numbersign", 0x7B }, { "at", 0x7C }, { "quotesingle", 0x7D }, { "equal", 0x7E },
		{ "quotedbl", 0x7F },
		{ "a", 0x81 }, { "b", 0x82 }, { "c", 0x83 }, { "d", 0x84 }, { "e", 0x85 },
		{ "f", 0x86 }, { "g", 0x87 }, { "h", 0x88 }, { "i", 0x89 },
		{ "j", 0x91 }, { "k", 0x92 }, { "l", 0x93 }, { "m", 0x94 }, { "n", 0x95 },
		{ "o", 0x96 }, { "p", 0x97 }, { "q", 0x98 }, { "r", 0x99 },
		{ "asciitilde", 0xA1 },
		{ "s", 0xA2 }, { "t", 0xA3 }, { "u", 0xA4 }, { "v", 0xA5 },
		{ "w", 0xA6 }, { "x", 0xA7 }, { "y", 0xA8 }, { "z", 0xA9 },
		{ "braceleft", 0xC0 },
		{ "A", 0xC1 }, { "B", 0xC2 }, { "C", 0xC3 }, { "D", 0xC4 }, { "E", 0xC5 },
		{ "F", 0xC6 }, { "G", 0xC7 }, { "H", 0xC8 }, { "I", 0xC9 },
		{ "endash", 0xCA }, { "braceright", 0xD0 },
		{ "J", 0xD1 }, { "K", 0xD2 }, { "L", 0xD3 }, { "M", 0xD4 }, { "N", 0xD5 },
		{ "O", 0xD6 }, { "P", 0xD7 }, { "Q", 0xD8 }, { "R", 0xD9 },
		{ "backslash", 0xE0 }, { "space", 0xE1 },
		{ "S", 0xE2 }, { "T", 0xE3 }, { "U", 0xE4 }, { "V", 0xE5 },
		{ "W", 0xE6 }, { "X", 0xE7 }, { "Y", 0xE8 }, { "Z", 0xE9 },
		{ "zero", 0xF0 }, { "one", 0xF1 }, { "two", 0xF2 }, { "three", 0xF3 }, { "four", 0xF4 },
		{ "five", 0xF5 }, { "six", 0xF6 }, { "seven", 0xF7 }, { "eight", 0xF8 }, { "nine", 0xF9 },
		{ "bar", 0xFA }
	};

	Output_Driver ("TeXEncoding");
	for (Index = 0; Index < arraysize (Ebcdic_Subs); Index++) {
		sprintf (Temp_Str, "dup %u /%s put", Ebcdic_Subs[Index].Encoding_Pos,
			 Ebcdic_Subs[Index].Char_Name);
		Output_Driver (Temp_Str);
	}
	Output_Driver ("pop");
}

unsigned short Connect (Device_Name)
char *Device_Name;
{
	static struct dsc$descriptor Name_Desc;
	static unsigned short Chan;
	extern unsigned long Sys$Assign();
	extern int Check_System_Status();

	Make_VMS_Descriptor (Device_Name, &Name_Desc);
	if (Check_System_Status (Sys$Assign (&Name_Desc, &Chan, 0, 0)) == 0)
		return (0);
	return (Chan);
}

Disconnect (Chan)
unsigned short Chan;
{
	extern unsigned long Sys$DAssgn();
	extern int Check_System_Status();

	Check_System_Status (Sys$DAssgn (Chan));
}

Write_Line (Buffer, Count)
char *Buffer;
unsigned short Count;
{
	auto   unsigned long Status;
	static struct Write_IO_Status_Block IO_Status;
	extern unsigned long Sys$QIOW();
	extern int Check_System_Status();
	globalvalue IO$_WRITEVBLK, IO$M_NOFORMAT;

	Status = Sys$QIOW (0, Channel, IO$_WRITEVBLK|IO$M_NOFORMAT, &IO_Status,
			   0, 0, Buffer, Count, 0, 0, 0, 0);
	if ((Status & 0x01) != 0)
		Status = IO_Status.Status;
	Check_System_Status (Status);
}

unsigned int Read_Line (Buffer, Size)
char *Buffer;
unsigned short Size;
{
	auto   unsigned long Status;
	auto   unsigned short Count;
	static struct Read_IO_Status_Block IO_Status;
	static unsigned long Terminator_Mask[2] = { 0, 0x00000400 };
	extern unsigned long Sys$QIOW();
	extern int Check_System_Status();
	globalvalue IO$_READVBLK, IO$M_NOECHO, IO$M_TIMED, SS$_TIMEOUT;

	Status = Sys$QIOW (0, Channel, IO$_READVBLK|IO$M_NOECHO|IO$M_TIMED,
			   &IO_Status, 0, 0, Buffer, Size-1, 5, Terminator_Mask,
			   0, 0);
	if ((Status & 0x01) != 0)
		Status = IO_Status.Status;
	if (Status == SS$_TIMEOUT || Check_System_Status (Status) == 0) {
		Buffer[0] = '\0';
		return (0);
	}
	Count = IO_Status.Byte_Count;
	if (Count > 0 && Buffer[Count-1] == '\r')
		Count--;
	while (Count > 0 && Buffer[Count-1] == ' ')
		Count--;
	Buffer[Count] = '\0';
	return (Count);
}

char *Read_String ()
{
	static char Buffer[256];

	Read_Line (Buffer, sizeof (Buffer));
	if ((Options & MONITOR) != 0)
		To_Log (Buffer);
	return (Buffer);
}

int Read_Boolean ()
{
	auto   char *Ptr;
	extern char *Read_String();
	extern int strcmp();

	Ptr = Read_String ();
	if (strcmp (Ptr, "true") == 0)
		return (1);
	return (0);
}

int Read_Int ()
{
	extern int atoi();
	extern char *Read_String();

	return (atoi (Read_String ()));
}

double Read_Float ()
{
	extern double atof();
	extern char *Read_String();

	return (atof (Read_String ()));
}

int Read_Float_Vector (Vector, Max_Size)
double Vector[];
int Max_Size;
{
	auto   char *Ptr, *VPtr, Term_Char;
	auto   int Count;
	static char Value[20];
	extern char *Read_String();
	extern double atof();

	Ptr = Read_String ();
	if (*Ptr == '[') {
		Ptr++;
		Term_Char = ']';
	} else if (*Ptr == '{') {
		Ptr++;
		Term_Char = '}';
	} else
		Term_Char = '\0';
	for (Count = 0; Count < Max_Size; Count++) {
		while (*Ptr == ' ')
			Ptr++;
		if (*Ptr == Term_Char)
			break;
		VPtr = Value;
		while (*Ptr != ' ' && *Ptr != Term_Char)
			*VPtr++ = *Ptr++;
		*VPtr = '\0';
		Vector[Count] = atof (Value);
	}
	return (Count);
}

int Read_Int_Vector (Vector, Max_Size)
int Vector[];
int Max_Size;
{
	auto   char *Ptr, *VPtr, Term_Char;
	auto   int Count;
	static char Value[20];
	extern char *Read_String();
	extern int atoi();

	Ptr = Read_String ();
	if (*Ptr == '[') {
		Ptr++;
		Term_Char = ']';
	} else if (*Ptr == '{') {
		Ptr++;
		Term_Char = '}';
	} else
		Term_Char = '\0';
	for (Count = 0; Count < Max_Size; Count++) {
		while (*Ptr == ' ')
			Ptr++;
		if (*Ptr == Term_Char)
			break;
		VPtr = Value;
		while (*Ptr != ' ' && *Ptr != Term_Char)
			*VPtr++ = *Ptr++;
		*VPtr = '\0';
		Vector[Count] = atoi (Value);
	}
	return (Count);
}

Output_Driver (Driver_String)
char *Driver_String;
{
	auto   char c;
	auto   int Len;
	extern int strlen();

	if ((Options & MONITOR) != 0)
		To_Log (Driver_String);
	Len = strlen (Driver_String);
	Write_Line (Driver_String, Len);
	if ((c = Driver_String[Len-1]) != ' ' && c != '\n' && c != '\r')
		Write_Line (" ", 1);
}

int Query_Int (Driver_String)
char *Driver_String;
{
	extern int Read_Int();

	Output_Driver (Driver_String);
	return (Read_Int ());
}

double Query_Float (Driver_String)
char *Driver_String;
{
	extern double Read_Float();

	Output_Driver (Driver_String);
	return (Read_Float ());
}

int Query_Float_Vector (Driver_String, Vector, Max_Size)
char *Driver_String;
double Vector[];
int Max_Size;
{
	extern int Read_Float_Vector();

	Output_Driver (Driver_String);
	return (Read_Float_Vector (Vector, Max_Size));
}

int Query_Int_Vector (Driver_String, Vector, Max_Size)
char *Driver_String;
int Vector[];
int Max_Size;
{
	extern int Read_Int_Vector();

	Output_Driver (Driver_String);
	return (Read_Int_Vector (Vector, Max_Size));
}

int Query_Boolean (Driver_String)
char *Driver_String;
{
	extern int Read_Boolean();

	Output_Driver (Driver_String);
	return (Read_Boolean ());
}

char *Query_String (Driver_String)
char *Driver_String;
{
	char *Read_String();

	Output_Driver (Driver_String);
	return (Read_String ());
}

Query_Int_Array (Driver_String, Array, Count)
char *Driver_String;
int Array[];
unsigned int Count;
{
	auto   unsigned int Cnt;
	extern int Read_Int();

	Output_Driver (Driver_String);
	for (Cnt = 0; Cnt < Count; Cnt++)
		Array[Cnt] = Read_Int ();
}

Query_Float_Array (Driver_String, Array, Count)
char *Driver_String;
double Array[];
unsigned int Count;
{
	auto   unsigned int Cnt;
	extern double Read_Float();

	Output_Driver (Driver_String);
	for (Cnt = 0; Cnt < Count; Cnt++)
		Array[Cnt] = Read_Float ();
}

Query_String_Array (Driver_String, Array, Count)
char *Driver_String;
char *Array[];
unsigned int Count;
{
	auto   char *Ptr;
	auto   unsigned int Cnt;
	extern int strlen();
	extern char *Read_String(), *Mem_Alloc();

	Output_Driver (Driver_String);
	for (Cnt = 0; Cnt < Count; Cnt++) {
		Ptr = Read_String ();
		Array[Cnt] = Mem_Alloc (strlen (Ptr) + 1);
		strcpy (Array[Cnt], Ptr);
	}
}

#include stdio

To_Log (String)
char *String;
{
	auto   int Len;
	extern int strlen();

	if ((Len = strlen (String)) > 0)
		fprintf (stderr, (String[Len-1] == '\n') ? "%s" : "%s\n", String);
}

int Check_System_Status (Status)
unsigned long Status;
{
	if ((Status & 0x01) == 0) {
		printf("Status: %08X\n", Status);
		return (0);
	}
	return (1);
}

/*
 *	Routine Generate_PL converts the font information obtained
 *	from the LaserWriter to a property list file, which can then
 *	be converted using TFtoPL into a TeX Font Metric (TFM) file.
 */

Generate_PL ()
{
	auto   struct Kern_Entry *K_Ptr;
	auto   struct Lig_Entry *L_Ptr;
	auto   char *Ptr;
	auto   int Index;
	static char Family[40], Typeface[40], Temp_Str[128], Face[4];
	static struct {
		char *Face_Name;
		char Face_Code;
	} Face_Data[] = {
		{ "light", 'L' },
		{ "medium", 'M' },
		{ "bold", 'B' },
		{ "demi", 'M' },
		{ "book", 'B' }
	};
	static char *Param_Names[] = {
		"SLANT", "SPACE", "STRETCH", "SHRINK", "XHEIGHT", "QUAD", "EXTRASPACE",
		"NUM1", "NUM2", "NUM3", "DENOM1", "DENOM2", "SUP1", "SUP2", "SUP3",
		"SUB1", "SUB2", "SUPDROP", "SUBDROP", "DELIM1", "DELIM2", "AXISHEIGHT"
	};
	static char *Encoding_Names[] = {
		"TEX TEXT", "TEX TYPEWRITER TEXT", "POSTSCRIPT TEXT", "POSTSCRIPT SYMBOL",
		"ASCII", "EBCDIC"
	};
	extern char *Code_C();
	extern int sscanf(), strcmp(), strlen();
/*
 *	Parse the font name to get the family and face name. These
 *	can be used by the DVI-reading program to reconstruct the
 *	original font name.
 */
	Typeface[0] = '\0';
	stringcpy_m (Family, Font_Name, sizeof (Family));
	if (strcmp (Family, "Helvetica-Narrow") != 0) {	/* an exception */
		for (Ptr = &Font_Name[strlen(Font_Name)]; Ptr > Font_Name; )
		if (*--Ptr == '-') {
			*Ptr = '\0';
			stringcpy_m (Family, Font_Name, sizeof (Family));
			*Ptr++ = '-';
			stringcpy_m (Typeface, Ptr, sizeof (Typeface));
			break;
		}
	}
/*
 *	Encode (weight,slope,expansion) triplet. This is later
 *	used to reconstruct the original full font name.
 */
	for (Index = 0; Index < arraysize (Face_Data); Index++)
	if (strcmp (Face_Data[Index].Face_Name, Weight) == 0) {
		Face[0] = Face_Data[Index].Face_Code;
		break;
	}
	if (Index >= arraysize (Face_Data)) {
		sprintf (Temp_Str, "%%QUERYLW-W-UNKFACE, face weight %s not recognized, using medium", Weight);
		To_Log (Temp_Str);
		Face[0] = 'M';
	}
	Face[1] = (Slant_Angle == 0.0) ? 'R' : 'I';
	Face[2] = 'R';
	Face[3] = '\0';
/*
 *	Output header information:
 */
	printf ("(FAMILY %s)\n", Family);
	printf ("(FACE F %s)\n", Face);
	printf ("(CODINGSCHEME %s)\n", Encoding_Names[Encoding_Scheme]);
	printf ("(DESIGNSIZE R %f)\n", Design_Size);
	printf ("(CHECKSUM H %08X)\n", Checksum);
	printf ("(FONTDIMEN\n");
	for (Index = 0; Index < 7; Index++)
		printf ("  (%s R %f)\n", Param_Names[Index], Param_Array[Index] / Design_Size);
	printf ("  )\n");
/*
 *	Output character information:
 */
	for (Index = 0; Index < 256; Index++)
	if (Char_Vector[Index] != 0) {
		printf ("(CHARACTER %s\n", Code_C (Index, Temp_Str));
		if (Char_Defs[Index].Char_Width != 0.0)
			printf ("  (CHARWD R %f)\n", Char_Defs[Index].Char_Width / Design_Size);
		if (Char_Defs[Index].Char_Height != 0.0)
			printf ("  (CHARHT R %f)\n", Char_Defs[Index].Char_Height / Design_Size);
		if (Char_Defs[Index].Char_Depth != 0.0)
			printf ("  (CHARDP R %f)\n", Char_Defs[Index].Char_Depth / Design_Size);
		if (Char_Defs[Index].Char_Italic != 0.0)
			printf ("  (CHARIC R %f)\n", Char_Defs[Index].Char_Italic / Design_Size);
		printf ("  )\n");
	}
/*
 *	Output Kern and Ligature information:
 */
	if (Lig_Kern_Count > 0) {
		printf ("(LIGTABLE\n");
		for (Index = 0; Index < 256; Index++)
		if (Char_Defs[Index].Lig_Ptr != 0 || Char_Defs[Index].Kern_Ptr != 0) {
			printf ("  (LABEL %s)\n", Code_C (Index, Temp_Str));
			for (L_Ptr = Char_Defs[Index].Lig_Ptr; L_Ptr != 0; L_Ptr = L_Ptr->Link)
				printf ("  (LIG %s %s)\n", Code_C (L_Ptr->Lig_Char, Temp_Str),
					Code_C (L_Ptr->New_Char, &Temp_Str[20]));
			for (K_Ptr = Char_Defs[Index].Kern_Ptr; K_Ptr != 0; K_Ptr = K_Ptr->Link)
				printf ("  (KRN %s R %f)\n", Code_C (K_Ptr->Kern_Char, Temp_Str), K_Ptr->Kern);
			printf ("  (STOP)\n");
		}
		printf ("  )\n");
	}
}

char *Code_C (Char_Code, String)
unsigned char Char_Code;
char *String;
{
	if (Char_Code < 041 || Char_Code > 0176 || Char_Code == 050 || Char_Code == 051)
		sprintf (String, "O %o", Char_Code);
	else
		sprintf (String, "C %c", Char_Code);
	return (String);
}
