/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains VMS I/O routines.
 */

#include descrip
#include "vms.h"
#include "iostatus.h"
#include "vmsio.h"

#define FILE_BUFFER_SIZE 8192

/*
 *	A word of explanation is in order about file processing. Mostly,
 *	the files used by the DVIOUT program and the symbiont will be
 *	binary files like font files, tfm files, dvi files, etc. These
 *	files are all accessed using block I/O and a rather large buffer
 *	to make things speedy. However, about half-way through this, it
 *	was realized that the capability to process normal text files
 *	(say, containing PostScript stuff generated by an editor) would
 *	be an attractive feature. So now there is a lot of extra stuff
 *	cluttering up the file I/O routines to handle this, but it's
 *	probably worth it.
 *
 *	So, normal sequential files are handled by treating each record
 *	in the file as a 'block', so that block 10 of a sequential file
 *	is the 10th record in the file. Additionally, the caller of
 *	'Open_File' can specify an optional string that will be appended
 *	to the end of each record read in from a sequential file. This is
 *	to facilitate, for example, automatically inserting a line feed
 *	at the end of each record in a PostScript file to serve as the
 *	necessary parameter separator for PostScript processors like the
 *	LaserWriter.
 *
 *	Some of the more important values stored in the control blocks
 *	that are rigorously maintained:
 *
 *	RAB: rab$l_rbf   - address of current buffer
 *	RAB: rab$w_usz   - maximum size of buffer
 *	RAB: rab$w_rsz   - actual size of current buffer
 *	BAB: bab$l_bfp   - address of next character to be read or written
 *                         in current buffer
 *	BAB: bab$w_hsz   - size of 'high' part of buffer, from bab$l_bfp
 *                         to the end of the buffer. The relationship
 *                         bab$w_hsz = &rab$l_rbf[rab$w_rsz] - bab$l_bfp
 *                         should always hold
 *	BAB: bab$l_eof   - address of end of file byte, if in the current
 *                         buffer, zero if not
 *	XAB: xab$l_ebk   - end of file block, in terms of our block size,
 *                         not VMS's
 *	XAB: xab$l_ffb   - first free byte in the end of file block.
 *
 *	Routine Open_File opens a file for read or write access. If the
 *	file name is not present, the default name should contain the
 *	address of a file identification field used in the NAM block.
 */

struct BAB *Open_File (Name, Default_Name, Mode, EOLStr)
char *Name, *Default_Name, *Mode, *EOLStr;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct NAM *Nam_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned long (*Open_F)(), EOF_Block;
	auto   unsigned short EOLSLen;
	extern struct FAB *Fab_Alloc();
	extern struct RAB *Rab_Alloc();
	extern struct BAB *Bab_Alloc();
	extern struct NAM *Nam_Alloc();
	extern struct XABFHC *Xabfhc_Alloc();
	extern unsigned long Sys$Open(), Sys$Create(), Sys$Close();
	extern unsigned long Sys$Connect();
	extern int Check_System_Status();
	extern int strlen();
/*
 *	Allocate File-access data structures:
 */
	Nam_Ptr = Nam_Alloc ();
	Xab_Ptr = Xabfhc_Alloc (0);
	Fab_Ptr = Fab_Alloc (Nam_Ptr, Xab_Ptr, Name, Default_Name, Mode);
/*
 *	Open the file:
 */
	Open_F = ((Fab_Ptr->fab$b_fac & FAB$M_GET) != 0) ? &Sys$Open : &Sys$Create;
	if (((*Open_F) (Fab_Ptr) & 0x01) == 0) {
		Free_Fab (Fab_Ptr);
		return (0);
	}
/*
 *	Put string terminators on resultant and expanded file names:
 */
	Nam_Ptr->nam$l_rsa[Nam_Ptr->nam$b_rsl] = '\0';
	Nam_Ptr->nam$l_esa[Nam_Ptr->nam$b_esl] = '\0';
/*
 *	Allocate record-access data structures:
 */
	EOLSLen = (EOLStr == 0) ? 0 : strlen (EOLStr);
	Rab_Ptr = Rab_Alloc (Fab_Ptr, Mode, FILE_BUFFER_SIZE, EOLSLen);
/*
 *	Connect the record stream:
 */
	if ((Sys$Connect (Rab_Ptr) & 0x01) == 0) {
		Check_System_Status (Sys$Close (Fab_Ptr));
		Free_Rab (Rab_Ptr);
		return (0);
	}
	Bab_Ptr = Bab_Alloc (Rab_Ptr);
	Bab_Ptr->bab$w_eolslen = EOLSLen;
	Bab_Ptr->bab$l_eolstr = EOLStr;
/*
 *	If the file is accessed sequentially, build the RFA index; else
 *	reset EOF block and first free byte to correspond to the buffer
 *	size in use;
 */
	if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0) {
		Build_File_Record_Index (Bab_Ptr, Xab_Ptr->xab$l_ebk, Xab_Ptr->xab$w_ffb);
		Xab_Ptr->xab$l_ebk = Bab_Ptr->bab$l_nrecs + 1;
		Xab_Ptr->xab$w_ffb = 0;
	} else {
		if ((Fab_Ptr->fab$b_fac & FAB$M_GET) == 0) {	/* NOT set by RMS */
			Xab_Ptr->xab$l_ebk = 1;
			Xab_Ptr->xab$w_ffb = 0;
		} else {
			EOF_Block = Xab_Ptr->xab$l_ebk - 1;
			Xab_Ptr->xab$l_ebk = EOF_Block / (Rab_Ptr->rab$w_usz >> 9) + 1;
			Xab_Ptr->xab$w_ffb += (EOF_Block % (Rab_Ptr->rab$w_usz >> 9)) << 9;
		}
	}
/*
 *	Prime the buffer by reading in the first block of the file
 *	(for new files, this sets up the proper buffer pointers, the
 *	first buffer being at the end of file):
 */
	Read_File_Buffer (1, Bab_Ptr);
	return (Bab_Ptr);
}

int Delete_File (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   unsigned long Sys_Status;
	extern unsigned long Sys$Disconnect(), Sys$Close();
	extern int Check_System_Status();
	globalvalue RMS$_MKD, RMS$_PRV, RMS$_WLK;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Fab_Ptr->fab$l_fop |= FAB$M_DLT;
	Check_System_Status (Sys$Disconnect (Rab_Ptr));
	Sys_Status = Sys$Close (Fab_Ptr);
	Free_Bab (Bab_Ptr);
	if (Sys_Status == RMS$_MKD || Sys_Status == RMS$_PRV || Sys_Status == RMS$_WLK)
		return (0);
	else {
		Check_System_Status (Sys_Status);
		return (1);
	}
}

Close_File (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	extern unsigned long Sys$Disconnect(), Sys$Close();
	extern int Check_System_Status();

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	if ((Fab_Ptr->fab$b_fac & FAB$M_PUT) != 0)
		Write_File_Buffer (Bab_Ptr);
	Check_System_Status (Sys$Disconnect (Rab_Ptr));
	Check_System_Status (Sys$Close (Fab_Ptr));
	Free_Bab (Bab_Ptr);
}

Free_Bab (Bab_Ptr)
struct BAB *Bab_Ptr;
{
	if (Bab_Ptr != 0) {
		Free_Rab (Bab_Ptr->bab$l_rab);
		if (Bab_Ptr->bab$l_rcidx != 0)
			Mem_Free (Bab_Ptr->bab$l_rcidx);
		Mem_Free (Bab_Ptr);
	}
}

Free_Rab (Rab_Ptr)
struct RAB *Rab_Ptr;
{
	if (Rab_Ptr != 0) {
		Free_Fab (Rab_Ptr->rab$l_fab);
		Mem_Free (Rab_Ptr);
	}
}

Free_Fab (Fab_Ptr)
struct FAB *Fab_Ptr;
{
	if (Fab_Ptr != 0) {
		Mem_Free (Fab_Ptr->fab$l_xab);
		Mem_Free (Fab_Ptr->fab$l_nam);
		Mem_Free (Fab_Ptr);
	}
}

struct NAM *Nam_Alloc ()
{
	auto   struct NAM *Nam_Ptr;
	auto   char *Ptr;
	extern char *Mem_Alloc();

	Nam_Ptr = (struct NAM *) Mem_Alloc (NAM$C_BLN + 2 * NAM$C_MAXRSS + 2);
	Table_Init (Nam_Ptr, NAM$C_BID, NAM$C_BLN);
	Ptr = (char *) Nam_Ptr;
	Nam_Ptr->nam$l_rsa = &Ptr[NAM$C_BLN];
	Nam_Ptr->nam$b_rss = NAM$C_MAXRSS;
	Nam_Ptr->nam$l_esa = &Ptr[NAM$C_BLN + NAM$C_MAXRSS + 1];
	Nam_Ptr->nam$b_ess = NAM$C_MAXRSS;
	return (Nam_Ptr);
}

struct XABFHC *Xabfhc_Alloc (Next_Xab_Ptr)
char *Next_Xab_Ptr;
{
	auto   struct XABFHC *Xab_Ptr;
	extern char *Mem_Alloc();

	Xab_Ptr = (struct XABFHC *) Mem_Alloc (XAB$C_FHCLEN);
	Table_Init (Xab_Ptr, XAB$C_FHC, XAB$C_FHCLEN);
	Xab_Ptr->xab$l_nxt = Next_Xab_Ptr;
	return (Xab_Ptr);
}

struct FAB *Fab_Alloc (Nam_Ptr, Xab_Ptr, Name, Default_Name, Mode)
struct NAM *Nam_Ptr;
struct XABFHC *Xab_Ptr;
char *Name, *Default_Name, *Mode;
{
	auto   struct FAB *Fab_Ptr;
	extern char *Mem_Alloc();
	extern int strlen();
	struct nam_file_id { char a[NAM$C_DVI]; short b[3]; short c[3]; };

	Fab_Ptr = (struct FAB *) Mem_Alloc (FAB$C_BLN);
	Table_Init (Fab_Ptr, FAB$C_BID, FAB$C_BLN);
	Fab_Ptr->fab$l_nam = Nam_Ptr;
	Fab_Ptr->fab$l_xab = Xab_Ptr;
	Fab_Ptr->fab$l_fop = 0;
	Fab_Ptr->fab$b_fac = FAB$M_BRO;
	Fab_Ptr->fab$b_shr = 0;
	if (Name != 0) {
		Fab_Ptr->fab$l_fna = Name;
		Fab_Ptr->fab$b_fns = (char) strlen (Name);
		if (Default_Name != 0) {
			Fab_Ptr->fab$l_dna = Default_Name;
			Fab_Ptr->fab$b_dns = (char) strlen (Default_Name);
		}
	} else {	/* Use NAM block (file id) to open the file */
		*(struct nam_file_id *) &Nam_Ptr->nam$t_dvi[0] = *(struct nam_file_id *) Default_Name;
		Fab_Ptr->fab$l_fop |= FAB$M_NAM;
	}
	if (*Mode == 'r') {
		Fab_Ptr->fab$b_shr |= FAB$M_SHRGET;
		Fab_Ptr->fab$b_fac |= FAB$M_GET;
	} else {
		Fab_Ptr->fab$l_fop |= FAB$M_TEF;
		Fab_Ptr->fab$b_fac |= FAB$M_TRN | FAB$M_PUT;
		if (*Mode == 'u')
			Fab_Ptr->fab$b_fac |= FAB$M_GET;
		else {		/* New file, for write */
			Fab_Ptr->fab$l_fop |= FAB$M_MXV;
			Fab_Ptr->fab$b_org = FAB$C_SEQ;
			Fab_Ptr->fab$b_rfm = FAB$C_FIX;
			Fab_Ptr->fab$w_mrs = 512;
		}
	}
	return (Fab_Ptr);
}

struct RAB *Rab_Alloc (Fab_Ptr, Mode, Block_Buf_Size, EOLSLen)
struct FAB *Fab_Ptr;
char *Mode;
unsigned short Block_Buf_Size, EOLSLen;
{
	auto   struct RAB *Rab_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned long Rop;
	auto   unsigned int Size;
	auto   unsigned char Rac;
	auto   char *Ptr;
	extern char *Mem_Alloc();
/*
 *	Determine whether to use block I/O or record I/O, and the corresponding
 *	buffer size. If the file has variable length records, we access
 *	the file sequentially by record (at least at first); if not, we use
 *	raw block i/o (a file with a defined record format must be an
 *	existing file, since new files that we create have no defined
 *	record format). Note that the fab$w_mrs field in the FAB is not
 *	reliable to determine the maximum buffer size needed to store one
 *	record of the file.
 *
 *	There is a possible problem here if the longest record in the
 *	file is near the maximum of 65535. The end of line string length
 *	added to the longest record size could cause overflow, since the
 *	size (in rab$w_usz) is a 16 bit quantity. It is considered
 *	sufficiently improbable that this will happen that the consequences
 *	(i.e. various errors from RMS services) are acceptable.
 */
	Xab_Ptr = Fab_Ptr->fab$l_xab;
	if ((Fab_Ptr->fab$b_rfm == FAB$C_FIX || Fab_Ptr->fab$b_rfm == FAB$C_UDF) &&
	    Fab_Ptr->fab$b_org == FAB$C_SEQ) {
		Rop = RAB$M_BIO;
		Rac = 0;
		Size = Block_Buf_Size;
	} else {
		Rop = 0;
		Rac = RAB$C_SEQ;
		Size = Xab_Ptr->xab$w_lrl + EOLSLen;
		if (Size > 65535)
			Size = 65535;
	}
	Rab_Ptr = (struct RAB *) Mem_Alloc (RAB$C_BLN + (int) Size);
	Table_Init (Rab_Ptr, RAB$C_BID, RAB$C_BLN);
	Ptr = (char *) Rab_Ptr;
	Rab_Ptr->rab$l_fab = Fab_Ptr;
	Rab_Ptr->rab$l_ubf = &Ptr[RAB$C_BLN];
	Rab_Ptr->rab$w_usz = Size;
	Rab_Ptr->rab$l_rop = Rop;
	Rab_Ptr->rab$b_rac = Rac;
	return (Rab_Ptr);
}

struct BAB *Bab_Alloc (Rab_Ptr)
struct RAB *Rab_Ptr;
{
	auto   struct BAB *Bab_Ptr;
	extern char *Mem_Alloc();

	Bab_Ptr = (struct BAB *) Mem_Alloc (BAB$C_BLN);
	Table_Init (Bab_Ptr, BAB$C_BID, BAB$C_BLN);
	Bab_Ptr->bab$l_rab = Rab_Ptr;
	Bab_Ptr->bab$l_vbn = 0;
	return (Bab_Ptr);
}

Table_Init (Tab_Ptr, Id, Size)
unsigned char *Tab_Ptr;
unsigned char Id, Size;
{
	auto   unsigned char *Table_Ptr;

	Table_Ptr = Tab_Ptr;
	*Table_Ptr++ = Id;
	*Table_Ptr++ = Size;
	while (Table_Ptr < Tab_Ptr + Size)
		*Table_Ptr++ = 0;
}

/*
 *	Routine Build_File_Record_Index reads through a sequential
 *	file and records the record file addresses of each record:
 */

Build_File_Record_Index (Bab_Ptr, EOF_Block, EOF_Byte)
struct BAB *Bab_Ptr;
unsigned long EOF_Block;
unsigned short EOF_Byte;
{
	auto   struct RAB *Rab_Ptr;
	auto   struct RFA {
		struct RFA *Link;
		unsigned long Record_Address;
	} *RFA_Ptr0, *RFA_Ptr;
	auto   unsigned long Count, Sys_Status;
	extern unsigned long Sys$Get();
	extern char *Mem_Alloc();
	globalvalue RMS$_EOF;
/*
 *	Assign some things; set for sequential access and locate mode:
 */
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Rab_Ptr->rab$b_rac = RAB$C_SEQ;
	Rab_Ptr->rab$l_rop |= RAB$M_LOC;
/*
 *	Read in each record of the file and record it's file address:
 */
	RFA_Ptr0 = 0;
	Count = 0;
	while (((Sys_Status = Sys$Get (Rab_Ptr)) & 0x01) != 0) {
		RFA_Ptr = (struct RFA *) Mem_Alloc (sizeof (struct RFA));
		RFA_Ptr->Link = RFA_Ptr0;
		RFA_Ptr->Record_Address =
			((*(unsigned long *) &Rab_Ptr->rab$w_rfa[0] - 1) << 9) +
			(Rab_Ptr->rab$w_rfa[2] & 0x01FF);
		RFA_Ptr0 = RFA_Ptr;
		Count++;
	}
/*
 *	Allocate the permanent index and fill it in:
 */
	Bab_Ptr->bab$l_nrecs = Count;
	Bab_Ptr->bab$l_rcidx = (unsigned long *) Mem_Alloc ((Count + 1) * sizeof (unsigned long));
	Bab_Ptr->bab$l_rcidx[Count] = (EOF_Block << 9) + (EOF_Byte & 0x01FF);
	while ((RFA_Ptr = RFA_Ptr0) != 0) {
		Bab_Ptr->bab$l_rcidx[--Count] = RFA_Ptr->Record_Address;
		RFA_Ptr0 = RFA_Ptr->Link;
		Mem_Free (RFA_Ptr);
	}
/*
 *	Reset RAB parameters:
 */
	Rab_Ptr->rab$l_rop &= ~RAB$M_LOC;
	Rab_Ptr->rab$b_rac = RAB$C_RFA;
}

Set_File_Open_Status (Bab_Ptr, Status)
struct BAB *Bab_Ptr;
unsigned long Status[2];
{
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;

	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Status[0] = Fab_Ptr->fab$l_sts;
	Status[1] = Fab_Ptr->fab$l_stv;
}

char *Full_File_Name (Bab_Ptr)
struct BAB *Bab_Ptr;
{
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct NAM *Nam_Ptr;

	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Nam_Ptr = Fab_Ptr->fab$l_nam;
	return (Nam_Ptr->nam$l_rsa);
}

unsigned char Read_Character (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   unsigned char c;

	Bab_Ptr = Bab_Pointer;
	if (Bab_Ptr->bab$l_bfp == Bab_Ptr->bab$l_eof)
		c = 0;
	else if (Bab_Ptr->bab$w_hsz > 0) {
		c = *Bab_Ptr->bab$l_bfp++;
		Bab_Ptr->bab$w_hsz--;
	} else {
		Read_File_Buffer (Bab_Ptr->bab$l_vbn + 1, Bab_Ptr);
		if (Bab_Ptr->bab$l_bfp == Bab_Ptr->bab$l_eof)
			c = 0;
		else {
			c = *Bab_Ptr->bab$l_bfp++;
			Bab_Ptr->bab$w_hsz--;
		}
	}
	return (c);
}

unsigned char *Read_Character_Reverse (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   unsigned char c;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	if (Bab_Ptr->bab$l_bfp > Rab_Ptr->rab$l_rbf) {
		c = *--Bab_Ptr->bab$l_bfp;
		Bab_Ptr->bab$w_hsz++;
	} else if (Bab_Ptr->bab$l_vbn == 1)
		c = 0;
	else {
		Read_File_Buffer (Bab_Ptr->bab$l_vbn - 1, Bab_Ptr);
		Bab_Ptr->bab$l_bfp = &Rab_Ptr->rab$l_rbf[Rab_Ptr->rab$w_rsz];
		c = *--Bab_Ptr->bab$l_bfp;
		Bab_Ptr->bab$w_hsz = 1;
	}
	return (c);
}

int Read_Block (Block_Ptr, Count, Bab_Pointer)
char *Block_Ptr;
unsigned int Count;
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   char *Ptr;
	auto   unsigned int Cnt;

	Ptr = Block_Ptr;
	Bab_Ptr = Bab_Pointer;
	Cnt = Count;
	while (Cnt > 0 && Bab_Ptr->bab$l_bfp != Bab_Ptr->bab$l_eof)
	if (Bab_Ptr->bab$w_hsz == 0)
		Read_File_Buffer (Bab_Ptr->bab$l_vbn + 1, Bab_Ptr);
	else while (Cnt > 0 && Bab_Ptr->bab$w_hsz > 0 &&
		    Bab_Ptr->bab$l_bfp != Bab_Ptr->bab$l_eof) {
		*Ptr++ = *Bab_Ptr->bab$l_bfp++;
		Bab_Ptr->bab$w_hsz--;
		Cnt--;
	}
	return (Count - Cnt);
}

Write_Character (c, Bab_Pointer)
char c;
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned short Size;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Xab_Ptr = Fab_Ptr->fab$l_xab;
/*
 *	Put the character into the buffer:
 */
	*Bab_Ptr->bab$l_bfp++ = c;
	Bab_Ptr->bab$w_hsz--;
	Bab_Ptr->bab$w_flags |= BAB_BUFFER_WRITTEN;
/*
 *	Update the end of file, if necessary:
 */
	Size = Bab_Ptr->bab$l_bfp - Rab_Ptr->rab$l_rbf;
	if (Bab_Ptr->bab$l_vbn == Xab_Ptr->xab$l_ebk && Size > Xab_Ptr->xab$w_ffb) {
		if (Size == Rab_Ptr->rab$w_usz) {
			Xab_Ptr->xab$l_ebk++;
			Xab_Ptr->xab$w_ffb = 0;
		} else
			Xab_Ptr->xab$w_ffb = Size;
		Rab_Ptr->rab$w_rsz = Size;
		Bab_Ptr->bab$w_hsz = 0;
		Bab_Ptr->bab$l_eof = Bab_Ptr->bab$l_bfp;
	}
/*
 *	Check if the buffer is full. If so, write out the
 *	buffer to the file and read in the next:
 */
	if (Size == Rab_Ptr->rab$w_usz)
		Read_File_Buffer (Bab_Ptr->bab$l_vbn + 1, Bab_Ptr);
}

int Write_Block (Block_Ptr, Count, Bab_Pointer)
char *Block_Ptr;
unsigned int Count;
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   char *Ptr;
	auto   unsigned int Cnt;

	Bab_Ptr = Bab_Pointer;
	for (Ptr = Block_Ptr, Cnt = Count; Cnt > 0; Cnt--)
		Write_Character (*Ptr++, Bab_Ptr);
}

Set_File_IO_Status (Bab_Ptr, Status)
struct BAB *Bab_Ptr;
unsigned long Status[2];
{
	auto   struct RAB *Rab_Ptr;

	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Status[0] = Rab_Ptr->rab$l_sts;
	Status[1] = Rab_Ptr->rab$l_stv;
}

unsigned long Set_File_Position (File_Address, Bab_Pointer)
unsigned long File_Address;
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   unsigned long EOF_Addr, Addr, Block_Number;
	auto   unsigned short Rest;
	extern unsigned long Get_EOF_Address();

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	EOF_Addr = Get_EOF_Address (Bab_Ptr);
	Addr = (File_Address < EOF_Addr) ? File_Address : EOF_Addr;
	if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0) {
		for (Block_Number = 0; Block_Number < Bab_Ptr->bab$l_nrecs &&
			Bab_Ptr->bab$l_rcidx[Block_Number+1] <= Addr; Block_Number++)
			;
		Rest = Addr - Bab_Ptr->bab$l_rcidx[Block_Number];
	} else {
		Block_Number = Addr / Rab_Ptr->rab$w_usz;
		Rest = Addr % Rab_Ptr->rab$w_usz;
	}
	Read_File_Buffer (Block_Number + 1, Bab_Ptr);
	Bab_Ptr->bab$l_bfp = &Bab_Ptr->bab$l_bfp[Rest];
	Bab_Ptr->bab$w_hsz -= Rest;
	return (Addr);
}

unsigned long Set_EOF (Bab_Ptr)
struct BAB *Bab_Ptr;
{
	extern unsigned long Set_File_Position(), Get_EOF_Address();

	return (Set_File_Position (Get_EOF_Address (Bab_Ptr), Bab_Ptr));
}

int File_At_EOF (Bab_Ptr)
struct BAB *Bab_Ptr;
{
	return ((Bab_Ptr->bab$l_bfp == Bab_Ptr->bab$l_eof) ? 1 : 0);
}

Read_File_Buffer (Block_Number, Bab_Pointer)
unsigned long Block_Number;
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned long EOF_Block, Sys_Status, Rec_Addr;
	auto   unsigned short Size;
	auto   char *StrPtr;
	extern unsigned long Sys$Read(), Sys$Get();
	extern int Check_System_Status();
	globalvalue RMS$_NORMAL, RMS$_EOF;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Xab_Ptr = Fab_Ptr->fab$l_xab;
/*
 *	Flush the current buffer, if necessary:
 */
	if ((Fab_Ptr->fab$b_fac & FAB$M_PUT) != 0 && Bab_Ptr->bab$l_vbn != Block_Number)
		Write_File_Buffer (Bab_Ptr);
/*
 *	Skip the actual read if the current block is the one desired;
 *	otherwise set the virtual block number and read it in;
 */
	if (Bab_Ptr->bab$l_vbn == Block_Number)
		Sys_Status = RMS$_NORMAL;
	else if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0) {
		Rec_Addr = Bab_Ptr->bab$l_rcidx[Block_Number-1];
		*(unsigned long *) &Rab_Ptr->rab$w_rfa[0] = (Rec_Addr >> 9) + 1;
		Rab_Ptr->rab$w_rfa[2] = Rec_Addr & 0x01FF;
		Sys_Status = Sys$Get (Rab_Ptr);
		Bab_Ptr->bab$l_nreads++;
	} else {
		Rab_Ptr->rab$l_bkt = (Block_Number - 1) * (Rab_Ptr->rab$w_usz >> 9) + 1;
		Sys_Status = Sys$Read (Rab_Ptr);
		Bab_Ptr->bab$l_nreads++;
	}
	if (Sys_Status == RMS$_EOF)
		Sys_Status = RMS$_NORMAL;
/*
 *	Set pointers according to amount read in:
 */
	if (Check_System_Status (Sys_Status) == 0) {	/* Error other than EOF */
		Rab_Ptr->rab$w_rsz = 0;
		Bab_Ptr->bab$l_eof = Rab_Ptr->rab$l_rbf = Rab_Ptr->rab$l_ubf;
	} else {
		if (Xab_Ptr->xab$l_ebk == Block_Number)
			Bab_Ptr->bab$l_eof = &Rab_Ptr->rab$l_rbf[Rab_Ptr->rab$w_rsz];
		else
			Bab_Ptr->bab$l_eof = 0;
/*
 *	Append the End-of-Line string for non-block reads:
 */
		if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0 && Bab_Ptr->bab$w_eolslen != 0) {
			for (StrPtr = Bab_Ptr->bab$l_eolstr; *StrPtr != '\0'; )
				Rab_Ptr->rab$l_rbf[Rab_Ptr->rab$w_rsz++] = *StrPtr++;
		}
	}
	Bab_Ptr->bab$w_hsz = Rab_Ptr->rab$w_rsz;
	Bab_Ptr->bab$l_bfp = Rab_Ptr->rab$l_rbf;
	Bab_Ptr->bab$l_vbn = Block_Number;
}

Write_File_Buffer (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned long Sys_Status;
	extern unsigned long Sys$Write();
	extern int Check_System_Status();
	globalvalue RMS$_NORMAL;
/*
 *	If the buffer is defined and has been written to, write the
 *	buffer out to the file. Unlike Read_File_Buffer, this routine
 *	leaves all associated buffer pointers defined exactly as they
 *	are upon entry:
 */
	Bab_Ptr = Bab_Pointer;
	if (Bab_Ptr->bab$l_vbn != 0 && (Bab_Ptr->bab$w_flags & BAB_BUFFER_WRITTEN) != 0) {
		Rab_Ptr = Bab_Ptr->bab$l_rab;
		Fab_Ptr = Rab_Ptr->rab$l_fab;
		Xab_Ptr = Fab_Ptr->fab$l_xab;
		Rab_Ptr->rab$l_bkt = (Bab_Ptr->bab$l_vbn - 1) * (Rab_Ptr->rab$w_usz >> 9) + 1;
		if (Rab_Ptr->rab$w_rsz == 0)	/* Zero-length write not permitted */
			Sys_Status = RMS$_NORMAL;
		else {
			Sys_Status = Sys$Write (Rab_Ptr);
			Bab_Ptr->bab$l_nwrites++;
		}
		Check_System_Status (Sys_Status);
		Bab_Ptr->bab$w_flags &= ~BAB_BUFFER_WRITTEN;
	}
}

unsigned long File_Position (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   unsigned long Addr;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	if (Bab_Ptr->bab$l_vbn == 0)
		Addr = 0;
	else {
		if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0)
			Addr = Bab_Ptr->bab$l_rcidx[Bab_Ptr->bab$l_vbn-1];
		else
			Addr = (Bab_Ptr->bab$l_vbn - 1) * Rab_Ptr->rab$w_usz;
		Addr += Bab_Ptr->bab$l_bfp - Rab_Ptr->rab$l_rbf;
	}
	return (Addr);
}

unsigned long Get_EOF_Address (Bab_Pointer)
struct BAB *Bab_Pointer;
{
	auto   struct BAB *Bab_Ptr;
	auto   struct RAB *Rab_Ptr;
	auto   struct FAB *Fab_Ptr;
	auto   struct XABFHC *Xab_Ptr;
	auto   unsigned long EOF_Addr;

	Bab_Ptr = Bab_Pointer;
	Rab_Ptr = Bab_Ptr->bab$l_rab;
	Fab_Ptr = Rab_Ptr->rab$l_fab;
	Xab_Ptr = Fab_Ptr->fab$l_xab;
	if ((Rab_Ptr->rab$l_rop & RAB$M_BIO) == 0)
		EOF_Addr = Bab_Ptr->bab$l_rcidx[Bab_Ptr->bab$l_nrecs];
	else
		EOF_Addr = (Xab_Ptr->xab$l_ebk - 1) * Rab_Ptr->rab$w_usz + Xab_Ptr->xab$w_ffb;
	return (EOF_Addr);
}

/*
 *	Routine Get_File_Stats returns the number of reads and writes
 *	from/to the file:
 */

Get_File_Stats (Bab_Ptr, Stats)
struct BAB *Bab_Ptr;
unsigned long Stats[2];
{
	Stats[0] = Bab_Ptr->bab$l_nreads;
	Stats[1] = Bab_Ptr->bab$l_nwrites;
}
