(* VDU routines for a VT100 terminal in 132 column mode. *)

#include 'globals.h';
#include 'screenio.h';
#include 'vdu.h';

VAR cursrow, curscol : INTEGER;   (* ShowChar remembers cursor location *)

(******************************************************************************)

PROCEDURE StartText;

(* We are about to draw text in dialogue region. *)

BEGIN
(* VT100 treats text and graphics the same, so do nothing *)
END; (* StartText *)

(******************************************************************************)

PROCEDURE MoveAbs (row, col : INTEGER);

(* Move cursor to given screen position. *)

BEGIN
WriteChar(ESC); WriteChar('[');
WriteInt(row);
WriteChar(';');
WriteInt(col);
WriteChar('H');
END; (* MoveAbs *)

(******************************************************************************)

PROCEDURE MoveToTextLine (line : INTEGER);

(* Move current position to start of given line. *)

BEGIN
MoveAbs(line,1);
END; (* MoveToTextLine *)

(******************************************************************************)

PROCEDURE ClearTextLine (line : INTEGER);

(* Erase given line; note that DVItoVDU does not assume anything about the
   current position at the end of this routine.
*)

BEGIN
MoveAbs(line,1);
WriteChar(ESC);
WriteString('[K');   (* erase to end of line *)
END; (* ClearTextLine *)

(******************************************************************************)

PROCEDURE ClearScreen;

BEGIN
WriteChar(ESC);
WriteString('[2J');   (* erase entire screen *)
END; (* ClearScreen *)

(******************************************************************************)

PROCEDURE StartGraphics;

(* We are about to draw in window region.
   VT100 makes no distinction between text and graphics.
   All we do is reset the current cursor position to some undefined state for
   use in the next ShowChar call.
*)

BEGIN
cursrow := 0;
END; (* StartGraphics *)

(******************************************************************************)

PROCEDURE LoadFont (fontname : string;
                    fontsize : INTEGER;
                    mag, hscale, vscale : REAL);

BEGIN
(* only one character size available on VT100s, so do nothing *)
END; (* LoadFont *)

(******************************************************************************)

PROCEDURE ShowChar (screenh, screenv : INTEGER;
                    ch : CHAR);

(* Show the given Terse character (mapped to ASCII) at the given position.
   We remember the cursor position (cursrow,curscol) so we can reduce the
   output bytes needed to position the next Terse character.
   StartGraphics resets the position to an undefined state (cursrow = 0).
   We also reset when the cursor reaches the right edge (= windowwd) to
   avoid possibility of any auto wrap.
*)

VAR amount : INTEGER;

BEGIN
(* first translate DVItoVDU coordinates into actual screen location *)
screenh := screenh + 1;
screenv := screenv + 1;
IF cursrow = screenv THEN BEGIN
   (* The cursor is on the same line as in previous ShowChar call so we only
      need to move left or right, and probably just a small amount (if at all).
   *)
   IF screenh = curscol THEN       (* cursor in correct location *)
      curscol := curscol + 1       (* cursor will move right when ch written *)
   ELSE IF screenh < curscol THEN BEGIN      (* move cursor left *)
      amount := curscol - screenh;
      WriteChar(ESC); WriteChar('[');
      IF amount > 1 THEN BEGIN               (* default is 1 col *)
         WriteInt(amount);
         curscol := curscol - (amount-1);    (* no need if amount = 1 *)
      END;
      WriteChar('D');
   END
   ELSE BEGIN                                (* move cursor right *)
      amount := screenh - curscol;
      WriteChar(ESC); WriteChar('[');
      IF amount > 1 THEN WriteInt(amount);   (* default is 1 col *)
      curscol := curscol + (amount+1);
      WriteChar('C');
   END;
END
ELSE BEGIN                         (* cursrow undefined or ch on a new line *)
   MoveAbs(screenv,screenh);
   cursrow := screenv;
   curscol := screenh + 1;         (* cursor will move right when ch written *)
END;
IF screenh = windowwd THEN         (* ch will be written at right edge *)
   cursrow := 0;                   (* so avoid auto wrap next time around *)
WriteChar(TeXtoASCII[ch]);
END; (* ShowChar *)

(******************************************************************************)

PROCEDURE ShowRectangle (screenh, screenv,          (* top left pixel *)
                         width, height : INTEGER;   (* size of rectangle *)
                         ch : CHAR);                (* black pixel *)

(* Display the given rectangle using the given black pixel character.
   DVItoVDU ensures the entire rectangle is visible.
*)

VAR i, j : INTEGER;

BEGIN
ch := TeXtoASCII[ch];            (* first convert TeX ch to ASCII *)
screenh := screenh + 1;
FOR i := 1 TO height DO BEGIN    (* adding 1 to screenv here *)
   MoveAbs(screenv+i,screenh);   (* move cursor to start of next row *)
   FOR j := 1 TO width DO
      WriteChar(ch);
END;
END; (* ShowRectangle *)

(******************************************************************************)

PROCEDURE ResetVDU;

(* We assume user wants to return to 80 column mode. *)

BEGIN
WriteChar(ESC); WriteString('[?3l');
END; (* ResetVDU *)

(******************************************************************************)

PROCEDURE InitVDU;

(* The main program only calls this routine after it has parsed the command
   line and successfully opened the given DVI file.
*)

CONST
   (* On a VT100 a "pixel" is a character. *)
   screenwd = 132;  (* width of wide VT100 screen in pixels *)
   screenht = 24;   (* height of VT100 screen in pixels *)
   lineht   = 1;    (* height of one text line in pixels *)

BEGIN
(* initialize the VDU parameters *)
DVIstatusl    := 1;
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := screenht DIV lineht;   (* = number of text lines *)
windowh       := 0;
windowv       := 4 * lineht;
windowwd      := screenwd;
windowht      := screenht - (4 * lineht);

WriteChar(ESC); WriteString('[?3h');    (* 132 column mode *)
END; (* InitVDU *)
