(* VDU routines for a VIS603 terminal, by David Beard. *)

#include 'globals.h';
#include 'screenio.h';
#include 'vdu.h';
#include 'tek4010vdu.h';

VAR cursrow, curscol : INTEGER;   (* ShowChar remembers cursor location *)
    textmode : BOOLEAN;           (* are we in VT200 mode? *)

(******************************************************************************)

PROCEDURE StartText;

(* We are about to draw text in dialogue region. *)

BEGIN
WriteChar(CAN);
textmode := true;
END; (* StartText *)

(******************************************************************************)

PROCEDURE MoveAbs (row, col : INTEGER);

(* Move cursor to given screen position. *)

BEGIN
WriteChar(ESC); WriteChar('[');
WriteInt(row);
WriteChar(';');
WriteInt(col);
WriteChar('H');
END; (* MoveAbs *)

(******************************************************************************)

PROCEDURE MoveToTextLine (line : INTEGER);

(* Move current position to start of given line. *)

BEGIN
WriteChar(CAN);
textmode := true;
MoveAbs(line,1);
END; (* MoveToTextLine *)

(******************************************************************************)

PROCEDURE ClearTextLine (line : INTEGER);

(* Erase given line; note that DVItoVDU does not assume anything about the
   current position at the end of this routine.
*)

BEGIN
WriteChar(CAN);
textmode := true;
MoveAbs(line,1);
WriteChar(ESC);
WriteString('[K');   (* erase to end of line *)
END; (* ClearTextLine *)

(******************************************************************************)

PROCEDURE ClearScreen;

BEGIN
WriteChar(CAN);
textmode := true;
WriteChar(ESC);
WriteString('[2J');   (* erase all alphanumerics *)
TEK4010ClearScreen;
textmode := false;
END; (* ClearScreen *)

(******************************************************************************)

PROCEDURE StartGraphics;

(* We are about to draw in window region. *)

BEGIN
TEK4010StartGraphics;
textmode := false;
cursrow := 0;
END; (* StartGraphics *)

(******************************************************************************)

PROCEDURE LoadFont (fontname : string;
                    fontsize : INTEGER;
                    mag, hscale, vscale : REAL);

BEGIN
TEK4010LoadFont(fontname,fontsize,mag,hscale,vscale);
textmode := false;
END; (* LoadFont *)

(******************************************************************************)

PROCEDURE ShowChar (screenh, screenv : INTEGER;
                    ch : CHAR);

(* We use VT200 text mode because it is much faster. *)

VAR amount : INTEGER;

BEGIN
(* first translate DVItoVDU coordinates into actual screen location *)
if not textmode then
   begin
   WriteChar(CAN);
   textmode := true
   end;
screenh := trunc (screenh * 132 / 1024);
screenv := trunc (screenv * 24 / (780-35));

screenh := screenh + 1;
screenv := screenv + 1;
IF cursrow = screenv THEN BEGIN
   (* The cursor is on the same line as in previous ShowChar call so we only
      need to move left or right, and probably just a small amount (if at all).
   *)
   IF screenh = curscol THEN       (* cursor in correct location *)
      curscol := curscol + 1       (* cursor will move right when ch written *)
   ELSE IF screenh < curscol THEN BEGIN      (* move cursor left *)
      amount := curscol - screenh;
      WriteChar(ESC); WriteChar('[');
      IF amount > 1 THEN BEGIN               (* default is 1 col *)
         WriteInt(amount);
         curscol := curscol - (amount-1);    (* no need if amount = 1 *)
      END;
      WriteChar('D');
   END
   ELSE BEGIN                                (* move cursor right *)
      amount := screenh - curscol;
      WriteChar(ESC); WriteChar('[');
      IF amount > 1 THEN WriteInt(amount);   (* default is 1 col *)
      curscol := curscol + (amount+1);
      WriteChar('C');
   END;
END
ELSE BEGIN                         (* cursrow undefined or ch on a new line *)
   MoveAbs(screenv,screenh);
   cursrow := screenv;
   curscol := screenh + 1;         (* cursor will move right when ch written *)
END;
IF screenh = windowwd THEN         (* ch will be written at right edge *)
   cursrow := 0;                   (* so avoid auto wrap next time around *)
WriteChar(TeXtoASCII[ch]);
END; (* ShowChar *)

(******************************************************************************)

PROCEDURE ShowRectangle (screenh, screenv,          (* top left pixel *)
                         width, height : INTEGER;   (* size of rectangle *)
                         ch : CHAR);                (* black pixel *)

(* Display the given rectangle. *)

VAR pos : INTEGER;

BEGIN
textmode := false;
IF height = 1 THEN BEGIN            (* show row vector *)
   pos := 779 - screenv;
   WriteChar(GS);
   SendXY(screenh,pos);             (* move cursor to start of row *)
   SendXY(screenh+width-1,pos);     (* draw vector to end of row *)
END
ELSE IF width = 1 THEN BEGIN        (* show column vector *)
   pos := 779 - screenv;
   WriteChar(GS);
   SendXY(screenh,pos);             (* move cursor to start of column *)
   SendXY(screenh,pos-height+1);    (* draw vector to end of column *)
END
ELSE BEGIN
   (* assume height and width > 1; draw and fill rectangle *)
   pos := 779 - (screenv+height-1);
   WriteChar(ESC);         WriteChar('/');
   WriteInt(screenh);      WriteChar(';');   (* left *)
   WriteInt(pos);          WriteChar(';');   (* bottom *)
   WriteInt(width-1);      WriteChar(';');
   WriteInt(height+1);     WriteChar('y');
   (* Note that there are a few problems with this command:
      - we need to subtract 1 from width.  While this prevents exceeding the
        right edge (reason unknown), it causes missing pixel columns.
      - we need to ADD 1 to height to avoid missing pixel rows.
      - the smallest rectangle drawn is 2 by 2.
      - the new cursor position is undefined.
      IS THIS TRUE FOR VIS 603???
      These funnies are outweighed by the improved efficiency in drawing large
      rectangles.
   *)
   havesentxy := FALSE;   (* need to re-synch cursor position *)
END;
END; (* ShowRectangle *)

(******************************************************************************)

PROCEDURE ResetVDU;

BEGIN
WriteChar(ESC); WriteString('[?50h');   (* 80 COL ITAG + graphics *)
WriteChar(CAN);
textmode := true;
END; (* ResetVDU *)

(******************************************************************************)

PROCEDURE InitVDU;

(* The dialog region will be the top 4 text lines in VT200 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 24 in VT200 mode.
*)

BEGIN
InitTEK4010VDU;
DVIstatusl    := 1;      (* DVItoVDU assumes top text line = 1 *)
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := 24;     (* also number of text lines on screen *)
(* The above values assume the VIS603 is in VT200 mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
windowv  := 125;         (* approx. height in TEK4010 pixels of 4 text lines
                            i.e. ~ 4 * 780/25 *)
windowh  := 0;
windowht := 780-windowv-35;   (* avoid drawing in status line *)
windowwd := 1024;

WriteChar(GS);
WriteChar(ESC);
WriteChar('@');          (* solid fill for rectangular draw and fill *)
WriteChar(ESC);
WriteString('[?40h');    (* 132 COL ITAG + graphics *)
WriteChar(CAN);
textmode := true;
END; (* InitVDU *)
