
IMPLEMENTATION MODULE VTTekVDU;

(*
   Author:         Christopher P. Andrasic
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   February, 1987

   Description:
   Implements the InitVTTek procedure that initializes the terminal dependent
   routines and parameters used in DVItoVDU.
   The VT200 VDU has a Transparent mode in which it behaves like a VT100, but
   it can also emulate a Tektronix 4010/14 terminal in which the screen is
   1024 pixels wide by 780 pixels high.
   (The actual resolution of the VT200 screen is 800 by 260 pixels;
   the appropriate scaling is done automatically.)

   The bottom left pixel is the point (x=0,y=0). Note that x coordinates
   increase to the right and y coordinates increase up the screen.
   DVItoVDU uses a coordinate scheme in which horizontal (=h) coordinates
   also increase to the right but vertical (=v) coordinates increase DOWN the
   screen, i.e. the top left pixel on the screen is the point (h=0,v=0).
   This means that the Tektronix 4010 routines will have to do a
   simple translation of the vertical coordinates passed by DVItoVDU.
*)

  FROM VDUInterface IMPORT 
   (* PROCEDURE variables: *)
                           MoveToTextLine,
                           ClearTextLine,
                           ClearScreen,
                           StartText,
                           StartGraphics,
                           LoadFont,
                           ShowChar,
                           ShowRectangle,
                           ResetVDU,
   (* INTEGER variables: *)
                           DVIstatusl,
                           windowstatusl,
                           messagel,
                           commandl,
                           bottoml,
                           windowv,
                           windowh,
                           windowht,
                           windowwd;

  FROM TEK4010VDU IMPORT 
                         TEK4010ClearScreen,
                         TEK4010StartGraphics,
                         TEK4010LoadFont,
                         TEK4010ShowChar,
                         TEK4010MoveToTextLine,
                         TEK4010ShowRectangle;

  FROM ScreenIO IMPORT 
                       Write, WriteCard,
                       WriteBuffer,WriteString;

  CONST
    CR  =  15C;
    CAN =  30C;
    ESC =  33C;
    GS  =  35C;
    SP  =  40C;
    VT  =  13C;
    BS  =  10C;
    LF  =  12C;
    US  =  27C;
    FF  =  14C;

  VAR
    curcol : CARDINAL;


(**********************************************************************)

  PROCEDURE VTTekMoveToTextLine( line : CARDINAL);

  BEGIN
    TEK4010MoveToTextLine(line);
    Write(ESC);
    WriteString('8');
  END VTTekMoveToTextLine;

(**********************************************************************)

  PROCEDURE MoveAbs (row, col : CARDINAL);

(* Assuming VTTek is in VT100 mode, we move the cursor to the
   given position.  VT100 numbers text lines the same way as DVItoVDU.
*)

    VAR
      i, movcol : CARDINAL;

  BEGIN
    TEK4010MoveToTextLine(row);
    Write(ESC);
    WriteString('8');
    movcol := curcol-col;
    IF movcol > 0 THEN
      FOR i:=1 TO col DO
        Write(SP)
      END;
    END;
    IF movcol < 0 THEN
      FOR i:=1 TO col DO
        Write(BS)
      END;
    END;
    curcol := col;

  END MoveAbs;

(**********************************************************************)

  PROCEDURE VTTekClearTextLine (line : CARDINAL);

    VAR
      i : CARDINAL;

  BEGIN
    VTTekMoveToTextLine(line);
    Write(US);
    Write(ESC);
    WriteString('8');
    FOR i:=1 TO 73 DO
      Write(BS);
      Write(SP);
      Write(SP);
    END;
  END VTTekClearTextLine;

(**********************************************************************)

  PROCEDURE VTTekStartText;

  BEGIN
    Write(US);
    Write(ESC);
    WriteString('8');
  END VTTekStartText;

(**********************************************************************)

  PROCEDURE VTTekStartGraphics;

  BEGIN
    Write(GS);
  END VTTekStartGraphics;

(**********************************************************************)

  PROCEDURE VTTekClearScreen;

  BEGIN
    Write(GS);
    Write(ESC);
    Write(FF);
    Write(ESC);
    WriteString('8');
  END VTTekClearScreen;

(**********************************************************************)

  PROCEDURE VTTekResetVDU;

  BEGIN
    Write(ESC);
    WriteString('[?38l');
    WriteString(' ');
  END VTTekResetVDU;

(**********************************************************************)

  PROCEDURE InitVTTek;

(* The dialog region will be the top 4 text lines in VT100 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 24 in VT100 mode.
*)

  BEGIN
    curcol := 1;
    DVIstatusl := 1;      (* DVItoVDU assumes top text line = 1 *)
    windowstatusl := 2;
    messagel := 3;
    commandl := 4;
    bottoml := 24;        (* also number of text lines on screen *)
(* The above values assume the VTTek is in VT100 mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
    windowv := 130;       (* approx. height in TEK4010 pixels of 4 text lines
                         i.e. 4 * 780/24 *)
    windowh := 0;
    windowht := 780-windowv;
    windowwd := 1024;

    MoveToTextLine := VTTekMoveToTextLine;
    ClearTextLine  := VTTekClearTextLine;
    ClearScreen    := VTTekClearScreen;
    StartText      := VTTekStartText;
    StartGraphics  := VTTekStartGraphics;
    LoadFont       := TEK4010LoadFont;
    ShowChar       := TEK4010ShowChar;
    ShowRectangle  := TEK4010ShowRectangle;
    ResetVDU       := VTTekResetVDU;

    Write(ESC);
    WriteString('[?38h');

  END InitVTTek;

(**********************************************************************)

BEGIN
END VTTekVDU.
