/*	DVIAPOLLO -- Apollo TeX DVI File Display Driver using GPR	     */
/*									     */
/*	Copyright (C) 1987 by Leonard N. Zubkoff, All Rights Reserved	     */
/*									     */
/*	This software is provided free and without any warranty.	     */
/*	Permission to copy for any purpose is hereby granted so		     */
/*	long as this copyright notice remains intact.			     */
/*									     */
/*	Revision:	 2-Jun-88 21:57:19				     */


#define begin	    {
#define end	    }
#define then
#define do
#define hidden	    static
#define visible
#define procedure   void


#include "header.h"
#include "genwindow.h"
#include <sys/file.h>


#include "/sys/ins/base.ins.c"
#include "/sys/ins/gpr.ins.c"
#include "/sys/ins/ios.ins.c"
#include "/sys/ins/kbd.ins.c"
#include "/sys/ins/pad.ins.c"


#define MaxFontID		64
#define MaxTextObjectsPerPage	10000
#define MaxBoxObjectsPerPage	10000
#define MaxCharactersPerPage	10000
#define AutomaticXoffset	-32768
#define VisibleListSize		64
#define NIL			0L


hidden DVIwindow
	CurrentDVIwindow;

hidden boolean
	ScreenUnobscured;


hidden short
	Xoffset,
	Yoffset,
	ScreenVisibleCount,
	ScreenFontID = -1,
	CurrentPage = 0,
	FontCharacterWidths[MaxFontID][128];

hidden int
	TextObjectCount,
	BoxObjectCount,
	CharacterCount,
	MinPageX;


hidden struct
    begin
	short Start;
	short Count;
	short X;
	short Y;
	short FontID;
	short Xlimit;
    end
	TextObjects[MaxTextObjectsPerPage];


hidden struct
    begin
	short X;
	short Y;
	short Width;
	short Height;
    end
	BoxObjects[MaxBoxObjectsPerPage];


hidden char
	Characters[MaxCharactersPerPage];


hidden gpr_$pixel_value_t
	Background,
	Foreground;


hidden gpr_$bitmap_desc_t
	ScreenBitmap;


hidden gpr_$offset_t
	ScreenBitmapSize;


hidden gpr_$window_t
	ScreenVisibleList[VisibleListSize];

visible procedure ShowError(Message)
	char *Message;
    begin
	status_$t Status;
	short AcquireCount;
	gpr_$force_release(AcquireCount,Status);
	fprintf(stderr,"dviapollo: %s\n",Message);
	exit(1);
    end;


hidden procedure AcquireDisplay()
    begin
	status_$t Status;
	ScreenUnobscured = gpr_$acquire_display(Status);
	if (!ScreenUnobscured) then
	    gpr_$inq_vis_list(VisibleListSize,ScreenVisibleCount,
			      ScreenVisibleList,Status);
	else ScreenVisibleCount = 1;
    end


hidden procedure ReleaseDisplay()
    begin
	static gpr_$window_t FullClipWindow =
	    { 0, 0, gpr_$max_x_size, gpr_$max_y_size };
	status_$t Status;
	if (!ScreenUnobscured) then
	    gpr_$set_clip_window(FullClipWindow,Status);
	gpr_$release_display(Status);
    end


visible procedure ClearScreen()
    begin
	status_$t Status;
	int i;
	AcquireDisplay();
	for (i=0; i<ScreenVisibleCount; i++) do
	    begin
		if (!ScreenUnobscured) then
		    gpr_$set_clip_window(ScreenVisibleList[i],Status);
		gpr_$clear(Background,Status);
	    end;
	ReleaseDisplay();
    end


visible procedure BeginPageDisplay()
    begin
	TextObjectCount = 0;
	BoxObjectCount = 0;
	CharacterCount = 0;
	MinPageX = 999999;
    end;

visible procedure EndPageDisplay()
    begin
	gpr_$position_t WindowOrigin;
	gpr_$window_t Rectangle;
	status_$t Status;
	int Index, i;
	AcquireDisplay();
	if (Xoffset == AutomaticXoffset) then
	    Xoffset = MinPageX-20;
	WindowOrigin.x_coord = -Xoffset;
	WindowOrigin.y_coord = -Yoffset;
	gpr_$set_coordinate_origin(WindowOrigin,Status);
	for (Index=0; Index<BoxObjectCount; Index++) do
	    begin
		Rectangle.window_base.x_coord = BoxObjects[Index].X;
		Rectangle.window_base.y_coord = BoxObjects[Index].Y;
		Rectangle.window_size.x_size = BoxObjects[Index].Width;
		Rectangle.window_size.y_size = BoxObjects[Index].Height;
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],Status);
			gpr_$rectangle(Rectangle,Status);
		    end;
	    end;
	for (Index=0; Index<TextObjectCount; Index++) do
	    begin
		short Start = TextObjects[Index].Start;
		short Count = TextObjects[Index].Count;
		short X = TextObjects[Index].X;
		short Y = TextObjects[Index].Y;
		short FontID = TextObjects[Index].FontID;
		if (ScreenFontID != FontID) then
		    begin
			gpr_$set_text_font(FontID,Status);
			ScreenFontID = FontID;
		    end;
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],Status);
			gpr_$move(X,Y,Status);
			gpr_$text(Characters[Start],Count,Status);
		    end;
	    end;
	WindowOrigin.x_coord = 0;
	WindowOrigin.y_coord = 0;
	gpr_$set_coordinate_origin(WindowOrigin,Status);
	ReleaseDisplay();
    end;

visible procedure DrawBox(X,Y,Width,Height)
	int X, Y, Width, Height;
    begin
	if (X < MinPageX) then MinPageX = X;
	BoxObjects[BoxObjectCount].X = X;
	BoxObjects[BoxObjectCount].Y = Y;
	BoxObjects[BoxObjectCount].Width = Width;
	BoxObjects[BoxObjectCount].Height = Height;
	BoxObjectCount++;
	if (BoxObjectCount == MaxBoxObjectsPerPage) then
	    ShowError("too many box objects on page");
    end


visible procedure DrawCharacter(Character,X,Y,FontID)
	char Character;
	short X, Y, FontID;
    begin
	status_$t Status;
	if (X < MinPageX) then MinPageX = X;
	if (TextObjectCount > 0
		&& X == TextObjects[TextObjectCount-1].Xlimit
		&& Y == TextObjects[TextObjectCount-1].Y
		&& FontID == TextObjects[TextObjectCount-1].FontID) then
	    begin
		TextObjects[TextObjectCount-1].Count++;
		TextObjects[TextObjectCount-1].Xlimit +=
		    FontCharacterWidths[FontID][Character];
	    end
	else
	    begin
		TextObjects[TextObjectCount].Start = CharacterCount;
		TextObjects[TextObjectCount].Count = 1;
		TextObjects[TextObjectCount].X = X;
		TextObjects[TextObjectCount].Y = Y;
		TextObjects[TextObjectCount].FontID = FontID;
		TextObjects[TextObjectCount].Xlimit =
		    X+FontCharacterWidths[FontID][Character];
		TextObjectCount++;
		if (TextObjectCount == MaxTextObjectsPerPage) then
		    ShowError("too many text objects on page");
	    end;
	Characters[CharacterCount] = Character;
	CharacterCount++;
	if (CharacterCount == MaxCharactersPerPage) then
	    ShowError("too many characters on page");
    end

visible short LoadFontFile(FontFileName)
	char *FontFileName;
    begin
	status_$t Status;
	short FontID, HorizontalSpacing, CharacterWidth;
	unsigned char Character;
	gpr_$load_font_file(*FontFileName,(short)strlen(FontFileName),
			    FontID,Status);
	if (Status.all != status_$ok) then
	    begin
		fprintf(stderr,"dviapollo: cannot load font %s\n",
			FontFileName);
		exit(1);
	    end;
	if (FontID >= MaxFontID) then
	    ShowError("Font ID too large");
	gpr_$inq_horizontal_spacing(FontID,HorizontalSpacing,Status);
	for (Character=0; Character<128; Character++) do
	    begin
		gpr_$inq_character_width(FontID,Character,CharacterWidth,Status);
		FontCharacterWidths[FontID][Character] =
		    CharacterWidth+HorizontalSpacing;
	    end;
	return FontID;
    end


visible procedure ClearError()
    begin
    end


hidden procedure ResetPageOffsets()
    begin
	Xoffset = AutomaticXoffset;
	Yoffset = 25;
    end


hidden procedure RedisplayScreen()
    begin
	CurrentPage = ShowDviPage(&CurrentDVIwindow,CurrentPage,0,0);
    end


hidden procedure RefreshProcedure(Unobscured,PositionChanged)
	boolean *Unobscured, *PositionChanged;
    begin
	RedisplayScreen();
    end;

hidden procedure CommandLoop()
    begin
	gpr_$event_t EventType;
	unsigned char EventData[1];
	gpr_$position_t EventPosition;
	status_$t Status;
	int Argument = 0;
	while (true) do
	    begin
		gpr_$event_wait(EventType,EventData[0],EventPosition,Status);
		switch (EventData[0])
		    begin
		case KBD_$EXIT:		    return;
		case KBD_$NEXT_WIN:	    ResetPageOffsets();
					    if (Argument > 0) then
						CurrentPage = Argument;
					    break;
		case KBD_$UP_BOX_ARROW2:    ResetPageOffsets();
					    if (Argument == 0) then CurrentPage--;
					    else CurrentPage -= Argument;
					    break;
		case KBD_$DOWN_BOX_ARROW2:  ResetPageOffsets();
					    if (Argument == 0) then CurrentPage++;
					    else CurrentPage += Argument;
					    break;
		case KBD_$UP_ARROW:	    Yoffset = -123;
					    break;
		case KBD_$DOWN_ARROW:	    Yoffset =
						11*123-ScreenBitmapSize.y_size;
					    break;
		case KBD_$LEFT_ARROW:	    Xoffset = -123;
					    break;
		case KBD_$RIGHT_ARROW:	    Xoffset =
						8*123+123/2-ScreenBitmapSize.x_size;
					    break;
		/* Shift UP BOX ARROW */
		case KBD_$LDS:		    ResetPageOffsets();
					    CurrentPage = 1;
					    break;
		/* Shift DOWN BOX ARROW */
		case KBD_$LFS:		    ResetPageOffsets();
					    CurrentPage = 999999;
					    break;
		case '0': case '1': case '2': case '3':
		case '4': case '5': case '6': case '7':
		case '8': case '9':	    Argument *= 10;
					    Argument += EventData[0]-'0';
					    break;
		    end;
		if (EventData[0] < '0' || EventData[0] > '9') then
		    begin
			Argument = 0;
			RedisplayScreen();
		    end;
	    end;
    end;

visible procedure main(ArgCount,ArgVector,Environment)
	int ArgCount;
	char *ArgVector[], *Environment[];
    begin
	static pad_$window_desc_t Window = { 0, 0, 800, 1024 };
	ios_$id_t StreamID;
	gpr_$plane_t HiPlane;
	gpr_$keyset_t KeySet;
	status_$t Status;
	boolean ColorDisplay;
	char InputFileName[256];
	short InputFileNameLength, Keystroke;
	int InitialPageNumber = 1;
	if (ArgCount < 2 || ArgCount > 3) then
	    begin
		fprintf(stderr,"Usage: dviapollo dvifilename [pagenumber]\n");
		exit(1);
	    end;
	strcpy(InputFileName,ArgVector[1]);
	InputFileNameLength = strlen(InputFileName);
	if (InputFileNameLength < 5
		|| strcmp(&InputFileName[InputFileNameLength-4],".dvi") != 0) then
	    strcat(InputFileName,".dvi");
	if ((AAFpath=getenv("TEX118AAF")) == NULL) then
	    AAFpath = FONTAREA;
	if (access(InputFileName,R_OK) == -1) then
	    begin
		fprintf(stderr,"dviapollo: cannot access %s\n",InputFileName);
		exit(1);
	    end;
	if (ArgCount == 3) then
	    InitialPageNumber = atoi(ArgVector[2]);

	pad_$create_window("",0,pad_$transcript,1,Window,StreamID,Status);
	pad_$set_border(StreamID,1,false,Status);
	pad_$set_auto_close(StreamID,1,true,Status);
	ScreenBitmapSize.x_size = gpr_$max_x_size;
	ScreenBitmapSize.y_size = gpr_$max_y_size;
	gpr_$init(gpr_$direct,StreamID,ScreenBitmapSize,
		  gpr_$highest_plane,ScreenBitmap,Status);
	gpr_$inq_bitmap_dimensions(ScreenBitmap,ScreenBitmapSize,HiPlane,Status);
	gpr_$set_obscured_opt(gpr_$input_ok_if_obs,Status);
	gpr_$set_refresh_entry(RefreshProcedure,NIL,Status);
	ColorDisplay = (HiPlane > 0);
	Background = (ColorDisplay ? gpr_$background : 0);
	Foreground = (ColorDisplay ? 0 : 1);
	gpr_$set_text_value(Foreground,Status);
	gpr_$set_text_background_value(gpr_$transparent,Status);
	gpr_$set_fill_value(Foreground,Status);
	lib_$init_set(KeySet,256);
	lib_$add_to_set(KeySet,256,KBD_$EXIT);
	lib_$add_to_set(KeySet,256,KBD_$NEXT_WIN);
	lib_$add_to_set(KeySet,256,KBD_$UP_BOX_ARROW2);
	lib_$add_to_set(KeySet,256,KBD_$DOWN_BOX_ARROW2);
	lib_$add_to_set(KeySet,256,KBD_$UP_ARROW);
	lib_$add_to_set(KeySet,256,KBD_$DOWN_ARROW);
	lib_$add_to_set(KeySet,256,KBD_$LEFT_ARROW);
	lib_$add_to_set(KeySet,256,KBD_$RIGHT_ARROW);
	lib_$add_to_set(KeySet,256,KBD_$LDS);
	lib_$add_to_set(KeySet,256,KBD_$LFS);
	for (Keystroke='0'; Keystroke<='9'; Keystroke++) do
	    lib_$add_to_set(KeySet,256,Keystroke);
	gpr_$enable_input(gpr_$keystroke,KeySet,Status);
	ResetPageOffsets();
	CurrentDVIwindow = OpenDvi(InputFileName);
	if (InitialPageNumber >= 0) then
	    begin
		CurrentPage = InitialPageNumber;
		RedisplayScreen();
		CommandLoop();
	    end
	else
	    begin
		InitialPageNumber = 0;
		CurrentPage = 0;
		while (CurrentPage == InitialPageNumber) do
		    begin
			CurrentPage++;
			InitialPageNumber = CurrentPage;
			RedisplayScreen();
		    end;
	    end;
	gpr_$terminate(false,Status);
	ios_$close(StreamID,Status);
    end
