// Python interface to TeXtfm
// $Id: tfmpy.cc,v 1.4 2000/11/21 22:38:43 yotam Exp $

#include <Python.h>
#include "TeXtfm.h"
#include "pygoods.h"
#include "debug.h"


////////////////////////////////////////////////////////////////////////
static void  sDeleteTFM(void* pv)
{  DHERE;
   TeXtfm*  pTFM = static_cast<TeXtfm*>(pv);
   delete pTFM;
} // sDeleteTFM

////////////////////////////////////////////////////////////////////////
static PyObject*  tfmOpen(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* ret;
   char*  fn;
   if (rPyArgParse(ret, args, "s", &fn))
   {
      TeXtfm*  pTeXtfm = new TeXtfm(fn);
      ret = PyCObject_FromVoidPtr(pTeXtfm, sDeleteTFM);
   }
   return ret;
} // tfmOpen


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmOK(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* ret;
   PyObject* obj;
   if (rPyArgParse(ret, args, "O", &obj))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      bool     active = pTeXtfm->active();
      ret = PyInt_FromLong((int)active);
   }
   return ret;
} // tfmOK


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmLengths(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   PyObject* tuple; 
   if (rPyArgParse(tuple, args, "O", &obj))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      tuple = PyTuple_New(TeXtfm::nLen);
#if 0
      dict = PyDict_New();
      static const struct {const char* name; unsigned ei;} k[TeXtfm::nLen] = {
         {"lf", TeXtfm::lf},
         {"lh", TeXtfm::lh},
         {"bc", TeXtfm::bc},
         {"ec", TeXtfm::ec},
         {"nw", TeXtfm::nw},
         {"nh", TeXtfm::nh},
         {"nd", TeXtfm::nd},
         {"ni", TeXtfm::ni},
         {"nl", TeXtfm::nl},
         {"nk", TeXtfm::nk},
         {"ne", TeXtfm::ne},
         {"np", TeXtfm::np}
      };
      for (unsigned i = 0;  i != TeXtfm::nLen;  ++i)
      {
         unsigned  len = pTeXtfm->length(k[i].ei);
         PyDict_SetItemString(dict, (char*)k[i].name, PyInt_FromLong(len));
      }
#endif
      for (unsigned i = 0;  i != TeXtfm::nLen;  ++i)
      {
         unsigned  len = pTeXtfm->length(i);
         PyTuple_SetItem(tuple, i, PyInt_FromLong(len));
      }
   }
   return tuple;
} // tfmLengths


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmHeader(PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* ret;
   if (rPyArgParse(ret, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      US32     h = pTeXtfm->header((US16)i);
      DPR0("i=" << i << ", h=" << h);
      ret = PyInt_FromLong(h);
   }
   return ret;
} // tfmHeader


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmHdrDesignSize(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   PyObject* ret;
   if (rPyArgParse(ret, args, "O", &obj))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      US32     ds = pTeXtfm->hDesignSize();
      ret = PyInt_FromLong(ds);
   }
   return ret;
} // tfmHdrDesignSize


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmDesignSize(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   PyObject* ret;
   if (rPyArgParse(ret, args, "O", &obj))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      double   ds = pTeXtfm->designSize();
      ret = PyFloat_FromDouble(ds);
   }
   return ret;
} // tfmDesignSize


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmExtraHeader(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   PyObject* ret;
   int       replaceChar;
   if (rPyArgParse(ret, args, "Oi", &obj, &replaceChar))
   {
      TeXtfm*      pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      char         c = (char)(replaceChar & 0x7f);
      string       seh = pTeXtfm->extraHeader(c);
      const char*  eh = seh.c_str();
      ret = PyString_FromString(eh);
   }
   return ret;
} // tfmExtraHeader


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharInfo(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   int       c;
   PyObject* dict;
   if (rPyArgParse(dict, args, "Oi", &obj, &c))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      US8      c8 = c;
      TeXtfm::CharInfo  charInfo = pTeXtfm->char_info(c8);
      dict = PyDict_New();
      PyDict_SetItemString(dict, "width_index",
                           PyInt_FromLong(charInfo.width_index()));
      PyDict_SetItemString(dict, "height_index",
                           PyInt_FromLong(charInfo.height_index()));
      PyDict_SetItemString(dict, "depth_index",
                           PyInt_FromLong(charInfo.depth_index()));
      PyDict_SetItemString(dict, "italic_index",
                           PyInt_FromLong(charInfo.italic_index()));
      PyDict_SetItemString(dict, "tag",
                           PyInt_FromLong(charInfo.tag()));
      PyDict_SetItemString(dict, "reminder",
                           PyInt_FromLong(charInfo.reminder()));
   }
   return dict;
} // tfmCharInfo


////////////////////////////////////////////////////////////////////////
enum {eWidth, eHeight, eDepth, eItalic, eKern};


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmDimension(int eDim, PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* ret;
   if (rPyArgParse(ret, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      long     l32 = 0;
      switch (eDim)
      {
       case eWidth:
         l32 = pTeXtfm->width(i);
         break;
       case eHeight:
         l32 = pTeXtfm->height(i);
         break;
       case eDepth:
         l32 = pTeXtfm->depth(i);
         break;
       case eItalic:
         l32 = pTeXtfm->italic(i);
         break;
       case eKern:
         l32 = pTeXtfm->kern(i);
         break;
      }
      ret = PyInt_FromLong(l32);
   }
   return ret;
} // tfmDimension


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmWidth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmDimension(eWidth, self, args));
} // tfmWidth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmHeight(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmDimension(eHeight, self, args));
} // tfmHeight


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmDepth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmDimension(eDepth, self, args));
} // tfmDepth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmItalic(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmDimension(eItalic, self, args));
} // tfmItalic


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharDimension(int eDim, PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       c;
   PyObject* ret;
   if (rPyArgParse(ret, args, "Oi", &obj, &c))
   {
      TeXtfm*          pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      US8              c8 = c;
      switch (eDim)
      {
       case eWidth:
         ret = PyInt_FromLong(pTeXtfm->cWidth(c8));
         break;
       case eHeight:
         ret = PyInt_FromLong(pTeXtfm->cHeight(c8));
         break;
       case eDepth:
         ret = PyInt_FromLong(pTeXtfm->cDepth(c8));
         break;
       case eItalic:
         ret = PyInt_FromLong(pTeXtfm->cItalic(c8));
         break;
      }
   }
   return ret;
} // tfmCharDimension


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharWidth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmCharDimension(eWidth, self, args));
} // tfmCharWidth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharHeight(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmCharDimension(eHeight, self, args));
} // tfmCharHeight


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharDepth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmCharDimension(eDepth, self, args));
} // tfmCharDepth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmCharItalic(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmCharDimension(eItalic, self, args));
} // tfmCharItalic


////////////////////////////////////////////////////////////////////////
PyObject*  tfmMaxDimension(int eDim, PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   PyObject* ret;
   if (rPyArgParse(ret, args, "O", &obj))
   {
      TeXtfm*          pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      switch (eDim)
      {
       case eWidth:
         ret = PyInt_FromLong(pTeXtfm->maxWidth());
         break;
       case eHeight:
         ret = PyInt_FromLong(pTeXtfm->maxHeight());
         break;
       case eDepth:
         ret = PyInt_FromLong(pTeXtfm->maxDepth());
         break;
       case eItalic:
         ret = PyInt_FromLong(pTeXtfm->maxItalic());
         break;
      }
   }
   return ret;
} // tfmMaxDimension


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmMaxWidth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmMaxDimension(eWidth, self, args));
} // tfmMaxWidth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmMaxHeight(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmMaxDimension(eHeight, self, args));
} // tfmMaxHeight


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmMaxDepth(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmMaxDimension(eDepth, self, args));
} // tfmMaxDepth


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmMaxItalic(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmMaxDimension(eItalic, self, args));
} // tfmMaxItalic


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmListAscend(PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* ret;
   if (rPyArgParse(ret, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      ret = PyInt_FromLong(pTeXtfm->listAscend(i));
   }
   return ret;
} // tfmListAscend


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmLigKernOwners(PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* ownersTuple;
   if (rPyArgParse(ownersTuple, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      TeXtfm::OwnerIter  b, e;
      pTeXtfm->ligKernOwners(b, e, i);
      unsigned nOwners = e - b;
      ownersTuple = PyTuple_New(nOwners);
      for (unsigned  oi = 0;  oi != nOwners;  ++oi, ++b)
      {
         PyTuple_SetItem(ownersTuple, oi, PyInt_FromLong(*b));
      }
   }
   return ownersTuple;
} // tfmLigKernOwners


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmLigKern(PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* tuple;
   if (rPyArgParse(tuple, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      TeXtfm::LigKern  ligKern = pTeXtfm->lig_kern(i);
      tuple = PyTuple_New(4);
      for (unsigned bi = 0;  bi != 4;  ++bi)
      {
         PyTuple_SetItem(tuple, bi, PyInt_FromLong(ligKern.byte(bi)));
      }
   }
   return tuple;
} // tfmLigKern


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmKern(PyObject* self, PyObject* args)
{  DHERE0;
   return(tfmDimension(eKern, self, args));
#if 0
   PyObject* obj;
   int       i;
   PyObject* ret;  DHERE;
   if (rPyArgParse(ret, args, "Oi", &obj, &i))
   {  DHERE;
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj)); DHERE;
      Int32    i32 = pTeXtfm->kern(i);
      DPR("i=" << i << ", i32=" << i32);
      ret = PyInt_FromLong(i32);
   } DHERE;
   return ret;
#endif
} // tfmKern


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmExten(PyObject* self, PyObject* args)
{  DHERE0;
   PyObject* obj;
   int       i;
   PyObject* tuple;
   if (rPyArgParse(tuple, args, "Oi", &obj, &i))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      TeXtfm::ExtensibleRecipe  e = pTeXtfm->exten(i);
      tuple = PyTuple_New(4);
      for (unsigned  bi = 0;  bi != 4;  ++bi)
      {
         PyTuple_SetItem(tuple, bi, PyInt_FromLong(e.byte(bi)));
      }
   }
   return tuple;
} // tfmExten


////////////////////////////////////////////////////////////////////////
static PyObject*  tfmParams(PyObject* self, PyObject* args)
{  DHERE;
   PyObject* obj;
   PyObject* dict;
   if (rPyArgParse(dict, args, "O", &obj))
   {
      TeXtfm*  pTeXtfm = static_cast<TeXtfm*>(PyCObject_AsVoidPtr(obj));
      dict = PyDict_New();
      PyDict_SetItemString(dict, "slant_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         slant_code)));
      PyDict_SetItemString(dict, "space_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         space_code)));
      PyDict_SetItemString(dict, "space_stretch_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         space_stretch_code)));
      PyDict_SetItemString(dict, "space_shrink_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         space_shrink_code)));
      PyDict_SetItemString(dict, "x_height_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         x_height_code)));
      PyDict_SetItemString(dict, "quad_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         quad_code)));
      PyDict_SetItemString(dict, "extra_space_code",
                           PyInt_FromLong(pTeXtfm->param(TeXtfm::
                                                         extra_space_code)));
   }
   return dict;
} // tfmParams


////////////////////////////////////////////////////////////////////////
static PyMethodDef
       tfm_methods[] =
       {
          {"open",           (PyCFunction)&tfmOpen,          METH_VARARGS},
          {"ok",             (PyCFunction)&tfmOK,            METH_VARARGS},
          {"lengths",        (PyCFunction)&tfmLengths,       METH_VARARGS},
          {"header",         (PyCFunction)&tfmHeader,        METH_VARARGS},
          {"headdesignsize", (PyCFunction)&tfmHdrDesignSize, METH_VARARGS},
          {"designsize",     (PyCFunction)&tfmDesignSize,    METH_VARARGS},
          {"extraheader",    (PyCFunction)&tfmExtraHeader,   METH_VARARGS},
          {"charinfo",       (PyCFunction)&tfmCharInfo,      METH_VARARGS},

          {"width",          (PyCFunction)&tfmWidth,         METH_VARARGS},
          {"height",         (PyCFunction)&tfmHeight,        METH_VARARGS},
          {"depth",          (PyCFunction)&tfmDepth,         METH_VARARGS},
          {"italic",         (PyCFunction)&tfmItalic,        METH_VARARGS},

          {"cwidth",         (PyCFunction)&tfmCharWidth,     METH_VARARGS},
          {"cheight",        (PyCFunction)&tfmCharHeight,    METH_VARARGS},
          {"cdepth",         (PyCFunction)&tfmCharDepth,     METH_VARARGS},
          {"citalic",        (PyCFunction)&tfmCharItalic,    METH_VARARGS},

          {"maxwidth",       (PyCFunction)&tfmMaxWidth,      METH_VARARGS},
          {"maxheight",      (PyCFunction)&tfmMaxHeight,     METH_VARARGS},
          {"maxdepth",       (PyCFunction)&tfmMaxDepth,      METH_VARARGS},
          {"maxitalic",      (PyCFunction)&tfmMaxItalic,     METH_VARARGS},

          {"listascend",     (PyCFunction)&tfmListAscend,    METH_VARARGS},
          {"ligkernowners",  (PyCFunction)&tfmLigKernOwners, METH_VARARGS},
          {"ligkern",        (PyCFunction)&tfmLigKern,       METH_VARARGS},
          {"kern",           (PyCFunction)&tfmKern,          METH_VARARGS},
          {"exten",          (PyCFunction)&tfmExten,         METH_VARARGS},
          {"params",         (PyCFunction)&tfmParams,        METH_VARARGS},
          {0, 0}
       };

////////////////////////////////////////////////////////////////////////
PyMethodDef* tfmMethods()
{
   return tfm_methods;
} // tfmMethods

////////////////////////////////////////////////////////////////////////
extern "C" void init_tfm()
{
   DHERE;
   Py_InitModule("_tfm", tfm_methods);
   DHERE;
} // init_tfm
