import java.awt.*;
import java.awt.event.*;
import java.util.Hashtable;
import java.util.Enumeration;
import java.util.StringTokenizer;

/**
 EditDialog is a dialog window for editing one record.  The
 first part of the dialog pertains to getting the critical
 information, such as the reference type and the key.  Once
 these have been selected, then the correct fields are put
 in place.
 */

public class EditPanel extends Panel implements ActionListener, ItemListener, TextListener {
    private Hashtable m_Required, m_Optional, m_Extra;  // storage for parts of display
    private Hashtable m_ReqList, m_OptList, m_HelpList;  // storage for fields of each type
    private Hashtable m_Holding;  // storage for data in transition
    private String m_Type, m_Key, m_CrossRef, m_Preamble;  // storage for record details
    private static boolean m_Debug;  // control whether or not to print debug messages
    // AWT objects for holding important information
    Label m_HelpLabel;  // label at bottom of top panel, with help
    Choice m_TypeChoice;  // holder of type choice
    List m_ExtraList;  // holder for different extra fields
    TextField m_CrossRefField;  // holder for crossreference
    TextField m_KeyField;  // holder for key field
    TextField m_CustomField;  // holder for other field
    TextField m_ExtraField;  // holder for new name of extra field
    Button m_ExtraButton;  // button to request adding extra field
    TextArea m_ExtraArea;  // place holder on screen for text area
    ScrollPane m_RecordPane;  // scroll pane for record data
    Panel m_RecordPanel;  // panel to hold main records
    Object m_RecordList;  // reference to list or records
    Object m_Record;  // reference to record being
    String m_EOLN;  // storage for end of line in this system
    Hashtable m_CrossRefHash; // storage for crossreference
    BibTeXRecord m_InitialRec; // storage for original record
    /**
     Constructor, expects a frame.
     */
    EditPanel()
    {
        super(); // set to modal
        m_Debug = false; // turn off debug by default
//        m_Changed = false;
        m_RecordList = null; // set to null for now
        m_Record = null;
        m_EOLN = System.getProperty("line.separator");
        setDefaultTypes();
        setLayout(new BorderLayout());
        setBackground(Color.lightGray);
        m_Required = new Hashtable();
        m_Optional = new Hashtable();
        m_Extra = new Hashtable();
        m_Type = "ARTICLE";
        m_Key = "";
        m_Preamble = "%" + m_EOLN + "%JavaBib Record" + m_EOLN + "%" + m_EOLN;
        updateType(m_Type);
        add(makeTopPanel(), "North");
        m_RecordPane = new ScrollPane(ScrollPane.SCROLLBARS_ALWAYS);
        m_RecordPanel = new Panel();
        m_RecordPane.add(m_RecordPanel);
        add(m_RecordPane, "Center");
        buildRecordPane();
//        add(buildRecordPane(), "Center", 1);
    }

    /**
     Add an item to a choice component alphabetically
     */
    public void addItemByOrder(Choice ch, String item) {
        // add the choices alphabetically
        if(ch.getItemCount() > 0) {
            int pos = 0;
            while(pos < ch.getItemCount()) {
                if(ch.getItem(pos).compareTo(item) > 0) {
                    ch.insert(item, pos);
                    break;}
                pos++;}
            if(pos == ch.getItemCount()) {
                ch.addItem(item);}}
        else {
            ch.addItem(item);}
    }

    /**
     Add an item to a choice component alphabetically
     */
    public void addItemByOrder(List li, String item) {
        // add the choices alphabetically
        if(li.getItemCount() > 0) {
            int pos = 0;
            while(pos < li.getItemCount()) {
                if(li.getItem(pos).compareTo(item) > 0) {
                    li.addItem(item, pos);
                    break;}
                pos++;}
            if(pos == li.getItemCount()) {
                li.addItem(item);}}
        else {
            li.addItem(item);}
    }

    /**
     Set up the basic layout.  The top will be the same as the earlier version,
     with a label, a type chooser, a key, a crossreference, and room for an
     alternate type.
     */
    private Panel makeTopPanel() {
        // in this incarnation, buttons controlling the use of the
        // data are not used.
        // now the panel that specifies the type of record
        Panel midPan = new Panel(new GridLayout(0,4));
        Label lab = null;
//        Choice ch = null;
//        TextField t = null;
        lab = new Label("Type:");
        lab.setAlignment(Label.RIGHT);
        midPan.add(lab);
        m_TypeChoice = new Choice();
        // fill the choices from the defaults
        Enumeration enum = m_HelpList.keys();
        while(enum.hasMoreElements()) {
            // sort items alphabetically
            String label = (String)enum.nextElement();
            addItemByOrder(m_TypeChoice, label);}
        // add 'OTHER' to the end
        m_TypeChoice.addItem("OTHER");
        // now select the current type
        if(m_HelpList.containsKey(m_Type)) {
            m_TypeChoice.select(m_Type);}
        else {
            m_TypeChoice.select("OTHER");}
        // add an item listener and add it to the panel
        m_TypeChoice.addItemListener(this);
        midPan.add(m_TypeChoice);
        lab = new Label("Key:");
        lab.setAlignment(Label.RIGHT);
        midPan.add(lab);
        m_KeyField = new TextField(m_Key, 10);
        m_KeyField.setBackground(Color.white);
//        m_KeyField.addTextListener(this);
        midPan.add(m_KeyField);
        lab = new Label("Custom:");
        lab.setAlignment(Label.RIGHT);
        midPan.add(lab);
        m_CustomField = new TextField(10);
        m_CustomField.setBackground(Color.white);
//        m_CustomField.addTextListener(this);
        if(m_TypeChoice.getSelectedItem().equals("OTHER")) {
            m_CustomField.setText(m_Type);}
        midPan.add(m_CustomField);
        lab = new Label("CrossRef:");
        lab.setAlignment(Label.RIGHT);
        midPan.add(lab);
        m_CrossRefField = new TextField(10);
        m_CrossRefField.setBackground(Color.white);
//        m_CrossRefField.addTextListener(this);
        midPan.add(m_CrossRefField);
        // create another panel for the bottom, with the type of record
        Panel botPan = new Panel(new GridLayout(1,1));
        m_HelpLabel = new Label((String)m_HelpList.get(m_Type));
        m_HelpLabel.setAlignment(Label.CENTER);
        botPan.add(m_HelpLabel);
        // correctly set up the top panel
        setTopPanel(m_TypeChoice.getSelectedItem());
        // now put them together
        Panel topPan = new Panel(new BorderLayout());
        topPan.add("Center", midPan);
        topPan.add("South", botPan);
        return topPan;
    }

    /**
     Add a listener for focus events on the key text area.
     */
    public void addKeyFocusListener(FocusListener list) {
        if(m_KeyField != null) {
            m_KeyField.addFocusListener(list);}
    }

    /**
     Add a listener for focus events on the crossref text field.
     */
    public void addCrossRefFocusListener(FocusListener list) {
        if(m_CrossRefField != null) {
            m_CrossRefField.addFocusListener(list);}
    }

    /**
     Pass a reference to the key field for comparison.
     */
    public TextField getKeyTextField() {
        return m_KeyField;}

    /**
     Pass a reference to the crossreference field, for comparison.
     */
    public TextField getCrossRefTextField() {
        return m_CrossRefField;}

    /**
     Little method to extract the text information from the lists.  Elements
     out of the from hash are removed as they are added to the to hash.
     */
    private void getFromTextComponents(Hashtable fromHash, Hashtable toHash) {
        Enumeration enum = fromHash.keys();
        while(enum.hasMoreElements()) {
            String name = (String)enum.nextElement();
            String txt = ((TextComponent)fromHash.get(name)).getText();
            if(txt.length() > 0) {
                toHash.put(name, txt);}
            fromHash.remove(name);}
    }

    /**
     Little method to extract the text information from the lists.  Elements
     out of the from hash are retained.
     */
    private void copyFromTextComponents(Hashtable fromHash, Hashtable toHash) {
        Enumeration enum = fromHash.keys();
        while(enum.hasMoreElements()) {
            String name = (String)enum.nextElement();
            String txt = ((TextComponent)fromHash.get(name)).getText();
            if(txt.length() > 0) {
                toHash.put(name, txt);}}
    }

    /**
     Add a set of text fields to a hash.  This method does not do any
     checking.  It goes through the list, adding the required field
     items, and extracting text from the hash.  Elements out of the
     from hash are removed as they are transferred to the to hash.
     */
    private void addTextFields(Hashtable fromHash, Hashtable toHash, String list) {
        if(list != null) {
            StringTokenizer tokens = new StringTokenizer(list, "*/, ");
            while(tokens.hasMoreTokens()) {
                String name = tokens.nextToken();
                if(fromHash.containsKey(name)) {
                    toHash.put(name, new TextField((String)fromHash.get(name), 72));}
                else {
                    toHash.put(name, new TextField("", 72));}
                // add a listener to the text component
                ((TextField)toHash.get(name)).setBackground(Color.white);;
                fromHash.remove(name);}}
    }

    /**
     Add a set of text areas to a hash.  As for the above, this method
     does not do any checking.  There is nothing to check in this case, it
     is a simple transfer from the one hash to the other.  Elements out of
     the from hash are removed as they are added to the to hash.
     */
    private void addTextAreas(Hashtable fromHash, Hashtable toHash) {
        Enumeration enum = fromHash.keys();
        while(enum.hasMoreElements()) {
            String name = (String)enum.nextElement();
            toHash.put(name, new TextArea((String)fromHash.get(name), 8, 72,
                                          TextArea.SCROLLBARS_BOTH));
            // will try not adding the text area listener, as this is
            // done on the one that is displayed
            ((TextArea)toHash.get(name)).setBackground(Color.white);
            fromHash.remove(name);}
    }

    /**
     Method that changes the type shown on the display
     */
    private void updateType(String newType) {
        m_Type = newType;
        // get the data from the old fields
        Hashtable storage = new Hashtable();
        getFromTextComponents(m_Required, storage);
        getFromTextComponents(m_Optional, storage);
        getFromTextComponents(m_Extra, storage);
        // remove the cross reference
        if(m_CrossRefHash != null) {
            Enumeration enum = storage.keys();
            while(enum.hasMoreElements()) {
                String item = (String)enum.nextElement();
                if(m_CrossRefHash.containsKey(item)) {
                    // item present in crossref, does it
                    // contain same text
                    if(((String)storage.get(item))
                       .equals((String)m_CrossRefHash.get(item))) {
                        // have a match, remove it from the storage hash
                        storage.remove(item);}}}}
        // check if the current type includes cross reference
        if(!m_Type.endsWith("(Xref)")) {
            m_CrossRef = "";
            m_CrossRefHash = null;
            if(m_CrossRefField != null) m_CrossRefField.setText("");}
        // build the new fields
        addTextFields(storage, m_Required, (String)m_ReqList.get(m_Type));
        addTextFields(storage, m_Optional, (String)m_OptList.get(m_Type));
        addTextAreas(storage, m_Extra);
        // add the cross reference information
        setCrossReference(m_CrossRefHash);
    }

    /**
     Method to create a block of fields.  These are text fields, for the
     required and optional parts of the record.
     @param p the panel to which the text areas are added
     @param gb the gridbag used for laying out the text areas
     @param gc the constraints to define the position
     @param label the label for this block
     @param hash the hashtable to add the components to
     */
    private void buildTextFieldBlock(Panel p, GridBagLayout gb, GridBagConstraints gc,
                                     String order, String label, Hashtable hash) {
        // return without doing anything if the order string is empty
        if(order == null) return;
        // add fields only if there are fields to add
        if(!hash.isEmpty()) {
            Label lab = new Label(label);
            gc.gridwidth = GridBagConstraints.REMAINDER;
            gb.setConstraints(lab, gc);
            p.add(lab);
            // add the required components
            String separator = "";
            StringTokenizer str = new StringTokenizer(order, "*/, ", true);
            while(str.hasMoreTokens()) {
                String name = (String)str.nextToken();
                if(name.length() == 1) {
                    // have a separator
                    separator = name;}
                else {
                    // not a separator, check if need to add 'or' to label
                    debug("Field name = " + name);
                    // first a button, to be formatted for help.
                    String butLab = "";
                    if(separator.equals("/")) {
                        butLab = "or " + name;}
                    else {
                        if(separator.equals("*")) {
                            butLab = "and/or " + name;}
                        else {
                            butLab = name;}}
                    Button bt = new Button(butLab);
                    gc.fill = GridBagConstraints.BOTH;
                    gc.gridwidth = GridBagConstraints.RELATIVE;
                    gb.setConstraints(bt, gc);
                    p.add(bt);
                    // now add the text field
                    TextField tf = (TextField)hash.get(name);
                    gc.gridwidth = GridBagConstraints.REMAINDER;
                    gb.setConstraints(tf, gc);
                    p.add(tf);}}
            gc.fill = GridBagConstraints.NONE;}
    }
    
    /**
     Method to create a block of fields.  These are text areas, for the
     extra fields.  All the fields remaining in the hash table are added
     to the toHash.
     @param p the panel to which the text areas are added
     @param gb the gridbag used for laying out the text areas
     @param gc the constraints to define the position
     @param label the label for this block
     @param hash the hashtable to add the components to
     */
    private void buildTextAreaBlock(Panel p, GridBagLayout gb, GridBagConstraints gc,
                                     String label, Hashtable hash) {
        Label lab = new Label(label);
        gc.gridwidth = GridBagConstraints.REMAINDER;
        gc.gridheight = GridBagConstraints.RELATIVE;
        gb.setConstraints(lab, gc);
        p.add(lab);
        // create a new panel, to hold the buttons and stuff
        Panel butPan = new Panel(new BorderLayout());
        // generate a chooser for the components that are
        // part of the list
        m_ExtraList = new List(4);
        m_ExtraList.setBackground(Color.lightGray);
        Enumeration enum = hash.keys();
        while(enum.hasMoreElements()) {
            // add the choices alphabetically
            String name = (String)enum.nextElement();
            addItemByOrder(m_ExtraList, name);}
        // add a text field, for the extra name.  This field goes on
        // the top, so that new values can be entered.
        m_ExtraField = new TextField(10);
        m_ExtraField.addTextListener(this);
        m_ExtraField.setBackground(Color.white);
        // last item to add, option for adding a new field
        m_ExtraList.addItemListener(this);
        // now select the first item in the list, by default I think
        butPan.add(m_ExtraField, "North");
        butPan.add(m_ExtraList, "Center");
        // set the constraints and add the panel
        gc.gridwidth = GridBagConstraints.RELATIVE;
        gc.gridheight = GridBagConstraints.REMAINDER;
        gc.anchor = GridBagConstraints.NORTH;
        gb.setConstraints(butPan, gc);
        p.add(butPan);
        // now add the text area
        m_ExtraArea = new TextArea(8, 72);
        m_ExtraArea.addTextListener(this);
        m_ExtraArea.setBackground(Color.white);
        m_ExtraArea.setVisible(false);
        gc.gridwidth = GridBagConstraints.REMAINDER;
        gc.gridheight = GridBagConstraints.REMAINDER;
        gc.anchor = GridBagConstraints.CENTER;
        gb.setConstraints(m_ExtraArea, gc);
        // check to see if this is a new field, in which case the
        // text area should not be visible, or if it is an old one
        if(m_ExtraList.getItemCount() > 0) {
            // list not empty, select top item and show data
            if(m_ExtraList.getSelectedIndex() >= 0) {
                // deselect selected item, if we have one
                m_ExtraList.deselect(m_ExtraList.getSelectedIndex());}
            // select the first item
            m_ExtraList.select(0);
            String item = m_ExtraList.getSelectedItem();
            m_ExtraArea = (TextArea)m_Extra.get(item);
            m_ExtraField.setText(item);
            debug("TextArea " + item + ":" + (TextArea)m_Extra.get(item));
            m_ExtraArea.setVisible(true);}
        p.add(m_ExtraArea);
    }

    /**
     Set up the custom and cross reference field according to
     whether or not the type permits it.
     */
    public void setTopPanel(String item) {
        if(item.endsWith("(Xref)")) {
            m_CrossRefField.setVisible(true);}
        else {
            m_CrossRefField.setVisible(false);}
        if(item.equals("OTHER")) {
            m_CustomField.setVisible(true);
            if(m_Type.length() > 0) {
                m_CustomField.setText(m_Type);}}
        else {
            m_CustomField.setVisible(false);}
        if(m_HelpList.containsKey(item)) {
            m_HelpLabel.setText((String)m_HelpList.get(item));}
        else {
            m_HelpLabel.setText("");}
    }
            
    /**
     Build the scroll pane that contains the fields, and add it.
     */
    public void buildRecordPane() {
        m_RecordPanel.removeAll();
        // need to put together a gridbagLayout
        GridBagLayout gb = new GridBagLayout();
        m_RecordPanel.setLayout(gb);
        GridBagConstraints gc = new GridBagConstraints();
        // build the required part
        buildTextFieldBlock(m_RecordPanel, gb, gc, (String)m_ReqList.get(m_Type),
                            "********** REQUIRED **********", m_Required);
        buildTextFieldBlock(m_RecordPanel, gb, gc, (String)m_OptList.get(m_Type),
                            "********** OPTIONAL **********", m_Optional);
        buildTextAreaBlock(m_RecordPanel, gb, gc, "********** EXTRA **********", m_Extra);
        // transfer the focus to the key, if possible
//        m_KeyField.requestFocus();
    }

    /**
     Set the contents from a BibTeXRecord object
     */
    public void setContents(BibTeXRecord data) {
        // get the data out of the record and into a hashtable
        Hashtable hash = new Hashtable();
        for(Enumeration e = data.keys(); e.hasMoreElements();) {
            String key = (String)e.nextElement();
            hash.put(key, ((BibTeXField)data.get(key)).getBareText());}
        // set the preamble
        m_Preamble = data.getPreamble();
        // cleave the '@' from the start, if it is there
        if(data.getType().startsWith("@")) {
            setContents(data.getType().substring(1), data.getKey(), hash);}
        else {
            // call the other setContents routine
            setContents(data.getType(), data.getKey(), hash);}
    }

    /**
     Get the data in the panel as a BibTeXRecord
     */
    public BibTeXRecord getBibTeXRecord() {
        // transfer all the stuff into the variables
        copyDataValues();
        // create a new record
        BibTeXRecord rec = new BibTeXRecord();
        // fix type for terminating crossref
        if(m_Type.endsWith("(Xref)")) {
            rec.setType("@" + m_Type.substring(0,m_Type.length()-6));}
        else {
            rec.setType("@" + m_Type);}
        // get the other stuff
        rec.setKey(m_Key);
        rec.setPreamble(m_Preamble);
        rec.setBareData(m_Holding);
        // take out any stuff from the crossreference
        if(rec.containsKey("CROSSREF")) {
            if(m_CrossRefHash != null) {
                if(m_CrossRefHash.size() > 0) {
                    // have stuff, pull out matches
                    Enumeration enum = m_CrossRefHash.keys();
                    while(enum.hasMoreElements()) {
                        String key = (String)enum.nextElement();
                        if(rec.containsKey(key)) {
                            // have match, compare contents
                            if(((BibTeXField)rec.get(key)).getBareText()
                               .equals((String)m_CrossRefHash.get(key))) {
                                rec.remove(key);}}}}}}
        // clear out holding and type so that there are no problems
        m_Holding = new Hashtable();
//        m_Type = "";
//        m_Key = "";
        // return the record
        return rec;
    }

    /**
     Set the information contained in this dialog
     */
    public void setContents(String type, String key, Hashtable data) {
//        m_Changed = false; // flag that there have not been any changes made
        m_Type = type.toUpperCase();
        if(m_HelpList.containsKey(new String(type + "(Xref)"))) {
            m_Type = type + "(Xref)";}
        m_KeyField.setText(key);
        m_CrossRefHash = null;
        // convert data into holding hash, changing field
        // names to upper case
        m_Holding = new Hashtable();
        Enumeration enum = data.keys();
        while(enum.hasMoreElements()) {
            String name = ((String)enum.nextElement());
            m_Holding.put(new String(name.toUpperCase()),
                          data.get(name));}
        // take out the cross reference, if there is one
        if(m_Holding.containsKey("CROSSREF")) {
            m_CrossRefField.setText((String)m_Holding.get("CROSSREF"));
            m_Holding.remove("CROSSREF");}
        else {
            m_CrossRefField.setText("");}
        // put the data in the required fields
        m_Required = new Hashtable();
        addTextFields(m_Holding, m_Required, (String)m_ReqList.get(m_Type));
        m_Optional = new Hashtable();
        addTextFields(m_Holding, m_Optional, (String)m_OptList.get(m_Type));
        m_Extra = new Hashtable();
        addTextAreas(m_Holding, m_Extra);
        // update the display
        if(m_HelpList.containsKey(m_Type)) {
            m_TypeChoice.select(m_Type);
            setTopPanel(m_Type);}
        else {
            m_TypeChoice.select("OTHER");
            setTopPanel("OTHER");}
        buildRecordPane();
//        validate();
    }

    /**
     Set the cross reference from a bibtex record.
     */
    public void setCrossReference(BibTeXRecord data) {
        Hashtable hash = new Hashtable();
        Enumeration enum = data.keys();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            hash.put(key, ((BibTeXField)data.get(key)).getBareText());}
        setCrossReference(hash);
    }

    /**
     Set the for a crossreference.  Take the contents of the passed hash and
     set the values of the m_Required and m_Optional fields which are blank
     at the present moment to the values in the crossreference.
     */
    public void setCrossReference(Hashtable data) {
        // null or empty, return
        if(data == null) return;
        if(data.size() == 0) return;
        // store the reference to the crossreference, if not stored already
        if(!data.equals(m_CrossRefHash)) {
            m_CrossRefHash = data;}
        // check the selected type to see if this is an 'inbook'
        if(!m_TypeChoice.getSelectedItem().startsWith("INBOOK")) {
            // not inbook, ditch title and set booktitle
            if(data.containsKey("TITLE")) {
                data.put("BOOKTITLE", data.get("TITLE"));
                data.remove("TITLE");}}
        // enumerate through the required list
        Enumeration enum = m_Required.keys();
        while(enum.hasMoreElements()) {
            String name = (String)enum.nextElement();
            // does crossreference contain this key?
            if(data.containsKey(name)) {
                TextField ta = (TextField)m_Required.get(name);
                // does text area contain any text
                if(ta.getText().equals("")) {
                    // set text to that passed
                    ta.setText((String)(data.get(name)));}}}
        // now do it for the optional list
        enum = m_Optional.keys();
        while(enum.hasMoreElements()) {
            String name = (String)enum.nextElement();
            // does crossreference contain this key?
            if(data.containsKey(name)) {
                TextField ta = (TextField)m_Optional.get(name);
                // does text area contain any text
                if(ta.getText().equals("")) {
                    // set text to that passed
                    ta.setText((String)(data.get(name)));}}}
        // build the record pane
        buildRecordPane();
        // validate this thing to make sure text is displayed
//        validate();
    }

    /**
     Remove cross reference.  Take the stored cross reference, and
     go through the fields, removing those that have not been changed
     */
    public void removeCrossReference() {
        // if crossreference is null, return
        if(m_CrossRefHash == null) return;
        if(m_CrossRefHash.size() == 0) return;
        // get the data from the old fields
        Hashtable storage = new Hashtable();
        getFromTextComponents(m_Required, storage);
        getFromTextComponents(m_Optional, storage);
        getFromTextComponents(m_Extra, storage);
        // remove the cross reference stuff from the holdings
        Enumeration enum = storage.keys();
        while(enum.hasMoreElements()) {
            String item = (String)enum.nextElement();
            if(m_CrossRefHash.containsKey(item)) {
                // item present in crossref, does it
                // contain same text
                String fieldText = (String)storage.get(item);
                if(((String)m_CrossRefHash.get(item)).equals(fieldText)) {
                    // have a match, remove it from the storage hash
                    storage.remove(item);}}}
        // empty the crossreference field
        m_CrossRefHash = null;
        // build the new fields
        addTextFields(storage, m_Required, (String)m_ReqList.get(m_Type));
        addTextFields(storage, m_Optional, (String)m_OptList.get(m_Type));
        addTextAreas(storage, m_Extra);
        // validate
        validate();
    }

    /**
     Process the action events generated by the buttons on the top.
     */
    public void actionPerformed(ActionEvent e)
    {
        debug(e.toString());
        String cmd = e.getActionCommand();
    }

    /**
     Process text events.  These are events passed by either the
     text field m_ExtraField or the text area m_ExtraArea.  The process
     logic is:
     If text field is source, and is not empty, activate area.  Otherwise,
     deactivate area.  Do not activate if name is in required or optional.
     If name is in extra, then display text area for this field.
     If text field is the source, and is empty, remove field from list.  If
     it is not empty, and field name is not in list, add field name to list
     and text to hash.
     */
    public void textValueChanged(TextEvent evt) {
//        m_Changed = true; // record that record has changed
        // focus flag
        boolean focusFlag = false;
        // check for source
        debug(evt.toString());
        String name = m_ExtraField.getText().toUpperCase();
        debug(name);
        if(evt.getSource().equals(m_ExtraField)) {
            // change in name, check to see if it is blank
            if(name.equals("")) {
                // yes, hide field
                debug("blank");
                m_ExtraArea.setVisible(false);}
            else {
                // has a value, check if exists
                if((m_Required.containsKey(name)
                    || m_Optional.containsKey(name))) {
                    // in other fields, hide
                    debug("elsewhere");
                    focusFlag = true;
                    m_ExtraArea.setVisible(false);}
                else {
                    if(m_Extra.containsKey(name)) {
                        // in extra list, switch text
                        GridBagLayout gb = (GridBagLayout)m_RecordPanel.getLayout();
                        GridBagConstraints gc = gb.getConstraints(m_ExtraArea);
                        // remove the component
                        m_RecordPanel.remove(m_ExtraArea);
                        m_ExtraArea.removeTextListener(this);
                        m_ExtraArea = (TextArea)m_Extra.get(name);
                        m_ExtraArea.addTextListener(this);
                        // add the component
                        gb.setConstraints(m_ExtraArea, gc);
                        m_RecordPanel.add(m_ExtraArea);}
                    else {
                        // have name that does not appear in any list,
                        // add it to extra list and display
                        m_Extra.put(name, new TextArea(8, 72));
                        ((TextArea)m_Extra.get(name)).setBackground(Color.white);
                        // get the constraints for the text area
                        GridBagLayout gb = (GridBagLayout)m_RecordPanel.getLayout();
                        GridBagConstraints gc = gb.getConstraints(m_ExtraArea);
                        // remove the component
                        m_RecordPanel.remove(m_ExtraArea);
                        m_ExtraArea.removeTextListener(this);
                        // check to see if this one is to be removed from list
                        if(m_ExtraArea.getText().equals("")) {
                            // name should already be removed
                            m_Extra.remove(m_ExtraArea);}
                        m_ExtraArea = (TextArea)m_Extra.get(name);
                        m_ExtraArea.addTextListener(this);
                        // add the component
                        gb.setConstraints(m_ExtraArea, gc);
                        m_RecordPanel.add(m_ExtraArea);}}
                focusFlag = true;}}
        if(evt.getSource().equals(m_ExtraArea)) {
            // is a text area, so add or remove list name as appropriate
            if(m_ExtraArea.getText().equals("")) {
                // blank, remove name from list
                debug("blank text area");
                // this bit may get hit early, apparently due to a focus
                // request.  Check to see that name is contained or not
                // blank before trying to remove
                if(name != null)
                    if(!name.equals("")) {
                        String[] extra = m_ExtraList.getItems();
                        boolean flag = false;
                        for(int i=0; i<extra.length; i++) {
                            if(extra[i].equals(name)) flag = true;}
                        if(flag) m_ExtraList.remove(name);}
            }
            else {
                // not blank, add name if not already there
                debug("text area not blank");
                String[] extra = m_ExtraList.getItems();
                boolean flag = false;
                for(int i=0; i<extra.length; i++) {
                    if(extra[i].equals(name)) flag = true;}
                if(!flag) {
                    // name not here, so add
                    addItemByOrder(m_ExtraList, name);}}}
        validate();
        if(focusFlag) m_ExtraField.requestFocus();
    }

    /**
     Process the item events.  For now, these will only come from the
     type chooser.  However, they may also come from the extra field
     chooser in future.
     */
    public void itemStateChanged(ItemEvent e)
    {
        debug(e.toString());
        if(e.getItemSelectable().equals(m_TypeChoice)) {
            debug("Caught ItemEvent at m_TypeChoice");
            String item = m_TypeChoice.getSelectedItem();
            if(item.equals("OTHER")) {
                m_Type = m_CustomField.getText();}
            else {
                m_Type = item;}
            setTopPanel(item);
            // take the data from the fields and update
            updateType(m_Type);
            // replace the scroll panel contents
            buildRecordPane();
//            remove(1);
//            add(buildRecordPane(), "Center", 1);
            validate();
        }
        // check for results from extra field item selected choice
        if(e.getItemSelectable().equals(m_ExtraList)) {
            debug("Caught ItemEvent at m_ExtraList");
            String item = m_ExtraList.getSelectedItem();
            // get the constraints for the text area
            GridBagLayout gb = (GridBagLayout)m_RecordPanel.getLayout();
            GridBagConstraints gc = gb.getConstraints(m_ExtraArea);
            // remove the component
            m_RecordPanel.remove(m_ExtraArea);
            // display the appropriate new component
            m_ExtraArea = (TextArea)m_Extra.get(item);
            // update the name
            m_ExtraField.setText(item);
            // add the component
            gb.setConstraints(m_ExtraArea, gc);
            m_RecordPanel.add(m_ExtraArea);
            validate();
        }
    }

    /**
     Check the status of the fields, making sure that all those which
     are required are present.
     Things to check:
     1) That the key is not a duplicate
     2) That the crossreference exists (if eligible)
     3) That the custom field is not a standard type
     4) That the required fields are all filled
     The method returns a string indicating the problem.  If the string
     is null, then the record has passed.
     */
    public String checkEntry() {
        String ret = null;
        String key = m_KeyField.getText();
        if(key.length() == 0) {
            return "Each record must contain a key.";}
        if(m_RecordList != null) {
            // check if list contains the key
            // check if list contains the crossreference
        }
        // check the custom type against the list
        String type = m_TypeChoice.getSelectedItem();
        if(type.equals("OTHER")) {
            type = m_CustomField.getText();
            if(m_HelpList.containsKey(type)) {
                return "Custom type cannot be among list of known record types.";}
            if(type.length() == 0) {
                return "Custom type must be indicated.";}}
        // check the contents of the required field
        if(m_ReqList.containsKey(type)) {
            // go through each token, splitting at spaces and commas
            StringTokenizer str = new StringTokenizer((String)m_ReqList.get(type), ", ");
            while(str.hasMoreTokens()) {
                String token = str.nextToken();
                // now check to see if the token contains a '/'
                int pos = token.indexOf('/');
                if(pos >= 0) {
                    // have a pair of fields where one of them, but not
                    // both, is needed
                    String left = token.substring(0, pos);
                    String right = token.substring(pos+1);
                    // check to see if both fields exist
                    if((((TextField)m_Required.get(left)).getText().length() > 0)
                       && (((TextField)m_Required.get(right)).getText().length() > 0)) {
                        ret = "Only one of " + left + " or " + right
                            + " may contain text.";
                        break;}
                    else {
                        // check to see if there are no entries
                        if((((TextField)m_Required.get(left)).getText().length() == 0)
                           && (((TextField)m_Required.get(right)).getText().length() == 0)) {
                            ret = "Either " + left + " or " + right
                                + " must contain text.";
                            break;}}}
                else {
                    // check for a '*' in the token
                    pos = token.indexOf('*');
                    if(pos >= 0) {
                        // have a pair of fields where one of them, but not
                        // both, is needed
                        String left = token.substring(0, pos);
                        String right = token.substring(pos+1);
                        // now we need at least one
                        if((((TextField)m_Required.get(left)).getText().length() == 0)
                           && (((TextField)m_Required.get(right)).getText().length() == 0)) {
                            ret = "At least one of " + left + " or " + right
                                + " must contain text.";
                            break;}}
                    else {
                        // have neither '/' or '*', so check for this
                        // one directly
                        if(((TextField)m_Required.get(token)).getText().length() == 0) {
                            ret = "The field " + token + " must contain an entry.";
                            break;}}}}}
        return ret;
    }

    /**
     Set the defaults for the record types and their fields
     */
    private void setDefaultTypes() {
        // just add the fields to the hashtables
        m_ReqList = new Hashtable();
        m_OptList = new Hashtable();
        m_HelpList = new Hashtable();
        m_HelpList.put("ARTICLE", "article from a journal or magazine");
        m_ReqList.put("ARTICLE", "AUTHOR, TITLE, JOURNAL, YEAR");
        m_OptList.put("ARTICLE", "VOLUME, NUMBER, PAGES, MONTH");
        m_HelpList.put("BOOK", "book with a definite publisher");
        m_ReqList.put("BOOK", "AUTHOR/EDITOR, TITLE, PUBLISHER, YEAR");
        m_OptList.put("BOOK", "VOLUME/NUMBER, SERIES, ADDRESS, EDITION, MONTH, NOTE");
        m_HelpList.put("BOOKLET", "printed and bound work without the name of a publisher");
        m_ReqList.put("BOOKLET", "TITLE");
        m_OptList.put("BOOKLET", "AUTHOR, HOWPUBLISHED, ADDRESS, MONTH, YEAR, NOTE");
        m_HelpList.put("INBOOK(Xref)", "part of a book");
        m_ReqList.put("INBOOK(Xref)", "AUTHOR/EDITOR, TITLE, CHAPTER*PAGES, PUBLISHER, YEAR");
        m_OptList.put("INBOOK(Xref)", "VOLUME/NUMBER, SERIES, TYPE, ADDRESS, EDITION, MONTH, NOTE");
        m_HelpList.put("INCOLLECTION(Xref)", "part of a book with its own title");
        m_ReqList.put("INCOLLECTION(Xref)", "AUTHOR, TITLE, BOOKTITLE, PUBLISHER, YEAR");
        m_OptList.put("INCOLLECTION(Xref)", "EDITOR, VOLUME/NUMBER, SERIES, TYPE, CHAPTER, PAGES, ADDRESS, EDITION, MONTH, NOTE");
        m_HelpList.put("INPROCEEDINGS(Xref)", "article in conference proceedings");
        m_ReqList.put("INPROCEEDINGS(Xref)", "AUTHOR, TITLE, BOOKTITLE, YEAR");
        m_OptList.put("INPROCEEDINGS(Xref)", "EDITOR, VOLUME/NUMBER, SERIES, PAGES, ADDRESS, MONTH, ORGANIZATION, PUBLISHER, NOTE");
        m_HelpList.put("MANUAL", "techical documentation");
        m_ReqList.put("MANUAL", "TITLE");
        m_OptList.put("MANUAL", "AUTHOR, ORGANIZATION, ADDRESS, EDITION, MONTH, YEAR, NOTE");
        m_HelpList.put("MASTERSTHESIS", "Master's thesis");
        m_ReqList.put("MASTERSTHESIS", "AUTHOR, TITLE, SCHOOL, YEAR");
        m_OptList.put("MASTERSTHESIS", "TYPE, ADDRESS, MONTH, NOTE");
        m_HelpList.put("MISC", "doesn't fit under other categories");
        m_ReqList.put("MISC", "");
        m_OptList.put("MISC", "AUTHOR, TITLE, HOWPUBLISHED, MONTH, YEAR, NOTE");
        m_HelpList.put("PHDTHESIS", "Doctoral thesis");
        m_ReqList.put("PHDTHESIS", "AUTHOR, TITLE, SCHOOL, YEAR");
        m_OptList.put("PHDTHESIS", "TYPE, ADDRESS, MONTH, NOTE");
        m_HelpList.put("PROCEEDINGS", "conference proceedings");
        m_ReqList.put("PROCEEDINGS", "TITLE, YEAR");
        m_OptList.put("PROCEEDINGS", "EDITOR, VOLUME/NUMBER, SERIES, ADDRESS, MONTH, ORGANIZATION, PUBLISHER, NOTE");
        m_HelpList.put("TECHREPORT", "report published by school or institution");
        m_ReqList.put("TECHREPORT", "AUTHOR, TITLE, INSTITUTION, YEAR");
        m_OptList.put("TECHREPORT", "TYPE, NUMBER, ADDRESS, MONTH, NOTE");
        m_HelpList.put("UNPUBLISHED", "unpublished work with author and title");
        m_ReqList.put("UNPUBLISHED", "AUTHOR, TITLE, NOTE");
        m_OptList.put("UNPUBLISHED", "MONTH, YEAR");
//        m_HelpList.put("OTHER", "category not part of current list");
//        m_ReqList.put("OTHER", "");
//        m_OptList.put("OTHER", "");
    }

    /**
     Set up all the data into the fields reserved globally
     */
    private void getDataValues() {
        m_Type = m_TypeChoice.getSelectedItem();
        m_CrossRef = m_CrossRefField.getText();
        m_Key = m_KeyField.getText();
        if(m_Type.equals("OTHER")) {
            m_Type = m_CustomField.getText().toUpperCase();}
        if(!m_Type.endsWith("(Xref)")) {
            m_CrossRef = "";}
        m_Holding = new Hashtable();
        // add values from fields to hash
        if(m_CrossRef.length() > 0) {
            m_Holding.put("CROSSREF", m_CrossRef);}
        getFromTextComponents(m_Required, m_Holding);
        getFromTextComponents(m_Optional, m_Holding);
        getFromTextComponents(m_Extra, m_Holding);
    }
    
    /**
     Copy the data values into the holding array
     */
    private void copyDataValues() {
        m_Type = m_TypeChoice.getSelectedItem();
        m_CrossRef = m_CrossRefField.getText();
        m_Key = m_KeyField.getText();
        if(m_Type.equals("OTHER")) {
            m_Type = m_CustomField.getText().toUpperCase();}
        if(!m_Type.endsWith("(Xref)")) {
            m_CrossRef = "";}
        m_Holding = new Hashtable();
        // add values from fields to hash
        if(m_CrossRef.length() > 0) {
            m_Holding.put("CROSSREF", m_CrossRef);}
        copyFromTextComponents(m_Required, m_Holding);
        copyFromTextComponents(m_Optional, m_Holding);
        copyFromTextComponents(m_Extra, m_Holding);
    }
    
    /**
     Output a string with the critical information
     */
    public String toString() {
        String ret = "EditDialog:" + m_EOLN + getRecordString();
        return ret;
    }

    /**
     Return the type for this record
     */
    public String getRecordType() {
        getDataValues(); // get the data into the fields
        return m_Type;}

    /**
     Return the key for this record
     */
    public String getRecordKey() {
        getDataValues(); // get the data into the fields
        return m_Key;}

    /**
     Return a hashtable with a copy of the data contained in this
     record.  A copy is used to ensure that damage cannot occur to the
     actual data contained in the editor.
     */
    public Hashtable getRecordFields() {
        getDataValues(); // get the data into the fields
        Hashtable hash = new Hashtable();
        Enumeration enum = m_Holding.keys();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            hash.put(key, m_Holding.get(key));}
        return hash;
    }

    /**
     Generate a string representation of the information
     contained in this dialog
     */
    public String getRecordString() {
        getDataValues(); // get the data into the fields
        // figure out the type
        String type = new String();
        if(m_Type.endsWith("(Xref)")) {
            type = m_Type.substring(0, m_Type.length()-6);}
        else {
            type = m_Type;}
        // get the record information for this type, so that
        // the output can be put in a pretty arrangement
        String fields = new String();
        if(m_ReqList.containsKey(type)) fields = (String)m_ReqList.get(type);
        if(m_OptList.containsKey(type)) fields = fields + " " + (String)m_OptList.get(type);
        // copy the hashtable so that we can remove elements
        Hashtable copy = new Hashtable();
        Enumeration enum = m_Holding.keys();
        while(enum.hasMoreElements()) {
            String key = (String)enum.nextElement();
            copy.put(key, m_Holding.get(key));}
        // create space for the output field
        StringBuffer ret = new StringBuffer("@" + type + "(" + m_Key);
        // add the cross reference, if needed
        if(copy.containsKey("CROSSREF")) {
            ret.append("," + m_EOLN + "    CROSSREF = {" + m_CrossRef + "}");
            copy.remove("CROSSREF");}
        // go through the possible records and compile them in order into
        // the output string
        StringTokenizer st = new StringTokenizer(fields, ", ");
        while(st.hasMoreTokens()) {
            String field = st.nextToken();
            if(copy.containsKey(field)) {
                ret.append("," + m_EOLN + "   " + field + " = {" + (String)copy.get(field) + "}");}
            copy.remove(field);}
        // have listed all the required and optional fields, now add anything
        // that is left
        enum = copy.keys();
        while(enum.hasMoreElements()) {
            String field = (String)enum.nextElement();
            ret.append("," + m_EOLN + "    " + field + " = {" + (String)copy.get(field) + "}");}
        // all done, just close it
        ret.append(m_EOLN + ")" + m_EOLN);
        return ret.toString();
    }

    
    /**
     Method to output a string when a debug flag is set
     */
    public static void debug(String message)
    {
        if(m_Debug) System.out.println("EditDialog:"+message);}

    /**
     Method to set the debug flag
     */
    public static void setDebug(boolean flag) {m_Debug = flag;}
}
