//  Copyright Takuma Yoshimura 2024.
//  Copyright Matt Borland 2024.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#define BOOST_TEST_MAIN
#define BOOST_TEST_MODULE StatsLandauTest

#ifdef BOOST_MATH_ENABLE_SYCL
#include "sycl/sycl.hpp"
#endif

#include <boost/math/tools/config.hpp>
#include <boost/test/included/unit_test.hpp>
#include <boost/test/tools/floating_point_comparison.hpp>
#include <boost/math/distributions/landau.hpp>

#if __has_include(<stdfloat>)
# include <stdfloat>
#endif

#ifdef __clang__
#  pragma clang diagnostic push 
#  pragma clang diagnostic ignored "-Wliteral-range"
#elif defined(__GNUC__)
#  pragma GCC diagnostic push 
#  pragma GCC diagnostic ignored "-Woverflow"
#endif

using boost::math::landau_distribution;

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
#  include <boost/multiprecision/cpp_bin_float.hpp>
   using boost::multiprecision::cpp_bin_float_quad;
#endif

template<class RealType, int N>
void do_test_landau_pdf(){
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    // Left tail of Landau distribution inherently limits accuracy due to the rapid decay of the function value.
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.23085262843901249987787600097675204323741277288698e-2763), tolerance * 10000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.43092685592935588160074717060232841559786193515624e-1259), tolerance * 10000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.07984079126853424748872672315932614660166330244758e-574), tolerance * 4000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51902330649665702862232056199116883579782363388698e-261), tolerance * 1000);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.61740160083645125413680224234682096739098348513281e-119), tolerance * 400);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.40268429420551752841762522647338593851384890728514e-54), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.64264014182893964260735316412669280242258553581563e-37), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.96628298286997713279446280099737948363623849272499e-25), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.65014695258589331535696024118503069088391324558285e-17), tolerance * 100);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52577680004870416264378840612037346351517011876564e-11), tolerance * 40);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.96564073685328676511751663666044321538322085645615e-8), tolerance * 40);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.49195679110887105000100573294850045215882388658736e-5), tolerance * 10);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.83029281547111049679644135145824504175531296639784e-4), tolerance * 4);

    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.50763682207511020788551990942118742909694977020766e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55239934591810979484111502955882518822410677336301e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.11811339800999276275816189148400161297598608940370e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.43803188361205250108898134179433045737003842752098e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.44559645555354621908225839934967437642938589048641e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.19256762240252210606670736977933945194758525070962e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55747609478534232829752995639617852824070107389867e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.90787204474682320276094536207112207798402106183626e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.21762208692280384264052188465103527015057255360522e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.35142856681038695962612277740850797545496160091346e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46926317225475336949604234617889242016284403135263e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.57038182663311646092167022800387128596042855954974e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65453539644067140109079490907422208395106068408806e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72189928839617935958276513066615896148477769341984e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.77299781354951097794140321722510611592577477809046e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.80862887196928314574763226767834515262714169253341e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.82979296472337068393182796635835781531645972141685e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83762914935252825962037404558307269544013453409236e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83335939453844198706047097920194738266342072083183e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.81824188244607845783889480215304948211453087161311e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79353314321623987234651193548815637401927993615201e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.76045845604337224786926223067521339118457783620200e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.72018967730828877761590993107231423707402335516208e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.67382951900254916684467526412940597211696659408252e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.62240126375351657025589608183516471314881294729647e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.56684293488331550048284390907274219670798777890274e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.50800501583089077855151395221936193088464543391697e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.44665091350810416315066697862911894015423062289459e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.38345947054726504652084392231776081590084152956389e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31902894209756674501863785530002142859374765415539e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25388195748200100617512573536171938322886836672500e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.18847108191054606649972320910802001046885937665349e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.12318467675064265241298421202132871385510191955048e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.05835282809935995378486652717710487064456876565464e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.99425317296731752992133425091298094257535743484505e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.93111650117890312500196499480137441707124661561802e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.86913205029713491869481191910148340512494429155677e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.80845244177744590342641324845464181512725512270901e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74919823041494123736512984399481253444800810061398e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.69146205716679200390505774997534008647397580964682e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.63531240868022603475813051802104652763421266792797e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52794577355035866144175272867439043108097332289280e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.42728268648990888425150769069939371653574963683898e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.33330183140387812967271143318337448709293578674499e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.24583903791438503777896988292725453902854934482082e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.16463504238864475787300731138600083506835647657585e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08937118376337417303260299595016407050621597377842e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01969562586719050610986409853288944869103385155807e-1), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.55242261334771588093967856464157010583515904813711e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40542106335080455284302185631726193959498759868958e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.42466467030240081155174776845818529710340047591917e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.58484837369864336802514847217316987349368653299870e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.86394883380361824062302169684513676586056038951383e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.24319810137134739489567689474925534356381531907667e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.70679858898932226345344507218372775665959269901845e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.24154519527191616123631238437619039275267092832219e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.83643820409470770350079809236512802617514831283958e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.17158939997042257734107084716347218188987519860810e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.65588931293442756274955123891841674957134577995432e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.24999943998132461512162382109845468953569057653206e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.92615450486233381461674333853015589016995103903169e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.66449631910567440889641030798080985765446733145615e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.45062281189987454243698017666196791149621836621069e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.27394302531621603387264220385271200998719076373774e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12656323880287532946687856443190592954884366662750e-2), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(9)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.97272209098442387115881795542777687494895710641442e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(10)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.29822142754406367527698808542596968321101482611499e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.04211802337476854086853204096220197132228559528499e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.07803739765703855539464392586871287241424478552475e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.32333875346680119671101261202737313738117089411627e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.72232982334134875374745579629692504175963269123550e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.23647379895109194018623503694593516089449736564227e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.83847488747490686627461184914507143000166295798209e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.23229209720541068914767304360845920096605060550204e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79947182256974676114073539538738666749318597550126e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48016777321149926163568000853843271440982208135806e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.23815521917416110142990010737370397993296057042020e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05051356188948975788809310742705170449199446295996e-3), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.02189494175551818802120780043908400302499693753383e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.82977743390109357722237932712596740553146168838163e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.85767880395157523314894776472286059373130513345678e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.38436988071020010518465973524617926941561115862543e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.33733686912592492663864652100824543220296291087705e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.56719760886187032655497099894304037738414747694058e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98455504950479524812911915761096343833926775180855e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.53330379508737569512814290970113441709685617438320e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.17681386666988585981678825113751906947724221061916e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89036019000414403525553377301848537121144402020797e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65677119200426097313111689557286038389044055520291e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.30262785101049126879812527584486523431435416855092e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05075497824543686508583330151701088746046662556142e-4), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.65309719784826735811404845249006631571011107397937e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.24862987881373408810690419771199915049078409763957e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.15969937273460908511004283053502890256890552857434e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.29851658065131948803621145305770537695198696080417e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.60580149897015438520712002341005415255634795262650e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.04036158768936643116910162586744782570492472991852e-5), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.94016243990046849166189460556288227597025594512208e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(512)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46086028677885635430916943188657821915692719236659e-6), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, static_cast<RealType>(1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.11684517285572531637079275580373275248746393740505e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 11)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.52416040197318095218142515144159361915369773070323e-7), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 12)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.80328507092562833387828627033077037362079287147494e-8), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 13)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.49830894979328205589801770626380874036837651152185e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 14)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37321260457507890170942420030228551149027146711858e-9), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 15)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.93116689130330056188281287044879751157310863062584e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 16)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48253875064413349633513149673423379659751420343007e-10), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 17)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.70600576035028939688935887999924497768090063213495e-11), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 18)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.26455685393631638684357809746815117207931914505548e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 19)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31607812424720075650435345952742430841572532807506e-12), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 20)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.79011407780867471284567547634162699267979186606188e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 21)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.44751775633854916204285667335984781835956253195982e-13), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 22)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.61878017561309094548695350627490496501579775589554e-14), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 23)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.04693171826032841875566733962302122545285900020005e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 24)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26173047049833308211942124367857471386868403241961e-15), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.65432295369693602811408648241147072923809751085578e-16), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 26)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41358031701599971473805281134574751063094181933632e-16), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 27)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.53395024254254681824332124678431347548990493315457e-17), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 28)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.83487488981301080647710951024578189621167614209000e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 29)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20871862925450931967259224559170596670053211110073e-18), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 30)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.52179645209930996243372289840247886946662593536877e-19), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 31)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38044909732789001695763980339161333087125784369381e-19), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 32)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.45112272298940668060590155421055597940788628104792e-20), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 34)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15695169189532579197171519055168026495351913764385e-21), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 36)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34809480577241011630081955001585893790541249979882e-22), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 38)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.42559253331811413189334788330851699569177453819777e-24), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 40)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.26599533286729634727335420905860617977183492898697e-25), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 42)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.29124708296676552183969416435576047604811850887072e-26), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 44)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.05702942684184448841776719264034349959339461930208e-27), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 46)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28564339177412106630199552439532586497065103592483e-28), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 48)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.03527119858493090854162194048072858076706076597819e-30), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 50)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.02204449911503854915599875932648541933495423396290e-31), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 52)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13877781194681051697029360385733987849026091896815e-32), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 54)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96173613246674215642318845771797429014235991071697e-33), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 56)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22608508279171150505713607335390403334626262247470e-34), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 58)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.66303176744819310534466860343470655112837733629379e-36), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 60)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.78939485465512007489594618239133015330653514470962e-37), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 62)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.99337178415944994713157754959412500112801769504923e-38), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 64)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87085736509965620084544629248142254049400680133870e-39), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 68)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.30803658242053201409813080005840450554429514097447e-42), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 72)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85470179000802031747569733233640023770585023325816e-44), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 76)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.11511788672188293650018950543585978468059122696737e-46), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 80)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.35592924500735522070031636324180548468184125270321e-49), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 84)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70153486133099813308596977841944913997064236009234e-51), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 88)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.64662055207421145736704601944518511418678161667227e-54), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 92)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.59633615315398885053400175162079403845778513781261e-56), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 96)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01419380982577689473984441890730791873276374028644e-58), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 100)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.96169456963194099507751725745105477863201113021124e-61), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 104)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54753694126247695120215517859242605013890436588657e-63), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 108)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.04506617680655059063341866635141218965712131228669e-66), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 112)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.36135397531505882446617916654287979923104956976029e-68), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 116)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.22403896607444853307101236930796194301091622511538e-71), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 120)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.60314022112283145823086420676091852871339922342198e-73), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 124)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40747664887610603837143133076598369654786648237564e-75), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.49795565967228921238840363580462378845964351238859e-78), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 128)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.49795565967228921238840363580462378845964351238859e-78), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 136)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.38921456859174989683289128998508268228293005240118e-83), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 144)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28009255502193449353529225005875895420403010118164e-87), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 152)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.95326622775563734975477943429376061127322830738396e-92), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 160)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.98044773522283531151547154891015718272892496044375e-97), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 168)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.54780233035710954515910575700402402149799340844629e-102), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 176)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.93939564568650748467881127472537845077208466864701e-107), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 184)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.05886774378761405711041431804281287395814280222275e-111), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 192)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.61570395475404976976076403509950694878867004733696e-116), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 200)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.46536858330390895044061895004197227293193061422265e-121), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 208)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.76185391739488060064791709906306804341420076633095e-126), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 216)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.74013354094677826026598678445902716585418818104698e-131), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 224)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.75874868918880960123594174874729486977262600867765e-136), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 232)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.33647898699780419940733974437672346035348907603114e-140), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 240)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.03930509490631744294332846737170938164289714970571e-145), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 248)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.11173262772570410605366282252763272345412773087419e-150), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist, ldexp(static_cast<RealType>(1), 256)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74812717853653580635629703144475208046589314403411e-155), tolerance);
}

template<class RealType, int N>
void do_test_landau_cdf() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    // Left tail of Landau distribution inherently limits accuracy due to the rapid decay of the function value.
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.23104384186197610398535377648417544130503923472088e-2767), tolerance * 10000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13839831060468958638389597955042905895011279005609e-1263), tolerance * 10000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.44312475898376675389279659203392726913179988221856e-577), tolerance * 4000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.60161744250590449897633679409320688863793657438650e-264), tolerance * 1000);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.35715172357835600843326068305277508896293574359993e-122), tolerance * 400);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21485044777822987831938208827397125595048645106450e-56), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.20700596052912975894895710721207297741913643002012e-39), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.68977785355043338853968150291349097131276031028880e-27), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.08280503661515358192902976596786876511904900370188e-18), tolerance * 100);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.65792002575428625860974216754165889856915439054540e-13), tolerance * 40);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.10050102765608360408879765554028186586411956977646e-9), tolerance * 40);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.68815894016077213783105942021589578300879813020189e-7), tolerance * 10);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.37625644367407768681214121219960213300943983402353e-5), tolerance * 4);

    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.07114056489178077422539043012078031612755414599729e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.02378089289225642153717146519785629105551593533471e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.86628849116890981757518380028315283124563329691336e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01356086445894450901462536543096095003540865562528e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87500433294484935737405475465845379363381335629325e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.14477362642888337899595114315256777323730741351157e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.86347714259952478386139532766899266845283140865196e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.03233833485353985646171581359634291113172343250638e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.61609610406317335842332400044553397267403259400925e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10447280320452875363390566033897366825208362273976e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.25520504813750944299743577454809066282274647597463e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41278210604233003854999957016294448201095585229262e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.57614904197281705412681135912362432813551089393050e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.74424900339390561779123457277662541525802871208804e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.91604740159711318114657552509684829710278507526176e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.09055135109096066379824832581601920047332653592899e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26682451996131584587603100587452166035163093744954e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.44399775009966844789856291613140149240683895719411e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.62127593214618528740100724985588598153581542828585e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.79794168130771479096847395825066083537476191027208e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.97335637253762835584202042505471763077481208909960e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.14695907126043189562824247063432944143645211079990e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.31826385110143246791390839018674857130873325623297e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(-0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.48685593270949525320895832911820568633181874504622e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.65238701512374797371944390202236332910560491002867e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.81457010849525523978613828303353744448250289768502e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.97317411782647886281236276761522493696113175680302e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.12801837383425412356107316013533713934091886254748e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.27896726012877622151287254340779424837571495657384e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.42592504593813016256394437035333320729928769135529e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.56883100040026405468976154826963286030986607276898e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.70765483748602800807348700535078666283426208025338e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.84239251923251354944511010208659148773285992261636e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.97306242841106189982720542754596269997825882200963e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.09970190928358551894563742001208165448751595738393e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.22236416601074576846725329522076603667792610069772e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.34111550192882688466795267571966376590108144590146e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.45603287874932276581779533145259002587664864932810e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.56720177227709322443036736236667681466823855576748e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.67471430008752600464561254217339060305933972234066e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.77866759641952347636729485475686950346612605081233e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.97630190201768046886571833364686445094779890812683e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.16093363694296970300414411298982724729800950998126e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.33340119589825831941744118804669746898228023705595e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.49453081078357569536385447775302397639174227456126e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.64512181956668895495628599815612188484872430606076e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.78593712196194817010501161391740217697424135219932e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(1.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.91769748027973587988050487247823119315646010788285e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.04107862044208783622223577234526499721208130809085e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.26517695571963502030771662018373539384937999898516e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.46273425405663094259919090350425049446731674380517e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(2.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.63758376080325935199241149369372014196158210794207e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.79296673358868394263845167623854238605663977145579e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.93161494811251760796024569455411548871678150466575e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.05582884526567009703468807923041602465523081207835e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(3.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.16754715036218405381848078882559199991501527210887e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.26840681332434061224397365001110201431714574161004e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(4.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.44288421192372625436336887550283154670590764164969e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.58804227080862094903927728732613232538756639404536e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(5.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.71029700052871041881284261613115047981162112317573e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.81440531622793612780308998937907666985394243647458e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(6.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.90394591759649482553720310181182407678855125658003e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7)), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.98164927634689821536641290454567745500263396394663e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(7.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.04962661826577677675750333226666504803112871241155e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist, static_cast<RealType>(8)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.10953085258825170758957718650744990707007748273190e-1), tolerance);
}

template<class RealType, int N>
void do_test_landau_ccdf() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.99292885943510821922577460956987921968387244585400e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.97976219107107743578462828534802143708944484064665e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.95133711508831090182424816199716847168754366703087e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.89864391355410554909853746345690390499645913443747e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.81249956670551506426259452453415462063661866437068e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.68552263735711166210040488568474322267626925864884e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.51365228574004752161386046723310073315471685913480e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.29676616651464601435382841864036570888682765674936e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.03839038959368266415766759995544660273259674059908e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.89552719679547124636609433966102633174791637726024e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.74479495186249055700256422545190933717725352402537e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.58721789395766996145000042983705551798904414770738e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.42385095802718294587318864087637567186448910606950e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.25575099660609438220876542722337458474197128791196e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.08395259840288681885342447490315170289721492473824e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.90944864890903933620175167418398079952667346407101e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.73317548003868415412396899412547833964836906255046e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.55600224990033155210143708386859850759316104280589e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.37872406785381471259899275014411401846418457171415e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.20205831869228520903152604174933916462523808972792e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.02664362746237164415797957494528236922518791090040e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.85304092873956810437175752936567055856354788920010e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.68173614889856753208609160981325142869126674376703e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(-0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.51314406729050474679104167088179431366818125495378e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.34761298487625202628055609797763667089439508997133e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.0625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.18542989150474476021386171696646255551749710231498e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.02682588217352113718763723238477506303886824319698e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.87198162616574587643892683986466286065908113745252e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.72103273987122377848712745659220575162428504342616e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.57407495406186983743605562964666679270071230864471e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.43116899959973594531023845173036713969013392723102e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.29234516251397199192651299464921333716573791974662e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.15760748076748645055488989791340851226714007738364e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.5625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.02693757158893810017279457245403730002174117799037e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.90029809071641448105436257998791834551248404261607e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.6875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.77763583398925423153274670477923396332207389930228e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.65888449807117311533204732428033623409891855409854e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.8125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.54396712125067723418220466854740997412335135067190e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.43279822772290677556963263763332318533176144423252e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(0.9375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.32528569991247399535438745782660939694066027765934e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.22133240358047652363270514524313049653387394918767e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.02369809798231953113428166635313554905220109187317e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.83906636305703029699585588701017275270199049001874e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.66659880410174168058255881195330253101771976294405e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.50546918921642430463614552224697602360825772543874e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.35487818043331104504371400184387811515127569393924e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.21406287803805182989498838608259782302575864780068e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1.875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.08230251972026412011949512752176880684353989211715e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.95892137955791216377776422765473500278791869190915e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.73482304428036497969228337981626460615062000101484e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.53726574594336905740080909649574950553268325619483e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(2.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.36241623919674064800758850630627985803841789205793e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20703326641131605736154832376145761394336022854421e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.06838505188748239203975430544588451128321849533425e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.94417115473432990296531192076958397534476918792165e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(3.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.83245284963781594618151921117440800008498472789113e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.73159318667565938775602634998889798568285425838996e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(4.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55711578807627374563663112449716845329409235835031e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41195772919137905096072271267386767461243360595464e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(5.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28970299947128958118715738386884952018837887682427e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18559468377206387219691001062092333014605756352542e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(6.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.09605408240350517446279689818817592321144874341997e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01835072365310178463358709545432254499736603605337e-1), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(7.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.50373381734223223242496667733334951968871287588448e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.90469147411748292410422813492550092929922517268105e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(9))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.89904178519375419646729692713027452766956304810423e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.08967063825624536726393015854367927398248532043928e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.42552960269528172637745151674852945361026757129572e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.87155991603325430279966781361705985837149401924358e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.40297179699419503759356245733027728020687579530781e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.00178775061898975362196226343808042189550214610720e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.65467961100606469227978737829290437667112287748795e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.35157264931262089761621934621402648953813567255367e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.84807181681993477724365847791930879377149510380933e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.44719141256297825765636022076703604777398281522436e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.12076533048217442252783357779404422496811288961954e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.84999992100002427148546326508663805148388380497970e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.62189540632143171631461472152513248987041904718627e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42718378313424593748302632910098340183096121887744e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.25908549106503559691958010353777133320615316433912e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.11253031965493064317003259449214452744697000964364e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.86945568960396455137037115431688883900622129241921e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.67614509428664592918239573763993132586708338097148e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51880213758692367036733937634308433186738240289411e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38828350661980875444222979837389265249992909781147e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.27829420024685114258602774435891775801837844690349e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18436085060190590919501220038943005140750783789755e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.10321854418155548568175979136702978995567636390268e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.03242932255189471157625857327944947615628400066886e-2), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.14901073563432515854150573798153430678589620957278e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.21304799594562916903235283213953478595851140543801e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.45021033455396813779806884621309261695988381836678e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.81662081970375351344432645398857600988792278643367e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.28205524656921301117582740567146809114392866561466e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.82502022536734166688286248837161828142031676980706e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.42981578145699797251974720633677573816663490439276e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.08470827750959570805565668839133674990776529552199e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.51875926400324844459822808369891291211052352126700e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(512))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.25243798184186193073508388843632555296804735961418e-3), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, static_cast<RealType>(1024))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.24222666597149382309278694960823422316591681354937e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 11))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.11546656864925721385400506911714668228522959074988e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55615661646764419866569094853675423840407090817980e-4), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 13))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.77642694561457848096607089076550607777923417509897e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.88702045347145575328162812652239839108265167027629e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 15))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.94318591305792177162247519775866592794374293269769e-5), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.71505353250968890581644704744607604904036466331858e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 17))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.85729142998688039060495209150963975787147384530557e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 18))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.42858279647060361837158514176420066606677711840386e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 19))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.21427464712490151175367196840586966659512986509712e-6), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.07132880360935675845351982773327267915998066915668e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 21))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03565265516245957916479321487698564090018898313690e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 22))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.51782323124974173858669998521646771346131838415381e-7), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 23))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.58910801622650317351735034158503382297981706223572e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.79455187330704471571840926063516392410676855026673e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.89727537800148613712401816975344760624742045931026e-8), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 26))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.48637543100068173667331269634864926724205314304352e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 27))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.74318733515442703770226252397963326193722512832281e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.37159356859216986241429458639296184020669300874754e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 29))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.18579675857518154364748998529259019922036982534955e-9), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 30))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.92898372613704146131858586390099325449659727514527e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 31))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.96449184577465200097456706940455493448424599625049e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.48224591841157074706460083370955443589105978681308e-10), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 34))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.70561478725616554450782995120929832467409189826579e-11), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 36))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.26403696230051280181276990208319690098486010067856e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 38))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.31600924018782655652761250713319794226217154394605e-12), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.79002310021356047142575775154740548981748412961642e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 42))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.44750577503651834833347404492486283164453776697661e-13), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 44))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.61876443758020639017543150875739203546579850550506e-14), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 46))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.04691109394324465958560412889011979187196424398413e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.26172777348533543325299108739056247374472601114618e-15), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 50))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.65431943371302795435372191153510741742082062236667e-16), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 52))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.41357985842823674325840669958475387008031322630868e-16), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 54))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.53394964607057868542015717421467433254478008250729e-17), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.83487411517643815597510900540151314745726776276537e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 58))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.20871852879410898385647064972035961074675685323943e-18), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 60))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.52179632198527209999982790078399565586115610945978e-19), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 62))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.38044908049631800172983942388255378445601347506470e-19), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.45112270124079498928544180507695627740193986743475e-20), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 68))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15695168827549686524997208584894563633442321098092e-21), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 72))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.34809480517218554065439979557746061380338130338956e-22), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 76))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.42559253232615962903750225978472507493251963237152e-24), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.26599533270384976814627294052855716278192786930198e-25), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 84))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.29124708293990610509133147452548575130220036516594e-26), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 88))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.05702942683744131568207851461222366485608388136168e-27), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 92))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.28564339177340082230129892190826718627845694793237e-28), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.03527119858375513938311820075537974687844953537787e-30), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 100))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.02204449911484696211444887297771791655738720297691e-31), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 104))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.13877781194677935132153054550953911471083914854259e-32), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 108))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.96173613246673709457595659093933570122399892106597e-33), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.22608508279171068410997286933691737981629848926873e-34), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 116))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.66303176744819177568733043335566577797931906017599e-36), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 120))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.78939485465511985980458152084728836560995745045025e-37), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 124))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.99337178415944991237786345052955511752869068868957e-38), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87085736509965619523616465658097194397480643905015e-39), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.87085736509965619523616465658097194397480643905015e-39), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 136))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.30803658242053201264126818976942165494933536744296e-42), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 144))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.85470179000802031743799538662868033396263440343802e-44), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 152))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.11511788672188293649921694790182825545415091296783e-46), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.35592924500735522070006620274151662286777695301906e-49), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 168))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70153486133099813308596336044590493080772537219152e-51), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 176))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.64662055207421145736704437674181613596767723512183e-54), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 184))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.59633615315398885053400170966477192811237391996946e-56), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.01419380982577689473984441783780153441889606248807e-58), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 200))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.96169456963194099507751725717891224382381274409403e-61), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 208))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.54753694126247695120215517858551259524367685316173e-63), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 216))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.04506617680655059063341866634965857517061270766301e-66), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.36135397531505882446617916654283538092602058893086e-68), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 232))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.22403896607444853307101236930795070674226792551118e-71), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 240))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.60314022112283145823086420676091824482119840840280e-73), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 248))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.40747664887610603837143133076598368938328062828235e-75), tolerance);
    BOOST_CHECK_CLOSE(cdf(complement(dist, ldexp(static_cast<RealType>(1), 256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.49795565967228921238840363580462378665343995422791e-78), tolerance);
}

template<class RealType, int N>
void do_test_landau_quantile_nearzero() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.03125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.37666142124664870427736212608782014348092232942703e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.0625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.16727296241754547290632950718657110865371011883488e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.09375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.01093317611810417245689707499918702758654056685650e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.77109518013577849065583862782160134594408514905936e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.15625)), BOOST_MATH_BIG_CONSTANT(RealType, N, -7.55147864363163697663604936395052484592948168261254e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.1875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.39830737061008092918841929339491480637487863539219e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.21875)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.28068260002115642528019835718799250663117082682413e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.25)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.17764764050720242897742634974454174978089714315837e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.28125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.07332710058680816601395586920882835843791948709923e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.3125)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.95448171270527198765810574858446724358448730774699e-1), tolerance);

    // Relative error decreases near the root.
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.34375)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.09096582988749397639311472333288802663134525410086e-2), tolerance * 10);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.74557416577759248536854968412795127813716796439183e-2), tolerance * 10);

    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.40625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.60865269786540938863996714417993425933643390360701e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.4375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.90647080532071673100911236876323489196027799942628e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.46875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.28310640852292004320545764804480585932279888826464e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.75630143945078214396279308922575172688187402052064e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.53125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.34751649957416497391423738312790099231331963193670e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.5625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.08338732735341567163440035550389980881822036530441e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.59375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.09977916217503813825408682237719484284367467701518e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31348919222343858173602105619413807214737739605647e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.65625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55537778016228176635489075791437025790508273018206e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.6875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.83358201486120130332744845371832465533289389791770e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.71875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15968268411007811908706240304578653239207877720217e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.75)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.55081568282045925871949387822806899205587760602861e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.78125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03356573722052416577951184486453626014326770622663e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.8125)), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.65170109428623528831428068089551728119930570543517e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.84375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.48307637103506875745328267812359172615004959210209e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68160868054034088524891526884683024288168516466432e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.90625)), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.60235856635301171960591513042024596858455255485505e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.9375)), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12983583770683756688253603663598582344177172936943e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, static_cast<RealType>(0.96875)), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19714287107110229779819680655029619622254635434578e1), tolerance);
}

template<class RealType, int N>
void do_test_landau_quantile_lower() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3)), BOOST_MATH_BIG_CONSTANT(RealType, N, -8.77109518013577849065583862782160134594408514905936e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.16727296241754547290632950718657110865371011883488e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.37666142124664870427736212608782014348092232942703e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.53937945687640569563878953898367418895418286968892e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.67193057806636817209385515556513905369159305215132e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.78348038398799867332294266481364804262899249535026e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10)), BOOST_MATH_BIG_CONSTANT(RealType, N, -1.96398712546539547900868300204740158406402253850740e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.10672834330580928387396530276881975373360375562766e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.22455234225296140347183338705883529117630663916504e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -16)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.32474749499506228416012679106564721359216355271060e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -20)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.48880222197195858123676152921728490355916489072920e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -24)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.62012266244208104039475013076613716084830146896364e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -28)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.72950770317262402197221324973838006547359087175177e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -32)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.82318656228158372073367735499500997206084370651970e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -40)), BOOST_MATH_BIG_CONSTANT(RealType, N, -2.97778747939490003780916437343292886429806927997697e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -48)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.10253941103391799610961378724535481625827036745191e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -56)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.20706965504048009454149883353144814083641366738928e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -64)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.29700011190686230364493911161520662289614047608888e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -80)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.44616534249558131319374859709618689440842790052762e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -96)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.56714397893479094507321490741760621098238204564655e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -112)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.66888826418737343076580628480419340009764391741662e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -128)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.75666995985336007747791649448887717847452667971341e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -160)), BOOST_MATH_BIG_CONSTANT(RealType, N, -3.90272386559500090814947503869007637264448398505091e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.02154690255519314578158695811138813197102888432118e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -224)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.12170243205222902851790251956181678240666087312573e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -256)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.20826069989721596260510558511263030379850705902299e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -320)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.35254674338923723387138280464388000015841039600555e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -384)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.47014796872616976009165833857464389146621180677518e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -448)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.56940486432189047495132801898571896454756873328884e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -512)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.65527239540648657446629479052874024150007745617425e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -640)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.79856256497052731627737920479094219620428520302497e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -768)), BOOST_MATH_BIG_CONSTANT(RealType, N, -4.91547732272180866354170476973276286391282782030491e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -896)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.01423042502756027207185677104980034013809957436307e0), tolerance);

    // The test is terminated because p = 0 after this in fp64.
    if(N <= 53){
        return;
    }

    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1024)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.09971143249822249471944441552701750744803726485204e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1280)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.24244558247619917128857214117595378487244291900164e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1536)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.35897805692044314674029465486888992071638929853610e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -1792)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.45745119510512195777968048015384189694395480185084e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -2048)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.54271778755494231572464179212263712867955955556373e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -2560)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.68514415447058604503373799640504720298538164608415e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3072)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.80146552760710346457341432134646648154958472048223e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -3584)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.89978436824284434538360953970051552537044157804319e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -4096)), BOOST_MATH_BIG_CONSTANT(RealType, N, -5.98493298246627952401490656857159297529934857258768e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -5120)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.12719032649420608124001447392893816734788637468278e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -6144)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.24339602515596353391422898038238569270302761832424e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -7168)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.34163046670299704746360546090336718985578875063595e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -8192)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.42671464308364892089984144203590287404801291341155e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -10240)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.56887983067842563164669431493815471028291272019257e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -12288)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.68502258391108555173104319503169367212234002338911e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(dist, ldexp(static_cast<RealType>(1), -14336)), BOOST_MATH_BIG_CONSTANT(RealType, N, -6.78321117119788305968192558554365316820805283962570e0), tolerance);
}

template<class RealType, int N>
void do_test_landau_quantile_upper() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //
    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist(static_cast<RealType>(0), static_cast<RealType>(1));

    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.75630143945078214396279308922575172688187402052064e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.46875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.34751649957416497391423738312790099231331963193670e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.4375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.08338732735341567163440035550389980881822036530441e-1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.40625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.09977916217503813825408682237719484284367467701518e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.31348919222343858173602105619413807214737739605647e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.34375))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.55537778016228176635489075791437025790508273018206e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.3125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.83358201486120130332744845371832465533289389791770e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.28125))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.15968268411007811908706240304578653239207877720217e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.55081568282045925871949387822806899205587760602861e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.21875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.03356573722052416577951184486453626014326770622663e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.1875))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.65170109428623528831428068089551728119930570543517e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, static_cast<RealType>(0.15625))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.48307637103506875745328267812359172615004959210209e0), tolerance);

    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -3))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.68160868054034088524891526884683024288168516466432e0), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -4))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.12983583770683756688253603663598582344177172936943e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -5))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.19714287107110229779819680655029619622254635434578e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -6))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.28089946846201448479186405990829330152038273940336e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -7))), BOOST_MATH_BIG_CONSTANT(RealType, N, 8.40066013407733304728124190173163568113870838023174e1), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -8))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.65941656579816517264930679482280545790703168918131e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -10))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.55753077855274500558851746639632584040672341383365e2), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -12))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.61233279175261252785344820794971528959731572291875e3), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -14))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.04359994051668217545134727183452999841836322543240e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -16))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.17280170760481128545941744972976378785527707571987e4), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -20))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.67552483213163052181906697701099131817014739251016e5), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -24))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.06807174647511043368265795578221114430723661526053e7), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -28))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.70891330693062153503879914631960898643362032945242e8), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -32))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.73426111586976542228936210986127280848086481329720e9), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -40))), BOOST_MATH_BIG_CONSTANT(RealType, N, 6.99970842207360156603062974124919047698394307869345e11), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -48))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.79192535600728707050158319385235151176991681631485e14), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -56))), BOOST_MATH_BIG_CONSTANT(RealType, N, 4.58732891137812933185516807516215074813520642490265e16), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -64))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.17435620131280049336696928005840807489266101470318e19), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -80))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.69626080092356929518666088104130567689669972090860e23), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -96))), BOOST_MATH_BIG_CONSTANT(RealType, N, 5.04382147849327037329330237397632338076523897478536e28), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -112))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.30551884414534967184149864106941373195910184715021e33), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -128))), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.16630482969909636093804454941121895928783047320959e38), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -160))), BOOST_MATH_BIG_CONSTANT(RealType, N, 9.30420839672446839098151322191224148323438073311282e47), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -192))), BOOST_MATH_BIG_CONSTANT(RealType, N, 3.99612707791001852622513406287046677525461975485236e57), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -224))), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.71632351102835736008910691332442626839731739200064e67), tolerance);
    BOOST_CHECK_CLOSE(quantile(complement(dist, ldexp(static_cast<RealType>(1), -256))), BOOST_MATH_BIG_CONSTANT(RealType, N, 7.37155334922269019018360983857591746072979653277477e76), tolerance);
}

template<class RealType, int N>
void do_test_landau_locscale_param() {
    //
    // Basic sanity checks, tolerance is 3 epsilon
    // expressed as a percentage:
    //

    BOOST_MATH_STD_USING
    RealType tolerance = boost::math::tools::epsilon<RealType>() * 100 * 3;

    std::cout << "Testing acurracy[%]: " << tolerance << std::endl;

    landau_distribution<RealType> dist_0_1(static_cast<RealType>(0), static_cast<RealType>(1));
    landau_distribution<RealType> dist_1_3(static_cast<RealType>(1), static_cast<RealType>(3));
    landau_distribution<RealType> dist_0_pihalf(static_cast<RealType>(0), boost::math::constants::pi<RealType>() / 2);

    BOOST_CHECK_CLOSE(entropy(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.3726364400044818244844049010588577710), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_1_3), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.3726364400044818244844049010588577710) + log(static_cast<RealType>(3)), tolerance);
    BOOST_CHECK_CLOSE(entropy(dist_0_pihalf), BOOST_MATH_BIG_CONSTANT(RealType, N, 2.8242191452939366892106001309537399145), tolerance);

    BOOST_CHECK_CLOSE(median(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.57563014394507821439627930892257517269), tolerance);
    BOOST_CHECK_CLOSE(
        median(dist_1_3),
        (1 + 3 * (BOOST_MATH_BIG_CONSTANT(RealType, N, 0.57563014394507821439627930892257517269) + 2 / boost::math::constants::pi<RealType>() * log(static_cast<RealType>(3)))),
        tolerance
    );
    BOOST_CHECK_CLOSE(median(dist_0_pihalf), BOOST_MATH_BIG_CONSTANT(RealType, N, 1.3557804209908013250320928093906509105), tolerance);

    BOOST_CHECK_CLOSE(mode(dist_0_1), BOOST_MATH_BIG_CONSTANT(RealType, N, -0.42931452986133525016556463510885028346), tolerance);
    BOOST_CHECK_CLOSE(
        mode(dist_1_3),
        (1 + 3 * (BOOST_MATH_BIG_CONSTANT(RealType, N, -0.42931452986133525016556463510885028346) + 2 / boost::math::constants::pi<RealType>() * log(static_cast<RealType>(3)))),
        tolerance
    );
    BOOST_CHECK_CLOSE(mode(dist_0_pihalf), BOOST_MATH_BIG_CONSTANT(RealType, N, -0.222782981256408504061824283124805665631673572953417648794046), tolerance);

    BOOST_CHECK_CLOSE(pdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.16353124086802260347581305180210465276342), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.09034495984298569669047391552324642172044), tolerance);
    BOOST_CHECK_CLOSE(pdf(dist_0_pihalf, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.14520663709640194253543670745173917702186), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.57786675964195234763672948547568695034661), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.17119311431882309272302517476020685228892), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_pihalf, static_cast<RealType>(1)), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.45101809281952585982591839302215356640746), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_pihalf, quantile(dist_0_pihalf, static_cast<RealType>(0.25))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.25), tolerance);

    BOOST_CHECK_CLOSE(cdf(dist_0_1, quantile(dist_0_1, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_1_3, quantile(dist_1_3, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
    BOOST_CHECK_CLOSE(cdf(dist_0_pihalf, quantile(dist_0_pihalf, static_cast<RealType>(0.75))), BOOST_MATH_BIG_CONSTANT(RealType, N, 0.75), tolerance);
}

BOOST_AUTO_TEST_CASE(landau_pdf_fp64)
{
    do_test_landau_pdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_pdf_fp64_std)
{
    do_test_landau_pdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_pdf_fp128)
{
    do_test_landau_pdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_cdf_fp64)
{
    do_test_landau_cdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_cdf_fp64_std)
{
    do_test_landau_cdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_cdf_fp128)
{
    do_test_landau_cdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_ccdf_fp64)
{
    do_test_landau_ccdf<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_ccdf_fp64_std)
{
    do_test_landau_ccdf<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_ccdf_fp128)
{
    do_test_landau_ccdf<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_quantile_nearzero_fp64)
{
    do_test_landau_quantile_nearzero<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_quantile_nearzero_fp64_std)
{
    do_test_landau_quantile_nearzero<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_quantile_nearzero_fp128)
{
    do_test_landau_quantile_nearzero<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_quantile_lower_fp64)
{
    do_test_landau_quantile_lower<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_quantile_lower_fp64_std)
{
    do_test_landau_quantile_lower<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_quantile_lower_fp128)
{
    do_test_landau_quantile_lower<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_quantile_upper_fp64)
{
    do_test_landau_quantile_upper<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_quantile_upper_fp64_std)
{
    do_test_landau_quantile_upper<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_quantile_upper_fp128)
{
    do_test_landau_quantile_upper<cpp_bin_float_quad, 113>();
}
#endif

BOOST_AUTO_TEST_CASE(landau_locscale_fp64)
{
    do_test_landau_locscale_param<double, 53>();
}

#ifdef __STDCPP_FLOAT64_T__
BOOST_AUTO_TEST_CASE(landau_locscale_fp64_std)
{
    do_test_landau_locscale_param<std::float64_t, 53>();
}
#endif

#ifndef BOOST_MATH_HAS_GPU_SUPPORT
BOOST_AUTO_TEST_CASE(landau_locscale_fp128)
{
    do_test_landau_locscale_param<cpp_bin_float_quad, 113>();
}
#endif

