/***************************************************************************
                            DCOPClient.h -  description
                             -------------------
    begin                : Fri Mar  8 13:55:08 2002
    copyright            : (C) 2000-2002 Lost Highway Ltd. All rights reserved.
    email                : Lost_Highway@tipitina.demon.co.uk
    generated by         : duke@tipitina on Fri Mar  8 13:55:08 2002, using kalyptus 0.9.
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 *                                                                         *
 ***************************************************************************/

#ifndef DCOPCLIENT_H
#define DCOPCLIENT_H

#include <Foundation/NSObject.h>
#include <Foundation/NSString.h>
#include <qtobjc/QObject.h>

/**

 Inter-process communication and remote procedure calls
 for KDE applications.

 This class provides IPC and RPC for KDE applications.  Usually you
 will not have to instantiate one yourself because @ref KApplication
 contains a method to return a pointer to a DCOPClient object which
 can be used for your whole application.

 Before being able to send or receive any DCOP messages, you will have
 to attach your client object to the DCOP server, and then register
 your application with a specific name. See @ref -attach
 and @ref -registerAs for
 more information.

 Data to be sent should be serialized into a @ref QDataStream which was
 initialized with the @ref QByteArray that you actually intend to send
 the data in.  An example of how you might do this:

 <pre>
   QByteArray data;
   QDataStream arg(data, IO_WriteOnly);
   arg << QString("This is text I am serializing");
   -[client send]("someApp", "someObject", "someFunction", arg);
 </pre>

 @author Preston Brown <pbrown@kde.org>, Matthias Ettrich <ettrich@kde.org>
 
 @short 
 Inter-process communication and remote procedure calls
 for KDE applications.
*/

@class DCOPClientTransaction; 


@interface DCOPClient : QObject 

/** 
 Constructs a new DCOP client, but does not attach to any server.  
*/
- init;

/** 
 Cleans up any open connections and dynamic data.
   
*/
- (void) dealloc;

/** 
 Attaches to the DCOP server.

 If the connection was already attached,
 the connection will be re-established with the current server address.

 Naturally, only attached application can use DCOP services.

 If a @ref QApplication object exists then client registers itself as
 @ref -[QApplication name] + "-" + <pid>.
 If no @ref QApplication object exists the client registers itself as
 "anonymous".

 If you want to register differently, you should use @ref -registerAs
 instead.

*/
- (BOOL) attach;

/** 

 Internal function for @ref KUniqueApplication to register the @ref DCOPClient
 with the application in case the application didn't exist at the
 time the @ref DCOPClient was created.
   
*/
- bindToApp;

/** 
 Detaches from the DCOP server.
   
*/
- (BOOL) detach;

/** 
 Returns whether or not the client is attached to the server.
   
*/
- (BOOL) isAttached;

/** 
 Returns whether the client is attached to a server owned by
 another user.
   
*/
- (BOOL) isAttachedToForeignServer;

/** 
 Returns whether the client handles incoming calls.
   
*/
- (BOOL) acceptCalls;

/** 
 Specify whether the client should accept incoming calls.
 By default clients accept incoming calls, but not when connected
 to a foreign server.
   
*/
- setAcceptCalls: (BOOL)b;

/** 
 Registers at the DCOP server.

 If the application was already registered,
 the registration will be re-done with the new @ref appId.

 @p appId is a @p unique application/program id that the server
 will use to associate requests with. If there is already an application
 registered with the same name, the server will add a number to the
 id to unify it. If @p addPID is true, the PID of the current process
 will be added to id.

 Registration is necessary if you want to allow other clients to talk
 to you.  They can do so using your @p appId as first parameter
 for @ref -send or @ref -call. If you just want to talk to
  other clients, you
 do not need to register at the server. In that case @ref -attach is
 enough.
 It will implicitly register you as "anonymous".

*/
- (NSString *) registerAs: (NSString *)appId addPID: (BOOL)addPID;

/** 
 Returns whether or not the client is registered at the server.
   
*/
- (BOOL) isRegistered;

/** 
 Returns the current app id or a null string if the application
 hasn't yet been registered.
   
*/
- (NSString *) appId;

/** 

*/
- (int) socket;

/** 
 Temporarily suspends processing of DCOP events.
 This can be usefull if you need to show e.g. a dialog before
 your application is ready to accept DCOP requests. Normally the
 dialog would start an event loop and in this event loop DCOP
 requests will be handled.

 Be aware that not responding to DCOP requests may cause other
 programs that want to communicate with your application, to hang.
   
*/
- suspend;

/** 
 Resumes the processing of DCOP events.
 See @ref -suspend.
   
*/
- resume;

/** 
 Sends a data block to the server.

*/
- (BOOL) send: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun data: (QByteArray*)data;

/** 
 This function acts exactly the same as the above, but the data
 parameter can be specified as a @ref QString for convenience.
   
*/
- (BOOL) send: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun dataString: (NSString *)data;

/** 
 Performs a synchronous send and receive.

  The parameters are the same as for send, with the exception of
  another @ref QByteArray being provided for results to be
  (optionally) returned in.

 A call blocks the application until the process receives the
 answer.

 If @p useEventLoop is true, a local event loop will be started after
 1/10th of a second in order to keep the user interface updated
 (by processing paint events and such) until an answer is received.

*/
- (BOOL) call: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun data: (QByteArray*)data replyType: (NSString *)replyType replyData: (QByteArray*)replyData useEventLoop: (BOOL)useEventLoop;
- (BOOL) call: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun data: (QByteArray*)data replyType: (NSString *)replyType replyData: (QByteArray*)replyData ;

/** 
 Searches for an object which matches a criteria.

 findObject calls @p remFun in the applications and objects identified
 by @p remApp and @p remObj until @p remFun returns true. The name of
 the application and object that returned true are returned in
 @p foundApp and @p foundObj respectively.

 If @p remFun is empty a default function is called in the object
 which always returns @p true.

 A findObject blocks the application until the process receives the
 answer.

 If @p useEventLoop is true, a local event loop will be started after
 1/10th of a second in order to keep the user interface updated
 (by processing paint events and such) until an answer is received.

*/
- (BOOL) findObject: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun data: (QByteArray*)data foundApp: (NSString *)foundApp foundObj: (NSString *)foundObj useEventLoop: (BOOL)useEventLoop;
- (BOOL) findObject: (NSString *)remApp remObj: (NSString *)remObj remFun: (NSString *)remFun data: (QByteArray*)data foundApp: (NSString *)foundApp foundObj: (NSString *)foundObj ;

/** 
 Emits @p signal as DCOP signal from object @p object with @p data as
 arguments.
   
*/
- emitDCOPSignal: (NSString *)object signal: (NSString *)signal data: (QByteArray*)data;
- emitDCOPSignal: (NSString *)signal data: (QByteArray*)data;

/** 
 Connects to a DCOP signal.

*/
- (BOOL) connectDCOPSignal: (NSString *)sender senderObj: (NSString *)senderObj signal: (NSString *)signal receiverObj: (NSString *)receiverObj slot: (NSString *)slot Volatile: (BOOL)Volatile;
- (BOOL) connectDCOPSignal: (NSString *)sender signal: (NSString *)signal receiverObj: (NSString *)receiverObj slot: (NSString *)slot Volatile: (BOOL)Volatile;

/** 
 Disconnects a DCOP signal.

 A special case is when both sender & signal are empty. In this
 case all connections related to @p receiverObj in the current client
 are disconnected. (Both connections from as well as to this object!)

*/
- (BOOL) disconnectDCOPSignal: (NSString *)sender senderObj: (NSString *)senderObj signal: (NSString *)signal receiverObj: (NSString *)receiverObj slot: (NSString *)slot;
- (BOOL) disconnectDCOPSignal: (NSString *)sender signal: (NSString *)signal receiverObj: (NSString *)receiverObj slot: (NSString *)slot;

/** 
 Reimplement this function to handle app-wide function calls unassociated w/an object.

 Note that @p fun is normalized. See @ref -normalizeFunctionSignature.

 If you do not want to reimplement this function for whatever reason,
 you can also use a default object  or a @ref DCOPObjectProxy.

*/
- (BOOL) process: (NSString *)fun data: (QByteArray*)data replyType: (NSString *)replyType replyData: (QByteArray*)replyData;

/** 
 Delays the reply of the current function call
 until @ref -endTransaction is called.

 This allows a server to queue requests.

 Note: Should be called from inside @ref -process only!
   
*/
- (DCOPClientTransaction*) beginTransaction;

/** 
 Sends the delayed reply of a function call.
   
*/
- endTransaction: (DCOPClientTransaction*)arg1 replyType: (NSString *)replyType replyData: (QByteArray*)replyData;

/** 
 Test whether the current function call is delayed.

 Note: Should be called from inside @ref -process only!

*/
- (int) transactionId;

/** 
 Checks whether @p remApp is registered with the @ref DCOPServer.

*/
- (BOOL) isApplicationRegistered: (NSString *)remApp;

/** 
 Retrieves the list of all currently registered applications
 from dcopserver.
   
*/
- (NSArray *) registeredApplications;

/** 
 Retrieves the list of objects of the remote application @p remApp.
   
*/
- (NSArray *) remoteObjects: (NSString *)remApp ok: (int*)ok;

/** 
 Retrieves the list of interfaces of the remote object @p remObj
 of application @p remApp.
  
*/
- (NSArray *) remoteInterfaces: (NSString *)remApp remObj: (NSString *)remObj ok: (int*)ok;

/** 
 Retrieves the list of functions of the remote object @p remObj
 of application @p remApp
  
*/
- (NSArray *) remoteFunctions: (NSString *)remApp remObj: (NSString *)remObj ok: (int*)ok;

/** 
 Receives a DCOPSend or DCOPCall message from the server.

   
*/
- (BOOL) receive: (NSString *)app obj: (NSString *)obj fun: (NSString *)fun data: (QByteArray*)data replyType: (NSString *)replyType replyData: (QByteArray*)replyData;

/** 
 Receives a @p DCOPFind message from the server.

   
*/
- (BOOL) find: (NSString *)app obj: (NSString *)obj fun: (NSString *)fun data: (QByteArray*)data replyType: (NSString *)replyType replyData: (QByteArray*)replyData;

/** 
 Returns the @p appId of the last application that talked to us.
   
*/
- (NSString *) senderId;

/** 
 Installs object @p objId as application-wide default object.

 All app-wide messages that have not been processed by the dcopclient
 will be send further to @p objId.
    
*/
- setDefaultObject: (NSString *)objId;

/** 
 Returns the current default object or an empty string if no object is
 installed as default object.

 A default object receives application-wide messages that have not
 been processed by the DCOPClient itself.
     
*/
- (NSString *) defaultObject;

/** 
 Enables / disables the @ref -applicationRegistered /
 @ref -applicationRemoved signals.
 Note that a counter is maintained about how often this method
 was called. If this method is called twice with @p enabled set to
 true, notifications will be enabled until it was called with
 @p enabled set to false as often.

 They are disabled by default.  
*/
- setNotifications: (BOOL)enabled;

/** 
 Tells the dcopserver to treat the client as daemon client, not
 as regular client.
 If the number of regular clients drops down to zero, the
 dcopserver will emit a KDE termination signal after 10
 seconds.
   
*/
- setDaemonMode: (BOOL)daemonMode;

/** 
 Sets the address of a server to use upon attaching.

 If no server address is ever specified, attach will try its best to
 find the server anyway.
   
*/
+ setServerAddress: (NSString *)addr;

/** 
 Normalizes the function signature @p fun.

 A normalized signature doesn't contain any unnecessary whitespace
 anymore. The remaining whitespace consists of single blanks only (0x20).

 Example for a normalized signature:
 <pre>
   "someFunction(QString,int)"
 </pre>

 When using @ref -send or @ref -call, normalization is done
 automatically for you.

   
*/
+ (NSString *) normalizeFunctionSignature: (NSString *)fun;

/** 
 Returns the application's main dcop client. The main client can
 be used by objects that do not have any specific access to a dcop
 client. In KDE applications, the main client usually is the same
 as -[KAppliction dcopClient].
   
*/
+ (DCOPClient*) mainClient;

/** 
 Sets the application's main dcop client. The main client can
 be used by objects that do not have any specific access to a dcop
 client. In KDE applications, the main client usually is the same
 as -[KAppliction dcopClient].
   
*/
+ setMainClient: (DCOPClient*)arg1;

/** 

    
*/
+ emergencyClose;

/** 

 Provides information about the last DCOP call for debugging purposes.
    
*/
+ (NSString *) postMortemSender;
+ (NSString *) postMortemObject;
+ (NSString *) postMortemFunction;

/** 
 File with information how to reach the dcopserver.

*/
+ (NSString *) dcopServerFile: (NSString *)hostname;

/** 
 For backwards compatibility with KDE 2.x
    
*/
+ (NSString *) dcopServerFileOld: (NSString *)hostname;
- processSocketData: (int)socknum;
@end

/** Override these methods in subclasses of DCOPClient to customise event handling behaviour */
@protocol DCOPClientEventHandling
- childEvent:(QChildEvent*) arg1;
- customEvent:(QCustomEvent*) arg1;
- timerEvent:(QTimerEvent*) arg1;
@end


#endif
