/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <qpushbutton.h>
#include <qlayout.h>
#include <qcursor.h>

#include <klistview.h>
#include <ksimpleconfig.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <kpopupmenu.h>
#include <kdebug.h>
#include <kmessagebox.h>

#include "netconfig.h"
#include "netconfig.moc"
#include "netdialog.h"

NetConfig::NetConfig(KSim::PluginObject *parent, const char *name)
   : KSim::PluginPage(parent, name)
{
  m_yes = i18n("yes");
  m_no = i18n("no");

  QVBoxLayout *mainLayout = new QVBoxLayout(this);
  mainLayout->setSpacing(6);

  usingBox = new KListView(this);
  usingBox ->addColumn(i18n("Interface"));
  usingBox ->addColumn(i18n("Timer"));
  usingBox ->addColumn(i18n("Commands"));
  usingBox ->addColumn(i18n("Graph"));
  usingBox ->addColumn(i18n("Load Label"));
  connect(usingBox, SIGNAL(contextMenu(KListView *,
     QListViewItem *, const QPoint &)),
     SLOT(menu(KListView *, QListViewItem *, const QPoint &)));
  connect(usingBox, SIGNAL(doubleClicked(QListViewItem *)), 
     SLOT(modifyItem(QListViewItem *)));
  mainLayout->addWidget(usingBox);

  layout = new QHBoxLayout;
  layout->setSpacing(6);

  QSpacerItem *spacer = new QSpacerItem(20, 20,
     QSizePolicy::Expanding, QSizePolicy::Minimum);
  layout->addItem(spacer);

  insertButton = new QPushButton(this);
  insertButton->setText(i18n("Add..."));
  connect(insertButton, SIGNAL(clicked()), SLOT(showNetDialog()));
  layout->addWidget(insertButton);

  modifyButton = new QPushButton(this);
  modifyButton->setText(i18n("Modify..."));
  connect(modifyButton, SIGNAL(clicked()), SLOT(modifyCurrent()));
  layout->addWidget(modifyButton);

  removeButton = new QPushButton(this);
  removeButton->setText(i18n("Remove..."));
  connect(removeButton, SIGNAL(clicked()), SLOT(removeCurrent()));
  layout->addWidget(removeButton);

  mainLayout->addLayout(layout);
}

NetConfig::~NetConfig()
{
}

void NetConfig::saveConfig()
{
  qHeapSort(m_deviceList);

  int i = 0;
  DeviceList::Iterator it;
  for(it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    config()->setGroup("Net");
    config()->setGroup("device-" + QString::number(i));
    config()->writeEntry("deviceName", (*it).deviceName);
    config()->writeEntry("showTimer", (*it).showTimer);
    config()->writeEntry("deviceFormat", (*it).timerFormat);
    config()->writeEntry("commands", (*it).commandsEnabled);
    config()->writeEntry("cCommand", (*it).cCommand);
    config()->writeEntry("dCommand", (*it).dCommand);
    config()->writeEntry("showGraph", (*it).showGraph);
    config()->writeEntry("showLabel", (*it).showLabel);
    i++;
  }

  config()->setGroup("Net");
  config()->writeEntry("deviceAmount", m_deviceList.count());
}

void NetConfig::readConfig()
{
  usingBox->clear();
  m_deviceList.clear();
  config()->setGroup("Net");
  int deviceAmount = config()->readNumEntry("deviceAmount");
  int i = 0;
  for (i = 0; i < deviceAmount; ++i) {
    if (config()->hasGroup("device-" + QString::number(i))) {
      config()->setGroup("device-" + QString::number(i));

      m_deviceList.append(NetDevice(
         config()->readBoolEntry("showTimer"),
         config()->readEntry("deviceFormat"),
         config()->readBoolEntry("commands"), 
         config()->readEntry("cCommand"),
         config()->readEntry("dCommand"), 
         config()->readBoolEntry("showGraph"),
         config()->readBoolEntry("showLabel"),
         config()->readEntry("deviceName")));

      (void) new QListViewItem(usingBox, 
         config()->readEntry("deviceName"),
         boolToString(config()->readBoolEntry("showTimer")),
         boolToString(config()->readBoolEntry("commands")),
         boolToString(config()->readBoolEntry("showGraph")),
         boolToString(config()->readBoolEntry("showLabel")));
    }
  }
}

void NetConfig::menu(KListView *, QListViewItem *item, const QPoint &)
{
  aboutMenu = new QPopupMenu(this);

  if (item) {
    aboutMenu->insertItem(i18n("&Add net device"), 3);
    aboutMenu->insertItem(i18n("&Modify '%1'").arg(item->text(0)), 2);
    aboutMenu->insertItem(i18n("&Remove '%1'").arg(item->text(0)), 1);
  }
  else {
    aboutMenu->insertItem(i18n("&Add net device"), 3);
    aboutMenu->insertItem(i18n("&Modify..."), 2);
    aboutMenu->insertItem(i18n("&Remove..."), 1);
    aboutMenu->setItemEnabled(1, false);
    aboutMenu->setItemEnabled(2, false);
  }

  switch (aboutMenu->exec(QCursor::pos())) {
    case 1:
      removeItem(item);
      break;
    case 2:
      modifyItem(item);
      break;
    case 3:
      showNetDialog();
      break;
  }

  delete aboutMenu;
}

void NetConfig::modifyItem(QListViewItem *item)
{
  if (!item)
    return;

  netDialog = new NetDialog(this);
  DeviceList::Iterator it, netDevice;
  for(it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if ((*it).deviceName == item->text(0)) {
      netDevice = it;
      netDialog->setDeviceName((*it).deviceName);
      netDialog->setShowTimer((*it).showTimer);
      netDialog->setFormat((*it).timerFormat);
      netDialog->setShowCommands((*it).commandsEnabled);
      netDialog->setCCommand((*it).cCommand);
      netDialog->setDCommand((*it).dCommand);
      netDialog->setShowGraph((*it).showGraph);
      netDialog->setShowLoadLabel((*it).showLabel);
      break;
    }
  }

  netDialog->exec();
  if (netDialog->okClicked()) {
    m_deviceList.remove(netDevice);
    m_deviceList.append(NetDevice(netDialog->timer(),
       netDialog->format(), netDialog->commands(),
       netDialog->cCommand(),
       netDialog->dCommand(),
       netDialog->graph(),
       netDialog->loadLabel(),
       netDialog->deviceName()));

    item->setText(0, netDialog->deviceName());
    item->setText(1, boolToString(netDialog->timer()));
    item->setText(2, boolToString(netDialog->commands()));
    item->setText(3, boolToString(netDialog->graph()));
    item->setText(4, boolToString(netDialog->loadLabel()));
  }

  delete netDialog;
}

void NetConfig::removeItem(QListViewItem *item)
{
  if (!item)
    return;

  int result = KMessageBox::questionYesNo(0, i18n("Are you sure you "
     "want to remove the net interface '%1' ?").arg(item->text(0)));

  if (result == KMessageBox::No)
    return;

  int i = 0;
  DeviceList::Iterator it;
  for(it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if (item->text(0) == (*it).deviceName) {
      m_deviceList.remove(it);
      if (config()->deleteGroup("device-" + QString::number(i)))
        kdDebug() << "device-" << i << " was deleted" << endl;
      break;
    }
    i++;
  }

  delete item;
}

void NetConfig::removeCurrent()
{
  removeItem(usingBox->currentItem());
}

void NetConfig::modifyCurrent()
{
  modifyItem(usingBox->currentItem());
}

void NetConfig::showNetDialog()
{
  netDialog = new NetDialog(this, "netDialog");
  netDialog->exec();
  if (netDialog->okClicked())
    getStats();

  delete netDialog;
}

void NetConfig::getStats()
{
  DeviceList::ConstIterator it;
  for (it = m_deviceList.begin(); it != m_deviceList.end(); ++it) {
    if ((*it).deviceName == netDialog->deviceName()) {
      KMessageBox::sorry(0, i18n("You already have a network "
        "interface by this name. please select a different interface"));
      return;
    }
  }

  m_deviceList.append(NetDevice(netDialog->timer(),
     netDialog->format(), 
     netDialog->commands(), 
     netDialog->cCommand(),
     netDialog->dCommand(), 
     netDialog->graph(),
     netDialog->loadLabel(),
     netDialog->deviceName()));

  (void) new QListViewItem(usingBox,
     netDialog->deviceName(),
     boolToString(netDialog->timer()),
     boolToString(netDialog->commands()),
     boolToString(netDialog->graph()),
     boolToString(netDialog->loadLabel()));
}

const QString &NetConfig::boolToString(bool value) const
{
  if (value)
    return m_yes;

  return m_no;
}
