/******************************************************************************
  File:     $Id: pclcomp.h,v 1.6 1999-09-02 08:06:08+02 Martin Rel $
  Contents: Header file for a PCL compression routine
  Author:   Martin Lottermoser, Metzgerfeldweg 9, 85737 Ismaning, Germany.
	    E-mail: martin.lottermoser@mch20.sbs.de

*******************************************************************************
*									      *
*	Copyright (C) 1997 by Martin Lottermoser			      *
*	All rights reserved						      *
*									      *
*******************************************************************************

  This file together with the implementation file pclcomp.c forms the "pclcomp
  library". It can be used under the terms of the GNU Library General Public
  License (LGPL), Version 2 (June 1991).

  USE IT AT YOUR OWN RISK.

******************************************************************************/

#ifndef _pclcomp_h	/* Inclusion protection */
#define _pclcomp_h

/* Configuration management identification */
#pragma ident "@(#)$Id: pclcomp.h,v 1.6 1999-09-02 08:06:08+02 Martin Rel $"

/*****************************************************************************/

/* PCL codes for raster graphics compression methods.
   These are the codes and methods defined in Hewlett-Packard's "Technical
   Reference Guide for the HP DeskJet 500 Series Printers" (1994).
*/
typedef enum {
  pcl_cm_none = 0,	/* Unencoded, non-compressed method */
  pcl_cm_rl = 1,	/* Run-Length Encoding */
  pcl_cm_tiff = 2,	/* Tagged Image File Format (TIFF) */
  pcl_cm_delta = 3,	/* Delta Row Compression */
  pcl_cm_crdr = 9	/* Compressed Replacement Delta Row Encoding */
} pcl_compression;


/* 'pcl_octet' is used to store 8-bit bytes as unsigned numbers. */
typedef unsigned char pcl_octet;


/* Octet strings */
typedef struct {
  pcl_octet *str;	/* Data area for storing the octet string. */
  int length;		/* Length of 'str' in octets (non-negative).
			   This is deliberately not "unsigned" in order to
			   catch overflows. */
} pcl_octet_string;
/* A "valid" pcl_octet_string is one where either 'length' is zero, or
   'length' is positive and 'str' is non-NULL and points to a storage
   area of at least 'length' octets. Invalid octet strings are neither accepted
   nor returned in the interface defined here.
   A "zero" pcl_octet_string is one where 'length' is zero.
*/


/******************************************************************************

  Function: pcl_compress

  This function compresses an octet string using the compression algorithm
  specified by 'method'.

  The arguments 'in' and 'out' must be non-NULL. They point to the data to be
  compressed ('in->length' octets starting at 'in->str') and the area to which
  the compressed result should be written (at most 'out->length' octets
  starting at 'out->str'). If '*in' and '*out' are both non-zero (see
  definitions above), their storage areas ('in->str' to
  'in->str + in->length - 1' and 'out->str' to 'out->str + out->length - 1')
  should not overlap.

  If 'method' refers to a method entailing "vertical" compression, i.e.
  compression with respect to an octet string previously sent, 'prev' must
  point to this previous string. This is the case for methods 3 and 9.
  The variable is ignored otherwise and may be NULL. If it is needed, it should
  not overlap with the storage area of '*out'.

  The function returns a non-zero value in case of insufficient space in '*out'.
  In that situation, part of 'out->str' may have been overwritten already.
  Otherwise, a value of zero is returned and 'out->length' is set to the number
  of octets the compressed result occupies in 'out->str'.

******************************************************************************/

extern int pcl_compress(pcl_compression method, const pcl_octet_string *in,
  const pcl_octet_string *prev, pcl_octet_string *out);

/*****************************************************************************/

#endif	/* Inclusion protection */
