/* Yo Emacs, this -*- C++ -*-

  Copyright (C) 1999,2000,2001 Jens Hoefkens
  jens@hoefkens.com
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  $Log: kfibschat.cpp,v $
  Revision 1.7  2001/02/10 15:29:34  hoefkens
  Proper use of i18n() allows for better translations.

  Revision 1.6  2001/01/23 08:42:03  kiefer
  a lot of i18n fixes ( I18N_NOOP -> i18n )

  Revision 1.5  2001/01/06 23:31:17  hoefkens
  Lots of small changes and fixed. Big thing is a rewrite of the join menu
  in kbgfibs.

  
*/

/*
  TODO
  ----

  - the "say" command should only be visible during actual games
    (same for kibitz)
  - there should be a better way of selecting users for the 
    context menu
  - make the colors a user setting
  - handle the message "You say to yourself:" properly
  - beep on private messages
  - get the splitter size right

*/

#include "kfibschat.moc"
#include "kfibschat.h"

#include <qlayout.h>
#include <qlabel.h>
#include <qmessagebox.h> 
#include <klocale.h>
#include <qpopupmenu.h>
#include <qgroupbox.h>
#include <kconfig.h>
#include <kcmenumngr.h>
#include <qregexp.h>
#include <qvaluelist.h> 
#include <kcmenumngr.h>
#include <kfontdialog.h>
#include <qfont.h>
#include <qwhatsthis.h>
#include <kcompletion.h>
#include <kcompletionbox.h>
#include <qdatetime.h>
#include <stdlib.h>
#include <stdio.h>

#include "clip.h"
#include "version.h"

#include <iostream.h>


// == constructor, destructor and general setup ================================

/*
 * Constructor of the chat window.
 */
KFibsChat::KFibsChat(QWidget *parent, const char *name)
	: QFrame(parent, name, false)
{ 
	setCaption(i18n("Chat Window"));

	/*
	 * Initialize the context menu and the text view
	 */
	contextMenu = new QPopupMenu();
	textWindow  = new KBgTextView(this, "chattext");

	QWhatsThis::add(textWindow, i18n("This is the display area of the chat window.\n\n"
					 "The text in this window is colored depending on whether "
					 "it is directed at you personally, shouted to the general "
					 "FIBS population, has been said by you, or is of general "
					 "interest. If you select the name of a player, the context "
					 "contains entries specifically geared towards that player."));
	/*
	 * Fully initialize the context menu
	 */
	KContextMenuManager::insert(textWindow->viewport(), contextMenu);
	
	contextMenu->insertItem(i18n("Info on"), this, SLOT(infoUser()),   0, Info);
	contextMenu->insertItem(i18n("Invite"),  this, SLOT(inviteUser()), 0, Invite);
	contextMenu->insertItem(i18n("Talk to"), this, SLOT(talkToUser()), 0, Talk);
	contextMenu->insertItem(i18n("Gag"),     this, SLOT(gagUser()),    0, Gag);
	contextMenu->insertSeparator();
	contextMenu->insertItem(i18n("Edit Gag List"),  this, SLOT(editGagList()),  0, GagEdit);
	contextMenu->insertItem(i18n("Clear Gag List"), this, SLOT(clearGagList()), 0, GagClear);
	contextMenu->insertItem(i18n("Silence All"),    this, SLOT(toggleSilent()), 0, Silent);
	contextMenu->insertSeparator();
	contextMenu->insertItem(i18n("Clear"),          textWindow, SLOT(clear()));
	
	contextMenu->insertItem(i18n("Change Font..."), textWindow, SLOT(selectFont()));
	contextMenu->insertItem(i18n("Close Window"),   this,       SLOT(hide()));

	connect(contextMenu, SIGNAL(aboutToShow()), this, SLOT(prepareMenu()));

	/*
	 * Setup the command bar
	 */
	cmdSelect = new QComboBox(this);

	cmdSelect->insertItem("kibitz" , Kibitz ); // these strings _not_ i18n'ed
	cmdSelect->insertItem("shout"  , Shout  );
	cmdSelect->insertItem("tell"   , Tell   );
	cmdSelect->insertItem("whisper", Whisper);
	cmdSelect->insertItem("message", Message);
	cmdSelect->insertItem("say"    , Say    );
	cmdSelect->setCurrentItem(Shout);

	cmdSelect->setFixedSize(cmdSelect->sizeHint());

	splitter = new QSplitter(this, "chatsplitter"); // will be resized in readConfig()
	
	cmdName  = new KLineEdit(splitter, "chatname");
	cmdName->setEnabled(false);
	cmdName->setText(i18n("  <player>  "));
	cmdName->setFixedHeight(cmdName->sizeHint().height());
	
	cmdText  = new KLineEdit(splitter, "chattext");
	cmdText->setFixedHeight(cmdText->sizeHint().height());
	cmdText->completionObject()->setOrder(KCompletion::Weighted);

	vl = new QVBoxLayout(this);
	hl = new QHBoxLayout();

	vl->addWidget(textWindow, 1);
	vl->addLayout(hl);

	hl->addWidget(cmdSelect);
	hl->addWidget(splitter);

	vl->activate();

	connect(cmdText, SIGNAL(returnPressed(const QString &)), this, SLOT(prepareCommand(const QString &)));
	connect(cmdSelect, SIGNAL(activated(const QString &)), this, SLOT(enableNameEdit(const QString &)));

	/*
	 * some eye candy :)
	 */
	setIcon(kapp->miniIcon());
}

/*
 * Destructor
 */
KFibsChat::~KFibsChat()
{
	delete contextMenu;
}


// == context menu handling ====================================================

/*
 * Reflect the current status in the context menu right before it's shown
 */
void KFibsChat::prepareMenu()
{
	currUser = textWindow->selectedText().stripWhiteSpace();
	if (currUser.isEmpty() || currUser.isNull()) currUser = "";

	contextMenu->changeItem(i18n("Info on ") + currUser, Info  );
	contextMenu->changeItem(i18n("Invite ")  + currUser, Invite);
	contextMenu->changeItem(i18n("Talk to ") + currUser, Talk  );
	contextMenu->changeItem(i18n("Gag ")     + currUser, Gag   );
	
	contextMenu->setItemEnabled(Info,   (currUser != ""));
	contextMenu->setItemEnabled(Invite, (currUser != ""));
	contextMenu->setItemEnabled(Talk,   (currUser != ""));
	contextMenu->setItemEnabled(Gag,    (currUser != ""));

	contextMenu->setItemEnabled(GagEdit,  !gagList.isEmpty());
	contextMenu->setItemEnabled(GagClear, !gagList.isEmpty());

	contextMenu->setItemChecked(Silent, silent);
}

/*
 * Add the user to the gag list, so we won't hear her shouting anymore.
 */
void KFibsChat::gagUser()
{
	if (gagList.contains(currUser.latin1()) <= 0)		//lukas: FIXME
		gagList.append(currUser.latin1());
	textWindow->write("<font color=\"blue\">" + i18n("You won't hear what %1 says and shouts.").arg(currUser) + "</font>");
}

/* 
 * Request information on the selected user
 */
void KFibsChat::infoUser()
{
	emit fibsCommand("whois " + currUser);
}

/* 
 * Request an invitation dialog for the selected user
 */
void KFibsChat::inviteUser()
{
	emit fibsRequestInvitation(currUser);
}

/*
 * Set the window up for talking to name
 */
void KFibsChat::fibsTalk(const QString &name)
{
	currUser = name;
	talkToUser();
}

/*
 * Clear all members of the gag list
 */
void KFibsChat::clearGagList()
{
	gagList.clear();
	textWindow->write("<font color=\"blue\">"  + i18n("Gag list is now empty.") + "</font>");
}

/*
 * Flip the silent flag
 */
void KFibsChat::toggleSilent()
{
	if (silent = !silent) 
		textWindow->write("<font color=\"blue\">" + i18n("You won't hear what people shout.") + "</font>");
	else
		textWindow->write("<font color=\"blue\">" + i18n("You will hear what people shout.") + "</font>");
}

// == various slots and functions ==============================================

/*
 * Catch hide events, so the engine's menu can be update. 
 */
void KFibsChat::showEvent(QShowEvent *e)
{
	QFrame::showEvent(e);
	emit windowVisible(true);
}

/*
 * Catch hide events, so the engine's menu can be update. 
 */
void KFibsChat::hideEvent(QHideEvent *e)
{
	emit windowVisible(false);
	QFrame::hideEvent(e);
}

/*
 * Set up the string for the server and emit it. Also add
 * the name in the history list.
 */
void KFibsChat::prepareCommand(const QString &s)
{
	if (!s.stripWhiteSpace().isEmpty()) {
		QString text(cmdSelect->currentText());
		text += " ";
		if (cmdName->isEnabled())
			text = text + cmdName->text() + " ";
		emit fibsCommand(text + s);
		cmdText->completionObject()->addItem(s);
	}
	cmdText->clear();
	cmdText->completionBox()->close();
}

/*
 * Enable the name field if tell has been selected in the combo box
 */
void KFibsChat::enableNameEdit(const QString &entry) 
{
	if ((entry != "tell") && (entry != "message")) {
		cmdName->setEnabled(false);
		cmdName->setFocusPolicy(QWidget::NoFocus);
	} else {
		cmdName->setEnabled(true);
		cmdName->setFocusPolicy(QWidget::StrongFocus);
	}
}

/*
 * Sets the name in the name entry field and sets the selection
 * to "kibitz" - for beginning of games.
 */
void KFibsChat::startGame(const QString &name)
{
	cmdName->setText(name);
	cmdName->setEnabled(false);
	cmdSelect->setCurrentItem(Kibitz);
	cmdName->setFocusPolicy(QWidget::StrongFocus);
}

/*
 * At the end of games, we switch from kibitz/say to tell. That way we
 * can talk even after the game is over
 */
void KFibsChat::endGame()
{
	if (cmdSelect->currentItem() == Kibitz || cmdSelect->currentItem() == Say)
		cmdSelect->setCurrentItem(Tell);
}

/*
 * Start talking to currUser
 */ 
void KFibsChat::talkToUser()
{
	cmdName->setEnabled(true);
	cmdName->setText(currUser);
	cmdSelect->setCurrentItem(Tell);  
	cmdName->setFocusPolicy(QWidget::StrongFocus);
}

/*
 * Fires up a small dialog that allows to remove users from the 
 * gag list.
 */
void KFibsChat::editGagList()
{
	QStrList tmp = gagList;		//lukas: FIXME

	KFibsEditGag *gl = new KFibsEditGag(&gagList, this, "gagedit");
	if (gl->exec()) {
		gagList = gl->getGagList();
		for (uint i = 0; i < tmp.count(); i++) {
			if (!gagList.contains(tmp.at(i))) {
				textWindow->write("<font color=\"blue\">"
						  + i18n("You will hear what %1 says and shouts.").arg(tmp.at(i))
						  + "</font>");
			}
		}
	}		
	delete gl;
}

/*
 * Remove all selected users from the gag list
 */
void KFibsChat::removeGagListEdit()
{
	for (uint i = 0; i < lbg->count(); i++) {
		if (lbg->isSelected(i)) {
			lbg->removeItem(i--);
		}
	}
}

/*
 * Clear the whole gag list
 */
void KFibsChat::clearGagListEdit()
{
	lbg->clear();
}

/*
 * Add a new name to the gag list
 */
void KFibsChat::addGagListEdit()
{
	for (uint i = 0; i < lbg->count(); i++) {
		if (strcmp(lbg->text(i).latin1(), leg->text().latin1()) == 0) {	//lukas: FIXME
			leg->clear();
			return;
		}
		if (strlen(lbg->text(i).latin1()) == 0) // TODO
			lbg->removeItem(i--);

	}
	lbg->insertItem(leg->text());
	leg->clear();
}


// == handle strings from the server ===========================================

/*
 * A message from the server that should be handled by us. If necessary, 
 * we replace the CLIP number by a string and put the line into the window.
 *
 * This function emits the string in rich text format with the signal 
 * personalMessage - again: the string contains rich text!
 */
void KFibsChat::handleData(const QString &msg)
{    
	QString clip = msg.left(msg.find(' ')), cMsg = msg;

	bool flag = false;
	int cmd = clip.toInt(&flag);

	QDateTime date;
 
	if (flag) {
		cMsg.replace(0, cMsg.find(' ')+1, "");

		QString user = cMsg.left(cMsg.find(' '));

		switch (cmd) {			
		case CLIP_SAYS:
			if (!gagList.contains(user.latin1())) {
				cMsg = i18n("<u>%1 tells you:</u> %2").arg(user).arg(cMsg.replace(QRegExp("^" + user), ""));
				cMsg = "<font color=\"red\">" + cMsg + "</font>";
				emit personalMessage(cMsg);
			} else 
				cMsg = "";
			break;

		case CLIP_SHOUTS:
			if ((!silent) && (!gagList.contains(user.latin1()))) {
				cMsg = i18n("<u>%1 shouts:</u> %2").arg(user).arg(cMsg.replace(QRegExp("^" + user), ""));
				cMsg = "<font color=\"black\">" + cMsg + "</font>";
			} else
				cMsg = "";
			break;
			
		case CLIP_WHISPERS:
			if (!gagList.contains(user.latin1())) {
				cMsg = i18n("<u>%1 whispers:</u> %2").arg(user).arg(cMsg.replace(QRegExp("^" + user), ""));
				cMsg = "<font color=\"red\">" + cMsg + "</font>";
				emit personalMessage(cMsg);
			} else 
				cMsg = "";
			break;

		case CLIP_KIBITZES:
			if (!gagList.contains(user.latin1())) {
				cMsg = i18n("<u>%1 kibitzes:</u> %2").arg(user).arg(cMsg.replace(QRegExp("^" + user), ""));
				cMsg = "<font color=\"red\">" + cMsg + "</font>";
				emit personalMessage(cMsg);
			} else 
				cMsg = "";
			break;

		case CLIP_YOU_SAY:		
			cMsg = i18n("<u>You tell %1:</u> %2").arg(user).arg(cMsg.replace(QRegExp("^" + user), ""));
			cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_YOU_SHOUT:
			cMsg = i18n("<u>You shout:</u> %1").arg(cMsg);
			cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_YOU_WHISPER:
			cMsg = i18n("<u>You whisper:</u> %1").arg(cMsg);
			cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_YOU_KIBITZ:
			cMsg = i18n("<u>You kibitz:</u> %1").arg(cMsg);
			cMsg = "<font color=\"darkgreen\">" + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_MESSAGE:
			user = cMsg.left(cMsg.find(' ')+1);
			cMsg.remove(0, cMsg.find(' ')+1);
			date.setTime_t(cMsg.left(cMsg.find(' ')+1).toUInt());
			cMsg.remove(0, cMsg.find(' '));
			cMsg = i18n("<u>User %1 left a message at %2</u>: %3").arg(user).arg(date.toString()).arg(cMsg);
			cMsg = "<font color=\"red\">" + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_MESSAGE_DELIVERED:
			cMsg = i18n("Your message for %1 has been delivered.").arg(user);
			cMsg = QString("<font color=\"darkgreen\">") + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;

		case CLIP_MESSAGE_SAVED:
			cMsg = i18n("Your message for %1 has been saved.").arg(user);
			cMsg = QString("<font color=\"darkgreen\">") + cMsg + "</font>";
			emit personalMessage(cMsg);
			break;
			
		default: // ignore the message
			return;
		}

	} else {
		// dump the string to cerr
		cerr << "KFibsChat::handleData unhandled message: " << cMsg.latin1() << endl;
		return;
	}

	if (cMsg != "")
		textWindow->write(cMsg);
}

// == configuration handling ===================================================

/*
 * Restore the previously stored settings
 */
void KFibsChat::readConfig()
{
	KConfig* config = kapp->config();
	config->setGroup(name());

	QFont kappFont = kapp->font();
	QPoint pos(10, 10);

	pos = config->readPointEntry("ori", &pos);
	setGeometry(pos.x(), pos.y(), config->readNumEntry("wdt",460), config->readNumEntry("hgt",200));
	config->readBoolEntry("vis", false) ? show() : hide();
	config->readListEntry("gag", gagList, ',');
	textWindow->setFont(config->readFontEntry("fon", &kappFont));
	silent = config->readBoolEntry("sil", false);

	QValueList<int> l;
	l.append(   config->readDoubleNumEntry("panner", 0.25) *splitter->width());
	l.append((1-config->readDoubleNumEntry("panner", 0.25))*splitter->width());
	splitter->setSizes(l);

	cmdText->completionObject()->setItems(config->readListEntry("history"));

	textWindow->readConfig();
}

/*
 * Save the current settings to disk
 */
void KFibsChat::saveConfig()
{
	KConfig* config = kapp->config();
	config->setGroup(name());

	config->writeEntry("ori", pos());
	config->writeEntry("hgt", height());
	config->writeEntry("wdt", width());
	config->writeEntry("vis", isVisible());
	config->writeEntry("fon", textWindow->font());
	config->writeEntry("gag", gagList, ',');
	config->writeEntry("sil", silent);

	config->writeEntry("panner", (double)cmdName->width()/(double)splitter->width());
	config->writeEntry("history", cmdText->completionObject()->items());

	textWindow->saveConfig();
}


// == gag list editor ==========================================================

/*
 * Constructor, creates the dialog but does not show nor execute it.
 */
KFibsEditGag::KFibsEditGag(const QStrList *gagList, QWidget *parent, const char *name)
	: QDialog(parent, name, true)
{
	setCaption(i18n("Gag List Editor"));
	
	QBoxLayout *vbox = new QVBoxLayout(this, 17);
	
	QLabel *info = new QLabel(this);
	
	lb     = new QListBox(this);
	ok     = new QPushButton(i18n("OK"), this);
	cancel = new QPushButton(i18n("Cancel"), this);
	
	QBoxLayout *hbox_1 = new QHBoxLayout();
	vbox->addLayout(hbox_1, 0);
	
	l = *gagList;
	
	lb->insertStrList(&l);
	lb->setMultiSelection(true);
	
	info->setText(i18n("Select all the users you want\n"
			   "to remove from the gag list\n"
			   "and then click Ok. Afterwards\n"
			   "you will again hear what they shout."));
	info->setMinimumSize(info->sizeHint());
	
	hbox_1->addWidget(lb, 0);
	hbox_1->addWidget(info, AlignTop);
	
	QBoxLayout *hbox_2 = new QHBoxLayout();
	vbox->addLayout(hbox_2);
	
	hbox_2->addWidget(ok);
	hbox_2->addWidget(cancel);
	
	lb->sizeHint().width();
	lb->sizeHint().height();
	
	lb->setMinimumSize(lb->sizeHint().width() > 100 ? 20+lb->sizeHint().width() : 100, 130);
	ok->setMinimumSize(ok->sizeHint());
	
	ok->setAutoDefault ( true );
	
	cancel->setMinimumSize(cancel->sizeHint());
	
	setMinimumSize( childrenRect().size() );
	
	vbox->activate();
	
	resize(minimumSize());
	
	connect(ok, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel, SIGNAL(clicked()), SLOT(reject()));
	
	lb->setFocus();
}

/*
 * Destructor is empty
 */
KFibsEditGag::~KFibsEditGag()
{
	// do nothing
}

/*
 * Return the new gag list
 */
QStrList KFibsEditGag::getGagList()		//lukas: FIXME BAD!!! - no QStrList at all
{
	l.clear();
	for (uint i=0; i<lb->count(); ++i) { 
		if (!lb->isSelected(i)) {
			l.append(lb->text(i).latin1());
		}
	}
	return l;
}


// EOF
