/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <klocale.h>

#include "conflictdisplay.h"


ConflictDisplay::ConflictDisplay(QWidget *parent, const char *name) : QWidget(parent, name),
								      _listwidth(0),
								      _listheight(0)
{
  _labellist.setAutoDelete(true);
  _labellist.clear();

  _scrollview = new QScrollView(this, "scrollview");
  _scrollview->setVScrollBarMode(QScrollView::AlwaysOn);
  _scrollview->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  _scrollview->setLineWidth(2);
  
  _background = new QWidget(_scrollview, "background");
  
  reset();
}


/*
 * Public methods
 *****************/
 
QSize  ConflictDisplay::minimumSizeHint() const
{
  return QSize(_listwidth, 70);
}


void  ConflictDisplay::reset()
{
  QLabel   *label;
  QWidget  *viewport = _scrollview->viewport();
  
  clear();
  
  label = new QLabel(i18n("No conflicts."), _background);
  label->adjustSize();
  
  _listwidth  = label->width();
  _listheight = label->height();
  
  _labellist.append(label);
  label->move(0, 0);
  label->show();
  
  _background->resize(QMAX(_listwidth,  (uint) viewport->width()), 
  		      QMAX(_listheight, (uint) viewport->height()));
}


void  ConflictDisplay::showConflicts(QList<Conflict>& conflictlist)
{
  Conflict  *conflict;
  QLabel    *label;
  QWidget   *viewport = _scrollview->viewport();
  
  clear();
  
  for (conflict = conflictlist.first(); conflict; conflict = conflictlist.next())
  {
    label = new QLabel(conflict->description(), _background);
    label->adjustSize();
    
    _labellist.append(label);
    label->move(0, _listheight);
    label->show();
    
    _listwidth   = QMAX(_listwidth, (uint) label->width());
    _listheight += label->height(); 
  }
  
  _background->resize(QMAX(_listwidth,  (uint) viewport->width()), 
  		      QMAX(_listheight, (uint) viewport->height())); 
}


QSize  ConflictDisplay::sizeHint() const
{
  return QSize(_listwidth, QMAX(70, _listheight));
}


/*
 * Protected methods
 ********************/
 
void  ConflictDisplay::resizeEvent(QResizeEvent *)
{
  uint     w = width();
  uint     h = height();
  uint     backw;
  uint     backh;
  QWidget  *viewport = _scrollview->viewport();
  
  _scrollview->setGeometry(0, 0, w, h);
  
  backw = QMAX(_listwidth-4,  (uint) viewport->width());
  backh = QMAX(_listheight-4, (uint) viewport->height());
  
  _background->setGeometry(2, 2, backw, backh); 
}


/*
 * Private methods
 ******************/
 
void  ConflictDisplay::clear()
{
  _labellist.clear();   
  _listwidth = _listheight = 0;
}

  
