/*
 *  Copyright (C) 2001 Matthew Mueller
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "state.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkclist.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-file-entry.h>

static void state_push_prefix (const gchar * name);

static void
state_push_prefix (const gchar * name)
{
	gchar *buf = g_strdup_printf ("/galeon/State/%s",name);
	gnome_config_push_prefix (buf);
	g_free (buf);
}

void
state_save_window_state (GdkWindow *window, const gchar *name)
{
	gint x, y, width, height;

	gdk_window_get_root_origin (window, &x, &y);
	gdk_window_get_size (window, &width, &height);

	state_push_prefix (name);
	gnome_config_set_int ("_x=", x);
	gnome_config_set_int ("_y=", y);
	gnome_config_set_int ("_width=", width);
	gnome_config_set_int ("_height=", height);
	gnome_config_pop_prefix ();
	gnome_config_sync ();
}

void
state_load_window_state (GtkWidget *window, const gchar *name,
			int defwidth, int defheight)
{
	GString *buf;
	gint x, y, width, height;
	
	buf = g_string_sized_new (16);
	
	state_push_prefix (name);

	x = gnome_config_get_int ("_x=-1");
	y = gnome_config_get_int ("_y=-1");
	if (!(x == -1 && y == -1))
		gtk_widget_set_uposition (window, x, y);

	g_string_sprintf (buf, "_width=%i", defwidth);
	width = gnome_config_get_int (buf->str);
	g_string_sprintf (buf, "_height=%i", defheight);
	height = gnome_config_get_int (buf->str);
	if (!(width == -1 && height == -1))
		gtk_window_set_default_size (GTK_WINDOW (window),
					     width, height);

	g_string_free (buf, TRUE);
	gnome_config_pop_prefix ();
}

void
state_save_column_widths (GtkWidget *w, const gchar *name)
{
	gint i;
	GtkCList *clist = GTK_CLIST (w);
	GString *cwidths = g_string_sized_new (32);
	
	state_push_prefix (name);

	g_string_sprintf (cwidths, "%i", clist->column[0].width);
	for (i=1; i < clist->columns; i++)
	{
		g_string_sprintfa (cwidths, ",%i", clist->column[i].width);
	}
	
	gnome_config_set_string ("_cwidths=", cwidths->str);
	g_string_free (cwidths, TRUE);
	gnome_config_pop_prefix ();
	gnome_config_sync ();
}

void
state_load_column_widths (GtkWidget *w, const gchar *name)
{
	int i;
	gchar *cwidths, *cp;
	
	state_push_prefix (name);
	cp = cwidths = gnome_config_get_string ("_cwidths=");

	for (i=0; cp != NULL && *cp != '\0'; i++)
	{
		gtk_clist_set_column_width (GTK_CLIST (w), i, atoi(cp));
		cp = strchr (cp, ',');
		if (cp) cp++;
	}
	g_free (cwidths);
	gnome_config_pop_prefix ();
}

void state_load_pane_pos (GtkWidget *pane, const gchar *name)
{
	if (pane != NULL)
	{
		gint pane_pos;
		gchar *buf = g_strdup_printf ("/galeon/State/%s", name);
		pane_pos = gnome_config_get_int (buf);
		g_free (buf);

		if (pane_pos != -1)
			gtk_paned_set_position (GTK_PANED (pane), pane_pos);
	}
}

void state_save_pane_pos (GtkWidget *pane, const gchar *name)
{
	if (pane != NULL)
	{
		gchar *buf = g_strdup_printf ("/galeon/State/%s", name);
		gnome_config_set_int (buf, GTK_PANED (pane)->child1_size);
		g_free (buf);
		gnome_config_sync ();
	}
}
