/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "Galeon.h"
#include "prefs.h"
#include "history.h"
#include "embed.h"
#include "session.h"
#include "misc.h"
#include "bookmarks.h"
#include "window.h"

#include <gtkmozembed.h>
#include <glade/glade.h>

/* GNOME includes */
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnorba/gnorba.h>
#include <liboaf/liboaf.h>
#ifdef ENABLE_APPLET
#include <applet-widget.h>
#endif

#include <errno.h>

/* local function prototypes */
static gint translate_url_arguments (poptContext context, gchar ***url);
static CORBA_ORB corba_init (int *argc, char *argv[]);
static gboolean  new_view_on_running_shell (gint n_urls, gchar **url);
static void new_shell (CORBA_ORB orb);
static gboolean galeon_init (int argc, char *argv[]);
static void create_default_browser (void);
static ViewState get_viewstate (void);
static gint read_mozilla_version (void);
static gint check_mozilla_version (void);

/* import from Galeon-impl.c */
Galeon_Browser impl_Galeon_Browser__create (PortableServer_POA poa, 
					    CORBA_Environment *corba_env);

/* global variables */
gboolean galeon_server_mode = FALSE;
gboolean galeon_panel_mode = FALSE;
GConfClient *gconf_client = NULL;

/* file local variables */
static CORBA_Environment corba_env;          /* corba environment for errors */
static gboolean open_in_existing   = FALSE;  /* load in existing window?     */
static gboolean open_in_new_tab    = FALSE;  /* force open in a new tab?     */
static gboolean open_in_new_window = FALSE;  /* force open in a new window?  */
static gboolean open_fullscreen    = FALSE;  /* open galeon in full screen ? */
static gchar *geometry_string      = NULL;   /* the geometry string          */
static gchar *bookmark_url         = NULL;   /* the temp bookmark to add     */

/* mozilla version detected at runtime */
int mozilla_version=0;

/* command line argument parsing structure */
static struct poptOption popt_options[] =
{
	{ NULL, '\0', POPT_ARG_INCLUDE_TABLE, &oaf_popt_options, 0, NULL,
	  NULL },
	{ "new-tab", 'n', POPT_ARG_NONE, &open_in_new_tab, 0,
	  N_("Open a new tab in an existing Galeon window"),
	  NULL },
	{ "new-window", 'w', POPT_ARG_NONE, &open_in_new_window, 0,
	  N_("Open a new window in an existing Galeon process"),
	  NULL },
	{ "fullscreen", 'f', POPT_ARG_NONE, &open_fullscreen, 0,
	  N_("Run galeon in full screen mode"),
	  NULL },
	{ "existing", 'x', POPT_ARG_NONE, &open_in_existing, 0,
	  N_("Attempt to load URL in existing Galeon window"),
	  NULL },
#ifdef ENABLE_APPLET
	{ "panel", 'p', POPT_ARG_NONE, &galeon_server_mode, 0,
	  N_("Open as a panel applet"),
	  NULL },		  
#endif
	{ "server", 's', POPT_ARG_NONE, &galeon_server_mode, 0,
	  N_("Don't open any windows; instead act as a server "
	     "for quick startup of new Galeon instances"),
	  NULL },		  
	{ "add-temp-bookmark", 't', POPT_ARG_STRING, &bookmark_url,
	  0, N_("Add a temp bookmark (don't open any window)"), 
	  N_("URL")},
	{ "geometry", 'g', POPT_ARG_STRING, &geometry_string,
	  0, N_("Create the initial window with the given geometry.\n"
		"see X(1) for the GEOMETRY format"),
	  N_("GEOMETRY")},

	/* terminator, must be last */
	{ NULL, 0, 0, NULL, 0, NULL, NULL }
};

/**
 * main: main entry point
 *
 * Set values to standard values, and change later by reading the 
 * options defined at the command line.
 * 
 * Initialize localization, glade, gnome, etc.
 */
int
main (int argc, char *argv[])
{
	GaleonEmbed *last_embed = NULL;
	gboolean tabbed_mode;
	poptContext context;
	gboolean no_default;
	CORBA_ORB orb;
	gint i, n_urls;
	gchar **url;

	 /* make stdout line buffered - we only use it for debug info */
	setvbuf (stdout, NULL, _IOLBF, 0);

	/* initialise localisation */
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain (PACKAGE);

#ifdef ENABLE_APPLET
	/* panel mode? this check has to be done here since we
	 * don't use gnome_init_* when in panel mode */
	for (i = 1; i < argc; i++)
	{
		if (strcmp (argv[i], "-p") == 0 || 
		    strcmp (argv[i], "--panel") == 0)
		{
			galeon_panel_mode = TRUE;
		}
	}

	/* initialise GNOME */
	if (galeon_panel_mode)
	{
		applet_widget_init (PACKAGE, VERSION, argc, argv, 
				    popt_options, 0, &context);
	}
	else
#endif
	{
		gnome_init_with_popt_table (PACKAGE, VERSION, argc, argv,
					    popt_options, 0, &context);
	}

	if (check_mozilla_version ())
		return 1;

	/* load arguments that aren't regular options (urls to load) */
	n_urls = translate_url_arguments (context, &url);

	/* initialise CORBA */
	orb = corba_init (&argc, argv);
	
	/* check if galeon is already running and use that instead */
	if (new_view_on_running_shell (n_urls, url))
	{
		/* done */
		return 0;
	}

	/* add a temp bookmark and exit if needed */
	if (bookmark_url != NULL)
	{
		bookmarks_load ();
		add_bookmark_default (BM_SITE, NULL, bookmark_url, NULL);
		bookmarks_save ();
		return 0;
	}
	
	/* start a new CORBA shell */
	new_shell (orb);

	/* initialise required external and internal Galeon services */
	no_default = galeon_init (argc, argv);

	/* if we have command-line supplied URLs */
	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED) ? 1 : 0;
	for (i = 0; i < n_urls; i++)
	{
		last_embed = embed_create_from_url (last_embed, url[i],
						    FALSE, !tabbed_mode);
	}
	g_strfreev (url);

	/* check if browser exists, or create default one */
	if (!no_default && !galeon_server_mode && !galeon_panel_mode)
	{
		create_default_browser ();
	}

	/* start main loop */
#ifdef ENABLE_APPLET
	if (galeon_panel_mode)
	{
		/* build panel and run special main loop */
		panel_main ();
	}
	else
#endif
	{
		/* enter the main GTK event processing loop */
		gtk_main ();
	}

	/* exit cleanly */
	return 0;
}


static gint
read_mozilla_version (void)
{
	gchar agentstr[160], *p, *endptr;
	gint ver= 0, v;
	gchar *preffile;
	FILE *f;

	preffile = g_strconcat (getenv ("MOZILLA_FIVE_HOME"),
				"/defaults/pref/all.js", NULL);
	f = fopen (preffile, "r");
	
	if (f == NULL)
	{
		g_warning ("error opening %s: %s", preffile, strerror (errno));
		g_free (preffile);
		return 0;
	}

	while ((p = fgets (agentstr, 160, f)) != NULL)
	{
		if ((p = strstr (agentstr, "useragent.misc")) != NULL &&
			(p = strstr (p, "rv:")) != NULL)
			break;
	}
	fclose(f);

	if (p == NULL)
	{
		g_warning ("mozilla version not found in %s", preffile);
	}
	else
	{
		p += 3;
		v = strtol (p, &endptr, 10);
		ver = v << 24;
		if (*endptr == '.')
		{
			p = endptr + 1;
			v = strtol (p, &endptr, 10);
			ver += v << 16;
			if (*endptr == '.')
			{
				p = endptr + 1;
				v = strtol (p, &endptr, 10);
				ver += v << 8;
			}
		}
		if (*endptr == '+')
		{
			ver++;
		}
	}

	g_free (preffile);
	return ver;
}

static gint
check_mozilla_version (void)
{
	mozilla_version = read_mozilla_version ();

	if (MOZILLA_VERSION != mozilla_version)
	{
		g_warning ("compiled MOZILLA_VERSION ("VERSIONFMT") != detected mozilla_version ("VERSIONFMT")",
			VERSIONFMTARGS (MOZILLA_VERSION), VERSIONFMTARGS (mozilla_version));
		//return -1;
	}

	return 0;
}

/**
 * translate_url_arguments: gather URL arguments and expand them fully
 * with realpath if they're filenames
 */
static gint
translate_url_arguments (poptContext context, gchar ***urls)
{
	gchar buffer[PATH_MAX];
	gchar **args;
	gint i, n;

	/* any context remaining? */
	if (context == NULL)
	{
		*urls = NULL;
		return 0;
	}

	/* get the args and check */
	args = (gchar **) poptGetArgs (context);
	if (args == NULL)
	{
		poptFreeContext (context);
		*urls = NULL;
		return 0;
	}

	/* count args */
	for (n = 0; args[n] != NULL; n++)
		/* nothing */;

	/* allocate pointer array */
	*urls = g_new0 (gchar *, n + 1);
	
	/* translate each one */
	for (i = 0; i < n; i++)
	{
		/* try to expand as files */
		if (realpath (args[i], buffer) != NULL)
		{
			(*urls)[i] = g_strconcat ("file://", buffer, NULL);
		}
		else
		{
			(*urls)[i] = g_strdup (args[i]);
		}
	}
	poptFreeContext (context);
	(*urls)[i] = NULL;

	/* return the number of urls */
	return n;
}

/**
 * corba_init: initialise Gnorba (for gtm) and OAF (for remote
 * Galeon startups).
 */
static CORBA_ORB
corba_init (int *argc, char *argv[])
{
	CORBA_ORB orb;

	/* catch exceptions */
	CORBA_exception_init (&corba_env);

	/* initialise CORBA (not for panel mode, this does its own init) */
	if (!galeon_panel_mode)
	{
		gnorba_CORBA_init (argc, argv, GNORBA_INIT_SERVER_FUNC, 
				   &corba_env);
	}

	/* initialise OAF */
	orb = oaf_init (*argc, argv);

	/* return handle */
	return orb;
}

/**
 * new_view_on_running_shell: 
 *
 * Create new view on existing shell. 
 * Returns false if no shell is currently running.
 */
static gboolean 
new_view_on_running_shell (gint n_urls, gchar **url)
{
	CORBA_Object client;
	ViewState viewstate;
	gboolean result = TRUE;
	gint i;
	
	/* compute target viewstate */
	viewstate = get_viewstate ();

	/* try to find an existing shell to run on */
	client = oaf_activate ("repo_ids.has('IDL:galeon/browser:1.0')", NULL,
			       OAF_FLAG_EXISTING_ONLY, NULL, &corba_env);

	/* check if we found one */
	if (CORBA_Object_is_nil (client, &corba_env))
	{
		return FALSE;
	}

	/* if found and we're given a bookmark to add... */
	if (bookmark_url != NULL)
	{
		Galeon_Browser_addTempBookmark (client, bookmark_url, 
						bookmark_url, &corba_env);
		return TRUE;
	}

	/* otherwise we'll be opening windows, so warn */
	g_message (_("Galeon already running, using existing process"));

	/* provided with urls? */
	if (n_urls == 0)
	{
		/* no, open a default window */
		result = Galeon_Browser_loadurl (client, "", viewstate, 
						 &corba_env);
	}

	/* open all of the urls */
	for (i = 0; i < n_urls; i++)
	{
		result = Galeon_Browser_loadurl (client, url[i], viewstate, 
						 &corba_env);
		if (!result)
		{
			break;
		}
	}

#if 0
	/* check */
	if (result)
	{
		GtkWidget *dialog = gnome_error_dialog 
			(_("Galeon remote startup failed.\n\n"
			   "Try exiting other OAF applications\n"
			   "(e.g. Evolution and Nautilus), and running\n"
			   "\"oaf-slay\" from the command line."));
			 
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
	}
#endif

	/* done */
	return TRUE;
}

/**
 * new_shell: create a new CORBA shell
 */
static void 
new_shell (CORBA_ORB orb)
{
	PortableServer_POA poa;
	Galeon_Browser client;
	gchar *reg_id;

	/* make a client */
	poa = (PortableServer_POA) CORBA_ORB_resolve_initial_references 
		(orb, "RootPOA", &corba_env);
	client = impl_Galeon_Browser__create (poa, &corba_env);

	/* check */
	if (client == NULL)
	{
		g_warning (_("Galeon object already active, "
			     "server registration failed\n"));
		return;
	}

	/* register server with oaf */
	reg_id = oaf_make_registration_id ("OAFIID:Galeon:20010802",
					   g_getenv ("DISPLAY"));
	oaf_active_server_register (reg_id, client);
	g_free (reg_id);
}

/**
 * galeon_init: initialise Galeon and the services on which it depends
 */
static gboolean
galeon_init (int argc, char *argv[])
{
	static gchar *restart_argv[] = { "galeon", "--server", NULL };
	GnomeClient *session_client;
	gboolean session_recovery;
	gboolean vfs_success;
	gboolean new_user;
	gchar *profile;
	extern gboolean pushed_startup;

	/* initialise GNOME VFS */
	vfs_success = gnome_vfs_init ();
	g_assert (vfs_success);

	/* initialise gconf */
	gconf_init (argc, argv, NULL);
	gconf_client = gconf_client_get_default ();
	g_assert (gconf_client != NULL);

	/* initialise GLADE */
	glade_gnome_init ();

	/* initialise Galeon itself */
	preferences_load ();
	history_init ();
	bookmarks_init ();
	initialise_colours ();

	/* check if this is the first time this user has run Galeon */
	new_user = newuser_check ();

	/* init mozilla home */
	gtk_moz_embed_set_comp_path (g_getenv ("MOZILLA_FIVE_HOME"));

	/* set the profile dir for gtkmozembed */
	profile = g_concat_dir_and_file (g_get_home_dir (), ".galeon/mozilla");
	gtk_moz_embed_set_profile_path (profile, "galeon");
	g_free (profile);

	/* initialise embed */
	if (!pushed_startup)
	{
		embed_startup_init ();
	}

	/* resume a crashed session, if any */
	session_recovery = session_autoresume ();

	/* connect our session recovery to GNOME session */
	session_client = gnome_master_client ();
	gnome_client_set_restart_command 
		(session_client, galeon_server_mode ? 2 : 1, restart_argv);
	gnome_client_set_restart_style (GNOME_CLIENT (session_client),
					GNOME_RESTART_IF_RUNNING);
	gtk_signal_connect (GTK_OBJECT (session_client), "save_yourself",
			    (GtkSignalFunc)client_save_yourself_cb, NULL);

	/* if this flag doesn't get cleared, then Galeon crashed */
	gnome_config_set_int ("/galeon/General/crashed", TRUE);
	gnome_config_sync ();

	/* return saying whether or not to override default window creation */
	return (new_user || session_recovery);
}

/**
 * create_default_browser: called at the end to make sure at least one
 * browser is created in normal modes.
 */
static void
create_default_browser (void)
{
	GaleonEmbed *embed;
	GaleonWindow *window;
	gint xpos = -1, ypos;
	gint width, height;
	gchar *url;
	
	/* don't bother if there's already at least one */
	if (g_list_length (all_embeds) != 0)
	{
		return;
	}

	/* create an empty window */
	window = window_create (GTK_MOZ_EMBED_FLAG_ALLCHROME);
	
	/* make sure the user can see it */
	set_settings_menu_window (window);
	
	/* show as fullscreen? */
	/* THE WINDOW NEEDS TO BE VISIBLE HERE */
	if (open_fullscreen)
	{
		gtk_widget_show (GTK_WIDGET (window->WMain));
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->view_fullscreen),
			 TRUE);
	}
	/* collect geometry information */
	else if (gnome_parse_geometry (geometry_string, &xpos, &ypos,
				       &width, &height))
	{
		gtk_window_set_default_size 
			(GTK_WINDOW (window->WMain), width, height);
		window->set_size = TRUE;
		gtk_widget_show (GTK_WIDGET (window->WMain));
	}
	
	/* reposition */
	if (xpos > 0 && ypos > 0)
	{
		gdk_window_move (window->WMain->window, xpos, ypos);
	}
	
	/* make sure it's visible */
	while (gtk_events_pending ())
	{
		gtk_main_iteration ();
	}
	
	/* build the default embed inside it */
	embed = embed_create_in_window (window, FALSE);
	embed_set_visibility (embed, TRUE);
	url = embed_get_default_url (NULL);
	embed_load_url (embed, url);
	g_free (url);
}

/**
 * get_viewstate:
 *
 * Determine the viewstate. A viewstate defines how the URI should be opened.
 * Seemingly cryptic. Might be redone if additional viewstates are added 
 * -Nate
 *
 * You're evil ;-)
 *   -- MattA
 * 
 * VIEWSTATE_DEFAULT: Use default (setting) viewstate
 * VIEWSTATE_WINDOW:  Open up in window
 * VIEWSTATE_TAB:     Open up in tab
 * VIEWSTATE_NONE:    Open up in existing window if there is one
 */
static ViewState
get_viewstate (void)
{
	return (2 * (!open_in_existing & (open_in_new_tab ^ 
					  open_in_new_window)) + 
		(!open_in_new_window & !open_in_existing));
}
 
