/******************************************************************************
  File:     $Id: hpdjparm.c,v 1.15 1998-11-21 17:46:12+01 mjl Rel $
  Contents: Ghostscript device driver "hpdj": Methods for parameter handling
  Author:   Martin Lottermoser, Metzgerfeldweg 9, 85737 Ismaning, Germany

*******************************************************************************
*									      *
*	Copyright (C) 1996, 1997, 1998 by Martin Lottermoser		      *
*	All rights reserved						      *
*									      *
*******************************************************************************

  Preprocessor symbols:

    HPDJ_INPUTATTRIBUTES (integer)
	The driver will install its own definition of 'InputAttributes' in its
	page device dictionary iff this is non-zero. The default is zero.

    HPDJ_INPUTMEDIA_PRN (integer)
	Somewhere between gs 5.10 and gs 5.50, the gdev_prn_*_input_media()
	macros and the gdev_prn_input_page_size() function were replaced by
	general functions without the "_prn" part in their names.
	Define this to be non-zero if your version of ghostscript expects the
	old macros and zero if it offers the new interface.
	The default is zero unless GS_REVISION is defined and at most 510,
	in which case the default is 1.

    HPDJ_NO_PAGECOUNTFILE
	If this symbol is defined, the driver will not offer the parameter
	'PagecountFile'.

******************************************************************************/

/* Configuration management identification */
#ifndef lint
static const char
  cm_id[] = "@(#)$Id: hpdjparm.c,v 1.15 1998-11-21 17:46:12+01 mjl Rel $";
#endif

/*****************************************************************************/

#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE	500
#endif

/* Special Aladdin header, must be included before <sys/types.h> on some
   platforms (e.g., FreeBSD). Here apparently needed because of <stdio.h>. */
#include "std.h"

/* Standard headers */
#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* hpdj headers */
#include "hpdj.h"

/* Additional ghostscript headers besides those included from hpdj.h */
#include "gsparam.h"

/*****************************************************************************/

#ifndef HPDJ_INPUTATTRIBUTES
#define HPDJ_INPUTATTRIBUTES	0
#endif

#ifndef HPDJ_INPUTMEDIA_PRN
#if defined(GS_REVISION) && GS_REVISION <= 510
#define HPDJ_INPUTMEDIA_PRN	1
#else
#define HPDJ_INPUTMEDIA_PRN	0
#endif
#endif

#ifndef lint
/* Permit deduction of compilation options from the binary */
static const char info_string[] = "@(#)HPDJ_INPUTATTRIBUTES is "
#if (HPDJ_INPUTATTRIBUTES) != 0
  "non-"
#endif
  "zero.";
#endif	/* lint */

/*****************************************************************************/

/*  Data structures for command line arguments to options

    The tables defined here are used to map names to integers.
    It is possible to have several names for the same value. The first name
    given is the canonical one, i.e. the one returned by the get_params routine.

    The lists are terminated by entries with a NULL pointer for the name.
 */

const StringToInt
  /* Colour modes */
  hpdj_colour_mode_list[] = {
    /* Values of type 'colour_type' are assumed to be usable as indices into
       this array in order to find string representations for them. */
    { "mono",	mono },
    { "CMY",	cmy },
    { "CMY+K",	cmy_plus_k },
    { "CMYK",	cmyk },
    /* Synonyms */
    { "monochrome", mono },
    { NULL,	0 }
  };
static const StringToInt
  /* Names for media types */
  media_type_list[] = {
    /* Canonical names */
    { "plain paper",	0 },
    { "bond paper",	1 },
    { "HP Premium paper", 2 },
    { "glossy paper",	3 },
    { "transparency film", 4 },
    { "photo paper",	5 },
    /* Shortened names */
    { "plain",		0 },
    { "bond",		1 },
    { "Premium",	2 },
    { "glossy",		3 },
    { "transparency",	4 },
    { "photo",		5 },
    { NULL,		0 }
  },
  /* Print Quality */
  print_quality_list[] = {
    { "draft",	       -1 },
    { "normal",		0 },
    { "presentation",	1 },
    /* Start of synonyms */
    { "best",		1 },
    { NULL,		0 }
  };

/******************************************************************************

  Function: get_option_string

  This function returns a string representation of 'in_value' in '*out_value',
  based on 'table'. If 'in_value' cannot be found in 'table', the function
  returns a decimal representation of 'in_value'.

  The string buffer in '*out_value' will be a statically allocated area which
  must not be modified.

******************************************************************************/

static void get_option_string(int in_value, const StringToInt *table,
  gs_param_string *out_value)
{
  while (table->name != NULL && table->value != in_value) table++;
  if (table->name != NULL) {
    out_value->data = (const byte *)table->name;
    out_value->size = strlen(table->name);
    out_value->persistent = true;	/* static const is persistent */
  }
  else {
    static char buffer[25];	/* Must be sufficient for an 'int' */

    sprintf(buffer, "%d", in_value);
    assert(strlen(buffer) < sizeof(buffer));
    out_value->data = (const byte *)buffer;
    out_value->size = strlen(buffer);
    out_value->persistent = false;
  }

  return;
}

/******************************************************************************

  Function: get_option_value

  This function parses 'in_value' based on 'table' and returns the result in
  '*out_value'.

  'in_value' must either be present in 'table' or must be a decimal
  representation of an integer. In these cases, the function returns 0,
  otherwise a non-zero ghostscript error value.

  On returning 'gs_error_VMerror', the function will have issued an error
  message.

******************************************************************************/

static int get_option_value(const gs_param_string *in_value,
  const StringToInt *table, int *out_value)
{
  char *s;
  int read;	/* counter */

  /* First we construct a properly NUL-terminated string */
  s = (char *) malloc(in_value->size + 1);
  if (s == NULL) {
    fprintf(stderr, ERRPREF
      "Memory allocation failure in get_option_value(), errno is %d.\n",
      errno);
    return_error(gs_error_VMerror);
  }
  strncpy(s, (const char *)in_value->data, in_value->size);
  s[in_value->size] = '\0';

  /* Check for a numerical value */
#if defined(_MSC_VER) && _MSC_VER <= 1100
  /* Bug in Visual C++ 5.0 SP1 (_MSC_VER is 1100): %n gives always 0. */
  if (sscanf(s, "%d", out_value) != 1) {
#else
  if (sscanf(s, "%d %n", out_value, &read) != 1 || s[read] != '\0') {
#endif /* _MSC_VER */
    /* What the user specified is not a valid numerical value */
    while (table->name != NULL && strcmp(table->name, s) != 0) table++;
    if (table->name != NULL) *out_value = table->value;
    else {
      free(s); s = NULL;
      return_error(gs_error_rangecheck);
    }
  }

  free(s); s = NULL;

  return 0;
}

/******************************************************************************

  Function: hpdj_get_params

  This function returns to the caller information about the values of
  parameters defined for the device. This includes parameters defined in base
  classes.

  The function returns zero on success and a negative value on error.

******************************************************************************/

int hpdj_get_params(gx_device *device, gs_param_list *plist)
{
  gs_param_string string_value;
  hpdj_device *dev = (hpdj_device *)device;
  int
    temp,  /* Used as an intermediate for various reasons */
    rc;

#ifdef HPDJ_TRACE
  fputs("! hpdj_get_params()...\n", stderr);
#endif

#ifndef NDEBUG
  /* Consistency checks on internal data structures. See comments for type
     'Model' and the variable 'hpdj_colour_mode_list[]'. */
  {
    int j;
    for (j = 0; hpdj_model[j].name != NULL; j++)
      assert(hpdj_model[j].model == j);
    for (j = mono; j <= cmyk; j++) assert(hpdj_colour_mode_list[j].value == j);
  }
#endif /* NDEBUG */

  /* Base class parameters */
  rc = gdev_prn_get_params(device, plist);
  if (rc < 0) return rc;

  /* Class-specific parameters. I have not checked for the reason, but in
     gs 4.03 and 5.10 one must return a parameter value here in order to be
     able to receive it in put_params. For example, if I return CMYLevels only
     if the colour mode is not monochrome, I'm never able to set it...
     Weird. */

  /* Colour levels */
  if ((rc = param_write_int(plist, "BlackLevels", &dev->black_levels)) < 0)
    return rc;
  if ((rc = param_write_int(plist, "CMYLevels", &dev->cmy_levels)) < 0)
    return rc;

  /* Colour mode */
  get_option_string(dev->colour_mode, hpdj_colour_mode_list, &string_value);
  if ((rc = param_write_string(plist, "ColourMode", &string_value)) < 0 ||
      (rc = param_write_string(plist, "ColorMode", &string_value)) < 0)
    return rc;

  /* Compression method */
  temp = dev->compression_method;
  if ((rc = param_write_int(plist, "CompressionMethod", &temp)) < 0) return rc;

  /* Dry time */
  if ((rc = param_write_int(plist, "DryTime", &dev->dry_time)) < 0) return rc;

  /* Manual feed */
  {
    bool temp;
    if (dev->manualfeed == automatic)
      temp = (*hpdj_model[dev->model].name == '5' && is_envelope(dev->ps_code));
    else temp = (dev->manualfeed == requested);
    if ((rc = param_write_bool(plist, "ManualFeed", &temp)) < 0) return rc;
  }

  /* Margin description file */
  if (dev->margin_overrides == NULL) string_value.data = (const byte *)"";
  else string_value.data = (const byte *)dev->margin_file;
  string_value.size = strlen((const char *)string_value.data);
  string_value.persistent = false;
  if ((rc = param_write_string(plist, "MarginFile", &string_value)) < 0)
    return rc;

  /* Media type */
  get_option_string(dev->media_type, media_type_list, &string_value);
  if ((rc = param_write_string(plist, "MediaType", &string_value)) < 0)
    return rc;

  /* Model name */
  string_value.data = (const byte *)hpdj_model[dev->model].name;
  string_value.size = strlen(hpdj_model[dev->model].name);
  string_value.persistent = true;
  if ((rc = param_write_string(plist, "Model", &string_value)) < 0) return rc;

#ifndef HPDJ_NO_PAGECOUNTFILE
  /* Page count file */
  if (dev->pagecount_file == NULL) string_value.data = (const byte *)"";
  else string_value.data = (const byte *)dev->pagecount_file;
  string_value.size = strlen((const char *)string_value.data);
  string_value.persistent = false;
  if ((rc = param_write_string(plist, "PagecountFile", &string_value)) < 0)
    return rc;
#endif	/* HPDJ_NO_PAGECOUNTFILE */

  /* Print quality */
  get_option_string(dev->print_quality, print_quality_list, &string_value);
  if ((rc = param_write_string(plist, "PrintQuality", &string_value)) < 0)
    return rc;

  /* Undocumented PCL command 1 */
  if ((rc = param_write_int(plist, "PCLundoc1", &dev->undoc1)) < 0) return rc;

#if HPDJ_INPUTATTRIBUTES != 0
  /* InputAttributes dictionary. We set only PageSize in each entry. */
  {
    int key;
    unsigned int entries = 0;
    const margin_desc *margins, *mp;
    gs_param_dict pdict;
    bool custom_page_size_supported = false;

    /* Set pointer to whichever margin list is the right one */
    if (dev->margin_overrides == NULL) margins = hpdj_model[dev->model].margin;
    else margins = dev->margin_overrides;

    /*  Count the number of media sizes supported and check whether there is an
	entry for custom page sizes.
	There is an unsightly discrepancy here between builtin margins and
	margins read from a margin file: for builtin margins, custom page sizes
	are supported if the 'custom' pointer for the model is non-NULL, for
	margins read there must in addition be an entry in the margin list.
    */
    for (mp = margins; mp->code != pcl_ps_none; mp++)
      if (mp->code == pcl_ps_custom) custom_page_size_supported = true;
      else entries++;
    if (custom_page_size_supported || dev->margin_file == NULL)
      custom_page_size_supported = (hpdj_model[dev->model].custom != NULL);

    /* Also add the number of custom page size descriptions */
    if (custom_page_size_supported) {
      const custom_page_desc *cp;
      for (cp = hpdj_model[dev->model].custom; cp->width_max > 0; cp++)
	entries++;
    }

    /*  Allocate and initialize a dictionary to be used as an InputAttributes
	dictionary with 'entries' entries and return a reference to it in
	'pdict'.
    */
#if HPDJ_INPUTMEDIA_PRN
    rc = gdev_prn_begin_input_media(plist, &pdict, entries);
#else
    rc = gdev_begin_input_media(plist, &pdict, entries);
#endif
    if (rc < 0) return rc;

    /* Write an entry for each supported media size to 'pdict' */
    key = 0;
    for (mp = margins; mp->code != pcl_ps_none; mp++, key++)
      if (mp->code != pcl_ps_custom) {
	/* Find entry in hpdj_mdim[] */
	const med_dim *dp = hpdj_mdim;
	while (dp->code != pcl_ps_none && dp->code != mp->code) dp++;
	if (dp->code != pcl_ps_none) {
#if HPDJ_INPUTMEDIA_PRN
	  rc = gdev_prn_input_page_size(key, &pdict, dp->width, dp->height);
#else
	  rc = gdev_write_input_page_size(key, &pdict, dp->width, dp->height);
#endif
	  if (rc < 0) return rc;
	}
      }

    /* Do the same for custom page sizes */
    if (custom_page_size_supported) {
      const custom_page_desc *cp;

      /*  In order to prevent a custom page size entry to be chosen in the case
	  where a rotation of a discrete page size entry would be possible, it
	  would be nice if we could also set a 'Priority' entry in
	  'InputAttributes' preferring all discrete sizes to the custom size.
	  However, until recently (gs 5.50) ghostscript's dictionary API
          permitted either integers or names as keys in dictionaries but not a
	  mixture (the two cases were distinguished by the parameter 'int_keys'
	  passed in param_begin_{read,write}_dict()); I have not yet found
	  the time to use the new feature.
	  This problem would disappear if one could require setpagedevice to
	  always choose portrait orientation with respect to device space (see
	  below), because hpdj defines 'InputAttributes' only to let a user
	  take advantage of setpagedevice's ability to match media sizes.
	  In that case, the input slot chosen is irrelevant.
       */

      for (cp = hpdj_model[dev->model].custom; cp->width_max > 0; cp++, key++) {
	/*  In this case we have to put a PageSize entry with a value of
	    [mxmin mymin mxmax mymax] into the dictionary we create for the
	    present key (see zmedia2.c).
	    This is not standard PostScript Level 2, though.
	    It would be nice if ghostscript were to support some means of
	    stating "use always portrait orientation in device space" here.
	    I sent a mail with such a suggestion to L. Peter Deutsch on
	    1997-11-05.
	*/
	char buffer[10];
	gs_param_float_array size_range;
	float range_data[4];
	gs_param_dict value;	/* << /PageSize [mxmin mymin mxmax mymax] >> */

	/* Create a new dictionary of length 1 having 'key' as its key in
	   whatever list it will be put into in a moment and return a reference
	   to it in 'value'. */
	sprintf(buffer, "%d", key);
	assert(strlen(buffer) < sizeof(buffer));
	value.size = 1;
	rc = param_begin_write_dict(pdict.list, buffer, &value, false);
	if (rc < 0) return rc;

	/* Initialize range_data[] */
	range_data[0] = cp->width_min;
	range_data[1] = cp->height_min;
	range_data[2] = cp->width_max;
	range_data[3] = cp->height_max;

	/* Now fill 'value' with one entry */
	size_range.data = range_data;
	size_range.size = 4;
	size_range.persistent = false;
	rc = param_write_float_array(value.list, "PageSize", &size_range);
	if (rc < 0) return rc;

	/* Put 'value' into 'pdict' */
	rc = param_end_write_dict(pdict.list, buffer, &value);
	if (rc < 0) return rc;
      }
    }

    /* Put 'pdict' into 'plist' as an InputAttributes dictionary */
#if HPDJ_INPUTMEDIA_PRN
    rc = gdev_prn_end_input_media(plist, &pdict);
#else
    rc = gdev_end_input_media(plist, &pdict);
#endif
    if (rc < 0) return rc;
  }
#endif	/* HPDJ_INPUTATTRIBUTES */

  return 0;
}

/******************************************************************************

  Function: is_word

  This function returns a non-zero value iff the string beginning at 's' is
  identical with the string pointed to by 'word' and is followed either by a
  blank character or '\0'.

******************************************************************************/

static int is_word(const char *s, const char *word)
{
  size_t l = strlen(word);
  if (strncmp(s, word, l) != 0) return 0;
  return s[l] == '\0' || isspace(s[l]);
}

/******************************************************************************

  Function: next_word

  This function returns a pointer to the *next* blank-separated word in the
  string pointed to by 's'. If the first character is not blank, it is
  considered to be part of the current word, i.e. the word to be returned is
  the one following.

  If there is no next word in this sense, the function returns NULL.

******************************************************************************/

static char *next_word(char *s)
{
  /* Skip current word */
  while (*s != '\0' && !isspace(*s)) s++;

  /* Skip intermediate blanks */
  while (*s != '\0' && isspace(*s)) s++;

  return *s == '\0'? NULL: s;
}

/******************************************************************************

  Function: hpdj_read_margins

  This function reads a margin description file and stores the result in '*dev'.
  The file name must already have been stored in 'dev->margin_file',
  'dev->margin_overrides' should be NULL.

  The function returns zero on success and a non-zero ghostscript error value
  otherwise. In the latter case, an error message will have been issued on
  stderr.

******************************************************************************/

#define BUFFER_SIZE	100

#define cleanup()	(free(list), fclose(f))


int hpdj_read_margins(hpdj_device *dev)
{
  char buffer[BUFFER_SIZE];
  FILE *f;
  float conversion_factor = BP_PER_IN;
    /* values read have to be multiplied by this value to obtain bp */
  int
    line = 0,	/* line number */
    read = 0;	/* number of margin entries read so far */
  margin_desc *list = NULL;

  /* Open the file */
  if ((f = fopen(dev->margin_file, "r")) == NULL) {
    const char *s;
    if ((s = strerror(errno)) == NULL) s = "unknown reason";
    fprintf(stderr, ERRPREF "Error opening the margin file\n"
      "    `%s'\n  for reading (%s).\n",
      dev->margin_file, s);
    return_error(gs_error_invalidfileaccess);
  }

  /* Loop over input lines */
  while (fgets(buffer, BUFFER_SIZE, f) != NULL) {
    char *s;
    margin_desc *current;
    int chars_read, temp;

    line++;

    /* Check for buffer overflow */
    if ((s = strchr(buffer, '\n')) == NULL && fgetc(f) != EOF) {
      fprintf(stderr,
	ERRPREF "Exceeding line length %d in margin file\n  %s, line %d.\n",
	BUFFER_SIZE - 2, dev->margin_file, line);
      cleanup();
      return_error(gs_error_limitcheck);
    }

    /* Eliminate the newline character */
    if (s != NULL) *s = '\0';

    /* Ignore blank and comment lines */
    s = buffer;
    while (isspace(*s)) s++;
    if (*s == '\0' || *s == '#') continue;

    /* Check for unit specification */
    if (is_word(s, "unit")) {
      char *unit_name = next_word(s);
      if (unit_name != NULL) s = next_word(unit_name);
      if (s == NULL) {
	if (is_word(unit_name, "in")) {
	  conversion_factor = BP_PER_IN;
	  continue;
	}
	if (is_word(unit_name, "mm")) {
	  conversion_factor = BP_PER_MM;
	  continue;
	}
      }
      /* The error message will be generated when the attempt to read the whole
	 line as a margin specification will fail. */
    }

    /* Extend the list */
    {
      margin_desc *new_list;
      new_list = (margin_desc *)realloc(list, (read+1)*sizeof(margin_desc));
      if (new_list == NULL) {
	fprintf(stderr, ERRPREF
	  "Memory allocation failure in hpdj_read_margins(), errno is %d.\n",
	  errno);
	cleanup();
	return_error(gs_error_VMerror);
      }
      list = new_list;
    }

    /* Read the line */
    current = list + read;
#if defined(_MSC_VER) && _MSC_VER <= 1100
    /* Bug in Visual C++ 5.0 SP1 (_MSC_VER is 1100): %n gives always 0. */
    if (sscanf(buffer, "%d %g %g %g %g", &temp, &current->left,
	  &current->bottom, &current->right, &current->top) != 5) {
#else
    if (sscanf(buffer, "%d %g %g %g %g %n", &temp, &current->left,
	  &current->bottom, &current->right, &current->top, &chars_read) != 5 ||
	buffer[chars_read] != '\0') {
#endif	/* _MSC_VER */
      fprintf(stderr,
	ERRPREF "Syntax error in margin file %s, line %d:\n    %s\n",
	dev->margin_file, line, buffer);
      cleanup();
      return_error(gs_error_rangecheck);
    }
    current->code = temp;
    read++;

    /* Convert to bp */
    current->left   *= conversion_factor;
    current->bottom *= conversion_factor;
    current->right  *= conversion_factor;
    current->top    *= conversion_factor;

    /* A margin for custom page sizes without the corresponding capability in
       the model is useless although it would not lead to a failure of hpdj.
       The same is true for unknown PCL page size codes. The user might not
       notice the reason without help, hence we check. */
    if (current->code == pcl_ps_custom) {
      if (hpdj_model[dev->model].custom == NULL) {
	fprintf(stderr, ERRPREF "The margin file %s\n"
	  "  contains a custom page size entry in line %d, "
	    "but custom page sizes\n"
	  "  are not supported by the DeskJet %s.\n",
	  dev->margin_file, line, hpdj_model[dev->model].name);
	cleanup();
	return_error(gs_error_rangecheck);
      }
    }
    else {
      int j = 0;
      while (hpdj_mdim[j].code != pcl_ps_none &&
	  current->code != hpdj_mdim[j].code) j++;
      if (hpdj_mdim[j].code == pcl_ps_none) {
	fprintf(stderr, ERRPREF
	  "Unknown PCL page size code (%d) in margin file\n  %s, line %d.\n",
	  (int)current->code, dev->margin_file, line);
	cleanup();
	return_error(gs_error_rangecheck);
      }
    }

    /* Too small a right margin is possibly harmful */
    if (current->right <= 0.124998*BP_PER_IN)	/* deliberately fuzzy */
      fprintf(stderr, WARNPREF
	"Margin file %s, line %d:\n  The right margin is less than 0.125 in.\n"
	"  This can lead to the print head being caught at the paper's edge.\n",
	dev->margin_file, line);
  }
  if (ferror(f)) {
    fprintf(stderr,
      ERRPREF "Unknown system error in reading `%s'.\n", dev->margin_file);
    cleanup();
    return_error(gs_error_invalidfileaccess);
  }
  fclose(f);

  /* Was the file empty? */
  if (read == 0) {
    fprintf(stderr, ERRPREF
      "The margin file %s\n  does not contain any margin definitions.\n",
      dev->margin_file);
    return_error(gs_error_rangecheck);
  }

  /* Create a list in the device structure */
  dev->margin_overrides = (margin_desc *) gs_malloc(read + 1,
    sizeof(margin_desc), "hpdj_read_margins");
  if (dev->margin_overrides == NULL) {
    fputs(ERRPREF
      "Memory allocation failure from gs_malloc() in hpdj_read_margins().\n",
      stderr);
    free(list);
    return_error(gs_error_VMerror);
  }

  /* Copy the list and append a sentinel entry */
  memcpy(dev->margin_overrides, list, read*sizeof(margin_desc));
  dev->margin_overrides[read].code = pcl_ps_none;

  /* Cleanup */
  free(list);

  return 0;
}

#undef BUFFER_SIZE
#undef cleanup

/******************************************************************************

  Function: set_derived_colour_data

  This routine determines and sets various derived values in the device
  structure based on the number of black and CMY levels requested.

  The values to be set are 'bits_per_components' and the fields 'depth',
  'max_gray', 'max_color', 'dither_grays' and 'dither_colors' in the
  'color_info' structure.

  The parameters 'black_levels' and 'cmy_levels' must be in the range
  0 to 256 without 1. At least one of the two must be positive.

******************************************************************************/

static void set_derived_colour_data(hpdj_device *dev)
{
  int
    factor = (dev->cmy_levels == 0? 1: 4),
      /* No distinction between CMY and CMY(+)K */
    levels,
    n;

  /* Choose equal number of bits for all components present */
  if (dev->black_levels >= dev->cmy_levels) levels = dev->black_levels;
  else levels = dev->cmy_levels;
  for (dev->bits_per_component = 1, n = 2; n < levels;
    dev->bits_per_component++) n *= 2;

  /* For the depth, consider all components and adjust to possible values.
     Ghostscript permits pixel depths 1, 2, 4, 8, 16, 24 and 32. */
  dev->color_info.depth = factor * dev->bits_per_component;
  if (dev->color_info.depth > 2) {
    if (dev->color_info.depth <= 4) dev->color_info.depth = 4;
    else if (dev->color_info.depth <= 8) dev->color_info.depth = 8;
    else dev->color_info.depth = ((dev->color_info.depth + 7)/8)*8;
      /* Next multiple of 8 */
  }

  /* max_color */
  if (dev->cmy_levels == 0) dev->color_info.max_color = 0;
  else dev->color_info.max_color = dev->cmy_levels - 1;

  /* max_gray */
  if (dev->black_levels > 0) dev->color_info.max_gray = dev->black_levels - 1;
  else dev->color_info.max_gray = dev->color_info.max_color;

  /* The dither parameters */
  if (dev->black_levels > 0) dev->color_info.dither_grays = dev->black_levels;
  else dev->color_info.dither_grays = dev->cmy_levels;
  if (dev->color_info.num_components == 1) dev->color_info.dither_colors = 0;
  else dev->color_info.dither_colors = dev->cmy_levels;

  return;
}

/******************************************************************************

  Function: hpdj_put_params

  This function reads a parameter list, extracts the parameters known to the
  device, and configures the device appropriately. This includes parameters
  defined by base classes.

  If an error occurs in the processing of parameters, the function will
  return a negative value, otherwise zero.

  This function does *not* exhibit transactional behaviour as requested in
  gsparam.h, i.e. on error the parameter values in the device structure
  might have changed. However, all values will be individually valid.

  The function does not check for all dependencies between values, see
  final_checks().

  Some of the parameters determine derived data in base classes or are relevant
  for printer initialization. Setting any of these parameters closes the
  device if it is open.

******************************************************************************/

int hpdj_put_params(gx_device *device, gs_param_list *plist)
{
  bool colour_mode_given_and_valid = false;
  gs_param_name pname;
  gs_param_string string_value;
  hpdj_device *dev = (hpdj_device *)device;
  int
    height = dev->height,	/* see comments at the end of this function */
    last_error = 0,
    temp,
    rc,
    width = dev->width;		/* see comments at the end of this function */

#ifdef HPDJ_TRACE
  fputs("! hpdj_put_params()...\n", stderr);
#endif

  /* BlackLevels. Various depending values will be adjusted below. */
  if ((rc = param_read_int(plist, (pname = "BlackLevels"), &temp)) == 0) {
    if (temp == 0 || 2 <= temp && temp <= 256) {
      if (dev->black_levels != temp) {
	gs_closedevice(device);
	  /* The close_device method can fail, but what should I do then? */
	dev->initialized = false;
      }
      dev->black_levels = temp;
    }
    else {
      fprintf(stderr, ERRPREF
	"The value for BlackLevels is outside the range permitted: %d.\n",
	temp);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
  }
  else if (rc < 0) last_error = rc;

  /* CMYLevels */
  if ((rc = param_read_int(plist, (pname = "CMYLevels"), &temp)) == 0) {
    if (temp == 0 || 2 <= temp && temp <= 256) {
      if (dev->cmy_levels != temp) {
      	gs_closedevice(device);
	dev->initialized = false;
      }
      dev->cmy_levels = temp;
    }
    else {
      fprintf(stderr, ERRPREF
	"The value for CMYLevels is outside the range permitted: %d.\n", temp);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
  }
  else if (rc < 0) last_error = rc;

  /* Colour mode. For those colonials across the pond we also accept the
     barbarized spelling variant.
  */
#define colour_mode(option)						\
  if ((rc = param_read_string(plist, (pname = option), &string_value)) == 0) { \
    rc = get_option_value(&string_value, hpdj_colour_mode_list, &temp); \
    if (rc != 0 || temp < mono || cmyk < temp) {			\
      if (rc != gs_error_VMerror) {					\
	fputs(ERRPREF "Unknown colour mode: `", stderr);		\
	fwrite(string_value.data, sizeof(char), string_value.size, stderr); \
	fputs("'.\n", stderr);						\
      }									\
      if (rc != 0) last_error = rc; else last_error = gs_error_rangecheck; \
      param_signal_error(plist, pname, last_error);			\
    }									\
    else colour_mode_given_and_valid = true;				\
  }									\
  else if (rc < 0) last_error = rc;

  colour_mode("ColorMode");
  colour_mode("ColourMode");	/* overrides if both are given */

#undef colour_mode

  if (colour_mode_given_and_valid) {
    if (dev->colour_mode != temp) {
      gs_closedevice(device);
      dev->initialized = false;
    }
    dev->colour_mode = temp;

    /* Set the colour model */
    switch(dev->colour_mode) {
    case mono:
      dev->color_info.num_components = 1; break;
    case cmy:
      /*FALLTHROUGH*/
    case cmy_plus_k:
      dev->color_info.num_components = 3; break;
    case cmyk:
      dev->color_info.num_components = 4; break;
    default:
      assert(0);
    }

    /* Adjust CMY levels if they are too small for colour */
    if (dev->color_info.num_components > 1 && dev->cmy_levels <= 0)
      dev->cmy_levels = 2;
    /* Adjust black levels for CMY */
    if (dev->colour_mode == cmy && dev->black_levels != 0)
      dev->black_levels = 0;

    /* Adjustments to pixel depth, 'max_gray', 'max_color' and dither levels
       will occur at the end of this routine */
  }

  /* Compression method */
  if ((rc = param_read_int(plist, (pname = "CompressionMethod"), &temp))
      == 0) {
    if (temp != pcl_cm_none && temp != pcl_cm_rl && temp != pcl_cm_tiff &&
	temp != pcl_cm_delta && temp != pcl_cm_crdr) {
      fprintf(stderr,
	ERRPREF "Unsupported compression method: %d.\n", temp);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
    else
      dev->compression_method = temp;
  }
  else if (rc < 0) last_error = rc;

  /* Dry time */
  if ((rc = param_read_int(plist, (pname = "DryTime"), &temp)) == 0) {
    if (-1 <= temp && temp <= 1200) dev->dry_time = temp;
    else {
      fprintf(stderr,
	ERRPREF "Illegal value for the dry time: %d.\n", temp);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
  }
  else if (rc < 0) last_error = rc;

  /* Manual feed */
  {
    bool temp;
    if ((rc = param_read_bool(plist, (pname = "ManualFeed"), &temp)) == 0)
      dev->manualfeed = (temp? requested: declined);
    else if (rc < 0) last_error = rc;
  }

  /* Margin description file */
  if ((rc = param_read_string(plist, (pname = "MarginFile"), &string_value))
      == 0) {
    /* Free old storage */
    if (dev->margin_file != NULL) {
      gs_free(dev->margin_file, strlen(dev->margin_file) + 1, sizeof(char),
	"hpdj_put_params");
      dev->margin_file = NULL;
    }
    if (dev->margin_overrides != NULL) {
      int n = 0;
      while (dev->margin_overrides[n].code != pcl_ps_none) n++;
      gs_free(dev->margin_overrides, n+1, sizeof(margin_desc),
	"hpdj_put_params");
      dev->margin_overrides = NULL;
    }

    /* Read margin file, unless the name is the empty string */
    if (string_value.size > 0) {
      dev->margin_file = (char *)gs_malloc(string_value.size + 1, sizeof(char),
	"hpdj_put_params");
      if (dev->margin_file == NULL) {
	fputs(ERRPREF
	  "Memory allocation failure from gs_malloc() in hpdj_put_params().\n",
	  stderr);
	last_error = gs_error_VMerror;
	param_signal_error(plist, pname, last_error);
      }
      else {
	strncpy(dev->margin_file, (const char *)string_value.data,
	  string_value.size);
	dev->margin_file[string_value.size] = '\0';
	if ((rc = hpdj_read_margins(dev)) != 0) {
	  last_error = rc;
	  param_signal_error(plist, pname, last_error);
	  gs_free(dev->margin_file, strlen(dev->margin_file) + 1, sizeof(char),
	    "hpdj_put_params");
	  dev->margin_file = NULL;
	}
      }
    }
  }
  else if (rc < 0) last_error = rc;

  /* Media type */
  if ((rc = param_read_string(plist, (pname = "MediaType"), &string_value))
      == 0) {
    /*  We accept numerical and string values. Numerical values defined for
	Series 500 and 800 DeskJets are 0-4, and newer DeskJets use 5 for photo
	paper. We give the user the benefit of the doubt, though, because the
	value is simply passed through to the printer, except for the older
	DeskJets where we map illegal values to "plain paper".
	If the user specifies a string, however, it must be a known one.
    */
    rc = get_option_value(&string_value, media_type_list, &dev->media_type);
    if (rc != 0) {
      if (rc != gs_error_VMerror) {
	fputs(ERRPREF "Unknown media type: `", stderr);
	fwrite(string_value.data, sizeof(char), string_value.size, stderr);
	fputs("'.\n", stderr);
      }
      last_error = rc;
      param_signal_error(plist, pname, last_error);
    }
    else if (dev->media_type < 0 || 5 < dev->media_type)
      fprintf(stderr, WARNPREF "Unknown media type: %d.\n", dev->media_type);
  }
  else if (rc < 0) last_error = rc;

  /* Model Name */
  if ((rc = param_read_string(plist, (pname = "Model"), &string_value)) == 0) {
    /* This property must be a known string. */
    int j = 0;
    while (hpdj_model[j].name != NULL &&
	(string_value.size != strlen(hpdj_model[j].name) ||
	  strncmp((const char *)string_value.data, hpdj_model[j].name,
	    string_value.size) != 0))
      j++;
      /* param_read_string() does not return null-terminated strings. */
    if (hpdj_model[j].name != NULL) {
      if (dev->model != hpdj_model[j].model) {
      	gs_closedevice(device);
	dev->initialized = false;
	dev->model = hpdj_model[j].model;
      }
    }
    else {
      fputs(ERRPREF "Unknown model name: `", stderr);
      fwrite(string_value.data, sizeof(char), string_value.size, stderr);
      fputs("'.\n", stderr);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
  }
  else if (rc < 0) last_error = rc;

#ifndef HPDJ_NO_PAGECOUNTFILE
  /* Page count file */
  if ((rc = param_read_string(plist, (pname = "PagecountFile"), &string_value))
      == 0) {
    /* Free old storage */
    if (dev->pagecount_file != NULL) {
      gs_free(dev->pagecount_file, strlen(dev->pagecount_file) + 1,
	sizeof(char), "hpdj_put_params");
      dev->pagecount_file = NULL;
    }

    /* Store file name unless it is the empty string */
    if (string_value.size > 0) {
      dev->pagecount_file = (char *)gs_malloc(string_value.size + 1,
	sizeof(char), "hpdj_put_params");
      if (dev->pagecount_file == NULL) {
	fputs(ERRPREF
	  "Memory allocation failure from gs_malloc() in hpdj_put_params().\n",
	  stderr);
	last_error = gs_error_VMerror;
	param_signal_error(plist, pname, last_error);
      }
      else {
	strncpy(dev->pagecount_file, (const char *)string_value.data,
	  string_value.size);
	dev->pagecount_file[string_value.size] = '\0';
      }
    }
  }
#endif	/* HPDJ_NO_PAGECOUNTFILE */

  /* Print Quality */
  if ((rc = param_read_string(plist, (pname = "PrintQuality"), &string_value))
      == 0) {
    /*  The only known values are -1, 0 and 1. Again, however, we assume the
	user knows what s/he is doing if another value is given. */
    rc = get_option_value(&string_value, print_quality_list,
      &dev->print_quality);
    if (rc != 0) {
      if (rc != gs_error_VMerror) {
	fputs(ERRPREF "Unknown print quality: `", stderr);
	fwrite(string_value.data, sizeof(char), string_value.size, stderr);
	fputs("'.\n", stderr);
      }
      last_error = rc;
      param_signal_error(plist, pname, last_error);
    }
    else if (dev->print_quality < -1 || 1 < dev->print_quality)
      fprintf(stderr, WARNPREF "Unknown print quality: %d.\n",
	dev->print_quality);
  }
  else if (rc < 0) last_error = rc;

  /* Undocumented PCL command 1 */
  if ((rc = param_read_int(plist, (pname = "PCLundoc1"), &dev->undoc1)) < 0)
    last_error = rc;

  /* Determine various derived colour parameters */
  set_derived_colour_data(dev);

  /* Catch a specification of BitsPerPixel --- otherwise a wrong value gives
     just a rangecheck error from gs without any readily understandable
     information. It's a bit confusing for the user here if we silently
     tolerate this if the value happens to be right, but at least in gs 3.33
     BitsPerPixel turns up at this point in certain situations even if the
     user did not specify it.
  */
  if ((rc = param_read_int(plist, (pname = "BitsPerPixel"), &temp)) == 0) {
    if (temp != dev->color_info.depth) {
      fprintf(stderr, ERRPREF "Inconsistent value for `BitsPerPixel': %d.\n"
	"  You should never specify this parameter when using hpdj.\n",
	temp);
      last_error = gs_error_rangecheck;
      param_signal_error(plist, pname, last_error);
    }
  } else if (rc < 0) last_error = rc;

  /* Process parameters defined by base classes (should occur after treating
     parameters defined for the derived class, see gsparam.h) */
  if ((rc = gdev_prn_put_params(device, plist)) < 0 ||
    rc > 0 && last_error >= 0) last_error = rc;

  /* Are there any parameters left unprocessed? Can one check that?
     The main part of ghostscript does not check for unrecognized options in
     the command line. */

  if (last_error < 0) return_error(last_error);

  /* In PostScript, you can redefine the media size within the PostScript
     program. The hpdj device accepts this only before the first page has
     been sent to the output file. Unfortunately, I can't simply put this in
     the open_device routine because, although the main part of ghostscript
     closes the device when this occurs, gdev_prn_put_params() sets the device
     to "closed" before calling gx_default_put_params() and resets it to its
     previous state afterwards.
     I therefore assume that changes in media size either (a) really close the
     device, leading to a later call to hpdj_open_device(), or (b) always pass
     through this function.
  */
  if (dev->is_open && (dev->width != width || dev->height != height)) {
    /* A change on the fly. I can't use gdev_prn_file_is_new() here, hence the
       need for 'initialized'. */
    if (!dev->initialized) rc = hpdj_set_page_layout(dev);
    else {
      /* Change after writing the first page */
      fputs(ERRPREF
	"This driver does not support changes in media size or resolution "
	"within a document.\n",
	stderr);
      /*  Actually, one could easily support this, at least between pages.
	  DeskJets, however, do not have the ability to automatically select
	  one of several input trays based on the media size demanded. And it's
	  easy to split a PostScript file, e.g. with psselect from Angus
	  Duggan's psutils package. */
      rc = -1;
    }
    if (rc != 0) {
      /* If one simply returns a non-zero code, ghostscript will ignore it.
	 I've tried to signal errors for "PageSize" (core dump) and ".MediaSize"
	 (ignored), and now I've had enough: I set a global flag which will be
	 checked in hpdj_print_page().
      */
      dev->is_valid = false;
      return_error(gs_error_rangecheck);
    }
  }

  return last_error;
}
