/*
 * extio.c
 *
 * Copyright 1999 Silicon Graphics, Inc.
 *
 * Derived from e2fsprogs lib/ext2fs/unix_io.c
 * Copyright (C) 1993, 1994, 1995 Theodore Ts'o.
 */
/* #define ARC_IO_ALLOW_WRITE */

#include <stddef.h>
#include <stdio.h>
#include <string.h>

#include <sys/types.h>
#include <linux/ext2_fs.h>
#include <ext2fs/ext2fs.h> /* In e2fsprogs-devel with Red Hat Linux */
#include <arc.h>

  
struct arc_private_data {
    int	magic;
    OPENMODE mode;
    ULONG fileID;
};

static errcode_t arc_open(const char *name, int flags, io_channel *channel);
static errcode_t arc_close(io_channel channel);
static errcode_t arc_set_blksize(io_channel channel, int blksize);
static errcode_t arc_read_blk
    (io_channel channel, unsigned long block, int count, void *data);
static errcode_t arc_write_blk
    (io_channel channel, unsigned long block, int count, const void *data);
static errcode_t arc_flush(io_channel channel);

static struct struct_io_manager struct_arc_manager = {
    EXT2_ET_MAGIC_IO_MANAGER,
    "ARC PROM I/O Manager",
    arc_open,
    arc_close,
    arc_set_blksize,
    arc_read_blk,
    arc_write_blk,
    arc_flush
};
io_manager arc_io_manager = &struct_arc_manager;


static errcode_t
arc_open(const char *name, int flags, io_channel *pchannel)
{
    io_channel channel = NULL;
    struct arc_private_data *priv;
    errcode_t status;

    if (name == NULL)
	return EXT2_ET_BAD_DEVICE_NAME;

    status =
	ext2fs_get_mem(sizeof(struct struct_io_channel), (void **)&channel);
    
    if (status == 0) {
	memset(channel, 0, sizeof(struct struct_io_channel));

	channel->name = NULL;
	channel->private_data = NULL;
	channel->magic = EXT2_ET_MAGIC_IO_CHANNEL;
	channel->manager = arc_io_manager;
	channel->block_size = 1024;
	channel->read_error = NULL;
	channel->write_error = NULL;
	channel->refcount = 1;

	status = ext2fs_get_mem(strlen(name)+1, (void **)&channel->name);
	if (status == 0) {
	    strcpy(channel->name, name);

	    status =
		ext2fs_get_mem(sizeof(struct arc_private_data),(void **) &priv);
	    if (status == 0) {
		channel->private_data = priv;
		priv->magic = EXT2_ET_BAD_MAGIC;
		priv->mode =(flags & IO_FLAG_RW) ? OpenReadWrite : OpenReadOnly;
		status = ArcOpen((char *)name, priv->mode, &priv->fileID);
	    }
	}
    }

    if (status == 0) {
	*pchannel = channel;
    } else if (channel != NULL) {
	if (channel->name != NULL)
	    ext2fs_free_mem((void **)&channel->name);
	if (channel->private_data != NULL)
	    ext2fs_free_mem((void **)&channel->private_data);
	ext2fs_free_mem((void **)&channel);
    }
    return status;
}


static errcode_t
arc_close(io_channel channel)
{
    struct arc_private_data *priv;
    errcode_t status = 0;

    EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
    priv = (struct arc_private_data *) channel->private_data;
    EXT2_CHECK_MAGIC(priv, EXT2_ET_BAD_MAGIC);

    if (--channel->refcount == 0) {
	status = ArcClose(priv->fileID);
	if (channel->name != NULL)
		ext2fs_free_mem((void **)&channel->name);
	if (channel->private_data != NULL)
		ext2fs_free_mem((void **)&channel->private_data);
	ext2fs_free_mem((void **)&channel);
    }
    return status;
}


static errcode_t
arc_set_blksize(io_channel channel, int blksize)
{
    struct arc_private_data *priv;

    EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
    priv = (struct arc_private_data *) channel->private_data;
    EXT2_CHECK_MAGIC(priv, EXT2_ET_BAD_MAGIC);

    if (channel->block_size != blksize)
	channel->block_size = blksize;
    return 0;
}


void
mul64(unsigned long block, int blocksize, LARGEINTEGER *result)
{
    ULONG m1l = block & 0x0FFFF, m1h = (block >> 16) & 0x0FFFF;
    ULONG m2l = blocksize & 0x0FFFF, m2h = (blocksize >> 16) & 0x0FFFF;
    ULONG i1 = m1l * m2h, i2 = m1h * m2l;

    result->HighPart = (m1h*m2h) + ((i1>>16) & 0x0FFFF) + ((i2>>16) & 0x0FFFF);
    i1 = (i1 & 0x0FFFF) + (i2 & 0x0FFFF) + (((m1l * m2l) >> 16) & 0x0FFFF);
    result->LowPart = ((i1 & 0x0FFFF) << 16) + ((m1l * m2l) & 0x0FFFF);
    result->HighPart += (i1 >> 16) & 0x0FFFF;
}


static errcode_t
arc_seek(io_channel channel, unsigned long block)
{
    struct arc_private_data *priv;
    LARGEINTEGER position;
    errcode_t status;

    priv = (struct arc_private_data *) channel->private_data;
    mul64(block, channel->block_size, &position);
    return ArcSeek(priv->fileID, &position, SeekAbsolute);
}


static errcode_t
arc_read_blk(io_channel channel, unsigned long block, int count, void *buf)
{
    struct arc_private_data *priv;
    errcode_t status;

    EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
    priv = (struct arc_private_data *) channel->private_data;
    EXT2_CHECK_MAGIC(priv, EXT2_ET_BAD_MAGIC);

    status = arc_seek(channel, block);
    if (status == 0) {
	size_t length = (count < 0) ? -count : count * channel->block_size;
	ULONG nread = 0;

	status = ArcRead(priv->fileID, buf, length, &nread);
	if ((nread > 0) && (nread < length)) {
	    status = EXT2_ET_SHORT_READ;
	    memset(((char *)buf) + nread, 0, length - nread);
	}
	if ((status != ESUCCESS) && (channel->read_error != NULL)) {
	    status = (channel->read_error)
			(channel, block, count, buf, length, nread, status);
	}
    }

    return status;
}


static errcode_t
arc_write_blk
    (io_channel channel, unsigned long block, int count, const void *buf)
{
    struct arc_private_data *priv;
    errcode_t status;

    EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
    priv = (struct arc_private_data *) channel->private_data;
    EXT2_CHECK_MAGIC(priv, EXT2_ET_BAD_MAGIC);

    status = arc_seek(channel, block);
#ifdef ARC_IO_ALLOW_WRITE
    if (status == 0) {
	size_t length = (count < 0) ? -count : count * channel->block_size;
	ULONG nwritten = 0;

	status = ArcWrite(priv->fileID, (void *) buf, length, &nwritten);
	if ((nwritten > 0) && (nwritten < length))
	    status = EXT2_ET_SHORT_WRITE;
	if ((status != ESUCCESS) && (channel->write_error != NULL)) {
	    status = (channel->write_error)
			(channel, block, count, buf, length, nwritten, status);
	}
    }
#endif /* ARC_IO_ALLOW_WRITE */

    return status;
}


static errcode_t
arc_flush(io_channel channel)
{
    struct arc_private_data *priv;
    
    EXT2_CHECK_MAGIC(channel, EXT2_ET_MAGIC_IO_CHANNEL);
    priv = (struct arc_private_data *) channel->private_data;
    EXT2_CHECK_MAGIC(priv, EXT2_ET_BAD_MAGIC);
    
    return 0;
}


/* Hack in some stuff to make ext2fs library work */

struct error_table {
    char const * const * msgs;
    long base;
    int n_msgs;
};
struct et_list {
    struct et_list *next;
    const struct error_table * table;
};
struct et_list *_et_list = NULL;

void
com_err(const char *whoami, long error, const char *format, ...)
{
}

const char *
ext2fs_strerror(long error)
{
    struct et_list *list = _et_list;

    while (list != NULL) {
	if ((error >= list->table->base)
	    && (error < (list->table->base + list->table->n_msgs)))
	{
	    return list->table->msgs[error - list->table->base];
	}

	list = list->next;
    }
}


int
sprintf(void)
{
    return EOF;
}


/* Only one file open at a time */
static ext2_file_t file;
static ULONG fd;

static void
arc_file_open(const CHAR *partition, const CHAR *filename)
{
	LONG status;
	ULONG nb = strlen(partition) + strlen(filename);
	CHAR *path = (CHAR *)malloc(nb);
	if (path == NULL) {
		Fatal("Malloc of %d for %s and %s failed\n",
						nb, partition, filename);
	}
	strcpy(path, partition);
	strcpy(path + strlen(partition), filename + 1); /* skip leading '/' */
printf("ARC opening %s\n", path);
	status = ArcOpen(path, OpenReadOnly, &fd);
	if (status != ESUCCESS) {
		Fatal("Open of %s failed\n", path);
	}
}

static void
arc_seek_read(ULONG off, VOID *cp, ULONG nb, ULONG *got)
{
	LONG status;
	LARGEINTEGER li;
	li.HighPart = 0;
	li.LowPart = off;
	status = ArcSeek(fd, &li, SeekAbsolute);
	if (status != ESUCCESS) {
		Fatal("Seek to 0x%x failed\n", off);
	}
printf("ARC reading %u bytes at offset %u\n", nb, off);
	status = ArcRead(fd, cp, nb, got);
	if (status != ESUCCESS) {
		Fatal("Read of 0x%x bytes at 0x%x failed\n", nb, off);
	}
}


void
file_open(const char *partition, const char *filename)
{
    extern io_manager arc_io_manager;
    ext2_filsys fs;
    ino_t file_inode;
    errcode_t status;

    initialize_ext2_error_table();

    status = ext2fs_open(partition, 0, 0, 0, arc_io_manager, &fs);
    if (status != 0) {
        /* Maybe ARC can read it directly (FAT floppy, etc) */
        return arc_file_open(partition, filename);
    }
printf("EXT2FS opening %s in partition %s\n", filename, partition);

    status = ext2fs_namei_follow
	(fs, EXT2_ROOT_INO, EXT2_ROOT_INO, filename, &file_inode);
    if (status != 0) {
	printf("ext2fs - %s\n", ext2fs_strerror(status));
	Fatal("Can't find file %s in partition %s\n", filename, partition);
    }

    status = ext2fs_file_open(fs, file_inode, 0, &file);
    if (status != 0) {
	printf("ext2fs - %s\n", ext2fs_strerror(status));
	Fatal("Can't open file %s in partition %s\n", filename, partition);
    }
}

void
file_seek_read(ext2_off_t off, char *cp, unsigned int nb, unsigned int *got)
{
    errcode_t status;

    if (fd != 0) {
	return arc_seek_read((ULONG)off, (VOID*)cp, (ULONG)nb, (ULONG*)got);
    }

printf("EXT2FS reading %u bytes at offset %u\n", nb, off);
    status = ext2fs_file_lseek(file, off, EXT2_SEEK_SET, NULL);
    if (status != 0) {
	printf("ext2fs - %s\n", ext2fs_strerror(status));
	Fatal("Cannot seek to 0x%\n", off);
    }

    status = ext2fs_file_read(file, cp, nb, got);
    if (status != 0) {
	printf("ext2fs - %s\n", ext2fs_strerror(status));
	Fatal("Cannot read 0x% bytes at 0x%x\n", nb, off);
    }
}
