/*
 * ARC Linux kernel loader main line -- see _start().
 *
 * All ELF stuff in here.
 * File i/o abstracted to file_open() and file_seek_read().
 *
 * Copyright 1999 Silicon Graphics, Inc.
 */

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <types.h>

#include <arc.h>
#include <elf.h>

#include <sys/types.h>

#define ANSI_CLEAR	"\033[2J"

#define PAGE_SIZE	4096
#define STACK_PAGES	16

typedef enum { False = 0, True } Boolean;

/* Difference between virtual and physical addresses */
ULONG voffset = 0xc0000000;

/*
 *  Reserve this memory for loading kernel
 *  Don't put loader structures there because they would be overwritten
 */
ULONG reserve_base = 0x0100000;
ULONG reserve_size = 0x0200000;


void
InitMalloc(void)
{
    MEMORYDESCRIPTOR *current = NULL;
    ULONG stack = (ULONG) &current;

    current = ArcGetMemoryDescriptor(current);
    while (current != NULL) {
	/*
	 *  The spec says we should have an adjacent FreeContiguous
	 *  memory area that includes our stack.  It would be much
	 *  easier to just look for that and give it to malloc, but
	 *  the 320 only shows FreeMemory areas, no FreeContiguous.
	 *  Oh well.
	 */
	if (current->Type == FreeMemory) {
	    ULONG start = current->BasePage * PAGE_SIZE;
	    ULONG end = start + (current->PageCount * PAGE_SIZE);

	    /* Leave some space for our stack */
	    if ((stack >= start) && (stack < end))
		end = (stack - (STACK_PAGES * PAGE_SIZE)) & ~(PAGE_SIZE - 1);

	    /* Don't use memory from reserved region */
	    if ((start >= reserve_base)&&(start < (reserve_base+reserve_size))) 
		start = reserve_base + reserve_size;
	    if ((end > reserve_base) && (end <= (reserve_base + reserve_size))) 
		end = reserve_base;

	    if (end > start)
		arclib_malloc_add(start, end - start);
	}

	current = ArcGetMemoryDescriptor(current);
    }
}


CHAR *LoadIdentifier = NULL;
CHAR *OSLoadPartition = NULL;
CHAR *OSLoadFilename = NULL;
CHAR *OSLoadOptions = "";

void
ProcessArguments(LONG argc, CHAR *argv[])
{
    LONG arg;
    CHAR *equals;
    size_t len;

    for (arg = 1; arg < argc; arg++) {
	equals = strchr(argv[arg], '=');
	if (equals != NULL) {
	    len = equals - argv[arg];
	    if (strncmp(argv[arg], "LoadIdentifier", len) == 0)
		LoadIdentifier = equals + 1;
	    if (strncmp(argv[arg], "OSLoadPartition", len) == 0)
		OSLoadPartition = equals + 1;
	    if (strncmp(argv[arg], "OSLoadFilename", len) == 0)
		OSLoadFilename = equals + 1;
	    if (strncmp(argv[arg], "OSLoadOptions", len) == 0) {
		/* Copy options to local memory to avoid overwrite later */
		OSLoadOptions = strdup(equals + 1);
		if (OSLoadOptions == NULL)
		    Fatal("Cannot allocate memory for options string\n");
	    }
	}
    }

    /* Workaround for annoying OSLoadPartion behavior in GUI */
    if ((OSLoadFilename != NULL) && (*OSLoadFilename != '/')) {
	CHAR *path = strchr(OSLoadFilename, '/');

	if (path != NULL) {
	    OSLoadPartition = malloc((path - OSLoadFilename) + 1);

	    if (OSLoadPartition == NULL)
		Fatal("Cannot allocate memory for argument processing\n");

	    strncpy(OSLoadPartition, OSLoadFilename, path-OSLoadFilename);
	    OSLoadPartition[path - OSLoadFilename] = '\0';
	}

	OSLoadFilename = path;
    }
}


void
LoadProgramSegments(Elf32_Ehdr *header)
{
    int idx;
    Boolean loaded = False;
    Elf32_Phdr *segment, *segments;
    size_t size = header->e_phentsize * header->e_phnum;

    if (size <= 0)
	Fatal("No program segments\n");

    segments = malloc(size);
    if (segments == NULL)
	Fatal("Cannot allocate memory for segment headers\n");

    file_seek_read(header->e_phoff, segments, size, NULL);

    segment = segments;
    for (idx = 0; idx < header->e_phnum; idx++) {
        void *memp;
	if (segment->p_type == PT_LOAD) {
	    printf("Loading program segment %u at 0x%x, size = 0x%x\n",
		    idx + 1, segment->p_vaddr - voffset, segment->p_filesz);

            memp = (void *) (segment->p_vaddr - voffset);
            file_seek_read(segment->p_offset, memp, segment->p_filesz, NULL);

	    size = segment->p_memsz - segment->p_filesz;
	    if (size > 0) {
		printf("Zeroing memory at 0x%x, size = 0x%x\n",
			(segment->p_vaddr + segment->p_filesz) - voffset,
			size);
		memset((void *)((segment->p_vaddr+segment->p_filesz) - voffset),
			0,
			size);
	    }

	    loaded = True;
	}

	segment = (Elf32_Phdr *) (((char *)segment) + header->e_phentsize);
    }

    if (!loaded)
	Fatal("No loadable program segments found\n");

    free(segments);
}

#ifdef VERIFY
/*
 * Something naive and simplistic that we know and trust
 */
int
cmp(char *a, char *b, int nb)
{
	while (nb--) {
		if (*a++ != *b++) {
			return nb;	/* bad */
		}
	}
	return 0; /* good */
}

int
iszero(char *a, int nb)
{
	while (nb--) {
		if (*a++ != 0) {
			return nb;	/* bad */
		}
	}
	return 0; /* good */
}

/*
 * Compare the segment to mem.
 * If all is well we reward the caller by returning, else Fatal().
 */
void
VerifySegment(ULONG off, unsigned int nbytes, char* mem)
{
#define NBYTES 4096 /* Can't eat too much stack (?) */
	char buf[NBYTES];
	unsigned int got;

	while (nbytes) {
#define MIN(a,b) (a < b ? a : b)
		int n = MIN(nbytes, NBYTES);
#undef MIN
		file_seek_read(off, buf, n, &got);
		if (n=cmp(buf, mem, got)) {
			Fatal("Failed compare at 0x%x, n 0x%x\n", off, n);
		}
		nbytes -= got;
		off += got;
		mem += got;
	}
#undef NBYTES
}

void
VerifyProgramSegments(Elf32_Ehdr *header)
{
    int idx;
    Boolean loaded = False;
    Elf32_Phdr *segment, *segments;
    size_t size = header->e_phentsize * header->e_phnum;

    if (size <= 0)
	Fatal("No program segments\n");

    segments = malloc(size);
    if (segments == NULL)
	Fatal("Cannot allocate memory for segment headers\n");

    file_seek_read(header->e_phoff, segments, size, NULL);

    segment = segments;
    for (idx = 0; idx < header->e_phnum; idx++) {
	if (segment->p_type == PT_LOAD) {
	    printf("Verifying program segment %u at 0x%x, size = 0x%x\n",
		    idx + 1, segment->p_vaddr - voffset, segment->p_filesz);

	    VerifySegment(segment->p_offset, segment->p_filesz,
				(void*)(segment->p_vaddr - voffset));

	    size = segment->p_memsz - segment->p_filesz;
	    if (size > 0) {
		int n;
                void *p = (void*)
			((segment->p_vaddr + segment->p_filesz) - voffset);
		printf("Verifying zeroed memory at 0x%x, size = 0x%x\n",
			p,
			size);
                if (n=iszero(p, size)) {
			Fatal("Failed bss zero compare at 0x%x\n", n);
		}
	    }
	}

	segment = (Elf32_Phdr *) (((char *)segment) + header->e_phentsize);
    }

    free(segments);
}
#endif /* VERIFY */



void
LoadKernel(const char *partition, const char *filename)
{
    Elf32_Ehdr header;
    extern void StartKernel(ULONG entry, const char *cmdline);

    file_open(partition, filename);

    file_seek_read(0, (void *) &header, sizeof(header), NULL);

    if (memcmp(&(header.e_ident[EI_MAG0]), ELFMAG, SELFMAG) != 0)
	Fatal("Not an ELF file\n");

    if (header.e_ident[EI_CLASS] != ELFCLASS32)
	Fatal("Not a 32-bit file\n");

    if (header.e_ident[EI_DATA] != ELFDATA2LSB)
	Fatal("Not a little-endian file\n");

    if (header.e_ident[EI_VERSION] != EV_CURRENT)
	Fatal("Wrong ELF version\n");

    if (header.e_type != ET_EXEC)
	Fatal("Not an executable file\n");

    if ((header.e_machine != EM_386) && (header.e_machine != EM_486))
	Fatal("Unsupported machine type\n");

    if (header.e_version != EV_CURRENT)
	Fatal("Wrong ELF version\n");

    LoadProgramSegments(&header);

#ifdef VERIFY
    VerifyProgramSegments(&header);
#endif

    printf("Starting kernel; entry point = 0x%x\n",
	    ((ULONG)header.e_entry) - voffset);
    StartKernel(((ULONG)header.e_entry) - voffset, OSLoadOptions);
}


void
_start(LONG argc, CHAR *argv[], CHAR *envp[])
{
    /* Print identification */
    printf(ANSI_CLEAR "\nARC Linux unified loader, 21jul99\n\n");

    InitMalloc();

    ProcessArguments(argc, argv);

    if (LoadIdentifier != NULL)
	printf("LoadIdentifier  :  %s\n", LoadIdentifier);
    if (OSLoadPartition != NULL)
	printf("OSLoadPartition :  %s\n", OSLoadPartition);
    if (OSLoadFilename != NULL)
	printf("OSLoadFilename  :  %s\n", OSLoadFilename);
    if (OSLoadOptions != NULL)
	printf("OSLoadOptions   :  %s\n", OSLoadOptions);
    printf("\n");

    if (OSLoadPartition == NULL)
	Fatal("Invalid load parition\n");
    if (OSLoadFilename == NULL)
	Fatal("Invalid load filename\n");

    LoadKernel(OSLoadPartition, OSLoadFilename);

    /* Not likely to get back here in a functional state, but what the heck */
    Wait("\n--- Press <spacebar> to restart ---");
    ArcRestart();
}
