/* $Id: utils.c,v 1.22 2001/03/31 01:23:35 mfleming Exp $
 *
 * Misc. routines which are used by the various functions to handle strings
 * and memory allocation and pretty much anything else we can think of. Also,
 * the load cutoff routine is in here, along with the HTML show stats
 * function. Could not think of a better place for it, so it's in here.
 *
 * Copyright (C) 1998  Steven Young
 * Copyright (C) 1999  Robert James Kaes (rjkaes@flarenet.com)
 * Copyright (C) 2000  Eazel, Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Authors: Stephen Young
 *          Robert James Kaes <rjkaes@flarenet.com>
 *          Robey Pointer <robey@eazel.com>
 *          Mike Fleming <mfleming@eazel.com>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>
#include <glib.h>
#include <stdarg.h>
#include <string.h>

#include "sock.h"
#include "proxy.h"
#include "log.h"
#include "utils.h"
#include "compat.h"
#include "session.h"

#define DEFAULT_SESSION_DIRECTORY_MODE (0700)
#define DEFAULT_SESSION_DIRECTORY ".eazel-proxy"
#define LOCKFILE_NAME "eazel-proxy-pid"
#define PORTFILE_NAME "eazel-proxy-port"
#define AUTHFILE_NAME "eazel-auth-token"


/*
 * Display the statics of the proxy server.
 */
int show_stats(Socket *sock)
{
	char *outbuf;
	static char *msg = "HTTP/1.0 200 OK\r\n" \
	    "Content-type: text/html\r\n\r\n" \
	    "<html><head><title>%s stats</title></head>\r\n" \
	    "<body>\r\n" \
	    "<center><h2>%s %s run-time statistics</h2></center><hr>\r\n" \
	    "<blockquote>\r\n" \
	    "Number of requests: %lu<br>\r\n" \
	    "Number of connections: %lu<br>\r\n" \
	    "Number of bad connections: %lu<br>\r\n" \
	    "Number of opens: %lu<br>\r\n" \
	    "Number of listens: %lu<br>\r\n" \
	    "Number of bytes (tx): %lu<br>\r\n" \
	    "Number of bytes (rx): %lu<br>\r\n" \
	    "Number of garbage collects: %lu<br>\r\n" \
	    "Number of idle connection kills: %lu<br>\r\n" \
	    "</blockquote>\r\n</body></html>\r\n";

	g_assert(sock);

	outbuf = g_strdup_printf(msg, PACKAGE, PACKAGE, VERSION, stats.num_reqs, stats.num_cons,
				 stats.num_badcons, stats.num_opens, stats.num_listens,
				 stats.num_tx, stats.num_rx, stats.num_garbage, stats.num_idles);

	socket_write(sock, outbuf, strlen(outbuf));
	g_free(outbuf);

	return 0;
}

/*
 * Display an error to the client.
 */
int http_error(Socket *sock, int err, const char *msg)
{
	char *outbuf;
	static char *premsg = "HTTP/1.0 %d %s\r\n" \
	    "Content-type: text/html\r\n\r\n" \
	    "<html><head><title>%s</title></head>\r\n" \
	    "<body>\r\n" \
	    "<font size=\"+2\">Fatal Error!</font><br>\r\n" \
	    "An error of type %d occurred: %s\r\n" \
	    "<hr>\r\n" \
	    "<font size=\"-1\"><em>Generated by %s %s</em></font>\r\n" \
	    "</body></html>\r\n";

	g_assert(sock);
	g_assert(err > 0);
	g_assert(msg);

	outbuf = g_strdup_printf(premsg, err, msg, msg, err, msg, PACKAGE, VERSION);

	socket_write(sock, outbuf, strlen(outbuf));
	g_free(outbuf);

	return 0;
}

/* Note that header_list is *freed* */
int http_error_headers(Socket *sock, int err, const char *msg, GList *header_list)
{
	char *outbuf;
	GList *current_position;
	static char *premsg = 
	    "<html><head><title>%s</title></head>\r\n"
	    "<body>\r\n"
	    "<font size=\"+2\">Fatal Error!</font><br>\r\n"
	    "An error of type %d occurred: %s\r\n"
	    "<hr>\r\n"
	    "<font size=\"-1\"><em>Generated by %s %s</em></font>\r\n"
	    "</body></html>\r\n";

	g_assert(sock);
	g_assert(err > 0);
	g_assert(msg);

	socket_cork (sock);

	outbuf = g_strdup_printf ("HTTP/1.0 %d %s\r\n", err, msg);
	socket_write (sock, outbuf, strlen (outbuf));
	g_free (outbuf);

	socket_write (sock, "Content-Type: text/html\r\n", strlen ("Content-Type: text/html\r\n"));

	for ( 	current_position = header_list ;
		current_position ; 
		current_position = g_list_next (current_position)
	) {
		err = socket_write (sock, (char *)current_position->data, strlen ((char *)current_position->data));
		err = socket_write (sock, "\r\n", 2);
		g_free (current_position->data);
		current_position->data = NULL;
	}
	err = socket_write (sock, "\r\n", 2);
	g_list_free (header_list);

	outbuf = g_strdup_printf(premsg, msg, err, msg, PACKAGE, VERSION);
	socket_write(sock, outbuf, strlen(outbuf));
	g_free(outbuf);

	socket_uncork (sock);

	return 0;
}

void make_daemon(void)
{
	if (fork() != 0)
		exit(0);

	setsid();
	signal(SIGHUP, SIG_IGN);

	if (fork() != 0)
		exit(0);

	chdir("/");
	umask(0);

	close(0);
	close(1);
	close(2);
}

/* Called when a new client connection occurs, returns TRUE if the 
 * connection should be allowd
 */ 
int eazel_check_connection (int fd)
{
#ifdef SIMPLE_PROXY
	return TRUE;
#else /* SIMPLE_PROXY */
	struct sockaddr_in peer_name;
	struct sockaddr_in our_name;
	int struct_length;
	int err;
	int result;
	gboolean success;
	uid_t uid;

	struct_length = sizeof (peer_name);
	err = getpeername (fd, (struct sockaddr *) &peer_name, &struct_length);

	if (0 != err) {
		log ("can't get peer name: %s", g_strerror(errno));
		result = 0;
		goto error;
	}

	struct_length = sizeof (our_name);
	err = getsockname (fd, (struct sockaddr *) &our_name, &struct_length);
	if (0 != err) {
		log ("can't get local name: %s", g_strerror(errno));
		result = 0;
		goto error;
	}

	if (AF_INET != peer_name.sin_family) {
		log ("wrong socket type");
		result = 0;
		goto error;
	}

	if (inet_addr ("127.0.0.1") != peer_name.sin_addr.s_addr) {
		log ("connection from non-loopback");
		result = 0;
		goto error;
	}

	success = uid_from_local_sockaddr_in (&peer_name, &our_name, &uid);

	if (!success) {
		log ("Couldn't get socket peer UID");
		result = 0;
		goto error;
	}

	if (uid != getuid()) {
		log ("UID of peer process is %d, uid of proxy is %d; disallowed", uid, getuid());
		result = 0;
	} else {
		result = 1;
	}

error:	
	return result;
#endif /* SIMPLE_PROXY */
}

char *
to_hex_string (guchar *binary, size_t length) 
{
	char *ret;
	size_t i;

	ret = g_malloc(sizeof(char) * (2*length + 1));

	for (i = 0; i < length; i++) {
		sprintf ( ret+(2*i), "%02x", binary[i] );
	}
	ret[length*2] = '\0';

	return ret;
}

typedef struct {
	const char *chrs;
	gboolean inverted;
	gboolean primed;
	char bv[32];
} UriStrspnSet; 

UriStrspnSet uri_escape_set = {
	"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ01234567890_.- ",
	TRUE, FALSE, ""
};

#define BV_SET(bv, idx) (bv)[((guchar)(idx))>>3] |= (1 << ( (idx) & 7) )
#define BV_UNSET(bv, idx) (bv)[((guchar)(idx))>>3] &= ~(1 << ( (idx) & 7) )
#define BV_IS_SET(bv, idx) ((bv)[(idx)>>3] & (1 << ( (idx) & 7)))

static const char *
uri_strspn_to(const char *str, UriStrspnSet *set, const char *path_end)
{
	const char *cur;
	const char *cur_chr;

	if ( ! set->primed ) {
		if (set->inverted) {
			memset (set->bv, 0xff, sizeof(set->bv));
			for ( cur_chr = set->chrs; '\0' != *cur_chr; cur_chr++) {
				BV_UNSET (set->bv, *cur_chr);
			}			
		} else {
			memset (set->bv, 0, sizeof(set->bv));
		
			for ( cur_chr = set->chrs; '\0' != *cur_chr; cur_chr++) {
				BV_SET (set->bv, *cur_chr);
			}
		}
		BV_SET (set->bv, '\0');
		set->primed = TRUE;
	}
	
	for ( cur = str; (path_end && cur < path_end) && ! BV_IS_SET (set->bv, *cur); cur++ ) ;

	if ( (path_end && cur >= path_end) || '\0' == *cur ) {
		return NULL;
	} else {
		return cur;
	}
}

/**
 * util_url_encode
 * produce a string that is x-url-encoded
 */
char *
util_url_encode (const char *to_escape)
{	
	const char * current_src;
	char *current_dest;
	char * ret;
	size_t escape_count;
	/* ' ' -> '+' */
	/* [^a-zA-Z0-9_.-] -> %xx */

	if (NULL == to_escape) {
		return NULL;
	}

	escape_count = 0;
	current_src = to_escape;
	while ( NULL != (current_src = uri_strspn_to (current_src, &uri_escape_set, NULL))
		&& '\0' != *current_src
	) {
		escape_count++;
		current_src++;
	}

	/* an escape character turns into three characters */
	ret = g_malloc( strlen (to_escape) + 1 + 2*escape_count );

	current_src = to_escape;
	current_dest = ret;

	while ('\0' != *current_src)  {
		if ( BV_IS_SET (uri_escape_set.bv, *current_src)) {
			sprintf (current_dest, "%%%02X", (unsigned int) *current_src);
			current_dest += 3;
		} else if (' ' == *current_src) {
			*current_dest++ = '+';
		} else {
			*current_dest++ = *current_src;
		}
		current_src++;
	}

	*current_dest = '\0';
	return ret;
}




typedef struct {
	char *data;
	size_t length;
} DataLengthTuple;

GList * 
piece_response_add (GList *list_response_pieces, char *piece, size_t piece_len)
{
	DataLengthTuple *tuple;

	tuple = g_new0 (DataLengthTuple, 1);
	tuple->length = piece_len;
	tuple->data = piece;

	return g_list_prepend (list_response_pieces, tuple);
}

char *
piece_response_combine (GList *list_response_pieces, size_t *p_length)
{
	GList *last_node;
	GList *current_node;
	size_t response_length;
	char *current_offset;
	char *ret;

	/* Note that the list was prepended to, not appended to */

	/* count up response size */
	response_length = 0;
	last_node = g_list_last (list_response_pieces);
	for ( 	current_node = last_node ; 
		NULL != current_node ;
		current_node = g_list_previous (current_node)
	) {
		response_length += ((DataLengthTuple*)current_node->data)->length;
	}

	if (0 == response_length) {
		ret = g_strdup ("");
	} else {
		ret = g_malloc (response_length);
	}

	if ( NULL != p_length ) {
		*p_length = response_length;
	}

	for ( 	current_node = last_node, current_offset = ret ; 
		NULL != current_node ;
		current_offset +=  ((DataLengthTuple*)current_node->data)->length, 
			current_node = g_list_previous (current_node)
	) {
		memcpy (current_offset, ((DataLengthTuple*)current_node->data)->data, 
			((DataLengthTuple*)current_node->data)->length
		);
	}

	return ret;
}

void
piece_response_free (GList *list_response_pieces)
{
	GList *current_node;

	for ( 	current_node = list_response_pieces ; 
		NULL != current_node ;
		current_node = g_list_next (current_node)
	) {
		free (((DataLengthTuple*)current_node->data)->data);
		free ((DataLengthTuple*)current_node->data);
	}
	g_list_free (list_response_pieces);

}


/**
 *
 * g_list_remove_all_custom
 *
 * Like g_list_remove_custom, but removes *all* matching nodes
 * Note, of course, the nodes are removed after list iteration is complete.
 */
GList *
g_list_remove_all_custom (GList * list, gpointer data, GCompareFunc func)
{
	GList *current;
	GList *free_list;	/* free_list contains the list of GList *'s in
				 * original list that need to be freed
				 */

	g_assert (NULL != func);

	for (current = list, free_list = NULL;
		NULL != current;
		current = g_list_next (current)
	) {
		if (0 == func(current->data, data) ) {
			free_list = g_list_prepend (free_list, current);
		}
	}

	/* note -- current now iterates "free_list" */
	for (current = free_list; NULL != current; current = g_list_next(current)) {
		list = g_list_remove_link (list, current->data);
		g_list_free (current->data);
	}

	g_list_free (free_list);

	return list;
}

pid_t
util_fork_exec (const char *path, char *const argv[])
{
	pid_t pid;

	pid = fork ();

	if ( 0 == pid ) {
		execv (path, argv);
		exit (-1);	
	}
	
	return pid;

}

gboolean
util_validate_url (const char * url)
{
	HTTPRequestLine *req;
	gboolean success;

	req = request_new();

	success = request_parse_url (url, req);

	request_free (req);

	return success;
}

gint /* GCompareFunc */
util_glist_string_starts_with (gconstpointer a, gconstpointer b)
{
	if ( NULL != a && NULL != b) {
		return strncmp ( (const char *)a, (const char *)b, strlen((const char *)b));
	} else {
		return -1;
	}
}

gint /* GCompareFunc */
util_glist_string_starts_with_case_insensitive (gconstpointer a, gconstpointer b)
{
	if ( NULL != a && NULL != b) {
		return strncasecmp ( (const char *)a, (const char *)b, strlen((const char *)b));
	} else {
		return -1;
	}
}


/* BASE64 code ported from neon (http://www.webdav.org/neon) */
static const gchar b64_alphabet[65] = {
    "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    "abcdefghijklmnopqrstuvwxyz"
    "0123456789+/=" };

gchar *
util_base64_encode (const gchar *text)
{
    /* The tricky thing about this is doing the padding at the end,
     *      * doing the bit manipulation requires a bit of concentration only */
    gchar *buffer, *point;
    gint inlen, outlen;

    /* Use 'buffer' to store the output. Work out how big it should be...
     *      * This must be a multiple of 4 bytes */

    inlen = strlen( text );
    outlen = (inlen*4)/3;
    if( (inlen % 3) > 0 ) /* got to pad */
        outlen += 4 - (inlen % 3);

    buffer = g_malloc( outlen + 1 ); /* +1 for the \0 */

    /* now do the main stage of conversion, 3 bytes at a time,
     *      * leave the trailing bytes (if there are any) for later */

    for( point=buffer; inlen>=3; inlen-=3, text+=3 ) {
        *(point++) = b64_alphabet[ (*text)>>2 ];
        *(point++) = b64_alphabet[ ((*text)<<4 & 0x30) | (*(text+1))>>4 ];
        *(point++) = b64_alphabet[ ((*(text+1))<<2 & 0x3c) | (*(text+2))>>6 ];
        *(point++) = b64_alphabet[ (*(text+2)) & 0x3f ];
    }

    /* Now deal with the trailing bytes */
    if( inlen ) {
        /* We always have one trailing byte */
        *(point++) = b64_alphabet[ (*text)>>2 ];
        *(point++) = b64_alphabet[ ( ((*text)<<4 & 0x30) |
                                     (inlen==2?(*(text+1))>>4:0) ) ];
        *(point++) = (inlen==1?'=':b64_alphabet[ (*(text+1))<<2 & 0x3c ] );
        *(point++) = '=';
    }

    /* Null-terminate */
    *point = '\0';

    return buffer;
}
