/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2001 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contributor: Ben Collins <bcollins@debian.org>
*/

#include "config.h"

#include <parted/parted.h>
#include <parted/endian.h>
#include <parted/disk_sun.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <unistd.h>
#include <string.h>


#define UFS_MAGIC	0x00011954
#define UFS_MAGIC_LFN	0x00095014
#define UFS_MAGIC_FEA	0x00195612
#define UFS_MAGIC_4GB	0x05231994

struct ufs_super_block {
	int8_t		space1[1372];
	int32_t		fs_magic;
	int8_t		space2[159];
};

static int ufs_probe_hp (const PedGeometry* geom);
static int ufs_probe_sun (const PedGeometry* geom);
static int ufs_clobber (PedGeometry* geom);
static int ufs_set_system (const PedFileSystem* fs, PedPartition* part,
			   const PedDiskType* disk_type);

static PedFileSystemOps ufs_ops_sun = {
	probe:		ufs_probe_sun,
	clobber:	ufs_clobber,
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_resize_constraint:	NULL,
	set_system:	ufs_set_system
};

static PedFileSystemOps ufs_ops_hp = {
	probe:		ufs_probe_hp,
	clobber:	ufs_clobber,
	open:		NULL,
	create:		NULL,
	close:		NULL,
	check:		NULL,
	copy:		NULL,
	resize:		NULL,
	get_resize_constraint:	NULL,
	set_system:	ufs_set_system
};

static PedFileSystemType ufs_type_sun = {
	next:	NULL,
	ops:	&ufs_ops_sun,
	name:	"sun-ufs"
};

static PedFileSystemType ufs_type_hp = {
	next:   NULL,
	ops:    &ufs_ops_hp,
	name:   "hp-ufs"
};

void
ped_file_system_ufs_init ()
{
	ped_file_system_type_register (&ufs_type_sun);
	ped_file_system_type_register (&ufs_type_hp);
}

void
ped_file_system_ufs_done ()
{
	ped_file_system_type_unregister (&ufs_type_hp);
	ped_file_system_type_unregister (&ufs_type_sun);
}

static int
ufs_probe_sun (const PedGeometry* geom)
{
	int8_t buf[1536];
	struct ufs_super_block *sb;

	if (geom->length < 5)
		return 0;
	if (!ped_geometry_read (geom, buf, 16, 3))
		return 0;

	sb = (struct ufs_super_block *)buf;

	if (PED_BE32_TO_CPU(sb->fs_magic) == UFS_MAGIC
	    || PED_LE32_TO_CPU(sb->fs_magic) == UFS_MAGIC)
		return 1;

	return 0;
}

static int
ufs_probe_hp (const PedGeometry* geom)
{
	int8_t buf[1536];
	struct ufs_super_block *sb;

	if (geom->length < 5)
		return 0;
	if (!ped_geometry_read (geom, buf, 16, 3))
		return 0;

	sb = (struct ufs_super_block *)buf;

	/* Try sane bytesex */
	switch (PED_BE32_TO_CPU(sb->fs_magic)) {
		case UFS_MAGIC_LFN:
		case UFS_MAGIC_FEA:
		case UFS_MAGIC_4GB:
			return 1;
	}
#if 0 /* Really need the _to_cpup function for this */
	/* Try perverted bytesex */
	switch (PED_LE32_TO_CPU(sb->fs_magic)) {
		case UFS_MAGIC_LFN:
		case UFS_MAGIC_FEA:
		case UFS_MAGIC_4GB:
			return 1;
	}
#endif
	/* Ok, give up... */
	return 0;
}

static int
ufs_clobber (PedGeometry* geom)
{
	char	buf[1536];

	if (!ped_geometry_read (geom, buf, 16, 3))
		return 0;

	memset (buf, 0, sizeof(struct ufs_super_block));

	return ped_geometry_write (geom, buf, 16, 3);
}

static int
ufs_set_system (const PedFileSystem* fs, PedPartition* part,
		const PedDiskType* disk_type)
{
	/* Right now we have a lot of questions to answer before doing
	   this. For one, a "UFS" filesystem, even on sparc, does not have
	   a single ID. Solaris abuses partition ID's and has seperate
	   ones for root, usr, stand, home, etc..instead of being smart
	   like Mac labels and using string ID's along with numeric fs
	   type ID's. Not only that but what ID's we use depend a lot on
	   the OS, not just the partition type. For now, we just ignore
	   this. We only want to "identify" these partitions, not really
	   mark them.
	*/

	if (strcmp (disk_type->name, SUN_NAME) == 0) {
		SunPartitionData* sun_data = part->disk_specific;

		if (sun_data->is_lvm) {
			sun_data->type = 0x8e;
			return 1;
		}

		if (sun_data->is_root) {
			sun_data->type = 0x2;
			return 1;
		}

		sun_data->type = 0x6;
		return 1;
	}

	return 0;
}

