/*
3dfx Voodoo Banshee driver 


 
*/


#include <stdlib.h>
#include <stdio.h>		/* for printf */
#include <string.h>		/* for memset */
#include <unistd.h>
#include "vga.h"
#include "libvga.h"
#include "driver.h"


/* New style driver interface. */
#include "timing.h"
#include "vgaregs.h"
#include "interface.h"
#include "accel.h"
#include "vgapci.h"

#define BANSHEEREG_SAVE(i) (VGA_TOTAL_REGS+i)
#define BANSHEE_TOTAL_REGS (VGA_TOTAL_REGS + 2 + 32)


static int banshee_init(int, int, int);
static void banshee_unlock(void);
static void banshee_lock(void);

void __svgalib_bansheeaccel_init(AccelSpecs * accelspecs, int bpp, int width_in_pixels);

static int banshee_memory,banshee_chiptype;
static int banshee_is_linear, banshee_linear_base, banshee_io_base;

static CardSpecs *cardspecs;

static void banshee_setpage(int page)
{
   page<<=1;
   outl(banshee_io_base+0x2c,(inl(banshee_io_base+0x2c)&0xfff00000)|(page)|(page<<10));
}

static int __svgalib_banshee_inlinearmode(void)
{
return banshee_is_linear;
}

/* Fill in chipset specific mode information */

static void banshee_getmodeinfo(int mode, vga_modeinfo *modeinfo)
{

    if(modeinfo->colors==16)return;

    modeinfo->maxpixels = banshee_memory*1024/modeinfo->bytesperpixel;
    modeinfo->maxlogicalwidth = 4088;
    modeinfo->startaddressrange = banshee_memory * 1024 - 1;
    modeinfo->haveblit = 0;
    modeinfo->flags &= ~HAVE_RWPAGE;

    if (modeinfo->bytesperpixel >= 1) {
	if(banshee_linear_base)modeinfo->flags |= CAPABLE_LINEAR;
        if (__svgalib_banshee_inlinearmode())
	    modeinfo->flags |= IS_LINEAR;
    }
}

/* Read and save chipset-specific registers */

typedef struct {
   unsigned int pllCtrl0, pllCtrl1, dacMode, dacAddr,
      		vidProcCfg, vidScreenSize, vgaInit0,
                vgaInit1, vidDesktopStartAddr,vidDesktopOverlayStride;
} *HWRecPtr;

static int banshee_saveregs(unsigned char regs[])
{ 
  HWRecPtr save;

  banshee_unlock();		/* May be locked again by other programs (e.g. X) */
  
  save=(HWRecPtr)(regs+62);
  
  regs[BANSHEEREG_SAVE(0)]=__svgalib_inCR(0x1a);
  regs[BANSHEEREG_SAVE(1)]=__svgalib_inCR(0x1b);
  save->pllCtrl0=inl(banshee_io_base+0x40);
  save->pllCtrl1=inl(banshee_io_base+0x44);
  save->dacMode=inl(banshee_io_base+0x4c);
  save->dacAddr=inl(banshee_io_base+0x50);
  save->vidProcCfg=inl(banshee_io_base+0x5c);
  save->vidScreenSize=inl(banshee_io_base+0x98);
  save->vgaInit0=inl(banshee_io_base+0x28);
  save->vgaInit1=inl(banshee_io_base+0x2c);
  save->vidDesktopStartAddr=inl(banshee_io_base+0xe4);
  save->vidDesktopOverlayStride=inl(banshee_io_base+0xe8);
  
  return BANSHEE_TOTAL_REGS - VGA_TOTAL_REGS;
}

/* Set chipset-specific registers */

static void banshee_setregs(const unsigned char regs[], int mode)
{  
    HWRecPtr restore;

    banshee_unlock();		/* May be locked again by other programs (eg. X) */
  
    restore=(HWRecPtr)(regs+62);
  
    __svgalib_outCR(0x1a,regs[BANSHEEREG_SAVE(0)]);
    __svgalib_outCR(0x1b,regs[BANSHEEREG_SAVE(1)]);
    outl(banshee_io_base+0x40,restore->pllCtrl0);
    outl(banshee_io_base+0x44,restore->pllCtrl1);
    outl(banshee_io_base+0x4c,restore->dacMode);
    outl(banshee_io_base+0x50,restore->dacAddr);
    outl(banshee_io_base+0x5c,restore->vidProcCfg);
    outl(banshee_io_base+0x98,restore->vidScreenSize);
    outl(banshee_io_base+0x28,restore->vgaInit0);
    outl(banshee_io_base+0x2c,restore->vgaInit1);
    outl(banshee_io_base+0xe4,restore->vidDesktopStartAddr);
    outl(banshee_io_base+0xe8,restore->vidDesktopOverlayStride);
    outl(banshee_io_base+0x5c,restore->vidProcCfg&0xfffffffe);
    outl(banshee_io_base+0x5c,restore->vidProcCfg|1);
    outl(banshee_io_base+0x5c,restore->vidProcCfg);

}


/* Return nonzero if mode is available */

static int banshee_modeavailable(int mode)
{
    struct info *info;
    ModeTiming *modetiming;
    ModeInfo *modeinfo;

    modeinfo = __svgalib_createModeInfoStructureForSvgalibMode(mode);

    if ((mode < G640x480x256 )
	|| mode == G720x348x2)
	return __svgalib_vga_driverspecs.modeavailable(mode);

    info = &__svgalib_infotable[mode];
    if (banshee_memory * 1024 < info->ydim * info->xbytes)
	return 0;

    modeinfo = __svgalib_createModeInfoStructureForSvgalibMode(mode);

    modetiming = malloc(sizeof(ModeTiming));
    if (__svgalib_getmodetiming(modetiming, modeinfo, cardspecs)) {
	free(modetiming);
	free(modeinfo);
	return 0;
    }
    free(modetiming);
    free(modeinfo);

    return SVGADRV;
}

static unsigned comp_lmn(unsigned clock) ;
/* Set a mode */

/* Local, called by banshee_setmode(). */

static void banshee_initializemode(unsigned char *moderegs,
			    ModeTiming * modetiming, ModeInfo * modeinfo, int mode)
{ /* long k; */
/*    int tmp, tmptot, tmpss, tmpse, tmpbs, tmpbe, k;
    int offset; */
    HWRecPtr banshee_regs;

    banshee_regs=(HWRecPtr)(moderegs+62);
   
    banshee_saveregs(moderegs);

    __svgalib_setup_VGA_registers(moderegs, modetiming, modeinfo);
/*
    offset = modeinfo->lineWidth >> 3;
    moderegs[0x13] = offset&0xff;
    moderegs[0x14]=0x40;
    moderegs[0x17]=0xA3;

    tmp    = (modetiming->CrtcHDisplay >> 3) - 1;
    tmptot = (modetiming->CrtcHTotal >> 3) - 5;
    tmpss  = (modetiming->CrtcHSyncStart >> 3);
    tmpse  = (modetiming->CrtcHSyncEnd >> 3);
    tmpbs  = (modetiming->CrtcHSyncStart >> 3) - 1;
    tmpbe  = (modetiming->CrtcHSyncEnd >> 3);
    moderegs[MXREG_SAVE(1)]=((tmptot & 0x100)>>8) | 
       			    ((tmp & 0x100)>>7) |
                            ((tmpss & 0x100)>>6) |
                            ((tmpbs & 0x100)>>5) |
                            ((tmpse & 0x20)>>1) |
                            ((tmpbe & 0x40));
    tmp    = modetiming->CrtcVDisplay - 1;
    tmptot = modetiming->CrtcVTotal - 2;
    tmpss  = modetiming->CrtcVSyncStart;
    tmpse  = modetiming->CrtcVSyncEnd;
    tmpbs  = modetiming->CrtcVSyncStart - 1;
    tmpbe  = offset; 
    moderegs[MXREG_SAVE(2)]=((tmptot & 0x400)>>10) | 
		            ((tmp & 0x400)>>9) |
		            ((tmpss & 0x400)>>8) |
		            ((tmpbs & 0x400)>>7) |
		            ((tmpse & 0x10)<<1) |
		            ((tmpbe & 0x300)>>2);
    if (modetiming->flags & INTERLACED)
	    moderegs[MXREG_SAVE(3)] |= 0x8;
*/
    banshee_regs->vidProcCfg&=0xf7e30000;
    banshee_regs->vidProcCfg|=0x00000c81;

    banshee_regs->vidScreenSize=modeinfo->width|(modeinfo->height<<12);

    if (modetiming->flags & DOUBLESCAN)
	    banshee_regs->vidProcCfg |= 0x10;
    switch (modeinfo->bitsPerPixel)
    {
	    case 8: 
               	    banshee_regs->vidProcCfg|=0<<18;
		    break;
	    case 15: 
	    case 16:if(modeinfo->greenWeight==5){
                        banshee_regs->vidProcCfg|=1<<18;
                    } else banshee_regs->vidProcCfg|=1<<18;
		    break;
	    case 24: 
               	    banshee_regs->vidProcCfg|=2<<18;
		    break;
	    case 32: 
               	    banshee_regs->vidProcCfg|=3<<18;
		    break;
	    default: 
		    break;
    }
    
    banshee_regs->vgaInit0&=0xfffffffb;
    if(modeinfo->bitsPerPixel!=8){
        banshee_regs->vgaInit0|=4;
    };

    banshee_regs->pllCtrl0=comp_lmn(modetiming->pixelClock);
    moderegs[59]|=0x0c;

    banshee_regs->vidDesktopStartAddr=0;
    banshee_regs->vidDesktopOverlayStride=modeinfo->lineWidth;    

    banshee_is_linear=0;

return ;

}

static int banshee_setmode(int mode, int prv_mode)
{
    unsigned char *moderegs;
    ModeTiming *modetiming;
    ModeInfo *modeinfo;

    if ((mode < G640x480x256 /*&& mode != G320x200x256*/)
	|| mode == G720x348x2) {

	return __svgalib_vga_driverspecs.setmode(mode, prv_mode);
    }
    if (!banshee_modeavailable(mode))
	return 1;

    modeinfo = __svgalib_createModeInfoStructureForSvgalibMode(mode);

    modetiming = malloc(sizeof(ModeTiming));
    if (__svgalib_getmodetiming(modetiming, modeinfo, cardspecs)) {
	free(modetiming);
	free(modeinfo);
	return 1;
    }

    moderegs = malloc(BANSHEE_TOTAL_REGS);

    banshee_initializemode(moderegs, modetiming, modeinfo, mode);
    free(modetiming);

    __svgalib_setregs(moderegs);	/* Set standard regs. */
    banshee_setregs(moderegs, mode);		/* Set extended regs. */
    free(moderegs);

    __svgalib_InitializeAcceleratorInterface(modeinfo);

    free(modeinfo);
    return 0;
}


/* Unlock chipset-specific registers */

static void banshee_unlock(void)
{
    int vgaIOBase, temp;

    vgaIOBase = (inb(0x3CC) & 0x01) ? 0x3D0 : 0x3B0;
    outb(vgaIOBase + 4, 0x11);
    temp = inb(vgaIOBase + 5);
    outb(vgaIOBase + 5, temp & 0x7F);
    
    outl(banshee_io_base+0x28,(inl(banshee_io_base+0x28)&0xffffffbf)|(1<<6));
}

static void banshee_lock(void)
{
    int vgaIOBase, temp;

    vgaIOBase = (inb(0x3CC) & 0x01) ? 0x3D0 : 0x3B0;
    outb(vgaIOBase + 4, 0x11);
    temp = inb(vgaIOBase + 5);
    outb(vgaIOBase + 5, temp & 0x7F);
    
    outl(banshee_io_base+0x28,(inl(banshee_io_base+0x28)&0xffffffbf));
}


/* Indentify chipset, initialize and return non-zero if detected */

static int banshee_test(void)
{
   int found,_ioperm=0;
   unsigned long buf[64];

   if (getenv("IOPERM") == NULL) {
      _ioperm=1;
      if (iopl(3) < 0) {
        printf("svgalib: banshee: cannot get I/O permissions\n");
        exit(1);
      }
   }
   found=(!__svgalib_pci_find_vendor_vga(0x121a,buf))&&((buf[0]>>16)==0x0003);
    
   if (_ioperm) iopl(0);

   if(found)banshee_init(0,0,0); 
   return found;
}


/* Set display start address (not for 16 color modes) */
/* Cirrus supports any address in video memory (up to 2Mb) */

static void banshee_setdisplaystart(int address)
{ 
  address=address >> 2;
  outw(0x3d4, (address & 0x00FF00) | 0x0C);
  outw(0x3d4, ((address & 0x00FF) << 8) | 0x0D);
  outl(banshee_io_base+0xe4,address>>2);

}


/* Set logical scanline length (usually multiple of 8) */

static void banshee_setlogicalwidth(int width)
{   
    int offset = width >> 3;
 
    __svgalib_outCR(0x13,offset&0xff);
    outl(banshee_io_base+0xe8,width);
}

static int banshee_linear(int op, int param)
{
if (op==LINEAR_ENABLE || op==LINEAR_DISABLE){ banshee_is_linear=1-banshee_is_linear; return 0;}
if (op==LINEAR_QUERY_BASE) return banshee_linear_base;
if (op == LINEAR_QUERY_RANGE || op == LINEAR_QUERY_GRANULARITY) return 0;		/* No granularity or range. */
    else return -1;		/* Unknown function. */
}

static int banshee_match_programmable_clock(int clock)
{
return clock ;
}

static int banshee_map_clock(int bpp, int clock)
{
return clock ;
}

static int banshee_map_horizontal_crtc(int bpp, int pixelclock, int htiming)
{
return htiming;
}

/* Function table (exported) */

DriverSpecs __svgalib_banshee_driverspecs =
{
    banshee_saveregs,
    banshee_setregs,
    banshee_unlock,
    banshee_lock,
    banshee_test,
    banshee_init,
    banshee_setpage,
    NULL,
    NULL,
    banshee_setmode,
    banshee_modeavailable,
    banshee_setdisplaystart,
    banshee_setlogicalwidth,
    banshee_getmodeinfo,
    0,				/* old blit funcs */
    0,
    0,
    0,
    0,
    0,				/* ext_set */
    0,				/* accel */
    banshee_linear,
    0,				/* accelspecs, filled in during init. */
    NULL,                       /* Emulation */
};

/* Initialize chipset (called after detection) */

static int banshee_init(int force, int par1, int par2)
{
   unsigned long buf[64];
   int found=0;
   int _ioperm=0;

    if (force) {
	banshee_memory = par1;
        banshee_chiptype = par2;
    } else {

    };
    if(banshee_memory==0)banshee_memory=16*1024;

    if (getenv("IOPERM") == NULL) {
      _ioperm=1;
      if (iopl(3) < 0) {
        printf("svgalib: mx: cannot get I/O permissions\n");
        exit(1);
      }
     }
     found=(!__svgalib_pci_find_vendor_vga(0x121a,buf))&&((buf[0]>>16)==0x0003);
    
/*     if (_ioperm) iopl(0);*/
     if (found){
       banshee_linear_base=buf[5]&0xffffff00;
       banshee_io_base=buf[6]&0xff00;
     };

     banshee_unlock();
     if (__svgalib_driver_report) {
	printf("Using Banshee driver, %iKB. ",banshee_memory);
     }
    
    cardspecs = malloc(sizeof(CardSpecs));
    cardspecs->videoMemory = banshee_memory;
    cardspecs->maxPixelClock4bpp = 250000;	
    cardspecs->maxPixelClock8bpp = 250000;	
    cardspecs->maxPixelClock16bpp = 250000;	
    cardspecs->maxPixelClock24bpp = 250000;
    cardspecs->maxPixelClock32bpp = 250000;
    cardspecs->flags = INTERLACE_DIVIDE_VERT | CLOCK_PROGRAMMABLE;
    cardspecs->maxHorizontalCrtc = 2040;
    cardspecs->maxPixelClock4bpp = 0;
    cardspecs->nClocks =0;
    cardspecs->mapClock = banshee_map_clock;
    cardspecs->mapHorizontalCrtc = banshee_map_horizontal_crtc;
    cardspecs->matchProgrammableClock=banshee_match_programmable_clock;
    __svgalib_driverspecs = &__svgalib_banshee_driverspecs;
    __svgalib_banked_mem_base=0xa0000;
    __svgalib_banked_mem_size=0x10000;
    __svgalib_linear_mem_base=banshee_linear_base;
    __svgalib_linear_mem_size=banshee_memory*0x400;
    return 0;
}

#define WITHIN(v,c1,c2) (((v) >= (c1)) && ((v) <= (c2)))

static unsigned
comp_lmn(unsigned clock)
{
  int     n, m, l;
  double  fvco;
  double  fout;
  double  fmax;
  double  fref;
  double  fvco_goal;

    fmax = 162000.0;

  fref = 14318.18; 

  for (m = 1; m <= 61; m++)
  {
    for (l = 3; l >= 0; l--)
    {
      for (n = 1; n <= 253; n++)
      {
        fout = ((double)(n + 2) * fref)/((double)(m + 2) * (1 << l));
        fvco_goal = (double)clock * (double)(1 << l);
        fvco = fout * (double)(1 << l);
        if (!WITHIN(fvco, 0.995*fvco_goal, 1.005*fvco_goal))
          continue;
#if 0
        if (!WITHIN(fvco, 32000.0, fmax))
          continue;
#endif
#if 0
printf("clock=%i n=%i m=%i l=%i",clock,n,m,l);
#endif
        return (n << 8) | (m<<2) | l;
      }
    }
  }
printf("BANSHEE driver: Can't do clock=%i\n",clock);
  return 0;
}

