/* Copyright (c) 1992, 2000 John E. Davis
 * This file is part of JED editor library source.
 *
 * You may distribute this file under the terms the GNU General Public
 * License.  See the file COPYING for more information.
 */
#include "config.h"
#include "jed-feat.h"

/*{{{ #include files */

#include <stdio.h>

#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#include <string.h>
#include <slang.h>

#include "hooks.h"
#include "misc.h"

typedef struct Hook_Type
{
   SLang_Name_Type *nt;
   struct Hook_Type *next;
}
Hook_Type;

typedef struct Hook_List_Type
{
   char *hooks_name;		       /* slstring */
   Hook_Type *hooks;
   struct Hook_List_Type *next;
}
Hook_List_Type;

static Hook_List_Type *Hook_List;

static Hook_Type *hook_already_exists (Hook_List_Type *h, 
				       SLang_Name_Type *nt, 
				       Hook_Type **p)
{
   Hook_Type *next, *last;
   
   last = NULL;
   next = h->hooks;
   
   while (next != NULL)
     {
	if (next->nt == nt)
	  break;
	
	last = next;
	next = next->next;
     }
   
   if (p != NULL) *p = last;
   return next;
}

static void free_hook (Hook_Type *h)
{
   if (h == NULL)
     return;
   
   SLfree ((char *) h);
}


static Hook_List_Type *new_hook_list (char *name)
{
   Hook_List_Type *h;

   h = (Hook_List_Type *) jed_malloc0 (sizeof (Hook_List_Type));
   if (h == NULL)
     return NULL;

   if (NULL == (h->hooks_name = SLang_create_slstring (name)))
     {
	SLfree ((char *) h);
	return NULL;
     }
   
   h->next = Hook_List;
   Hook_List = h;
   return h;
}

static Hook_List_Type *find_hook_list (char *name)
{
   Hook_List_Type *l;
   char ch;

   l = Hook_List;
   ch = name[0];
   while (l != NULL)
     {
	if ((ch == l->hooks_name[0])
	    && (0 == strcmp (l->hooks_name, name)))
	  break;
	
	l = l->next;
     }
   return l;
}

	  
static int insert_hook_1 (char *hook, SLang_Name_Type *nt, int append)
{
   Hook_List_Type *l;
   Hook_Type *h, *p;

   if (NULL == (l = find_hook_list (hook)))
     {
	if (NULL == (l = new_hook_list (hook)))
	  return -1;
     }

   if (NULL != hook_already_exists (l, nt, &p))
     return 0;

	
   if (NULL == (h = (Hook_Type *) jed_malloc0 (sizeof (Hook_Type))))
     return -1;
	
   h->nt = nt;
   if (append && (p != NULL))
     p->next = h;
   else
     {
	h->next = l->hooks;
	l->hooks = h;
     }
   return 0;
}

static int pop_hooks_info (char **s, SLang_Name_Type **nt)
{
   if (NULL == (*nt = SLang_pop_function ()))
     return -1;
   
   if (-1 == SLang_pop_slstring (s))
     return -1;
   
   return 0;
}

static int insert_hook (int append)
{
   SLang_Name_Type *nt;
   char *name;
   int status;

   if (-1 == pop_hooks_info (&name, &nt))
     return -1;
   
   status = insert_hook_1 (name, nt, append);
   
   SLang_free_slstring (name);
   return status;
}

static void add_hook_cmd (void)
{
   (void) insert_hook (0);
}
static void append_hook_cmd (void)
{
   (void) insert_hook (1);
}
static void remove_hook_cmd (void)
{
   char *name;
   SLang_Name_Type *nt;
   Hook_Type *h, *p;
   Hook_List_Type *l;

   if (-1 == pop_hooks_info (&name, &nt))
     return;
   
   if ((NULL == (l = find_hook_list (name)))
       || (NULL == (h = hook_already_exists (l, nt, &p))))
     {
	SLang_free_slstring (name);
	return;
     }
   if (p != NULL)
     p->next = h->next;
   else
     l->hooks = h->next;

   free_hook (h);
   SLang_free_slstring (name);
}

static int execute_fun_with_args (SLang_Name_Type *nt, unsigned int argc,
				  char **argv)
{
   unsigned int i;

   (void) SLang_start_arg_list ();
   for (i = 0; i < argc; i++)
     (void) SLang_push_string (argv[i]);
   (void) SLang_end_arg_list ();

   return SLexecute_function (nt);
}

static int jed_run_hooks (char *name, int method, unsigned int argc, char **argv)
{
   Hook_List_Type *l;
   Hook_Type *h;

   if (NULL == (l = find_hook_list (name)))
     h = NULL;
   else
     h = l->hooks;

   while (h != NULL)
     {
	int status;
	
	status = execute_fun_with_args (h->nt, argc, argv);
	h = h->next;

	if (status == -1)
	  return -1;
	
	if (method == JED_HOOKS_RUN_ALL)
	  continue;
	
	if (-1 == SLang_pop_integer (&status))
	  return -1;

	if (method == JED_HOOKS_RUN_UNTIL_0)
	  {
	     if (status == 0)
	       return 0;
	     continue;
	  }
	
	/* else method = JED_HOOKS_RUN_UNTIL_NON_0 */
	if (status)
	  return 1;
     }

   if (method == JED_HOOKS_RUN_UNTIL_0)
     return 1;

   return 0;
}


static char **build_arg_list (unsigned int n, va_list ap)
{
   char **s;
   unsigned int i;

   s = (char **)SLmalloc ((n+1) * sizeof (char *));
   if (s == NULL)
     return NULL;
   
   for (i = 0; i < n; i++)
     s[i] = va_arg(ap, char *);
   s[n] = NULL;
   return s;
}


int jed_va_run_hooks (char *name, int method, unsigned int nargs, ...)
{
   va_list ap;
   char **arglist;
   int status;

   va_start (ap, nargs);
   if (NULL == (arglist = build_arg_list (nargs, ap)))
     {
	va_end (ap);
	return -1;
     }
   va_end (ap);

   status = jed_run_hooks (name, method, nargs, arglist);
   SLfree ((char *) arglist);
   return status;
}

static SLang_Intrin_Fun_Type Hooks_Intrinsics [] =
{
   MAKE_INTRINSIC_0("add_to_hook", add_hook_cmd, SLANG_VOID_TYPE),
   MAKE_INTRINSIC_0("append_to_hook", append_hook_cmd, SLANG_VOID_TYPE),
   MAKE_INTRINSIC_0("remove_from_hook", remove_hook_cmd, SLANG_VOID_TYPE),
   MAKE_INTRINSIC_0(NULL, NULL, 0)
};

int jed_init_user_hooks (void)
{
   return SLadd_intrin_fun_table (Hooks_Intrinsics, NULL);
}
