/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "tlib.h"
#include "clip.h"
#include "xforms.h"

//#define USE_ASM

// local variables
const float ONE = 1.0f;
static Matrix gMat;
static int gNumXformedVerts;
static unsigned short g_fpu_cw_old, g_fpu_cw_new;
static VertexData *gPrevVert;

// external variables
extern float gMinClipX, gMinClipY, gMinClipZ, gMaxClipX, gMaxClipY, gMaxClipZ;
extern float gMinClipZInv, gMaxClipZInv;
extern int sign_mask_table[8];

void SetCurrMatrix(Matrix m)
{
	MatrixCopy(gMat, m);
}

void BeginXforms()
{
	gPrevVert = NULL;
	gNumXformedVerts = 0;

	// change the FPU control word to 24-bit precision
	// bits 8 and 9 determine precision:
	//   00 - 24-bits
	//   10 - 53-bits
	//   11 - 64-bits
	/*
	__asm
	{
		fnstcw	word ptr [g_fpu_cw_old]
		mov			ax, [g_fpu_cw_old]
		and			ax, 0xfcff // 24-bit precision
		mov			[g_fpu_cw_new], ax
		fldcw		word ptr [g_fpu_cw_new]
	} */
}

int EndXforms()
{
	if (gPrevVert)
	{
		gPrevVert->outcode = ComputeOutcode(&gPrevVert->vertex);
	}

	// restore the FPU control word
	/*
	__asm
	{
		fldcw		word ptr [g_fpu_cw_old]
	}
	*/
	return gNumXformedVerts;
}

// this function will transform the vector v by the matrix m
// it'll then compute 1/w and homogenize (i.e. compute x/w and y/w)
// it'll also calculate the clipping outcodes (in integer while wating for the divide)
// the vert's x, y, z, oow, and outcode fields get set accordingly (z must be 1/oow or w, and is used when clipping)
// it assumes that v[3] is 1.0f pre-transform
#ifdef USE_ASM
__declspec(naked) void XformVertex(VertexData *vert, Vector *v)
{
	/*
	__asm // 82 cycles
	{
		// ******** MATRIX MULTIPLY ********
		mov			edx, [esp + 8] // edx = v
		push		esi

		fld			dword ptr [gMat]
		// m00
		fmul		dword ptr [edx]
		// m00*v0 (2)
		fld			dword ptr [gMat + 4]
		// m01
		// m00*v0 (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		fxch		st(1)
		// m00*v0 (0)
		// m01*v1 (2)
		fadd		dword ptr [gMat + 12]
		// m00*v0 + m03 (2)
		// m01*v1 (1)
		fld			dword ptr [gMat + 8]
		// m02
		// m00*v0 + m03 (1)
		// m01*v1 (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m00*v0 + m03 (0)
		// m01*v1 (0)
		fxch		st(2)
		// m01*v1 (0)
		// m00*v0 + m03 (0)
		// m02*v2 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m03 (2)
		// m02*v2 (1)
		fld			dword ptr [gMat + 16]
		// m10
		// m00*v0 + m01*v1 + m03 (1)
		// m02*v2 (0)
		fmul		dword ptr [edx]
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m03 (0)
		// m02*v2 (0)
		fxch		st(2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 + m03 (0)
		// m10*v0 (2)
		faddp		st(1), st
		// m00*v0 + m01*v1 + m02*v2 + m03 (2)
		// m10*v0 (1)
		fld			dword ptr [gMat + 16 + 4]
		// m11
		// m00*v0 + m01*v1 + m02*v2 + m03 (1)
		// m10*v0 (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m10*v0 (0)
		fxch		st(2)
		// m10*v0 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (2)
		fadd		dword ptr [gMat + 16 + 12]
		// m10*v0 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (1)
		fld			dword ptr [gMat + 16 + 8]
		// m12
		// m10*v0 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m11*v1 (0)
		fxch		st(3)
		// m11*v1 (0)
		// m10*v0 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (1)
		fld			dword ptr [gMat + 48]
		// m30
		// m10*v0 + m11*v1 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fmul		dword ptr [edx]
		// m30*v0 (2)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m12*v2 (0)
		fxch		st(3)
		// m12*v2 (0)
		// m10*v0 + m11*v1 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (2)
		faddp		st(1), st
		// m10*v0 + m11*v1 + m12*v2 + m13 (2)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (1)
		fld			dword ptr [gMat + 48 + 4]
		// m31
		// m10*v0 + m11*v1 + m12*v2 + m13 (1)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (0)
		fmul		dword ptr [edx + 4]
		// m31*v1 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m30*v0 (0)
		fxch		st(3)
		// m30*v0 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (2)
		fadd		dword ptr [gMat + 48 + 12]
		// m30*v0 + m33 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (1)
		fld			dword ptr [gMat + 48 + 8]
		// m32
		// m30*v0 + m33 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (0)
		fmul		dword ptr [edx + 8]
		// m32*v2 (2)
		// m30*v0 + m33 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m31*v1 (0)
		fxch		st(4)
		// m31*v1 (0)
		// m30*v0 + m33 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (2)
		faddp		st(1), st
		// m30*v0 + m31*v1 + m33 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (1)
		fld			dword ptr [gMat + 32]
		// m20
		// m30*v0 + m31*v1 + m33 (1)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (0)
		fmul		dword ptr [edx]
		// m20*v0 (2)
		// m30*v0 + m31*v1 + m33 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m32*v2 (0)
		fxch		st(4)
		// m32*v2 (0)
		// m30*v0 + m31*v1 + m33 (0)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (2)
		faddp		st(1), st
		// m30*v0 + m31*v1 + m32*v2 + m33 (2)
		// m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (1)
		fld			dword ptr [gMat + 32 + 4]
		// m21
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (1)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fmul		dword ptr [edx + 4]
		// m21*v1 (2)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m20*v0 (0)
		fxch		st(4)
		// m20*v0 (0)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (2)
		fadd		dword ptr [gMat + 32 + 12]
		// m20*v0 + m23 (2)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (1)
		fld			dword ptr [gMat + 32 + 8]
		// m22
		// m20*v0 + m23 (1)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fmul		dword ptr [edx + 8]
		// m22*v2 (2)
		// m20*v0 + m23 (0)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m21*v1 (0)
		fxch		st(5)
		// m21*v1 (0)
		// m20*v0 + m23 (0)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (2)
		faddp		st(1), st
		// m20*v0 + m21*v1 + m23 (2)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (1)
		mov			esi, [gPrevVert]
		// m20*v0 + m21*v1 + m23 (1)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (0)
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (0)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (0)
		// m22*v2 (0)
		fld			dword ptr [ONE]
		// 1
		// m20*v0 + m21*v1 + m23 (0)
		// W (0)
		// Y (0)
		// X (0)
		// m22*v2 (0)
		fxch		st(1)
		// m20*v0 + m21*v1 + m23 (0)
		// 1
		// W (0)
		// Y (0)
		// X (0)
		// m22*v2 (0)
		faddp		st(5), st
		// 1
		// W (0)
		// Y (0)
		// X (0)
		// Z = m20*v0 + m21*v1 + m22*v2 + m23  (2)

		// ******** DIVIDE AND COMPUTE OUTCODES ********
		fdiv		st, st(1)
		// 1/W (18) - 18+1 cycles in 24-bit mode
		// W (0)
		// Y (0)
		// X (0)
		// Z (0)

		// if (gPrevVert)
		test		esi, esi
		jz			SETUP_VERTEX

		// compute outcodes
		push		edi
		push		ebx

		mov			edi, [giMinClipX]
		nop  // ******** need any integer math ???? ********

		mov			eax, [esi + 4*GR_VERTEX_X_OFFSET] // eax = x
		mov			edx, [esi + 4*GR_VERTEX_OOW_OFFSET] // edi = w

		mov			ebx, [esi + 4*GR_VERTEX_Y_OFFSET] // ebx = y
		and			edx, 0x80000000

		mov			ecx, [esi + 4*GR_VERTEX_Z_OFFSET] // ecx = z
		xor			eax, edx // eax = +/- x

		sar			eax, 1
		xor			ebx, edx // ebx = +/- y

		sar			ebx, 1
		xor			ecx, edx // ecx = +/- z

		sar			ecx, 1
		mov			edx, [giMaxClipX]

		sub			edx, eax // edx = maxx - x
		sub			eax, edi // eax = x - minx

		shr			edx, 30
		mov			edi, [giMinClipY]

		shr			eax, 31
		and			dl, 0x2

		or			al, dl
		mov			edx, [giMaxClipY]

		sub			edx, ebx // edx = maxy - y
		sub			ebx, edi // ebx = y - miny

		shr			edx, 28
		mov			edi, [giMinClipZ]

		shr			ebx, 29
		and			dl, 0x8

		and			bl, 0x4
		or			al, dl

		or			al, bl
		mov			edx, [giMaxClipZ]

		sub			edx, ecx // edx = maxz - z
		sub			ecx, edi // ecx = z - minz

		shr			edx, 26
		nop  // ******** need any integer math ???? ********

		shr			ecx, 27
		and			dl, 0x20

		and			cl, 0x10
		or			al, dl

		or			al, cl
		pop			ebx

		mov			dword ptr (VertexData)[esi].outcode, eax
		pop			edi

SETUP_VERTEX:
		mov			ecx, [gNumXformedVerts]
		pop			esi
		// 1/W (0)
		// W (0)
		// Y (0)
		// X (0)
		// Z (0)
		fmul		st(3), st
		// 1/W (0)
		// W (0)
		// Y (0)
		// X/W (2)
		// Z (0)
		mov			eax, [esp + 4] // eax = vert
		inc			ecx
		// 1/W (0)
		// W (0)
		// Y (0)
		// X/W (1)
		// Z (0)
		fmul		st(2), st
		// 1/W (0)
		// W (0)
		// Y/W (2)
		// X/W (0)
		// Z (0)
		mov			[gPrevVert], eax
		mov			[gNumXformedVerts], ecx
		// 1/W (0)
		// W (0)
		// Y/W (1)
		// X/W (0)
		// Z (0)
		fmul		st(4), st
		// 1/W (0)
		// W (0)
		// Y/W (0)
		// X/W (0)
		// Z/W (2)
		fxch		st(1)
		// W (0)
		// 1/W (0)
		// Y/W (0)
		// X/W (0)
		// Z/W (2)
		fstp		dword ptr (VertexData)[eax].w
		fstp		dword ptr (GrVertex)[eax].oow
		fstp		dword ptr (GrVertex)[eax].y
		fstp		dword ptr (GrVertex)[eax].x
		fstp		dword ptr (GrVertex)[eax].z

		ret
	}
	*/
	__asm
	{
		mov			edx, [esp + 8] // edx = v
		lea			ecx, [gMat]

		push		edi
		push		esi

		fld			dword ptr [ecx + 48 + 0]
		// m30
		fmul		dword ptr [edx + 0]
		// m30*v0 (2)
		fld			dword ptr [ecx + 48 + 4]
		// m31
		// m30*v0 (1)
		fmul		dword ptr [edx + 4]
		// m31*v1 (2)
		// m30*v0 (0)
		fxch		st(1)
		// m30*v0 (0)
		// m31*v1 (2)
		fadd		dword ptr [ecx + 48 + 12]
		// m30*v0 + m33 (2)
		// m31*v1 (1)
		fld			dword ptr [ecx + 48 + 8]
		// m32
		// m30*v0 + m33 (1)
		// m31*v1 (0)
		fmul		dword ptr [edx + 8]
		// m32*v2 (2)
		// m30*v0 + m33 (0)
		// m31*v1 (0)
		fxch		st(1)
		// m30*v0 + m33 (0)
		// m32*v2 (2)
		// m31*v1 (0)
		faddp		st(2), st
		// m32*v2 (1)
		// m30*v0 + m31*v1 + m33 (2)
		fld			dword ptr [ecx + 0 + 0]
		// m00
		// m32*v2 (0)
		// m30*v0 + m31*v1 + m33 (1)
		fmul		dword ptr [edx + 0]
		// m00*v0 (2)
		// m32*v2 (0)
		// m30*v0 + m31*v1 + m33 (0)
		fxch		st(1)
		// m32*v2 (0)
		// m00*v0 (2)
		// m30*v0 + m31*v1 + m33 (0)
		faddp		st(2), st
		// m00*v0 (1)
		// W = m30*v0 + m31*v1 + m32*v2 + m33 (2)
		fld			dword ptr [ecx + 0 + 4]
		// m01
		// m00*v0 (0)
		// W (1)
		fmul		dword ptr [edx + 4]
		// m01*v1 (2)
		// m00*v0 (0)
		// W (0)
		fxch		st(1)
		// m00*v0 (0)
		// m01*v1 (2)
		// W (0)
		fadd		dword ptr [ecx + 0 + 12]
		// m00*v0 + m03 (2)
		// m01*v1 (1)
		// W (0)
		fld			dword ptr [ecx + 0 + 8]
		// m02
		// m00*v0 + m03 (1)
		// m01*v1 (0)
		// W (0)
		fmul		dword ptr [edx + 8]
		// m02*v2 (2)
		// m00*v0 + m03 (0)
		// m01*v1 (0)
		// W (0)
		fxch		st(1)
		// m00*v0 + m03 (0)
		// m02*v2 (2)
		// m01*v1 (0)
		// W (0)
		faddp		st(2), st
		// m02*v2 (1)
		// m00*v0 + m01*v1 + m03 (2)
		// W (0)
		fld			dword ptr [ecx + 16 + 0]
		// m10
		// m02*v2 (0)
		// m00*v0 + m01*v1 + m03 (1)
		// W (0)
		fmul		dword ptr [edx + 0]
		// m10*v0 (2)
		// m02*v2 (0)
		// m00*v0 + m01*v1 + m03 (0)
		// W (0)
		fxch		st(1)
		// m02*v2 (0)
		// m10*v0 (2)
		// m00*v0 + m01*v1 + m03 (0)
		// W (0)
		faddp		st(2), st
		// m10*v0 (1)
		// X = m00*v0 + m01*v1 + m02*v2 + m03 (2)
		// W (0)
		fld			dword ptr [ecx + 16 + 4]
		// m11
		// m10*v0 (0)
		// X (1)
		// W (0)
		fmul		dword ptr [edx + 4]
		// m11*v1 (2)
		// m10*v0 (0)
		// X (0)
		// W (0)
		fxch		st(1)
		// m10*v0 (0)
		// m11*v1 (2)
		// X (0)
		// W (0)
		fadd		dword ptr [ecx + 16 + 12]
		// m10*v0 + m13 (2)
		// m11*v1 (1)
		// X (0)
		// W (0)
		fld			dword ptr [ecx + 16 + 8]
		// m12
		// m10*v0 + m13 (1)
		// m11*v1 (0)
		// X (0)
		// W (0)
		fmul		dword ptr [edx + 8]
		// m12*v2 (2)
		// m10*v0 + m13 (0)
		// m11*v1 (0)
		// X (0)
		// W (0)
		fxch		st(1)
		// m10*v0 + m13 (0)
		// m12*v2 (2)
		// m11*v1 (0)
		// X (0)
		// W (0)
		faddp		st(2), st
		// m12*v2 (1)
		// m10*v0 + m11*v1 + m13 (2)
		// X (0)
		// W (0)

		push		ebx
		mov			edx, [gPrevVert]

		fld			dword ptr [ONE]
		// 1.0f
		// m12*v2 (0)
		// m10*v0 + m11*v1 + m13 (0)
		// X (0)
		// W (0)
		fxch		st(1)
		// m12*v2 (0)
		// 1.0f
		// m10*v0 + m11*v1 + m13 (0)
		// X (0)
		// W (0)
		faddp		st(2), st
		// 1.0f
		// Y = m10*v0 + m11*v1 + m12*v2 + m13 (2)
		// X (0)
		// W (0)
		fdiv		st, st(3)
		// 1/W (18)
		// Y (1)
		// X (0)
		// W (0)

		// ******** do some integer math why the float divide is computing ********
		// the divide should only take 18 clocks since we should be in 24-bit fpu mode
		test		edx, edx
		jz			SETUP_VERTEX

		xor			eax, eax // eax = outcodes
		xor			ecx, ecx // ecx = sign mask

		mov			ebx, (GrVertex)[edx].z // ebx = z
		mov			esi, [gMaxClipZ]

		mov			edi, [gMinClipZ]
		cmp			esi, ebx // maxz - z

		adc			eax, eax
		cmp			ebx, edi // z - minz

		adc			eax, eax
		add			ebx, ebx // carry if ebx = z < 0

		adc			ecx, ecx
		mov			esi, [gMaxClipY]

		mov			ebx, (GrVertex)[edx].y // ebx = y
		mov			edi, [gMinClipY]

		cmp			esi, ebx // maxy - y
		mov			esi, [gMaxClipX]

		adc			eax, eax
		cmp			ebx, edi // y - miny

		adc			eax, eax
		add			ebx, ebx // carry if ebx = y < 0

		adc			ecx, ecx
		mov			ebx, (GrVertex)[edx].x // ebx = x

		mov			edi, [gMinClipX]
		cmp			esi, ebx // maxx - x

		adc			eax, eax
		cmp			ebx, edi // x - minx

		adc			eax, eax
		add			ebx, ebx // carry if ebx = x < 0

		adc			ecx, ecx
		mov			ebx, (GrVertex)[edx].z // ebx = z

		sar			ebx, 31 // sign mask of z
		mov			ecx, [sign_mask_table + 4*ecx] // sign mask of zzyyxx

		xor			eax, ecx // outcodes if z > 0
		and			ebx, 0xf

		xor			eax, ebx

		mov			dword ptr (VertexData)[edx].outcode, eax

SETUP_VERTEX:
		mov			eax, [esp + 16] // eax = vert
		mov			ecx, [gNumXformedVerts]

		pop			ebx
		inc			ecx

		// 1/W (0)
		// Y (0)
		// X (0)
		// W (0)
		fmul		st(2), st
		// 1/W (0)
		// Y (0)
		// X/W (2)
		// W (0)

		pop			esi
		mov			[gPrevVert], eax

		fmul		st(1), st
		// 1/W (0)
		// Y/W (2)
		// X/W (0)
		// W (0)

		pop			edi
		mov			[gNumXformedVerts], ecx

		fstp		dword ptr (GrVertex)[eax].oow
		fstp		dword ptr (GrVertex)[eax].y
		fstp		dword ptr (GrVertex)[eax].x
		fstp		dword ptr (GrVertex)[eax].z

		ret
	}
}
#else
void XformVertex(VertexData *vert, Vector *v)
{
	Vector pt;

	pt[W] = gMat[3][0]*(*v)[0] + gMat[3][1]*(*v)[1] + gMat[3][2]*(*v)[2] + gMat[3][3];
	pt[X] = gMat[0][0]*(*v)[0] + gMat[0][1]*(*v)[1] + gMat[0][2]*(*v)[2] + gMat[0][3];
	pt[Y] = gMat[1][0]*(*v)[0] + gMat[1][1]*(*v)[1] + gMat[1][2]*(*v)[2] + gMat[1][3];

	vert->vertex.oow = 1.0f/pt[W];

	if (gPrevVert)
	{
		gPrevVert->outcode = ComputeOutcode(&gPrevVert->vertex);
	}

	pt[X] *= vert->vertex.oow;
	pt[Y] *= vert->vertex.oow;

	vert->vertex.x = pt[X];
	vert->vertex.y = pt[Y];
	vert->vertex.z = pt[W];

	gPrevVert = vert;

	gNumXformedVerts++;
}
#endif // USE_ASM
