/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"
#include "tlib.h"
#include "texcache.h"
#include "util.h"
#include "linux_utils.h"

// local variables
void (*gCurrUIFunction)(XEvent);
static int gNumTMUs, gNumColorBuffers;
static TextureCache *gTexCache[2];
#ifdef USE_GLIDE3
static void *gGlideState = NULL;
#else
static GrState gGlideState;
#endif // USE_GLIDE3

void SetCurrUIFunction( void (*func)(XEvent) )
{
	gCurrUIFunction = func;
}


BOOL GlideInitialize()
{
	int i;
#ifndef USE_GLIDE3
	GrHwConfiguration hwconfig;
#endif

	grGlideInit();

	gNumColorBuffers = 3;

#ifdef USE_GLIDE3
	// set the number of TMUs available
	if (grGet(GR_NUM_TMU, 4, (long *)&gNumTMUs) != 4)
	{
		grGlideShutdown();
		return FALSE;
	}

	// HACK: Rush doesn't allow resolution changes when triple buffering
	if (strcmp(grGetString(GR_HARDWARE), "Voodoo Rush") == 0)
	{
		gNumColorBuffers = 2;
	}

	if (!grGet(GR_GLIDE_STATE_SIZE, 4, (long *)&i))
	{
		return FALSE;
	}
	if (gGlideState)
	{
		delete [] gGlideState;
	}
	gGlideState = new FxU8[i];
	if (!gGlideState)
	{
		return FXFALSE;
	}
#else // USE_GLIDE3
	if (!grSstQueryHardware(&hwconfig))
	{
		grGlideShutdown();
		return FXFALSE;
	}

	// set the number of TMUs available
	switch (hwconfig.SSTs[0].type)
	{
		case GR_SSTTYPE_VOODOO:
			gNumTMUs = hwconfig.SSTs[0].sstBoard.VoodooConfig.nTexelfx;
			break;

		case GR_SSTTYPE_SST96:
			gNumTMUs = hwconfig.SSTs[0].sstBoard.SST96Config.nTexelfx;
			break;

		case GR_SSTTYPE_AT3D:
			gNumTMUs = 1;
			break;

		case GR_SSTTYPE_Voodoo2:
			gNumTMUs = hwconfig.SSTs[0].sstBoard.Voodoo2Config.nTexelfx;
			break;

		default:
			return FXFALSE;
	}

	// HACK: Rush doesn't allow resolution changes when triple buffering
	if (hwconfig.SSTs[0].type == GR_SSTTYPE_SST96)
	{
		gNumColorBuffers = 2;
	}
#endif // USE_GLIDE3

	// create a texture cache for each tmu
	for (i=0; i<gNumTMUs; i++)
	{
		gTexCache[i] = new TextureCache(GR_TMU0+i);
		if (!gTexCache[i])
		{
			grGlideShutdown();
			return FXFALSE;
		}
	}

	grSstSelect(0);

	return FXTRUE;
}

void GlideCleanup()
{
	int i;

#ifdef USE_GLIDE3
	if (gGlideState)
	{
		delete [] gGlideState;
		gGlideState = NULL;
	}
#endif // USE_GLIDE3

	for (i=0; i<gNumTMUs; i++)
	{
		delete gTexCache[i];
		gTexCache[i] = NULL;
	}

	grGlideShutdown();
}

void StoreGlideState()
{
	// store the current glide state before changing it
#ifdef USE_GLIDE3
	grGlideGetState(gGlideState);
	grVertexLayout(GR_PARAM_ST1, 0, GR_PARAM_DISABLE);
#else
	grGlideGetState(&gGlideState);
	grHints(GR_HINT_STWHINT, 0);
#endif // USE_GLIDE3
}

void RestoreGlideState()
{
#ifdef USE_GLIDE3
  grGlideSetState(gGlideState);
#else
  grGlideSetState(&gGlideState);
#endif // USE_GLIDE3
}

int GetNumTMUs()
{
	return gNumTMUs;
}

int GetNumColorBuffers()
{
	return gNumColorBuffers;
}

TextureCache *GetTexCache(GrChipID_t tmu)
{
	return gTexCache[tmu-GR_TMU0];
}

// this should be the last drawing call before
// swapping buffers, because it will ignore the z-buffer
void DrawCursor(int x, int y)
{
	const int NUM_VERTS = 16;
	const float RADIUS = 12.0f;
	GrVertex verts[NUM_VERTS+2];
	float sin_theta, cos_theta;
	int i, color;

	// store the current glide state before changing it
#ifdef USE_GLIDE3
	grGlideGetState(gGlideState);
	grVertexLayout(GR_PARAM_ST1, 0, GR_PARAM_DISABLE);
#else
	grGlideGetState(&gGlideState);
	grHints(GR_HINT_STWHINT, 0);
#endif // USE_GLIDE3

	grAlphaBlendFunction(GR_BLEND_SRC_ALPHA, GR_BLEND_ONE_MINUS_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO);
	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grColorCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_NONE, FXFALSE);
	grDepthMask(FXFALSE);
	grDepthBufferFunction(GR_CMP_ALWAYS);

	i = linux_timeGetTime()>>2;

	color = ((i & 256) ? (i & 255) : 255 - (i & 255));
	color = (color<<16) | 0x00ff00 | (255-color);
	grConstantColorValue(color);

	verts[0].a = 127.0f + (float)((i & 128) ? (i & 127) : 127 - (i & 127));
	verts[1 + 0*(NUM_VERTS>>2)].a = 0.0f;
	verts[1 + 1*(NUM_VERTS>>2)].a = 0.0f;
	verts[1 + 2*(NUM_VERTS>>2)].a = 0.0f;
	verts[1 + 3*(NUM_VERTS>>2)].a = 0.0f;

	// compute the 4 trivial points
#ifdef USE_GLIDE3
	verts[0].x = (float)x;
	verts[0].y = (float)y;
	verts[1 + 0*(NUM_VERTS>>2)].x = (float)x + RADIUS;
	verts[1 + 0*(NUM_VERTS>>2)].y = (float)y;
	verts[1 + 1*(NUM_VERTS>>2)].x = (float)x;
	verts[1 + 1*(NUM_VERTS>>2)].y = (float)y + RADIUS;
	verts[1 + 2*(NUM_VERTS>>2)].x = (float)x - RADIUS;
	verts[1 + 2*(NUM_VERTS>>2)].y = (float)y;
	verts[1 + 3*(NUM_VERTS>>2)].x = (float)x;
	verts[1 + 3*(NUM_VERTS>>2)].y = (float)y - RADIUS;
#else
	verts[0].x = SNAP((float)x);
	verts[0].y = SNAP((float)y);
	verts[1 + 0*(NUM_VERTS>>2)].x = SNAP((float)x + RADIUS);
	verts[1 + 0*(NUM_VERTS>>2)].y = SNAP((float)y);
	verts[1 + 1*(NUM_VERTS>>2)].x = SNAP((float)x);
	verts[1 + 1*(NUM_VERTS>>2)].y = SNAP((float)y + RADIUS);
	verts[1 + 2*(NUM_VERTS>>2)].x = SNAP((float)x - RADIUS);
	verts[1 + 2*(NUM_VERTS>>2)].y = SNAP((float)y);
	verts[1 + 3*(NUM_VERTS>>2)].x = SNAP((float)x);
	verts[1 + 3*(NUM_VERTS>>2)].y = SNAP((float)y - RADIUS);
#endif // USE_GLIDE3

	// to avoid wrapping, make an extra copy of the first vert at the end
	verts[NUM_VERTS + 1].x = verts[1].x;
	verts[NUM_VERTS + 1].y = verts[1].y;
	verts[NUM_VERTS + 1].a = verts[1].a;

	// compute the ramaining points using the symmetry about each quadrant
	for (i=1; i<(NUM_VERTS>>2); i++)
	{
		fsincos(i*(2.0f*PI)*(1.0f/(float)NUM_VERTS), &sin_theta, &cos_theta);
		sin_theta *= RADIUS;
		cos_theta *= RADIUS;

		verts[1 + i + 0*(NUM_VERTS>>2)].a = 0.0f;
		verts[1 + i + 1*(NUM_VERTS>>2)].a = 0.0f;
		verts[1 + i + 2*(NUM_VERTS>>2)].a = 0.0f;
		verts[1 + i + 3*(NUM_VERTS>>2)].a = 0.0f;

#ifdef USE_GLIDE3
		verts[1 + i + 0*(NUM_VERTS>>2)].x = (float)x + cos_theta;
		verts[1 + i + 0*(NUM_VERTS>>2)].y = (float)y + sin_theta;
		verts[1 + i + 1*(NUM_VERTS>>2)].x = (float)x - sin_theta;
		verts[1 + i + 1*(NUM_VERTS>>2)].y = (float)y + cos_theta;
		verts[1 + i + 2*(NUM_VERTS>>2)].x = (float)x - cos_theta;
		verts[1 + i + 2*(NUM_VERTS>>2)].y = (float)y - sin_theta;
		verts[1 + i + 3*(NUM_VERTS>>2)].x = (float)x + sin_theta;
		verts[1 + i + 3*(NUM_VERTS>>2)].y = (float)y - cos_theta;
#else
		verts[1 + i + 0*(NUM_VERTS>>2)].x = SNAP((float)x + cos_theta);
		verts[1 + i + 0*(NUM_VERTS>>2)].y = SNAP((float)y + sin_theta);
		verts[1 + i + 1*(NUM_VERTS>>2)].x = SNAP((float)x - sin_theta);
		verts[1 + i + 1*(NUM_VERTS>>2)].y = SNAP((float)y + cos_theta);
		verts[1 + i + 2*(NUM_VERTS>>2)].x = SNAP((float)x - cos_theta);
		verts[1 + i + 2*(NUM_VERTS>>2)].y = SNAP((float)y - sin_theta);
		verts[1 + i + 3*(NUM_VERTS>>2)].x = SNAP((float)x + sin_theta);
		verts[1 + i + 3*(NUM_VERTS>>2)].y = SNAP((float)y - cos_theta);
#endif // USE_GLIDE3
	}

	// draw the fan
#ifdef USE_GLIDE3
	grDrawVertexArrayContiguous(GR_TRIANGLE_FAN, NUM_VERTS+2, &verts[0], sizeof(GrVertex));
#else
	for (i=0; i<NUM_VERTS; i++)
	{
		grDrawTriangle(&verts[0], &verts[1 + i], &verts[1 + i+1]);
	}
#endif // USE_GLIDE3

	// restore the original glide state
#ifdef USE_GLIDE3
	grGlideSetState(gGlideState);
#else
	grGlideSetState(&gGlideState);
#endif // USE_GLIDE3
}

// this should be drawn first, replacing grBufferClear
// so that it remains below all other drawings
// this ignores the z-buffer
void DrawBackground(float width, float height)
{
	GrVertex verts[4];

	// store the current glide state before changing it
#ifdef USE_GLIDE3
	grGlideGetState(gGlideState);
	grVertexLayout(GR_PARAM_ST1, 0, GR_PARAM_DISABLE);
#else
	grGlideGetState(&gGlideState);
	grHints(GR_HINT_STWHINT, 0);
#endif // USE_GLIDE3

	grTexCombine(GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
							 GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE, FXFALSE, FXFALSE);
	grColorCombine(GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
								 GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grAlphaBlendFunction(GR_BLEND_ONE, GR_BLEND_ZERO, GR_BLEND_ZERO, GR_BLEND_ZERO);

	grDepthMask(FXTRUE);
	grDepthBufferFunction(GR_CMP_ALWAYS);

	verts[0].oow = verts[1].oow = verts[2].oow = verts[3].oow = 1.0f/65535.0f;

	verts[0].x = 0.0f;
	verts[0].y = 0.0f;
	verts[0].tmuvtx[0].sow = 0.0f*verts[0].oow;
	verts[0].tmuvtx[0].tow = 256.0f*verts[0].oow;

	verts[1].x = width;
	verts[1].y = 0.0f;
	verts[1].tmuvtx[0].sow = 256.0f*verts[1].oow;
	verts[1].tmuvtx[0].tow = 256.0f*verts[1].oow;

	verts[2].x = 0.0f;
	verts[2].y = height;
	verts[2].tmuvtx[0].sow = 0.0f*verts[2].oow;
	verts[2].tmuvtx[0].tow = 0.0f*verts[2].oow;

	verts[3].x = width;
	verts[3].y = height;
	verts[3].tmuvtx[0].sow = 256.0f*verts[3].oow;
	verts[3].tmuvtx[0].tow = 0.0f*verts[3].oow;

#ifdef USE_GLIDE3
	grDrawVertexArrayContiguous(GR_TRIANGLE_STRIP, 4, &verts[0], sizeof(GrVertex));
#else
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);
#endif // USE_GLIDE3

	// restore the original glide state
#ifdef USE_GLIDE3
	grGlideSetState(gGlideState);
#else
	grGlideSetState(&gGlideState);
#endif // USE_GLIDE3
}

// this should be drawn first, replacing grBufferClear
// so that it remains below all other drawings
// this ignores the z-buffer
// the fade value must be in the range [0 255], and the higher
// the fade value, the more it'll fade the background to black
void FadeBackground(float width, float height, float fade)
{
	GrVertex verts[4];

	// store the current glide state before changing it
#ifdef USE_GLIDE3
	grGlideGetState(gGlideState);
	grVertexLayout(GR_PARAM_ST1, 0, GR_PARAM_DISABLE);
#else
	grGlideGetState(&gGlideState);
	grHints(GR_HINT_STWHINT, 0);
#endif // USE_GLIDE3

	grColorCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grAlphaCombine(GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
								 GR_COMBINE_LOCAL_ITERATED, GR_COMBINE_OTHER_NONE, FXFALSE);
	grAlphaBlendFunction(GR_BLEND_ZERO, GR_BLEND_ONE_MINUS_SRC_ALPHA, GR_BLEND_ZERO, GR_BLEND_ZERO);

	grDepthMask(FXTRUE);
	grDepthBufferFunction(GR_CMP_ALWAYS);

	verts[0].oow = verts[1].oow = verts[2].oow = verts[3].oow = 1.0f/65535.0f;

	verts[0].x = 0.0f;
	verts[0].y = 0.0f;
	verts[0].a = fade;

	verts[1].x = width;
	verts[1].y = 0.0f;
	verts[1].a = fade;

	verts[2].x = 0.0f;
	verts[2].y = height;
	verts[2].a = fade;

	verts[3].x = width;
	verts[3].y = height;
	verts[3].a = fade;

#ifdef USE_GLIDE3
	grDrawVertexArrayContiguous(GR_TRIANGLE_STRIP, 4, &verts[0], sizeof(GrVertex));
#else
	grDrawTriangle(&verts[0], &verts[1], &verts[2]);
	grDrawTriangle(&verts[1], &verts[3], &verts[2]);
#endif // USE_GLIDE3

	// restore the original glide state
#ifdef USE_GLIDE3
	grGlideSetState(gGlideState);
#else
	grGlideSetState(&gGlideState);
#endif // USE_GLIDE3
}
