/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "basics.h"
#include "mathutil.h"
#include "matstack.h"

// local variables
static int gStackSize = 0;
static Matrix *gMatrices = NULL;
static Matrix *gTopMatrix = NULL;


// **********************************************
// ****   POST CONCATENATING MATRIX STACK    ****
// **********************************************
int InitMatrixStack(int stack_size)
{
	gStackSize = stack_size;
	gMatrices = new Matrix[gStackSize];
	if (!gMatrices)
	{
		gStackSize = 0;
		gTopMatrix = NULL;
		return 0;
	}

	gTopMatrix = &gMatrices[0];
	IdentityMat(*gTopMatrix);

	return 1;
}

void CleanupMatrixStack()
{
	if (gMatrices)
	{
		delete [] gMatrices;
		gMatrices = NULL;
	}
	gStackSize = 0;
	gTopMatrix = NULL;
}

int PushMatrix()
{
	if (gTopMatrix < &gMatrices[gStackSize-1])
	{
		MatrixCopy(*(gTopMatrix+1), *gTopMatrix);
		gTopMatrix++;
		return 1;
	}
	// no more room on stack
	return 0;
}

int PopMatrix()
{
	if (gTopMatrix > &gMatrices[0])
	{
		gTopMatrix--;
		return 1;
	}
	// stack is empty
	return 0;
}

void GetMatrix(Matrix m)
{
	MatrixCopy(m, *gTopMatrix);
}

void LoadMatrix(Matrix m)
{
	MatrixCopy(*gTopMatrix, m);
}

void MultMatrix(Matrix m)
{
	MatMultMat4x4(*gTopMatrix, *gTopMatrix, m);
}

void LoadIdentity()
{
	IdentityMat(*gTopMatrix);
}

void Scale(float sx, float sy, float sz)
{
	(*gTopMatrix)[0][0] *= sx;
	(*gTopMatrix)[0][1] *= sy;
	(*gTopMatrix)[0][2] *= sz;
	(*gTopMatrix)[1][0] *= sx;
	(*gTopMatrix)[1][1] *= sy;
	(*gTopMatrix)[1][2] *= sz;
	(*gTopMatrix)[2][0] *= sx;
	(*gTopMatrix)[2][1] *= sy;
	(*gTopMatrix)[2][2] *= sz;
	(*gTopMatrix)[3][0] *= sx;
	(*gTopMatrix)[3][1] *= sy;
	(*gTopMatrix)[3][2] *= sz;
}

void Translate(float dx, float dy, float dz)
{
	(*gTopMatrix)[0][3] = (*gTopMatrix)[0][0]*dx + (*gTopMatrix)[0][1]*dy + (*gTopMatrix)[0][2]*dz + (*gTopMatrix)[0][3];
	(*gTopMatrix)[1][3] = (*gTopMatrix)[1][0]*dx + (*gTopMatrix)[1][1]*dy + (*gTopMatrix)[1][2]*dz + (*gTopMatrix)[1][3];
	(*gTopMatrix)[2][3] = (*gTopMatrix)[2][0]*dx + (*gTopMatrix)[2][1]*dy + (*gTopMatrix)[2][2]*dz + (*gTopMatrix)[2][3];
	(*gTopMatrix)[3][3] = (*gTopMatrix)[3][0]*dx + (*gTopMatrix)[3][1]*dy + (*gTopMatrix)[3][2]*dz + (*gTopMatrix)[3][3];
}

void RotateX(float angle)
{
	Matrix m;
	float cos_angle, sin_angle;

	fsincos(angle, &sin_angle, &cos_angle);

	m[0][1] = (*gTopMatrix)[0][1]*cos_angle + (*gTopMatrix)[0][2]*sin_angle;
	m[1][1] = (*gTopMatrix)[1][1]*cos_angle + (*gTopMatrix)[1][2]*sin_angle;
	m[2][1] = (*gTopMatrix)[2][1]*cos_angle + (*gTopMatrix)[2][2]*sin_angle;
	m[3][1] = (*gTopMatrix)[3][1]*cos_angle + (*gTopMatrix)[3][2]*sin_angle;
	(*gTopMatrix)[0][2] = (*gTopMatrix)[0][2]*cos_angle - (*gTopMatrix)[0][1]*sin_angle;
	(*gTopMatrix)[1][2] = (*gTopMatrix)[1][2]*cos_angle - (*gTopMatrix)[1][1]*sin_angle;
	(*gTopMatrix)[2][2] = (*gTopMatrix)[2][2]*cos_angle - (*gTopMatrix)[2][1]*sin_angle;
	(*gTopMatrix)[3][2] = (*gTopMatrix)[3][2]*cos_angle - (*gTopMatrix)[3][1]*sin_angle;
	(*gTopMatrix)[0][1] = m[0][1];
	(*gTopMatrix)[1][1] = m[1][1];
	(*gTopMatrix)[2][1] = m[2][1];
	(*gTopMatrix)[3][1] = m[3][1];
}

void RotateY(float angle)
{
	Matrix m;
	float cos_angle, sin_angle;

	fsincos(angle, &sin_angle, &cos_angle);

	m[0][0] = (*gTopMatrix)[0][0]*cos_angle - (*gTopMatrix)[0][2]*sin_angle;
	m[1][0] = (*gTopMatrix)[1][0]*cos_angle - (*gTopMatrix)[1][2]*sin_angle;
	m[2][0] = (*gTopMatrix)[2][0]*cos_angle - (*gTopMatrix)[2][2]*sin_angle;
	m[3][0] = (*gTopMatrix)[3][0]*cos_angle - (*gTopMatrix)[3][2]*sin_angle;
	(*gTopMatrix)[0][2] = (*gTopMatrix)[0][0]*sin_angle + (*gTopMatrix)[0][2]*cos_angle;
	(*gTopMatrix)[1][2] = (*gTopMatrix)[1][0]*sin_angle + (*gTopMatrix)[1][2]*cos_angle;
	(*gTopMatrix)[2][2] = (*gTopMatrix)[2][0]*sin_angle + (*gTopMatrix)[2][2]*cos_angle;
	(*gTopMatrix)[3][2] = (*gTopMatrix)[3][0]*sin_angle + (*gTopMatrix)[3][2]*cos_angle;
	(*gTopMatrix)[0][0] = m[0][0];
	(*gTopMatrix)[1][0] = m[1][0];
	(*gTopMatrix)[2][0] = m[2][0];
	(*gTopMatrix)[3][0] = m[3][0];
}

void RotateZ(float angle)
{
	Matrix m;
	float cos_angle, sin_angle;

	fsincos(angle, &sin_angle, &cos_angle);

	m[0][0] = (*gTopMatrix)[0][0]*cos_angle + (*gTopMatrix)[0][1]*sin_angle;
	m[1][0] = (*gTopMatrix)[1][0]*cos_angle + (*gTopMatrix)[1][1]*sin_angle;
	m[2][0] = (*gTopMatrix)[2][0]*cos_angle + (*gTopMatrix)[2][1]*sin_angle;
	m[3][0] = (*gTopMatrix)[3][0]*cos_angle + (*gTopMatrix)[3][1]*sin_angle;
	(*gTopMatrix)[0][1] = (*gTopMatrix)[0][1]*cos_angle - (*gTopMatrix)[0][0]*sin_angle;
	(*gTopMatrix)[1][1] = (*gTopMatrix)[1][1]*cos_angle - (*gTopMatrix)[1][0]*sin_angle;
	(*gTopMatrix)[2][1] = (*gTopMatrix)[2][1]*cos_angle - (*gTopMatrix)[2][0]*sin_angle;
	(*gTopMatrix)[3][1] = (*gTopMatrix)[3][1]*cos_angle - (*gTopMatrix)[3][0]*sin_angle;
	(*gTopMatrix)[0][0] = m[0][0];
	(*gTopMatrix)[1][0] = m[1][0];
	(*gTopMatrix)[2][0] = m[2][0];
	(*gTopMatrix)[3][0] = m[3][0];
}
