/*****************************************************************************/
/*  rfc959.c - General purpose routines for the FTP protocol (RFC 959)       */
/*  Copyright (C) 1999 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This library is free software; you can redistribute it and/or            */
/*  modify it under the terms of the GNU Library General Public              */
/*  License as published by the Free Software Foundation; either             */
/*  version 2 of the License, or (at your option) any later version.         */
/*                                                                           */
/*  This library is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        */
/*  Library General Public License for more details.                         */
/*                                                                           */
/*  You should have received a copy of the GNU Library General Public        */
/*  License along with this library; if not, write to the Free Software      */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "rfc959.h"

static time_t parse_time (char **str);
static int gftp_parse_ls_eplf (char *str, gftp_file *fle);
static int gftp_parse_ls_unix (char *str, int cols, gftp_file *fle);
static int gftp_parse_ls_nt (char *str, gftp_file *fle);
static int gftp_parse_ls_novell (char *str, gftp_file *fle);
static char *goto_next_token (char *pos);
static char *copy_token (char **dest, char *source);
static char *parse_ftp_proxy_string (gftp_request *request);

gftp_request *gftp_request_new (void) {
   gftp_request *request;
   
   request = g_malloc0 (sizeof (gftp_request));
   request->sockfd = request->datafd = -1;
   request->data_type = gftp_type_binary;
   request->transfer_type = gftp_transfer_passive;
   return (request);
}

void gftp_request_destroy (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   gftp_disconnect (request);
   if (request->hostname) g_free (request->hostname);
   if (request->username) g_free (request->username);
   if (request->password) g_free (request->password);
   if (request->account) g_free (request->account);
   if (request->directory) g_free (request->directory);
   if (request->proxy_config) g_free (request->proxy_config);
   if (request->proxy_hostname) g_free (request->proxy_hostname);
   if (request->proxy_username) g_free (request->proxy_username);
   if (request->proxy_password) g_free (request->proxy_password);
   if (request->proxy_account) g_free (request->proxy_account);
   if (request->last_ftp_response) g_free (request->last_ftp_response);
   if (request->command_buffer.buffer) g_free (request->command_buffer.buffer);
   if (request->data_buffer.buffer) g_free (request->data_buffer.buffer);
   g_free (request);
}

void gftp_file_destroy (gftp_file *file) {
   g_return_if_fail (file != NULL);
   
   if (file->file) g_free (file->file);
   if (file->user) g_free (file->user);
   if (file->group) g_free (file->group);
   if (file->attribs) g_free (file->attribs);
}

void gftp_logging (gftp_request *request, int logging, gftp_logging_func logging_function, void *ptr) {
   g_return_if_fail (request != NULL);

   request->logging = logging;
   request->logging_function = logging_function;
   request->user_data = ptr;
}

int gftp_connect (gftp_request *request) {
   char tempchar, resp, *startpos, *endpos, *tempstr, *connect_host, *dir;
   struct sockaddr_in servaddr;
   struct servent *service;
   struct hostent *host;
   unsigned int port;
   int curhost;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->hostname != NULL, -2);
   g_return_val_if_fail (request->username != NULL, -2);
   
   if ((request->sockfd = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Failed to create a socket: %s\n"),
         	g_strerror (errno));
      }
      request->sockfd = -1;
      return (-1);
   }
   memset (&servaddr, 0, sizeof(servaddr));
   servaddr.sin_family = AF_INET;

   port = *request->proxy_config != '\0' ? htons (request->proxy_port) : htons (request->port);
   connect_host = *request->proxy_config != '\0' ? request->proxy_hostname : request->hostname;
   if (ntohs (port) == 0) {
      if ((service = getservbyname ("ftp", "tcp")) == NULL) {
         port = htons (21);
         request->port = 21;
      }
      else {
         port = service->s_port;
         request->port = ntohs (service->s_port);
      }
   }
   servaddr.sin_port = port;

   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data, _("Looking up %s\n"),
      	connect_host);
   }
   if (!(host = gethostbyname (connect_host))) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Cannot look up hostname %s: %s\n"),
         	connect_host, g_strerror (errno));
      }
      close (request->sockfd);
      request->sockfd = -1;
      return (-1);
   }

   curhost = 0;
   while (host->h_addr_list[curhost] != NULL) {
      memcpy (&servaddr.sin_addr, host->h_addr_list[curhost], host->h_length);
      if (*request->proxy_config == '\0') {
         g_free (request->hostname);
         request->hostname = g_malloc (strlen (host->h_name) + 1);/*FIXME*/
         strcpy (request->hostname, host->h_name);
         connect_host = request->hostname;
      }
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, _("Trying %s:%d\n"),
         	connect_host, ntohs (port));
      }
      if (connect(request->sockfd, (struct sockaddr *) &servaddr, sizeof(servaddr)) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot connect to %s: %s\n"),
         	connect_host, g_strerror (errno));
         }
      }
      else break;
      curhost++;
   }

   if (host->h_addr_list[curhost] == NULL) {
      close (request->sockfd);
      request->sockfd = -1;
      return (-1);
   }

   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data, _("Connected to %s:%d\n"),
      	connect_host, ntohs (port));
   }

   /* Get the banner */
   while (gftp_read_response (request) == -2);
   
   /* Login the proxy server if available */
   if (*request->proxy_config != '\0') {
      resp = '3';
      startpos = endpos = tempstr = parse_ftp_proxy_string (request);
      while ((resp == '3' || resp == '2') && *startpos != '\0') {
         if (*endpos == '\n' || *endpos == '\0') {
            tempchar = *(endpos + 1);
            if (*endpos != '\0') *(endpos + 1) = '\0';
            resp = gftp_send_command (request, startpos);
            if (*endpos != '\0') *(endpos + 1) = tempchar;
            else break;
            startpos = endpos + 1;
         }
         endpos++;
      }
      g_free (tempstr);
   }
   else {
      tempstr = g_strconcat ("USER ", request->username, "\r\n", NULL);
      resp = gftp_send_command (request, tempstr);
      g_free (tempstr);
      if (resp == '3') {
         tempstr = g_strconcat ("PASS ", request->password, "\r\n", NULL);
         resp = gftp_send_command (request, tempstr);
         g_free (tempstr);
      }
      if (resp == '3' && request->account) {
         tempstr = g_strconcat ("ACCT ", request->account, "\r\n", NULL);
         resp = gftp_send_command (request, tempstr);
         g_free (tempstr);
      }
   }
   
   if (resp == '2') {
      tempstr = g_malloc (9);
      if (request->data_type == gftp_type_binary) {
         strcpy (tempstr, "TYPE I\r\n");
      }
      else {
         strcpy (tempstr, "TYPE A\r\n");
      }
      gftp_send_command (request, tempstr);

      if (!(request->directory != NULL && gftp_set_directory (request, request->directory) == 0)) {
         g_free (tempstr);
         if (gftp_send_command (request, "PWD\r\n") != '2') return (0);
      
         tempstr = strchr (request->last_ftp_response, '"');
         if (tempstr != NULL) dir = tempstr + 1;
         else return (0);
      
         tempstr = strchr (dir, '"');
         if (tempstr != NULL) *tempstr = '\0';
         else return (0);
      
         request->directory = g_malloc (strlen (dir) + 1);
         strcpy (request->directory, dir);
      }
   }
   else {
      close (request->sockfd);
      request->sockfd = -1;
      return (-2);
   }
   return (0);
}

void gftp_disconnect (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   if (request->sockfd >= 0) {
      if (request->logging) {
         request->logging_function (gftp_logging_misc, request->user_data, _("Disconnecting from site %s\n"),
         	request->hostname);
      }
      close (request->sockfd);
      request->sockfd = -1;
   }
}

int gftp_data_connection_new (gftp_request *request) {
   char *pos, *pos1, resp, command[50];
   struct sockaddr_in data_addr;
   size_t data_addr_len;
   unsigned int temp[6];
   unsigned char ad[6];
   int i;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   if ((request->datafd = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Failed to create a socket: %s\n"),
         	g_strerror (errno));
      }
      return (-1);
   }

   data_addr_len = sizeof (data_addr);
   memset(&data_addr, 0, data_addr_len);
   data_addr.sin_family = AF_INET;

   if(request->transfer_type == gftp_transfer_passive) {
      if ((resp = gftp_send_command(request, "PASV\r\n")) != '2') {
         if(resp != '5') {
            close (request->datafd);
            request->datafd = -1;
            return (-2);
         }
         else {
            request->transfer_type = gftp_transfer_active;
            return (gftp_data_connection_new (request));
         }
      }
      else {
         pos = request->last_ftp_response + 4;
         while (!isdigit (*pos) && *pos != '\0') pos++;
         if(*pos == '\0') {
            close (request->datafd);
            request->datafd = -1;
            return (-2);
         }
         if (sscanf (pos, "%d,%d,%d,%d,%d,%d", &temp[0], &temp[1], &temp[2], 
         	&temp[3], &temp[4], &temp[5]) != 6) {
            close (request->datafd);
            request->datafd = -1;
            return (-2);
         }
         for (i=0; i<6; i++) ad[i] = (unsigned char) (temp[i] & 0xff);

         memcpy (&data_addr.sin_addr, &ad[0], 4);
         memcpy (&data_addr.sin_port, &ad[4], 2);
         if (connect(request->datafd, (struct sockaddr *) &data_addr, data_addr_len) == -1) {
            if (request->logging) {
               request->logging_function (gftp_logging_error, request->user_data, _("Cannot create a data connection: %s\n"),
               		g_strerror (errno));
            }
            close (request->datafd);
            request->datafd = -1;
            return (-1);
         }
      }
   }
   else {
      getsockname (request->sockfd, (struct sockaddr *) &data_addr, &data_addr_len);
      data_addr.sin_port = 0;
      if (bind (request->datafd, (struct sockaddr *) &data_addr, data_addr_len) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot bind a port: %s\n"),
            	g_strerror (errno));
         }
         close (request->datafd);
         request->datafd = -1;
         return (-1);
      }
      getsockname (request->datafd, (struct sockaddr *) &data_addr, &data_addr_len);
      if (listen (request->datafd, 1) == -1) {
         if (request->logging) {
            request->logging_function (gftp_logging_error, request->user_data, _("Cannot listen on port %d: %s\n"), 
            	ntohs (data_addr.sin_port), g_strerror (errno));
         }
         close (request->datafd);
         request->datafd = -1;
         return (-1);
      }
      pos = (char *) &data_addr.sin_addr;
      pos1 = (char *) &data_addr.sin_port;
      g_snprintf (command, sizeof (command), "PORT %d,%d,%d,%d,%d,%d\r\n",
      		pos[0] & 0xff, pos[1] & 0xff, pos[2] & 0xff, pos[3] & 0xff,
      		pos1[0] & 0xff, pos1[1] & 0xff);
      command[sizeof (command) - 1] = '\0';
      if (gftp_send_command (request, command) != '2') {
         close (request->datafd);
         request->datafd = -1;
         return(-2);
      }
   }
   return(0);
}

int gftp_init_fxp_transfer (gftp_request *pasv_request, gftp_request *port_request) {
   char *pos, *endpos, tempstr[255];

   g_return_val_if_fail (pasv_request != NULL, -2);
   g_return_val_if_fail (port_request != NULL, -2);
   g_return_val_if_fail (pasv_request->sockfd >= 0, -2);
   g_return_val_if_fail (port_request->sockfd >= 0, -2);
   
   pasv_request->transfer_type = gftp_transfer_passive;
   port_request->transfer_type = gftp_transfer_active;
   
   if (gftp_send_command (pasv_request, "PASV\r\n") != '2') return (-2);

   pos = pasv_request->last_ftp_response + 4;
   while (!isdigit (*pos) && *pos != '\0') pos++;
   if (*pos == '\0') return (-2);
   
   endpos = pos;
   while (*endpos != ')' && *endpos != '\0') endpos++;
   if (*endpos == ')') *endpos = '\0';
   
   g_snprintf (tempstr, sizeof (tempstr), "PORT %s\r\n", pos);
   tempstr[sizeof (tempstr) - 1] = '\0';
   if (gftp_send_command (port_request, tempstr) != '2') return (-2);

   return (0);
}

int gftp_accept_active_connection (gftp_request *request) {
   struct sockaddr_in cli_addr;
   size_t cli_addr_len;
   int infd;
  
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   g_return_val_if_fail (request->transfer_type == gftp_transfer_active, -2);
   
   cli_addr_len = sizeof (cli_addr);
   if ((infd = accept (request->datafd, (struct sockaddr *) &cli_addr, &cli_addr_len)) == -1) {
      close (request->datafd);
      request->datafd = -1;
      return (-1);
   }
   close (request->datafd);
   request->datafd = infd;
   return (0);
}

int gftp_get_file (gftp_request *request, const char *filename, long startsize) {
   char command[30], *tempstr;
   int ret;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (filename != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   if (request->datafd < 0 && (ret = gftp_data_connection_new (request)) < 0) return (ret);
   if (startsize > 0) {
      g_snprintf (command, sizeof (command), "REST %ld\r\n", startsize);
      command[sizeof (command) - 1] = '\0';
      if (gftp_send_command (request, command) != '3') {
         close (request->datafd);
         request->datafd = -1;
         return (-2);
      }
   }
   tempstr = g_strconcat ("RETR ", filename, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '1') {
      close (request->datafd);
      request->datafd = -1;
      return (-2);
   }
   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = gftp_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   return (0);
}

int gftp_put_file (gftp_request *request, const char *filename, long startsize) {
   char command[30], *tempstr;
   int ret;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (filename != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   if (request->datafd < 0 && (ret = gftp_data_connection_new (request)) < 0) return (ret);
   if (startsize > 0) {
      g_snprintf (command, sizeof (command), "REST %ld\r\n", startsize);
      command[sizeof (command) - 1] = '\0';
      if (gftp_send_command (request, command) != '3') {
         close (request->datafd);
         request->datafd = -1;
         return (-2);
      }
   }
   tempstr = g_strconcat ("STOR ", filename, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '1') {
      close (request->datafd);
      request->datafd = -1;
      return (-2);
   }
   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = gftp_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   return (0);
}

int gftp_list_files (gftp_request *request) {
   char ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   
   if ((ret = gftp_data_connection_new (request)) < 0) return (ret);
   if (gftp_send_command (request, "LIST -L\r\n") != '1') {
      close (request->datafd);
      request->datafd = -1;
      return (-2);
   }

   if (request->transfer_type == gftp_transfer_active) {
      if ((ret = gftp_accept_active_connection (request)) < 0) {
         return (ret);
      }
   }
   return (0);
}

int gftp_parse_url (gftp_request *request, const char *url) {
   char *pos, *endpos, *endhostpos, *str, tempchar;
   const char *stpos;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (url != NULL, -2);
   
   for (stpos = url; *stpos == ' '; stpos++);
   str = g_malloc (strlen (stpos) + 1);
   strcpy (str, stpos);
   for (pos = str + strlen (str) - 1; *pos == ' '; pos--) *pos = '\0';
   
   pos = str;
   while (*pos == ' ') pos++;
   if (strncmp (pos, "ftp://", 6) == 0) pos += 6;
   
   if (strchr (pos, '@') != NULL) {
      /* A user/password was entered */
      if ((endpos = strchr (pos, ':')) == NULL) {
         g_free (str);
         return (-2);
      }
      *endpos = '\0';
      gftp_set_username (request, pos);

      pos = endpos+1;

      if ((endpos = strchr (pos, '@')) == NULL) {
         g_free (str);
         return (-2);
      }
      if (strchr (endpos+1, '@') != NULL) endpos = strchr (endpos+1, '@');
      *endpos = '\0';
      gftp_set_password (request, pos);

      pos = endpos+1;
   }
   else {
      gftp_set_username (request, "anonymous");
      if (request->password) g_free (request->password);
      request->password = NULL;
   }
   
   /* Now get the hostname and an optional port and optional directory */
   endhostpos = pos + strlen (pos);
   if ((endpos = strchr (pos, ':')) != NULL) {
      endhostpos = endpos;
   }
   else if((endpos = strchr(pos, '/')) != NULL) {
      endhostpos = endpos;
   }
   tempchar = *endhostpos;
   *endhostpos = '\0';
   gftp_set_hostname (request, pos);
   *endhostpos = tempchar;

   if ((endpos = strchr (pos, ':')) != NULL) {
      /* A port was entered */
      pos = endpos + 1;
      gftp_set_port (request, strtol (pos, NULL, 10));
   }
   if ((endpos = strchr (pos, '/')) != NULL) {
      gftp_set_directory (request, endpos);
   }
   g_free (str);
   return(0);
}

static time_t parse_time (char **str) {
   const char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", 
   			   "Aug", "Sep", "Oct", "Nov", "Dec"};
   char *startpos, *endpos, *datepos;
   struct tm curtime, tt;
   time_t t;
   int i;

   startpos = *str;
   memset (&tt, 0, sizeof (tt));
   if (isdigit (startpos[0]) && startpos[2] == '-') {
      /* This is how NT will return the date/time */
      /* 07-06-99  12:57PM */
      if((endpos = strchr (startpos, '-')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_mon = strtol (startpos, NULL, 10) - 1;

      startpos = endpos + 1;
      if((endpos = strchr (startpos, '-')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_mday = strtol (startpos, NULL, 10);

      startpos = endpos + 1;
      if((endpos = strchr (startpos, ' ')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_year = strtol (startpos, NULL, 10);

      while (*endpos == ' ') endpos++;

      startpos = endpos + 1;
      if ((endpos = strchr (startpos, ':')) == NULL) {
         g_free (str);
         return (0);
      }
      tt.tm_hour = strtol (startpos, NULL, 10);

      startpos = endpos + 1;
      while ((*endpos != 'A') && (*endpos != 'P')) endpos++;
      if (*endpos == 'P') {
         if(tt.tm_hour == 12) tt.tm_hour = 0;
         else tt.tm_hour += 12;
      }
      tt.tm_min = strtol (startpos, NULL, 10);
   }
   else {
      /* This is how most UNIX, Novell, and MacOS ftp servers send their time */
      /* Jul 06 12:57 */
      t = time (NULL);
      curtime = *localtime (&t);
   
      /* Get the month */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (0);
      }
      for (i=0; i<12; i++) {
         if (strncmp (months[i], startpos, 3) == 0) {
            tt.tm_mon = i;
            break;
         }
      }
   
      /* Skip the blanks till we get to the next entry */
      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      /* Get the day */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (0);
      }
      tt.tm_mday = strtol (startpos, NULL, 10);

      /* Skip the blanks till we get to the next entry */
      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      if ((datepos = strchr (startpos, ':')) != NULL) {
         /* No year was specified. We will use the current year */
         tt.tm_year = curtime.tm_year;

         /* If the date is in the future, than the year is from last year */
         if ((tt.tm_mon > curtime.tm_mon) || 
      		((tt.tm_mon == curtime.tm_mon) && tt.tm_mday > curtime.tm_mday)) {
           tt.tm_year--;
         }

         /* Get the hours and the minutes */
         tt.tm_hour = strtol (startpos, NULL, 10);
         tt.tm_min = strtol (datepos + 1, NULL, 10);
      }
      else {
         /* Get the year. The hours and minutes will be assumed to be 0 */
         tt.tm_year = strtol (startpos, NULL, 10) - 1900;
      }
   }
   *str = startpos;
   return (mktime (&tt));
}

static int gftp_parse_ls_eplf (char *str, gftp_file *fle) {
   char *startpos;
 
   startpos = str;
   fle->attribs = g_malloc (11);
   strcpy (fle->attribs, "----------");
   while (startpos) {
      startpos++;
      switch (*startpos) {
         case '/' :
            *fle->attribs = 'd';
            break;
         case 's' :
            fle->size = strtol (startpos+1, NULL, 10);
            break;
         case 'm' :
            fle->datetime = strtol (startpos+1, NULL, 10);
            break;
      }
      startpos = strchr (startpos, ',');
   }
   if ((startpos = strchr (str, 9)) == NULL) {
      return (-2);
   }
   fle->file = g_malloc (strlen (startpos));
   strcpy (fle->file, startpos+1);
   fle->user = g_malloc (8);
   strcpy (fle->user, _("unknown"));
   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));
   return (0);
}

static int gftp_parse_ls_unix (char *str, int cols, gftp_file *fle) {
   char *endpos, *startpos;

   startpos = str;
   /* Copy file attributes */
   if ((startpos = copy_token (&fle->attribs, startpos)) == NULL) {
      return (-2);
   }

   if (cols >= 9) {
      /* Skip the number of links */
      startpos = goto_next_token (startpos);

      /* Copy the user that owns this file */
      if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
         return (-2);
      }

      /* Copy the group that owns this file */
      if ((startpos = copy_token (&fle->group, startpos)) == NULL) {
         return (-2);
      }
   }
   else {
      fle->group = g_malloc (8);
      strcpy (fle->group, _("unknown"));
      if (cols == 8) {
         if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
            return (-2);
         }
      }
      else {
         fle->user = g_malloc (8);
         strcpy (fle->user, _("unknown"));
      }
      startpos = goto_next_token (startpos);
   }
   

   if(fle->attribs[0] == 'b' || fle->attribs[0] == 'u' || fle->attribs[0] == 'c') {
      /* This is a block or character device. We will store the major number
         in the high word and the minor number in the low word */

      /* Get the major number */
      if ((endpos = strchr (startpos, ',')) == NULL) {
         return (-2);
      }
      fle->size = strtol (startpos, NULL, 10) << 16;

      startpos = endpos + 1;
      while (*startpos == ' ') startpos++;

      /* Get the minor number */
      if ((endpos = strchr(startpos, ' ')) == NULL) {
         return (-2);
      }
      fle->size |= strtol (startpos, NULL, 10);
   }
   else {
      /* This is a regular file  */
      if ((endpos = strchr (startpos, ' ')) == NULL) {
         return (-2);
      }
      fle->size = strtol (startpos, NULL, 10);
   }

   /* Skip the blanks till we get to the next entry */
   startpos = endpos + 1;
   while (*startpos == ' ') startpos++;

   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   }
   
   /* Skip the blanks till we get to the next entry */
   startpos = goto_next_token (startpos);

   /* Parse the filename. If this file is a symbolic link, remove the -> part */
   if (fle->attribs[0] == 'l' && ((endpos = strstr (startpos, "->")) != NULL)) {
      *(endpos-1) = '\0';
   }
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);   
   return (0);
}

static int gftp_parse_ls_nt (char *str, gftp_file *fle) {
   char *startpos;
 
   startpos = str;
   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   };

   /* No such thing on Windoze.. */
   fle->user = g_malloc (8);
   strcpy (fle->user, _("unknown"));
   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));

   startpos = goto_next_token (startpos);
   fle->attribs = g_malloc (11);
   if(startpos[0] == '<') {
      strcpy (fle->attribs, "drwxrwxrwx");
   }
   else {
      strcpy (fle->attribs, "-rw-rw-rw-");
      fle->size = strtol (startpos, NULL, 10);
   }
   startpos = goto_next_token (startpos);
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);
   return (0);
}

static int gftp_parse_ls_novell (char *str, gftp_file *fle) {
   char *startpos;
   
   if (str[12] != ' ') return (-2);
   str[12] = '\0';
   fle->attribs = g_malloc (13);
   strcpy (fle->attribs, str);
   startpos = str + 13;

   while ((*startpos == ' ' || *startpos == '\t') && *startpos != '\0') startpos++;
   if ((startpos = copy_token (&fle->user, startpos)) == NULL) {
      return (-2);
   }

   fle->group = g_malloc (8);
   strcpy (fle->group, _("unknown"));
   
   fle->size = strtol (startpos, NULL, 10);

   startpos = goto_next_token (startpos);
   if ((fle->datetime = parse_time (&startpos)) == 0) {
      return (-2);
   }

   startpos = goto_next_token (startpos);
   fle->file = g_malloc (strlen (startpos) + 1);
   strcpy (fle->file, startpos);
   return (0);
}

int gftp_parse_ls (const char *lsoutput, gftp_file *fle) {
   int result, cols;
   char *str, *pos;

   g_return_val_if_fail (lsoutput != NULL, -2);
   g_return_val_if_fail (fle != NULL, -2);
   
   str = g_malloc (strlen (lsoutput) + 1);
   strcpy (str, lsoutput);
   memset (fle, 0, sizeof (gftp_file));

   if (str[strlen (str) - 1] == '\n') str[strlen (str) - 1] = '\0';
   if (str[strlen (str) - 1] == '\r') str[strlen (str) - 1] = '\0';
   if (*lsoutput == '+') {
      /* EPLF format */
      result = gftp_parse_ls_eplf (str, fle);
   }
   else if (isdigit (str[0]) && str[2] == '-') {
      /* WinNT format */
      result = gftp_parse_ls_nt (str, fle);
   }
   else if (str[1] == ' ' && str[2] == '[') {
      /* Novell format */
      result = gftp_parse_ls_novell (str, fle);
   }
   else {
     /* UNIX/MacOS format */
      
     /* If there is no space between the attribs and links field, just make one */
     if (strlen (str) > 10) str[10] = ' ';
      
     /* Determine the number of columns */
     cols = 0;
     pos = str;
     while (*pos != '\0') {
        while (*pos != '\0' && *pos != ' ' && *pos != '\t') {
           if (*pos == ':') break;
           pos++;
        }
        cols++;
        if (*pos == ':') {
           cols++;
           break;
        }
        while (*pos != '\0' && (*pos == ' ' || *pos == '\t')) pos++;
     }
     if (cols > 6) result = gftp_parse_ls_unix (str, cols, fle);
     else result = -2;
   }
   g_free (str);
   return (result);
}

static char *goto_next_token (char *pos) {
   while (*pos != ' ' && *pos != '\t' && *pos != '\0') pos++;
   if (pos == '\0') return (pos);
   while ((*pos == ' ' || *pos == '\t') && *pos != '\0') pos++;
   return (pos);
}

static char *copy_token (char **dest, char *source) {
   /* This function is used internally by gftp_parse_ls () */
   char *endpos;

   endpos = source;
   while (*endpos != ' ' && *endpos != '\t' && *endpos != '\0') endpos++;
   if (*endpos == '\0') {
      return (NULL);
   }
   *endpos = '\0';
   *dest = g_malloc (endpos - source + 1);
   strcpy (*dest, source);
   
   /* Skip the blanks till we get to the next entry */
   source = endpos + 1;
   while ((*source == ' ' || *source == '\t') && *source != '\0') source++;
   return (source);
}

int gftp_get_next_file (gftp_request *request, gftp_file *fle) {
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (fle != NULL, -2);
   g_return_val_if_fail (request->datafd >= 0, -2);

   if (request->last_dir_entry) {
      g_free (request->last_dir_entry);
      request->last_dir_entry = NULL;
   }
   if ((request->last_dir_entry = gftp_read_line (request->datafd, &request->data_buffer))) {
      if (gftp_parse_ls (request->last_dir_entry, fle) != 0) {
         /* If we cannot parse this file listing, we will just clear the
            entire structure. The calling routine can just check for a 
            NULL pointer in the fle->file field */
         memset (fle, 0, sizeof (gftp_file));
      }
      
      return (strlen (request->last_dir_entry));
   }
   return (0);
}

int gftp_end_transfer (gftp_request *request) {
   char ret;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   g_return_val_if_fail (request->datafd >= 0, -2);
   
   close (request->datafd);
   ret = gftp_read_response (request);
   request->datafd = -1;
   if (request->data_buffer.buffer) {
      g_free (request->data_buffer.buffer);
   }
   if (request->command_buffer.buffer) {
      g_free (request->command_buffer.buffer);
   }
   request->data_buffer.buffer = request->data_buffer.pos = NULL;
   request->command_buffer.buffer = request->command_buffer.pos = NULL;
   return (ret == '2' ? 0 : -2);
}

int gftp_send_command (gftp_request *request, const char *command) {
   int ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (command != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   
   if (request->logging) { 
      request->logging_function (gftp_logging_send, request->user_data, command);
   }

   do {
      ret = write (request->sockfd, command, strlen (command));
      if (ret == -1 && errno != EINTR) {
         return (-1);
      }
   } while (ret == -1 && errno == EINTR);
   
   return (gftp_read_response (request));
}

char *gftp_read_line (int sockfd, gftp_readline_buffer *buffer) {
   char readstr[255], *tempstr, *retstr, *pos;
   int ret;
   
   g_return_val_if_fail (buffer != NULL, NULL);

   ret = 1;
   retstr = NULL;
   while (ret != 0) {
      if (!buffer->buffer || !buffer->pos || *buffer->pos == '\0'
      		|| strchr (buffer->pos, '\n') == NULL) {
         if (buffer->pos && *buffer->pos != '\0') {
            tempstr = g_malloc (strlen (buffer->pos) + 1);
            strcpy (tempstr, buffer->pos);
            g_free (buffer->buffer);
         }
         else tempstr = NULL;

         memset (readstr, 0, sizeof (readstr));
         do {
            ret = read (sockfd, readstr, sizeof (readstr) - 1);
            if (ret == -1 && errno != EINTR) {
               return (NULL);
            }
         } while (ret == -1 && errno == EINTR);

         if (tempstr) {
            buffer->buffer = g_strconcat (tempstr, readstr, NULL);
            buffer->pos = buffer->buffer;
            g_free (tempstr);
         }
         else {
            buffer->buffer = g_malloc (strlen (readstr) + 1);
            strcpy (buffer->buffer, readstr);
         }
         if (!buffer->pos) {
            buffer->pos = buffer->buffer;
         }
      }

      if ((pos = strchr (buffer->pos, '\n'))) {
         if (*(pos-1) == '\r') *(pos-1) = '\0';
         else *pos = '\0';
         retstr = g_malloc (strlen (buffer->pos) + 1);
         strcpy (retstr, buffer->pos);
         buffer->pos = pos + 1;
         while (*buffer->pos == '\n' || *buffer->pos == '\r') buffer->pos++;
         if (*buffer->pos == '\0') {
            g_free (buffer->buffer);
            buffer->buffer = buffer->pos = NULL;
         }
         break;
      }
   }
   return (ret == 0 && buffer->pos && *buffer->pos == '\0' ? NULL : retstr);
}

int gftp_read_response (gftp_request *request) {
   char code[4];
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   *code = '\0';
   if (request->last_ftp_response) {
      g_free (request->last_ftp_response);
      request->last_ftp_response = NULL;
   }
   do {
      if (!(request->last_ftp_response = gftp_read_line (request->sockfd, &request->command_buffer))) {
         break;
      }
      if (*code == '\0') {
         strncpy (code, request->last_ftp_response, 3);
         code[3] = ' ';
      }
      if (request->logging) { 
         request->logging_function (gftp_logging_recv, request->user_data, "%s\n", request->last_ftp_response);
      }
   } while (strncmp (code, request->last_ftp_response, 4) != 0);
   
   return (request->last_ftp_response ? *request->last_ftp_response : -2);
}
 
int gftp_set_data_type (gftp_request *request, gftp_data_type type) {
   char tempstr[9];

   g_return_val_if_fail (request != NULL, -2);

   if (request->sockfd >= 0 && request->data_type != type) {
      if (type == gftp_type_binary) {
         strcpy (tempstr, "TYPE I\r\n");
      }
      else {
         strcpy (tempstr, "TYPE A\r\n");
      }
      if (gftp_send_command (request, tempstr) != '2') return (-2);
   }
   request->data_type = type;
   return (0);
}

void gftp_set_hostname (gftp_request *request, const char *hostname) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (hostname != NULL);
   
   if (request->hostname) g_free (request->hostname);
   if (*hostname == '\0') {
      request->hostname = NULL;
      return;
   }

   request->hostname = g_malloc (strlen (hostname) + 1);
   strcpy (request->hostname, hostname);
}

void gftp_set_username (gftp_request *request, const char *username) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (username != NULL);
   
   if (request->username) g_free (request->username);
   if (*username == '\0') {
      request->username = NULL;
      return;
   }

   request->username = g_malloc (strlen (username) + 1);
   strcpy (request->username, username);
}

void gftp_set_password (gftp_request *request, const char *password) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (password != NULL);
   
   if (request->password) g_free (request->password);
   if (*password == '\0') {
      request->password = NULL;
      return;
   }

   request->password = g_malloc (strlen (password) + 1);
   strcpy (request->password, password);
}

void gftp_set_account (gftp_request *request, const char *account) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (account != NULL);
   
   if (request->account) g_free (request->account);
   if (*account == '\0') {
      request->account = NULL;
      return;
   }

   request->account = g_malloc (strlen (account) + 1);
   strcpy (request->account, account);
}

int gftp_set_directory (gftp_request *request, const char *directory) {
   char ret, *tempstr, *dir;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);

   if (*directory == '\0') {
      if (request->directory) g_free (request->directory);
      request->directory = NULL;
      return (0);
   }
   
   /* See if we are connected to a remote site */
   if (request->sockfd >= 0) {
      if (strcmp (directory, "..") == 0) {
         ret = gftp_send_command (request, "CDUP\r\n");
      }
      else {
         tempstr = g_strconcat ("CWD ", directory, "\r\n", NULL);
         ret = gftp_send_command (request, tempstr);
         g_free (tempstr);
      }
      if (ret != '2') return (-2);
      if (request->directory) {
         g_free (request->directory);
         request->directory = NULL;
      }
      
      /* We successfully changed to the new directory. Now get the new cwd */
      if (gftp_send_command (request, "PWD\r\n") != '2') return (-2);
      
      tempstr = strchr (request->last_ftp_response, '"');
      if (tempstr != NULL) dir = tempstr + 1;
      else return (-2);
      
      tempstr = strchr (dir, '"');
      if (tempstr != NULL) *tempstr = '\0';
      else return (-2);
      
      request->directory = g_malloc (strlen (dir) + 1);
      strcpy (request->directory, dir);
   }
   else {
      request->directory = g_malloc (strlen (directory) + 1);
      strcpy (request->directory, directory);
   }
   return (0);
}

void gftp_set_port (gftp_request *request, unsigned int port) {
   g_return_if_fail (request != NULL);
   
   request->port = port;
}

void gftp_set_proxy_hostname (gftp_request *request, const char *hostname) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (hostname != NULL);
   
   if (request->proxy_hostname) g_free (request->proxy_hostname);
   if (*hostname == '\0') {
      request->proxy_hostname = NULL;
      return;
   }

   request->proxy_hostname = g_malloc (strlen (hostname) + 1);
   strcpy (request->proxy_hostname, hostname);
}

void gftp_set_proxy_username (gftp_request *request, const char *username) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (username != NULL);
   
   if (request->proxy_username) g_free (request->proxy_username);
   if (*username == '\0') {
      request->proxy_username = NULL;
      return;
   }

   request->proxy_username = g_malloc (strlen (username) + 1);
   strcpy (request->proxy_username, username);
}

void gftp_set_proxy_password (gftp_request *request, const char *password) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (password != NULL);
   
   if (request->proxy_password) g_free (request->proxy_password);
   if (*password == '\0') {
      request->proxy_password = NULL;
      return;
   }

   request->proxy_password = g_malloc (strlen (password) + 1);
   strcpy (request->proxy_password, password);
}

void gftp_set_proxy_account (gftp_request *request, const char *account) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (account != NULL);
   
   if (request->proxy_account) g_free (request->proxy_account);
   if (*account == '\0') {
      request->proxy_account = NULL;
      return;
   }

   request->proxy_account = g_malloc (strlen (account) + 1);
   strcpy (request->proxy_account, account);
}

void gftp_set_proxy_port (gftp_request *request, unsigned int port) {
   g_return_if_fail (request != NULL);
   
   request->proxy_port = port;
}

int gftp_remove_directory (gftp_request *request, const char *directory) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   tempstr = g_strconcat ("RMD ", directory, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}

int gftp_remove_file (gftp_request *request, const char *file) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (file != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   tempstr = g_strconcat ("DELE ", file, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}

int gftp_make_directory (gftp_request *request, const char *directory) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   tempstr = g_strconcat ("MKD ", directory, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}

int gftp_rename_file (gftp_request *request, const char *oldname, const char *newname) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (oldname != NULL, -2);
   g_return_val_if_fail (newname != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   tempstr = g_strconcat ("RNFR ", oldname, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   if (ret != '3') return (-2);

   tempstr = g_strconcat ("RNTO ", newname, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}

int gftp_chmod (gftp_request *request, const char *file, int mode) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (file != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);

   tempstr = g_malloc (strlen (file) + (mode / 10) + 16);
   sprintf (tempstr, "SITE CHMOD %d %s\r\n", mode, file);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret == '2' ? 0 : -2);
}

char gftp_site_cmd (gftp_request *request, const char *command) {
   char *tempstr, ret;
   
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (command != NULL, -2);
   g_return_val_if_fail (request->sockfd >= 0, -2);
   
   tempstr = g_strconcat ("SITE ", command, "\r\n", NULL);
   ret = gftp_send_command (request, tempstr);
   g_free (tempstr);
   return (ret);
}

unsigned long gftp_get_file_size (gftp_request *request, const char *filename) {
   char *tempstr;
   
   g_return_val_if_fail (request != NULL, 0);
   g_return_val_if_fail (filename != NULL, 0);
   g_return_val_if_fail (request->sockfd >= 0, 0);
   
   tempstr = g_strconcat ("SIZE ", filename, "\r\n", NULL);
   gftp_send_command (request, tempstr);
   g_free (tempstr);

   if (*request->last_ftp_response != '2') return (0);
   return (strtol (request->last_ftp_response + 4, NULL, 10));
}

static char *parse_ftp_proxy_string (gftp_request *request) {
   char *startpos, *endpos, *oldstr, *newstr, *newval, tempport[10];

   g_return_val_if_fail (request != NULL, NULL);
   
   newstr = g_malloc (1);
   *newstr = '\0';
   startpos = endpos = request->proxy_config;
   while (*endpos != '\0') {
      if (*endpos == '%' && tolower (*(endpos + 1)) == 'p') {
         switch (tolower (*(endpos + 2))) {
            case 'u':
               newval = request->proxy_username;
               break;
            case 'p':
               newval = request->proxy_password;
               break;
            case 'h':
               newval = request->proxy_hostname;
               break;
            case 'o':
               g_snprintf (tempport, sizeof (tempport), "%d", request->proxy_port);
               newval = tempport;
               break;
            case 'a':
               newval = request->proxy_account;
               break;
            default:
               endpos++;
               continue;
         }
      }
      else if (*endpos == '%' && tolower (*(endpos + 1)) == 'h') {
         switch (tolower (*(endpos + 2))) {
            case 'u':
               newval = request->username;
               break;
            case 'p':
               newval = request->password;
               break;
            case 'h':
               newval = request->hostname;
               break;
            case 'o':
               g_snprintf (tempport, sizeof (tempport), "%d", request->port);
               newval = tempport;
               break;
            case 'a':
               newval = request->account;
               break;
            default:
               endpos++;
               continue;
         }
      }
      else if (*endpos == '%' && tolower (*(endpos + 1)) == 'n') {
         *endpos = '\0';
         oldstr = newstr;
         newstr = g_strconcat (oldstr, startpos, "\r\n", NULL);
         g_free (oldstr);
         endpos += 2;
         startpos = endpos;
         continue;
      }
      else {
         endpos++;
         continue;
      }

      *endpos = '\0'; 
      oldstr = newstr;
      if (!newval) newstr = g_strconcat (oldstr, startpos, NULL);
      else newstr = g_strconcat (oldstr, startpos, newval, NULL);
      g_free (oldstr);
      endpos += 3;
      startpos = endpos;
   }
   return (newstr);
}

void gftp_set_proxy_config (gftp_request *request, const char *proxy_config) {
   int len;

   g_return_if_fail (request != NULL);
   g_return_if_fail (proxy_config != NULL);
   
   if (request->proxy_config) g_free (request->proxy_config);
   len = strlen (proxy_config);

   if (len > 0 && (proxy_config[len - 1] != 'n' || 
   	proxy_config[strlen (proxy_config) - 2] != '%')) {
      
      len += 2;
   }

   request->proxy_config = g_malloc (len + 1);
   strcpy (request->proxy_config, proxy_config);
   if (len != strlen (proxy_config)) {
      request->proxy_config[len - 2] = '%';
      request->proxy_config[len - 1] = 'n';
      request->proxy_config[len] = '\0';
   }
}
