/* vi:set ts=8 sts=0 sw=8:
 * $Id: doc.c,v 1.34 1999/01/21 21:10:12 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include "main.h"
#include <gdk/gdkkeysyms.h>
#include "doc.h"
#include "menu.h"
#include "misc.h"
#include "file.h"
#include "dialog.h"
#include "toolbar.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "recent.h"
#ifdef APP_GNP
#include "project.h"
#include "print.h"
#endif

/*** external functions ***/
extern void win_set_title(doc_t *);


/*** local types ***/
#define INSERT	0
#define APPEND	1
#define FLW_NUM_COLUMNS	3

typedef struct {
	GtkWidget *toplev;
	GSList *adv_list;
	GtkWidget *adv_button;
} doc_info_t;


/*** local variables ***/
static unsigned nextdocid = 0;
static toolbar_data_t dlw_tbdata[] = {
#ifdef APP_GNP
	{ " Save ",	"Save file",		"dlw Save",	"tb_save.xpm",
			(GtkSignalFunc)doc_save_cb },
#endif
	{ " Close ",	"Close current file",	"dlw Close",	"tb_cancel.xpm",
			(GtkSignalFunc)doc_close_cb },
#ifdef APP_GNP
	{ " Print ",	"Print file",		"dlw Print",	"tb_print.xpm",
			(GtkSignalFunc)print_cb },
#endif
	{ " SPACE " },
	{ " Ok ",	"Close list window",	"dlw Ok",	"tb_exit.xpm",
			(GtkSignalFunc)doc_list_destroy },
	{ NULL }
};


/*** local function prototypes ***/
static void	doc_add_to_notebook(doc_t *d);
#ifdef USE_AUTO_INDENT
static gboolean doc_auto_indent_cb(GtkWidget *, GdkEventKey *, win_t *w);
#endif
static bool_t	doc_close_common(win_t *w);
static bool_t	doc_close_execute(win_t *w);
#ifdef APP_GNP
static bool_t	doc_close_verify_yes(win_t *w);
static bool_t	doc_close_verify(win_t *w);
#endif
static void	doc_free(doc_t *);
static void	doc_info_advanced(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_close(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_info_table_add(GtkWidget *, doc_info_t *, char *, int, int,
				   int , int, bool_t);
static int	doc_list_select(GtkWidget *, int, int, GdkEventButton *,
				gpointer);
static void	doc_list_update(win_t *w, doc_t *d, int rownum, char *text);
static void	doc_open_filesel_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_open_ok(GtkWidget *wgt, gpointer cbdata);
static doc_t *	doc_remove(win_t *w);
#ifdef APP_GNP
static void	doc_saveas_destroy(GtkWidget *wgt, gpointer cbdata);
static void	doc_saveas_ok(GtkWidget *wgt, gpointer data);
#endif
static void	doc_open_dialog_create(win_t *w, char *title,
				       void (*open_ok)(GtkWidget *, gpointer));


/*** global function definitions ***/
/*
 * PUBLIC: doc_basefname
 *
 * convenience routine.  should probably be a macro, but oh well.
 */
char *
doc_basefname(doc_t *d)
{
	return (char *)my_basename(d->fname);
} /* doc_current */


/*
 * PUBLIC: doc_by_name
 *
 * goes to a document by its filename.  if the document is not already opened,
 * then open it.  currently used only as a menu callback for the recent doc
 * list.  see recent_list_add() and recent_list_init().
 */
void
doc_by_name(GtkWidget *wgt, gpointer cbdata)
{
	GSList *dlp;
	char *full;
	int num;
	doc_t *d;
	doc_new_name_t *dnnp = (doc_new_name_t *)cbdata;

	/* see if we already have it open.  if so, go to it immediately */
	num = 0;
	g_assert(dnnp->w != NULL);
	dlp = dnnp->w->doclist;
	g_assert(dlp != NULL);
	while (dlp) {
		d = (doc_t *)(dlp->data);
		g_assert(d != NULL);
		g_assert(d->fname != NULL);

		/*
		 * since full pathnames are always stored in dnnp, we need to
		 * construct a full pathname for the doc's fname as well.
		 */
		full = file_full_pathname_make(d->fname);

		if (strcmp(full, dnnp->fname) == 0) {
			g_free(full);
			gtk_notebook_set_page(GTK_NOTEBOOK(dnnp->w->nb), num);
			break;
		}

		num++;
		dlp = dlp->next;
		g_free(full);
	} /* while dlp */

	/* not open, so open it, and close 'Untitled' if necessary */
	if (dlp == NULL) {
		d = doc_current(dnnp->w);
		if (dnnp->w->numdoc == 1 &&
#ifdef APP_GNP
		    !d->changed &&
#endif
		    strcmp(d->fname, UNTITLED) == 0) {
			doc_close_common(dnnp->w);
		}

		doc_new(dnnp->w, dnnp->fname, TRUE);
	}
} /* doc_by_name */


#ifdef APP_GNP
/*
 * PUBLIC: doc_changed_cb
 *
 * callback to indicate document changed.  also called by file_save_execute(),
 * since after saving a file, the document is no longer "changed".
 */
void
doc_changed_cb(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;

	d->changed = TRUE;
	gtk_signal_disconnect(GTK_OBJECT(d->data), d->changed_id);
	d->changed_id = FALSE;
	doc_info_label_update(d->w);
} /* doc_changed_cb */


/*
 * PUBLIC: doc_check_if_any_changed
 *
 * checks if any documents in the given window has been changed/updated.
 * returns TRUE if there are, FALSE if not.
 */
bool_t
doc_check_if_any_changed(win_t *w)
{
	GSList *dp;
	doc_t *d;
	bool_t ret = FALSE;

	g_assert(w != NULL);
	dp = w->doclist;

	while (dp) {
		d = (doc_t *)(dp->data);
		g_assert(d != NULL);
		if (d->changed) {
			ret = TRUE;
			break;
		}
		dp = dp->next;
	}

	return ret;
} /* doc_check_if_any_changed */
#endif


/*
 * PUBLIC: doc_close_cb
 *
 * close file callback
 */
void
doc_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	/* if all we have is a blank, Untitled doc, then return immediately */
#ifdef APP_GNP
	if (!d->changed && w->numdoc == 1 && strcmp(d->fname, UNTITLED) == 0)
#endif
		return;

	(void)doc_close_common(w);

	recent_list_write(w);

	if (w->numdoc < 1)
		doc_new(w, UNTITLED, TRUE);
} /* doc_close_cb */


/*
 * PUBLIC: doc_close_all_cb
 *
 * callback for closing all files
 */
void
doc_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);

	/* if all we have is a blank, Untitled doc, then return immediately */
#ifdef APP_GNP
	if (!d->changed && w->numdoc == 1 && strcmp(d->fname, UNTITLED) == 0)
#endif
		return;

	(void)doc_close_all_common(w);

	/*
	 * doc_close_all_common() won't create an "untitled" document, so
	 * do it here.
	 */
	if (w->numdoc < 1)
		doc_new(w, UNTITLED, TRUE);
} /* doc_close_all_cb */


/*
 * PUBLIC: doc_close_all_common
 *
 * common routine for closing all open files.  called from the menu callback
 * routine as well as win_close_common().  returns TRUE if all the files were
 * closed, FALSE if not.
 */
bool_t
doc_close_all_common(win_t *w)
{
	bool_t allclosed = TRUE;

	g_assert(w != NULL);
	while (w->numdoc > 0) {
		/*
		 * checking if any docs have changed may slow things down a
		 * bit, but it's comestically better since it avoids having
		 * to hide the window, and then unhide it again when getting
		 * to a changed document.
		 */
#ifdef APP_GNP
		if (doc_check_if_any_changed(w) == FALSE)
#endif
			if (GTK_WIDGET_VISIBLE(w->nb))
				gtk_widget_hide(w->nb);

		allclosed = doc_close_common(w);
		if (allclosed == FALSE)	/* not closed */
			break;
	}

	if (!GTK_WIDGET_VISIBLE(w->nb))
		gtk_widget_show(w->nb);

	recent_list_write(w);

	return allclosed;
} /* doc_close_all_common */


/*
 * PUBLIC: doc_current
 *
 * convenience routine.  should probably be a macro, but since it isn't, be
 * sure to turn on compiler optimization for automatic inlining.
 */
doc_t *
doc_current(win_t *w)
{
	return (doc_t *)(w->curdoc);
} /* doc_current */


/*
 * PUBLIC: doc_foreach
 *
 * given a window, traverses all documents of that window and calls func()
 * on each.
 */
void
doc_foreach(win_t *wp, void (*func)(void *))
{
	GSList *dlp;
	doc_t *dp;

	for (dlp = wp->doclist; dlp; dlp = dlp->next) {
		dp = (doc_t *)dlp->data;
		func(dp);
	}
} /* doc_foreach */


/*
 * PUBLIC: doc_info_cb
 *
 * callback invoked when docinfo button is clicked
 */
void
doc_info_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *table, *tmp, *vbox;
	char *s;
	doc_info_t *di;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	if (di) {
		gdk_window_raise(di->toplev->window);
		return;
	}

	di = (doc_info_t *)g_malloc0(sizeof(doc_info_t));
	d->docinfo = di;
	di->toplev = gtk_dialog_new();
	gtk_signal_connect(GTK_OBJECT(di->toplev), "destroy",
			   GTK_SIGNAL_FUNC(doc_info_destroy), d);
	gtk_window_set_title(GTK_WINDOW(di->toplev), "File Information");

	vbox = GTK_DIALOG(di->toplev)->vbox;
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	table = gtk_table_new(19, 2, FALSE);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, FALSE, 0);
	gtk_widget_show(table);

	/* blech, this is a whole lot of code just for some basic info */
	/* i wish there was a more efficient way of doing this */
	doc_info_table_add(table, di, "Name:",            0, 1, 0, 1, FALSE);
	doc_info_table_add(table, di, " ",                0, 1, 1, 2, FALSE);
	doc_info_table_add(table, di, "Modified:",        0, 1, 2, 3, FALSE);
	doc_info_table_add(table, di, "Read Only:",       0, 1, 3, 4, FALSE);
	doc_info_table_add(table, di, " ",                0, 1, 4, 5, FALSE);
	doc_info_table_add(table, di, "Bytes in memory:", 0, 1, 5, 6, FALSE);
	doc_info_table_add(table, di, "Bytes on disk:",   0, 1, 6, 7, FALSE);

	doc_info_table_add(table, di, doc_basefname(d),   1, 2, 0, 1, FALSE);
	doc_info_table_add(table, di, "                ", 1, 2, 1, 2, FALSE);
#ifdef APP_GNP
	doc_info_table_add(table, di,
		(d->changed) ? "Yes" : "No",              1, 2, 2, 3, FALSE);
#endif
	doc_info_table_add(table, di, (d->sb) ?
		((file_is_readonly(d->fname)) ?
		"Yes" : "No" ) : "No",                    1, 2, 3, 4, FALSE);
	doc_info_table_add(table, di, " ",                1, 2, 4, 5, FALSE);
	doc_info_table_add(table, di,
		ltoa((long)gtk_text_get_length(GTK_TEXT(d->data))),
		1, 2, 5, 6, FALSE);
	doc_info_table_add(table, di,
		(d->sb) ? ltoa((long)d->sb->st_size) : UNKNOWN,
		1, 2, 6, 7, FALSE);

	tmp = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, 0, 2, 7, 8);
	di->adv_list = g_slist_prepend(di->adv_list, tmp);

	doc_info_table_add(table, di, "Full pathname:",     0, 1, 8, 9, TRUE);
	doc_info_table_add(table, di, "Inode:",             0, 1, 9, 10, TRUE);
	doc_info_table_add(table, di, "Mode:",              0, 1, 10, 11, TRUE);
	doc_info_table_add(table, di, "# of Links:",        0, 1, 11, 12, TRUE);
	doc_info_table_add(table, di, "UID:",               0, 1, 12, 13, TRUE);
	doc_info_table_add(table, di, "GID:",               0, 1, 13, 14, TRUE);
	doc_info_table_add(table, di, "Last accessed:",     0, 1, 14, 15, TRUE);
	doc_info_table_add(table, di, "Last modified:",     0, 1, 15, 16, TRUE);
	doc_info_table_add(table, di, "Last status change:",0, 1, 16, 17, TRUE);
	doc_info_table_add(table, di, "File block size:",   0, 1, 17, 18, TRUE);
	doc_info_table_add(table, di, "Blocks allocated:",  0, 1, 18, 19, TRUE);

	s = file_full_pathname_make(d->fname);
	doc_info_table_add(table, di, s, 1, 2, 8, 9, TRUE);
	g_free(s);
	if (d->sb) {
		doc_info_table_add(table, di, ltoa((long)d->sb->st_ino),
				   1, 2, 9, 10, TRUE);
		doc_info_table_add(table, di, file_perm_string(d->sb),
				   1, 2, 10, 11, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_nlink),
				   1, 2, 11, 12, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_uid),
				   1, 2, 12, 13, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_gid),
				   1, 2, 13, 14, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_atime, FALSE),
				   1, 2, 14, 15, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_mtime, FALSE),
				   1, 2, 15, 16, TRUE);
		doc_info_table_add(table, di, ttoa(d->sb->st_ctime, FALSE),
				   1, 2, 16, 17, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_blksize),
				   1, 2, 17, 18, TRUE);
		doc_info_table_add(table, di, ltoa((long)d->sb->st_blocks),
				   1, 2, 18, 19, TRUE);
	} else {
		int i;
		for (i = 9; i <= 18; i++)
			doc_info_table_add(
				table, di, UNKNOWN, 1, 2, i, i + 1, TRUE);
	}

	/* buttons */
	di->adv_button = gtk_button_new_with_label("Advanced");
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(di->toplev)->action_area),
			   di->adv_button, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(di->adv_button), "clicked",
			   GTK_SIGNAL_FUNC(doc_info_advanced), di);

	tmp = gtk_button_new_with_label(BUTTON_CLOSE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(di->toplev)->action_area),
			   tmp, TRUE, TRUE, 0);
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			   GTK_SIGNAL_FUNC(doc_info_close), d);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	gtk_widget_show_all(GTK_DIALOG(di->toplev)->action_area);
	gtk_widget_show(di->toplev);
} /* doc_info_cb */


/*
 * PUBLIC: doc_info_init
 *
 * creates the document info button
 */
void
doc_info_init(win_t *w, GtkWidget *parent)
{
	w->docinfo = gtk_button_new();
	w->docinfo_label = gtk_label_new(" File Info ");
	gtk_misc_set_alignment(GTK_MISC(w->docinfo_label), 0.5, 0.5);
	gtk_container_add(GTK_CONTAINER(w->docinfo), w->docinfo_label);
#if 0
	w->docinfo = gtk_button_new_with_label("File Info");
#endif

	gtk_box_pack_end(GTK_BOX(parent), w->docinfo, FALSE, FALSE, 0);
	gtk_signal_connect(GTK_OBJECT(w->docinfo), "clicked",
			GTK_SIGNAL_FUNC(doc_info_cb), w);

	/* set according to preferences */
	if (IS_SHOW_MSGBAR())
		gtk_widget_show_all(w->docinfo);

} /* docinfo_init */


/*
 * PUBLIC: doc_info_label_update
 *
 * updates the file info button's label to reflect readonly and modified status
 * of the current document.
 */
void
doc_info_label_update(win_t *w)
{
	doc_t *d;
	char *buf;
	bool_t ro;

	d = doc_current(w);
	buf = (char *)g_malloc(strlen(" File Info ") + 7);
	strcpy(buf, " File Info ");

	ro = file_is_readonly(d->fname);

#ifdef APP_GNP
	if (d->changed || ro) {
		strcat(buf, " (");
		if (d->changed)
			strcat(buf, "*");
		if (ro)
			strcat(buf, "R");
		strcat(buf, ")");
#endif
		strcat(buf, ")");
	}

	gtk_label_set(GTK_LABEL(w->docinfo_label), buf);
	g_free(buf);
} /* doc_info_label_update */


/*
 * PUBLIC: doc_list_add
 *
 * appends an entry to the files list window.
 */
void
doc_list_add(win_t *w, doc_t *d, int rownum, char *text, int which)
{
	char numstr[4], sizestr[16];
	char *rownfo[FLW_NUM_COLUMNS];
	GtkCList *clist;
	int i;
	
	g_assert(w != NULL);
	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);

	/*
	 * currently, when there are no documents opened, we always create a
	 * new document called "Untitled".  if all we have "open" right now is
	 * the "Untitled" doc, we need to remove it before appending the new
	 * entry to the list.  (alternatively, we could also just set the
	 * clist field to the new column info too.)
	 */
	if (rownum == 0 && text != NULL) {
		/*
		 * this variable shouldn't need to be static/global (AFAIK),
		 * but i get a segv on my Linux box at home if i do NOT have
		 * compiler optimization present.  yet, this works fine on
		 * my Digital Unix box at work.  there's either something
		 * screwy with GTK or my Linux box (which is a little dated:
		 * a Debian 1.2 system with gcc 2.7.2.1 and libc5).
		 */
		static char *t;
		gtk_clist_get_text(clist, rownum, FnameCol, &t);
		if (t != NULL && strcmp(t, UNTITLED) == 0) {
			gtk_clist_freeze(clist);
			gtk_clist_remove(clist, rownum);
			gtk_clist_thaw(clist);
		}
	}

	g_snprintf(numstr, 4, "%d", rownum + 1);
	rownfo[0] = numstr;
	if (d->sb == NULL)
		rownfo[1] = UNKNOWN;
	else {
		g_snprintf(sizestr, 16, "%8lu ", (gulong)(d->sb->st_size));
		rownfo[1] = sizestr;
	}

	if (text && strlen(text) > 0) {
		rownfo[2] =  text;
	} else {
		rownfo[1] = UNKNOWN;
		rownfo[2] = UNTITLED;
	}
	gtk_clist_freeze(clist);
	if (which == INSERT)
		gtk_clist_insert(clist, rownum, rownfo);
	else
		gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, rownum, FnumCol);

	/* renumber any existing entries */
	if (clist->rows > 1) {
		for (i = 1; i < clist->rows; i++) {
			g_snprintf(numstr, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, numstr);
		}
	}

	gtk_clist_thaw(clist);
} /* doc_list_add */


/*
 * PUBLIC: doc_list_destroy
 *
 * zap!
 */
void
doc_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (w->dlw) {
		gtk_widget_destroy(w->dlw);
		w->dlw = NULL;
		w->dlw_data = NULL;
		w->dlw_tb = NULL;
#ifndef GTK_HAVE_FEATURES_1_1_0
		w->dlw_tb_h = NULL;
#endif
	}
} /* doc_list_destroy */


/*
 * PUBLIC: doc_list_remove
 *
 * removes an entry from the files list window.
 */
void
doc_list_remove(win_t *w, int num)
{
	char buf[4];
	int i;
	GtkCList *clist;
	
	g_assert(w != NULL);
	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);
	g_assert(clist != NULL);
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, num);

	/* renumber any entries starting with the row just removed */
	if (clist->rows > 1) {
		for (i = num; i < clist->rows; i++) {
			g_snprintf(buf, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, buf);
		}
	}

	/* highlight the current document in the files list window */
	num = gtk_notebook_current_page(GTK_NOTEBOOK(w->nb));
	gtk_clist_select_row(clist, num, FnumCol);
	g_snprintf(buf, 4, "%d", num + 1);
	gtk_clist_set_text(clist, num, FnumCol, buf);
	gtk_clist_thaw(clist);
} /* doc_list_remove */


/*
 * PUBLIC: doc_list_show
 *
 * creates a new popup window containing a list of open files/documents
 * for the window from which it was invoked.
 */
void
doc_list_show(GtkWidget *wgt, gpointer cbdata)
{
	int num;
	GSList *dp;
	doc_t *d;
	GtkWidget *tmp, *vbox, *scrolled_win;

	char *titles[] = { " # ", " Size (bytes) ", " File name " };
	win_t *w = (win_t *)cbdata;


	g_assert(w != NULL);

	if (w->dlw) {
		gdk_window_raise(w->dlw->window);
		return;
	}

	w->dlw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(w->dlw), "Currently Open Files");
	gtk_signal_connect(GTK_OBJECT(w->dlw), "destroy",
			   GTK_SIGNAL_FUNC(doc_list_destroy), cbdata);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(w->dlw), vbox);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);
	gtk_widget_show(vbox);

	w->dlw_data = gtk_clist_new_with_titles(FLW_NUM_COLUMNS, titles);
	gtk_widget_show(w->dlw_data);
	tmp = w->dlw_data;
	gtk_clist_column_titles_passive(GTK_CLIST(tmp));
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FnumCol, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FsizeCol, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tmp), FnameCol, TRUE);
	gtk_clist_set_column_min_width(GTK_CLIST(tmp), FnameCol, 64);
#else
	gtk_clist_set_column_width(GTK_CLIST(tmp), FnumCol, 25);
	gtk_clist_set_column_width(GTK_CLIST(tmp), FsizeCol, 90);
#endif
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled_win), tmp);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_win, TRUE, TRUE, 0);
	gtk_widget_show(scrolled_win);

	num = 0;
	dp = w->doclist;
	while (dp) {
		d = (doc_t *)(dp->data);
		doc_list_add(w, d, num, doc_basefname(d), APPEND);
		dp = dp->next;
		num++;
	}

	/*
	 * MUST connect this signal **after** adding fnames to list, else
	 * we keep changing the cur doc to the newly added file.
	 */
	gtk_signal_connect(GTK_OBJECT(w->dlw_data),
			   "select_row", GTK_SIGNAL_FUNC(doc_list_select), w);

	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, TRUE, 0);
	gtk_widget_show(tmp);

	toolbar_create(w, vbox, dlw_tbdata, &(w->dlw_tb), &(w->dlw_tb_h));

	/*
	 * there should be a way to auto-detect how wide to make these widgets
	 * instead of having to use arbitrary values
	 */
	gtk_widget_set_usize(w->dlw, 250, 350);

	/*
	 * since we're currently inserting documents at the head of the
	 * notebook, the notebook list and the list here are in reverse
	 * order.  hence, manually select the actual current file.
	 */
	num = g_slist_index(w->doclist, doc_current(w));
	gtk_clist_select_row(GTK_CLIST(w->dlw_data), num, FnumCol);

	gtk_widget_show(w->dlw);
} /* doc_list_show */


/*
 * PUBLIC: doc_new
 *
 * creates a new document
 */
void
doc_new(win_t *w, char *fname, bool_t update_msgbar)
{
#ifdef APP_GNP
	GtkText *text;
#endif
	char *cwd, *fn;
	doc_t *d;
	d = (doc_t *)g_malloc0(sizeof(doc_t));
	d->docid = nextdocid++;
	d->w = w;
	d->fname = g_strdup(fname);

	w->numdoc++;

#ifdef APP_GNP
	/* text widget */
	d->data = gtk_text_new(NULL, NULL);

	/* this is a hack to get consistent tab stops for the text widget */
	text = GTK_TEXT(d->data);
	text->default_tab_width = 8;
	text->tab_stops = g_list_remove(text->tab_stops, text->tab_stops->data);
	text->tab_stops = g_list_remove(text->tab_stops, text->tab_stops->data);
	text->tab_stops = NULL;
	text->tab_stops = g_list_prepend(text->tab_stops, (void *)8);
	text->tab_stops = g_list_prepend(text->tab_stops, (void *)8);

	gtk_text_set_editable(GTK_TEXT(d->data), TRUE);
	gtk_text_set_point(GTK_TEXT(d->data), 0);
	gtk_text_set_word_wrap(GTK_TEXT(d->data), IS_USE_WORDWRAP());
	gtk_widget_show(d->data);
#ifdef USE_AUTO_INDENT
	gtk_signal_connect_after(GTK_OBJECT(d->data), "key_press_event",
				 GTK_SIGNAL_FUNC(doc_auto_indent_cb), w);
#endif
	/* quickmenu */
	gtk_signal_connect_object(GTK_OBJECT(d->data), "event",
				  GTK_SIGNAL_FUNC(quickmenu_show),
				  GTK_OBJECT(w->quickmenu));
#endif

	/* tab label */
	d->tablabel = gtk_label_new(fname);
	GTK_WIDGET_UNSET_FLAGS(d->tablabel, GTK_CAN_FOCUS);
	if (IS_SHOW_TABS())
		gtk_widget_show(d->tablabel);

	/* add to notebook, update doclist, and update winlist popup */
	doc_add_to_notebook(d);
	w->doclist = g_slist_prepend(w->doclist, d);
	w->curdoc = d;
	win_list_set_curdoc(w);
	win_list_set_numdoc(w);

#ifdef APP_GNP
	/* now open the file and stuff the text into the text widget */
	gtk_text_freeze(GTK_TEXT(d->data));
	gtk_widget_realize(d->data);
	file_open_execute(w, d);
	doc_redraw(d);	/* hack alert!  it seems that even if the text
			 * properties (font, fg and bg colors) are updated, we
			 * can't use them in gtk_text_insert because the text
			 * widget itself already has the settings which were
			 * loaded at app startup.  so by putting the redraw
			 * here, we force it to load whatever the current
			 * settings are, as opposed to what was found during
			 * app startup.  sometimes, the way GTK does is just
			 * totally non-intuitive (chalk another one up to lack
			 * of documentation). */
	gtk_text_thaw(GTK_TEXT(d->data));
#endif

	/* various status updates */
	win_set_title(d);
	doc_list_add(w, d, 0, doc_basefname(d), INSERT);

	/* skip printing cwd in full pathname */
	fn = d->fname;
	if ((cwd = getcwd(NULL, 0)) != NULL) {
		if (strstr(d->fname, cwd))
			fn = d->fname + strlen(cwd) + 1;
		g_free(cwd);
	}

#ifdef APP_GNP
	/* changed */
	d->changed = FALSE;
	d->changed_id = gtk_signal_connect(GTK_OBJECT(d->data), "changed",
					   GTK_SIGNAL_FUNC(doc_changed_cb), d);
#endif

	doc_info_label_update(w);
	if (update_msgbar)
		msgbar_printf(w, "Opened %s", fn);
	msgbox_printf("opened %s", fn);
} /* doc_new */


/*
 * PUBLIC: doc_new_cb
 *
 * new document (file) callback
 */
void
doc_new_cb(GtkWidget *wgt, gpointer cbdata)
{
	doc_new((win_t *)cbdata, UNTITLED, TRUE);
} /* doc_new_cb */


/*
 * PUBLIC: doc_open_cb
 *
 * file open callback to create the file selection dialog box.
 */
void
doc_open_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	if (w->filesel == NULL)
		doc_open_dialog_create(w, "Open File...", doc_open_ok);
	else
		gdk_window_raise(w->filesel->window);

	if (GTK_WIDGET_VISIBLE(w->filesel))
		return;

	gtk_widget_show(w->filesel);
} /* doc_open_cb */


#if defined(WANT_PROJECT) && defined(APP_GNP)
void
doc_open_prj(win_t *w)
{
	if (w->filesel == NULL)
		doc_open_dialog_create(w, "Open Project...", prj_open);
	else
		gdk_window_raise(w->filesel->window);

	if (GTK_WIDGET_VISIBLE(w->filesel))
		return;

	gtk_widget_show(w->filesel);
} /* doc_open_prj */
#endif	/* WANT_PROJECT */


static void
doc_open_dialog_create(win_t *w, char *title,
	void (*open_ok)(GtkWidget *, gpointer))
{
	w->filesel = gtk_file_selection_new(title);
	gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(w->filesel));
	gtk_signal_connect(GTK_OBJECT(w->filesel), "destroy",
			(GtkSignalFunc)doc_open_filesel_destroy, w);
	gtk_signal_connect(GTK_OBJECT(
			GTK_FILE_SELECTION(w->filesel)->ok_button),
			"clicked", (GtkSignalFunc)open_ok, w);
	gtk_signal_connect(GTK_OBJECT(
			GTK_FILE_SELECTION(w->filesel)->cancel_button),
			"clicked", (GtkSignalFunc)doc_open_filesel_destroy, w);
} /* doc_open_dialog_create */


/*
 * PUBLIC: doc_open_in_new_win
 *
 * closes current doc and opens it in a new window.  it might be slower this
 * way because we have to reread the file from disk again, but there's no easy
 * way around reparenting all the widgets associated with a doc_t structure.
 * look at it this way: at least we're not copying the entire contents of the
 * text buffer and consuming a boatload of memory (this braindead method is
 * what's used in gedit).
 */
void
doc_open_in_new_win(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	win_t *neww;
	doc_t *d;
	char *fname;

	g_assert(w != NULL);
	d = doc_current(w);
#ifdef APP_GNP
	if (!d->changed && strcmp(d->fname, UNTITLED) == 0)
#endif
		return;

	fname = g_strdup(d->fname);
	if (!doc_close_common(w)) {
		g_free(fname);
		return;
	}
	if (w->numdoc < 1)
		doc_new(w, UNTITLED, TRUE);
	recent_list_write(w);

	neww = win_new();
	doc_new(neww, fname, TRUE);
	recent_list_add(neww, fname);
	g_free(fname);

	gtk_widget_show(neww->toplev);

	d = doc_current(neww);
	recent_list_add(neww, d->fname);

	msgbar_printf(neww, "Opened %s", doc_basefname(d));
	msgbox_printf("opened %s", doc_basefname(d));
} /* doc_open_in_new_win */


#ifdef APP_GNP
/*
 * PUBLIC: doc_redraw
 *
 * redraws the widgets for a text document
 */
void
doc_redraw(void *data)
{
	doc_t *dp = (doc_t *)data;

	prefs_update_text_widget_style(dp->data);
} /* doc_redraw */
#endif


#ifdef APP_GNP
/*
 * PUBLIC: doc_save_cb
 *
 * file save callback
 */
void
doc_save_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)doc_save_common((win_t *)cbdata);
} /* doc_save_cb */


/*
 * PUBLIC: doc_save_common
 *
 * common routine for saving a document.  called from the menu callback as
 * well as get_filename(), when saving a file before printing.
 */
bool_t
doc_save_common(win_t *w)
{
	doc_t *d;
	bool_t ret;
	int num;

	g_assert(w != NULL);
	d = doc_current(w);

	if (strcmp(d->fname, UNTITLED) == 0)
		ret = doc_saveas_common(w);
	else
		ret = file_save_execute(d, FALSE);

	if (ret) {
		num = g_slist_index(w->doclist, doc_current(w));
		doc_list_update(w, d, num, d->fname); 
		doc_info_label_update(w);
	}

	recent_list_write(w);

	return ret;
} /* doc_save_common */


/*
 * PUBLIC: doc_saveas_cb
 *
 * file save-as callback
 */
void
doc_saveas_cb(GtkWidget *wgt, gpointer cbdata)
{
	(void)doc_saveas_common((win_t *)cbdata);
} /* doc_save_cb */


/*
 * PUBLIC: doc_saveas_common
 *
 * common routine for doing a document save-as.  called from the menu callback
 * as well as get_filename(), when saving a file before printing.
 */
bool_t
doc_saveas_common(win_t *w)
{
	GtkWidget *tmp;
	doc_t *d;
	guint last_id;

	if (w->saveas != NULL)
		gdk_window_raise(w->saveas->window);

	/* allow only one saveas box.  wait until that one goes away */
	while (w->saveas != NULL)
		gtk_main_iteration_do(TRUE);

	d = doc_current(w);
	last_id = d->changed_id;

	w->saveas = gtk_file_selection_new("Save As...");
	tmp = w->saveas;
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(tmp)->ok_button),
			   "clicked", (GtkSignalFunc)doc_saveas_ok, w);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(tmp)->cancel_button),
			   "clicked", (GtkSignalFunc)doc_saveas_destroy, w);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(tmp)->cancel_button),
			   "destroy", (GtkSignalFunc)doc_saveas_destroy, w);

	gtk_widget_show(w->saveas);

	while (w->saveas != NULL)
		gtk_main_iteration_do(TRUE);

	return (d->changed_id != last_id);
} /* doc_saveas_common */
#endif


/*** local function definitions ***/
/*
 * PRIVATE: doc_add_to_notebook
 *
 * add a newly created document to the notebook
 */
static void
doc_add_to_notebook(doc_t *d)
{
#ifdef APP_GNP
	GtkWidget *vsb, *vbox2;
#endif
	GtkWidget *table, *vbox1;
	GtkStyle *style;

	vbox1 = gtk_vbox_new(TRUE, TRUE);
	gtk_widget_show(vbox1);

	table = gtk_table_new(2, 2, FALSE);
	gtk_table_set_row_spacing(GTK_TABLE(table), 0, 2);
	gtk_table_set_col_spacing(GTK_TABLE(table), 0, 2);
	gtk_box_pack_start(GTK_BOX(vbox1), table, TRUE, TRUE, 1);
	gtk_widget_show(table);

	gtk_table_attach_defaults(GTK_TABLE(table), d->data, 0, 1, 0, 1);

	style = gtk_style_new();
	gtk_widget_set_style(GTK_WIDGET(d->data), style);
	gtk_widget_set_rc_style(GTK_WIDGET(d->data));
	gtk_widget_ensure_style(GTK_WIDGET(d->data));

#ifdef APP_GNP
	vbox2 = gtk_vbox_new(TRUE, TRUE);
	gtk_widget_show(vbox2);

	vsb = gtk_vscrollbar_new(GTK_TEXT(d->data)->vadj);
	gtk_box_pack_start(GTK_BOX(vbox2), vsb, TRUE, TRUE, 0);
	gtk_table_attach(GTK_TABLE(table), vbox2, 1, 2, 0, 1,
#ifdef GTK_HAVE_FEATURES_1_1_0
			 (GtkAttachOptions)GTK_FILL,
			 (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
#else
			 GTK_FILL, GTK_EXPAND | GTK_FILL,
#endif
			 0, 0);
	GTK_WIDGET_UNSET_FLAGS(vsb, GTK_CAN_FOCUS);
	gtk_widget_show(vsb);
#endif

	/*
	 * if there are a lot of files open, appending is really slow since
	 * it traverses the child list from start to end.  until this changes
	 * (e.g., the notebook widget keeps a pointer to the last child so
	 * appending is fast), we'll prepend documents.  i know this looks
	 * kind of weird, but try opening a couple hundred files and see
	 * what happens.
	 *
	 * gtk_notebook_append_page(GTK_NOTEBOOK(d->w->notebook), vbox1,
	 * 	d->tablabel);
	 */
	gtk_notebook_insert_page(GTK_NOTEBOOK(d->w->nb), vbox1, d->tablabel, 0);

	gtk_notebook_set_page(GTK_NOTEBOOK(d->w->nb), 0);

	gtk_widget_grab_focus(d->data);
} /* doc_add_to_notebook */


#ifdef USE_AUTO_INDENT
/*
 * PRIVATE: doc_auto_indent_cb
 *
 * auto-indent callback.  this routine was taken straight from gedit.
 * is there a better way to do this?
 */
static gboolean
doc_auto_indent_cb(GtkWidget *txt, GdkEventKey *event, win_t *w)
{
	int i, newlines, newline_1 = 0;
	char *buffer, *whitespace;

	/*
	 * if we return TRUE, we're saying that this routine has processed the
	 * keystroke entered.  hence, menu accelerators will not work.
	 *
	 * if we return FALSE, that means we did NOT process the keystroke,
	 * and the event key will be passed to another widget because we have
	 * captured ALL keystroke events.  this results in the default key
	 * bindings for the text widget (e.g., the arrow keys, home, end,
	 * insert, et al) to not take any effect on the text widget.
	 *
	 * basically, what i'm saying is that auto-indentation is a feature
	 * that should be handled in the text widget's event key routine.
	 * in this case, it's gtk_text_key_press().  given that, i elect to
	 * not include auto-indentation as a default 'feature'.
	 */
	if (!IS_AUTO_INDENT() || event->keyval != GDK_Return)
		return TRUE;

	newlines = 0;
	for (i = GTK_EDITABLE(txt)->current_pos; i > 0; i--) {
		buffer = gtk_editable_get_chars(GTK_EDITABLE(txt), i - 1, i);
		if (buffer == NULL)
			continue;

		if (buffer[0] == 10) {
			if (newlines > 0) {
				g_free(buffer);
				break;
			} else {
				newlines++;
				newline_1 = i;
				g_free(buffer);
			}
		}
	}

	whitespace = g_malloc0(newline_1 - i + 2);

	for (i = i; i <= newline_1; i++) {
		buffer = gtk_editable_get_chars(GTK_EDITABLE(txt), i, i + 1);
		if ((buffer[0] != 32) & (buffer[0] != 9))
			break;
		strncat(whitespace, buffer, 1);
		g_free(buffer);
	}

	if (strlen(whitespace) > 0) {
		i = GTK_EDITABLE(txt)->current_pos;
		gtk_editable_insert_text(GTK_EDITABLE(txt), whitespace,
					 strlen(whitespace), &i);
	}
	g_free(whitespace);

	return TRUE;
} /* doc_auto_indent_cb */
#endif


/*
 * PRIVATE: doc_close_common
 *
 * common routine for closing a document.
 */
static bool_t
doc_close_common(win_t *w)
{
	doc_t *d;

	g_assert(w != NULL);
	d = doc_current(w);

#ifdef APP_GNP
	if (d->changed)
		return doc_close_verify(w);
#endif

	return doc_close_execute(w);
} /* doc_close_common */


/*
 * PRIVATE: doc_close_execute
 *
 * actually close the document (remove from the notebook, remove from the
 * doclist, etc.  always return TRUE.
 */
static bool_t
doc_close_execute(win_t *w)
{
	doc_t *d;

	d = doc_remove(w);
	g_assert(d != NULL);
	doc_free(d);

	return TRUE;
} /* doc_close_execute */


#ifdef APP_GNP
/*
 * PRIVATE: doc_close_verify
 *
 * creates popup verify box to ask user whether or not to save a file before
 * closing.
 */
#define CLOSE_TITLE     "Save File"
#define CLOSE_MSG       "has been modified. \n Do you wish to save it?"
static bool_t
doc_close_verify(win_t *w)
{
	int ret;
	char *title, *msg;
	doc_t *d = doc_current(w);

	title = (char *)g_malloc(strlen(CLOSE_TITLE) + strlen(d->fname) + 5);
	msg =   (char *)g_malloc(strlen(CLOSE_MSG) + strlen(d->fname) + 6);
	sprintf(title, "%s '%s'?", CLOSE_TITLE, d->fname);
	sprintf(msg  , " '%s' %s ", d->fname, CLOSE_MSG);

	ret = do_dialog_yes_no_cancel(title, msg);

	g_free(title);
	g_free(msg);

	switch (ret) {
	case DIALOG_YES:
		return doc_close_verify_yes(w);
	case DIALOG_NO:
		return doc_close_execute(w);
	case DIALOG_CANCEL:
		break;
	default:
		printf("doc_close_verify: do_dialog returned %d\n", ret);
		exit(-1);
	} /* switch */

	return FALSE;
} /* doc_close_verify */


/*
 * PRIVATE: doc_close_verify_yes
 *
 * callback invoked when user selected "Yes" to save a file before closing
 */
static bool_t
doc_close_verify_yes(win_t *w)
{
	bool_t saved;
	doc_t *d = doc_current(w);

	if (strcmp(d->fname, UNTITLED) == 0) {
		doc_saveas_cb(NULL, w);
		if (d->changed == FALSE)
			return doc_close_execute(w);
	} else {
		saved = file_save_execute(d, FALSE);
		if (saved)
			return doc_close_execute(w);
	}

	return FALSE;
} /* doc_close_verify_yes */
#endif


/*
 * PRIVATE: doc_free
 *
 * free space used by a doc_t.  currently, only used after doc_remove(),
 * so there is some g_slist magic happening.
 */
static void
doc_free(doc_t *d)
{
	doc_info_t *di = (doc_info_t *)(d->docinfo);

	if (di)
		doc_info_close(NULL, d);

	/*
	 * oddly enough, when a glist item is removed, it gets put on this
	 * "free list".  my guess is that it gets recycled somehow.  the main
	 * point is that in the process, it appears that i don't have to
	 * free any memory.  i find this incredulous, which is why i'm keeping
	 * this code snippet (and comment) here as a reminder.
	 */
	if (d->fname)
		g_free(d->fname);
	if (d->sb)
		g_free(d->sb);
	if (d->printwin)
		gtk_widget_destroy(d->printwin);
	if (d->printentry)
		gtk_widget_destroy(d->printentry);
#if GLIST_MAGIC
	if (d->data)
		gtk_widget_destroy(d->data);
	if (d->tablabel)
		gtk_widget_destroy(d->tablabel);
	g_free(d);
#endif
} /* doc_free */


/*
 * PRIVATE: doc_info_advanced
 *
 * callback invoked when user clicks on "Advanced" button in the doc info
 * popup window.  traverses the list of widgets not shown and shows them.
 */
static void
doc_info_advanced(GtkWidget *wgt, gpointer cbdata)
{
	doc_info_t *di = (doc_info_t *)cbdata;
	GSList *alp;
	GtkWidget *wp;

	g_assert(di != NULL);
	alp = di->adv_list;
	while (alp) {
		wp = (GtkWidget *)(alp->data);
		gtk_widget_show(wp);
		alp = alp->next;
	}

	gtk_widget_destroy(di->adv_button);
	gtk_window_set_title(GTK_WINDOW(di->toplev),
			     "Advanced File Information");
} /* doc_info_advanced */


/*
 * PRIVATE: doc_info_close
 *
 * callback for "Close" button on doc info window.
 */
static void
doc_info_close(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	doc_info_t *di;

	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	gtk_widget_destroy(di->toplev);	/* goes to doc_info_destroy */
} /* doc_info_close */


/*
 * PRIVATE: doc_info_destroy
 *
 * callback for the "destroy" event on the top level doc info window.
 */
static void
doc_info_destroy(GtkWidget *wgt, gpointer cbdata)
{
	doc_t *d = (doc_t *)cbdata;
	doc_info_t *di;
	GSList *alp;
	GtkWidget *wp;

	g_assert(d != NULL);
	di = (doc_info_t *)(d->docinfo);
	alp = di->adv_list;
	while (alp) {
		wp = (GtkWidget *)(alp->data);
		gtk_widget_destroy(wp);
		alp = alp->next;
	}
	g_slist_free(di->adv_list);
	g_free(di);
	d->docinfo = NULL;
} /* doc_info_destroy */


/*
 * PRIVATE: doc_info_table_add
 *
 * adds new entry into table.  since we keep all the widgets in one table
 * (so that alignment is consistent), we build a list of widgets representing
 * the "advanced" file infomation.  these widgets are *not* shown initially.
 */
static void
doc_info_table_add(GtkWidget *table, doc_info_t *di, char *text,
	int lft, int rht, int top, int bot, bool_t advanced)
{
	GtkWidget *tmp;

	tmp = gtk_label_new(text);
	gtk_misc_set_alignment(GTK_MISC(tmp), (rht % 2), 0.5);
	gtk_table_attach_defaults(GTK_TABLE(table), tmp, lft, rht, top, bot);

	if (advanced)
		di->adv_list = g_slist_prepend(di->adv_list, tmp);
	else
		gtk_widget_show(tmp);
} /* doc_info_table_add */


/*
 * PRIVATE: doc_list_select
 *
 * callback routine when selecting a file in the files list window.
 * effectively, sets the notebook to show the selected file.
 * see select_row() in gtkclist.c to match function parameters.
 */
static int
doc_list_select(GtkWidget *wgt, int row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	/*
	 * note: gtk_notebook_set_page() emits a switch_page signal.
	 * consequently, we do not have to manually update our curdoc
	 * pointer in the win_t field at this point since it's down
	 * in our switch_page callback (see notebook_switch_page()).
	 */
	gtk_notebook_set_page(GTK_NOTEBOOK(w->nb), row);

	return TRUE;
} /* doc_list_select */


/*
 * PRIVATE: doc_list_update
 *
 * currently only called from doc_save_common(), this routine updates the
 * clist entry after saving a file.  this is necessary because the filename
 * could have changed, and most likely, the file size would have changed
 * as well.
 */
static void
doc_list_update(win_t *w, doc_t *d, int rownum, char *text)
{
	GtkCList *clist;
	char buf[16];
	
	g_assert(w != NULL);

	if (w->dlw == NULL)
		return;

	clist = GTK_CLIST(w->dlw_data);

	if (d->sb == NULL) {
		d->sb = (struct stat *)g_malloc(sizeof(struct stat));
		if (stat(text, d->sb) == -1) {
			/* print warning */
			gtk_clist_set_text(clist, rownum, FsizeCol, UNKNOWN);
			g_free(d->sb);
			d->sb = NULL;
		} else {
			g_snprintf(buf, 16, "%8lu ", (gulong)(d->sb->st_size));
			gtk_clist_set_text(clist, rownum, FsizeCol, buf);
		}
	} else {
		g_snprintf(buf, 16, "%8lu ", (gulong)(d->sb->st_size));
		gtk_clist_set_text(clist, rownum, FsizeCol, buf);
	}
	gtk_clist_set_text(clist, rownum, FnameCol, text);
} /* doc_list_update */


/*
 * PRIVATE: doc_open_filesel_destroy
 *
 * a destroy signal was sent from the window manager.  we need to set the
 * filesel field to NULL so we know we need to recreate it next time.
 */
static void
doc_open_filesel_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	gtk_widget_destroy(w->filesel);
	w->filesel = NULL;
} /* doc_open_destroy */


/*
 * PRIVATE: doc_open_ok
 *
 * callback invoked when user has clicked the "Ok" button from the file
 * selector for opening a file
 */
static void
doc_open_ok(GtkWidget *wgt, gpointer cbdata)
{
	char *fname;
	struct stat sb;
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(w->filesel));
	if (fname != NULL) {
		if (stat(fname, &sb) == -1)
			goto out;	/* XXX - need to print warning */

		if (S_ISDIR(sb.st_mode)) {
			gtk_file_selection_set_filename(
				GTK_FILE_SELECTION(w->filesel), fname);
			return;
		}

		/*
		 * if there's only one document, and it's an empty "Untitled"
		 * doc, then remove it before adding the one we want
		 */
		d = doc_current(w);
		if (w->numdoc == 1 &&
#ifdef APP_GNP
		    !d->changed &&
#endif
		    strcmp(d->fname, UNTITLED) == 0) {
			doc_close_common(w);
		}

		doc_new(w, fname, TRUE);
		recent_list_add(w, fname);
	} /* fname != NULL */

out:
	doc_open_filesel_destroy(NULL, w);
} /* doc_open_ok */


/*
 * PRIVATE: doc_remove
 *
 * removes current document from window and returns pointer to it.
 * currently only called from doc_close_execute().
 */
static doc_t *
doc_remove(win_t *w)
{
	GSList *dlp;
	GtkNotebook *nb;
	char *cwd, *fn;
	doc_t *d;
	int num;

	d = doc_current(w);
	nb = GTK_NOTEBOOK(w->nb);
	g_assert(nb != NULL);

	/* remove from notebook */
	num = gtk_notebook_current_page(nb);
	gtk_notebook_remove_page(nb, num);

	/*
	 * remove from doclist and doc list window.  update numdoc in winlist.
	 * contents of the doc gets freed in doc_free().
	 */
	dlp = g_slist_find(w->doclist, d);
	w->doclist = g_slist_remove_link(w->doclist, dlp);
	w->numdoc--;
	doc_list_remove(w, num);
	win_list_set_numdoc(w);

	/* sanity checks */
	g_assert(w->numdoc == g_list_length(nb->children));
	g_assert(w->numdoc == g_slist_length(w->doclist));

	/* skip printing cwd in full pathname */
	fn = d->fname;
	if ((cwd = getcwd(NULL, 0)) != NULL) {
		if (strstr(d->fname, cwd))
			fn = d->fname + strlen(cwd) + 1;
		g_free(cwd);
	}

	msgbar_printf(w, "Closed %s", fn);
	msgbox_printf("closed %s", fn);

	/* update curdoc pointer */
	num = gtk_notebook_current_page(nb);
	if (num == -1) {	/* closed last doc */
		w->curdoc = NULL;
		return (doc_t *)(dlp->data);
	}

	w->curdoc = g_slist_nth_data(w->doclist, num);
	win_set_title(doc_current(w));

	return (doc_t *)(dlp->data);
} /* doc_remove */


#ifdef APP_GNP
/*
 * PRIVATE: doc_saveas_destroy
 *
 * blow away the save-as dialog
 */
static void
doc_saveas_destroy(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_widget_destroy(w->saveas);
	w->saveas = NULL;
} /* doc_saveas_destroy */
#endif


#ifdef APP_GNP
/*
 * PRIVATE: doc_saveas_ok
 *
 * callback invoked when user clicked "Ok" on the file save-as dialog.
 */
#define OW_TITLE	"Overwrite"
#define OW_MSG		"exists.  Overwrite?"
static void
doc_saveas_ok(GtkWidget *wgt, gpointer cbdata)
{
	int ret;
	doc_t *d;
	bool_t saved;
	char *fname, *title, *msg;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);
	g_assert(w->saveas != NULL);

	fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(w->saveas));
	if (fname == NULL || fname[0] == '\0')
		return;

	if (file_exist(fname)) {
		title = (char *)g_malloc(strlen(OW_TITLE) + strlen(fname) + 5);
		msg = (char *)g_malloc(strlen(OW_MSG) + strlen(fname) + 6);
		sprintf(title, "%s '%s'?", OW_TITLE, fname);
		sprintf(msg,   " '%s' %s ", fname, OW_MSG);

		ret = do_dialog_yes_no_cancel(title, msg);

		g_free(title);
		g_free(msg);

		switch (ret) {
		case DIALOG_YES:
			goto saveas_do_save;	/* ugh, just had to use goto */
			break;
		case DIALOG_NO:
			/* don't overwrite; let user decide another name */
			break;
		case DIALOG_CANCEL:
			/* canceled */
			gtk_widget_destroy(w->saveas);
			w->saveas = NULL;
			break;
		default:
			printf("doc_saveas_ok: do_dialog returnd %d\n", ret);
			exit(-1);
		} /* switch */
	} else {
saveas_do_save:
		d = doc_current(w);
		if (d->fname)
			g_free(d->fname);
		d->fname = g_strdup(fname);
		saved = file_save_execute(d, FALSE);
		if (saved) {
			gtk_widget_destroy(w->saveas);
			w->saveas = NULL;
			doc_info_label_update(w);
		}
	}
} /* doc_saveas_ok */
#endif


/* the end */
