/*
 * Program write.c - dump memory  structures to  file for iso9660 filesystem.

   Written by Eric Youngdale (1993).

   Copyright 1993 Yggdrasil Computing, Incorporated

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

static char rcsid[] ="$Id: write.c,v 1.4 1997/04/10 03:33:25 eric Rel $";

/* APPLE_HYB James Pearson j.pearson@@ps.ucl.ac.uk 19/11/97 */

/* JOLIET James Pearson j.pearson@@ps.ucl.ac.uk 31/7/97 */

#include <string.h>
#include <stdlib.h>
#include "mkisofs.h"
#include "iso9660.h"
#include <time.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
 
#ifdef __svr4__
extern char * strdup(const char *);
#endif

#ifdef VMS
extern char * strdup(const char *);
#endif


/* Max number of sectors we will write at  one time */
#define NSECT 16

/* Counters for statistics */

static int table_size = 0;
static int total_dir_size = 0;
static int rockridge_size = 0;
static struct directory ** pathlist;
static next_path_index = 1;
#ifdef JOLIET
/* Joliet versions of the above */
static int j_table_size = 0;
static int j_total_dir_size = 0;
static int j_rockridge_size = 0;
static struct directory ** j_pathlist;
static j_next_path_index = 1;
#endif /* JOLIET */

/* Used to fill in some  of the information in the volume descriptor. */
static struct tm *local;
/* JCP 13/8/97 - "local" get trashed by the iso9660_date() function,
so declare Ctime here to store the value */
static time_t Ctime;

/* Routines to actually write the disc.  We write sequentially so that
   we could write a tape, or write the disc directly */


#define FILL_SPACE(X)   memset(vol_desc.X, ' ', sizeof(vol_desc.X))
#ifdef JOLIET
#define J_FILL_SPACE(X)   memset(vol_sdesc.X, 0, sizeof(vol_sdesc.X))
#endif /* JOLIET */

void FDECL2(set_721, char *, pnt, unsigned int, i)
{
     pnt[0] = i & 0xff;
     pnt[1] = (i >> 8) &  0xff;
}

void FDECL2(set_722, char *, pnt, unsigned int, i)
{
     pnt[0] = (i >> 8) &  0xff;
     pnt[1] = i & 0xff;
}

void FDECL2(set_723, char *, pnt, unsigned int, i)
{
     pnt[3] = pnt[0] = i & 0xff;
     pnt[2] = pnt[1] = (i >> 8) &  0xff;
}

void FDECL2(set_731, char *, pnt, unsigned int, i)
{
     pnt[0] = i & 0xff;
     pnt[1] = (i >> 8) &  0xff;
     pnt[2] = (i >> 16) &  0xff;
     pnt[3] = (i >> 24) &  0xff;
}

void FDECL2(set_732, char *, pnt, unsigned int, i)
{
     pnt[3] = i & 0xff;
     pnt[2] = (i >> 8) &  0xff;
     pnt[1] = (i >> 16) &  0xff;
     pnt[0] = (i >> 24) &  0xff;
}

int FDECL1(get_733, char *, p)
{
     return ((p[0] & 0xff)
	     | ((p[1] & 0xff) << 8)
	     | ((p[2] & 0xff) << 16)
	     | ((p[3] & 0xff) << 24));
}

void FDECL2(set_733, char *, pnt, unsigned int, i)
{
     pnt[7] = pnt[0] = i & 0xff;
     pnt[6] = pnt[1] = (i >> 8) &  0xff;
     pnt[5] = pnt[2] = (i >> 16) &  0xff;
     pnt[4] = pnt[3] = (i >> 24) &  0xff;
}

void FDECL4(xfwrite, void *, buffer, int, count, int, size, FILE *, file)
{
     while(count) 
     {
	  int got = fwrite(buffer,size,count,file);

	  if(got<=0) 
	  {
	       fprintf(stderr,"cannot fwrite %d*%d\n",size,count);
	       exit(1);
	  }
	  count-=got,*(char**)&buffer+=size*got;
     }
}

struct deferred_write
{
  struct deferred_write * next;
  char			* table;
  unsigned int		  extent;
  unsigned int		  size;
  char			* name;
#ifdef APPLE_HYB
  struct directory_entry *s_entry;
  unsigned int            pad;
#endif /* APPLE_HYB */
};

static struct deferred_write * dw_head = NULL, * dw_tail = NULL;

static struct directory_entry * sort_dir;
static struct eltorito_boot_descriptor boot_desc;

unsigned int last_extent_written  =0;
static struct iso_primary_descriptor vol_desc;
static path_table_index;
#ifdef JOLIET
/* Joliet uses a supplementary volume descriptor */
static struct iso_supplementary_descriptor vol_sdesc;
static j_path_table_index;
#endif /* JOLIET */
static time_t begun;

/* We recursively walk through all of the directories and assign extent
   numbers to them.  We have already assigned extent numbers to everything that
   goes in front of them */

void FDECL1(assign_directory_addresses, struct directory *, node)
{
     int		dir_size;
     struct directory * dpnt;

     dpnt = node;
     
     while (dpnt)
     {
	  /*
	   * If we already have an extent for this (i.e. it came from
	   * a multisession disc), then don't reassign a new extent.
	   */
	  dpnt->path_index = next_path_index++;
	  if( dpnt->extent == 0 )
	  {
	       dpnt->extent = last_extent;
	       dir_size = (dpnt->size + (SECTOR_SIZE - 1)) >> 11;
	       
	       last_extent += dir_size;
	       
	       /* 
		* Leave room for the CE entries for this directory.  Keep them
		* close to the reference directory so that access will be 
		* quick. 
		*/
	       if(dpnt->ce_bytes)
	       {
		    last_extent += ROUND_UP(dpnt->ce_bytes) >> 11;
	       }
	  }

	  if(dpnt->subdir) 
	  {
	       assign_directory_addresses(dpnt->subdir);
	  }

	  dpnt = dpnt->next;
     }
}

#ifdef JOLIET
/* the address of the Joliet directories are non the same as the
   primary, but we don't have to worry about RR here */
void FDECL1(j_assign_directory_addresses, struct directory *, node)
{
     int		dir_size;
     struct directory * dpnt;

     dpnt = node;
     
     while (dpnt)
     {
	  /*
	   * If we already have an extent for this (i.e. it came from
	   * a multisession disc), then don't reassign a new extent.
	   */
	  dpnt->j_path_index = j_next_path_index++;
	  if( dpnt->j_extent == 0 )
	  {
	       dpnt->j_extent = last_extent;
	       dir_size = (dpnt->j_size + (SECTOR_SIZE - 1)) >> 11;
	       
	       last_extent += dir_size;
	  }

	  if(dpnt->subdir) 
	  {
	       j_assign_directory_addresses(dpnt->subdir);
	  }

	  dpnt = dpnt->next;
     }
}
#endif /* JOLIET */

static void FDECL3(write_one_file, char *, filename, 
		   unsigned int, size, FILE *, outfile)
{
     char		  buffer[SECTOR_SIZE * NSECT];
     FILE		* infile;
     int		  remain;
     int		  use;


     if ((infile = fopen(filename, "rb")) == NULL) 
     {
#if defined(sun) || defined(_AUX_SOURCE)
	  fprintf(stderr, "cannot open %s: (%d)\n", filename, errno);
#else
	  fprintf(stderr, "cannot open %s: %s\n", filename, strerror(errno));
#endif
	  exit(1);
     }
     remain = size;

     while(remain > 0)
     {
	  use =  (remain >  SECTOR_SIZE * NSECT - 1 ? NSECT*SECTOR_SIZE : remain);
	  use = ROUND_UP(use); /* Round up to nearest sector boundary */
	  memset(buffer, 0, use);
	  if (fread(buffer, 1, use, infile) == 0) 
	  {
	       fprintf(stderr,"cannot read from %s\n",filename); 
	       exit(1);
	  }
	  xfwrite(buffer, 1, use, outfile);
	  last_extent_written += use/SECTOR_SIZE;
#if 0
	  if((last_extent_written % 1000) < use/SECTOR_SIZE) 
	  {
	       fprintf(stderr,"%d..", last_extent_written);
	  }
#else
	  if((last_extent_written % 5000) < use/SECTOR_SIZE)
	  {
	       time_t now;
	       time_t the_end;
	       double frac;
	       
	       time(&now);
	       frac = last_extent_written / (double)last_extent;
	       the_end = begun + (now - begun) / frac;
	       fprintf(stderr, "%6.2f%% done, estimate finish %s",
		       frac * 100., ctime(&the_end));
	  }
#endif
	  remain -= use;
     }
     fclose(infile);
} /* write_one_file(... */

static void FDECL1(write_files, FILE *, outfile)
{
     struct deferred_write * dwpnt, *dwnext;
     dwpnt = dw_head;
     while(dwpnt)
     {
	  if(dwpnt->table) 
	  {
	       xfwrite(dwpnt->table,  1, ROUND_UP(dwpnt->size), outfile);
	       last_extent_written += ROUND_UP(dwpnt->size) / SECTOR_SIZE;
	       table_size += dwpnt->size;
/*		  fprintf(stderr,"Size %d ", dwpnt->size); */
	       free(dwpnt->table);
	  } 
	  else 
	  {

#ifdef VMS
	       vms_write_one_file(dwpnt->name, dwpnt->size, outfile);
#else
	       write_one_file(dwpnt->name, dwpnt->size, outfile);
#endif
	       free(dwpnt->name);
	  }

#ifdef APPLE_HYB
	  if (apple_hyb)
	  {
		/* we may have to pad out ISO files to work with 
		   HFS clump sizes */
		char blk[SECTOR_SIZE];
		int i;

		for(i=0;i<dwpnt->pad;i++)
		    xfwrite(blk, 1, SECTOR_SIZE, outfile);

		last_extent_written += dwpnt->pad;
	  }
#endif /* APPLE_HYB */

	  dwnext = dwpnt;
	  dwpnt = dwpnt->next;
	  free(dwnext);
     }
} /* write_files(... */

#if 0
static void dump_filelist()
{
     struct deferred_write * dwpnt;
     dwpnt = dw_head;
     while(dwpnt)
     {
	  fprintf(stderr, "File %s\n",dwpnt->name);
	  dwpnt = dwpnt->next;
     }
     fprintf(stderr,"\n");
}
#endif

int FDECL2(compare_dirs, const void *, rr, const void *, ll) 
{
     char * rpnt, *lpnt;
     struct directory_entry ** r, **l;
     
     r = (struct directory_entry **) rr;
     l = (struct directory_entry **) ll;
     rpnt = (*r)->isorec.name;
     lpnt = (*l)->isorec.name;

#ifdef APPLE_HYB
     /* resource fork MUST (not sure if this is true for HFS volumes) be
	before the data fork - so force it here */
     if ((*r)->assoc && (*r)->assoc == (*l))
	return 1;
#endif /* APPLE_HYB */
     
     /*
      *  Put the '.' and '..' entries on the head of the sorted list.
      *  For normal ASCII, this always happens to be the case, but out of
      *  band characters cause this not to be the case sometimes.
      */

     if( strcmp(rpnt, ".") == 0 ) return -1;
     if( strcmp(lpnt, ".") == 0 ) return  1;

     if( strcmp(rpnt, "..") == 0 ) return -1;
     if( strcmp(lpnt, "..") == 0 ) return  1;

     while(*rpnt && *lpnt) 
     {
	  if(*rpnt == ';' && *lpnt != ';') return -1;
	  if(*rpnt != ';' && *lpnt == ';') return 1;
	  
	  if(*rpnt == ';' && *lpnt == ';') return 0;
	  
	  if(*rpnt == '.' && *lpnt != '.') return -1;
	  if(*rpnt != '.' && *lpnt == '.') return 1;
	  
	  if(*rpnt < *lpnt) return -1;
	  if(*rpnt > *lpnt) return 1;
	  rpnt++;  lpnt++;
     }
     if(*rpnt) return 1;
     if(*lpnt) return -1;
     return 0;
}

void FDECL1(sort_directory, struct directory_entry **, sort_dir)
{
     int dcount = 0;
     int i, len;
     struct directory_entry * s_entry;
     struct directory_entry ** sortlist;
     
     s_entry = *sort_dir;
     while(s_entry)
     {
	  dcount++;
	  s_entry = s_entry->next;
     }

     /*
      * OK, now we know how many there are.  Build a vector for sorting. 
      */
     sortlist =   (struct directory_entry **) 
	  e_malloc(sizeof(struct directory_entry *) * dcount);

     dcount = 0;
     s_entry = *sort_dir;
     while(s_entry)
     {
	  sortlist[dcount] = s_entry;
	  len = s_entry->isorec.name_len[0];
	  s_entry->isorec.name[len] = 0;
	  dcount++;
	  s_entry = s_entry->next;
     }
  
     qsort(sortlist, dcount, sizeof(struct directory_entry *), 
	   (int (*)(const void *, const void *))compare_dirs);
     
     /* 
      * Now reassemble the linked list in the proper sorted order 
      */
     for(i=0; i<dcount-1; i++)
     {
	  sortlist[i]->next = sortlist[i+1];
#ifdef DEBUG
	  fprintf(stderr,"%d %s %s\n", i, sortlist[i]->whole_name, sortlist[i]->isorec.name);
#endif /* DEBUG */
     }

     sortlist[dcount-1]->next = NULL;
     *sort_dir = sortlist[0];
     
     free(sortlist);
     
}

#ifdef JOLIET
/* Joliet version of compare_dirs(), seems to be some redundant code
   in the original version ... */
int FDECL2(j_compare_dirs, const void *, rr, const void *, ll) 
{
     char * rpnt, *lpnt;
     struct directory_entry ** r, **l;
     int 	rlen, llen;
     
     r = (struct directory_entry **) rr;
     l = (struct directory_entry **) ll;
     rpnt = (*r)->j_isorec.name;
     lpnt = (*l)->j_isorec.name;

     rlen = (*r)->j_isorec.name_len[0];
     llen = (*l)->j_isorec.name_len[0];

     /*
      *  Put the '.' and '..' entries on the head of the sorted list.
      *  For normal ASCII, this always happens to be the case, but out of
      *  band characters cause this not to be the case sometimes.
      */
/*
	Not sure why "." and ".." are checked here, as (j_)isorec.name
	never has "." and ".." ...

     if( strcmp(rpnt, ".") == 0 ) return -1;
     if( strcmp(lpnt, ".") == 0 ) return  1;

     if( strcmp(rpnt, "..") == 0 ) return -1;
     if( strcmp(lpnt, "..") == 0 ) return  1;
*/
     if (*rpnt == 0 && rlen == 1) return -1;	/* "." */
     if (*lpnt == 0 && llen == 1) return  1;

     if (*rpnt == 1 && rlen == 1) return -1;
     if (*lpnt == 1 && llen == 1) return  1;	/* ".." */

     /* skip over first null */
     rpnt++;  lpnt++;
     
     while(*rpnt && *lpnt) 
     {
	  if(*rpnt == ';' && *lpnt != ';') return -1;
	  if(*rpnt != ';' && *lpnt == ';') return 1;
	  
	  if(*rpnt == ';' && *lpnt == ';') return 0;

	  /* we don't worry about "." in Joliet filenames */
	  
	  if(*rpnt < *lpnt) return -1;
	  if(*rpnt > *lpnt) return 1;
	  rpnt+=2;  lpnt+=2;
     }
     if(*rpnt) return 1;
     if(*lpnt) return -1;
     return 0;
}

/* Joliet version of sort_directory() */
void FDECL1(j_sort_directory, struct directory_entry **, sort_dir)
{
     int dcount = 0;
     int i, len;
     struct directory_entry * s_entry;
     struct directory_entry ** sortlist;
     
     s_entry = *sort_dir;
     while(s_entry)
     {
	  dcount++;
	  s_entry = s_entry->next;
     }

     /*
      * OK, now we know how many there are.  Build a vector for sorting. 
      */
     sortlist =   (struct directory_entry **) 
	  e_malloc(sizeof(struct directory_entry *) * dcount);

     dcount = 0;
     s_entry = *sort_dir;
     while(s_entry)
     {
	  sortlist[dcount] = s_entry;
	  /* end of j_isorec already set to NULL */
	  dcount++;
	  s_entry = s_entry->next;
     }
  
     qsort(sortlist, dcount, sizeof(struct directory_entry *), 
	   (int (*)(const void *, const void *))j_compare_dirs);
     
     /* 
      * Now reassemble the linked list in the proper sorted order 
      */
     for(i=0; i<dcount-1; i++)
     {
	  sortlist[i]->j_next = sortlist[i+1];
#ifdef DEBUG
	  fprintf(stderr,"%d %s\n", i, sortlist[i]->whole_name);
#endif /* DEBUG */
     }

     sortlist[dcount-1]->j_next = NULL;
     *sort_dir = sortlist[0];
     
     free(sortlist);
     
}
#endif /* JOLIET */

void generate_root_record()
{
/*   time_t ctime; */
     
     time (&Ctime);
     local = localtime(&Ctime);
     
     root_record.length[0] = 1 + sizeof(struct iso_directory_record)
	  - sizeof(root_record.name);
     root_record.ext_attr_length[0] = 0;
     set_733((char *) root_record.extent, root->extent);
     set_733((char *) root_record.size, ROUND_UP(root->size));
     iso9660_date(root_record.date, Ctime);
     root_record.flags[0] = 2;
     root_record.file_unit_size[0] = 0;
     root_record.interleave[0] = 0;
     set_723(root_record.volume_sequence_number, DEF_VSN);
     root_record.name_len[0] = 1;
}
#ifdef JOLIET
/* Joliet version of generate_root_record() */
void j_generate_root_record()
{
/*   time_t ctime;
     
     time (&ctime);
     local = localtime(&ctime);
*/
     j_root_record.length[0] = 1 + sizeof(struct iso_directory_record)
	  - sizeof(j_root_record.name);
     j_root_record.ext_attr_length[0] = 0;
     set_733((char *) j_root_record.extent, root->j_extent);
     set_733((char *) j_root_record.size, ROUND_UP(root->j_size));
     iso9660_date(j_root_record.date, Ctime);
     j_root_record.flags[0] = 2;
     j_root_record.file_unit_size[0] = 0;
     j_root_record.interleave[0] = 0;
     set_723(j_root_record.volume_sequence_number, DEF_VSN);
     j_root_record.name_len[0] = 1;
}
#endif /* JOLIET */

static void FDECL1(assign_file_addresses, struct directory *, dpnt)
{
     struct directory * finddir;
     struct directory_entry * s_entry;
     struct file_hash *s_hash;
     struct deferred_write * dwpnt;
     char whole_path[1024];
     
     while (dpnt)
     {
	  s_entry = dpnt->contents;
	  for(s_entry = dpnt->contents; s_entry; s_entry = s_entry->next)
	  {
	       
	       /*
		* If we already have an  extent for this entry,
		* then don't assign a new one.  It must have come
		* from a previous session on the disc.  Note that
		* we don't end up scheduling the thing for writing
		* either.
		*/
	       if( isonum_733(s_entry->isorec.extent) != 0 )
	       {
		    continue;
	       }
	       
	       /* 
		* This saves some space if there are symlinks present 
		*/
	       s_hash = find_hash(s_entry->dev, s_entry->inode);
	       if(s_hash)
	       {
#ifdef APPLE_HYB
		    if (apple_hyb && s_entry->isorec.flags[0] != 2) {
			fprintf(stderr,"can't have muliple linked files in HFS volume: %s\n", s_entry->whole_name);
			exit (-1);
		    }
#endif /* APPLE_HYB */
		    if(verbose)
		    {
			 fprintf(stderr, "Cache hit for %s%s%s\n",s_entry->filedir->de_name, 
				 SPATH_SEPARATOR, s_entry->name);
		    }
		    set_733((char *) s_entry->isorec.extent, s_hash->starting_block);
		    set_733((char *) s_entry->isorec.size, s_hash->size);
#ifdef JOLIET
		    if (use_Joliet) {
		      set_733((char *) s_entry->j_isorec.extent, s_hash->j_starting_block);
		      set_733((char *) s_entry->j_isorec.size, s_hash->j_size);
		    }
#endif /* JOLIET */
		    continue;
	       }

	       /*
		* If this is for a directory that is not a . or a .. entry, 
		* then look up the information for the entry.  We have already
		* assigned extents for directories, so we just need to
		* fill in the blanks here.
		*/
	       if (strcmp(s_entry->name,".") && strcmp(s_entry->name,"..") && 
		   s_entry->isorec.flags[0] == 2)
	       {
		    finddir = dpnt->subdir;
		    while(1==1)
		    {
			 if(finddir->self == s_entry) break;
			 finddir = finddir->next;
			 if(!finddir) 
			 {
			      fprintf(stderr,"Fatal goof\n"); exit(1);
			 }
		    }
		    set_733((char *) s_entry->isorec.extent, finddir->extent);
		    s_entry->starting_block = finddir->extent;
		    s_entry->size = ROUND_UP(finddir->size);
		    total_dir_size += s_entry->size;
		    set_733((char *) s_entry->isorec.size, ROUND_UP(finddir->size));
#ifdef JOLIET
		    if (use_Joliet) {
		      set_733((char *) s_entry->j_isorec.extent, finddir->j_extent);
		      s_entry->j_starting_block = finddir->j_extent;
		      s_entry->j_size = ROUND_UP(finddir->j_size);
		      set_733((char *) s_entry->j_isorec.size, ROUND_UP(finddir->j_size));
		    }
#endif /* JOLIET */
		    add_hash(s_entry);
		    continue;
	       }


	       /*
		* If this is . or .., then look up the relevant info from the
		* tables.
		*/
	       if(strcmp(s_entry->name,".") == 0) 
	       {
		    set_733((char *) s_entry->isorec.extent, dpnt->extent);
#ifdef JOLIET
		    if (use_Joliet) {
		      set_733((char *) s_entry->j_isorec.extent, dpnt->j_extent);
		      set_733((char *) s_entry->j_isorec.size, ROUND_UP(dpnt->j_size));
		      s_entry->j_starting_block = dpnt->j_extent;
		      s_entry->j_size = ROUND_UP(dpnt->j_size);
		    }
#endif /* JOLIET */
		    
		    /* 
		     * Set these so that the hash table has the
		     * correct information
		     */
		    s_entry->starting_block = dpnt->extent;
		    s_entry->size = ROUND_UP(dpnt->size);
		    
		    add_hash(s_entry);
		    s_entry->starting_block = dpnt->extent;
		    set_733((char *) s_entry->isorec.size, ROUND_UP(dpnt->size));
		    continue;
	       }

	       if(strcmp(s_entry->name,"..") == 0) 
	       {
		    if(dpnt == root)
		    { 
			 total_dir_size += root->size;
		    }
		    set_733((char *) s_entry->isorec.extent, dpnt->parent->extent);
#ifdef JOLIET
		    if (use_Joliet) {
		      set_733((char *) s_entry->j_isorec.extent, dpnt->parent->j_extent);
		      set_733((char *) s_entry->j_isorec.size, ROUND_UP(dpnt->parent->j_size));
		      s_entry->j_starting_block = dpnt->parent->j_extent;
		      s_entry->j_size = ROUND_UP(dpnt->parent->j_size);
		    }
#endif /* JOLIET */
		    
		    /* 
		     * Set these so that the hash table has the
		     * correct information
		     */
		    s_entry->starting_block = dpnt->parent->extent;
		    s_entry->size = ROUND_UP(dpnt->parent->size);
		    
		    add_hash(s_entry);
		    s_entry->starting_block = dpnt->parent->extent;
		    set_733((char *) s_entry->isorec.size, ROUND_UP(dpnt->parent->size));
		    continue;
	       }

	       /* 
		* Some ordinary non-directory file.  Just schedule the
		* file to be written.  This is all quite
		* straightforward, just make a list and assign extents
		* as we go.  Once we get through writing all of the
		* directories, we should be ready write out these
		* files
		*/
	       if(s_entry->size) 
	       {
		    dwpnt = (struct deferred_write *) 
			 e_malloc(sizeof(struct deferred_write));
#ifdef APPLE_HYB
		    /* save this directory entry for later use */
		    dwpnt->s_entry = s_entry;
#endif /* APPLE_HYB */
		    if(dw_tail)
		    {
			 dw_tail->next = dwpnt;
			 dw_tail = dwpnt;
		    } 
		    else 
		    {
			 dw_head = dwpnt;
			 dw_tail = dwpnt;
		    }
		    if(s_entry->inode  ==  TABLE_INODE) 
		    {
			 dwpnt->table = s_entry->table;
			 dwpnt->name = NULL;
			 sprintf(whole_path,"%s%sTRANS.TBL",
				 s_entry->filedir->whole_name, SPATH_SEPARATOR);
		    } 
		    else 
		    {
			 dwpnt->table = NULL;
			 strcpy(whole_path, s_entry->whole_name);
			 dwpnt->name = strdup(whole_path);
		    }
		    dwpnt->next = NULL;
		    dwpnt->size = s_entry->size;
		    dwpnt->extent = last_extent;
		    set_733((char *) s_entry->isorec.extent, last_extent);
#ifdef APPLE_HYB
		    /* set the initial padding to zero */
		    dwpnt->pad = 0;
#endif /* APPLE_HYB */
#ifdef JOLIET
		    if (use_Joliet) {
		      /* ordinary files have the same extent in both
			 the primary and Joliet */
		      set_733((char *) s_entry->j_isorec.extent, last_extent);
		      s_entry->j_starting_block = last_extent;
		      s_entry->j_size = s_entry->size;
		    }
#endif /* JOLIET */
		    s_entry->starting_block = last_extent;
		    add_hash(s_entry);
		    last_extent += ROUND_UP(s_entry->size) >> 11;
		    if(verbose)
		    {
			 fprintf(stderr,"%d %d %s\n", s_entry->starting_block,
				 last_extent-1, whole_path);
		    }
#ifdef DBG_ISO
		    if((ROUND_UP(s_entry->size) >> 11) > 500)
		    {
			 fprintf(stderr,"Warning: large file %s\n", whole_path);
			 fprintf(stderr,"Starting block is %d\n", s_entry->starting_block);
			 fprintf(stderr,"Reported file size is %d extents\n", s_entry->size);
			 
		    }
#endif
		    if(last_extent > (800000000 >> 11)) 
		    { 
			 /*
			  * More than 800Mb? Punt 
			  */
			 fprintf(stderr,"Extent overflow processing file %s\n", whole_path);
			 fprintf(stderr,"Starting block is %d\n", s_entry->starting_block);
			 fprintf(stderr,"Reported file size is %d extents\n", s_entry->size);
			 exit(1);
		    }
		    continue;
	       }

	       /*
		* This is for zero-length files.  If we leave the extent 0,
		* then we get screwed, because many readers simply drop files
		* that have an extent of zero.  Thus we leave the size 0,
		* and just assign the extent number.
		*/
	       set_733((char *) s_entry->isorec.extent, last_extent);
#ifdef JOLIET
	       if (use_Joliet)
		 set_733((char *) s_entry->j_isorec.extent, last_extent);
#endif /* JOLIET */
	  }
	  if(dpnt->subdir) 
	  {
	       assign_file_addresses(dpnt->subdir);
	  }
	  dpnt = dpnt->next;
     }
} /* assign_file_addresses(... */

void FDECL2(generate_one_directory, struct directory *, dpnt, FILE *, outfile)
{
     unsigned int			  ce_address = 0;
     char				* ce_buffer;
     unsigned int			  ce_index = 0;
     unsigned int			  ce_size;
     unsigned int			  dir_index;
     char				* directory_buffer;
     int				  new_reclen;
     struct directory_entry		* s_entry;
     struct directory_entry		* s_entry_d;
     unsigned int			  total_size;
     
     total_size = (dpnt->size + (SECTOR_SIZE - 1)) &  ~(SECTOR_SIZE - 1);
     directory_buffer = (char *) e_malloc(total_size);
     memset(directory_buffer, 0, total_size);
     dir_index = 0;
     
     ce_size = (dpnt->ce_bytes + (SECTOR_SIZE - 1)) &  ~(SECTOR_SIZE - 1);
     ce_buffer = NULL;
     if(ce_size)
     { 
          ce_buffer = (char *) e_malloc(ce_size);
          memset(ce_buffer, 0, ce_size);
	  ce_index = 0;
	  /*
	   * Absolute byte address of CE entries for this directory 
	   */
	  ce_address = last_extent_written + (total_size >> 11);
	  ce_address = ce_address << 11;
     }
     
     s_entry = dpnt->contents;
     while(s_entry) 
     {

	  /* 
	   * We do not allow directory entries to cross sector boundaries.  
	   * Simply pad, and then start the next entry at the next sector 
	   */
	  new_reclen = s_entry->isorec.length[0];
	  if( (dir_index & (SECTOR_SIZE - 1)) + new_reclen >= SECTOR_SIZE )
	  {
	       dir_index = (dir_index + (SECTOR_SIZE - 1)) & 
		    ~(SECTOR_SIZE - 1);
	  }

	  memcpy(directory_buffer + dir_index, &s_entry->isorec, 
		 sizeof(struct iso_directory_record) -
		 sizeof(s_entry->isorec.name) + s_entry->isorec.name_len[0]);
	  dir_index += sizeof(struct iso_directory_record) - 
	       sizeof (s_entry->isorec.name)+ s_entry->isorec.name_len[0];

	  /*
	   * Add the Rock Ridge attributes, if present 
	   */
	  if(s_entry->rr_attr_size)
	  {
	       if(dir_index & 1)
	       {
		    directory_buffer[dir_index++] = 0;
	       }

	       /* 
		* If the RR attributes were too long, then write the
		* CE records, as required.
		*/
	       if(s_entry->rr_attr_size != s_entry->total_rr_attr_size) 
	       {
		    unsigned char * pnt;
		    int len, nbytes;
		    
		    /* 
		     * Go through the entire record and fix up the CE entries
		     * so that the extent and offset are correct 
		     */
		    
		    pnt = s_entry->rr_attributes;
		    len = s_entry->total_rr_attr_size;
		    while(len > 3)
		    {
#ifdef DEBUG
			 if (!ce_size)
			 {
			      fprintf(stderr,"Warning: ce_index(%d) && ce_address(%d) not initialized\n",
				      ce_index, ce_address);
			 }
#endif
			 
			 if(pnt[0] == 'C' && pnt[1] == 'E') 
			 {
			      nbytes = get_733( (char *) pnt+20);
			      
			      if((ce_index & (SECTOR_SIZE - 1)) + nbytes >=
				 SECTOR_SIZE) 
			      {
				   ce_index = ROUND_UP(ce_index);
			      }
			      
			      set_733( (char *) pnt+4, 
				       (ce_address + ce_index) >> 11);
			      set_733( (char *) pnt+12, 
				       (ce_address + ce_index) & (SECTOR_SIZE - 1));
			      
			      
			      /* 
			       * Now store the block in the ce buffer 
			       */
			      memcpy(ce_buffer + ce_index, 
				     pnt + pnt[2], nbytes);
			      ce_index += nbytes;
			      if(ce_index & 1) 
			      {
				   ce_index++;
			      }
			 }
			 len -= pnt[2];
			 pnt += pnt[2];
		    }
		    
	       }

	       rockridge_size += s_entry->total_rr_attr_size;
	       memcpy(directory_buffer + dir_index, s_entry->rr_attributes, 
		      s_entry->rr_attr_size);
	       dir_index += s_entry->rr_attr_size;
	  }
	  if(dir_index & 1)
	  {
	       directory_buffer[dir_index++] = 0;
	  }
	  
	  s_entry_d = s_entry;
	  s_entry = s_entry->next;
	  
	  if (s_entry_d->rr_attributes) free(s_entry_d->rr_attributes);
#ifndef JOLIET
	  if (!use_Joliet) {
	  /* don't free up these yet if we are using Joliet! */
	    free (s_entry_d->name);
	    free (s_entry_d);
	  }
#endif /* JOLIET */
     }
     sort_dir = NULL;

     if(dpnt->size != dir_index)
     {
	  fprintf(stderr,"Unexpected directory length %d %d %s\n",dpnt->size, 
	    dir_index, dpnt->de_name);
     }

     xfwrite(directory_buffer, 1, total_size, outfile);
     last_extent_written += total_size >> 11;
     free(directory_buffer);

     if(ce_size)
     {
	  if(ce_index != dpnt->ce_bytes)
	  {
	       fprintf(stderr,"Continuation entry record length mismatch (%d %d).\n",
		       ce_index, dpnt->ce_bytes);
	  }
	  xfwrite(ce_buffer, 1, ce_size, outfile);
	  last_extent_written += ce_size >> 11;
	  free(ce_buffer);
     }
     
} /* generate_one_directory(... */

static 
void FDECL1(build_pathlist, struct directory *, node)
{
     struct directory * dpnt;
     
     dpnt = node;
     
     while (dpnt)
     {
	  pathlist[dpnt->path_index] = dpnt;
	  if(dpnt->subdir) build_pathlist(dpnt->subdir);
	  dpnt = dpnt->next;
     }
} /* build_pathlist(... */

int FDECL2(compare_paths, void const *, r, void const *, l) 
{
  struct directory const *ll = *(struct directory * const *)l;
  struct directory const *rr = *(struct directory * const *)r;

  if (rr->parent->path_index < ll->parent->path_index)
  {
       return -1;
  }

  if (rr->parent->path_index > ll->parent->path_index) 
  {
       return 1;
  }

  return strcmp(rr->self->isorec.name, ll->self->isorec.name);
  
} /* compare_paths(... */

void generate_path_tables()
{
  struct directory_entry * de;
  struct directory	 * dpnt;
  int			   fix;
  int			   i;
  int			   j;
  int			   namelen;
  char			 * npnt;
  char			 * npnt1;
  int			   tablesize;

  /*
   * First allocate memory for the tables and initialize the memory 
   */
  tablesize = path_blocks << 11;
  path_table_m = (char *) e_malloc(tablesize);
  path_table_l = (char *) e_malloc(tablesize);
  memset(path_table_l, 0, tablesize);
  memset(path_table_m, 0, tablesize);

  /*
   * Now start filling in the path tables.  Start with root directory 
   */
  path_table_index = 0;
  pathlist = (struct directory **) e_malloc(sizeof(struct directory *) 
					    * next_path_index);
  memset(pathlist, 0, sizeof(struct directory *) * next_path_index);
  build_pathlist(root);

  do
  {
       fix = 0;
       qsort(&pathlist[1], next_path_index-1, sizeof(struct directory *), 
	     (int (*)(const void *, const void *))compare_paths);

       for(j=1; j<next_path_index; j++)
       {
	    if(pathlist[j]->path_index != j)
	    {
		 pathlist[j]->path_index = j;
		 fix++;
	    }
       }
  } while(fix);

  for(j=1; j<next_path_index; j++)
  {
       dpnt = pathlist[j];
       if(!dpnt)
       {
	    fprintf(stderr,"Entry %d not in path tables\n", j);
	    exit(1);
       }
       npnt = dpnt->de_name;
       
       /* 
	* So the root comes out OK 
	*/
       if( (*npnt == 0) || (dpnt == root) ) 
       {
	    npnt = ".";  
       }
       npnt1 = strrchr(npnt, PATH_SEPARATOR);
       if(npnt1) 
       { 
	    npnt = npnt1 + 1;
       }
       
       de = dpnt->self;
       if(!de) 
       {
	    fprintf(stderr,"Fatal goof\n"); 
	    exit(1);
       }
#ifdef DEBUG
	fprintf(stderr, "%d %s  %d\n",j, dpnt->whole_name,
		dpnt->parent->path_index);
#endif /* DEBUG */
       
       namelen = de->isorec.name_len[0];
       
       path_table_l[path_table_index] = namelen;
       path_table_m[path_table_index] = namelen;
       path_table_index += 2;
       
       set_731(path_table_l + path_table_index, dpnt->extent); 
       set_732(path_table_m + path_table_index, dpnt->extent); 
       path_table_index += 4;
       
       set_721(path_table_l + path_table_index, 
	       dpnt->parent->path_index); 
       set_722(path_table_m + path_table_index, 
	       dpnt->parent->path_index); 
       path_table_index += 2;
       
       for(i =0; i<namelen; i++)
       {
	    path_table_l[path_table_index] = de->isorec.name[i];
	    path_table_m[path_table_index] = de->isorec.name[i];
	    path_table_index++;
       }
       if(path_table_index & 1) 
       {
	    path_table_index++;  /* For odd lengths we pad */
       }
  }
  
  free(pathlist);
  if(path_table_index != path_table_size)
  {
       fprintf(stderr,"Path table lengths do not match %d %d\n",
	       path_table_index,
	       path_table_size);
  }
} /* generate_path_tables(... */

#ifdef JOLIET
/* Joliet version of generate_one_directory() - don't worry about RR here */
void FDECL2(j_generate_one_directory, struct directory *, dpnt, FILE *, outfile)
{
     unsigned int			  dir_index;
     char				* directory_buffer;
     int				  new_reclen;
     struct directory_entry		* s_entry;
     struct directory_entry		* s_entry_d;
     unsigned int			  total_size;
     
     total_size = (dpnt->j_size + (SECTOR_SIZE - 1)) &  ~(SECTOR_SIZE - 1);
     directory_buffer = (char *) e_malloc(total_size);
     memset(directory_buffer, 0, total_size);
     dir_index = 0;
     
     s_entry = dpnt->j_contents;
     while(s_entry) 
     {

	  /* 
	   * We do not allow directory entries to cross sector boundaries.  
	   * Simply pad, and then start the next entry at the next sector 
	   */
	  new_reclen = s_entry->j_isorec.length[0];
	  if( (dir_index & (SECTOR_SIZE - 1)) + new_reclen >= SECTOR_SIZE )
	  {
	       dir_index = (dir_index + (SECTOR_SIZE - 1)) & 
		    ~(SECTOR_SIZE - 1);
	  }

	  memcpy(directory_buffer + dir_index, &s_entry->j_isorec, 
		 sizeof(struct iso_directory_record) -
		 sizeof(s_entry->j_isorec.name) + s_entry->j_isorec.name_len[0]);
	  dir_index += sizeof(struct iso_directory_record) - 
	       sizeof (s_entry->j_isorec.name)+ s_entry->j_isorec.name_len[0];

	  if(dir_index & 1)
	  {
	       directory_buffer[dir_index++] = 0;
	  }
	  
	  s_entry_d = s_entry;
	  s_entry = s_entry->j_next;
	  
	  free (s_entry_d->name);
	  free (s_entry_d);
     }
     sort_dir = NULL;

     if(dpnt->j_size != dir_index)
     {
	  fprintf(stderr,"Unexpected directory length %d %d %s\n",dpnt->size, 
	    dir_index, dpnt->de_name);
     }

     xfwrite(directory_buffer, 1, total_size, outfile);
     last_extent_written += total_size >> 11;
     free(directory_buffer);

} /* j_generate_one_directory(... */

/* Joliet version of build_pathlist() */
static 
void FDECL1(j_build_pathlist, struct directory *, node)
{
     struct directory * dpnt;
     
     dpnt = node;
     
     while (dpnt)
     {
	  pathlist[dpnt->j_path_index] = dpnt;
	  if(dpnt->subdir) j_build_pathlist(dpnt->subdir);
	  dpnt = dpnt->next;
     }
} /* j_build_pathlist(... */

/* Joliet version of compare_paths() */
int FDECL2(j_compare_paths, void const *, r, void const *, l) 
{
  struct directory const *ll = *(struct directory * const *)l;
  struct directory const *rr = *(struct directory * const *)r;

  if (rr->parent->j_path_index < ll->parent->j_path_index)
  {
       return -1;
  }

  if (rr->parent->j_path_index > ll->parent->j_path_index) 
  {
       return 1;
  }

  return memcmp(rr->self->j_isorec.name, ll->self->j_isorec.name,
	sizeof(rr->self->j_isorec.name));
  
} /* j_compare_paths(... */

/* Joliet version of generate_path_tables() */
void j_generate_path_tables()
{
  struct directory_entry * de;
  struct directory	 * dpnt;
  int			   fix;
  int			   i;
  int			   j;
  int			   namelen;
  char			 * npnt;
  char			 * npnt1;
  int			   tablesize;

  /*
   * First allocate memory for the tables and initialize the memory 
   */
  tablesize = j_path_blocks << 11;
  j_path_table_m = (char *) e_malloc(tablesize);
  j_path_table_l = (char *) e_malloc(tablesize);
  memset(j_path_table_l, 0, tablesize);
  memset(j_path_table_m, 0, tablesize);

  /*
   * Now start filling in the path tables.  Start with root directory 
   */
  j_path_table_index = 0;
  pathlist = (struct directory **) e_malloc(sizeof(struct directory *) 
					    * j_next_path_index);
  memset(pathlist, 0, sizeof(struct directory *) * j_next_path_index);
  j_build_pathlist(root);

  do
  {
       fix = 0;
       qsort(&pathlist[1], j_next_path_index-1, sizeof(struct directory *), 
	     (int (*)(const void *, const void *))j_compare_paths);

       for(j=1; j<j_next_path_index; j++)
       {
	    if(pathlist[j]->j_path_index != j)
	    {
		 pathlist[j]->j_path_index = j;
		 fix++;
	    }
       }
  } while(fix);

  for(j=1; j<j_next_path_index; j++)
  {
       dpnt = pathlist[j];
       if(!dpnt)
       {
	    fprintf(stderr,"Entry %d not in path tables\n", j);
	    exit(1);
       }
       npnt = dpnt->de_name;
       
       /* 
	* So the root comes out OK 
	*/
       if( (*npnt == 0) || (dpnt == root) ) 
       {
	    npnt = ".";  
       }
       npnt1 = strrchr(npnt, PATH_SEPARATOR);
       if(npnt1) 
       { 
	    npnt = npnt1 + 1;
       }
       
       de = dpnt->self;
       if(!de) 
       {
	    fprintf(stderr,"Fatal goof\n"); 
	    exit(1);
       }

#ifdef DEBUG
	fprintf(stderr, "%d %s  %d\n",j, dpnt->whole_name,
		dpnt->parent->j_path_index);
#endif /* DEBUG */ 

       namelen = de->j_isorec.name_len[0];
       
       j_path_table_l[j_path_table_index] = namelen;
       j_path_table_m[j_path_table_index] = namelen;
       j_path_table_index += 2;
       
       set_731(j_path_table_l + j_path_table_index, dpnt->j_extent); 
       set_732(j_path_table_m + j_path_table_index, dpnt->j_extent); 
       j_path_table_index += 4;
       
       set_721(j_path_table_l + j_path_table_index, 
	       dpnt->parent->j_path_index); 
       set_722(j_path_table_m + j_path_table_index, 
	       dpnt->parent->j_path_index); 
       j_path_table_index += 2;
       
       for(i =0; i<namelen; i++)
       {
	    j_path_table_l[j_path_table_index] = de->j_isorec.name[i];
	    j_path_table_m[j_path_table_index] = de->j_isorec.name[i];
	    j_path_table_index++;
       }
       if(j_path_table_index & 1) 
       {
	    j_path_table_index++;  /* For odd lengths we pad */
       }
  }
  
  free(pathlist);
  if(j_path_table_index != j_path_table_size)
  {
       fprintf(stderr,"Joliet path table lengths do not match %d %d\n",
	       j_path_table_index,
	       j_path_table_size);
  }
} /* j_generate_path_tables(... */

/* Joliet version of memcpy_max() */
void
FDECL3(j_memcpy_max, char *, tt, char *, ff, int, max)
{
  int spare;
  int n;

  if (ff)
       n = strlen(ff);
  else
       /* empty string */
       n = 0;

  /* truncate if necessary */
  if (n > max/2)
  {
       n = max/2;
  }

  /* find the remainder of the string */
  spare = max/2 - n;

  /* copy 8 bit chars to 16 bit */
  tt++;
  while (n-- && *ff)
  {
       *tt = *ff;

	tt += 2;
	ff++;
  }

  /* set the rest to '0 ' */
  while(spare--)
  {
	*tt = ' ';
	tt += 2;
  }
} /* j_memcpy_max(... */

#endif /* JOLIET */

void
FDECL3(memcpy_max, char *, to, char *, from, int, max)
{
  int n = strlen(from);
  if (n > max)
  {
       n = max;
  }
  memcpy(to, from, n);

} /* memcpy_max(... */

int FDECL1(iso_write, FILE *, outfile)
{
  char				buffer[2048];
  int				i;
  char				iso_time[17];
  int				should_write;
#ifdef APPLE_HYB
  int				start_extent;   /* orig ISO files start */
  int				pad = 0;	/* initial padding */
  int				Csize;		/* clump size for HFS vol */
  int				n, r;		/* HFS hdr output */

  /* get the expected start of the ISO files */
  start_extent = last_extent;

#endif /* APPLE_HYB */

  time(&begun);
  assign_file_addresses(root);

#ifdef APPLE_HYB
  if (apple_hyb)
  {
    int loop = CTC_LOOP;
    int last_extent_save = last_extent;

    /* allocate memory for the libhfs/mkisofs extra info */
    hce = (hce_mem *)e_malloc(sizeof(hce_mem));

    hce->error = (char *)e_malloc(1024);

    /* mark as unallocated for use later */
    hce->hfs_ce = hce->hfs_hdr = 0;

    /* set the intial factor to increase Catalog file size */
    hce->ctc_size = CTC;

    /* "create" the HFS volume (just the header, catalog/extents files)
	if there's a problem with the Catalog file being too small,
	we keep on increasing the size (up to CTC_LOOP) times and try again.
	Unfortunately I don't know enough about the inner workings of
	HFS, so I can't workout the size of the Catalog file in
	advance (and I don't want to "grow" as is is normally allowed to),
	therefore, this approach is a bit over the top as it involves
	throwing away the "volume" we have created and trying again ...  */
    do
    {
	hce->error[0] = '\0';

	/* attempt to create the Mac volume */
	Csize = make_mac_volume(root, start_extent);

	/* if we have a problem ... */
	if (Csize < 0)
	{
	    /* we've made too many attempts, or got some other error */
	    if (loop == 0 || errno != HCE_ERROR)
	    {
		/* HCE_ERROR is not a valid errno value */
		if (errno == HCE_ERROR)
		     errno = 0;

		/* exit with the error */
		if (*hce->error)
		    fprintf(stderr, "%s\n", hce->error);
		perr(hfs_error);
	    }
	    else
	    {
		/* increase Catalog file size factor */
		hce->ctc_size *= CTC;
 
		/* reset the initial "last_extent" and try again */
		last_extent = last_extent_save;
	    }
	}
	else
	    /* everything OK - just carry on ... */
	    loop = 0;
    }
    while (loop--);

    /* finished with any HFS type errors */
    free(hce->error);
    hce->error = 0;

    /* the ISO files need to start on a multiple of the HFS allocation
       blocks, so find out how much padding we need */
    pad = V_ROUND_UP(start_extent*SECTOR_SIZE, Csize)/SECTOR_SIZE;

    pad -= start_extent;
  }
#endif /* APPLE_HYB */

  memset(buffer, 0, sizeof(buffer));

  /*
   * This will break  in the year  2000, I supose, but there is no good way
   * to get the top two digits of the year. 
   */
  local = localtime(&Ctime);
  sprintf(iso_time, "%4.4d%2.2d%2.2d%2.2d%2.2d%2.2d00", 1900 + local->tm_year,
	  local->tm_mon+1, local->tm_mday,
	  local->tm_hour, local->tm_min, local->tm_sec);

#ifdef APPLE_HYB
  n = 0;

  if (apple_hyb)
  {
    /* write out HFS volume header info */
    xfwrite(hce->hfs_hdr, hce->hfs_hdr_size, HFS_BLOCKSZ, outfile);

    /* get size in CD blocks == 4xHFS_BLOCKSZ == 2048 */
    n = hce->hfs_hdr_size/4;
    r = hce->hfs_hdr_size%4;

    /* write out any partial CD block */
    if (r)
    {
      xfwrite(buffer, 4-r, HFS_BLOCKSZ, outfile);
      n++;
    }
  }

  /* write out the remainder of the ISO header */
  for(i=n; i<16; i++)
#else

  /*
   * First, we output 16 sectors of all zero 
   */

  for(i=0; i<16; i++)
#endif /* APPLE_HYB */
    {
      xfwrite(buffer, 1, sizeof(buffer), outfile);
    }

  last_extent_written += 16;

  /*
   * Next we write out the primary descriptor for the disc 
   */
  memset(&vol_desc, 0, sizeof(vol_desc));
  vol_desc.type[0] = ISO_VD_PRIMARY;
  memcpy(vol_desc.id, ISO_STANDARD_ID, sizeof(ISO_STANDARD_ID));
  vol_desc.version[0] = 1;
  
  memset(vol_desc.system_id, ' ', sizeof(vol_desc.system_id));
  memcpy_max(vol_desc.system_id, system_id, strlen(system_id));
  
  memset(vol_desc.volume_id, ' ', sizeof(vol_desc.volume_id));
  memcpy_max(vol_desc.volume_id, volume_id, strlen(volume_id));
  
  should_write = last_extent - session_start;
#ifdef APPLE_HYB
  if (apple_hyb)
    /* add in the catalog/extents/dt file sizes */
    set_733((char *) vol_desc.volume_space_size,
        should_write + H_ROUND_UP(hce->hfs_tot_size)/SECTOR_SIZE);
  else
#endif /* APPLE_HYB */
    set_733((char *) vol_desc.volume_space_size, should_write);
  set_723(vol_desc.volume_set_size, 1);
  set_723(vol_desc.volume_sequence_number, DEF_VSN);
  set_723(vol_desc.logical_block_size, 2048);
  
  /*
   * The path tables are used by DOS based machines to cache directory
   * locations 
   */

  set_733((char *) vol_desc.path_table_size, path_table_size);
  set_731(vol_desc.type_l_path_table, path_table[0]);
  set_731(vol_desc.opt_type_l_path_table, path_table[1]);
  set_732(vol_desc.type_m_path_table, path_table[2]);
  set_732(vol_desc.opt_type_m_path_table, path_table[3]);

  /*
   * Now we copy the actual root directory record 
   */
  memcpy(vol_desc.root_directory_record, &root_record, 
	 sizeof(struct iso_directory_record) + 1);

  /*
   * The rest is just fluff.  It looks nice to fill in many of these fields,
   * though.
   */
  FILL_SPACE(volume_set_id);
  if(volset_id)  memcpy_max(vol_desc.volume_set_id,  volset_id, strlen(volset_id));

  FILL_SPACE(publisher_id);
  if(publisher)  memcpy_max(vol_desc.publisher_id,  publisher, strlen(publisher));

  FILL_SPACE(preparer_id);
  if(preparer)  memcpy_max(vol_desc.preparer_id,  preparer, strlen(preparer));

  FILL_SPACE(application_id);
  if(appid) memcpy_max(vol_desc.application_id, appid, strlen(appid));

  FILL_SPACE(copyright_file_id);
  if(copyright) memcpy_max(vol_desc.copyright_file_id, copyright, 
		       strlen(copyright));

  FILL_SPACE(abstract_file_id);
  if(abstract) memcpy_max(vol_desc.abstract_file_id, abstract, 
			  strlen(abstract));

  FILL_SPACE(bibliographic_file_id);
  if(biblio) memcpy_max(vol_desc.bibliographic_file_id, biblio, 
		       strlen(biblio));

  FILL_SPACE(creation_date);
  FILL_SPACE(modification_date);
  FILL_SPACE(expiration_date);
  FILL_SPACE(effective_date);
  vol_desc.file_structure_version[0] = 1;
  FILL_SPACE(application_data);

  memcpy(vol_desc.creation_date,  iso_time, 17);
  memcpy(vol_desc.modification_date,  iso_time, 17);
  memcpy(vol_desc.expiration_date, "0000000000000000", 17);
  memcpy(vol_desc.effective_date,  iso_time,  17);

  /*
   * if not a bootable cd do it the old way 
   */
  xfwrite(&vol_desc, 1, 2048, outfile);
  if (!use_eltorito) 
    {
      /*
       * For some reason, Young Minds writes this twice.  Aw, what the heck 
       */
      xfwrite(&vol_desc, 1, 2048, outfile);
    }
  else
    {
      /*
       * Next we write out the boot volume descriptor for the disc 
       */
      get_torito_desc(&boot_desc);
      xfwrite(&boot_desc, 1, 2048, outfile);
    }
  
  /* 
   * either way, we've written two more 
   */

  last_extent_written += 2;

  /*
   * Now write the end volume descriptor.  Much simpler than the other one 
   */
  memset(&vol_desc, 0, sizeof(vol_desc));
  vol_desc.type[0] = ISO_VD_END;
  memcpy(vol_desc.id, ISO_STANDARD_ID, sizeof(ISO_STANDARD_ID));
  vol_desc.version[0] = 1;

#ifdef JOLIET
  if (use_Joliet) {
    /* set and write out the supplementary descriptor */
    memset(&vol_sdesc, 0, sizeof(vol_sdesc));
    vol_sdesc.type[0] = ISO_VD_SUPPLEMENTARY;
    memcpy(vol_sdesc.id, ISO_STANDARD_ID, sizeof(ISO_STANDARD_ID));
    vol_sdesc.version[0] = 1;
    
    memset(vol_sdesc.system_id, 0, sizeof(vol_sdesc.system_id));
    j_memcpy_max(vol_sdesc.system_id, system_id, sizeof(vol_sdesc.system_id));
  
    memset(vol_sdesc.volume_id, 0, sizeof(vol_sdesc.volume_id));
    j_memcpy_max(vol_sdesc.volume_id, volume_id, sizeof(vol_sdesc.volume_id));
  
    should_write = last_extent - session_start;
#ifdef APPLE_HYB
    if (apple_hyb)
      /* add in the catalog/extents/dt file sizes */
      set_733((char *) vol_sdesc.volume_space_size,
        should_write + H_ROUND_UP(hce->hfs_tot_size)/SECTOR_SIZE);
    else
#endif /* APPLE_HYB */
      set_733((char *) vol_sdesc.volume_space_size, should_write);

    /* UCS-2 Level 1 escape sequence */
    vol_sdesc.escape[0] = 0x25;
    vol_sdesc.escape[1] = 0x2f;
    vol_sdesc.escape[2] = 0x40;

    set_723(vol_sdesc.volume_set_size, 1);
    set_723(vol_sdesc.volume_sequence_number, DEF_VSN);
    set_723(vol_sdesc.logical_block_size, 2048);
  
    /*
     * The path tables are used by DOS based machines to cache directory
     * locations 
     */

    set_733((char *) vol_sdesc.path_table_size, j_path_table_size);
    set_731(vol_sdesc.type_l_path_table, j_path_table[0]);
    set_731(vol_sdesc.opt_type_l_path_table, j_path_table[1]);
    set_732(vol_sdesc.type_m_path_table, j_path_table[2]);
    set_732(vol_sdesc.opt_type_m_path_table, j_path_table[3]);

    /*
     * Now we copy the actual root directory record 
     */
    memcpy(vol_sdesc.root_directory_record, &j_root_record, 
	 sizeof(struct iso_directory_record) + 1);

    J_FILL_SPACE(volume_set_id);
    j_memcpy_max(vol_sdesc.volume_set_id, volset_id, sizeof(vol_sdesc.volume_set_id));

    J_FILL_SPACE(publisher_id);
    j_memcpy_max(vol_sdesc.publisher_id, publisher, sizeof(vol_sdesc.publisher_id));

    J_FILL_SPACE(preparer_id);
    j_memcpy_max(vol_sdesc.preparer_id, preparer, sizeof(vol_sdesc.preparer_id));

    J_FILL_SPACE(application_id);
    j_memcpy_max(vol_sdesc.application_id, appid, sizeof(vol_sdesc.application_id));

    J_FILL_SPACE(copyright_file_id);
    j_memcpy_max(vol_sdesc.copyright_file_id, copyright, sizeof(vol_sdesc.copyright_file_id));

    J_FILL_SPACE(abstract_file_id);
    j_memcpy_max(vol_sdesc.abstract_file_id, abstract, sizeof(vol_sdesc.abstract_file_id));

    J_FILL_SPACE(bibliographic_file_id);
    j_memcpy_max(vol_sdesc.bibliographic_file_id, biblio, sizeof(vol_sdesc.bibliographic_file_id));

    J_FILL_SPACE(creation_date);
    J_FILL_SPACE(modification_date);
    J_FILL_SPACE(expiration_date);
    J_FILL_SPACE(effective_date);
    vol_sdesc.file_structure_version[0] = 1;
    J_FILL_SPACE(application_data);

    memcpy(vol_sdesc.creation_date,  iso_time, 17);
    memcpy(vol_sdesc.modification_date,  iso_time, 17);
    memcpy(vol_sdesc.expiration_date, "0000000000000000", 17);
    memcpy(vol_sdesc.effective_date,  iso_time,  17);

    xfwrite(&vol_sdesc, 1, 2048, outfile);
  }
  else
    /* not using Joliet, write out end descriptor twice */
#endif /* JOLIET */
    xfwrite(&vol_desc, 1, 2048, outfile);

  xfwrite(&vol_desc, 1, 2048, outfile);
  last_extent_written += 2;

  /*
   * Next we write the path tables 
   */
  xfwrite(path_table_l, 1, path_blocks << 11, outfile);
  xfwrite(path_table_m, 1, path_blocks << 11, outfile);
  last_extent_written += 2*path_blocks;
  free(path_table_l);
  free(path_table_m);
  path_table_l = NULL;
  path_table_m = NULL;

#ifdef JOLIET
  if (use_Joliet) {
    xfwrite(j_path_table_l, 1, j_path_blocks << 11, outfile);
    xfwrite(j_path_table_m, 1, j_path_blocks << 11, outfile);
    last_extent_written += 2*j_path_blocks;
    free(j_path_table_l);
    free(j_path_table_m);
    j_path_table_l = NULL;
    j_path_table_m = NULL;
  }
#endif /* JOLIET */

  /*
   * OK, all done with that crap.  Now write out the directories.
   * This is where the fur starts to fly, because we need to keep track of
   * each file as we find it and keep track of where we put it. 
   */

#ifdef DBG_ISO
  fprintf(stderr,"Total directory extents being written = %d\n", last_extent);
#endif
#if 0 
 generate_one_directory(root, outfile);
#endif
  generate_iso9660_directories(root, outfile);

#ifdef JOLIET
  if (use_Joliet)
    j_generate_iso9660_directories(root, outfile);
#endif /* JOLIET */

  if(extension_record) 
    {
      xfwrite(extension_record, 1, SECTOR_SIZE, outfile);
      last_extent_written++;
    }

  /* 
   * Now write all of the files that we need. 
   */
#ifdef APPLE_HYB
  if (apple_hyb) {
	int i;

	/* write out padding to round up to HFS allocation block */
	hfs_extra = H_ROUND_UP(hce->hfs_tot_size)/SECTOR_SIZE;

	for(i=0;i<pad;i++)
	    xfwrite(buffer, 1, sizeof(buffer), outfile);

	last_extent_written += pad;

	fprintf(stderr,"Total extents scheduled to be written (inc HFS) = %d\n",
	  last_extent - session_start + hfs_extra);
  }
  else
#endif /* APPLE_HYB */
    fprintf(stderr,"Total extents scheduled to be written = %d\n", 
	  last_extent - session_start);

  write_files(outfile);
 
#ifdef APPLE_HYB
  /* write out extents/catalog/dt file */
  if (apple_hyb) {

	xfwrite(hce->hfs_ce, hce->hfs_tot_size, HFS_BLOCKSZ, outfile);

	/* round up to a whole CD block */
	if (H_ROUND_UP(hce->hfs_tot_size) - hce->hfs_tot_size*HFS_BLOCKSZ)
	    xfwrite(buffer, 1, H_ROUND_UP(hce->hfs_tot_size) - hce->hfs_tot_size*HFS_BLOCKSZ, outfile);
	fprintf(stderr, "Total extents actually written (inc HFS) = %d\n",
	    last_extent_written - session_start + hfs_extra);
	fprintf(stderr, "(Size of ISO volume = %d, HFS extra = %d)\n",
	    last_extent_written - session_start, hfs_extra);
  }
  else
#endif /* APPLE_HYB */
    fprintf(stderr,"Total extents actually written = %d\n", 
	  last_extent_written - session_start);
  /* 
   * Hard links throw us off here 
   */
  if(should_write != last_extent - session_start)
    {
      fprintf(stderr,"Number of extents written not what was predicted.  Please fix.\n");
      fprintf(stderr,"Predicted = %d, written = %d\n", should_write, last_extent);
    }

  fprintf(stderr,"Total translation table size: %d\n", table_size);
  fprintf(stderr,"Total rockridge attributes bytes: %d\n", rockridge_size);
  fprintf(stderr,"Total directory bytes: %d\n", total_dir_size);
  fprintf(stderr,"Path table size(bytes): %d\n", path_table_size);

#ifdef DEBUG
  fprintf(stderr, "next extent, last_extent, last_extent_written %d %d %d\n",
	  next_extent, last_extent, last_extent_written);
#endif

  return 0;

} /* iso_write(... */

#ifdef APPLE_HYB

/*
**	get_adj_size:	get the ajusted size of the volume with the HFS
**			allocation block size for each file
*/			
int FDECL1(get_adj_size, int, Csize)
{
	struct deferred_write *dw;
	int     size = 0;

	/* loop through all the files finding the new total size */
	for(dw = dw_head; dw; dw = dw->next)
	{
	    size += V_ROUND_UP(dw->size, Csize);
	}

	return(size);
}
/*
**	adj_size:	adjust the ISO record entries for all files
**			based on the HFS allocation block size
*/
int FDECL2(adj_size, int, Csize, int, start_extent)
{
	struct deferred_write *dw;
	struct directory_entry *s_entry;
	int	size;

	/* get the adjusted start_extent (with padding) */
	start_extent = V_ROUND_UP(start_extent*SECTOR_SIZE, Csize)/SECTOR_SIZE;

	/* loop through all files changing their starting blocks and
	   finding any padding needed to written out latter */
	for(dw = dw_head; dw; dw = dw->next)
	{
	    s_entry = dw->s_entry;
	    s_entry->starting_block = dw->extent = start_extent;
	    set_733((char *) s_entry->isorec.extent, start_extent);
#ifdef JOLIET
	    s_entry->j_starting_block = start_extent;
	    set_733((char *) s_entry->j_isorec.extent, start_extent);
#endif /* JOLIET */
	    size = V_ROUND_UP(dw->size, Csize)/SECTOR_SIZE;
	    dw->pad = size - ROUND_UP(dw->size)/SECTOR_SIZE;

	    start_extent += size;
	}

	return(start_extent);
}
	
#endif /* APPLE_HYB */
