/*
**	volume.c: prepare HFS volume for mkhybrid
**
**	James Pearson 17/7/97
**	modified JCP 29/7/97 to improve allocation sizes to cut
**	down on wasted space. Now uses the HFS "allocation" size rounded
**	up to the nearest 2048 bytes. Savings can be significant with
**	a large volume containing lots of smallish files.
*/

#ifdef APPLE_HYB

#include "mkisofs.h"
#include "config.h"

/* ISO blocks == 2048, HFS blocks == 512 */
#define BLK_CONV SECTOR_SIZE/HFS_BLOCKSZ

static hfsvol *vol_save = 0;	/* used to "destroy" an HFS volume */

int DECL(copy_to_mac_vol, (hfsvol *, struct directory *, long drAlBlSt));

/*
**	AlcSiz: find allocation size for given volume size
*/
int
AlcSiz(int vlen)
{
	int	lpa, drAlBlkSiz;

	/* code extracted from hfs_format() */
	lpa = 1 + vlen / 65536;
	drAlBlkSiz = lpa * HFS_BLOCKSZ;

	/* now set our "allocation size" to the allocation block rounded
	   up to the nearest SECTOR_SIZE (2048 bytes)  */
	drAlBlkSiz = V_ROUND_UP(drAlBlkSiz, SECTOR_SIZE);

	return(drAlBlkSiz);
}

/*
**	XClpSiz: find the default size of the catalog/extent file
*/
int
XClpSiz(int vlen)
{
	int	olpa, lpa, drNmAlBlks, drAlBlkSiz;
	int	vbmsz, drXTClpSiz;

	/* code extracted from hfs_format() */

	/* get the lpa from our calculated allocation block size */
	drAlBlkSiz = AlcSiz(vlen);
	lpa = drAlBlkSiz/HFS_BLOCKSZ;

	vbmsz = (vlen / lpa + 4095) / 4096;
	drNmAlBlks = (vlen - 5 - vbmsz) / lpa;
	drXTClpSiz = drNmAlBlks / 128 * drAlBlkSiz;

	/* make allowances because we have possibly rounded up the
	   allocation size */

	/* get the "original" lpa " */
	olpa = 1 + vlen / 65536;

	/* adjust size upwards */
	drXTClpSiz = (drXTClpSiz*lpa)/olpa;

	/* round up to the nearest alloaction size */
	drXTClpSiz = V_ROUND_UP(drXTClpSiz, drAlBlkSiz);

	return(drXTClpSiz);
}

/*
**	get_vol_size: get the size of the volume including the extent/catalog
*/
int
get_vol_size(int vblen)
{
	int	drXTClpSiz, new_drXTClpSiz, drAlBlkSiz;
	int	new_vblen;

	/* try to estimate a "volume size" based on the code
	   in hfs_format - we need the size of the catalog/extents
	   and Desktop files included in the volume, as we add this
	   to the end of the ISO volume */

	drXTClpSiz = XClpSiz(vblen);
	drAlBlkSiz = AlcSiz(vblen);

	/* catalog file is set at CTC times (default twice) the extents file
	   size - hence the (ctc_size + 1) below. The Desktop starts of the
	   same size as the "clump size" == 4 x drAlBlkSiz, plus a spare
	   drAlBlkSiz for the alternative MDB */

	new_vblen = vblen + ((hce->ctc_size + 1)*drXTClpSiz + 5*drAlBlkSiz)/HFS_BLOCKSZ;

	return (new_vblen);
}

/*
**	write_fork: "write" file data to the volume
**
**	This is used to update the HFS file internal structures
**	but no data is actually wriiten (it's trapped deep down in
**	libhfs).
*/
int
write_fork(hfsfile *hfp, long tot)
{
	char	blk[HFS_BLOCKSZ];
	unsigned short start;
	long	len;

	len = tot;
	/* we need to know where this fork starts */
	start = hfs_get_drAllocPtr(hfp);

	/* loop through the data a block at a time */
	while (len >= HFS_BLOCKSZ)
	{
	    if(hfs_write(hfp, blk, HFS_BLOCKSZ) < 0)
		return(-1);
	    len -= HFS_BLOCKSZ;
	}
	/* write out anything left */
	if (len)
	    if(hfs_write(hfp, blk, len) < 0)
		return(-1);

	/* set the start of the allocation search to be immediately
	   after this fork */
	hfs_set_drAllocPtr(hfp, start, tot);

	return(0);
}

/*
**	make_mac_volume: "create" an HFS volume using the ISO data
**
**	The HFS volume structures are set up (but no data is written yet).
**
**	ISO volumes have a allocation size of 2048 bytes - regardless
**	of the size of the volume. HFS allocation size is depends on volume
**	size, so we may have to update the ISO structures to add in any
**	padding.
*/
int FDECL2(make_mac_volume, struct directory *, dpnt, int, start_extent)
{
	char vol_name[HFS_MAX_VLEN+1];	/* Mac volume name */
	hfsvol *vol;			/* Mac volume */
	int vlen, vblen;		/* vol length (bytes, blocks) */
	int Csize, lastCsize;		/* allocation sizes */
	int ret = 0;			/* return value */
	int loop = 1;

	/* umount volume if we have had a previous attempt */
	if (vol_save)
	    if (hfs_umount(vol_save, 0) < 0)
		return (-1);

	/* set the default clump size to the ISO block size */
	lastCsize = SECTOR_SIZE;

	if (verbose)
	    fprintf(stderr, "Creating HFS Volume info\n");

	/* copy ISO volume name to Mac Volume name */
	strncpy(vol_name, volume_id, HFS_MAX_VLEN);
	vol_name[HFS_MAX_VLEN] = '\0';

	/* get initial size of HFS volume (size of ISO volume) */
	vblen = last_extent * BLK_CONV;

	/* add on size of extents/catalog file, but this may mean
	   the allocation size will change, so loop round until the allocation
	   size doesn't change */
	while (loop) {
	    hce->XTCsize = XClpSiz(vblen);
	    vblen = get_vol_size(vblen);
	    Csize = AlcSiz(vblen);

	    if (Csize == lastCsize) {
		/* allocation size hasn't changed, so carry on */
		loop = 0;
	    }
	    else {
		/* allocation size has changed, so update ISO volume size */
		vlen = get_adj_size(Csize);
		vlen += V_ROUND_UP(start_extent * SECTOR_SIZE, Csize);
		vblen = vlen /  HFS_BLOCKSZ;
		lastCsize = Csize;
	    }
	}

	/* set vlen to size in bytes */
	vlen = hce->hfs_vol_size = vblen * HFS_BLOCKSZ;

	/* update the ISO structures with new start extents and any padding
	   required */
	if (Csize != SECTOR_SIZE)
	    last_extent = adj_size(Csize, start_extent);

	/* set the default allocation size for libhfs */
	hce->Csize = Csize;

	/* format and mount the "volume" */
	if (hfs_format(hce, 0, vol_name) < 0)
	{
	    sprintf(hce->error,"can't HFS format %s",vol_name);
	    return(-1);
	}

	if ((vol = hfs_mount(hce, 0, 0)) == 0)
	{
	    sprintf(hce->error,"can't HFS mount %s",vol_name);
	    return(-1);
	}

	/* save the volume for possible later use */
	vol_save = vol;

	/* Recursively "copy" the files to the volume - we need to
	   know the first allocation block in the volume as starting blocks
	   of files are relative to this.
	*/
	ret = copy_to_mac_vol(vol, dpnt, hce->hfs_hdr_size);
	if (ret < 0)
	    return(ret);

	/* make the Desktop files - I *think* this stops the Mac
	   rebuilding the desktop when the CD is mounted on a Mac
	   These will be ignored if they already exist */
	ret = make_desktop(vol, last_extent*BLK_CONV);
	if (ret < 0)
	    return(ret);

	/* close the volume */
	if (hfs_flush(vol) < 0)
	    return(-1);

	/* unmount and set the start blocks for the catalog/extents files */
	if (hfs_umount(vol, last_extent*BLK_CONV) < 0)
	    return(-1);

	return(Csize);
}

/*	copy_to_mac_vol: copy all files in a directory to corresponding
**			 Mac folder.
**
**	Files are copied recursively to corresponding folders on the Mac
**	volume. The caller routine needs to do a hfs_chdir before calling this
**	routine. 
*/
int FDECL3(copy_to_mac_vol, hfsvol, *vol, struct directory *, node, long, drAlBlSt)
{
	struct directory_entry * s_entry;	/* ISO directory entry */
	struct directory *dpnt;			/* ISO directory */

	hfsfile *hfp;				/* HFS file */
	hfsdirent *ent;				/* HFS file entities */
	long id;				/* current HFS folder */
	long dext, rext;			/* real data/rsrc start blk */
	int ret;				/* result code */

	int	i;

	/* store the current HFS directory ID */
	if ((id = hfs_getcwd(vol)) == 0)
	    return(-1);

#ifdef DEBUG
	fprintf(stderr,"directory = %s depth = %d\n", node->whole_name, node->depth);
#endif /* DEBUG */

	/* loop through the ISO directory entries and process files */
	for(s_entry = node->contents; s_entry; s_entry = s_entry->next)
	{
	    /* ignore directory and associated (rsrc) files */
	    if(s_entry->isorec.flags[0])
		    continue;

	    /* ignore any non-Mac type file */
	    if(!s_entry->hfs_ent)
		    continue;

#ifdef DEBUG
	    fprintf(stderr," Name = %s", s_entry->whole_name); 
	    fprintf(stderr,"   Startb =  %d\n", s_entry->starting_block);
#endif /* DEBUG */

	    ent = s_entry->hfs_ent;

	    /* create file */
	    if ((hfs_create(vol, ent->name, ent->type, ent->creator)) < 0)
	    {
		sprintf(hce->error, "can't HFS create file %s", s_entry->whole_name);
		return(-1);
	    }

	    /* open file */
	    if ((hfp = hfs_open(vol, ent->name)) == 0)
	    {
		sprintf(hce->error, "can't HFS create open %s",
			s_entry->whole_name);
		return(-1);
	    }

	    /* if it has a data fork, then "write" it out */
	    if (ent->dsize)
		write_fork(hfp, ent->dsize);

	    /* if it has a resource fork, set the fork and "write" it out */
	    if (ent->rsize)
	    {
		if ((hfs_setfork(hfp, 1)) < 0)
		    return(-1);
		write_fork(hfp, ent->rsize);
	    }

	    /* update any HFS file attributes */
	    if ((hfs_fsetattr(hfp, ent)) < 0)
	    {
		sprintf(hce->error, "can't HFS set attributes %s",
			s_entry->whole_name);
		return(-1);
	    }

	    /* get the ISO starting block of data fork (may be zero)
	       and convert to the equivalent HFS block */
	    if (ent->dsize)
		dext = s_entry->starting_block * BLK_CONV - drAlBlSt;
	    else
		dext = 0;

	    /* if the file has a resource fork (associated file), get it's
	       ISO starting block and convert as above */
	    if (s_entry->assoc && ent->rsize)
		rext = s_entry->assoc->starting_block * BLK_CONV - drAlBlSt;
	    else
		rext = 0;

	    /* close the file and update the starting blocks */
	    if (hfs_close(hfp, dext, rext) < 0)
	    {
		sprintf(hce->error, "can't HFS close file %s",
			s_entry->whole_name);
		return(-1);
	    }
	}

	/* process sub-directories */
	for(dpnt = node->subdir; dpnt; dpnt = dpnt->next)
	{
	    ent = dpnt->hfs_ent;

	    /* if we don't have hfs entries, then this is a "deep"
	       directory - this will be processed later */
	    if (!ent)
		continue;

	    /* make sub-folder */
	    if (hfs_mkdir(vol, ent->name) < 0)
		return(-1);

	    /* change to sub-folder */
	    if (hfs_chdir(vol, ent->name) < 0)
		return(-1);

	    /* recursively copy files ... */
	    ret = copy_to_mac_vol(vol, dpnt, drAlBlSt);
	    if (ret < 0)
		return(ret);

	    /* change back to this folder */
	    if (hfs_setcwd(vol, id) < 0)
		return(-1);
	}

	/* if we have a "reloc" entry, then we process it here -
	   this will only happen if we have deep directories that
	   have relocated to the "rr_moved" directory in the root
	   directory. This list of directory entries are not "real"
	   directory entries, they only hold the name and HFS entries,
	   the "real" entry is stored in the parent field */
	for(dpnt = node->reloc; dpnt; dpnt = dpnt->reloc)
	{
	    ent = dpnt->hfs_ent;

	    /* make sub-folder */
	    if (hfs_mkdir(vol, ent->name) < 0)
		return(-1);

	    /* change to sub-folder */
	    if (hfs_chdir(vol, ent->name) < 0)
		return(-1);

	    /* recursively copy files from the real actual directory */
	    ret = copy_to_mac_vol(vol, dpnt->parent, drAlBlSt);
	    if (ret < 0)
		return(ret);

	    /* change back to this folder */
	    if (hfs_setcwd(vol, id) < 0)
		return(-1);
	}

	return(0);
}
#endif /* APPLE_HYB */ 
