------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S I N P U T . L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--     Copyright (C) 1992,1993,1994,1995 Free Software Foundation, Inc.     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with Alloc;  use Alloc;
with Atree;  use Atree;
with Debug;  use Debug;
with Namet;  use Namet;
with Osint;  use Osint;
with Output; use Output;
with System; use System;

with Unchecked_Conversion;

package body Sinput.L is

   -------------------------------
   -- Adjust_Instantiation_Sloc --
   -------------------------------

   procedure Adjust_Instantiation_Sloc (N : Node_Id; A : Sloc_Adjustment) is
   begin
      Set_Sloc (N, Source_Ptr (Int (Sloc (N)) + Int (A)));
   end Adjust_Instantiation_Sloc;

   ---------------------------------
   -- Create_Instantiation_Source --
   ---------------------------------

   procedure Create_Instantiation_Source
     (X   : Source_File_Index;
      Loc : Source_Ptr;
      A   : out Sloc_Adjustment)
   is
      Xnew : Source_File_Index;
      Adj  : Source_Ptr;

   begin
      Source_File.Increment_Last;
      Xnew := Source_File.Last;

      Source_File.Table (Xnew) := Source_File.Table (X);

      Source_File.Table (Xnew).Instantiation := Loc;
      Source_File.Table (Xnew).Template      := X;

      --  Now we need to compute the new values of Source_First, Source_Last
      --  and adjust the source file pointer to have the correct virtual
      --  origin for the new range of values.

      Source_File.Table (Xnew).Source_First :=
        Source_File.Table (Xnew - 1).Source_Last + 1;

      Adj := Source_File.Table (Xnew).Source_First -
             Source_File.Table (X).Source_First;

      A := Sloc_Adjustment (Adj);

      Source_File.Table (Xnew).Source_Last :=
        Source_File.Table (X).Source_Last + Adj;

      Source_File.Table (Xnew).Sloc_Adjust :=
        Source_File.Table (X).Sloc_Adjust - Adj;

      --  For a given character in the source, a higher subscript will be
      --  used to access the instantiation, which means that the virtual
      --  origin must have a corresponding lower value. We compute this
      --  new origin by taking the address of the appropriate adjusted
      --  element in the old array. Since this adjusted element will be
      --  at a negative subscript, we must suppress checks.

      declare
         pragma Suppress (All_Checks);

         function To_Source_Buffer_Ptr is new
           Unchecked_Conversion (Address, Source_Buffer_Ptr);

      begin
         Source_File.Table (Xnew).Source_Text :=
           To_Source_Buffer_Ptr
             (Source_File.Table (X).Source_Text (-Adj)'Address);
      end;

   end Create_Instantiation_Source;

   ----------------------
   -- Load_Source_File --
   ----------------------

   function Load_Source_File
     (N    : File_Name_Type)
      return Source_File_Index
   is
      Src  : Source_Buffer_Ptr;
      Lptr : Lines_Table_Ptr;
      X    : Source_File_Index;
      Lo   : Source_Ptr;
      Hi   : Source_Ptr;

   begin
      for J in 1 .. Source_File.Last loop
         if Source_File.Table (J).File_Name = N then
            return J;
         end if;
      end loop;

      --  Here we must build a new entry in the file table

      Source_File.Increment_Last;
      X := Source_File.Last;

      if X = Source_File.First then
         Lo := First_Source_Ptr;
      else
         Lo := Source_File.Table (X - 1).Source_Last + 1;
      end if;

      Read_Source_File (N, Lo, Hi, Src);

      if Src = null then
         Source_File.Decrement_Last;
         return No_Source_File;

      else
         if Debug_Flag_L then
            Write_Str ("*** Build source file table entry, Index = ");
            Write_Int (Int (X));
            Write_Str (", file name = ");
            Write_Name (N);
            Write_Eol;
            Write_Str ("  Lo = ");
            Write_Int (Int (Lo));
            Write_Eol;
            Write_Str ("  Hi = ");
            Write_Int (Int (Hi));
            Write_Eol;

            Write_Str ("  First 10 chars -->");

            for J in Lo .. Lo + 9 loop
               Write_Char (Src (J));
            end loop;

            Write_Str ("<--");
            Write_Eol;

            Write_Str ("  Last 10 chars  -->");

            for J in Hi - 10 .. Hi - 1 loop
               Write_Char (Src (J));
            end loop;

            Write_Str ("<--");
            Write_Eol;

            if Src (Hi) = EOF then
               Write_Str ("  OK, EOF at end");
            else
               Write_Str ("  error, no EOF at end");
            end if;
         end if;

         Lptr := new Lines_Table_Type (1 .. Alloc_Lines_Initial);
         Lptr (1) := Src'First;

         Source_File.Table (X).File_Name         := N;
         Source_File.Table (X).Full_File_Name    := Full_Source_Name;
         Source_File.Table (X).Reference_Name    := Full_Source_Name;
         Source_File.Table (X).Line_Offset       := 0;
         Source_File.Table (X).Source_Text       := Src;
         Source_File.Table (X).Source_First      := Lo;
         Source_File.Table (X).Source_Last       := Hi;
         Source_File.Table (X).Time_Stamp        := Current_Source_File_Stamp;
         Source_File.Table (X).Lines_Table       := Lptr;
         Source_File.Table (X).Num_Source_Lines  := 1;
         Source_File.Table (X).Keyword_Casing    := Unknown;
         Source_File.Table (X).Identifier_Casing := Unknown;
         Source_File.Table (X).Instantiation     := No_Location;
         Source_File.Table (X).Template          := No_Source_File;
         Source_File.Table (X).Sloc_Adjust       := 0;

         return X;
      end if;
   end Load_Source_File;

end Sinput.L;
