/*
 * $Id: init.c,v 1.49 1995/06/30 09:42:12 coleman Exp coleman $
 *
 * init.c - main loop and initialization routines
 *
 * This file is part of zsh, the Z shell.
 *
 * Copyright (c) 1992-1995 Paul Falstad
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * In no event shall Paul Falstad or the Zsh Development Group be liable
 * to any party for direct, indirect, special, incidental, or consequential
 * damages arising out of the use of this software and its documentation,
 * even if Paul Falstad and the Zsh Development Group have been advised of
 * the possibility of such damage.
 *
 * Paul Falstad and the Zsh Development Group specifically disclaim any
 * warranties, including, but not limited to, the implied warranties of
 * merchantability and fitness for a particular purpose.  The software
 * provided hereunder is on an "as is" basis, and Paul Falstad and the
 * Zsh Development Group have no obligation to provide maintenance,
 * support, updates, enhancements, or modifications.
 *
 */

#define GLOBALS

#include "zsh.h"

extern int sigtrapped[VSIGCOUNT];
extern List sigfuncs[VSIGCOUNT];

static int noexitct = 0;

/**/
void
main(int argc, char **argv)
{
#ifdef LC_ALL
    setlocale(LC_ALL, "");
#endif

    permalloc();

    if (!(zsh_name = strrchr(argv[0], '/')))
	zsh_name = argv[0];
    else
	zsh_name++;
    if (*zsh_name == '-')
	zsh_name++;

    setflags();
    parseargs(argv);
    init_io();
    setupvals();
    init_signals();
    heapalloc();
    runscripts();

    for (;;) {
	do
	    loop(1);
	while (tok != ENDINPUT);
	if (!(isset(IGNOREEOF) && interact)) {
#if 0
	    if (interact)
		fputs(islogin ? "logout\n" : "exit\n", stderr);
#endif
	    zexit(lastval);
	    continue;
	}
	noexitct++;
	if (noexitct >= 10) {
	    stopmsg = 1;
	    zexit(lastval);
	}
	zerrnam("zsh", (!islogin) ? "use 'exit' to exit."
		: "use 'logout' to logout.", NULL, 0);
    }
}

/* keep executing lists until EOF found */

/**/
void
loop(int toplevel)
{
    List list;

    pushheap();
    for (;;) {
	freeheap();
	errflag = 0;
	if (interact && isset(SHINSTDIN))
	    preprompt();
	hbegin();		/* init history mech        */
	intr();			/* interrupts on            */
	ainit();		/* init alias mech          */
	lexinit();              /* initialize lexical state */
	if (!(list = parse_event())) {	/* if we couldn't parse a list */
	    hend();
	    if (tok == ENDINPUT && !errflag)
		break;
	    continue;
	}
	if (hend()) {
	    if (stopmsg)	/* unset 'you have stopped jobs' flag */
		stopmsg--;
	    execlist(list, 0);
	    if (toplevel)
		noexitct = 0;
	}
	if (ferror(stderr)) {
	    zerr("write error", NULL, 0);
	    clearerr(stderr);
	}
	if (subsh)		/* how'd we get this far in a subshell? */
	    exit(lastval);
	if (((!interact || sourcelevel) && errflag) || retflag)
	    break;
	if (trapreturn) {
	    lastval = trapreturn;
	    trapreturn = 0;
	}
	if (isset('t') && toplevel) {
	    if (sigtrapped[SIGEXIT])
		dotrap(SIGEXIT);
	    exit(lastval);
	}
    }
    popheap();
}

/**/
void
setflags(void)
{
    int c;

    for (c = 0; c != 47; c++)
	opts[c] = OPT_UNSET;
    for (c = 47; c != 128; c++)
	opts[c] = OPT_INVALID;
    for (c = 'a'; c <= 'z'; c++)
	opts[c] = opts[c - 'a' + 'A'] = OPT_UNSET;
    for (c = '0'; c <= '9'; c++)
	opts[c] = OPT_UNSET;

    opts['A'] = OPT_INVALID;

    /* These are the flags we turn on by default in zsh */
    opts[BGNICE]      = OPT_SET;
    opts[NOTIFY]      = OPT_SET;
    opts[HASHCMDS]    = OPT_SET;
    opts[HASHLISTALL] = OPT_SET;
    opts[HASHDIRS]    = OPT_SET;
    opts[INTERACTIVE] = (isatty(0)) ? OPT_SET : OPT_UNSET;

    /* USEZLE is now set in init_io() after attempt to open terminal        */
    /* opts[USEZLE]      = (interact && SHTTY != -1) ? OPT_SET : OPT_UNSET; */

    if (getuid() != geteuid() || getgid() != getegid())
	opts[PRIVILEGED] = OPT_SET;

    /* The following options cause zsh to behave more like   *
     * Bourne and Korn shells when invoked as "sh" or "ksh". *
     * NOBANGHIST - don't recognize csh-style history subst  *
     * INTERACTIVECOMMENTS - allow interactive comments      *
     * IGNOREBRACES - don't perform brace expansion          *
     * NONOMATCH - don't print error for unmatched wildcards *
     * RMSTARSILENT - don't query 'rm *'                     *
     * SHWORDSPLIT - split parameters using IFS              *
     * KSHOPTIONPRINT - print options ksh-like               *
     * GLOBSUBST - text from parameters can be globbed       *
     * LOCALOPTIONS - option settings are local to functions *
     * NOEQUALS - Don't perform equals substitution          *
     * NOBADPATTERN - Leave bad glob patterns alone.         *
     * PROMPTSUBST - Perform substitutions in prompts        */

    if (strcmp(zsh_name, "sh") == 0 || strcmp(zsh_name, "ksh") == 0) {
	opts[NOBANGHIST]          = OPT_SET;
	opts[INTERACTIVECOMMENTS] = OPT_SET;
	opts[IGNOREBRACES]        = OPT_SET;
	opts[NONOMATCH]           = OPT_SET;
	opts[RMSTARSILENT]        = OPT_SET;
	opts[SHWORDSPLIT]         = OPT_SET;
	opts[KSHOPTIONPRINT]      = OPT_SET;
	opts[GLOBSUBST]           = OPT_SET;
	opts[LOCALOPTIONS]        = OPT_SET;
	opts[NOEQUALS]            = OPT_SET;
	opts[NOBADPATTERN]        = OPT_SET;
	opts[PROMPTSUBST]         = OPT_SET;
    }

    /* If we are invoked as "sh", ignore escapes in echo.         *
     * This is not the final solution: configure should be added  *
     * to make echo compatible with /bin/echo or the echo builtin *
     * of /bin/sh.  This hack makes echo compatible with the bash *
     * echo on linux.                                             */

    if (strcmp(zsh_name, "sh") == 0)
        opts[BSDECHO] = OPT_SET;
}

static char *cmd;
static int opti = OPT_UNSET;

/**/
void
parseargs(char **argv)
{
    char **x;
    int breakouter = 0, action;
    LinkList paramlist;

    hackzero = argzero = *argv;
    /* If started with name beginning with '-' *
     * then assume it is a login shell         */
    opts[LOGINSHELL] = (**(argv++) == '-') ? OPT_SET : OPT_UNSET;
    SHIN = 0;

    /* loop through command line options (begins with "-" or "+") */
    while (!breakouter && *argv && (**argv == '-' || **argv == '+')) {
	action = (**argv == '-') ? OPT_SET : OPT_UNSET;
	while (*++*argv) {
	    /* option '-b' or '--' signifies end of options */
	    if (**argv == 'b' || **argv == '-') {
		breakouter = 1;          /* fall out of the outer while loop */
		break;
	    }
	    if (opts[(int)**argv] == OPT_INVALID) {
		zerr("bad option: -%c", NULL, **argv);
		exit(1);
	    }
	    if (**argv == 'c') {         /* -c command */
		argv++;
		if (!*argv) {
		    zerr("string expected after -c", NULL, 0);
		    exit(1);
		}
		cmd = *argv;
		opts[INTERACTIVE] = opti;
		opts['c'] = OPT_SET;
		breakouter = 1;          /* fall out of the outer while loop */
		break;
	    } else if (**argv == 'o') {
		int c;

		if (!*++*argv)
		    argv++;
		if (!*argv) {
		    zerr("string expected after -o", NULL, 0);
		    exit(1);
		}
		c = optlookup(*argv);   /* find the code for this option */
		if (c == -1)
		    zerr("no such option: %s", *argv, 0);
		else {
		    if (c == 'i')
			opti = action;
		    opts[c] = action;
#ifdef HAVE_SETUID
		    if (c == PRIVILEGED && action == OPT_UNSET)
			setuid(getuid()), setgid(getgid());;
#endif
		}
		break;
	    } else {
		if (**argv == 'i')
		    opti = action;
		opts[(int)**argv] = action;
#ifdef HAVE_SETUID
		if ((int)**argv == PRIVILEGED && action == OPT_UNSET)
		    setuid(getuid()), setgid(getgid());;
#endif
	    }
	}
	argv++;
    }
    paramlist = newlinklist();
    if (*argv) {
	if (opts[SHINSTDIN] == OPT_UNSET) {
	    argzero = *argv;
	    if (opts['c'] == OPT_UNSET)
		SHIN = movefd(open(argzero, O_RDONLY));
	    if (SHIN == -1) {
		zerr("can't open input file: %s", argzero, 0);
		exit(1);
	    }
	    opts[INTERACTIVE] = opti;
	    argv++;
	}
	while (*argv)
	    addlinknode(paramlist, ztrdup(*argv++));
    } else
	opts[SHINSTDIN] = OPT_SET;
    pparams = x = (char **) zcalloc((countlinknodes(paramlist) + 1) * sizeof(char *));

    while ((*x++ = (char *)getlinknode(paramlist)));
    free(paramlist);
    argzero = ztrdup(argzero);
}

/**/
void
init_io(void)
{
    long ttpgrp;
    static char *outbuf, *errbuf;

    if (!outbuf)
	outbuf = (char *) malloc(BUFSIZ);
    if (!errbuf)
	errbuf = (char *) malloc(BUFSIZ);

/* stdout,stderr fully buffered */
#ifdef _IOFBF
    setvbuf(stdout, outbuf, _IOFBF, BUFSIZ);
    setvbuf(stderr, errbuf, _IOFBF, BUFSIZ);
#else
    setbuffer(stdout, outbuf, BUFSIZ);
    setbuffer(stderr, errbuf, BUFSIZ);
#endif

    if (shout) {
	fclose(shout);
	shout = 0;
    }

    /* Make sure the tty is opened read/write. */
    if (isatty(0))
	SHTTY = movefd(open(ttyname(0), O_RDWR));
    else
	SHTTY = movefd(open("/dev/tty", O_RDWR));

    if (SHTTY != -1) {
#if defined(JOB_CONTROL) && defined(TIOCSETD) && defined(NTTYDISC)
	int ldisc = NTTYDISC;

	ioctl(SHTTY, TIOCSETD, (char *)&ldisc);
#endif

	/* Associate terminal file descriptor with a FILE pointer */
	shout = fdopen(SHTTY, "w");

        /* We will only use zle if shell is interactive, *
         * SHTTY != -1, and shout != 0                   */
	opts[USEZLE] = (interact && shout) ? OPT_SET : OPT_UNSET;

	gettyinfo(&shttyinfo);	/* get tty state */
#if defined(__sgi)
	if (shttyinfo.tio.c_cc[VSWTCH] <= 0)	/* hack for irises */
	    shttyinfo.tio.c_cc[VSWTCH] = CSWTCH;
#endif
    } else {
	opts[USEZLE] = OPT_UNSET;
    }

#ifdef JOB_CONTROL
    if (interact && (SHTTY != -1)) {
	attachtty(GETPGRP());
	if ((mypgrp = GETPGRP()) > 0) {
	    opts[MONITOR] = OPT_SET;
	    while ((ttpgrp = gettygrp()) != -1 && ttpgrp != mypgrp) {
		sleep(1);
		mypgrp = GETPGRP();
		if (mypgrp == gettygrp())
		    break;
		killpg(mypgrp, SIGTTIN);
		mypgrp = GETPGRP();
	    }
	} else {
	    opts[MONITOR] = OPT_UNSET;
	}
    } else {
	opts[MONITOR] = OPT_UNSET;
    }
#else
    opts[MONITOR] = OPT_UNSET;
#endif
}

/**/
void
setupvals(void)
{
    struct passwd *pswd;
    struct timezone dummy_tz;
    char *ptr;
    int i;

    noeval = 0;
    curhist = 0;
    histsiz = DEFAULT_HISTSIZE;
    lithistsiz = 5;
    inithist();
    mailcheck = 60;
    logcheck = 60;
    keytimeout = 40;
    dirstacksize = -1;
    listmax = 100;
    clwords = (char **) zcalloc((clwsize = 16) * sizeof(char *));

    cmdstack = (unsigned char *) zalloc(256);
    cmdsp = 0;

    reporttime = -1;
    bangchar = '!';
    hashchar = '#';
    hatchar = '^';
    termok = 0;
    curjob = prevjob = coprocin = coprocout = -1;
    gettimeofday(&shtimer, &dummy_tz);	/* init $SECONDS */
    srand((unsigned int)(shtimer.tv_sec + shtimer.tv_usec)); /* seed $RANDOM */

    newreswdtable();    /* create hash table for reserved words */
    newaliastable();    /* create hash table for aliases        */
    newcmdnamtable();   /* create hash table for commands       */
    newcompctltable();  /* create hash table for compctls       */

    initxbindtab();  /* initialize key bindings */

    /* The following variable assignments cause zsh to behave more *
     * like Bourne and Korn shells when invoked as "sh" or "ksh".  *
     * NULLCMD=":" and READNULLCMD=":"                             */

    if (strcmp(zsh_name, "sh") == 0 || strcmp(zsh_name, "ksh") == 0) {
	nullcmd     = ztrdup(":");
	readnullcmd = ztrdup(":");
    } else {
	nullcmd     = ztrdup("cat");
	readnullcmd = ztrdup("more");
    }

    prompt  = ztrdup("%m%# ");
    prompt2 = ztrdup("> ");
    prompt3 = ztrdup("?# ");
    prompt4 = ztrdup("+ ");
    sprompt = ztrdup("zsh: correct `%R' to `%r' [nyae]? ");
    term    = ztrdup("");
    ppid    = getppid();

    baud = getbaudrate(&shttyinfo);  /* get the output baudrate */

#ifdef TIOCGWINSZ
    if (!(columns = shttyinfo.winsize.ws_col))
	columns = 80;
    if (!(lines = shttyinfo.winsize.ws_row))
	lines = 24;
#else
    columns = 80;
    lines = 24;
#endif

    if (!(ttystrname = ztrdup(ttyname(SHTTY))))
	ttystrname = ztrdup("");
    ifs         = ztrdup(" \t\n");
    timefmt     = ztrdup(DEFAULT_TIMEFMT);
    watchfmt    = ztrdup(DEFAULT_WATCHFMT);
    wordchars   = ztrdup(DEFAULT_WORDCHARS);
    fceditparam = ztrdup(DEFAULT_FCEDIT);
    tmpprefix   = ztrdup(DEFAULT_TMPPREFIX);
    postedit    = ztrdup("");
    underscore  = ztrdup("");

    mypid = getpid();

    cdpath   = mkarray(NULL);
    manpath  = mkarray(NULL);
    fignore  = mkarray(NULL);
    fpath    = mkarray(NULL);
    mailpath = mkarray(NULL);
    watch    = mkarray(NULL);
    psvar    = mkarray(NULL);

    hostnam     = (char *) zalloc(256);
    gethostname(hostnam, 256);
    hostvar     = ztrdup(hostnam);

    compctlsetup();
    namdirs     = (Nameddirs) zcalloc(sizeof(*namdirs) * 2);
    userdirsz = 2;
    userdirct = 0;

#ifdef CACHE_USERNAMES
    usernamescached = 0;
#endif /* CACHE_USERNAMES */

    zoptarg = ztrdup("");
    zoptind = 1;
    schedcmds = NULL;

    /* Set default path */
    path    = (char **) zalloc(sizeof(*path) * 4);
    path[0] = ztrdup("/bin");
    path[1] = ztrdup("/usr/bin");
    path[2] = ztrdup("/usr/ucb");
    path[3] = NULL;

    inittyptab();     /* initialize the ztypes table */
    initlextabs();

    /* We cache the uid so we know when to *
     * recheck the info for `USERNAME'     */
    cached_uid = getuid();

    /* Get password entry and set info for `HOME' and `USERNAME' */
    if ((pswd = getpwuid(cached_uid))) {
	home = ztrdup(pswd->pw_dir);
	cached_username = ztrdup(pswd->pw_name);
    } else {
	home = ztrdup("/");
	cached_username = ztrdup("");
    }

    /* Get info for `LOGNAME'.  If getlogin fails, *
     * we'll take whatever is in `USERNAME'.       */
    if (!(login_name = getlogin()))
	login_name = ztrdup(cached_username);

    /* Try a cheap test to see if we can *
     * initialize `PWD' from `HOME'      */
    if (ispwd(home))
	pwd = ztrdup(home);
    else if ((ptr = zgetenv("PWD")) && ispwd(ptr))
	pwd = ztrdup(ptr);
    else
	pwd = zgetcwd();

    oldpwd = ztrdup(pwd);  /* initialize `OLDPWD' = `PWD' */

    newparamtable();   /* Set up paramater hash table */

    if (!strcmp(term, "emacs"))   /* unset zle if using zsh under emacs */
	opts[USEZLE] = OPT_UNSET;
    times(&shtms);

#ifdef HAVE_GETRLIMIT
    for (i = 0; i != RLIM_NLIMITS; i++)
	getrlimit(i, limits + i);
#endif

    breaks = loops = 0;
    lastmailcheck = time(NULL);
    locallist = NULL;
    locallevel = sourcelevel = 0;
    trapreturn = 0;
    dirstack = newlinklist();
    bufstack = newlinklist();
    inbuf = (char *) zalloc(inbufsz = 256);
    inbufptr = inbuf + inbufsz;
    inbufct = 0;
    hsubl = hsubr = NULL;
    lastpid = 0;
    bshin = fdopen(SHIN, "r");
}

/**/
void
init_signals(void)
{
    intr();

#ifndef QDEBUG
    signal_ignore(SIGQUIT);
#endif

    install_handler(SIGHUP);
    install_handler(SIGCHLD);
    if (interact) {
	install_handler(SIGALRM);
#ifdef SIGWINCH
	install_handler(SIGWINCH);
#endif
	signal_ignore(SIGTERM);
    }
    if (jobbing) {
	long ttypgrp;

	while ((ttypgrp = gettygrp()) != -1 && ttypgrp != mypgrp)
	    kill(0, SIGTTIN);
	if (ttypgrp == -1) {
	    opts[MONITOR] = OPT_UNSET;
	} else {
	    signal_ignore(SIGTTOU);
	    signal_ignore(SIGTSTP);
	    signal_ignore(SIGTTIN);
	    signal_ignore(SIGPIPE);
	    attachtty(mypgrp);
	}
    }
    if (islogin) {
	signal_setmask(signal_mask(0));
    } else if (interact) {
	sigset_t set;

	sigemptyset(&set);
	sigaddset(&set, SIGINT);
	sigaddset(&set, SIGQUIT);
	signal_unblock(set);
    }
}

/**/
void
runscripts(void)
{
    /* Source the init scripts.  If called as "ksh" or "sh"  *
     * then we source the standard sh/ksh scripts instead of *
     * the standard zsh scripts                              */

    sourcelevel = 32768;	/* hack to avoid errexit in init scripts */

    if (strcmp(zsh_name, "ksh") == 0 || strcmp(zsh_name, "sh") == 0) {
	if (islogin)
	    source("/etc/profile");
	if (unset(PRIVILEGED)) {
	    if (islogin)
		sourcehome(".profile");
	    source(getsparam("ENV"));
	}
	else
	    source("/etc/suid_profile");
    } else {
#ifdef GLOBAL_ZSHENV
	source(GLOBAL_ZSHENV);
#endif
	if (!isset(NORCS)) {
	    if (unset(PRIVILEGED))
		sourcehome(".zshenv");
	    if (islogin) {
#ifdef GLOBAL_ZPROFILE
		source(GLOBAL_ZPROFILE);
#endif
		if (unset(PRIVILEGED))
		    sourcehome(".zprofile");
	    }
	    if (interact) {
#ifdef GLOBAL_ZSHRC
		source(GLOBAL_ZSHRC);
#endif
		if (unset(PRIVILEGED))
		    sourcehome(".zshrc");
	    }
	    if (islogin) {
#ifdef GLOBAL_ZLOGIN
		source(GLOBAL_ZLOGIN);
#endif
		if (unset(PRIVILEGED))
		    sourcehome(".zlogin");
	    }
	}
    }
    sourcelevel = 0;

    if (isset('c')) {
	if (SHIN >= 10)
	    close(SHIN);
	SHIN = movefd(open("/dev/null", O_RDONLY));
	execstring(cmd, 1);
	stopmsg = 1;
	zexit(lastval);
    }
    if (interact && !isset(NORCS))
	readhistfile(getsparam("HISTFILE"), 0);
    adjustwinsize();   /* check window size and adjust if necessary */
    if (isset('t') && opti == OPT_UNSET)
	prompt = ztrdup("");
}

/* source a file */

/**/
int
source(char *s)
{
    int fd        = SHIN;            /* store the shell input fd                   */
    int cj        = thisjob;         /* store our current job number               */
    int oldlineno = lineno;          /* store our current lineno                   */
    int oldshst   = opts[SHINSTDIN]; /* store current value of this option         */
    FILE *obshin  = bshin;           /* store file handle for buffered shell input */
    int osubsh    = subsh;           /* store whether we are in a subshell         */
    /* Current while, for, ... loops are invisible to the source'd script          */
    int oloops    = loops;           /* stored the # of nested loops we are in     */

    loops  = 0;
    lineno = 0;
    opts[SHINSTDIN] = OPT_UNSET;
    if ((SHIN = movefd(open(s, O_RDONLY))) == -1) {
	SHIN = fd;
	thisjob = cj;
	opts[SHINSTDIN] = oldshst;
	return 1;
    }
    bshin = fdopen(SHIN, "r");
    subsh = 0;
    sourcelevel++;
    loop(0);                    /* now we got everything set up, so loop through *
                                 * the file to be sourced                        */
    sourcelevel--;
    fclose(bshin);
    bshin = obshin;             /* restore file handle for buffered shell input */
    subsh = osubsh;             /* restore whether we are in a subshell         */
    opts[SHINSTDIN] = oldshst;  /* restore old value of this option             */
    SHIN = fd;                  /* restore the shell input fd                   */
    thisjob = cj;               /* restore current job number                   */
    loops = oloops;             /* restore the # of nested loops we are in      */
    lineno = oldlineno;         /* restore our current lineno                   */
    errflag = 0;
    retflag = 0;
    return 0;
}

/* Try to source a file in the home directory */

/**/
void
sourcehome(char *s)
{
    char buf[PATH_MAX];
    char *h;

    if (!(h = getsparam("ZDOTDIR")))
	h = home;
    sprintf(buf, "%s/%s", h, s);
    (void)source(buf);
}

/**/
void
ainit(void)
{
    alstackind = 0;		/* reset alias stack */
    alstat = 0;
    isfirstln = 1;
}

/**/
void
compctlsetup(void)
{
    static char
        *os[] =
    {"setopt", "unsetopt", NULL}, *vs[] =
    {"export", "typeset", "vared", "unset", NULL}, *cs[] =
    {"which", "builtin", NULL}, *bs[] =
    {"bindkey", NULL};

    compctl_process(os, CC_OPTIONS, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, 0);
    compctl_process(vs, CC_VARS, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, 0);
    compctl_process(bs, CC_BINDINGS, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, 0);
    compctl_process(cs, CC_COMMPATH, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
		    NULL, NULL, 0);
    cc_compos.mask = CC_COMMPATH;
    cc_default.refc = 10000;
    cc_default.mask = CC_FILES;
    cc_first.refc = 10000;
    cc_first.mask = 0;
}

